
#include <iostream>
#include <fstream>
#include <algorithm>
#include <string>
#include <utility>
#include <iterator>
#include <array>
#include <limits>
#include <vector>
#include <set>
#include <cassert>
#include <cfloat>
#include <list>
#include <cmath>
#include <map>
#include <filesystem>
#include <thread>
#include <mutex>
#include <unordered_set>
#include <unordered_map>
#include <cstring>

using namespace std;

static unsigned char getbyte(FILE *inf)
{
  unsigned char buf[1];
  fread(buf,1,1,inf);
  return buf[0];
}

static unsigned short getshort(FILE *inf)
{
  unsigned char buf[2];
  fread(buf,2,1,inf);
  return (buf[0]<<8)+(buf[1]<<0);
}

static unsigned long getlong(FILE *inf)
{
  unsigned char buf[4];
  fread(buf,4,1,inf);
  return (buf[0]<<24)+(buf[1]<<16)+(buf[2]<<8)+(buf[3]<<0);
}

static void putbyte(FILE *outf, unsigned char x)
{
  unsigned char buf[1] = { x };
  fwrite(buf,1,1,outf);
}

static void putshort(FILE *outf, unsigned short x)
{
  unsigned char buf[2] = { x >> 8, x & 0xffu };
  fwrite(buf,2,1,outf);
}

static void putlong(FILE *outf, unsigned long x)
{
  unsigned char buf[4] = { x >> 24, (x >> 16) & 0xffu , (x >> 8) & 0xffu, x & 0xffu  };
  fwrite(buf,4,1,outf);
}

// https://media.xiph.org/svt/SGIIMAGESPEC

void writeRGBAFile(FILE *const out, int w, int h, unsigned char* pixels)
{
    putshort(out, 474); // magic
    putbyte (out,   0); // storage
    putbyte (out,   1); // bpc
    putshort(out,   3); // dimensions
    putshort(out,   w); // xsize
    putshort(out,   h); // ysize
    putshort(out,   3); // zsize
    putlong (out,   0); // pixmin
    putlong (out,   0xffffff); // pixmax
    putlong (out,   0); // dummy
    char name[80] = { 0 };
    fwrite(name, 1, 80, out);
    putlong (out,   0); // colormap ID
    char dummy2[404];
    memset(dummy2, 0, sizeof(dummy2));
    fwrite(dummy2, 1, 404, out);

    unsigned char* pixels2 = (unsigned char*)malloc(w * h * 3);
    // re-interleave
    for(int y = 0; y < h; ++y)
    {
        for(int x = 0; x < w; ++x)
        {
            for(int z = 0; z < 3; ++z)
            {
                pixels2[(x + y * w) + z * w * h] = pixels[z + (x + (h - 1 - y) * w) * 3];
            }
        }
    }

    fwrite(pixels2, 1, w * h * 3, out);
    
    free(pixels2);
}

static void setCrumb(uint8_t* buffer, int pos, int crumb)
{
    assert(crumb >= 0 && crumb < 4);
    buffer[(pos / 8) * 2 + 0] |= (crumb & 1) << (7 - (pos & 7));
    buffer[(pos / 8) * 2 + 1] |= (crumb >> 1) << (7 - (pos & 7));
}

static int getCrumb(const uint8_t* buffer, int pos)
{
    int crumb = 0;
    crumb = ((buffer[(pos / 8) * 2 + 0] >> (7 - (pos & 7))) & 1) |
             (((buffer[(pos / 8) * 2 + 1] >> (7 - (pos & 7))) & 1) << 1);
    assert(crumb >= 0 && crumb < 4);
    return crumb;
}

static void writeBytesToFile(const char* filename, const vector<uint8_t>& bytes)
{
    FILE* out = fopen(filename, "wb");
    assert(out);
    fwrite(bytes.data(), 1, bytes.size(), out);
    fclose(out);
}

static const int screen_width = 160, screen_height = 144;
static const int half_screen_width = screen_width / 2;
static const int num_depths = 64;
static const double k = 1.5;

static void setPixel(uint8_t* buffer, int w, int h, int x, int y, int col)
{
    assert(x >= 0 && x < w);
    assert(y >= 0 && y < h);
    const int tile_x = x / 8, tile_y = y / 8;
    assert(tile_x < (w / 8));
    const int tile_index = tile_x + tile_y * (w / 8);
    setCrumb(buffer, tile_index * 64 + (x & 7) + (y & 7) * 8, col);
}

static int getPixel(const uint8_t* buffer, int w, int h, int x, int y)
{
    const int tile_x = x / 8, tile_y = y / 8;
    const int tile_index = tile_x + tile_y * (w / 8);
    return getCrumb(buffer, tile_index * 64 + (x & 7) + (y & 7) * 8);
}


int main(int argc, char** argv)
{
    const int crumbs_even[2] = { 0b00, 0b01 };
    const int crumbs_odd[2] = { 0b10, 0b11 };

    const size_t strip_table_buffer_size = half_screen_width / 8 * num_depths * 2;

    vector<uint8_t> strip_table_buffer(strip_table_buffer_size, 0);
    vector<uint8_t> U_step_table(num_depths, 0);
    vector<uint8_t> palette_table(num_depths * 4, 0);
    vector<uint8_t> sine_table(256, 0);

    for(int i = 0; i < 256; ++i)
        sine_table[i] = std::max(0, std::min(num_depths - 1, int(std::sin(double(i) / 256.0 * M_PI * 2.0) * num_depths / 2 + num_depths / 2)));

    cout << "strip_table_buffer.size() = " << strip_table_buffer.size() << endl;
    cout << "U_step_table.size() = " << U_step_table.size() << endl;
    cout << "palette_table.size() = " << palette_table.size() << endl;
    cout << "sine_table.size() = " << palette_table.size() << endl;

    cout << "total size = " << (palette_table.size() + U_step_table.size() + strip_table_buffer.size() + sine_table.size()) << endl;

    assert((strip_table_buffer.size() & 15) == 0);
    cout << "num strip tiles = " << strip_table_buffer.size() / 16 << endl;

    for(int depth = 0; depth < num_depths; ++depth)
    {
        const double Ud_step = (128.0 / double((depth + 1) * k + 128 - num_depths * k)) * 3.0;
        double Ud = 0.0 - Ud_step * half_screen_width;
        for(int x = 0; x < half_screen_width; ++x)
        {
            if(uint8_t(Ud) & 128)
            {
                setPixel(strip_table_buffer.data(), half_screen_width, num_depths, x, depth, crumbs_odd[x & 1]);
                //setCrumb(strip_table_buffer.data(), depth * half_screen_width + x, crumbs_odd[x & 1]);
            }
            else
            {
                setPixel(strip_table_buffer.data(), half_screen_width, num_depths, x, depth, crumbs_even[x & 1]);
                //setCrumb(strip_table_buffer.data(), depth * half_screen_width + x, crumbs_even[x & 1]);
            }
            Ud += Ud_step;
        }
        U_step_table[depth] = 255 - uint8_t(Ud_step + 0.5);

#define COMPOSE_PALETTE(c0, c1, c2, c3) ((c0) | ((c1) << 2) | ((c2) << 4) | ((c3) << 6))

        // First
        {
            const int depth_col = std::max(0, std::min(6, (depth * 6 + num_depths - 1) / num_depths));
            const bool dither = depth_col & 1;
            const int depth_crumb = depth_col / 2;

            // Even cells
            if(dither)
            {
                palette_table[depth * 4 + 0] |= COMPOSE_PALETTE(depth_crumb + 1, depth_crumb + 0, 0, 0); // Even rows
                palette_table[depth * 4 + 1] |= COMPOSE_PALETTE(depth_crumb + 0, depth_crumb + 1, 0, 0); // Odd rows
            }
            else
            {
                palette_table[depth * 4 + 0] |= COMPOSE_PALETTE(depth_crumb, depth_crumb, 0, 0); // Even rows
                palette_table[depth * 4 + 1] |= COMPOSE_PALETTE(depth_crumb, depth_crumb, 0, 0); // Odd rows
            }

            // Odd cells
            if(dither)
            {
                palette_table[depth * 4 + 2] |= COMPOSE_PALETTE(0, 0, depth_crumb + 1, depth_crumb + 0); // Even rows
                palette_table[depth * 4 + 3] |= COMPOSE_PALETTE(0, 0, depth_crumb + 0, depth_crumb + 1); // Odd rows
            }
            else
            {
                palette_table[depth * 4 + 2] |= COMPOSE_PALETTE(0, 0, depth_crumb, depth_crumb); // Even rows
                palette_table[depth * 4 + 3] |= COMPOSE_PALETTE(0, 0, depth_crumb, depth_crumb); // Odd rows
            }
        }

        // Second
        {
            const int depth_col = std::max(0, std::min(6, (depth * 6 + num_depths - 1) / num_depths - 2));
            const bool dither = depth_col & 1;
            const int depth_crumb = depth_col / 2;

            // Even cells
            if(dither)
            {
                palette_table[depth * 4 + 0] |= COMPOSE_PALETTE(0, 0, depth_crumb + 1, depth_crumb + 0); // Even rows
                palette_table[depth * 4 + 1] |= COMPOSE_PALETTE(0, 0, depth_crumb + 0, depth_crumb + 1); // Odd rows
            }
            else
            {
                palette_table[depth * 4 + 0] |= COMPOSE_PALETTE(0, 0, depth_crumb, depth_crumb); // Even rows
                palette_table[depth * 4 + 1] |= COMPOSE_PALETTE(0, 0, depth_crumb, depth_crumb); // Odd rows
            }

            // Odd cells
            if(dither)
            {
                palette_table[depth * 4 + 2] |= COMPOSE_PALETTE(depth_crumb + 1, depth_crumb + 0, 0, 0); // Even rows
                palette_table[depth * 4 + 3] |= COMPOSE_PALETTE(depth_crumb + 0, depth_crumb + 1, 0, 0); // Odd rows
            }
            else
            {
                palette_table[depth * 4 + 2] |= COMPOSE_PALETTE(depth_crumb, depth_crumb, 0, 0); // Even rows
                palette_table[depth * 4 + 3] |= COMPOSE_PALETTE(depth_crumb, depth_crumb, 0, 0); // Odd rows
            }
        }

#undef COMPOSE_PALETTE
    }

    for(uint8_t& x : palette_table)
        x ^= 0xFF;

    writeBytesToFile("board_strip_tiles.bin", strip_table_buffer);
    writeBytesToFile("board_u_step_table.bin", U_step_table);
    writeBytesToFile("board_palette_table.bin", palette_table);
    writeBytesToFile("board_sine_table.bin", sine_table);

    const uint32_t testimage_w = 160, testimage_h = 144;
    const uint32_t num_testimage_bytes = testimage_w * testimage_h * 3;
    uint8_t* testimage = (uint8_t*)malloc(num_testimage_bytes);
    memset(testimage, 0, num_testimage_bytes);

    //const uint8_t palettes[2] = { 0b11110000, 0b00001111 };

    uint8_t U = 0;
    for(int y = 0; y < screen_height; ++y)
    {
        //const int depth = (y * num_depths) / screen_height;
        //const int depth = std::max(0, std::min(num_depths - 1, int((std::cos(double(y) / 20.) * 0.5 + 0.5) * num_depths )));
        const int depth = y % num_depths;
        for(int x = 0; x < screen_width / 2; ++x)
        {
            //const int crumb = getCrumb(strip_table_buffer.data(), depth * half_screen_width + x);
            //const int pal_index = crumb;
            const int pal_index = getPixel(strip_table_buffer.data(), half_screen_width, num_depths, x, depth);

            //int pal_col = (palettes[(U & 128) ? 1 : 0] >> (pal_index * 2)) & 3;
            const uint8_t* pal = palette_table.data() + depth * 4 + (y & 1);
            const int pal_col = pal_index; // (pal[(U & 128) ? 2 : 0] >> (pal_index * 2)) & 3;

            const int col = std::max(0, std::min(255, (int)pal_col * (255 / 3)));
            testimage[(x + y * screen_width) * 3 + 0] = col;
            testimage[(x + y * screen_width) * 3 + 1] = col;
            testimage[(x + y * screen_width) * 3 + 2] = col;
        }
        const uint8_t U_step = U_step_table[depth];
        U += U_step;
    }

    {
        FILE* out = fopen("checkerboard_test.rgba", "wb");
        writeRGBAFile(out, testimage_w, testimage_h, testimage);
        fclose(out);
    }

    cout << "done." << endl;
}
