
; font_small.png came from https://opengameart.org/content/small-blocky-font
; https://opengameart.org/content/32x32-fantasy-tileset

; bit9x9 came from https://fontstruct.com/fontstructions/show/559959/bit9x9 and https://www.mattlag.com/bitfonts/

; Pocket Tangrams
; by Clockwork Cats

INCLUDE "hardware.inc"

DEF TANGRAM_BG_PALETTE              EQU %10011000
DEF HUD_PALETTE                     EQU %00110011
DEF INGAME_MENU_BG_PALETTE          EQU %00001111
DEF INGAME_MENU_BG_PALETTE_2        EQU %00110011
DEF TEMPS_SIZE                      EQU $A
DEF TILE_INDEX_ZERO                 EQU $28
DEF TILE_INDEX_COLON                EQU $3A
DEF TILE_INDEX_ARROW                EQU $7E
DEF TILE_INDEX_RETICLE              EQU $7D
DEF TILE_INDEX_BLANK                EQU $7B
DEF TILE_INDEX_EMPTY                EQU $7C
DEF TILE_INDEX_OPAQUE               EQU $7A
DEF NUM_IN_GAME_MENU_ITEMS          EQU 6
DEF TIME_DISPLAY_MAP_BASE_ADDRESS   EQU $9A2D
DEF STATEC_PATCHED_ALLOCATION_SIZE  EQU $0BD2
DEF LOGO_MAP_ALLOCATION_SIZE        EQU $0C00
DEF NUM_LOGO_TILES                  EQU 38
DEF MEGASCROLLER_TILE_BASE          EQU $A0
DEF NUM_TANGRAMS                    EQU 38

DEF INGAME_MENU_TRANSITION_COUNTER_INIT     EQU 72
DEF INGAME_MENU_TRANSITION_COUNTER_LIMIT    EQU (32 - 8)


NUM_PATCH_LOCATIONS = 0
MACRO AddPatchLocation
DEF \1_patch_{d:NUM_PATCH_LOCATIONS} EQU @ - \1
NUM_PATCH_LOCATIONS = NUM_PATCH_LOCATIONS + 1
ENDM

MACRO WriteBorderGraphicsToTileFF
    ld hl, $9800 - $10
    ld a, %11111111
    ld c, 8
:
    ld [hli], a     ; 2
    inc hl          ; 2
    dec c           ; 1
    jr nz, :-
/*
REPT 8
    ld [hli], a     ; 2
    inc hl          ; 2
ENDR
*/
ENDM


MACRO SpinWaitForLine
IF 1
    ld b, \1        ; 2
    rst SpinWaitForLineRST
ELSE
    ld a, \1
    ldh [rLYC], a
    ld a, STATF_LYC
    ldh [rSTAT], a
    xor a, a
    ld [rIF], a
    halt
ENDC
ENDM

MACRO SpinWaitForVBlank
    ld b, 144        ; 2
    rst SpinWaitForLineRST
ENDM

MACRO SpinWaitIfNotInVBlank
    ld b, 144       ; 2
:
    ldh a, [rLY]    ; 3                     ; 2B
    cp a, b         ; 1                     ; 1B
    jr c, :-        ; 3 taken / 2 untaken   ; 2B
ENDM


MACRO SpinWaitForMode0
IF 1
    ld hl, rSTAT
    ld a, 3
:
    and a, [hl]     ; 2
    jr nz, :-       ; 3 taken / 2 untaken
ELSE
    ld hl, rSTAT ; This is not really needed, but expected by other code unfortunately
    ld a, STATF_MODE00
    ldh [rSTAT], a
    xor a, a
    ld [rIF], a
    halt
ENDC
ENDM

MACRO SpinWaitForMode3
    ld hl, rSTAT
    ld a, 3
:
    and a, [hl]     ; 2
    cpl
    jr nz, :-       ; 3 taken / 2 untaken
ENDM


MACRO CopyTileToHRAMUnrolled
    ld hl, \2
    ld c, LOW(\1)
REPT 15
    ld a, [hli]     ; 2
    ldh [c], a      ; 2
    inc c           ; 1
ENDR
    ld a, [hli]     ; 2
    ldh [c], a      ; 2
ENDM

MACRO CopyTileFromHRAMUnrolled
    ld hl, \1
    ld c, LOW(\2)
REPT 15
    ldh a, [c]      ; 2
    ld [hli], a     ; 2
    inc c           ; 1
ENDR
    ldh a, [c]      ; 2
    ld [hli], a     ; 2
ENDM

MACRO FillTileUnrolled
    ld a, \2
    ld hl, \1
REPT 16
    ld [hli], a     ; 2
ENDR
ENDM

MACRO SetBackgroundPalette
    ld a, \1
    ldh [rBGP], a
ENDM

MACRO ConditionallySetBackgroundPalette
    ldh a, [DoNotChangePaletteFlag]
    or a, a
    jr nz, :+
    SetBackgroundPalette \1
:
ENDM

MACRO SetLCDControl
    ld a, \1
    ldh [rLCDC], a
ENDM

MACRO SetLCDControlModified
    ldh a, [LCDControlModified_Mask]
    and a, \1
    ld h, a
    ldh a, [LCDControlModified_Bits]
    or a, h
    ldh [rLCDC], a
ENDM

MACRO SetLCDCModifiedMaskAndBits
    ld a, \1
    ldh [LCDControlModified_Mask], a
    ld a, \2
    ldh [LCDControlModified_Bits], a
ENDM

MACRO MemCopyStride256Unrolled
REPT \1 - 1
    ld a, [hli]     ; 2
    ld [de], a      ; 2
    inc d           ; 1
ENDR
    ld a, [hl]      ; 2
    ld [de], a      ; 2
ENDM

MACRO LoadPiece
FOR I, Piece_SIZEOF
    ld a, [hli]
    ldh [HeldPieceCoords + I], a
ENDR
ENDM

MACRO LoadWordFromAddress
IF HIGH(\2) == $FF
    ldh a, [\2 + 0]
    ld  LOW(\1), a
    ldh a, [\2 + 1]
    ld HIGH(\1), a
ELSE
    ld  a, [\2 + 0]
    ld  LOW(\1), a
    ld  a, [\2 + 1]
    ld HIGH(\1), a
ENDC
ENDM

MACRO StoreWordToAddress
IF HIGH(\1) == $FF
    ld  a, LOW(\2)
    ldh [\1 + 0], a
    ld  a, HIGH(\2)
    ldh [\1 + 1], a
ELSE
    ld  a, LOW(\2)
    ld  [\1 + 0], a
    ld  a, HIGH(\2)
    ld  [\1 + 1], a
ENDC
ENDM

MACRO CopyTilesFixed
    ld hl, \2
    ld de, \1
    ld c, \3
    call CopyTiles
ENDM

MACRO CopyTilesFixedStride2
    ld hl, \2
    ld de, \1
    ld c, \3
    call CopyTilesStride2
ENDM

MACRO CopyTilesFixedStride2Source1BPP
    ld hl, \2
    ld de, \1
    ld c, \3
    call CopyTilesStride2Source1BPP
ENDM


MACRO ClearTilesFixed
    ld hl, \1
    ld c, \2
    call ClearTiles
ENDM

MACRO DrawTextFixed
    ld hl, \2
    ld de, \1
    call DrawText
ENDM

MACRO CallIotaMem
IF \3 == 1
    ld a, \2
    ld [\1], a
ELSE
    ld hl, \1
    ld a, \2
    ld c, \3
    call IotaMem
ENDC
ENDM


MACRO CallSetMem
    ld hl, \1
    ld a, \2
    ld bc, \3
    call SetMem
ENDM


MACRO CheckPendingStateAddress
    LoadWordFromAddress hl, PendingStateAddress
    ld a, h
    or a, l
    jr z, :+
    xor a, a
    ld [PendingStateAddress + 0], a
    ld [PendingStateAddress + 1], a
    jp hl
:
ENDM

MACRO Add8To16WithCarry
    ld a, LOW(\1)
    add a, \2
    ld LOW(\1), a
    ld a, HIGH(\1)
    adc 0
    ld HIGH(\1), a
ENDM


                        RSRESET
DEF Piece_ActiveMask    RB
DEF Piece_X             RB
DEF Piece_Y             RB
DEF Piece_Angle         RB
DEF Piece_Vertices      RW
DEF Piece_FGDraw        RW
DEF Piece_BGDraw        RW
DEF Piece_BoundXMin     RB
DEF Piece_BoundYMin     RB
DEF Piece_BoundXMax     RB
DEF Piece_BoundYMax     RB
DEF Piece_SIZEOF        RB 0

assert Piece_SIZEOF <= 16

MACRO PieceData
    db \1 ; ActiveMask
    db \2 ; X
    db \3 ; Y
    db \4 ; Angle
    dw \5 ; Vertices
    dw \6 ; FGDraw
    dw \7 ; BGDraw
    db \8 ; Bound X Min
IF \9 > 16
    db \9 - 16 ; Bound Y Min
ELSE
    db 0 ; Bound Y Min
ENDC
    db \<10> ; Bound X Max
IF \<11> < (127 - 16)
    db \<11> + 16 ; Bound Y Max
ELSE
    db 127 ; Bound Y Max
ENDC
    ds 16 - Piece_SIZEOF
ENDM

SECTION "StateC Patched Copy", WRAMX[$D000]
UNION
StateC_Patched:
    ds STATEC_PATCHED_ALLOCATION_SIZE
NEXTU
LogoMap:
    ds LOGO_MAP_ALLOCATION_SIZE
ENDU
RecordTimes:
    ds NUM_TANGRAMS * 4

SECTION "FGFrameBuffer", WRAM0[$C000]
UNION
FGFrameBuffer:
    ds $200
NEXTU
MegaScrollerMapBuffer:
    ds $100
ENDU

SECTION "Variables", WRAM0[$C000 + $200]
VariablesStart:
PieceCoords:
    ds 7 * 16
ActivePieceOffset:
    db
CurrentActionState:
    dw
LastJoypadState:
    db
PendingStateAddress:
    dw
TimeCounter:
    db
TangramToLoad:
    dw
InGameMenuState:
    db
InGameMenuTransitionCounter:
    db
InGameMenuScrollX:
    db
InGameMenuScrollY:
    db
InGameMenuPalette:
    db
SelectionScreen_FrameCounter:
    db
SelectionScreen_TangramIndex:
    db
SelectionScreen_IndexDigits:
    dw
SelectionScreen_DigitPos:
    db
SelectionScreen_RedrawMode:
    db
SelectionScreen_PreviewTimer:
    db
TimerFreeze:
    db
AutoSolveWasUsed:
    db

Elapsed:
.seconds
    db
.decaseconds
    db
.minutes
    db
.decaminutes
    db

VariablesEnd:

assert VariablesEnd - VariablesStart < 512
assert (LOW(PieceCoords) + 7 * 16) < 256

SECTION "Sprites", WRAM0[$C000 + $400]
Sprites:
    ds $A0

SECTION "TextTilesBuffer", WRAM0, ALIGN[8, 0]
TextTilesBuffer:
    ds $100
PolygonFillBuffer:
    ds 128
PostPolygonFillBuffer:
    ds $100 - 128

SECTION "BGFrameBuffer", WRAM0, ALIGN[8, 0]
PreBGFrameBuffer:
    ds $100
BGFrameBuffer:
    ds $800

SECTION "HRAM", HRAM[$FF80]
UNION
Temps:
    ds TEMPS_SIZE
NEXTU
DrawText_SourceAddr:
    dw
DrawText_DestAddr:
    dw
DrawText_PixelColumn:
    db
DrawText_RowCount:
    db
DrawText_CharWidth:
    db
LCDControlModified_Mask:
    db
LCDControlModified_Bits:
    db
DoNotChangePaletteFlag:
    db
NEXTU
TitleScreen_FrameCounter:
    db
TitleScreen_SineOffset:
    db
TitleScreen_MegaScrollPos:
    db
TitleScreen_MegaScrollSubPos:
    db
TitleScreen_MegaScrollReadAddr:
    dw
TitleScreen_RasterScrollY:
    db
TitleScreen_RasterPalette:
    db
ENDU
TempsEnd:
assert TempsEnd - Temps == TEMPS_SIZE
SpriteDMARoutine:
    ds 5
UNION
TileBackupA:
    ds 16
NEXTU
TangramTitleStringTemp:
    ds 16
ENDU
TileBackupB:
    ds 16
Vertices:
    ds 8
LineSlope:
    db
LineSlope2:
    db
LineParams: ; x0, y0, x1, y1
    ds 4
HeldPieceCoords:
    ds Piece_SIZEOF
EndHRAM:

assert (EndHRAM - $FF80) < 96 ; Allow space for the stack
;assert ($FFFE - $7A) >= (Temps + TEMPS_SIZE) ; Stack for hUGEDriver (?)


SECTION "restarts", ROM0[$0000]
SpinWaitForLineRST:
:
    ldh a, [rLY]    ; 3                     ; 2B
    cp a, b         ; 1                     ; 1B
    jr nz, :-       ; 3 taken / 2 untaken   ; 2B
    ret             ; 4                     ; 1B
RST00END:
assert (RST00END - $00) < 8
REPT $0008 - RST00END
    nop
ENDR
; $0008
ret
REPT 7
    nop
ENDR
; $0010
ret
REPT 7
    nop
ENDR
; $0018
ret
REPT 7
    nop
ENDR
; $0020
ret
REPT 7
    nop
ENDR
; $0028
ret
REPT 7
    nop
ENDR
; $0030
ret
REPT 7
    nop
ENDR
; $0038
ret
REPT 7
    nop
ENDR




SECTION "V-Blank Interrupt", ROM0[$40]
    nop
    jp VBlankHandler

SECTION	"STAT Handler",ROM0[$48]
    nop
    jp HBlankHandler


SECTION "Timer overflow interrupt", ROM0[$0050]
    reti

SECTION "Serial transfer completion interrupt", ROM0[$0058]
    reti

SECTION "P10-P13 signal low edge interrupt", ROM0[$0060]
    reti



SECTION "Header", ROM0[$100]
    jp EntryPoint
SECTION "Nintendo Logo", ROM0[$0104]
    NINTENDO_LOGO
SECTION "ROM Info", ROM0[$0134]
    db "POCKET TANGRAMS", 0 ; Title
    db "CC"                 ; New Licensee Code
    db 0                    ; SGB Flag
    db 3                    ; Cartridge Type (MBC1+RAM+BATTERY)
    db 0                    ; ROM Size (32 KB)
    db 2                    ; RAM Size (8 KB)
    db 1                    ; Destination (non-Japan)

    ds $150 - @, 0 ; Make room for the rest of the header



SECTION "EntryPoint", ROM0[$150]
EntryPoint:
    di

    ; Reset record times in WRAM
    CallSetMem RecordTimes, 9, NUM_TANGRAMS * 4

    ; Check SRAM for integrity string and clear records if it is not intact.

    ; Enable external RAM
    ld a, $0A
    ld [$0000], a
    ; Inspect SRAM for integrity
    ld hl, $A000 + NUM_TANGRAMS * 4
    ld de, SRAMCheckString
    ld c, 9
:
    ld a, [de]
    ld b, a
    ld a, [hli]
    inc de
    cp a, b
    jr nz, :+
    dec c
    jr nz, :-
    ; Copy record times from SRAM to WRAM
    ld de, RecordTimes
    ld hl, $A000
    ld bc, NUM_TANGRAMS * 4
    call CopyMem
    jr :+++ ; SRAM has passed integrity check
:
    ; Reset
    CallSetMem $A000, 9, NUM_TANGRAMS * 4
    ; Write the integrity string
    ld hl, $A000 + NUM_TANGRAMS * 4
    ld de, SRAMCheckString
:
    ld a, [de]
    ld [hli], a
    inc de
    or a, a
    jr nz, :-
:
    ; Disable external RAM
    ld a, $00
    ld [$0000], a



    xor a, a

    ld [InGameMenuState], a

    ld [ActivePieceOffset], a
    ld [LastJoypadState], a

    ld [PendingStateAddress + 0], a
    ld [PendingStateAddress + 1], a

    ld [TimeCounter], a
    ld [Elapsed.seconds], a
    ld [Elapsed.decaseconds], a
    ld [Elapsed.minutes], a
    ld [Elapsed.decaminutes], a

    ld [TimerFreeze], a
    ld [SelectionScreen_TangramIndex], a

    ld [SelectionScreen_IndexDigits + 0], a
    ld [SelectionScreen_IndexDigits + 1], a

    ldh [DoNotChangePaletteFlag], a

    call copy_dma_hrampart_to_hram


    ld a, %10000000     ; Turn sound on
    ldh [rNR52], a

    ; Do not turn the LCD off outside of VBlank
    SpinWaitIfNotInVBlank

    ; Turn the LCD off and clear LCD control registers
    xor	a
    ldh	[rLCDC],a
    ldh	[rIE],a
    ldh	[rIF],a
    ldh [rSTAT],a

    ; Initialize the window position to 255,255
    dec	a
    ldh	[rWY], a
    ldh	[rWX], a

    ; Initialize scroll registers to 0,0
    inc a
    ldh [rSCX], a
    ldh [rSCY], a

    ; Initialize palettes
    ld a, %11100100
    ld [rBGP], a

    ld a, %11111100
    ld [rOBP0], a

    ld a, %11111100
    ld [rOBP1], a

    ; Turn the LCD on
    SetLCDControl LCDCF_ON | LCDCF_BGON | LCDCF_BG8800 | LCDCF_OBJOFF | LCDCF_OBJ16 | LCDCF_WINOFF

    SpinWaitForVBlank

    ; set up the lcdc int
    xor a, a
    ;ld a,STATF_MODE00 | STATF_LYC
    ld a, STATF_MODE00
    ldh [rSTAT],a

    ; Enable  the desired interrupts
    xor a, a
    ld a, IEF_STAT
    ldh [rIE], a

    ;di
    ;halt
    xor	a, a
    ei
    ldh	[rIF],a

    di ; Not actually using interrupts for anything for now....

    jp TitleScreen



    ;jp TangramSelectionScreen


    ;ld hl, TangramAddressTable
    ;ld a, 3
    ;call IndirectLoadWordFromTable
    ;StoreWordToAddress TangramToLoad, de

    ;jp LoadTangramAndStartGame


SECTION "TangramSelectionScreen", ROM0
TangramSelectionScreen:

    ; Reset sound
    ld a, %00000000     ; Turn sound off
    ldh [rNR52], a
    ld a, %10000000     ; Turn sound on
    ldh [rNR52], a

    ; Enable the desired sound channels
    ld a, %01110111     ; Max. master volume, no Vin
    ldh [rNR50], a
    ld a, %10011001     ; Output sound 1 to both channels
    ldh [rNR51], a


    xor a, a
    ld [TimerFreeze], a

    SetLCDControl LCDCF_ON | LCDCF_BGON | LCDCF_BG8000 | LCDCF_OBJOFF | LCDCF_OBJ16 | LCDCF_WINOFF | LCDCF_BG9C00
    SetBackgroundPalette INGAME_MENU_BG_PALETTE

    ;ld a, STATF_MODE00
    ;ldh [rSTAT],a

    ; Enable  the desired interrupts
    ;ld a, IEF_STAT
    ;ldh [rIE], a

    di ; Not actually using interrupts for anything for now....

    ; Do not turn the LCD off outside of VBlank
    SpinWaitIfNotInVBlank

    ; Turn the LCD off
    xor	a
    ldh	[rLCDC],a

    ; Copy StateC to RAM so it can be patched
    ld de, StateC_Patched
    ld hl, StateC
    ld bc, StateCEnd - StateC
    call CopyMem
    call PatchIncrementStateCLoadAddresses


    ; Clear tiles in VRAM
    ld hl, $8000
    ld bc, $1800
    call ClearMem

    ; Initialize scroll registers to 0,0
    xor a, a
    ldh [rSCX], a
    ldh [rSCY], a

    ; Initialize palettes
    ld a, %11100100
    ld [rBGP], a

    ld a, %11111100
    ld [rOBP0], a

    ld a, %11111100
    ld [rOBP1], a



    ld hl, $9800
    ld bc, 256*256/64
.cleartilemap
    ld a, $7F
    ld [hli], a
    dec bc
    ld a, b
    or a, c
    jp nz, .cleartilemap

    ld a, $80 ; tile $00
    ld [$9A00], a
    ld [$9A01], a
    ld [$9A12], a
    ld [$9A13], a

    ld a, $7F
    ld [$9800], a
    ld [$9801], a

    call MakeFramebuffer

    ld hl, Sprites
    ld bc, $00A0
    call ClearMem


    ClearTilesFixed TextTilesBuffer, $20
    ClearTilesFixed BGFrameBuffer, 128
    ClearTilesFixed FGFrameBuffer, 32

    ClearTilesFixed PreBGFrameBuffer, 16


    ; Map: Tangram name in HUD
    CallIotaMem $9802, $08, 8
    CallIotaMem $980A, $18, 8

    ; Fill the "loading" screen with empty tiles
    CallSetMem $9C00, $7F, $20 * 18

    ; Map: Tangram name in HUD ($9C00 base address, to hide loading garbage screens)
    CallIotaMem $9C02, $08, 8
    CallIotaMem $9C0A, $18, 8

        ; Resume
    DrawTextFixed TextTilesBuffer + 1, TextStrings_0
    CopyTilesFixedStride2 $8080 + 1, TextTilesBuffer + 1 + $00, 4
        ; Check if solved
    DrawTextFixed TextTilesBuffer + 1, TextStrings_1
    CopyTilesFixedStride2 $80C0 + 1, TextTilesBuffer + 1 + $00, 4
    CopyTilesFixedStride2 $8180 + 1, TextTilesBuffer + 1 + $40, 5
        ; Select Tangram
    DrawTextFixed TextTilesBuffer + 1, TextStrings_2
    CopyTilesFixedStride2 $81D0 + 1, TextTilesBuffer + 1 + $00, 3
    CopyTilesFixedStride2 $8280 + 1, TextTilesBuffer + 1 + $30, 6
        ; Solve this piece
    DrawTextFixed TextTilesBuffer + 1, TextStrings_3
    CopyTilesFixedStride2 $82E0 + 1, TextTilesBuffer + 1 + $00, 2
    CopyTilesFixedStride2 $8380 + 1, TextTilesBuffer + 1 + $20, 7
        ; Solve all pieces
    DrawTextFixed TextTilesBuffer + 1, TextStrings_4
    CopyTilesFixedStride2 $83F0 + 1, TextTilesBuffer + 1 + $00, 1
    CopyTilesFixedStride2 $8480 + 1, TextTilesBuffer + 1 + $10, 8
        ; Quit to title screen
    DrawTextFixed TextTilesBuffer + 1, TextStrings_5
    CopyTilesFixedStride2 $8580 + 1, TextTilesBuffer + 1 + $00, 8
    CopyTilesFixedStride2 $8680 + 1, TextTilesBuffer + 1 + $80, 2



        ; Solved!
    DrawTextFixed TextTilesBuffer + 1, TextStrings_6
    CopyTilesFixedStride2 $8480 + 0, TextTilesBuffer + 1 + $00, 4
        ; Not solved...
    DrawTextFixed TextTilesBuffer + 1, TextStrings_7
    CopyTilesFixedStride2 $8580 + 0, TextTilesBuffer + 1 + $00, 7
        ; New Record!
    DrawTextFixed TextTilesBuffer + 1, TextStrings_8
    CopyTilesFixedStride2 $8680 + 0, TextTilesBuffer + 1 + $00, 7


    ; Numeric digits 0 through 7 inclusive
    CopyTilesFixedStride2Source1BPP $8000 + TILE_INDEX_ZERO * $10,  DigitFontTiles + $8 * $00, 8

    ; Numeric digits 8 and 9
    CopyTilesFixedStride2Source1BPP $8100 + TILE_INDEX_ZERO * $10,  DigitFontTiles + $8 * $08, 8

    ; Colon symbol
    CopyTilesFixedStride2Source1BPP $8000 + TILE_INDEX_COLON * $10, DigitFontTiles + $8 * $0A, 1

    ; Arrow symbol
    CopyTilesFixedStride2Source1BPP $8000 + TILE_INDEX_ARROW * $10 + 1, DigitFontTiles + $8 * $0B, 1

    ; Reticle symbol
    CopyTilesFixedStride2Source1BPP $8000 + TILE_INDEX_RETICLE * $10 + 1, DigitFontTiles + $8 * $0C, 1

    ; "Blank" tile
    CopyTilesFixedStride2Source1BPP $8000 + TILE_INDEX_BLANK * $10 + 1, DigitFontTiles + $8 * $0D, 1

    ; Opaque tile
    CopyTilesFixedStride2Source1BPP $8000 + TILE_INDEX_OPAQUE * $10 + 0, DigitFontTiles + $8 * $0E, 1


    ld a, TILE_INDEX_ZERO
    ld [TIME_DISPLAY_MAP_BASE_ADDRESS + 0], a
    ld [TIME_DISPLAY_MAP_BASE_ADDRESS + 1], a
    ld [TIME_DISPLAY_MAP_BASE_ADDRESS + 3], a
    ld [TIME_DISPLAY_MAP_BASE_ADDRESS + 4], a
    ld a, TILE_INDEX_COLON
    ld [TIME_DISPLAY_MAP_BASE_ADDRESS + 2], a

    xor a, a
    ld [SelectionScreen_FrameCounter], a
    ld [SelectionScreen_RedrawMode], a

    ld [LastJoypadState], a
    ld [PendingStateAddress + 0], a
    ld [PendingStateAddress + 1], a

    ld a, 1
    ld [SelectionScreen_DigitPos], a

    ClearTilesFixed BGFrameBuffer, 128
    ClearTilesFixed FGFrameBuffer, 32

    StoreWordToAddress CurrentActionState, AS_T0

.setupsprites
    ld hl, Sprites
    ld bc, $00A0
    call ClearMem

    ; Execute DMA transfer to OAM
    call run_dma

    call UpdateTangramSelectionNameInner

    CopyTilesFixedStride2 $8080 + 0, TextTilesBuffer + 0 + $00, 8
    CopyTilesFixedStride2 $8180 + 0, TextTilesBuffer + 0 + $80, 8


    ; Set "loading" frame tiles
FOR I, 16
    CallSetMem $9C22 + I * $20, TILE_INDEX_OPAQUE, 16
ENDR


    call UpdateTimeDisplayFromRecords

    ; Turn the LCD on
    SetLCDControl LCDCF_ON | LCDCF_BGON | LCDCF_BG8000 | LCDCF_OBJON | LCDCF_OBJ8 | LCDCF_WINOFF
    SetBackgroundPalette HUD_PALETTE
    
    ld a, %11111111
    ldh [rOBP0], a

    SpinWaitForVBlank

.loop

    ; Backup tile $FF
    CopyTileToHRAMUnrolled TileBackupB, $9800 - $10

    ; Write border graphics to tile $FF
    ;FillTileUnrolled $9800 - $10, %11111111
    WriteBorderGraphicsToTileFF

    SpinWaitForLine 7
    SpinWaitForMode0

    SetLCDControl LCDCF_ON | LCDCF_BGON | LCDCF_BG8800 | LCDCF_OBJOFF | LCDCF_OBJ8 | LCDCF_WINOFF
    SetBackgroundPalette TANGRAM_BG_PALETTE


    ld a, [SelectionScreen_RedrawMode]
    cp a, 1
    jp nz, .noredraw

    SpinWaitForLine 16

    ld bc, TextTilesBuffer + 0 + $00 ; 3
    ld de, $8080                     ; 3

.copyloop1
    SpinWaitForMode0

    ld h, b ; 1
    ld l, c ; 1

    ; Copy
REPT 4
    ld a, [hli]     ; 2
    ld [de], a      ; 2
    inc de          ; 2
    inc de          ; 2
    inc hl          ; 2
ENDR

    ld b, h
    ld c, l

    dec hl      ; 2
    xor a, a    ; 1
   ; Clear
REPT 4
    dec hl      ; 2
    ld [hld], a ; 2
ENDR

    bit 0, d
    jr z, .copyloop1


    ld bc, TextTilesBuffer + 0 + $80 ; 3
    ld de, $8180                     ; 3

.copyloop2
    SpinWaitForMode0

    ld h, b ; 1
    ld l, c ; 1

    ; Copy
REPT 4
    ld a, [hli]     ; 2
    ld [de], a      ; 2
    inc de          ; 2
    inc de          ; 2
    inc hl          ; 2
ENDR

    ld b, h
    ld c, l

    dec hl      ; 2
    xor a, a    ; 1
   ; Clear
REPT 4
    dec hl      ; 2
    ld [hld], a ; 2
ENDR

    bit 0, d
    jr nz, .copyloop2

    ld a, 30
    ld [SelectionScreen_PreviewTimer], a

    ld a, 2
    ld [SelectionScreen_RedrawMode], a

.noredraw

    call BorderTileJuggleBody

    SpinWaitForLine 135
    SpinWaitForMode0


    SetLCDControl LCDCF_ON | LCDCF_BGON | LCDCF_BG8000 | LCDCF_OBJON | LCDCF_OBJ8 | LCDCF_WINOFF
    SetBackgroundPalette HUD_PALETTE


    SpinWaitForVBlank

    ld a, [SelectionScreen_RedrawMode]
    cp a, 2
    jr nz, .notimedown

    ld a, [SelectionScreen_PreviewTimer]
    dec a
    ld [SelectionScreen_PreviewTimer], a
    jr nz, .notimedown

    ; Draw silhouette

    SetLCDControl LCDCF_ON | LCDCF_BGON | LCDCF_BG8000 | LCDCF_OBJOFF | LCDCF_OBJ8 | LCDCF_WINOFF | LCDCF_BG9C00
    SetBackgroundPalette HUD_PALETTE

    call LoadPieceCoordsForTangram
    ld bc, $0010
    call DrawSilhouette

    SpinWaitForVBlank

    ; Clear tile $FF, because the first thing that StateC does is create a backup of it (again)
    ;FillTileUnrolled $9800 - $10, 0

    SetLCDCModifiedMaskAndBits ($FF & ~LCDCF_OBJON), LCDCF_BG9C00 | LCDCF_OBJOFF
    call StateC_Patched

    SetLCDControl LCDCF_ON | LCDCF_BGON | LCDCF_BG8800 | LCDCF_OBJOFF | LCDCF_OBJ8 | LCDCF_WINOFF
    SetBackgroundPalette TANGRAM_BG_PALETTE

    xor a, a
    ld [SelectionScreen_RedrawMode], a

    ; Clear the unused bitplane
    xor a, a
FOR I, 8
    ldh [TileBackupA + (I * 2)], a
ENDR

.notimedown

    ; Restore tile $00
    CopyTileFromHRAMUnrolled $8800, TileBackupA

    ld a, 16
    ld [$FE00 + 0], a   ; Y

    ld a, 16 + 8
    ld [$FE00 + 1], a   ; X

    ld a, [SelectionScreen_DigitPos]
    bit 0, a
    jr z, :+
    ld a, 16 + 8 + 5
    ld [$FE00 + 1], a   ; X
:

    ld a, TILE_INDEX_EMPTY
    ld [$FE00 + 2], a   ; Tile
    ld a, [SelectionScreen_FrameCounter]
    bit 5, a
    jr nz, :+
    ld a, TILE_INDEX_BLANK
    ld [$FE00 + 2], a   ; Tile
:

    ld a, [SelectionScreen_FrameCounter]
    inc a
    ld [SelectionScreen_FrameCounter], a

    ; Poll joypad and handle inputs
    call HandleInputEvents

    CheckPendingStateAddress

    SpinWaitIfNotInVBlank

    SetLCDControl LCDCF_ON | LCDCF_BGON | LCDCF_BG8000 | LCDCF_OBJON | LCDCF_OBJ8 | LCDCF_WINOFF
    SetBackgroundPalette HUD_PALETTE

    jp .loop


SECTION "UpdateTangramSelectionName", ROM0
UpdateTangramSelectionName:
    SpinWaitForLine 7
    SpinWaitForMode0

    SetLCDControl LCDCF_ON | LCDCF_BGON | LCDCF_BG8800 | LCDCF_OBJOFF | LCDCF_OBJ8 | LCDCF_WINOFF
    SetBackgroundPalette TANGRAM_BG_PALETTE

    call UpdateTangramSelectionNameInner

    call BorderTileJuggleBody

    SpinWaitForLine 135
    SpinWaitForMode0

    SetLCDControl LCDCF_ON | LCDCF_BGON | LCDCF_BG8000 | LCDCF_OBJON | LCDCF_OBJ8 | LCDCF_WINOFF
    SetBackgroundPalette HUD_PALETTE

    SpinWaitForVBlank

    ; Restore tile $00
    CopyTileFromHRAMUnrolled $8800, TileBackupA

    call UpdateTimeDisplayFromRecords

    ret

SECTION "UpdateTangramSelectionNameInner", ROM0
UpdateTangramSelectionNameInner:
    ld a, [SelectionScreen_TangramIndex]
    ld hl, TangramAddressTable
    call IndirectLoadWordFromTable
    StoreWordToAddress TangramToLoad, de
    ld a, [SelectionScreen_IndexDigits + 0]
    ld b, a
    ld a, [SelectionScreen_IndexDigits + 1]
    ld c, a
    call LoadTangramName
    ld a, 1
    ld [SelectionScreen_RedrawMode], a
    ret


SECTION "UpdateTimeDisplayFromRecords", ROM0
UpdateTimeDisplayFromRecords:
    ; Get WRAM address of record time for this tangram
    ld hl, RecordTimes
    call GetRecordTimeAddress
    ; Read in the record time
    ld a, [hli]
    ld [Elapsed.decaminutes], a
    ld a, [hli]
    ld [Elapsed.minutes], a
    ld a, [hli]
    ld [Elapsed.decaseconds], a
    ld a, [hli]
    ld [Elapsed.seconds], a

    ld a, [Elapsed.decaminutes]
    ld hl, TIME_DISPLAY_MAP_BASE_ADDRESS + 0
    call WriteDecimalNumeral
    ld a, [Elapsed.minutes]
    ld hl, TIME_DISPLAY_MAP_BASE_ADDRESS + 1
    call WriteDecimalNumeral
    ld a, [Elapsed.decaseconds]
    ld hl, TIME_DISPLAY_MAP_BASE_ADDRESS + 3
    call WriteDecimalNumeral
    ld a, [Elapsed.seconds]
    ld hl, TIME_DISPLAY_MAP_BASE_ADDRESS + 4
    call WriteDecimalNumeral
    ret



SECTION "TitleScreen", ROM0
TitleScreen:

    xor a, a
    ld [TimerFreeze], a

    ; Do not turn the LCD off outside of VBlank
    SpinWaitIfNotInVBlank

    ; Turn the LCD off
    xor	a
    ldh	[rLCDC],a


    call InitializePieceStates

    ; Clear tiles in VRAM
    ld hl, $8000
    ld bc, $1800
    call ClearMem

    ClearTilesFixed TextTilesBuffer, $20

FOR I, 3
    ld hl, $9A00 + I * $20
    ld bc, $20
    ld a, $FF
    call SetMem
ENDR


    ; Set up scrollertext
    DrawTextFixed TextTilesBuffer + 1, TextStrings_9
    CopyTilesFixedStride2 $8D00 + 1, TextTilesBuffer + 1 + $00, 16
    CallIotaMem $9A20, $D0, 16

    ;DrawTextFixed TextTilesBuffer + 1, TextStrings_10
    ;CopyTilesFixedStride2 $8E00 + 1, TextTilesBuffer + 1 + $00, 9
    ;CallIotaMem $9A33, $E0, 16

    ;DrawTextFixed TextTilesBuffer + 1, TextStrings_11
    ;CopyTilesFixedStride2 $8F00 + 1, TextTilesBuffer + 1 + $00, 8
    ;CallIotaMem $9A20, $D0, 16




    xor a, a
    ldh [TitleScreen_MegaScrollPos], a
    ldh [TitleScreen_MegaScrollSubPos], a
    ldh [TitleScreen_FrameCounter], a
    ldh [TitleScreen_SineOffset], a


    ld de, LogoMapCompressed
    ld hl, LogoMap

    ; Decompress RLE
:
    ld a, [de]
    cp a, $FF
    jr z, :++
    inc de
    ld b, a
    rlca
    rlca
    and a, 3
    inc a
    ld c, a
    ld a, b
    and a, $3F
    add a, MEGASCROLLER_TILE_BASE
:
    ld [hli], a
    dec c
    jr nz, :-
    jr :--
:

    ld a, $FF
    ld [hl], a

    StoreWordToAddress TitleScreen_MegaScrollReadAddr, LogoMap


    ld hl, MegaScrollerMapBuffer
    ld bc, $100
:
    ld a, MEGASCROLLER_TILE_BASE       ; 2
    ld [hli], a     ; 2
    dec bc          ; 2
    ld a, b         ; 1
    or a, c         ; 1
    jr nz, :-


    CopyTilesFixed $8A00, LogoTiles, NUM_LOGO_TILES

    ; Set up graphics for the checkerboard zoom raster-effect
    CopyTilesFixed $8000, CheckerboardStripTiles, 80

    ld de, $8500
    ld hl, $8000
    ld bc, 80 * 16
    call CopyMem

    ; Mirror the strips
    ld hl, $8500
    ld de, 80 * 16 * 8
:
    ld a, [hl]
    ld c, a
    ld b, 0

:
    ld a, c
    and a, 1
    rlc b
    or a, b
    ld b, a
    rrc c
    dec de
    ld a, 7
    and a, e
    jr nz, :-

    ld a, b
    cpl
    ld [hli], a
    ;dec de
    ld a, d
    or a, e
    jr nz, :--


FOR I, 8
    CallIotaMem $9800 + I * $20,  I * 10, 10
    ld hl, $980A + I * $20
    ld a, $50 + I * 10 + 9
    ld c, 10
    call ReverseIotaMem
ENDR


    ; Set up sprites for the logo

.setupsprites
    ld hl, Sprites
    ld bc, $00A0
    call ClearMem

    ld hl, Sprites
    ld b, 10
:
    ld a, 16
    ld [hli], a ; Y
    ld a, b
    swap a
    rrca
    add a, 32 + 8
    ld [hli], a ; X
    ld a, $00
    ld [hli], a ; Tile
    xor a, a
    ld [hli], a ; Flags
    dec b
    jr nz, :-
    ld b, 6
:
    xor a, a
    ld [hli], a ; Y
    ld [hli], a ; X
    xor a, a
    ld [hli], a ; Tile
    ld [hli], a ; Flags
    dec b
    jr nz, :-
    ld b, 10
:
    ld a, 32
    ld [hli], a ; Y
    ld a, b
    swap a
    rrca
    add a, 32 + 8
    ld [hli], a ; X
    ld a, $00
    ld [hli], a ; Tile
    xor a, a
    ld [hli], a ; Flags
    dec b
    jr nz, :-
    ld b, 6
:
    xor a, a
    ld [hli], a ; Y
    ld [hli], a ; X
    ld [hli], a ; Tile
    ld [hli], a ; Flags
    dec b
    jr nz, :-


    ; Execute DMA transfer to OAM
    call run_dma

    ; Objects palette
    ld a, %10010000
    ;ld a, %11100100
    ld [rOBP0], a



    ; Turn the LCD on
    SetLCDControl LCDCF_ON | LCDCF_BGON | LCDCF_BG8000 | LCDCF_BG9800 | LCDCF_OBJON | LCDCF_OBJ16 | LCDCF_WINOFF

    ; Enable sound globally
    ld a, $80
    ld [rAUDENA], a
    ; Enable all channels in stereo
    ld a, $FF
    ld [rAUDTERM], a
    ; Set volume
    ld a, $77
    ld [rAUDVOL], a

    ld hl, tangram_music
    call hUGE_init

    ; set up the lcdc int
    ld a,STATF_MODE00 ; | STATF_LYC
    ldh [rSTAT],a

    xor	a, a
    ldh	[rIF],a

    ; Enable  the desired interrupts
    ld a, IEF_VBLANK | IEF_STAT
    ldh [rIE], a


    ; Set things up in preparation for the very first H-Blank
    ld hl, $FE00
    xor a, a
    ldh [TitleScreen_RasterScrollY], a
    ldh [TitleScreen_RasterPalette], a

    ei

TitleScreenLoop:

    halt

    jr TitleScreenLoop


SECTION "HBlankHandler", ROM0
HBlankHandler:
    ; Write sprite Y position
    ld [hli], a
    inc l
    ld a, b
    ; Write sprite tile index
    ld [hl], a

    ; Do the stuff for the background
    ;ldh a, [TitleScreen_RasterScrollY]
    ;dec a
    ;ldh [rSCY], a
    ;ldh a, [TitleScreen_RasterPalette]
    ;ldh [rBGP], a

    ldh a, [rLY]
    ld d, a
    ld h, HIGH(CheckerboardSineTable)
    ld l, a
    ldh a, [TitleScreen_SineOffset]
    add a, l
    ld l, a
    ld a, d
    cpl
    ld b, a
    ld a, [hl] ; Depth
    ld c, a
    add a, b
    ldh [rSCY], a
    ;ldh [TitleScreen_RasterScrollY], a
    ld a, d
    and a, 1
    ld b, a

    ld hl, CheckerboardPalettetable
assert LOW(CheckerboardPalettetable) == 0
    ld a, e     ; 1
    and a, 128  ; 2
    add a, c    ; 1
    rlca        ; 1
    rlca        ; 1
    add a, b    ; 1
    add a, l    ; 1
    ld l, a     ; 1


    ld a, [hl] ; Palette
    ldh [rBGP], a
    ;ldh [TitleScreen_RasterPalette], a
    ld hl, CheckerboardUStepTable
    ld a, c
    add a, l
    ld l, a
    ld a, [hl]
    add a, e
    ld e, a


    ld a, d
    cp a, 128
    jr nz, :++
    ld a, IEF_VBLANK
    ldh [rIE], a
REPT 8
;    nop
ENDR
    ld a, INGAME_MENU_BG_PALETTE
    ldh [rBGP], a
    ld a, 2
    ldh [rSCY], a
    ldh a, [TitleScreen_FrameCounter]
    ldh [rSCX], a
    SetLCDControl LCDCF_ON | LCDCF_BGON | LCDCF_BG8000 | LCDCF_BG9800 | LCDCF_OBJOFF | LCDCF_OBJ16 | LCDCF_WINOFF
    ;push af
    ;push hl
    ;push bc
    ;push de
    call _hUGE_dosound
    ;pop de
    ;pop bc
    ;pop hl
    ;pop af

    call readJoypad
    bit GB_JOY_BIT_START, a
    jr z, :+
    ; Fake my own RETI
    xor a, a
    ldh [rIF], a
    ei
    pop hl
    jp TangramSelectionScreen
:

    reti
:



    ; Do the stuff for the vertical megascroller

    ldh a, [TitleScreen_MegaScrollSubPos]
    cpl
    inc a
    ld c, a
    dec a
    cpl
    add a, d
    ld d, a

assert LOW(MegaScrollerMapBuffer) == 0
    ld h, HIGH(MegaScrollerMapBuffer)
    ldh a, [TitleScreen_MegaScrollPos]
    add a, d
    ld l, a
    ld a, [hl]
    ld b, a

    ; Calculate sprite address in OAM
    ld a, d
    and a, 31
    rlca
    rlca
    ld l, a
    ; Calculate sprite Y position
    ld a, d
    and a, $F0

    add a, 16 + 16
    add a, c

    ld h, $FE
    reti


SECTION "VBlankHandler", ROM0
VBlankHandler:

    ldh a, [TitleScreen_FrameCounter]
    cpl
    inc a
    ldh [TitleScreen_SineOffset], a

    ldh a, [TitleScreen_FrameCounter]
    ld b, a
    inc a
    ldh [TitleScreen_FrameCounter], a
    xor a, b
    bit 4, a

    jr z, :+
    ldh a, [TitleScreen_MegaScrollPos]
    swap a
    inc a
    and a, $0F
    swap a
    ldh [TitleScreen_MegaScrollPos], a
:

    ld a, b
    inc a
    and a, $0F
    ldh [TitleScreen_MegaScrollSubPos], a

assert LOW(MegaScrollerMapBuffer) == 0
    LoadWordFromAddress hl, TitleScreen_MegaScrollReadAddr
    ld d, HIGH(MegaScrollerMapBuffer)
    ld a, b
    add a, 128 + 16
    ld e, a
    ld a, [hli]
    ld [de], a

    ; Check for end of megascroll tile map data
    ld a, [hl]
    cp a, $FF
    jr nz, :+
    ld hl, LogoMap
:

    StoreWordToAddress TitleScreen_MegaScrollReadAddr, hl

    ldh a, [TitleScreen_MegaScrollSubPos]
    cpl
    inc a
    ld c, a


    ld l, e

    ; Set Y for first row of sprites
    ld a, 16
    add a, c
FOR I, 10
    ld [$FE40 + I * 4 + 0], a
ENDR

    ; Set tile index for first row of sprites
    ld a, e
    inc a
    sub a, 10 * 16
    and a, $F0
    ld e, a
FOR I, 10
    ld a, [de]
    inc e
    ld [$FE40 + I * 4 + 2], a
ENDR

    ld e, l

    ; Set Y for second row of sprites
    ld a, 32
    add a, c
FOR I, 10
    ld [$FE00 + I * 4 + 0], a
ENDR

    ; Set tile index for second row of sprites
    ld a, e
    inc a
    sub a, 9 * 16
    and a, $F0
    ld e, a
FOR I, 10
    ld a, [de]
    inc e
    ld [$FE00 + I * 4 + 2], a
ENDR


    ld a, $FF
    ldh [rBGP], a

    SetLCDControl LCDCF_ON | LCDCF_BGON | LCDCF_BG8000 | LCDCF_BG9800 | LCDCF_OBJON | LCDCF_OBJ16 | LCDCF_WINOFF

    ld a, IEF_VBLANK | IEF_STAT
    ldh [rIE], a

    xor a, a
    ldh [rIF], a
    ldh [rSCX], a

    xor a, a
    ld e, a
    ld b, a
    ld hl, $FE9C

    reti

SECTION "LoadTangramName", ROMX
LoadTangramName:
    ;push bc
    ;ClearTilesFixed TextTilesBuffer, 16
    ;pop bc

    ; Tiles: Tangram name in HUD
    LoadWordFromAddress hl, TangramToLoad
    ld de, 21
    add hl, de
    ld de, TangramTitleStringTemp
    ; Numeric digits
    ld a, $30
    add a, b
    ld [de], a
    inc de
    ld a, $30
    add a, c
    ld [de], a
    inc de

    ld a, $3A
    ld [de], a
    inc de
    ; Colon
    ld a, $20
    ld [de], a
    inc de
:
    ld a, [hli]
    ld [de], a
    inc de
    or a, a
    jp nz, :-

    ld hl, TangramTitleStringTemp
    ld de, TextTilesBuffer + 0
    call DrawText

    ret


SECTION "MakeFramebuffer", ROMX
MakeFramebuffer: ; Tiles increment down columns
    ld bc, $1000
    ld hl, $9802 + 32
    ld de, 16
    ld c, 0
.startrow
    ld a, c
.next
    xor a, 128
    ld [hli], a
    xor a, 128
    add a, 16
    jr nc, .next
    inc c
    dec b
    add hl, de
    jr nz, .startrow
    ret


SECTION "LoadTangramAndStartGame", ROM0
LoadTangramAndStartGame:

    xor a, a
    ld [Elapsed.seconds], a
    ld [Elapsed.decaseconds], a
    ld [Elapsed.minutes], a
    ld [Elapsed.decaminutes], a

    SetLCDControl LCDCF_ON | LCDCF_BGON | LCDCF_BG8000 | LCDCF_OBJOFF | LCDCF_OBJ8 | LCDCF_WINOFF | LCDCF_BG9C00
    SetBackgroundPalette HUD_PALETTE

    ClearTilesFixed BGFrameBuffer, 128
    ClearTilesFixed FGFrameBuffer, 32

    SpinWaitIfNotInVBlank

        ; Resume
    CallIotaMem $9CD5, $08, 4
        ; Check if solved
    CallIotaMem $9CF5, $0C, 4
    CallIotaMem $9CF9, $18, 5
        ; Select Tangram
    CallIotaMem $9D15, $1D, 3
    CallIotaMem $9D18, $28, 6
        ; Solve this piece
    CallIotaMem $9D35, $2E, 2
    CallIotaMem $9D37, $38, 7
        ; Solve all pieces
    CallIotaMem $9D55, $3F, 1
    CallIotaMem $9D56, $48, 8
        ; Quit to title screen
    CallIotaMem $9D75, $58, 8
    CallIotaMem $9D7D, $68, 2


        ; Solved!
    CallIotaMem $9F29, $48, 4
        ; Not solved...
    CallIotaMem $9F37, $58, 7



.setupsprites
    ld hl, Sprites
    ld b, 8
    ld d, 126
.row
    ld c, 4
    ld a, d
    sub a, 8
    ld d, a
.col
    xor a, a
    ld [hli], a ; Y
    ld [hli], a ; X
    ld a, d
    ld [hli], a ; Tile
    xor a, a
    ld [hli], a ; Flags
    dec d
    dec d
    dec c
    jr nz, .col
    dec b
    jr nz, .row


    ; Draw silhouette
    call LoadPieceCoordsForTangram
    ld bc, $0010
    call DrawSilhouette

    SpinWaitForVBlank

    ld a, [Elapsed.decaminutes]
    ld hl, TIME_DISPLAY_MAP_BASE_ADDRESS + 0
    call WriteDecimalNumeral
    ld a, [Elapsed.minutes]
    ld hl, TIME_DISPLAY_MAP_BASE_ADDRESS + 1
    call WriteDecimalNumeral
    ld a, [Elapsed.decaseconds]
    ld hl, TIME_DISPLAY_MAP_BASE_ADDRESS + 3
    call WriteDecimalNumeral
    ld a, [Elapsed.seconds]
    ld hl, TIME_DISPLAY_MAP_BASE_ADDRESS + 4
    call WriteDecimalNumeral

    SetLCDCModifiedMaskAndBits ($FF & ~LCDCF_OBJON), LCDCF_BG9C00 | LCDCF_OBJOFF
    call StateC_Patched

    jp StartGame


SECTION "InitializePieceStates", ROM0
InitializePieceStates:
    ld bc, 7 * 16
    ld hl, PieceInitialStates
    ld de, PieceCoords
    call CopyMem
    ret

SECTION "LoadPieceCoordsForTangram", ROM0
LoadPieceCoordsForTangram:
    LoadWordFromAddress hl, TangramToLoad
    ld de, PieceCoords
    ld c, 7
:
    inc de ; Skip Piece_ActiveMask
    ld a, [hli]
    ld [de], a ; Piece_X
    inc de
    ld a, [hli]
    ld [de], a ; Piece_Y
    inc de
    ld a, [hli]
    ld [de], a ; Piece_Angle
    ld a, e
    add a, 16 - 3
    ld e, a
    dec c
    jr nz, :-
    ret

SECTION "DrawSilhouette", ROM0
DrawSilhouette:
    ld a, b
    ldh [Temps + 6], a
    ld a, c
    ldh [Temps + 7], a

    ld a, LOW(PieceCoords)
    ldh [Temps + 2], a
    ld a, 7
    ldh [Temps + 3], a

.pieceloop
    ld h, HIGH(PieceCoords)
    ldh a, [Temps + 2]
    ld l, a
    LoadPiece

    ; Get FG DrawPolygon routine address
    LoadWordFromAddress hl, HeldPieceCoords + Piece_FGDraw

    ; Offset the piece position to account for offset caused by mask expansion (where applicable)
    ldh a, [Temps + 7]
    ld b, a
    ldh a, [HeldPieceCoords + Piece_Y]
    add a, b
    ldh [HeldPieceCoords + Piece_Y], a

    ldh a, [Temps + 6]
    ld b, a
    ldh a, [HeldPieceCoords + Piece_X]
    add a, b
    ldh [HeldPieceCoords + Piece_X], a

    ld de, @ + 5
    push de
    jp hl ; FGDrawPolygon

    ; Fill polygon FG

    ; BG framebuffer address
assert LOW(PreBGFrameBuffer) == 0
    ld a, [Temps + 7]
    ldh a, [HeldPieceCoords + Piece_Y]
    and a, $F8
    sub a, 32 - 4
    swap a
    ld e, a
    and a, $0F
    add a, HIGH(PreBGFrameBuffer)
    ld d, a
    ld a, e
    and a, $F0
    ld e, a
    ld a, [Temps + 6]
    ldh a, [HeldPieceCoords + Piece_X]
    rrca                ; 1
    rrca                ; 1
    rrca                ; 1
    and a, $0F          ; 2
    sub a, 4            ; 2
    add a, e            ; 1
    ld e, a
    ld a, d
    ldh [Temps + 4], a
    ld a, e
    ldh [Temps + 5], a


FOR I, 8

assert LOW(PolygonFillBuffer) == 0

    ; First pass - Extend pixels in FG downwards and write result to PolygonFillBuffer

    ld hl, FGFrameBuffer + I
    ld de, PolygonFillBuffer
    ld bc, +8
    ld a, $00
:
    or a, [hl]
    ld [de], a
    add hl, bc      ; 2
    inc e           ; 1
    bit 6, e        ; 2
    jr z, :-

    ; Second pass - Extend pixels in FG upwards and bitwise AND result with PolygonFillBuffer and clear FG

    ld bc, -8
    add hl, bc      ; 2
    dec e           ; 1
    ld a, $00
:
    or a, [hl]
    ld b, a
    ld a, [de]
    and a, b        ; 1
    ld [de], a      ; 2
    ld b, $FF       ; 2
    ld [hl], 0      ; 3
    add hl, bc      ; 2
    dec e           ; 1
    bit 7, e        ; 2
    jr z, :-

    ; Third pass - Bitwise OR the contents of PolygonFillBuffer into BG framebuffer at offset location

    ld hl, PolygonFillBuffer    ; 3
    ldh a, [Temps + 4]  ; 3
    ld d, a             ; 1
    ldh a, [Temps + 5]  ; 3
    ld e, a             ; 1
    ld c, 64            ; 2
:


IF 0
    ld a, d ; Hacky bounds test
    and a, $F0
    cp a, $C0
    jp nz, .boundstest\@
    ld a, [de]      ; 2
    or a, [hl]      ; 2
    ld [de], a      ; 2
.boundstest\@:
ELSE
    ld a, [de]      ; 2
    or a, [hl]      ; 2
    ld [de], a      ; 2
ENDC

    inc hl          ; 2
    ld a, e         ; 1
    add a, 16       ; 2
    ld e, a         ; 1
    ld a, d         ; 1
    adc 0           ; 2
    ld d, a         ; 1
    dec c
    jr nz, :-

    ldh a, [Temps + 5]  ; 3
    inc a               ; 1
    ldh [Temps + 5], a  ; 3

ENDR

    ldh a, [Temps + 2]
    add a, 16
    ldh [Temps + 2], a
    ldh a, [Temps + 3]
    dec a
    ldh [Temps + 3], a
    jp nz, .pieceloop

    ret


SECTION "ExpandBGFrameBuffer", ROM0
ExpandBGFrameBuffer:
    ; Note that the expanded image is offset 8 pixels right and 4 pixels up
    call ExpandBGFrameBufferH
    call ExpandBGFrameBufferV
    ret

SECTION "ExpandBGFrameBufferV", ROM0
ExpandBGFrameBufferV:
FOR I, 2
    ld c, 15
.col\@
    ld hl, BGFrameBuffer
    ld de, 16 << (I + 0)
    add hl, de
    ld de, BGFrameBuffer
    ld a, l
    add a, c
    ld l, a
    ld a, e
    add a, c
    ld e, a
.row\@
    ld a, [de]
    or a, [hl]
    ld [de], a
    Add8To16WithCarry hl, 16
    Add8To16WithCarry de, 16
    ld a, h
    cp a, HIGH(BGFrameBuffer) + 8
    jr nz, .row\@     ; 3 / 2
    dec c
    ld a, c
    cp a, $FF
    jr nz, .col\@
ENDR
    ret

SECTION "ExpandBGFrameBufferH", ROM0
ExpandBGFrameBufferH:
    ld hl, BGFrameBuffer
    ; c = current bitbuffer
    ; d = incoming bitbuffer, shifted in from right
    ; b = output bitbuffer, written right-to-left
.row
    ld e, 16
    ld b, 0
.col
    ld c, d         ; 1
    ld a, [hl]      ; 2 ; Read
    ld d, a         ; 1

FOR I, 4
    ld a, %00000111 ; 2
    ;ld a, c         ; 1
    ;;or a, a         ; 1
    ;and a, %00011111 ; 2
    and a, c        ; 1
    jr z, :+        ; 3 / 2
    set 7 - (I + 4), b    ; 2
:
    sla c           ; 2
    bit 7 - (I + 0), d    ; 2
    jr z, :+        ; 3 / 2
    set 0, c        ; 2
:
ENDR

    ld a, b         ; 1
    ld [hli], a     ; 2 ; Write
    ld b, 0         ; 2

FOR I, 4
    ld a, %00000111 ; 2
    ;ld a, c         ; 1
    ;;or a, a         ; 1
    ;and a, %00011111 ; 2
    and a, c        ; 1
    jr z, :+        ; 3 / 2
    set 7 - (I + 0), b    ; 2
:
    sla c           ; 2
    bit 7 - (I + 4), d    ; 2
    jr z, :+        ; 3 / 2
    set 0, c        ; 2
:
ENDR

    dec e           ; 1
    jp nz, .col     ; 3 / 2
    ld a, h
    cp a, HIGH(BGFrameBuffer) + 8
    jp nz, .row
    ret




SECTION "StartGame", ROM0
StartGame:
    call InitializePieceStates

    xor a, a
    ld [ActivePieceOffset], a
    ld [AutoSolveWasUsed], a
    ld [LastJoypadState], a

    ld [PendingStateAddress + 0], a
    ld [PendingStateAddress + 1], a

    ld [TimeCounter], a

    StoreWordToAddress CurrentActionState, AS_00

    call LoadActivePiece
    SpinWaitForVBlank

    call DrawPieceWireframesToBG

    ; Initialize timer

    ld a, %00000100 ; CPU Clock / 1024 = 4096 Hz
    ldh [rTAC], a
    xor a, a
    ld [$FE00], a
    ldh [rTIMA], a
    ldh [rTMA], a
    jp StateA

SECTION "StateA", ROM0
StateA:

    StoreWordToAddress CurrentActionState, AS_00

    ld a, 1
    ld [TimerFreeze], a

StateALoop: ; Interactive gameplay state

    ; Backup tile $FF
    CopyTileToHRAMUnrolled TileBackupB, $9800 - $10

    ; Write border graphics to tile $FF
    ;FillTileUnrolled $9800 - $10, %11111111
    WriteBorderGraphicsToTileFF

    SpinWaitForLine 1

FOR I, 7

    ld bc, FGFrameBuffer + (I + 8 + 8) * 8 ; 3
    ld de, $8000 + (I + 8 + 8) * 2         ; 3

    SpinWaitForMode0

    ld h, b ; 1
    ld l, c ; 1

    ; Copy 8 bytes to VRAM
    MemCopyStride256Unrolled 8

    xor a, a    ; 1
    ; Clear 8 bytes
REPT 8
    ld [hld], a ; 2
ENDR

ENDR

    ;SpinWaitForLine 7

    ;SpinWaitForMode0

    ; Switch to LCDCF_BG8800, enable sprites
    SetLCDControl LCDCF_ON | LCDCF_BGON | LCDCF_BG8800 | LCDCF_OBJON | LCDCF_OBJ16 | LCDCF_WINOFF
    SetBackgroundPalette TANGRAM_BG_PALETTE


FOR I, 64 - (8 + 8 + 7)

    ld bc, FGFrameBuffer + (I + 8 + 8 + 7) * 8 ; 3
    ld de, $8000 + (I + 8 + 8 + 7) * 2         ; 3

    SpinWaitForMode0

    ld h, b ; 1
    ld l, c ; 1

    ; Copy 8 bytes to VRAM
    MemCopyStride256Unrolled 8

    xor a, a    ; 1
    ; Clear 8 bytes
REPT 8
    ld [hld], a ; 2
ENDR

ENDR


    ; Get FG DrawPolygon routine address
    LoadWordFromAddress hl, HeldPieceCoords + Piece_FGDraw

    ;SetBackgroundPalette 255-TANGRAM_BG_PALETTE
    ld de, @ + 5
    push de
    jp hl ; DrawPolygon

    ; Update sprites
I = 0
FOR X, 8
FOR Y, 4
    ldh a, [HeldPieceCoords + Piece_Y]
    and a, 255-7
    add a, -32 + 64 - Y * 16 + 12
    ld [Sprites + I * 4 + 0], a ; Y
    ldh a, [HeldPieceCoords + Piece_X]
    and a, 255-7
    add a, -32 + 64 - X * 8 + 20
    ld [Sprites + I * 4 + 1], a ; X
I = I + 1
ENDR
ENDR

    ; Reticle
    ldh a, [HeldPieceCoords + Piece_Y]
    add a, 16 - 3
    ld [Sprites + 39 * 4 + 0], a ; Y
    ldh a, [HeldPieceCoords + Piece_X]
    add a, 16 + 8 - 3
    ld [Sprites + 39 * 4 + 1], a ; X
    ld a, TILE_INDEX_RETICLE
    ld [Sprites + 39 * 4 + 2], a ; Tile index


    ; Poll joypad and handle inputs
    call HandleInputEvents

    ;SetBackgroundPalette 255-TANGRAM_BG_PALETTE
    SpinWaitForLine 114
    ;SetBackgroundPalette TANGRAM_BG_PALETTE

FOR I, 8

    ld bc, FGFrameBuffer + (I) * 8 ; 3
    ld de, $8000 + (I) * 2         ; 3

    SpinWaitForMode0

    ld h, b ; 1
    ld l, c ; 1

    ; Copy 8 bytes to VRAM
    MemCopyStride256Unrolled 8

    xor a, a    ; 1
    ; Clear 8 bytes
REPT 8
    ld [hld], a ; 2
ENDR

ENDR

    call BorderTileJuggleBody

    SpinWaitForLine 135

FOR I, 8

    ld bc, FGFrameBuffer + (I + 8) * 8 ; 3
    ld de, $8000 + (I + 8) * 2         ; 3

    SpinWaitForMode0

    ld h, b ; 1
    ld l, c ; 1

    ; Copy 8 bytes to VRAM
    MemCopyStride256Unrolled 8

IF I == 0
    ; Switch to LCDCF_BG8000, turn sprites off, on scanline 135
    SetLCDControl LCDCF_ON | LCDCF_BGON | LCDCF_BG8000 | LCDCF_OBJOFF | LCDCF_OBJ16 | LCDCF_WINOFF
    SetBackgroundPalette HUD_PALETTE
ENDC

    xor a, a    ; 1
    ; Clear 8 bytes
REPT 8
    ld [hld], a ; 2
ENDR

ENDR


    SpinWaitForLine 144

    ;SetBackgroundPalette 255-HUD_PALETTE
    ;SpinWaitForLine 144 ; Start of vblank
    ;SetBackgroundPalette HUD_PALETTE
    ; Restore tile $00
    CopyTileFromHRAMUnrolled $8800, TileBackupA

    ; Execute DMA transfer to OAM
    call run_dma

    ; Poll joypad and handle inputs
    ;call HandleInputEvents

    ; Handle timer interrupt flag
    call HandleTimer

    CheckPendingStateAddress

    jp StateALoop



SECTION "LoadActivePiece", ROM0
LoadActivePiece:
    ld a, [ActivePieceOffset]
    swap a
    ld hl, PieceCoords
    add a, l
    ld l, a
    LoadPiece
    ret

SECTION "LoadActivePieceFromROM", ROM0
LoadActivePieceFromROM:
    ldh a, [HeldPieceCoords + Piece_ActiveMask]
    ld b, a
    ld de, PieceCoords

    LoadWordFromAddress hl, TangramToLoad
    ld de, PieceCoords
    ld c, 7
:
    rrc b
    bit 7, b
    jr z, :+
    swap e
    inc e
    swap e
    ld a, l
    add a, 3
    ld l, a
    ld a, h
    adc 0
    ld h, a
    dec c
    jr nz, :-
:
    inc de ; Skip Piece_ActiveMask
    ld a, [hli]
    sub a, 4 ; Move piece 4 pixels to the left due to reasons
    ld [de], a ; Piece_X
    inc de
    ld a, [hli]
    ld [de], a ; Piece_Y
    inc de
    ld a, [hli]
    ld [de], a ; Piece_Angle
    call LoadActivePiece
    ret

SECTION "SFX_Nudge", ROMX
SFX_Nudge:
    ; SFX: Nudge Piece Position
    ld a, %00100000    ; Noise length
    ldh [rNR41], a
    ld a, %01110001    ; Noise volume envelope
    ldh [rNR42], a
    ld a, %00100010    ; Noise polynomial counter
    ldh [rNR43], a
    ld a, %11000000    ; Trigger
    ldh [rNR44], a
    ret
SECTION "SFX_Nudge2", ROMX
SFX_Nudge2:
    ; SFX: Nudge Piece Angle
    ld a, %00100000    ; Noise length
    ldh [rNR41], a
    ld a, %01110001    ; Noise volume envelope
    ldh [rNR42], a
    ld a, %01000010    ; Noise polynomial counter
    ldh [rNR43], a
    ld a, %11000000    ; Trigger
    ldh [rNR44], a
    ret
SECTION "SFX_Switch", ROMX
SFX_Switch:
    ; SFX: Switch Piece
    ld a, %01110011     ; Frequency sweep
    ldh [rNR10], a
    ld a, %10000000     ; 50% duty, max. sound length
    ldh [rNR11], a
    ld a, %11110001     ; Volume envelope
    ldh [rNR12], a
    ld a, $AD     ; Lo 8 bits of 11-bit frequency
    ldh [rNR13], a
    ld a, %11000101     ; Hi 3 bits of 11-bit frequency and trigger
    ldh [rNR14], a
    ret
SECTION "SFX_Menu", ROMX
SFX_Menu:
    ; SFX: Change In-Game Menu Selection
    ld a, %00000000     ; Frequency sweep
    ldh [rNR10], a
    ld a, %10110000     ; 50% duty, max. sound length
    ldh [rNR11], a
    ld a, %10110000     ; Volume envelope
    ldh [rNR12], a
    ld a, $FD     ; Lo 8 bits of 11-bit frequency
    ldh [rNR13], a
    ld a, %11000101     ; Hi 3 bits of 11-bit frequency and trigger
    ldh [rNR14], a
    ret
SECTION "SFX_OpenMenu", ROMX
SFX_OpenMenu:
    ; SFX: Open In-Game Menu
    ld a, %01110110     ; Frequency sweep
    ldh [rNR10], a
    ld a, %10000000     ; 50% duty, max. sound length
    ldh [rNR11], a
    ld a, %11010001     ; Volume envelope
    ldh [rNR12], a
    ld a, $AD     ; Lo 8 bits of 11-bit frequency
    ldh [rNR13], a
    ld a, %11000110     ; Hi 3 bits of 11-bit frequency and trigger
    ldh [rNR14], a
    ret
SECTION "SFX_CloseMenu", ROMX
SFX_CloseMenu:
    ; SFX: Close In-Game Menu
    ld a, %01111110     ; Frequency sweep
    ldh [rNR10], a
    ld a, %10000000     ; 50% duty, max. sound length
    ldh [rNR11], a
    ld a, %11010001     ; Volume envelope
    ldh [rNR12], a
    ld a, $AD     ; Lo 8 bits of 11-bit frequency
    ldh [rNR13], a
    ld a, %11000110     ; Hi 3 bits of 11-bit frequency and trigger
    ldh [rNR14], a
    ret
SECTION "SFX_NotSolved", ROM0
SFX_NotSolved:
    ; SFX: Tangram Not Yet Solved
    ld a, %00000000     ; Frequency sweep
    ldh [rNR10], a
    ld a, %10000000     ; 50% duty, max. sound length
    ldh [rNR11], a
    ld a, %11110000     ; Volume envelope
    ldh [rNR12], a
    ld a, $00     ; Lo 8 bits of 11-bit frequency
    ldh [rNR13], a
    ld a, %11000100     ; Hi 3 bits of 11-bit frequency and trigger
    ldh [rNR14], a
    ret
SECTION "SFX_Solved", ROM0
SFX_Solved:
    ; SFX: Tangram Solved!
    ld a, %11110111     ; Frequency sweep
    ldh [rNR10], a
    ld a, %10000000     ; 50% duty, max. sound length
    ldh [rNR11], a
    ld a, %11110000     ; Volume envelope
    ldh [rNR12], a
    ld a, $AD     ; Lo 8 bits of 11-bit frequency
    ldh [rNR13], a
    ld a, %11000111     ; Hi 3 bits of 11-bit frequency and trigger
    ldh [rNR14], a
    ret


SECTION "SwitchToNextPiece", ROM0
SwitchToNextPiece:
    push hl
    push bc

    ; Play sound effect
    call SFX_Switch

    ; Store current piece

    ld a, [ActivePieceOffset]
    swap a
    ld hl, PieceCoords
    add a, l
    ld l, a

FOR I, Piece_SIZEOF
    ldh a, [HeldPieceCoords + I]
    ld [hli], a
ENDR

    ; Increment the active piece offset (with wraparound)

    ld hl, ActivePieceOffset     ; 3
    inc [hl]                     ; 3
    ld a, 7                      ; 2
    cp a, [hl]                   ; 2
    jp nz, :+
    xor a, a                     ; 1
    ld [hl], a                   ; 2
:

    ; Set LCDC for StateB
    SetLCDCModifiedMaskAndBits $FF, $00

    ; Load active mask from next piece

    ld a, [ActivePieceOffset]
    swap a
    ld hl, PieceCoords
    add a, l
    ld l, a
    ld a, [hl] ; Piece_ActiveMask
assert Piece_ActiveMask == 0

    ; Re-render the background wireframes
    call StateB

    ;SetLCDCModifiedMaskAndBits $FF, $00
    SetLCDCModifiedMaskAndBits ($FF & ~LCDCF_OBJON), LCDCF_OBJOFF
    call StateC

    ; Load next piece fully

    call LoadActivePiece

    pop bc
    pop hl
    ret


SECTION "DrawPieceWireframesToBG", ROM0
DrawPieceWireframesToBG:

    ; Store current piece

    ld a, [ActivePieceOffset]
    swap a
    ld hl, PieceCoords
    add a, l
    ld l, a

FOR I, Piece_SIZEOF
    ldh a, [HeldPieceCoords + I]
    ld [hli], a
ENDR

    ; Set LCDC for StateB
    SetLCDCModifiedMaskAndBits ($FF & ~LCDCF_OBJON), LCDCF_BG9C00 | LCDCF_OBJOFF | LCDCF_BG8000
    ld a, 1
    ldh [DoNotChangePaletteFlag], a

    ; Load active mask

    ld a, [ActivePieceOffset]
    swap a
    ld hl, PieceCoords
    add a, l
    ld l, a
    ld a, [hl] ; Piece_ActiveMask
assert Piece_ActiveMask == 0

    ; Re-render the background wireframes
    call StateB

    ;SetLCDCModifiedMaskAndBits $FF, $00
    SetLCDCModifiedMaskAndBits ($FF & ~LCDCF_OBJON), LCDCF_BG9C00 | LCDCF_OBJOFF | LCDCF_BG8000
    call StateC

    ; Re-load the current piece

    call LoadActivePiece

    xor a, a
    ldh [DoNotChangePaletteFlag], a

    ret


SECTION "CheckForPuzzleCompletion", ROM0
CheckForPuzzleCompletion:
    push hl
    push bc
    push de

    xor a, a
    ld [TimerFreeze], a

    ; Store current piece

    ld a, [ActivePieceOffset]
    swap a
    ld hl, PieceCoords
    add a, l
    ld l, a

FOR I, Piece_SIZEOF
    ldh a, [HeldPieceCoords + I]
    ld [hli], a
ENDR

    SetLCDControl LCDCF_ON | LCDCF_BGON | LCDCF_BG8000 | LCDCF_OBJOFF | LCDCF_OBJ8 | LCDCF_WINOFF | LCDCF_BG9C00
    SetBackgroundPalette HUD_PALETTE

    ; Render silhouette to BG framebuffer

    ClearTilesFixed FGFrameBuffer, 32
    ld bc, $FE11
    call DrawSilhouette
    call ExpandBGFrameBuffer

    ld a, 1
    ldh [DoNotChangePaletteFlag], a
    SetLCDCModifiedMaskAndBits ($FF & ~LCDCF_OBJON), LCDCF_BG9C00 | LCDCF_OBJOFF | LCDCF_BG8000
    SpinWaitForVBlank
    call StateC
    xor a, a
    ldh [DoNotChangePaletteFlag], a

    SetLCDControl LCDCF_ON | LCDCF_BGON | LCDCF_BG8000 | LCDCF_OBJOFF | LCDCF_OBJ8 | LCDCF_WINOFF | LCDCF_BG9C00
    SetBackgroundPalette HUD_PALETTE

    ; Restore active piece
    call LoadActivePiece

    ; Mark solved
    ld a, 1
    ldh [Temps + 0], a

    ; Check for coverage in 4 frames
FOR I, 4
    SpinWaitForLine 7

    ;ld bc, $8800 + I * $0400 + 8    ; 3
    ld b, HIGH($8800 + I * $0400 + 8)    ; 2
    ld e,  LOW($8800 + I * $0400 + 8)    ; 2
    ld d, 128       ; 2

.checkloop\@
    xor a, a        ; 1
    ld c, a         ; 1

    SpinWaitForMode0

    ld h, b         ; 1
    ld l, e         ; 1

    ; Reduce
FOR J, 4
    ld a, [hli]     ; 2 ; Load candidate mask
    cpl             ; 1
    ld b, a         ; 1
    ld a, [hld]     ; 2 ; Load target mask
    ;ld [hl], 0      ; 3 ; Clear old value
    and a, b        ; 1
    or a, c         ; 1
    ld c, a         ; 1
IF J < 3
    inc h           ; 1
ENDC
ENDR


    ld a, c
    or a, a
    jr z, :+
    ; Mark not-solved
    xor a, a
    ldh [Temps + 0], a
:

    ld h, $88 + I * $04
    inc l
    inc l

    ld b, h         ; 1
    ld e, l         ; 1

    dec d           ; 1
    jr nz, .checkloop\@
ENDR


    ; Return solved-or-not-solve flag in A
    ld a, [Temps + 0]
    pop de
    pop bc
    pop hl
    ret


SECTION "StateB", ROM0
StateB: ; Draw non-active polygons to the background framebuffer in 3 frames

    ;SpinWaitForVBlank

    ld b, a ; Piece_ActiveMask
    ld c, 7
    push bc
    ld a, LOW(PieceCoords)
    ldh [Temps + 2], a

StateBLoop:

    bit 0, b    ; 2
    jr nz, :+
    pop bc
    dec c
    ret z
    ld a, b
    sra a
    ld b, a
    push bc
    ldh a, [Temps + 2]  ; 3
    add a, 16           ; 2
    ldh [Temps + 2], a  ; 3
:

    ; Backup tile $FF
    CopyTileToHRAMUnrolled TileBackupB, $9800 - $10

    ; Write border graphics to tile $FF
    ;FillTileUnrolled $9800 - $10, %11111111
    WriteBorderGraphicsToTileFF

    SpinWaitForLine 7
    SpinWaitForMode0

    ; Switch to LCDCF_BG8800
    SetLCDControlModified LCDCF_ON | LCDCF_BGON | LCDCF_BG8800 | LCDCF_OBJON | LCDCF_OBJ16 | LCDCF_WINOFF
    ConditionallySetBackgroundPalette TANGRAM_BG_PALETTE

    ld h, HIGH(PieceCoords)
    ldh a, [Temps + 2]
    ld l, a
    LoadPiece   ; A


    ; Get BG DrawPolygon routine address
    LoadWordFromAddress hl, HeldPieceCoords + Piece_BGDraw

    ld de, @ + 5
    push de
    jp hl ; DrawPolygon A

    pop bc
    ld a, b
    sra a
    ld b, a
    dec c
    ldh a, [Temps + 2]  ; 3
    add a, 16           ; 2
    ldh [Temps + 2], a  ; 3

    bit 0, b    ; 2
    jr nz, :+
    ld a, b
    sra a
    ld b, a
    dec c
    ldh a, [Temps + 2]  ; 3
    add a, 16           ; 2
    ldh [Temps + 2], a  ; 3
:
    push bc

    ld h, HIGH(PieceCoords)
    ldh a, [Temps + 2]
    ld l, a
    LoadPiece   ; B

    ; Get BG DrawPolygon routine address
    LoadWordFromAddress hl, HeldPieceCoords + Piece_BGDraw

    ld de, @ + 5
    push de
    jp hl ; DrawPolygon B


    call BorderTileJuggleBody

    SpinWaitForLine 135
    SpinWaitForMode0

    ; Switch to LCDCF_BG8000, turn sprites off
    SetLCDControlModified LCDCF_ON | LCDCF_BGON | LCDCF_BG8000 | LCDCF_OBJOFF | LCDCF_OBJ16 | LCDCF_WINOFF
    ConditionallySetBackgroundPalette HUD_PALETTE

    SpinWaitForLine 144 ; Start of vblank
    ; Restore tile $00
    CopyTileFromHRAMUnrolled $8800, TileBackupA

    ; Handle timer interrupt flag
    call HandleTimer

    pop bc
    dec c
    ret z
    ld a, b
    sra a
    ld b, a
    push bc
    ldh a, [Temps + 2]  ; 3
    add a, 16           ; 2
    ldh [Temps + 2], a  ; 3
    jp StateBLoop


SECTION "StateC", ROM0
StateC: ; Copy tile data from background framebuffer to VRAM in 2 frames

StateC_Frame0:

    ; Backup tile $FF
    CopyTileToHRAMUnrolled TileBackupB, $9800 - $10

    ; Write border graphics to tile $FF
    ;FillTileUnrolled $9800 - $10, %11111111
    WriteBorderGraphicsToTileFF

    SpinWaitForLine 7
    SpinWaitForMode0

    ; Switch to LCDCF_BG8800
    SetLCDControlModified LCDCF_ON | LCDCF_BGON | LCDCF_BG8800 | LCDCF_OBJOFF | LCDCF_OBJ16 | LCDCF_WINOFF
    ConditionallySetBackgroundPalette TANGRAM_BG_PALETTE

    SpinWaitForLine 8


    ld bc, BGFrameBuffer ; 3
    AddPatchLocation StateC
    ld de, $8800         ; 3

REPT 7  ; 7 x 16 = 112
.label\@:
    SpinWaitForMode0

    ld h, b ; 1
    ld l, c ; 1

    ; Copy 8 bytes to VRAM
    MemCopyStride256Unrolled 8

    xor a, a    ; 1
    ; Clear 8 bytes
REPT 7
    ld [hld], a ; 2
ENDR
    ld [hl], a  ; 2

    ld a, 16    ; 2
    add a, c    ; 1
    ld c, a     ; 1

    inc e       ; 1
    inc e       ; 1
    ld d, $88   ; 2
    jr nc, .label\@ ; 3 taken / 2 untaken

    inc b       ; 1
ENDR


FOR I, 2

    ld bc, BGFrameBuffer + (112 + I) * 16 ; 3
    AddPatchLocation StateC
    ld de, $8800 + (112 + I) * 2          ; 3

    SpinWaitForMode0

    ld h, b ; 1
    ld l, c ; 1

    ; Copy 8 bytes to VRAM
    MemCopyStride256Unrolled 8

    xor a, a    ; 1
    ; Clear 8 bytes
    ld c, 8     ; 2
:
    ld [hld], a ; 2
    dec c       ; 1
    jr nz, :-   ; 3 taken / 2 untaken

ENDR

    call BorderTileJuggleBody

    SpinWaitForLine 128

FOR I, 7

    ld bc, BGFrameBuffer + (I + 114) * 16 ; 3
    AddPatchLocation StateC
    ld de, $8800 + (I + 114) * 2          ; 3

    SpinWaitForMode0

    ld h, b ; 1
    ld l, c ; 1

    ; Copy 8 bytes to VRAM
    MemCopyStride256Unrolled 8

    xor a, a    ; 1
    ; Clear 8 bytes
    ld c, 8     ; 2
:
    ld [hld], a ; 2
    dec c       ; 1
    jr nz, :-   ; 3 taken / 2 untaken

ENDR

    SpinWaitForMode0
    ; Switch to LCDCF_BG8000, turn sprites off, on scanline 135
    SetLCDControlModified LCDCF_ON | LCDCF_BGON | LCDCF_BG8000 | LCDCF_OBJOFF | LCDCF_OBJ16 | LCDCF_WINOFF
    ConditionallySetBackgroundPalette HUD_PALETTE

    SpinWaitForLine 136

FOR I, 7

    ld bc, BGFrameBuffer + (I + 121) * 16 ; 3
    AddPatchLocation StateC
    ld de, $8800 + (I + 121) * 2          ; 3

    SpinWaitForMode0

    ld h, b ; 1
    ld l, c ; 1

    ; Copy 8 bytes to VRAM
    MemCopyStride256Unrolled 8

    xor a, a    ; 1
    ; Clear 8 bytes
    ld c, 8     ; 2
:
    ld [hld], a ; 2
    dec c       ; 1
    jr nz, :-   ; 3 taken / 2 untaken

ENDR


    SpinWaitForLine 144 ; Start of vblank
    ; Restore tile $00
    CopyTileFromHRAMUnrolled $8800, TileBackupA

    ; Handle timer interrupt flag
    call HandleTimer



StateC_Frame1:

/*
    ; If this is the patched version of StateC, then:
    ; Backup bitplane A of tile $FF (since bitplane B is about to be over-written after writing border graphics)
    ;CopyTileToHRAMUnrolled TileBackupB, $9800 - $10
    ld hl, $9800 - $10
    ld c, LOW(TileBackupB)
REPT 7
    ld a, [hli]     ; 2
    ldh [c], a      ; 2
    inc hl          ; 2
    inc c           ; 1
    inc c           ; 1
ENDR
    ld a, [hli]     ; 2
    ldh [c], a      ; 2
*/


    ; Write border graphics to tile $FF
    ;FillTileUnrolled $9800 - $10, %11111111
    WriteBorderGraphicsToTileFF

    SpinWaitForLine 0
    SpinWaitForMode0

FOR I, 7

    ld bc, BGFrameBuffer + 8 + (I + 1) * 16 ; 3
    AddPatchLocation StateC
    ld de, $8800 + 8 * 256 + (I + 1) * 2          ; 3

    SpinWaitForMode0

    ld h, b ; 1
    ld l, c ; 1

    ; Copy 8 bytes to VRAM
    MemCopyStride256Unrolled 8

    xor a, a    ; 1
    ; Clear 8 bytes
    ld c, 8     ; 2
:
    ld [hld], a ; 2
    dec c       ; 1
    jr nz, :-   ; 3 taken / 2 untaken

ENDR

    ;SpinWaitForMode0

    ld bc, BGFrameBuffer + 8 + 8 * 16 ; 3
    AddPatchLocation StateC
    ld de, $8800 + 8 * 256 + 8 * 2          ; 3

FOR I, 7  ; 7 x 16 = 112
IF I == 0
    ; Switch to LCDCF_BG8800
    SetLCDControlModified LCDCF_ON | LCDCF_BGON | LCDCF_BG8800 | LCDCF_OBJOFF | LCDCF_OBJ16 | LCDCF_WINOFF
    ConditionallySetBackgroundPalette TANGRAM_BG_PALETTE
ENDC

.label\@:

    SpinWaitForMode0

    ld h, b ; 1
    ld l, c ; 1

    ; Copy 8 bytes to VRAM
    MemCopyStride256Unrolled 8

    xor a, a    ; 1
    ; Clear 8 bytes
REPT 7
    ld [hld], a ; 2
ENDR
    ld [hl], a  ; 2

    ld a, 16    ; 2
    add a, c    ; 1
    ld c, a     ; 1
    inc e       ; 1
    inc e       ; 1
    ld d, $88 + 8   ; 2
    jr nc, .label\@ ; 3 taken / 2 untaken

    inc b       ; 1
ENDR

FOR I, 8
    ld bc, BGFrameBuffer + 8 + (I + 112) * 16   ; 3
    AddPatchLocation StateC
    ld de, $8800 + 8 * 256 + (I + 112) * 2      ; 3

    SpinWaitForMode0

    ld h, b ; 1
    ld l, c ; 1

    ; Copy 8 bytes to VRAM
    MemCopyStride256Unrolled 8

    xor a, a    ; 1
    ; Clear 8 bytes
    ld c, 8     ; 2
:
    ld [hld], a ; 2
    dec c       ; 1
    jr nz, :-   ; 3 taken / 2 untaken
ENDR


    ;call BorderTileJuggleBody


    SpinWaitForLine 123
    ; Backup first 8 bytes of tile $00
    ld b, HIGH($8800 + 0)
    ld d,  LOW($8800 + 0)
    ld c, LOW(TileBackupA + 0)  ; 2
    SpinWaitForMode0

    ld h, b      ; 1
    ld l, d      ; 1
REPT 7
    ld a, [hli]  ; 2
    ldh [c], a   ; 2
    inc c        ; 1
ENDR
    ld a, [hl]   ; 2
    ld [c], a    ; 2


    SpinWaitForLine 124
    ; Backup second 8 bytes of tile $00
    ld b, $88
    ld d, $00 + 8
    ld c, LOW(TileBackupA + 8)  ; 2
    SpinWaitForMode0
    ld h, b      ; 1
    ld l, d      ; 1
REPT 7
    ld a, [hli]  ; 2
    ldh [c], a   ; 2
    inc c        ; 1
ENDR
REPT 1
    ld a, [hl]   ; 2
    ld [c], a    ; 2
ENDR


    SpinWaitForLine 125
    ; Write first bitplane of border graphics to tile $00 (write 8 bytes)
    ld b, $88     ; 2
    ld c, $00     ; 2
    SpinWaitForMode0
    ld a, h       ; 1  (a = $FF)
    ld h, b       ; 1
    ld l, c       ; 1
REPT 8
    ld [hli], a   ; 2
ENDR


    SpinWaitForLine 126
    ; Write second bitplane of border graphics to tile $00 (write 8 bytes)
    ld b, $88     ; 2
    ld c, $00 + 8 ; 2
    SpinWaitForMode0
    ld a, h       ; 1  (a = $FF)
    ld h, b       ; 1
    ld l, c       ; 1
REPT 8
    ld [hli], a   ; 2
ENDR


    SpinWaitForLine 127

FOR I, 8
    ld bc, BGFrameBuffer + 8 + (I + 120) * 16   ; 3
    AddPatchLocation StateC
    ld de, $8800 + 8 * 256 + (I + 120) * 2      ; 3

    SpinWaitForMode0

    ld h, b ; 1
    ld l, c ; 1

    ; Copy 8 bytes to VRAM
    MemCopyStride256Unrolled 8

    xor a, a    ; 1
    ; Clear 8 bytes
    ld c, 8     ; 2
:
    ld [hld], a ; 2
    dec c       ; 1
    jr nz, :-   ; 3 taken / 2 untaken
ENDR


    ;SpinWaitForLine 135
    SpinWaitForMode0

    ; Switch to LCDCF_BG8000, turn sprites off
    SetLCDControlModified LCDCF_ON | LCDCF_BGON | LCDCF_BG8000 | LCDCF_OBJOFF | LCDCF_OBJ16 | LCDCF_WINOFF
    ConditionallySetBackgroundPalette HUD_PALETTE

    SpinWaitForLine 144 ; Start of vblank
    ; Restore tile $00
    CopyTileFromHRAMUnrolled $8800, TileBackupA

    ; Handle timer interrupt flag
    call HandleTimer

    ret
StateCEnd:
assert StateCEnd - StateC <= STATEC_PATCHED_ALLOCATION_SIZE

PatchIncrementStateCLoadAddresses:
FOR I, NUM_PATCH_LOCATIONS
    ld hl, StateC_patch_{d:I} + StateC_Patched + 1  ; 3
    inc [hl]   ; 3
ENDR
    ret



SECTION "PreparePolygon_3", ROM0
PreparePolygon_3:
    LoadWordFromAddress bc, HeldPieceCoords + Piece_Vertices
    ldh a, [HeldPieceCoords + Piece_Angle]
    and a, 127
    add a, a    ; x2
    ld c, a
    ld a, b     ; 1
    adc 0       ; 2
    ld b, a     ; 1
    ld a, c
    add a, a    ; x4  (x6 = x(2 + 4) = x2 + x4)
    ld d, a
    ld a, b     ; 1
    adc 0       ; 2
    ld b, a     ; 1
    ld a, d
    add a, c
    ld c, a
    ld a, b     ; 1
    adc 0       ; 2
    ld b, a     ; 1

FOR I, 5
    ld a, [bc]
    ldh [Vertices + I], a
    inc bc
ENDR
    ld a, [bc]
    ldh [Vertices + 5], a
    ret


SECTION "PreparePolygon_4", ROM0
PreparePolygon_4:
    LoadWordFromAddress bc, HeldPieceCoords + Piece_Vertices
    ldh a, [HeldPieceCoords + Piece_Angle]
    and a, 63
    ld d, a
    ld a, b
    cp a, HIGH(VertexRotations2)
    jr nz, :+
    ld a, c
    cp a, LOW(VertexRotations2)
    jr nz, :+
    ld a, d
    and a, 31
    ld d, a
:
    ld a, d
    add a, a    ; x2
    add a, a    ; x4
    add a, a    ; x8

    ld c, a
    ld a, b     ; 1
    adc 0       ; 2
    ld b, a     ; 1

FOR I, 7
    ld a, [bc]
    ldh [Vertices + I], a
    inc bc
ENDR

    ld a, [bc]
    ldh [Vertices + 7], a
    ret



MACRO DrawPolygonTemplate

\1DrawPolygon_\2:

    call PreparePolygon_\2

FOR INDEX, \2
FOR OFFSET, 2
    ldh a, [Vertices + ((INDEX + OFFSET) % \2) * 2 + 0] ; local_x
    ld b, a
    ldh a, [HeldPieceCoords + Piece_X]
IF STRCMP("\1", "FG") == 0
    and a, 7
    add a, 32 - 4
ENDC
    add a, b
    ldh [LineParams + 0 + OFFSET * 2], a ; xn

    ldh a, [Vertices + ((INDEX + OFFSET) % \2) * 2 + 1] ; local_y
    ld b, a
    ldh a, [HeldPieceCoords + Piece_Y]
IF STRCMP("\1", "FG") == 0
    and a, 7
    add a, 32 - 4
ELSE
    add a, 16
ENDC
    add a, b
    ;add a, 16
    ldh [LineParams + 1 + OFFSET * 2], a ; yn
ENDR

    call \1DrawLineXY                 ; drawline
ENDR
    ret
ENDM


    DrawPolygonTemplate FG, 3
    DrawPolygonTemplate BG, 3

    DrawPolygonTemplate FG, 4
    DrawPolygonTemplate BG, 4



; --------------------------------------------------------------------------------------------

MACRO FGDrawLineIncX
    inc l
ENDM

MACRO FGDrawLineDecX
    dec l
ENDM

MACRO FGDrawLineIncY
    ld a, l     ; 1
    add a, 8    ; 2 (8 bytes per scanline)
    ld l, a     ; 1
    ld a, h     ; 1
    adc 0       ; 2
    ld h, a     ; 1
ENDM

MACRO BGDrawLineIncX
    inc l
ENDM

MACRO BGDrawLineDecX
    dec l
ENDM

MACRO BGDrawLineIncY
    ld a, l     ; 1
    add a, 16   ; 2 (16 bytes per scanline)
    ld l, a     ; 1
    ld a, h     ; 1
    adc 0       ; 2
    ld h, a     ; 1
ENDM

MACRO DrawLineTemplate

\1DrawLineXY:
    xor a, a
    ldh [Temps + 0], a
    ldh a, [LineParams + 0] ; x0
    ld b, a
    ldh a, [LineParams + 2] ; x1
    sub a, b
    jr nc, .noswapx
    ld d, a
    ld a, 1
    ldh [Temps + 0], a
    ld a, d
    cpl
    inc a
.noswapx
    ld d, a
    ldh a, [LineParams + 1] ; y0
    ld c, a
    ldh a, [LineParams + 3] ; y1
    sub a, c
    jr nc, .noswapy
    ld e, a ; box height
    ldh a, [LineParams + 3] ; swap y
    ldh [LineParams + 1], a
    ld a, c
    ldh [LineParams + 3], a

    ldh a, [LineParams + 2] ; swap x
    ldh [LineParams + 0], a
    ld a, b
    ldh [LineParams + 2], a
    ldh a, [Temps + 0]
    xor a, 1
    ldh [Temps + 0], a

    ld a, e
    cpl
    inc a
.noswapy
    ld e, a
    inc e
    inc d
    ld b, d
    ld a, e
    sub a, b
    jr nc, .tall

    ; wide
    ld a, e ; (box height)
    ldh [LineSlope2], a
    ld a, d ; requires LineSlope >= LineSlope2 (box width)
    ldh [LineSlope], a

    ld hl, PixelBitLUT
    ldh a, [LineParams + 0] ; x0
    ld b, a
    and a, 7
    add a, l
    ld l, a
    ld c, [hl]   ; initial pixel bit (basically X offset of start within a tile row)

    ld hl, RasterStartAddressTable\1
    ldh a, [LineParams + 1] ; y0
    add a, a
    ld e, a
    ld a, h
    adc 0
    ld h, a
    ld a, e
    add a, l
    ld l, a
    ld a, [hli]
    ld h, [hl]
    ld l, a


    srl b
    srl b
    srl b
    ld a, l
    add a, b
    ld l, a

    ld b, d    ; total number of columns

    ldh a, [Temps + 0]
    or a
    jr nz, .goleft
    call \1DrawLine3 ; right-right-down
    ret
.goleft
    call \1DrawLine4 ; left-left-down
    ret

.tall
    ; tall
    ld a, d ; (box width)
    ldh [LineSlope2], a
    ld a, e ; requires LineSlope >= LineSlope2 (box height)
    ldh [LineSlope], a

    ld hl, PixelBitLUT
    ldh a, [LineParams + 0] ; x0
    ld b, a
    and a, 7
    add a, l
    ld l, a
    ld c, [hl]   ; initial pixel bit (basically X offset of start within a tile row)

    ld hl, RasterStartAddressTable\1
    ldh a, [LineParams + 1] ; y0
    add a, a
    ld d, a
    ld a, h
    adc 0
    ld h, a
    ld a, d
    add a, l
    ld l, a
    ld a, [hli]
    ld h, [hl]
    ld l, a

    srl b
    srl b
    srl b
    ld a, l
    add a, b
    ld l, a


    ld b, e    ; total number of rows

    ldh a, [Temps + 0]
    or a
    jr nz, .goleft2
    call \1DrawLine ; down-down-right
    ret
.goleft2
    call \1DrawLine2 ; down-down-left
    ret


\1DrawLine: ; down-down-right lines
    ldh a, [LineSlope]
    ld e, a
    ;inc b
.next
    ; plot pixel
    ld a, c     ; 1
    or a, [hl]  ; 2
    ld [hl], a  ; 2

    ;ld a, [hl]  ; 2
    ;or a, c     ; 1
    ;ld [hl], a  ; 2

    \1DrawLineIncY
    ldh a, [LineSlope2] ; other slope
    ld d, a ; amount to subtract
    ld a, e
    sub a, d
    ld e, a
    jr z, .skipTileXInc2
    jr nc, .skipTileXInc
.skipTileXInc2
    ldh a, [LineSlope]
    ld d, a ; amount to add
    ld a, e
    add a, d
    ld e, a
    rrc c ; rotate
.skipTileXInc
    jr nc, .skipMapXInc
    \1DrawLineIncX
.skipMapXInc
    dec b
	jr nz, .next
    ret

\1DrawLine2: ; down-down-left lines
    ldh a, [LineSlope]
    ld e, a
    ;inc b
.next
    ld a, [hl]
    or a, c
    ld [hl], a ; plot pixel
    \1DrawLineIncY
    ldh a, [LineSlope2] ; other slope
    ld d, a ; amount to subtract
    ld a, e
    sub a, d
    ld e, a
    jr z, .skipTileXInc2
    jr nc, .skipTileXInc
.skipTileXInc2
    ldh a, [LineSlope]
    ld d, a ; amount to add
    ld a, e
    add a, d
    ld e, a
    rlc c ; rotate
.skipTileXInc
    jr nc, .skipMapXInc
    \1DrawLineDecX
.skipMapXInc
    dec b
	jr nz, .next
    ret


\1DrawLine3: ; right-right-down lines
    ldh a, [LineSlope]
    ld e, a
    ;inc b
.next
    ld a, [hl]
    or a, c
    ld [hl], a ; plot pixel
    rrc c ; rotate
    jr nc, .skipMapXInc
    \1DrawLineIncX
.skipMapXInc
    ldh a, [LineSlope2] ; other slope
    ld d, a ; amount to subtract
    ld a, e
    sub a, d
    ld e, a
    jr z, .skipTileYInc2
    jr nc, .skipTileYInc
.skipTileYInc2
    ldh a, [LineSlope]
    ld d, a ; amount to add
    ld a, e
    add a, d
    ld e, a
    \1DrawLineIncY
    ;inc l
.skipTileYInc
.skipMapYInc
    dec b
	jr nz, .next
    ret


\1DrawLine4: ; left-left-down lines
    ldh a, [LineSlope]
    ld e, a
    ;inc b
.next
    ld a, [hl]
    or a, c
    ld [hl], a ; plot pixel
    rlc c ; rotate
    jr nc, .skipMapXInc
    \1DrawLineDecX
.skipMapXInc
    ldh a, [LineSlope2] ; other slope
    ld d, a ; amount to subtract
    ld a, e
    sub a, d
    ld e, a
    jr z, .skipTileYInc2
    jr nc, .skipTileYInc
.skipTileYInc2
    ldh a, [LineSlope]
    ld d, a ; amount to add
    ld a, e
    add a, d
    ld e, a
    \1DrawLineIncY
    ;inc l
.skipTileYInc
.skipMapYInc
    dec b
	jr nz, .next
    ret

ENDM


SECTION "DrawLine", ROMX

    DrawLineTemplate FG
    DrawLineTemplate BG


SECTION "ReadJoypad", ROMX
; a = state of joypad
; - destroys b
readJoypad:
  ; Procedure from http://www.emulatronia.com/doctec/consolas/gameboy/gameboy.txt
  
  ; Read first part of joypad state
  ld a, P1F_5
  ld [rP1], a
  
  ; Read several times to burn cycles waiting for register to update
  ld a, [rP1]
  ld a, [rP1]
  cpl ; Invert so 1=on and 0=off
  and $0f ; Only want 4 least significant bits
  swap a  ; Swap nibbles
  ld b, a ; Store in b
  
  ; Read second part of joypad state
  ld a, P1F_4
  ld [rP1], a
  
  ; Read several times to burn cycles waiting for register to update
  ld a, [rP1]
  ld a, [rP1]
  ld a, [rP1]
  ld a, [rP1]
  ld a, [rP1]
  ld a, [rP1]
  cpl ; invert
  and $0f ; only 4 least significant bits
  or b ; Merge with b
  
  ret


SECTION "copy_dma_hrampart_to_hram", ROMX
copy_dma_hrampart_to_hram:
    ld de, SpriteDMARoutine
    ld hl, run_dma_hrampart
    ld b, run_dma_hrampart_end - run_dma_hrampart
.loop
    ld a, [hli]
    ld [de], a
    inc de
    dec b
    jr nz, .loop
    ret

SECTION "run_dma", ROMX
run_dma:          ; This part is in ROM
assert LOW(Sprites) == 0
    ld a, HIGH(Sprites)
    ld bc, $2946  ; B: wait time; C: LOW($FF46)
    ld hl, SpriteDMARoutine
    jp hl

run_dma_hrampart:
    ldh [c], a
.wait
    dec b
    jr nz, .wait
    ret
run_dma_hrampart_end:


SECTION "StateD", ROMX
StateD:

    xor a, a
    ld [InGameMenuState], a
    ld [InGameMenuScrollX], a
    ld [InGameMenuScrollY], a

    ld a, INGAME_MENU_BG_PALETTE
    ld [InGameMenuPalette], a

    ld a, INGAME_MENU_TRANSITION_COUNTER_INIT
    ld [InGameMenuTransitionCounter], a

    call ClearInGameMenuSelectorAll
    call UpdateInGameMenuSelector

StateDLoop: ; In-game menu

    SpinWaitIfNotInVBlank

    ; Backup tile $FF
    CopyTileToHRAMUnrolled TileBackupB, $9800 - $10

    ; Write border graphics to tile $FF
    ;FillTileUnrolled $9800 - $10, %11111111
    WriteBorderGraphicsToTileFF

    SpinWaitForLine 7
    SpinWaitForMode0

    ; Switch to LCDCF_BG8800
    SetLCDControl LCDCF_ON | LCDCF_BGON | LCDCF_BG8800 | LCDCF_OBJON | LCDCF_OBJ16 | LCDCF_WINOFF
    SetBackgroundPalette TANGRAM_BG_PALETTE

    ; In-game menu display

    ld a, [InGameMenuTransitionCounter]
    ld b, a
    rst SpinWaitForLineRST

    SpinWaitForMode0

    ; Set scroller registers to position where the desired dialog text becomes visible
    ld a, [InGameMenuScrollX]
    add a, $80
    ldh [rSCX], a
    ld a, [InGameMenuScrollY]
    ldh [rSCY], a

    SetLCDControl LCDCF_ON | LCDCF_BGON | LCDCF_BG8000 | LCDCF_OBJOFF | LCDCF_OBJ16 | LCDCF_WINOFF | LCDCF_BG9C00
    SetBackgroundPalette [InGameMenuPalette]

    ld a, [InGameMenuTransitionCounter]
    cpl
    inc a
    add a, 145
    ld b, a
    rst SpinWaitForLineRST

    SpinWaitForMode0

    ; Reset scroll registers
    xor a, a
    ldh [rSCX], a
    ldh [rSCY], a

    SetLCDControl LCDCF_ON | LCDCF_BGON | LCDCF_BG8800 | LCDCF_OBJON | LCDCF_OBJ16 | LCDCF_WINOFF
    SetBackgroundPalette TANGRAM_BG_PALETTE


    call BorderTileJuggleBody

    SpinWaitForLine 135
    SpinWaitForMode0

    ; Switch to LCDCF_BG8000
    SetLCDControl LCDCF_ON | LCDCF_BGON | LCDCF_BG8000 | LCDCF_OBJOFF | LCDCF_OBJ16 | LCDCF_WINOFF | LCDCF_BG9800

    SetBackgroundPalette HUD_PALETTE

    SpinWaitForLine 144 ; Start of vblank
    ; Restore tile $00
    CopyTileFromHRAMUnrolled $8800, TileBackupA

    ; Execute DMA transfer to OAM
    call run_dma

    ; Poll joypad and handle inputs
    call HandleInputEvents

    call UpdateInGameMenuTransitionAnimation

    CheckPendingStateAddress

    jp StateDLoop


SECTION "InGameMenuSelectorPositionTable", ROM0
InGameMenuSelectorPositionTable:
FOR I, NUM_IN_GAME_MENU_ITEMS
    dw $9C94 + $40 + $20 * I
ENDR

SECTION "UpdateInGameMenuSelector", ROM0
UpdateInGameMenuSelector:
    ld hl, InGameMenuSelectorPositionTable
    ld a, [InGameMenuState]
    call IndirectLoadWordFromTable
    ld a, TILE_INDEX_ARROW
    ld [de], a
    ret

SECTION "ClearInGameMenuSelector", ROMX
ClearInGameMenuSelector:
    ld hl, InGameMenuSelectorPositionTable
    ld a, [InGameMenuState]
    call IndirectLoadWordFromTable
    xor a, a
    ld [de], a
    ret

SECTION "ClearInGameMenuSelectorAll", ROMX
ClearInGameMenuSelectorAll:
    ld c, NUM_IN_GAME_MENU_ITEMS
:
    ld hl, InGameMenuSelectorPositionTable
    ld a, c
    dec a
    call IndirectLoadWordFromTable
    xor a, a
    ld [de], a
    dec c
    jr nz, :-
    ret

SECTION "UpdateInGameMenuTransitionAnimation", ROMX
UpdateInGameMenuTransitionAnimation:
    ld a, [InGameMenuState]
    cp a, $FF
    jr z, :+
    cp a, $FE
    jr z, :+
    jr :++++
:
    ; Transitioning out
    ld a, [InGameMenuTransitionCounter]
    cp a, INGAME_MENU_TRANSITION_COUNTER_INIT
    jr nz, :++
    ld a, [InGameMenuState]
    cp a, $FE
    jr z, :+
    StoreWordToAddress PendingStateAddress, StateA
    ret
:
    StoreWordToAddress PendingStateAddress, TangramSelectionScreen
    ret
:
    inc a
    inc a
    inc a
    ld [InGameMenuTransitionCounter], a
    ret
:
    cp a, $FD
    ret z
    ; Transitioning in
    ld a, [InGameMenuTransitionCounter]
    cp a, INGAME_MENU_TRANSITION_COUNTER_LIMIT
    ret z
    dec a
    dec a
    dec a
    ld [InGameMenuTransitionCounter], a
    ret


SECTION "BorderTileJuggleBody", ROMX
BorderTileJuggleBody:

    ld d,  LOW($8800 + 0)
    ld c, LOW(TileBackupA + 0)  ; 2
    ;SpinWaitForLine 122
    ld b, 122       ; 2
:
    ldh a, [rLY]    ; 3                     ; 2B
    cp a, b         ; 1                     ; 1B
    jr nz, :-       ; 3 taken / 2 untaken   ; 2B

    ; Backup first 8 bytes of tile $00
    ld b, HIGH($8800 + 0)
    SpinWaitForMode0

    ld h, b      ; 1
    ld l, d      ; 1
REPT 7
    ld a, [hli]  ; 2
    ldh [c], a   ; 2
    inc c        ; 1
ENDR
    ld a, [hl]   ; 2
    ldh [c], a   ; 2



    ;SpinWaitForLine 123
    ; Restore top 4 rows (both bitplanes) of tile $FF (copy 8 bytes)
    ld b, HIGH($9800 - 16)
    ld d,  LOW($9800 - 16)
    ld c, LOW(TileBackupB + 0)  ; 2
    SpinWaitForMode0
    ld h, b      ; 1
    ld l, d      ; 1
REPT 7
    ldh a, [c]   ; 2
    ld [hli], a  ; 2
    inc c        ; 1
ENDR
    ldh a, [c]   ; 2
    ld [hl], a   ; 2



    ;SpinWaitForLine 124
    ; Backup second 8 bytes of tile $00
    ld b, $88
    ld d, $00 + 8
    ld c, LOW(TileBackupA + 8)  ; 2
    SpinWaitForMode0
    ld h, b      ; 1
    ld l, d      ; 1
REPT 7
    ld a, [hli]  ; 2
    ldh [c], a   ; 2
    inc c        ; 1
ENDR
    ld a, [hl]   ; 2
    ldh [c], a   ; 2


    ;SpinWaitForLine 125
    ; Write first bitplane of border graphics to tile $00 (write 8 bytes)
    ld b, $88     ; 2
    ld c, $00     ; 2
    SpinWaitForMode0
    ld a, h       ; 1  (a = $FF)
    ld h, b       ; 1
    ld l, c       ; 1
REPT 8
    ld [hli], a   ; 2
ENDR


    ld c, $00 + 8 ; 2
    SpinWaitForLine 126
    ; Write second bitplane of border graphics to tile $00 (write 8 bytes)
    ld b, $88     ; 2
    SpinWaitForMode0
    ld a, h       ; 1  (a = $FF)
    ld h, b       ; 1
    ld l, c       ; 1
REPT 8
    ld [hli], a   ; 2
ENDR


    ld d,  LOW($9800 - 8)
    ld c, LOW(TileBackupB + 8)  ; 2
    SpinWaitForLine 127
    ; Restore bottom 4 rows (both bitplanes) of tile $FF (copy 8 bytes)
    ld b, HIGH($9800 - 8)
    SpinWaitForMode0
    ld h, b      ; 1
    ld l, d      ; 1
REPT 7
    ldh a, [c]   ; 2
    ld [hli], a  ; 2
    inc c        ; 1
ENDR
    ldh a, [c]   ; 2
    ld [hl], a   ; 2

    ret

SECTION "WriteDecimalNumeral", ROMX
WriteDecimalNumeral:
    add a, $28
    ld c, a
    and a, $10
    rrca
    add a, c
    ld [hl], a
    ret

SECTION "HandleTimer", ROMX
HandleTimer:
    ld a, [TimerFreeze]
    or a, a
    ret z

    ldh a, [rIF]
    bit 2, a
    jr z, .noupdate
    xor a, a
    ld [rIF], a
    ld hl, TimeCounter      ; 3
    ld a, [hl]              ; 2
    ld b, a                 ; 1
    inc [hl]                ; 3
    ld a, [hl]              ; 2
    xor a, b                ; 1
    bit 4, a                ; 2
    jr z, .noupdate

    ld a, [Elapsed.seconds]
    inc a
    ld [Elapsed.seconds], a
    cp a, 10
    jr nz, .updateseconds

    xor a, a
    ld [Elapsed.seconds], a
    ld a, [Elapsed.decaseconds]
    inc a
    ld [Elapsed.decaseconds], a
    cp a, 6
    jr nz, .updatedecaseconds

    xor a, a
    ld [Elapsed.decaseconds], a
    ld a, [Elapsed.minutes]
    inc a
    ld [Elapsed.minutes], a
    cp a, 10
    jr nz, .updateminutes

    xor a, a
    ld [Elapsed.minutes], a
    ld a, [Elapsed.decaminutes]
    inc a
    ld [Elapsed.decaminutes], a

.updatedecaminutes
    ld a, [Elapsed.decaminutes]
    ld hl, TIME_DISPLAY_MAP_BASE_ADDRESS + 0
    call WriteDecimalNumeral
.updateminutes
    ld a, [Elapsed.minutes]
    ld hl, TIME_DISPLAY_MAP_BASE_ADDRESS + 1
    call WriteDecimalNumeral
.updatedecaseconds
    ld a, [Elapsed.decaseconds]
    ld hl, TIME_DISPLAY_MAP_BASE_ADDRESS + 3
    call WriteDecimalNumeral
.updateseconds
    ld a, [Elapsed.seconds]
    ld hl, TIME_DISPLAY_MAP_BASE_ADDRESS + 4
    call WriteDecimalNumeral
.noupdate
    ret


SECTION "HandleInputEvents", ROMX
HandleInputEvents:
    ld a, [LastJoypadState]
    ld c, a
    call readJoypad
    ld [LastJoypadState], a
    ld b, a
    xor a, c
    ld d, a     ; d = b xor c
    and a, c
    ld c, a     ; 8 "Released" events
    ld a, d
    and a, b
    ld b, a     ; 8 "Pressed" events

    LoadWordFromAddress hl, CurrentActionState

    ; Perform action state transitions until all events have been consumed
:
    ld d, b
    ld e, c
    push de
    jp hl ; ActionState
ActionStateReturnAddress:

    pop hl
    pop bc
    pop de
IF 1
    ld a, b
    or a, c
    jp nz, :-
ELSE
    ld a, b
    cp a, d
    jr nz, :-
    ld a, c
    cp a, e
    jr nz, :-
ENDC


    StoreWordToAddress CurrentActionState, hl
    ret


MACRO ActionState
AS_\1:

IF STRCMP("\2", "__") != 0
    ld a, b
    or a, c
    ld hl, AS_\2
    jr z, .action
ENDC

    ld hl, AS_\1

    ; Calculate mask of non-self-referring state transitions
EVENT_MASK = 0
FOR I, 3, _NARG+1
J = I - 3
IF STRCMP("\<I>", "__") != 0
EVENT_MASK = EVENT_MASK | (1 << J)
ENDC
ENDR


    ld a, c
    and a, LOW(EVENT_MASK)
    ld c, a
    ld a, b
    and a, HIGH(EVENT_MASK)
    ld b, a


FOR I, 3, _NARG+1
;J = 15 - (I - 3) ; Reverse order so that A and B button presses have higher priority.
J = I - 3
K = 3 + J

IF STRCMP("\<K>", "__") != 0

/*
    ld a, (1 << (J & 7)) ; 2B
    and a, b             ; 1B
    jr z, :+             ; 2B
    ld hl, AS_\<K>       ; 3B
    jr .action           ; 2B
:
*/


    ; Test for event flag
IF J < 8
    bit J, c            ; 2B
ELSE
    bit J - 8, b        ; 2B
ENDC



    jr z, :+            ; 2B


    ; Clear event flag
IF J < 8
    res J, c            ; 2B
ELSE
    res J - 8, b        ; 2B
ENDC


    ;call ClearActionStateEventFlag.clearflag_{d:J} ; 3B
    
    ld hl, AS_\<K>      ; 3B
    jr .action          ; 2B
:
ENDC


ENDR

.action
    push bc
    push hl
    ld de, ActionStateReturnAddress
    push de
ENDM


/*
SECTION "ClearActionStateEventFlag", ROM0
ClearActionStateEventFlag:
    ; Clear event flag
FOR J, 16
.clearflag_{d:J}
IF J < 8
    res J, c            ; 2B
ELSE
    res J - 8, b        ; 2B
ENDC
ENDR
    ret
*/



MACRO IncHeldPieceCoord
    ldh a, [HeldPieceCoords + \1]   ; 3
    inc a                           ; 1
    ldh [HeldPieceCoords + \1], a   ; 3
ENDM

MACRO DecHeldPieceCoord
    ldh a, [HeldPieceCoords + \1]   ; 3
    dec a                           ; 1
    ldh [HeldPieceCoords + \1], a   ; 3
ENDM


MACRO BoundedPieceMovementTemplate
MovePieceBounded_\1_\3:
    ldh a, [HeldPieceCoords + \1]   ; 3
    \3 a                                ; 1
    ld b, a
    ldh a, [HeldPieceCoords + \2]
    cp a, b
    jr z, :+
    ld a, b
    ldh [HeldPieceCoords + \1], a   ; 3
:    
    ret
ENDM

SECTION "PieceMoveRoutines", ROM0
    BoundedPieceMovementTemplate Piece_X, Piece_BoundXMin, dec
    BoundedPieceMovementTemplate Piece_Y, Piece_BoundYMin, dec
    BoundedPieceMovementTemplate Piece_X, Piece_BoundXMax, inc
    BoundedPieceMovementTemplate Piece_Y, Piece_BoundYMax, inc


SECTION "ActionStates", ROMX

;   In-game interactivity
;
;                   Co  No  -A  -B  -s  -S  -R  -L  -U  -D  +A  +B  +s  +S  +R  +L  +U  +D
    ActionState     00, __, __, __, __, __, __, __, __, __, A0, B0, 09, S0, 01, 02, 03, 04      ; InGameRoot
    ret

;   Piece movement
;
;                   Co  No  -A  -B  -s  -S  -R  -L  -U  -D  +A  +B  +s  +S  +R  +L  +U  +D
    ActionState     01, __, __, __, __, __, 00, __, __, __, A0, B0, __, __, __, 00, 08, 05      ; MovePieceRight
    call MovePieceBounded_Piece_X_inc
    ret

;                   Co  No  -A  -B  -s  -S  -R  -L  -U  -D  +A  +B  +s  +S  +R  +L  +U  +D
    ActionState     02, __, __, __, __, __, __, 00, __, __, A0, B0, __, __, 00, __, 07, 06      ; MovePieceLeft
    call MovePieceBounded_Piece_X_dec
    ret

;                   Co  No  -A  -B  -s  -S  -R  -L  -U  -D  +A  +B  +s  +S  +R  +L  +U  +D
    ActionState     03, __, __, __, __, __, __, __, 00, __, __, B0, __, __, 08, 07, __, 00      ; MovePieceUp
    call MovePieceBounded_Piece_Y_dec
    ret

;                   Co  No  -A  -B  -s  -S  -R  -L  -U  -D  +A  +B  +s  +S  +R  +L  +U  +D
    ActionState     04, __, __, __, __, __, __, __, __, 00, __, B0, __, __, 05, 06, 00, __      ; MovePieceDown
    call MovePieceBounded_Piece_Y_inc
    ret

;                   Co  No  -A  -B  -s  -S  -R  -L  -U  -D  +A  +B  +s  +S  +R  +L  +U  +D
    ActionState     05, __, __, __, __, __, 04, __, __, 01, __, B0, __, __, __, __, __, __      ; MovePieceRightDown
    call MovePieceBounded_Piece_X_inc
    call MovePieceBounded_Piece_Y_inc
    ret

;                   Co  No  -A  -B  -s  -S  -R  -L  -U  -D  +A  +B  +s  +S  +R  +L  +U  +D
    ActionState     06, __, __, __, __, __, __, 04, __, 02, __, B0, __, __, __, __, __, __      ; MovePieceDownLeft
    call MovePieceBounded_Piece_X_dec
    call MovePieceBounded_Piece_Y_inc
    ret

;                   Co  No  -A  -B  -s  -S  -R  -L  -U  -D  +A  +B  +s  +S  +R  +L  +U  +D
    ActionState     07, __, __, __, __, __, __, 03, 02, __, __, B0, __, __, __, __, __, __      ; MovePieceLeftUp
    call MovePieceBounded_Piece_X_dec
    call MovePieceBounded_Piece_Y_dec
    ret

;                   Co  No  -A  -B  -s  -S  -R  -L  -U  -D  +A  +B  +s  +S  +R  +L  +U  +D
    ActionState     08, __, __, __, __, __, 03, __, 01, __, __, B0, __, __, __, __, __, __      ; MovePieceUpRight
    call MovePieceBounded_Piece_X_inc
    call MovePieceBounded_Piece_Y_dec
    ret

;                   Co  No  -A  -B  -s  -S  -R  -L  -U  -D  +A  +B  +s  +S  +R  +L  +U  +D
    ActionState     09, 00, __, __, __, __, __, __, __, __, __, __, __, __, __, __, __, __      ; SwitchToNextPiece
    StoreWordToAddress PendingStateAddress, .deferredSwitchToNextPiece
    ret
.deferredSwitchToNextPiece
    call SwitchToNextPiece
    jp StateALoop

;                   Co  No  -A  -B  -s  -S  -R  -L  -U  -D  +A  +B  +s  +S  +R  +L  +U  +D
    ActionState     A0, __, 00, __, __, __, __, __, __, __, __, __, __, __, A1, A2, A3, A4      ; AHeldDown
    ret

;                   Co  No  -A  -B  -s  -S  -R  -L  -U  -D  +A  +B  +s  +S  +R  +L  +U  +D
    ActionState     A1, __, 00, __, __, __, A0, __, __, __, __, __, __, __, __, __, __, __      ; RotatePieceRight
    IncHeldPieceCoord Piece_Angle
    ret

;                   Co  No  -A  -B  -s  -S  -R  -L  -U  -D  +A  +B  +s  +S  +R  +L  +U  +D
    ActionState     A2, __, 00, __, __, __, __, A0, __, __, __, __, __, __, __, __, __, __      ; RotatePieceLeft
    DecHeldPieceCoord Piece_Angle
    ret

;   In-game pause menu
;
;                   Co  No  -A  -B  -s  -S  -R  -L  -U  -D  +A  +B  +s  +S  +R  +L  +U  +D
    ActionState     S0, S1, __, __, __, __, __, __, __, __, __, __, __, __, __, __, __, __      ; EnterPauseMenu
    StoreWordToAddress PendingStateAddress, StateD
    ; Play sound effect
    call SFX_OpenMenu
    ret

;                   Co  No  -A  -B  -s  -S  -R  -L  -U  -D  +A  +B  +s  +S  +R  +L  +U  +D
    ActionState     S1, __, __, __, __, __, __, __, __, __, S5, __, __, S2, __, __, S4, S3      ; PauseMenuRoot
    ret

;                   Co  No  -A  -B  -s  -S  -R  -L  -U  -D  +A  +B  +s  +S  +R  +L  +U  +D
    ActionState     S2, S1, __, __, __, __, __, __, __, __, __, __, __, __, __, __, __, __      ; LeavePauseMenu
    ld a, [InGameMenuTransitionCounter]
    cp a, INGAME_MENU_TRANSITION_COUNTER_LIMIT
    jr nz, :+
    ld a, $FF
    ld [InGameMenuState], a
    ; Play sound effect
    call SFX_CloseMenu
:
    ret

;                   Co  No  -A  -B  -s  -S  -R  -L  -U  -D  +A  +B  +s  +S  +R  +L  +U  +D
    ActionState     S7, __, __, __, __, __, __, __, __, __, __, __, __, __, __, __, __, __      ; LeavePauseMenuAndGoToSelection
    ld a, [InGameMenuTransitionCounter]
    cp a, INGAME_MENU_TRANSITION_COUNTER_LIMIT
    jr nz, :+
    ld a, $FE
    ld [InGameMenuState], a
:
    ret

;                   Co  No  -A  -B  -s  -S  -R  -L  -U  -D  +A  +B  +s  +S  +R  +L  +U  +D
    ActionState     S3, S1, __, __, __, __, __, __, __, __, __, __, __, __, __, __, __, __      ; NudgePauseSelectorDown
    ld a, [InGameMenuState]
    cp a, $FF
    ret z
    cp a, NUM_IN_GAME_MENU_ITEMS - 1
    jr z, :+
    inc a
    ld b, a
    call ClearInGameMenuSelector
    ld a, b
    ld [InGameMenuState], a
    call UpdateInGameMenuSelector
    ; Play sound effect
    call SFX_Menu
:
    ret

;                   Co  No  -A  -B  -s  -S  -R  -L  -U  -D  +A  +B  +s  +S  +R  +L  +U  +D
    ActionState     S4, S1, __, __, __, __, __, __, __, __, __, __, __, __, __, __, __, __      ; NudgePauseSelectorUp
    ld a, [InGameMenuState]
    cp a, $FF
    ret z
    or a, a
    jr z, :+
    dec a
    ld b, a
    call ClearInGameMenuSelector
    ld a, b
    ld [InGameMenuState], a
    call UpdateInGameMenuSelector
    ; Play sound effect
    call SFX_Menu
:
    ret


;                   Co  No  -A  -B  -s  -S  -R  -L  -U  -D  +A  +B  +s  +S  +R  +L  +U  +D
    ActionState     S5, __, __, __, __, __, __, __, __, __, __, __, __, __, __, __, __, __      ; DoPauseMenuAction
    ld a, [InGameMenuState]
    cp a, $FF
    ret z
    cp a, 0
    jr z, .action_resume
    cp a, 1
    jr z, .action_check
    cp a, 2
    jr z, .action_reselect
    cp a, 3
    jr z, .action_solvethis
    cp a, 4
    jr z, .action_solveall
    cp a, 5
    jr z, .action_quit
    ret
.action_resume
    ; Play sound effect
    call SFX_CloseMenu
    ld a, $FF
    ld [InGameMenuState], a
    ret
.action_check
    StoreWordToAddress PendingStateAddress, .deferredCheckForPuzzleCompletion
    ret
.action_reselect
    StoreWordToAddress PendingStateAddress, TangramSelectionScreen
    ld a, $FF
    ld [InGameMenuState], a
    ret
.action_solvethis
    ; Load the active piece from ROM
    call LoadActivePieceFromROM
    ;call DrawPieceWireframesToBG
    ld a, $FF
    ld [AutoSolveWasUsed], a
    ld [InGameMenuState], a
    ret
.action_solveall
    call LoadPieceCoordsForTangram

    ; Move piece coords 4 pixels to the left, due to reasons
    ld de, PieceCoords
    ld c, 7
:
    inc de ; Skip Piece_ActiveMask
    ld a, [de]
    sub a, 4
    ld [de], a ; Piece_X
    ld a, e
    add a, 16 - 1
    ld e, a
    dec c
    jr nz, :-

    call LoadActivePiece
    call DrawPieceWireframesToBG
    ld a, $FF
    ld [AutoSolveWasUsed], a
    ld [InGameMenuState], a
    ret
.action_quit
    StoreWordToAddress PendingStateAddress, TitleScreen
    ld a, $FF
    ld [InGameMenuState], a
    ret
.deferredCheckForPuzzleCompletion
    call CheckForPuzzleCompletion
    cp a, 1
    jp nz, .notsolved
    ; Solved

    ; Get WRAM address of record time for this tangram
    ld hl, RecordTimes
    call GetRecordTimeAddress
    ; Check completion time against record time
    ld a, [Elapsed.decaminutes]
    ld b, a
    ld a, [hl]
    cp a, b
    jr c, .norecord
    jr nz, .record
    inc hl
    ld a, [Elapsed.minutes]
    ld b, a
    ld a, [hl]
    cp a, b
    jr c, .norecord
    jr nz, .record
    inc hl
    ld a, [Elapsed.decaseconds]
    ld b, a
    ld a, [hl]
    cp a, b
    jr c, .norecord
    jr nz, .record
    inc hl
    ld a, [Elapsed.seconds]
    ld b, a
    ld a, [hl]
    cp a, b
    jr c, .norecord
    jr z, .record ; Classify an identical time as being not a new record.
.record
    ; If the hint/auto mechanism was used then invalidate the new record
    ld a, [AutoSolveWasUsed]
    or a, a
    jr nz, .norecord

    ; Set "it's a new record" flag

    ; Write the record time to WRAM
    ld hl, RecordTimes
    call GetRecordTimeAddress
    ld a, [Elapsed.decaminutes]
    ld [hli], a
    ld a, [Elapsed.minutes]
    ld [hli], a
    ld a, [Elapsed.decaseconds]
    ld [hli], a
    ld a, [Elapsed.seconds]
    ld [hli], a

    ; Write record time to SRAM and write integrity string.
    ; Enable external RAM
    ld a, $0A
    ld [$0000], a
    ; Clear the integrity string
    ld hl, $A000 + NUM_TANGRAMS * 4
    ld bc, 9
    call ClearMem
    ; Get SRAM address of record time for this tangram
    ld hl, $A000
    call GetRecordTimeAddress
    ; Write the record time to SRAM
    ld a, [Elapsed.decaminutes]
    ld [hli], a
    ld a, [Elapsed.minutes]
    ld [hli], a
    ld a, [Elapsed.decaseconds]
    ld [hli], a
    ld a, [Elapsed.seconds]
    ld [hli], a
    ; Write the integrity string
    ld hl, $A000 + NUM_TANGRAMS * 4
    ld de, SRAMCheckString
:
    ld a, [de]
    ld [hli], a
    inc de
    or a, a
    jr nz, :-
    ; Disable external RAM
    ld a, $00
    ld [$0000], a
.norecord
    ; Switch to "Press A to continue" action state
    StoreWordToAddress CurrentActionState, AS_S8
    ;jp TangramSelectionScreen

    ; Set dialog to "Solved!"
    ld a, INGAME_MENU_BG_PALETTE_2
    ld [InGameMenuPalette], a
    ld a, -$60 - 3 * 8
    ld [InGameMenuScrollX], a
    ld a, $80
    ld [InGameMenuScrollY], a

    ; Play sound effect
    call SFX_Solved
    jp StateDLoop

.notsolved
    ; Not yet solved
    call LoadActivePiece
    SpinWaitForVBlank
    call DrawPieceWireframesToBG
    ; Switch to "Press A to continue" action state
    StoreWordToAddress CurrentActionState, AS_S6
    ld a, 1
    ld [TimerFreeze], a

    ; Reduce size of menu box
    ; ld a, $FD
    ; ld [InGameMenuState], a
    ; ld a, INGAME_MENU_TRANSITION_COUNTER_LIMIT + 3
    ; ld [InGameMenuTransitionCounter], a

    ; Set dialog to "Not solved"
    ld a, INGAME_MENU_BG_PALETTE_2
    ld [InGameMenuPalette], a
    ld a, $80
    ld [InGameMenuScrollY], a

    ; Play sound effect
    call SFX_NotSolved
    jp StateDLoop




;                   Co  No  -A  -B  -s  -S  -R  -L  -U  -D  +A  +B  +s  +S  +R  +L  +U  +D
    ActionState     S6, __, __, __, __, __, __, __, __, __, __, __, __, S2, __, __, __, __      ; ConfirmationDialog
    ret

;                   Co  No  -A  -B  -s  -S  -R  -L  -U  -D  +A  +B  +s  +S  +R  +L  +U  +D
    ActionState     S8, __, __, __, __, __, __, __, __, __, __, __, __, S7, __, __, __, __      ; ConfirmationDialog2
    ret

;                   Co  No  -A  -B  -s  -S  -R  -L  -U  -D  +A  +B  +s  +S  +R  +L  +U  +D
    ActionState     B0, __, __, 00, __, __, __, __, __, __, __, __, __, __, B1, B2, B3, B4      ; BHeldDown
    ret

;                   Co  No  -A  -B  -s  -S  -R  -L  -U  -D  +A  +B  +s  +S  +R  +L  +U  +D
    ActionState     B1, B0, __, __, __, __, __, __, __, __, __, __, __, __, __, __, __, __      ; NudgePieceRight
    ; Play sound effect
    call SFX_Nudge
    call MovePieceBounded_Piece_X_inc
    ret

;                   Co  No  -A  -B  -s  -S  -R  -L  -U  -D  +A  +B  +s  +S  +R  +L  +U  +D
    ActionState     B2, B0, __, __, __, __, __, __, __, __, __, __, __, __, __, __, __, __      ; NudgePieceLeft
    ; Play sound effect
    call SFX_Nudge
    call MovePieceBounded_Piece_X_dec
    ret

;                   Co  No  -A  -B  -s  -S  -R  -L  -U  -D  +A  +B  +s  +S  +R  +L  +U  +D
    ActionState     B3, B0, __, __, __, __, __, __, __, __, __, __, __, __, __, __, __, __      ; NudgePieceUp
    ; Play sound effect
    call SFX_Nudge
    call MovePieceBounded_Piece_Y_dec
    ret

;                   Co  No  -A  -B  -s  -S  -R  -L  -U  -D  +A  +B  +s  +S  +R  +L  +U  +D
    ActionState     B4, B0, __, __, __, __, __, __, __, __, __, __, __, __, __, __, __, __      ; NudgePieceDown
    ; Play sound effect
    call SFX_Nudge
    call MovePieceBounded_Piece_Y_inc
    ret

;                   Co  No  -A  -B  -s  -S  -R  -L  -U  -D  +A  +B  +s  +S  +R  +L  +U  +D
    ActionState     A3, A0, __, __, __, __, __, __, __, __, __, __, __, __, __, __, __, __      ; NudgeRotatePieceRight
    ; Play sound effect
    call SFX_Nudge2
    IncHeldPieceCoord Piece_Angle
    ret

;                   Co  No  -A  -B  -s  -S  -R  -L  -U  -D  +A  +B  +s  +S  +R  +L  +U  +D
    ActionState     A4, A0, __, __, __, __, __, __, __, __, __, __, __, __, __, __, __, __      ; NudgeRotatePieceLeft
    ; Play sound effect
    call SFX_Nudge2
    DecHeldPieceCoord Piece_Angle
    ret


;   Tangram selection screen
;

IncrementTangramSelectionIndex:
    ld a, [SelectionScreen_TangramIndex]
    cp a, NUM_TANGRAMS - 1
    jr z, :++
    inc a
    ld [SelectionScreen_TangramIndex], a
    ld a, [SelectionScreen_IndexDigits + 1]
    cp a, 9
    jr nz, :+
    ld a, [SelectionScreen_IndexDigits + 0]
    inc a
    ld [SelectionScreen_IndexDigits + 0], a
    ld a, -1
:
    inc a
    ld [SelectionScreen_IndexDigits + 1], a
    ret
:
    xor a, a
    ld [SelectionScreen_TangramIndex], a
    ld [SelectionScreen_IndexDigits + 0], a
    ld [SelectionScreen_IndexDigits + 1], a
    ret


DecrementTangramSelectionIndex:
    ld a, [SelectionScreen_TangramIndex]
    cp a, 0
    jr z, :++
    dec a
    ld [SelectionScreen_TangramIndex], a
    ld a, [SelectionScreen_IndexDigits + 1]
    cp a, 0
    jr nz, :+
    ld a, [SelectionScreen_IndexDigits + 0]
    dec a
    ld [SelectionScreen_IndexDigits + 0], a
    ld a, 10
:
    dec a
    ld [SelectionScreen_IndexDigits + 1], a
    ret
:
    ld a, NUM_TANGRAMS - 1
    ld [SelectionScreen_TangramIndex], a
    ld a, (NUM_TANGRAMS - 1) / 10
    ld [SelectionScreen_IndexDigits + 0], a
    ld a, (NUM_TANGRAMS - 1) % 10
    ld [SelectionScreen_IndexDigits + 1], a
    ret


;                   Co  No  -A  -B  -s  -S  -R  -L  -U  -D  +A  +B  +s  +S  +R  +L  +U  +D
    ActionState     T0, __, __, __, __, __, __, __, __, __, __, __, __, T4, T3, T3, T2, T1      ; SelectionScreenRoot
    ret

;                   Co  No  -A  -B  -s  -S  -R  -L  -U  -D  +A  +B  +s  +S  +R  +L  +U  +D
    ActionState     T1, T0, __, __, __, __, __, __, __, __, __, __, __, __, __, __, __, __      ; NudgeSelectionDown
    ; Play sound effect
    call SFX_Switch
    call DecrementTangramSelectionIndex
    ld a, [SelectionScreen_DigitPos]
    bit 0, a
    jr nz, :++
    ld c, 9
:
    call DecrementTangramSelectionIndex
    dec c
    jr nz, :-
:
    call UpdateTangramSelectionName
    ret


;                   Co  No  -A  -B  -s  -S  -R  -L  -U  -D  +A  +B  +s  +S  +R  +L  +U  +D
    ActionState     T2, T0, __, __, __, __, __, __, __, __, __, __, __, __, __, __, __, __      ; NudgeSelectionUp
    ; Play sound effect
    call SFX_Switch
    call IncrementTangramSelectionIndex
    ld a, [SelectionScreen_DigitPos]
    bit 0, a
    jr nz, :++
    ld c, 9
:
    call IncrementTangramSelectionIndex
    dec c
    jr nz, :-
:
    call UpdateTangramSelectionName
    ret


;                   Co  No  -A  -B  -s  -S  -R  -L  -U  -D  +A  +B  +s  +S  +R  +L  +U  +D
    ActionState     T3, T0, __, __, __, __, __, __, __, __, __, __, __, __, __, __, __, __      ; SwapSelectionDigit
    ld a, [SelectionScreen_DigitPos]
    xor a, 1
    ld [SelectionScreen_DigitPos], a
    ret

;                   Co  No  -A  -B  -s  -S  -R  -L  -U  -D  +A  +B  +s  +S  +R  +L  +U  +D
    ActionState     T4, 00, __, __, __, __, __, __, __, __, __, __, __, __, __, __, __, __      ; ChooseTangram
    ld hl, TangramAddressTable
    ld a, [SelectionScreen_TangramIndex]
    call IndirectLoadWordFromTable
    StoreWordToAddress TangramToLoad, de
    StoreWordToAddress PendingStateAddress, LoadTangramAndStartGame
    ret



SECTION "GetRecordTimeAddress", ROM0
GetRecordTimeAddress:
    ld a, [SelectionScreen_TangramIndex]
    rlca
    rlca
    ld b, a
    and a, $FC
    ld e, a
    ld a, b
    and a, $03
    ld d, a
    add hl, de
    ret


SECTION "IndirectLoadWordFromTable", ROM0
IndirectLoadWordFromTable:
    add a, a
    add a, l
    ld l, a
    ld a, h
    adc 0
    ld h, a
    ld a, [hli]
    ld e, a
    ld a, [hl]
    ld d, a
    ret


SECTION "ClearMem", ROM0
ClearMem:
:
    xor a, a        ; 1
    ld [hli], a     ; 2
    dec bc          ; 2
    ld a, b         ; 1
    or a, c         ; 1
    jr nz, :-
    ret

SECTION "SetMem", ROM0
SetMem:
    ld d, a         ; 1
:
    ld a, d         ; 1
    ld [hli], a     ; 2
    dec bc          ; 2
    ld a, b         ; 1
    or a, c         ; 1
    jr nz, :-
    ret

SECTION "CopyMem", ROM0
CopyMem:
:
    ld a, [hli]     ; 2
    ld [de], a      ; 2
    inc de          ; 2
    dec bc          ; 2
    ld a, b         ; 1
    or a, c         ; 1
    jr nz, :-
    ret

SECTION "IotaMem", ROMX
IotaMem:
:
    ld [hli], a     ; 2
    inc a
    dec c
    jr nz, :-
    ret

SECTION "ReverseIotaMem", ROMX
ReverseIotaMem:
:
    ld [hli], a     ; 2
    dec a
    dec c
    jr nz, :-
    ret

SECTION "CopyTiles", ROMX
CopyTiles:
:
REPT 16
    ld a, [hli]     ; 2
    ld [de], a      ; 2
    inc de          ; 2
ENDR
    dec c
    jr nz, :-
    ret

SECTION "CopyTilesStride2Source1BPP", ROMX
CopyTilesStride2Source1BPP:
:
REPT 8
    ld a, [hli]     ; 2
    ld [de], a      ; 2
    inc de          ; 2
    inc de          ; 2
ENDR
    dec c
    jr nz, :-
    ret

SECTION "CopyTilesStride2", ROMX
CopyTilesStride2:
:
REPT 8
    ld a, [hli]     ; 2
    ld [de], a      ; 2
    inc de          ; 2
    inc de          ; 2
    inc hl          ; 2
ENDR
    dec c
    jr nz, :-
    ret

SECTION "ClearTiles", ROMX
ClearTiles:
    xor a, a
:
REPT 16
    ld [hli], a     ; 2
ENDR
    dec c
    jr nz, :-
    ret


IF 0
calc84maniac — Yesterday at 11:35 PM
here's another take on the VWF stuff:
    ; Get power of 2 from column offset
    ldh a, [DrawText_PixelColumn]
assert (PowerOfTwoLut & $7) == 0
    ld hl, PowerOfTwoLut
    add a, l
    ld l, a
    ld h, [hl]
    ld l, b
.shiftloop
    add hl, hl
    jr nc, .shiftloop
    ; Render left side of character into first tile
    ld a, [de]
    or a, h
    ld [de], a
    ; Check for overflow into next tile...

    ; Optionally write right side of character into next tile
    ld a, l
    ld [de], a
basically it sets a bit in H as the loop terminator and shifts the font data across HL
jvsTSX — Yesterday at 11:36 PM
oh great now the asm highlight is working for LD
calc84maniac — Yesterday at 11:37 PM
it does if you use AVR asm
jvsTSX — Yesterday at 11:37 PM
alright i'll try it out
calc84maniac — Yesterday at 11:37 PM
technically this would be unrollable too but you can't use jp hl to jump into it
I personally like updating a JR offset, but it's less common to allocate code in RAM on GB
calc84maniac — Yesterday at 11:54 PM
I think something that could be more helpful is to draw each character independently instead of going row-by-row across the entire string
there's quite a bit of time spent looking up each source character row
when it could just be incremented in the loop
you may also be able to take advantage of the character width being the same in each row
calc84maniac — Today at 12:01 AM
so actually my code could end with this to keep DE in a good position to move to the next row
    ld a, [de]
    or a, h
    ld [de], a
    ld a, l
    ld hl, 16
    add hl, de
    ld [hl], a
fizzer — Today at 12:01 AM
This looks pretty good
fizzer — Today at 12:01 AM
I'm already doing this actually
I have some VRAM update stuff with tight timing and I needed two variants with different base addresses so I copied the code from ROM into RAM and patched the addresses.
calc84maniac — Today at 12:02 AM
ah yeah, if it's in RAM you could definitely just write the pixel column offset to a JR offset byte
fizzer — Today at 12:02 AM
That's a nice idea
calc84maniac — Today at 12:02 AM
instead of using a power-of-2 LUT
fizzer — Today at 12:03 AM
I'm likely to re-use this VWF code so it's probably worth improving it already
calc84maniac — Today at 12:03 AM
and if you do all character rows in a loop, you could update that offset outside the loop
fizzer — Today at 12:03 AM
I was thinking actually of re-factoring the loops so that the rows are the inner loop instead of the outer
So that I only need to fetch each character index from the string once
ENDC


;calc84maniac — Today at 10:47 PM
; I feel like it might be worth taking an extra byte+cycle of overhead to multiply by 3 instead of 4, to save a byte+cycle in each iteration

; Input:  Address of first char in text string : hl
;         Address of first tile in VRAM        : de
SECTION "DrawText", ROM0
DrawText:
    StoreWordToAddress DrawText_SourceAddr, hl
    StoreWordToAddress DrawText_DestAddr, de

    ld a, 8 ; Number of rows
.row
    ldh [DrawText_RowCount], a
    ld b, a
    LoadWordFromAddress hl, DrawText_SourceAddr
    push hl
    LoadWordFromAddress de, DrawText_DestAddr
    ld a, e     ; 1
    dec b       ; 1
    sla b       ; 2
    add a, b    ; 1
    ld e, a     ; 1

    xor a, a
    ldh [DrawText_PixelColumn], a ; Pixel column offset
    ld [de], a

.col
    ; Read next character from text string
    pop hl      ; 3
    ld a, [hl+] ; 2
    or a, a
    ; If character code is $00 then this is the end of the string
    jp z, .rowdone
    sub a, 31   ; 2
    push hl     ; 4

    ; Get pixel-width of character
    ld b, a                     ; 1
    ld hl, FontTileWidths       ; 3
    add a, l                    ; 1
    ld l, a                     ; 1
    ld a, [hl]                  ; 2
    ldh [DrawText_CharWidth], a ; 3
    ld c, a                     ; 1
    ld a, b                     ; 1

assert LOW(FontTiles) == 0
    ld h, HIGH(FontTiles)

    rlca       ; 1
    rlca       ; 1
    rlca       ; 1
    ld b, a
    and a, $F8
    ld l, a

    ld a, b
    and a, $07
    add a, h
    ld h, a

    ldh a, [DrawText_RowCount]
    dec a
    add a, l
    ld l, a

    ; Get one 8-pixel row from source tile
    ld a, [hl]
    ld b, a

    ld a, c ; Character width

    ; Get character pixels mask
assert (PixelMaskLUT & $7) == 0
    ld hl, PixelMaskLUT
    dec a       ; 1
    add a, l    ; 1
    ld l, a     ; 1
    ld a, [hl]  ; 2
    ld c, a     ; 1

    ldh a, [DrawText_PixelColumn]
    cpl         ; 1
    add a, 8    ; 2     ; a = 7 - position
    add a, a    ; 1
    add a, a    ; 1
DEF SHIFTLOOP_HEAD_SIZE EQU 3 + 1 + 1 + 1 + 1
DEF SHIFTLOOP_BODY_SIZE EQU 7 * 4
.shiftloop_base
    ld hl, @ + SHIFTLOOP_HEAD_SIZE
    add a, l
    ld l, a
    ld a, b
    jp hl
.shiftloop_body
assert HIGH(@ + SHIFTLOOP_BODY_SIZE) == HIGH(@)
REPT 7
    rrca        ; 1
    srl c       ; 2
    nop         ; 1
ENDR
.shiftloop_end
assert (.shiftloop_base + SHIFTLOOP_HEAD_SIZE) == .shiftloop_body
assert (.shiftloop_base + SHIFTLOOP_HEAD_SIZE + SHIFTLOOP_BODY_SIZE) == .shiftloop_end

    jr c, :++
    ; No carry - character does not span tiles

    ld h, d
    ld l, e
    or a, [hl]
    ld [hl], a

    ; Advance pixel column offset by character width
    ldh a, [DrawText_CharWidth]
    ld b, a
    ldh a, [DrawText_PixelColumn]  ; 3
    add a, b            ; 1
    and a, 7
    ldh [DrawText_PixelColumn], a  ; 3

    jr nz, :+
    ; Advance write address by 16 bytes (assume no carry!)
    swap e      ; 2
    inc e       ; 1
    swap e      ; 2
    xor a, a
    ld [de], a
:

    jr .col
:
    ; Carry - character spans across 2 tiles

    ld b, a
    and a, c

    ld h, d
    ld l, e
    or a, [hl]
    ld [hl], a

    ; Advance write address by 16 bytes (assume no carry!)
    swap e      ; 2
    inc e       ; 1
    swap e      ; 2

    ld a, c     ; 1
    cpl         ; 1
    ld c, a     ; 1
    ld a, b     ; 1
    and a, c    ; 1

    ld [de], a

    ; Advance pixel column offset by character width
    ldh a, [DrawText_CharWidth]
    ld b, a
    ldh a, [DrawText_PixelColumn]  ; 3
    add a, b            ; 1
    and a, 7
    ldh [DrawText_PixelColumn], a  ; 3

    jp .col
.rowdone

    ldh a, [DrawText_RowCount]
    dec a
    jp nz, .row

    ret




SECTION "LUTs", ROM0, ALIGN[3, 0]
PixelBitLUT:
    db %10000000, %01000000, %00100000, %00010000, %00001000, %00000100, %00000010, %00000001
PixelMaskLUT:
    db %10000000, %11000000, %11100000, %11110000, %11111000, %11111100, %11111110, %11111111

SECTION "RasterStartAddressTableFG", ROM0, ALIGN[8, 0]
RasterStartAddressTableFG:
FOR Y, 64
    dw FGFrameBuffer + (Y * 8)
ENDR
SECTION "RasterStartAddressTableBG", ROM0, ALIGN[8, 0]
RasterStartAddressTableBG:
FOR Y, (128 + 32)
    dw PreBGFrameBuffer + (Y * 16)
ENDR

SECTION "Font Tiles", ROMX, ALIGN[8, 0]
FontTiles:
    incbin "neosans.bin"
FontTileWidths:
    incbin "neosans_widths.bin"
DigitFontTiles:
    incbin "digits.bin"

SECTION "Text Strings", ROMX
; Auto-generated strings
TextStrings_0:
    db "Resume", 0
TextStrings_1:
    db "Check if solved", 0
TextStrings_2:
    db "Select Tangram", 0
TextStrings_3:
    db "Solve this piece", 0
TextStrings_4:
    db "Solve all pieces", 0
TextStrings_5:
    db "Quit to title screen", 0


; Hand-crafted strings
TextStrings_6:
    db "Solved!", 0
TextStrings_7:
    db "Not solved...", 0
TextStrings_8:
    db "New record!", 0
TextStrings_9:
    db "Welcome to Pocket Tangrams!", 0
TextStrings_12:
    db "(c) Edd Biddulph 2022", 0

SRAMCheckString:
    db "KNOBLAUCH", 0

PieceInitialStates:
INCLUDE "default_tangram.inc"

SECTION "Tangrams", ROMX
Tangrams:
INCLUDE "tangrams.inc"

SECTION "VertexRotations", ROMX, ALIGN[8, 0]
VertexRotations0:
    incbin "rotations0.bin"
VertexRotations1:
    incbin "rotations2.bin"
VertexRotations2:
    incbin "rotations3.bin"
VertexRotations3:
    incbin "rotations5.bin"
VertexRotations4:
    incbin "rotations6.bin"

SECTION "Checkerboard LUTs", ROMX
CheckerboardStripTiles:
    incbin "board_strip_tiles.bin"
SECTION "CheckerboardUStepTable", ROMX, ALIGN[6, 0]
CheckerboardUStepTable:
    incbin "board_u_step_table.bin"
SECTION "CheckerboardPalettetable", ROMX, ALIGN[8, 0]
CheckerboardPalettetable:
    incbin "board_palette_table.bin"
SECTION "CheckerboardSineTable", ROMX, ALIGN[8, 0]
CheckerboardSineTable:
    incbin "board_sine_table.bin"

LogoTiles:
    incbin "logo_tiledata.bin"
SECTION "LogoMapCompressed", ROM0
LogoMapCompressed:
    incbin "logo_tilemap_compressed.bin"
