; WIREFRAME

ALIGN: MACRO ?boundary
       ds ?boundary - 1 - ($ + ?boundary - 1) % ?boundary
       ENDM

ALIGN_VIRTUAL: MACRO ?boundary
       ds virtual (?boundary - 1 - ($ + ?boundary - 1) % ?boundary)
       ENDM

V9990_WAIT_FOR_COMMAND_READY: MACRO
      in a,(V9P_STATUS)
      rra
      jr c,$-3
      ENDM

V9990_WAIT_FOR_TRANSFER_READY: MACRO
      in a,(V9P_STATUS)
      rla
      jr nc,$-3
      ENDM

V9990_SET_BORDER_COLOUR: MACRO ?c
      push af
      ld a,?c
      call v9990_set_border_colour
      pop af
      ENDM

MAKE_LMMV: MACRO ?dx, ?dy, ?nx, ?ny, ?lop
      db #00,#00                ; R#32, R#33
      db #00,#00                ; R#34, R#35
      db ?dx&#FF,(?dx>>8)&7     ; R#36, R#37
      db ?dy&#FF,(?dy>>8)&15    ; R#38, R#39
      db ?nx&#FF,(?nx>>8)&7     ; R#40, R#41
      db ?ny&#FF,(?ny>>8)&15    ; R#42, R#43
      db %00000000              ; R#44
      db ?lop                   ; R#45
      db V9O_LMMV<<4            ; R#52
      ENDM

MAKE_LMMM: MACRO ?sx, ?sy, ?dx, ?dy, ?nx, ?ny, ?arg, ?lop
      db ?sx&#FF,(?sx>>8)&7     ; R#32, R#33
      db ?sy&#FF,(?sy>>8)&15    ; R#34, R#35
      db ?dx&#FF,(?dx>>8)&7     ; R#36, R#37
      db ?dy&#FF,(?dy>>8)&15    ; R#38, R#39
      db ?nx&#FF,(?nx>>8)&7     ; R#40, R#41
      db ?ny&#FF,(?ny>>8)&15    ; R#42, R#43
      db ?arg                   ; R#44
      db ?lop                   ; R#45
      db V9O_LMMM<<4            ; R#52
      ENDM

; V9990 command structure (specialised for LINEs)
COMMAND_DX:     equ  0 ; 2  R#36, R#37
COMMAND_DY:     equ  2 ; 2  R#38, R#39
COMMAND_MJ:     equ  4 ; 2  R#40, R#41
COMMAND_MI:     equ  6 ; 2  R#42, R#43
COMMAND_ARG:    equ  8 ; 1  R#44

COMMAND_SIZE:   equ  9

; BIOS ROM
CGTABL: equ #0004

; BIOS Routines
RSLREG: equ #0138
WSLREG: equ #013B
ENASLT: equ #0024

; System Variables
FORCLR: equ #F3E9
BAKCLR: equ #F3EA
BDRCLR: equ #F3EB
TXTNAM: equ #F3B3
TXTCGP: equ #F3B7
HKEY:   equ #FD9A
CLIKSW: equ #F3DB
T32NAM: equ #F3BD
T32COL: equ #F3BF
T32CGP: equ #F3C1
T32ATR: equ #F3C3
T32PAT: equ #F3C5
GRPNAM: equ #F3C7
GRPCOL: equ #F3C9
GRPCGP: equ #F3CB
GRPATR: equ #F3CD
GRPPAT: equ #F3CF

; VRAM
PATTBL: equ #0000
NAMTBL: equ #0800
SATRBL: equ #1400
COLTBL: equ #1480
SPATBL: equ #1800

; Ports
VDPDAT: equ #98 ; Port #0
VDPCTL: equ #99 ; Port #1
VDPPAL: equ #9A ; Port #2
VDPREG: equ #9B ; Port #3
SLOTSL: equ #A8
KEYROW: equ #A9
KEYSEL: equ #AA
PSGREG: equ #A0
PSGWRI: equ #A1
PSGREA: equ #A2
OPLLRG: equ #7C
OPLLDT: equ #7D

; PSG Registers
PSG_REG_TONE_0: equ 0
PSG_REG_TONE_1: equ 1
PSG_REG_TONE_2: equ 2
PSG_REG_TONE_3: equ 3
PSG_REG_TONE_4: equ 4
PSG_REG_TONE_5: equ 5
PSG_REG_NOISE:  equ 6
PSG_REG_MIXER:  equ 7 ; Important note: bit 6 must be 0, and bit 7 must be 1.
PSG_REG_AMP_0:  equ 8
PSG_REG_AMP_1:  equ 9
PSG_REG_AMP_2:  equ 10
PSG_REG_ENV_0:  equ 11
PSG_REG_ENV_1:  equ 12
PSG_REG_ENV_2:  equ 13

; V9990 Ports
V9P_VRAMDAT: equ #60 ; VRAM Data
V9P_PALDATA: equ #61 ; Palette Data
V9P_COMDATA: equ #62 ; Command Data
V9P_REGDATA: equ #63 ; Register Data
V9P_REGSLCT: equ #64 ; Register Select
V9P_STATUS:  equ #65 ; Status
V9P_IFLAGS:  equ #66 ; Interrupt Flags
V9P_SYSCTRL: equ #67 ; System Control

; V9990 Registers
V9R_VRAMWR0: equ  0  ; VRAM Write Address
V9R_VRAMWR1: equ  1
V9R_VRAMWR2: equ  2
V9R_VRAMRD0: equ  3  ; VRAM Read Address
V9R_VRAMRD1: equ  4
V9R_VRAMRD2: equ  5
V9R_SCNMOD0: equ  6  ; Screen Mode
V9R_SCNMOD1: equ  7
V9R_CONTROL: equ  8  ; Control
V9R_INTRPT0: equ  9  ; Interrupt
V9R_INTRPT1: equ 10
V9R_INTRPT2: equ 11
V9R_INTRPT3: equ 12
V9R_PALCTRL: equ 13  ; Palette Control
V9R_PALPNTR: equ 14  ; Palette Pointer
V9R_BDRPCOL: equ 15  ; Backdrop Colour
V9R_DISPADJ: equ 16  ; Display Adjust
V9R_SCROLL0: equ 17  ; Scroll Control
V9R_SCROLL1: equ 18
V9R_SCROLL2: equ 19
V9R_SCROLL3: equ 10
V9R_SCROLL4: equ 21
V9R_SCROLL5: equ 22
V9R_SCROLL6: equ 23
V9R_SCROLL7: equ 24
V9R_SPRBASE: equ 25  ; Sprite Generator Base Address
V9R_LCDCTRL: equ 26  ; LCD Control
V9R_PRICTRL: equ 27  ; Priority Control
V9R_CURPOFS: equ 28  ; Cursor Sprite Palette Offset
V9R_CMNDWR0: equ 32  ; Command (Write)
V9R_CMDLOGT: equ 45  ; Logic + TP
V9R_FONTCOL: equ 48  ; Font Colour
V9R_CMNDOPC: equ 52  ; Command Operation Code
V9R_CMNDRD0: equ 53  ; Command (Read)

; V9990 Opcodes
V9O_STOP:     equ  0
V9O_LMMC:     equ  1
V9O_LMMV:     equ  2
V9O_LMCM:     equ  3
V9O_LMMM:     equ  4
V9O_CMMC:     equ  5
V9O_CMMM:     equ  7
V9O_BMXL:     equ  8
V9O_BMLX:     equ  9
V9O_BMLL:     equ 10
V9O_LINE:     equ 11
V9O_SEARCH:   equ 12
V9O_POINT:    equ 13
V9O_PSET:     equ 14
V9O_ADVANCE:  equ 15

; Constants

  include "constants.asm"

FRAME_WIDTH:          equ 256 ; B2 mode
FRAME_HEIGHT:         equ 212
FRAME_BPP:            equ 2
IMAGE_SPACE_W:        equ 1024
IMAGE_SPACE_H:        equ 2048
MAX_COMMANDS:         equ 128

ROTATION_UNITS:       equ 1024
SCALE_ANIM_FRAMES:    equ 64

BACKGROUND_COLOUR:    equ %10101010
BORDER_COLOUR:        equ 12

ENABLE_STAT_COUNTERS: equ 0

MODE_FADING_IN:       equ 0
MODE_AWAITING_INPUT:  equ 1
MODE_FADING_OUT:      equ 2

  ; 32Kbyte ROM
  org #0000

  ds 16
  jp $

  ; Interrupt Service Routine
  ds #0038 - $
  push af
  in a,(V9P_IFLAGS)
  rrca
  jr c,handle_V_interrupt
  rrca
  jr c,handle_H_interrupt ; Must come before CE to ensure a close-to-horizon split.
  rrca
  jr c,handle_CE_interrupt
  pop af
  ei
  ret

handle_CE_interrupt:
  ld a,%00000100
  out (V9P_IFLAGS),a
  push hl
  ;V9990_SET_BORDER_COLOUR BORDER_COLOUR+4
  ; Check to see if the commandlist is empty
  ld hl,(commandlist_read_length_ptr)
  xor a
  cp (hl)
  jr z,handle_CE_interrupt.empty
  ; Execute a command from the commandlist
  push bc
  dec (hl)
  ld hl,(commandlist_read_ptr)
  ; Put the read head at the start address of the next command
  ld bc,-COMMAND_SIZE
  add hl,bc
  ld (commandlist_read_ptr),hl
  ; Execute a command
  ld a,V9R_CMNDWR0+4
  out (V9P_REGSLCT),a
  ld c,V9P_REGDATA
  REPT COMMAND_SIZE
    outi
  ENDM
  ld a,%00001111
  out (c),a ; R#45, LOP
  ld a,V9R_CMNDOPC
  out (V9P_REGSLCT),a
  ld a,V9O_LINE<<4
  out (c),a ; R#52, OP-CODE
  pop bc
  pop hl
  pop af
  ei
  ret
handle_CE_interrupt.empty:
  ld a,1
  ld (last_command_done_flag),a
  ;V9990_SET_BORDER_COLOUR BORDER_COLOUR+0
  pop hl
  pop af
  ei
  ret

handle_H_interrupt:
  ld a,%00000010
  out (V9P_IFLAGS),a
  pop af
  ei
  ret

handle_V_interrupt:
  ld a,%00000001
  out (V9P_IFLAGS),a
  ; Increment the frameskip count
  ld a,(frames_to_skip)
  inc a
  ld (frames_to_skip),a

  ; Check to see if the writing commandlist if full yet or not
  ld a,(commandlist_filled_flag)
  and a
  jp z,handle_V_interrupt.return
  ; It is full.

  ; Check to see if the reading commandlist is finished yet or not
  ld a,(last_command_done_flag)
  and a
  jp z,handle_V_interrupt.return
  ; It is empty.

  IF 0
  ; Horrible WebMSX
  in a,(V9P_STATUS)
  rra
  jp c,handle_V_interrupt.return
  push hl
  ld hl,(commandlist_read_length_ptr)
  xor a
  cp (hl)
  pop hl
  jp nz,handle_V_interrupt.return
  ENDIF

  ; Both conditions hold (all pending commands have executed and there is
  ; a complete commandlist ready to begin execution).

  ;V9990_SET_BORDER_COLOUR BORDER_COLOUR+1

  ; So do a swap and update the game logic.
  push hl
  push bc
  push ix

  ; Swap the commandlists
  push hl
  push de
  ; Swap the read/write heads.
  ld de,(commandlist_read_ptr)
  ld hl,(commandlist_write_ptr)
  ld (commandlist_write_ptr),de
  ld (commandlist_read_ptr),hl
  ld de,(commandlist_read_length_ptr)
  ld hl,(commandlist_write_length_ptr)
  ld (commandlist_write_length_ptr),de
  ld (commandlist_read_length_ptr),hl
  ; "Swap" the framebuffer. There is no need to disable interrupts here,
  ; because there is a guarantee that no commands are still being executed.
  ; Set Scroll Registers
  ld hl,buffer_swap
  ld a,V9R_SCROLL1
  out (V9P_REGSLCT),a
  ld a,(hl)
  xor 1
  out (V9P_REGDATA),a ; R#18 SCAY (bit 12-8)
  ld (hl),a
  pop de
  pop hl

  ; Start a command to clear the framebuffer
  ld ix,clear_framebuffer_lmmv_ram
  ld a,(buffer_swap)
  xor 1
  ld (ix+COMMAND_DY+4+1),a
  ld hl,clear_framebuffer_lmmv_ram
  ld a,V9R_CMNDWR0
  out (V9P_REGSLCT),a
  ld c,V9P_REGDATA
  REPT 14
    outi
  ENDM
  ld a,V9R_CMNDOPC
  out (V9P_REGSLCT),a
  outi

  ld a,(frames_to_skip)
  ld b,a
frameskip_loop:
  push bc

  ; Handle inputs
  in a,(KEYSEL)
  and #F0
  or 8 ; Select row 8
  out (KEYSEL),a
  in a,(KEYROW)
  bit 7,a ; Right
  call z,right_pressed
  bit 6,a ; Down
  call z,down_pressed
  bit 5,a ; Up
  call z,up_pressed
  bit 4,a ; Left
  call z,left_pressed

  ; Run one frame of logic
  ld hl,(theta)
  ld bc,(theta_delta)
  add hl,bc
  ld a,h
  and (ROTATION_UNITS>>8)-1
  ld h,a
  ld (theta),hl

  ld hl,(phi)
  ld bc,(phi_delta)
  add hl,bc
  ld a,h
  and (ROTATION_UNITS>>8)-1
  ld h,a
  ld (phi),hl

  ld a,(demo_mode)
  cp MODE_FADING_IN
  jr z,frameskip.fade_in
  cp MODE_AWAITING_INPUT
  jr z,frameskip.awaiting_input
  cp MODE_FADING_OUT
  jr z,frameskip.fade_out

frameskip.fade_in:
  ; Fade in
  ld hl,fade_frame_count
  inc (hl)
  ld a,SCALE_ANIM_FRAMES
  cp (hl)
  jr nz,$+2+2+2+3
  ld (hl),SCALE_ANIM_FRAMES-1
  ld a,MODE_AWAITING_INPUT  ; 2B
  ld (demo_mode),a          ; 3B
  jr frameskip_iter

frameskip.awaiting_input:
  ; Handle model selection
  in a,(KEYSEL)
  and #F0
  or 0 ; Select row 0
  out (KEYSEL),a
  in a,(KEYROW)
  bit 1,a ; 1
  call z,preselect_model_house
  bit 2,a ; 2
  call z,preselect_model_doe
  bit 3,a ; 3
  call z,preselect_model_cone
  bit 4,a ; 4
  call z,preselect_model_two
  jr frameskip_iter

frameskip.fade_out:
  ; Fade out
  ld hl,fade_frame_count
  dec (hl)
  ld a,-1
  cp (hl)
  jr nz,$+2+2+2+3+3+1
  ld (hl),0
  ld a,MODE_FADING_IN       ; 2B
  ld (demo_mode),a          ; 3B
  ld hl,(select_model_routine_addr) ; 3B
  jp (hl)                           ; 1B

frameskip_iter:
  pop bc
  dec b
  jp nz,frameskip_loop

  xor a
  ld (frames_to_skip),a

  pop ix
  pop bc
  pop hl

  ;V9990_SET_BORDER_COLOUR BORDER_COLOUR+2

  ; Reset the flags
  xor a
  ld (last_command_done_flag),a
  ld (commandlist_filled_flag),a
handle_V_interrupt.return:
  pop af
  ei
  ret

preselect_model_house:
  ld hl,select_model_house
  ld (select_model_routine_addr),hl
  ld hl,demo_mode
  ld (hl),MODE_FADING_OUT
  ret
select_model_house:
  ld hl,house_logsign_vertices
  ld (logsign_vertices_addr),hl
  ld hl,house_edges
  ld (edges_addr),hl
  ld a,10
  ld (num_vertices),a
  ld a,15
  ld (num_edges),a
  jp frameskip_iter

preselect_model_doe:
  ld hl,select_model_doe
  ld (select_model_routine_addr),hl
  ld hl,demo_mode
  ld (hl),MODE_FADING_OUT
  ret
select_model_doe:
  ld hl,doe_logsign_vertices
  ld (logsign_vertices_addr),hl
  ld hl,doe_edges
  ld (edges_addr),hl
  ld a,20
  ld (num_vertices),a
  ld a,30
  ld (num_edges),a
  jp frameskip_iter

preselect_model_cone:
  ld hl,select_model_cone
  ld (select_model_routine_addr),hl
  ld hl,demo_mode
  ld (hl),MODE_FADING_OUT
  ret
select_model_cone:
  ld hl,cone_logsign_vertices
  ld (logsign_vertices_addr),hl
  ld hl,cone_edges
  ld (edges_addr),hl
  ld a,22
  ld (num_vertices),a
  ld a,60
  ld (num_edges),a
  jp frameskip_iter

preselect_model_two:
  ld hl,select_model_two
  ld (select_model_routine_addr),hl
  ld hl,demo_mode
  ld (hl),MODE_FADING_OUT
  ret
select_model_two:
  ld hl,two_logsign_vertices
  ld (logsign_vertices_addr),hl
  ld hl,two_edges
  ld (edges_addr),hl
  ld a,30
  ld (num_vertices),a
  ld a,53
  ld (num_edges),a
  jp frameskip_iter



top:
  ; Initialise working variables RAM
  ld hl,RAM_START
  ld (hl),0
  ld de,RAM_START+1
  ld bc,RAM_END-RAM_START-1
  ldir

  ; Set up the V9990
  call v9990_reset
  ld b,1
  call v9990_setmode_B1_BP2_1024x2048
  call v9990_set_512k

  ld a,V9R_CONTROL
  out (V9P_REGSLCT),a
  ld a,%01000010
  out (V9P_REGDATA),a

  ; Set some common parameters

  V9990_WAIT_FOR_COMMAND_READY

  ; Set DIY+DIX
  ld a,44
  out (V9P_REGSLCT),a
  ld a,%00000000
  out (V9P_REGDATA),a ; R#44

  ; Set Write Mask
  ld a,46
  out (V9P_REGSLCT),a
  ld a,%11111111
  out (V9P_REGDATA),a ; R#46
  out (V9P_REGDATA),a ; R#47

  ; Set FC - this value will stay constant throughout.
  ld a,V9R_FONTCOL
  out (V9P_REGSLCT),a
  ld a,%11111111
  out (V9P_REGDATA),a ; R#48
  out (V9P_REGDATA),a ; R#49


  ; Clear all of VRAM
  V9990_WAIT_FOR_COMMAND_READY
  ld a,V9R_CMNDWR0+4
  out (V9P_REGSLCT),a
  ; Set DX
  xor a
  out (V9P_REGDATA),a ; R#36
  out (V9P_REGDATA),a ; R#37
  ; Set DY
  out (V9P_REGDATA),a ; R#38
  out (V9P_REGDATA),a ; R#39
  ; Set NX
  out (V9P_REGDATA),a ; R#40
  ld a,IMAGE_SPACE_W>>8
  out (V9P_REGDATA),a ; R#41
  ; Set NY
  xor a
  out (V9P_REGDATA),a ; R#42
  ld a,IMAGE_SPACE_H>>8
  out (V9P_REGDATA),a ; R#43

  ; Set Logical Operation, for clearing to zero.
  ld a,V9R_CMDLOGT
  out (V9P_REGSLCT),a
  ld a,%00000000
  out (V9P_REGDATA),a ; R#45

  ; Set OP
  ld a,V9R_CMNDOPC
  out (V9P_REGSLCT),a
  ld a,V9O_LMMV<<4
  out (V9P_REGDATA),a

  V9990_WAIT_FOR_COMMAND_READY

  ; ----------------------------------------------------

  ; Clear the commandlists
  ld hl,commandlists_start
  ld (hl),0
  ld de,commandlists_start+1
  ld bc,commandlists_end-commandlists_start-1
  ldir

  ; Initialise the commandlist pointers.
  ld hl,commandlist_1+1
  ld (commandlist_read_ptr),hl
  ld hl,commandlist_1+0
  ld (commandlist_read_length_ptr),hl
  ld (hl),0

  ld hl,commandlist_0+1
  ld (commandlist_write_ptr),hl
  ld hl,commandlist_0+0
  ld (commandlist_write_length_ptr),hl
  ld (hl),0

  ; Add 256 to DY in all of the commands in one commandlist.
  ld b,MAX_COMMANDS
  ld ix,commandlist_1+1
  ld de,COMMAND_SIZE
init_commandlists.loop:
  ld (ix+COMMAND_DY+1),1
  add ix,de
  djnz init_commandlists.loop

  xor a
  ld (buffer_swap),a

  ; ----------------------------------------------------

  ; Reset Scroll Registers
  ld a,V9R_SCROLL0
  out (V9P_REGSLCT),a
  ld a,#00
  out (V9P_REGDATA),a ; R#17
  ld a,#00
  out (V9P_REGDATA),a ; R#18
  ld a,#00
  out (V9P_REGDATA),a ; R#19
  ld a,#00
  out (V9P_REGDATA),a ; R#20
  ld a,#00
  out (V9P_REGDATA),a ; R#21
  ld a,#00
  out (V9P_REGDATA),a ; R#22
  ld a,#00
  out (V9P_REGDATA),a ; R#23
  ld a,#00
  out (V9P_REGDATA),a ; R#24

  ; Wait for all commands to be completed
  V9990_WAIT_FOR_COMMAND_READY

  ; Fill the palette
  ld a,V9R_PALPNTR
  out (V9P_REGSLCT),a
  xor a
  out (V9P_REGDATA),a
  ld c,V9P_PALDATA
  ld b,palette_rgb_end-palette_rgb_start
  ld hl,palette_rgb_start
  otir

  ;V9990_SET_BORDER_COLOUR BORDER_COLOUR

  ; Set interrupt enables
  ld a,V9R_INTRPT0
  out (V9P_REGSLCT),a
  ld a,%00000101 ; IECE, IEH, IEV
  out (V9P_REGDATA),a

  ld hl,clear_framebuffer_lmmv
  ld de,clear_framebuffer_lmmv_ram
  ld bc,15
  ldir

  ; Pre-select the house model
  ld hl,house_logsign_vertices
  ld (logsign_vertices_addr),hl
  ld hl,house_edges
  ld (edges_addr),hl
  ld a,10
  ld (num_vertices),a
  ld a,15
  ld (num_edges),a

  ; Enable the display
  call v9990_enable_display

  ld a,1
  ld (last_command_done_flag),a

  ei

mainloop:

  IF ENABLE_STAT_COUNTERS
    ld hl,stat_counters_start
    ld (hl),0
    ld de,stat_counters_start+1
    ld bc,stat_counters_end-stat_counters_start-1
    ldir
  ENDIF


  ; Camera constants

  ; log(cos(theta))
  ld bc,logcos_table
  ld hl,(theta)
  add hl,hl
  add hl,bc
  ld a,(hl)
  ld (log_cos_theta+0),a
  inc hl
  ld a,(hl)
  ld (log_cos_theta+1),a

  ; sign(cos(theta))
  ld bc,signcos_table
  ld hl,(theta)
  add hl,bc
  ld a,(hl)
  ld (sign_cos_theta),a

  ; log(sin(theta))
  ld bc,logsin_table
  ld hl,(theta)
  add hl,hl
  add hl,bc
  ld a,(hl)
  ld (log_sin_theta+0),a
  inc hl
  ld a,(hl)
  ld (log_sin_theta+1),a

  ; sign(sin(theta))
  ld bc,signsin_table
  ld hl,(theta)
  add hl,bc
  ld a,(hl)
  ld (sign_sin_theta),a



  ; log(cos(phi))
  ld bc,logcos_table
  ld hl,(phi)
  add hl,hl
  add hl,bc
  ld a,(hl)
  ld (log_cos_phi+0),a
  inc hl
  ld a,(hl)
  ld (log_cos_phi+1),a

  ; sign(cos(phi))
  ld bc,signcos_table
  ld hl,(phi)
  add hl,bc
  ld a,(hl)
  ld (sign_cos_phi),a

  ; log(sin(phi))
  ld bc,logsin_table
  ld hl,(phi)
  add hl,hl
  add hl,bc
  ld a,(hl)
  ld (log_sin_phi+0),a
  inc hl
  ld a,(hl)
  ld (log_sin_phi+1),a

  ; sign(sin(phi))
  ld bc,signsin_table
  ld hl,(phi)
  add hl,bc
  ld a,(hl)
  ld (sign_sin_phi),a


  ; sin_theta_sin_phi
  ld hl,(log_sin_theta)
  ld bc,(log_sin_phi)
  add hl,bc
  ld (log_sin_theta_sin_phi),hl
  ld a,(sign_sin_theta)
  ld b,a
  ld a,(sign_sin_phi)
  xor b
  ld (sign_sin_theta_sin_phi),a

  ; cos_theta_sin_phi
  ld hl,(log_cos_theta)
  ld bc,(log_sin_phi)
  add hl,bc
  ld (log_cos_theta_sin_phi),hl
  ld a,(sign_cos_theta)
  ld b,a
  ld a,(sign_sin_phi)
  xor b
  ld (sign_cos_theta_sin_phi),a

  ; sin_theta_cos_phi
  ld hl,(log_sin_theta)
  ld bc,(log_cos_phi)
  add hl,bc
  ld (log_sin_theta_cos_phi),hl
  ld a,(sign_sin_theta)
  ld b,a
  ld a,(sign_cos_phi)
  xor b
  ld (sign_sin_theta_cos_phi),a

  ; cos_theta_cos_phi
  ld hl,(log_cos_theta)
  ld bc,(log_cos_phi)
  add hl,bc
  ld (log_cos_theta_cos_phi),hl
  ld a,(sign_cos_theta)
  ld b,a
  ld a,(sign_cos_phi)
  xor b
  ld (sign_cos_theta_cos_phi),a

  ; Scaling animation constants
  ld bc,logscale_table
  ld a,(fade_frame_count)
  add a,a
  ld l,a
  ld h,0
  add hl,bc
  ld a,(hl)
  ld (log_animscale+0),a
  inc hl
  ld a,(hl)
  ld (log_animscale+1),a

  ;V9990_SET_BORDER_COLOUR BORDER_COLOUR+3
  call transform_mesh
  ;V9990_SET_BORDER_COLOUR BORDER_COLOUR+0

  ; Draw the edges
  ld iy,(commandlist_write_ptr)
  ld a,(num_edges)
  ld b,a
  ld hl,(edges_addr)
edges_loop:
  push bc
  ld b,(hl)
  inc hl
  ld c,(hl)
  inc hl
  push hl

  ; Calculate address of second vertex
  ld l,c
  ld h,0
  add hl,hl
  ld de,projected_vertices
  add hl,de
  ex de,hl

  ; Calculate address of first vertex
  ld l,b
  ld h,0
  add hl,hl
  ld bc,projected_vertices
  add hl,bc

  ; Construct a line command for this edge
  call construct_line_command

  ld bc,COMMAND_SIZE
  add iy,bc
  ld hl,(commandlist_write_length_ptr)
  inc (hl)

  pop hl
  pop bc
  djnz edges_loop
  ld (commandlist_write_ptr),iy

  ld hl,commandlist_filled_flag
  ld (hl),1
  xor a

  ;V9990_SET_BORDER_COLOUR BORDER_COLOUR+3

  cp (hl)
  jr nz,$-1

  jp mainloop

; ---------------------- Input Handling and Dynamics ----------------------

up_pressed:
  push af
  ld hl,(phi_delta)
  dec hl
  ld (phi_delta),hl
  pop af
  ret

down_pressed:
  push af
  ld hl,(phi_delta)
  inc hl
  ld (phi_delta),hl
  pop af
  ret

left_pressed:
  push af
  ld hl,(theta_delta)
  dec hl
  ld (theta_delta),hl
  pop af
  ret

right_pressed:
  push af
  ld hl,(theta_delta)
  inc hl
  ld (theta_delta),hl
  pop af
  ret

; ---------------------- V9990 Common ----------------------

v9990_setmode_B1_BP2_1024x2048:
  push af
  ;     |    Register #6    |          Register #7        |
  ; MCS DSPM DCKM XIMM CLRM    C25M SM1 SM PAL EO IL HSCN
  ;   0   10   00   10   00   0   0   0  0   0  0  0    0
  ld a,#00 ; MCS = 0
  out (V9P_SYSCTRL),a
  ld a,V9R_SCNMOD0
  out (V9P_REGSLCT),a
  ld a,%10001000
  out (V9P_REGDATA),a
  ld a,%00000000
  bit 0,b ; Shift
  jr nz,$+2+2
  or %00000110 ; Interlace
  out (V9P_REGDATA),a
  ; Palette Control
  ld a,V9R_PALCTRL
  out (V9P_REGSLCT),a
  ld a,%00000000
  out (V9P_REGDATA),a
  pop af
  ret

v9990_enable_display:
  ld a,V9R_CONTROL
  out (V9P_REGSLCT),a
  ld a,%11000010
  out (V9P_REGDATA),a
  ret

v9990_set_512k:
  ld a,V9R_CONTROL | %11000000
  out (V9P_REGSLCT),a
  in a,(V9P_REGDATA)
  and %11111000
  or  %00000010
  out (V9P_REGDATA),a
  ret

v9990_reset:
  ld a,#02
  out (V9P_SYSCTRL),a
  xor a
  out (V9P_SYSCTRL),a
  ld a,V9R_SCNMOD0
  out (V9P_REGSLCT),a
  ld a,%11000000 ; Stand-by mode
  out (V9P_REGDATA),a
  ret

v9990_set_border_colour:
  push af
  push bc
  ld b,a
  ld a,V9R_BDRPCOL
  out (V9P_REGSLCT),a
  ld a,b
  out (V9P_REGDATA),a
  pop af
  pop bc
  ret

; ---------------------- 3D Objects ----------------------

LOAD_COORDINATE: MACRO
      ld hl,(vertices_addr) ; 17c
      ld e,(hl)
      inc hl
      ld d,(hl)
      inc hl
      ld a,(hl) ; Sign
      inc hl
      ld (vertices_addr),hl
      ld b,a    ; 5c
      ex af,af' ; 5c
      ld a,b    ; 5c
      ex af,af' ; 5c
      ENDM

ACCUMULATE: MACRO ?log_trig, ?sign_trig, ?accum, ?neg
      ld l,e    ; 5c
      ld h,d    ; 5c
      ld bc,(?log_trig)
      and a
      sbc hl,bc
      ; Clamp to zero
      bit 7,h
      jr z,$+2+3
      ld hl,#0000
      ; HL ϵ [0, +4096) = max(0, log(abs(x)) - log(cos(theta)))
      add hl,hl
      ld bc,exp_table
      add hl,bc
      ld b,(hl)
      inc hl
      ld h,(hl)
      ld l,b
      ; HL ϵ [0, +4096) = abs(x) * abs(cos(theta))
      ; Sign of x is already stored in A
      ld b,a
      ld a,(?sign_trig)
      xor b
      jr z,$+2+1+1+3+2
      ; Negate HL
      ld c,l        ;  5c 1B
      ld b,h        ;  5c 1B
      ld hl,#0000   ; 11c 3B
      sbc hl,bc     ; 17c 2B
      ; HL ϵ (-4096, +4096)
      IF ?neg=0
        ld bc,(?accum)
        add hl,bc
        ld (?accum),hl
      ELSE
        ld c,l
        ld b,h
        ld hl,(?accum)
        and a
        sbc hl,bc
        ld (?accum),hl
      ENDIF
      ENDM



transform_vertex:

  ld hl,#0000
  ld (x_accum),hl
  ld (y_accum),hl
  ld (z_accum),hl

  ; X

  ; Load log(abs(x) from the vertex list
  LOAD_COORDINATE
  ; DE ϵ [0, +4096) = log(abs(x))

  ; Apply uniform scale factor to X
  ld hl,(log_animscale)
  ex de,hl
  and a
  sbc hl,de
  ; Clamp to zero
  bit 7,h
  jr z,$+2+3
  ld hl,#0000
  ex de,hl

  ; x_accum += +exp_table[MAX(0, log_x - log_cos_theta)] * sign_cos_theta * sign(x);
  ACCUMULATE log_cos_theta, sign_cos_theta, x_accum, 0

  ; Restore sign(x)
  ex af,af' ; 5c
  ld b,a    ; 5c
  ex af,af' ; 5c
  ld a,b    ; 5c

  ; z_accum += -exp_table[MAX(0, log_x - log_sin_theta)] * sign_sin_theta * sign(x);
  ACCUMULATE log_sin_theta, sign_sin_theta, z_accum, 1


  ; Y

  ; Load log(abs(y) from the vertex list
  LOAD_COORDINATE
  ; DE ϵ [0, +4096) = log(abs(y))

  ; Apply uniform scale factor to X
  ld hl,(log_animscale)
  ex de,hl
  and a
  sbc hl,de
  ; Clamp to zero
  bit 7,h
  jr z,$+2+3
  ld hl,#0000
  ex de,hl

  ; x_accum += -exp_table[MAX(0, log_y - log_sin_theta_sin_phi)] * sign_sin_theta_sin_phi * sign(y);
  ACCUMULATE log_sin_theta_sin_phi, sign_sin_theta_sin_phi, x_accum, 1

  ; Restore sign(x)
  ex af,af' ; 5c
  ld b,a    ; 5c
  ex af,af' ; 5c
  ld a,b    ; 5c

  ; y_accum += +exp_table[MAX(0, log_y - log_cos_phi)] * sign_cos_phi * sign(y);
  ACCUMULATE log_cos_phi, sign_cos_phi, y_accum, 0

  ; Restore sign(x)
  ex af,af' ; 5c
  ld b,a    ; 5c
  ex af,af' ; 5c
  ld a,b    ; 5c

  ; z_accum += -exp_table[MAX(0, log_y - log_cos_theta_sin_phi)] * sign_cos_theta_sin_phi * sign(y);
  ACCUMULATE log_cos_theta_sin_phi, sign_cos_theta_sin_phi, z_accum, 1


  ; Z

  ; Load log(abs(z) from the vertex list
  LOAD_COORDINATE
  ; DE ϵ [0, +4096) = log(abs(z))

  ; Apply uniform scale factor to X
  ld hl,(log_animscale)
  ex de,hl
  and a
  sbc hl,de
  ; Clamp to zero
  bit 7,h
  jr z,$+2+3
  ld hl,#0000
  ex de,hl

  ; x_accum += +exp_table[MAX(0, log_z - log_sin_theta_cos_phi)] * sign_sin_theta_cos_phi * sign(z);
  ACCUMULATE log_sin_theta_cos_phi, sign_sin_theta_cos_phi, x_accum, 0

  ; Restore sign(x)
  ex af,af' ; 5c
  ld b,a    ; 5c
  ex af,af' ; 5c
  ld a,b    ; 5c

  ; y_accum += +exp_table[MAX(0, log_z - log_sin_phi)] * sign_sin_phi * sign(z);
  ACCUMULATE log_sin_phi, sign_sin_phi, y_accum, 0

  ; Restore sign(x)
  ex af,af' ; 5c
  ld b,a    ; 5c
  ex af,af' ; 5c
  ld a,b    ; 5c

  ; z_accum += +exp_table[MAX(0, log_z - log_cos_theta_cos_phi)] * sign_cos_theta_cos_phi * sign(z);
  ACCUMULATE log_cos_theta_cos_phi, sign_cos_theta_cos_phi, z_accum, 0


  ; Projection

  ; z_accum += 4096 * 2;
  ld bc,4096*2
  ld hl,(z_accum)
  add hl,bc
  ; log_table[((z_accum)+2)>>2]
  inc hl
  srl h
  rr l
  ld a,l
  and #FE
  ld l,a
  ld bc,log_table
  add hl,bc
  ld e,(hl)
  inc hl
  ld d,(hl)
  ; DE ϵ [0, +4096) = log(abs(z_accum))

  ; log_table[abs(x_accum)]
  ld hl,(x_accum)
  ; abs(x_accum)
  bit 7,h
  jr z,$+2+1+1+3+2
  ; Negate HL
  ld c,l        ;  5c 1B
  ld b,h        ;  5c 1B
  ld hl,#0000   ; 11c 3B
  sbc hl,bc     ; 17c 2B
  ld bc,log_table
  add hl,hl
  add hl,bc
  ld a,(hl)
  inc hl
  ld h,(hl)
  ld l,a

  ; int16_t proj_x = exp_table[MAX(0, log_table[abs(x_accum)] - log_table[((z_accum)+2)>>2] + 2500)] * sign(x_accum);

  ; + 2500
  ld bc,2500
  add hl,bc

  and a
  sbc hl,de
  ; Clamp to zero
  bit 7,h
  jr z,$+2+3
  ld hl,#0000

  ld bc,exp_table
  add hl,hl
  add hl,bc
  ld c,(hl)
  inc hl
  ld b,(hl)

  ; Apply sign for X
  ld a,(x_accum+1)
  bit 7,a
  ld l,c
  ld h,b
  jr nz,$+2+3+2
  ; Negate HL
  ld hl,#0000   ; 11c 3B
  sbc hl,bc     ; 17c 2B

  ; projected_vertices[i][0] = (proj_x + 1) / 2 + VIRTUAL_FRAME_WIDTH / 2;
  inc hl
  sra h
  rr l
  ld bc,FRAME_WIDTH/2
  add hl,bc
  ld bc,(projected_vertices_addr)
  ld a,l ; Result of projection is ϵ [0, 256)
  ld (bc),a
  inc bc
  ld (projected_vertices_addr),bc

  ; log_table[abs(y_accum)]
  ld hl,(y_accum)
  ; abs(y_accum)
  bit 7,h
  jr z,$+2+1+1+3+2
  ; Negate HL
  ld c,l        ;  5c 1B
  ld b,h        ;  5c 1B
  ld hl,#0000   ; 11c 3B
  sbc hl,bc     ; 17c 2B
  ld bc,log_table
  add hl,hl
  add hl,bc
  ld a,(hl)
  inc hl
  ld h,(hl)
  ld l,a

  ; int16_t proj_y = exp_table[MAX(0, log_table[abs(y_accum)] - log_table[((z_accum)+2)>>2] + 2500)] * sign(y_accum);

  ; + 2500
  ld bc,2500
  add hl,bc

  and a
  sbc hl,de
  ; Clamp to zero
  bit 7,h
  jr z,$+2+3
  ld hl,#0000

  ld bc,exp_table
  add hl,hl
  add hl,bc
  ld c,(hl)
  inc hl
  ld b,(hl)

  ; Apply sign for Y
  ld a,(y_accum+1)
  bit 7,a
  ld l,c
  ld h,b
  jr nz,$+2+3+2
  ; Negate HL
  ld hl,#0000   ; 11c 3B
  sbc hl,bc     ; 17c 2B

  ; projected_vertices[i][1] = (proj_y + 1) / 2 + VIRTUAL_FRAME_HEIGHT / 2;
  inc hl
  sra h
  rr l
  ld bc,FRAME_HEIGHT/2
  add hl,bc
  ld bc,(projected_vertices_addr)
  ld a,l ; Result of projection is ϵ [0, 256)
  ld (bc),a
  inc bc
  ld (projected_vertices_addr),bc

  ret

transform_mesh:
  ld hl,projected_vertices
  ld (projected_vertices_addr),hl
  ld hl,(logsign_vertices_addr)
  ld (vertices_addr),hl

  ld a,(num_vertices)
  ld b,a
transform_mesh.loop:
  push bc
  call transform_vertex
  pop bc
  djnz transform_mesh.loop

  ret

construct_line_command:
  ; X delta -> B
  ld a,(hl)
  ld b,a
  ld a,(de)
  sub b
  ld b,a
  ; Y delta -> C
  inc hl
  ld a,(hl)
  ld c,a
  inc de
  ld a,(de)
  sub c
  ld c,a

  ; Determine octant for line direction

  ; ARG
  ld a,c
  rlca
  ld e,b
  rl e
  rla
  rla
  and %00000110 ; DIY+DIX
  ld e,a

  ; Get absolute value of X delta
  bit 7,b
  jr z,$+2+1+2+1
  ld a,b
  neg
  ld b,a
  ; Get absolute value of Y delta
  bit 7,c
  jr z,$+2+1+2+1
  ld a,c
  neg
  ld c,a

  ld a,c
  sub b
  ld (iy+COMMAND_MJ),b  ; Maj
  ld (iy+COMMAND_MI),c  ; Min
  jr c,$+2+3+3
  ld (iy+COMMAND_MJ),c  ; Maj
  ld (iy+COMMAND_MI),b  ; Min

  ; Rotate in the MAJ bit
  ; X is the longer side and Y the shorter side at "0" while
  ; Y is the longer side and X the shorter side at "1".
  ld a,e
  rla
  xor 1
  ld (iy+COMMAND_ARG),a

  ; XY start
  ld a,(hl)
  ld (iy+COMMAND_DY+0),a
  dec hl
  ld a,(hl)
  ld (iy+COMMAND_DX+0),a

  ret

; ---------------------- Misc. lookup tables ----------------------

log_table:
  incbin "log_table.bin"
logcos_table:
  incbin "logcos_table.bin"
signcos_table:
  incbin "signcos_table.bin"
logscale_table:
  incbin "logscale_table.bin"

  ds #4000 - $ ; Referred to page 0, i.e. address #0000

; ---------------------- MSX ROM header for BIOS ----------------------

  org #4000

  db "AB"
  dw INIT
  db 0,0,0,0,0,0,0,0,0,0,0,0

INIT:
  di
  im 1 ; Interrupt mode 1 - RST #38
  ld sp,#F000

  ; Turn off the TMS9918 / V9938 so that we don't get useless interrupts from it.
  ld a,%00000000
  out (VDPCTL),a
  ld a,#81
  out (VDPCTL),a

  ; Switch pages 0 and 2 to the (primary) slot in which this
  ; cartridge is inserted. It is assumed that page 1 is already mapped.
  call RSLREG
  ld b,a
  ;    33221100
  and %00001100
  ld c,a
  REPT 4
    rlca
  ENDM
  or c
  REPT 2
    rrca
  ENDM
  ld c,a
  ld a,b
  ;    33221100
  and %11001100
  or c
  ; Now, since WSLREG is mapped into page 0 currently it can't be called.
  ; So the slot select register port is used directly instead.
  out (SLOTSL),a

  ; Page 3 is kept as RAM, making 16Kbyte of total RAM available.
  ; WARNING: After this point, the BIOS is not available at all.
  ; Finally jump into page 0 to start.
  jp top


  db "Copyright (c) Edd Biddulph 2025"

; Note: When setting the color palette data in the B4, B5 and B6 modes,
;       use the same value for each corresponding pair of palette addresses
;       0 to 31 and 32 to 63, that is, 0 and 32, 1 and 33 and so on.
palette_rgb_start:
  ; Offset  0
  incbin "palette.bin"
  ; Offset  4
  db (31*0)/3, (31*0)/3, (31*0)/3
  db (31*3)/3, (31*3)/3, (31*3)/3
  db (31*3)/3, (31*2)/3, (31*1)/3
  db (31*1)/3, (31*2)/3, (31*3)/3
  ; Offset  8
  db (31*0)/3, (31*0)/3, (31*0)/3
  db (31*3)/3, (31*3)/3, (31*3)/3
  db (31*3)/3, (31*2)/3, (31*1)/3
  db (31*1)/3, (31*2)/3, (31*3)/3
  ; Offset 12
  db 0, 0, 0 ; Border
  db 255, 0, 0
  db 0, 255, 0
  db 0, 0, 255
  db 255, 255, 0

  ds 32*3-17*3
  incbin "palette.bin"
palette_rgb_end:

clear_framebuffer_lmmv:
  MAKE_LMMV 0, 0, FRAME_WIDTH, FRAME_HEIGHT, %00000000

exp_table:
  incbin "exp_table.bin"

  ; House
house_logsign_vertices:
  incbin "house_logsign_vertices.bin"
house_edges:
  incbin "house_edges.bin"

  ; Dodecahedron
doe_logsign_vertices:
  incbin "doe_logsign_vertices.bin"
doe_edges:
  incbin "doe_edges.bin"

  ; Cone
cone_logsign_vertices:
  incbin "cone_logsign_vertices.bin"
cone_edges:
  incbin "cone_edges.bin"

  ; Two
two_logsign_vertices:
  incbin "two_logsign_vertices.bin"
two_edges:
  incbin "two_edges.bin"

logsin_table:
  incbin "logsin_table.bin"
signsin_table:
  incbin "signsin_table.bin"

  ds #8000 - $

; ---------------------- Variables ----------------------

; 16Kbyte RAM
  org #C000
RAM_START:
  ; Commandlists
commandlists_start:
commandlist_0:
  ds virtual MAX_COMMANDS*COMMAND_SIZE+1
commandlist_1:
  ds virtual MAX_COMMANDS*COMMAND_SIZE+1
commandlists_end:
commandlist_read_ptr:
  ds virtual 2
commandlist_read_length_ptr:
  ds virtual 2
commandlist_write_ptr:
  ds virtual 2
commandlist_write_length_ptr:
  ds virtual 2

  ; Commandlist execution state
buffer_swap:
  ds virtual 1
last_command_done_flag:
  ds virtual 1
commandlist_filled_flag:
  ds virtual 1
frames_to_skip:
  ds virtual 1

  ; Templated commands
clear_framebuffer_lmmv_ram:
  ds virtual 15

  ; Misc.
demo_mode:
  ds virtual 1
fade_frame_count:
  ds virtual 1
log_animscale:
  ds virtual 2
select_model_routine_addr:
  ds virtual 2

  ; 3D objects
num_vertices:
  ds virtual 1
num_edges:
  ds virtual 1
logsign_vertices_addr:
  ds virtual 2
edges_addr:
  ds virtual 2
theta:
  ds virtual 2
theta_delta:
  ds virtual 2
phi:
  ds virtual 2
phi_delta:
  ds virtual 2
vertices_addr:
  ds virtual 2
projected_vertices_addr:
  ds virtual 2
log_cos_theta:
  ds virtual 2
sign_cos_theta:
  ds virtual 1
log_sin_theta:
  ds virtual 2
sign_sin_theta:
  ds virtual 1
log_sin_theta_sin_phi:
  ds virtual 2
sign_sin_theta_sin_phi:
  ds virtual 1
log_cos_phi:
  ds virtual 2
sign_cos_phi:
  ds virtual 1
log_cos_theta_sin_phi:
  ds virtual 2
sign_cos_theta_sin_phi:
  ds virtual 1
log_sin_theta_cos_phi:
  ds virtual 2
sign_sin_theta_cos_phi:
  ds virtual 1
log_sin_phi:
  ds virtual 2
sign_sin_phi:
  ds virtual 1
log_cos_theta_cos_phi:
  ds virtual 2
sign_cos_theta_cos_phi:
  ds virtual 1
x_accum:
  ds virtual 2
y_accum:
  ds virtual 2
z_accum:
  ds virtual 2
projected_vertices:
  ds virtual 256*2

  ; Stat counters
stat_counters_start:
stat_num_commands_written:
  ds virtual 1
stat_counters_end:
RAM_END:
  ds virtual #E000 - $

