#include <SDL.h>
#include <math.h>
#include <assert.h>

#include "font.h"
#include "defs.h"
#include "mem.h"
#include "render.h"
#include "game.h"
#include "input.h"

#include <iostream>
#include <fstream>
#include <algorithm>
#include <string>
#include <utility>
#include <iterator>
#include <array>
#include <limits>
#include <vector>
#include <set>
#include <cassert>
#include <cfloat>
#include <list>
#include <cmath>
#include <chrono>
#include <map>

using namespace std;

using Vec4 = array<Real, 4>;
using Vec3 = array<Real, 3>;
using Vec2 = array<Real, 2>;
using Face = vector<size_t>;
using Cell = vector<size_t>;

template<typename T, size_t N>
static T dot(const array<T, N>& a, const array<T, N>& b)
{
    T res = 0.0l;
    for(size_t i = 0; i < N; ++i)
        res += a[i] * b[i];
    return res;
}

template<typename T, size_t N>
static array<T, N> normalize(const array<T, N>& a)
{
    array<T, N> res;
    const auto l = sqrt(dot(a, a));
    for(size_t i = 0; i < N; ++i)
        res[i] = a[i] / l;
    return res;
}

template<typename T, size_t N>
static T length(const array<T, N>& a)
{
    return sqrt(dot(a, a));
}

template<typename T, size_t N>
static array<T, N> operator+(const array<T, N>& a, const array<T, N>& b)
{
    array<T, N> res;
    for(size_t i = 0; i < N; ++i)
        res[i] = a[i] + b[i];
    return res;
}

template<typename T, size_t N>
static array<T, N> operator-(const array<T, N>& a, const array<T, N>& b)
{
    array<T, N> res;
    for(size_t i = 0; i < N; ++i)
        res[i] = a[i] - b[i];
    return res;
}

template<typename T, size_t N>
static array<T, N> operator*(const array<T, N>& a, T s)
{
    array<T, N> res;
    for(size_t i = 0; i < N; ++i)
        res[i] = a[i] * s;
    return res;
}

template<typename T, size_t N>
static array<T, N> operator*(T s, const array<T, N>& a)
{
    return a * s;
}

static istream& operator>>(istream& s, Vec4& v)
{
    s >> v[0] >> v[1] >> v[2] >> v[3];
    return s;
}

static istream& operator>>(istream& s, Vec3& v)
{
    s >> v[0] >> v[1] >> v[2];
    return s;
}

template<typename T, size_t N>
static ostream& operator<<(ostream& s, const array<T, N>& v)
{
    for(size_t i = 0; i < N; ++i)
    {
        s << v[i];
        if(i != N - 1)
            s << " ";
    }
    return s;
}

static inline Vec3 cross(const Vec3& a, const Vec3& b)
{
   return { a[1] * b[2] - a[2] * b[1],
            a[2] * b[0] - a[0] * b[2],
            a[0] * b[1] - a[1] * b[0] };
}

static tuple<vector<Vec3>, list<vector<size_t>>> loadOFF(string path, ostream& logstream)
{
    ifstream in(path);

    string first_line;
    in >> first_line;

    if(first_line != "OFF")
        return {};

    logstream << "format: \"" << first_line << "\"" << endl;

    size_t num_vertices = 0, num_faces = 0, num_edges = 0;

    in >> num_vertices >> num_faces >> num_edges;

    logstream << "v: " << num_vertices << " f: " << num_faces << " e: " << num_edges << endl;

    vector<Vec3> vertices(num_vertices);
    list<vector<size_t>> faces(num_faces);

    logstream << "vertices" << endl;
    for(auto& v : vertices)
    {
        in >> v;
        logstream << v << endl;
    }

    logstream << "faces" << endl;
    for(auto& f : faces)
    {
        size_t n = 0;
        in >> n;
        logstream << n << " ";
        f.resize(n);
        for(auto& i : f)
        {
            in >> i;
            logstream << i << " ";
        }
        logstream << endl;
    }

    return { vertices, faces };
}

#define ENGINE_MODE_NULL            0
#define ENGINE_MODE_TITLE           1
#define ENGINE_MODE_TRACKSELECTION  2
#define ENGINE_MODE_INGAME          3

static bool_t game_paused = FALSE;
static int keep_going = 1;

static renderstate_t g_rs;
static player_t g_player;
static gamestate_t g_gs;

static int num_obj_drawn = 0;

uint32_t frame_time = 0;

// Byte Move to XY from Linear
static const uint8_t Command_BMXL = 0b10000000;
// Logical Move to Memory from VDP
static const uint8_t Command_LMMV = 0b00100000;
// Logical Move to Memory from Memory
static const uint8_t Command_LMMM = 0b01000000;

struct Command
{
  uint8_t SX_L;    // R#32
  uint8_t SX_H;    // R#33
  uint8_t SY_L;    // R#34
  uint8_t SY_H;    // R#35
  uint8_t DX_L;    // R#36 
  uint8_t DX_H;    // R#37
  uint8_t DY_L;    // R#38
  uint8_t DY_H;    // R#39
  uint8_t NX_L;    // R#40
  uint8_t NX_H;    // R#41
  uint8_t NY_L;    // R#42
  uint8_t NY_H;    // R#43
  uint8_t ARG;     // R#44
  uint8_t LOP;     // R#45
  uint8_t WM_L;    // R#46
  uint8_t WM_H;    // R#47
  uint8_t FC_L;    // R#48
  uint8_t FC_H;    // R#49
  uint8_t BC_L;    // R#50
  uint8_t BC_H;    // R#51
  uint8_t OP_CODE; // R#52
};

int g_pending_enginemode = ENGINE_MODE_NULL;
int g_enginemode = ENGINE_MODE_NULL;

static int sign(float x)
{
  return (x < 0.0) ? -1 : +1;
}

void transitionEngineMode(int old_mode, int new_mode)
{
  assert(old_mode != new_mode);
  assert(new_mode != ENGINE_MODE_NULL);
  
  switch(old_mode)
  {
    case ENGINE_MODE_NULL:
      break;
    
    case ENGINE_MODE_TRACKSELECTION:
      break;
      
    case ENGINE_MODE_INGAME:
      break;
      
    default:
      assert(0);
      break;
  }
  
  switch(new_mode)
  {
    case ENGINE_MODE_TRACKSELECTION:
      break;
      
    case ENGINE_MODE_INGAME:
      resetLevel(&g_player, &g_gs, SDL_GetTicks());
      break;
      
    default:
      assert(0);
      break;
  }
  
  g_enginemode = new_mode;
}


static void quitLevel()
{
  keep_going = 0;
}

struct xorshift32_state {
    uint32_t a;
};

/* The state must be initialized to non-zero */
uint32_t xorshift32(struct xorshift32_state *state)
{
	/* Algorithm "xor" from p. 4 of Marsaglia, "Xorshift RNGs" */
	uint32_t x = state->a;
	x ^= x << 13;
	x ^= x >> 17;
	x ^= x << 5;
	return state->a = x;
}


// https://media.xiph.org/svt/SGIIMAGESPEC

static unsigned char getbyte(FILE *inf)
{
  unsigned char buf[1];
  fread(buf,1,1,inf);
  return buf[0];
}

static unsigned short getshort(FILE *inf)
{
  unsigned char buf[2];
  fread(buf,2,1,inf);
  return (buf[0]<<8)+(buf[1]<<0);
}

static unsigned long getlong(FILE *inf)
{
  unsigned char buf[4];
  fread(buf,4,1,inf);
  return (buf[0]<<24)+(buf[1]<<16)+(buf[2]<<8)+(buf[3]<<0);
}

uchar* loadFromRGBAFile(FILE *const src, int* out_w, int* out_h)
{
  short magic = getshort(src);
  assert(magic == 474);
  char storage = getbyte(src);
  char bpc = getbyte(src);
  short dimension = getshort(src);
  short xsize = getshort(src);
  short ysize = getshort(src);
  short zsize = getshort(src);
  long pixmin = getlong(src);
  long pixmax = getlong(src);
  long dummy = getlong(src);
  char name[80] = { 0 };
  fread(name, 1, 80, src);
  long colormap = getlong(src);
  char dummy2[404];
  fread(dummy2, 1, 404, src);
  printf("magic = %u\n", magic);
  printf("storage = %u\n", storage);
  printf("bpc = %u\n", bpc);
  printf("dimension = %u\n", dimension);
  printf("xsize = %u\n", xsize);
  printf("ysize = %u\n", ysize);
  printf("zsize = %u\n", zsize);
  printf("pixmin = %u\n", pixmin);
  printf("pixmax = %u\n", pixmax);
  printf("name = '%s'\n", name);
  printf("colormap = %u\n", colormap);
  uchar* pixels = (uchar*)malloc(xsize * ysize * zsize);
  fread(pixels, 1, xsize * ysize * zsize, src);
  *out_w = xsize;
  *out_h = ysize;
  uchar* pixels2 = (uchar*)malloc(xsize * ysize * zsize);
  // re-interleave
  for(int y = 0; y < ysize; ++y)
  {
    for(int x = 0; x < xsize; ++x)
    {
      for(int z = 0; z < zsize; ++z)
      {
        pixels2[z + (x + (ysize-1-y) * xsize) * zsize] = pixels[(x + y * xsize) + z * xsize * ysize];
      }
    }
  }
  free(pixels);
  return pixels2;
}

static void putbyte(FILE *outf, unsigned char x)
{
  unsigned char buf[1] = { x };
  fwrite(buf,1,1,outf);
}

static void putshort(FILE *outf, unsigned short x)
{
  unsigned char buf[2] = { (unsigned char)(x >> 8), (unsigned char)(x & 0xff) };
  fwrite(buf,2,1,outf);
}

static void putlong(FILE *outf, unsigned long x)
{
  unsigned char buf[4] = { (unsigned char)(x >> 24),
                           (unsigned char)((x >> 16) & 0xff),
                           (unsigned char)((x >> 8) & 0xff),
                           (unsigned char)(x & 0xff)  };
  fwrite(buf,4,1,outf);
}

static void writeRGBAFile(FILE *const out, int w, int h, unsigned char* pixels)
{
    putshort(out, 474); // magic
    putbyte (out,   0); // storage
    putbyte (out,   1); // bpc
    putshort(out,   3); // dimensions
    putshort(out,   w); // xsize
    putshort(out,   h); // ysize
    putshort(out,   3); // zsize
    putlong (out,   0); // pixmin
    putlong (out,   0xffffff); // pixmax
    putlong (out,   0); // dummy
    char name[80] = { 0 };
    fwrite(name, 1, 80, out);
    putlong (out,   0); // colormap ID
    char dummy2[404];
    memset(dummy2, 0, sizeof(dummy2));
    fwrite(dummy2, 1, 404, out);

    unsigned char* pixels2 = (unsigned char*)malloc(w * h * 3);
    // re-interleave
    for(int y = 0; y < h; ++y)
    {
        for(int x = 0; x < w; ++x)
        {
            for(int z = 0; z < 3; ++z)
            {
                pixels2[(x + y * w) + z * w * h] = pixels[z + (x + (h-1-y) * w) * 3];
            }
        }
    }

    fwrite(pixels2, 1, w * h * 3, out);
    
    free(pixels2);
}

static uint8_t virtual_framebuffer[VIRTUAL_FRAME_WIDTH*VIRTUAL_FRAME_HEIGHT];

static void blitVirtualFramebuffer(uint16_t* pixel_buffer)
{
  static const uint16_t palette[4] = { 0x0000, RGB888toRGB565(255,128,0),
                       RGB888toRGB565(0,128,255), RGB888toRGB565(255,255,255) };
  for(int y=0;y<VIRTUAL_FRAME_HEIGHT;++y)
    for(int x=0;x<VIRTUAL_FRAME_WIDTH;++x)
    {
      for(int v=0;v<VIRTUAL_PIXEL_HEIGHT*FRAME_ZOOM_FACTOR;++v)
        for(int u=0;u<VIRTUAL_PIXEL_WIDTH*FRAME_ZOOM_FACTOR;++u)
        {
          plotPixel(pixel_buffer, x * VIRTUAL_PIXEL_WIDTH*FRAME_ZOOM_FACTOR + u, y * VIRTUAL_PIXEL_HEIGHT*FRAME_ZOOM_FACTOR + v, palette[virtual_framebuffer[x+y*VIRTUAL_FRAME_WIDTH]&3]);
        }
    }
}

static void clearVirtualFramebuffer()
{
  for(int y=0;y<VIRTUAL_FRAME_HEIGHT;++y)
    for(int x=0;x<VIRTUAL_FRAME_WIDTH;++x)
    {
      virtual_framebuffer[x+y*VIRTUAL_FRAME_WIDTH] = 0;
    }
}

static void plotPixelVirtual(uint32_t x, uint32_t y, const uint8_t colour)
{
    if(x < VIRTUAL_FRAME_WIDTH && y < VIRTUAL_FRAME_HEIGHT)
        virtual_framebuffer[x + y * VIRTUAL_FRAME_WIDTH] = colour;
}

static void plotLineVirtual(int x0, int y0, int x1, int y1, const uint8_t col)
{
   const bool_t steep = labs(y1 - y0) > labs(x1 - x0);

   if(steep)
   {
      SWAP(int, x0, y0);
      SWAP(int, x1, y1);
   }

   if(x0 > x1)
   {
      SWAP(int, x0, x1);
      SWAP(int, y0, y1);
   }

   const int deltax = x1 - x0;
   const int deltay = labs(y1 - y0);
   int error = deltax / 2;
   int y = y0;
   const int ystep = (y0 < y1) ? 1 : -1;

   for(int x = x0; x <= x1; ++x)
   {
      if(steep)
         plotPixelVirtual(y, x, col);
      else
         plotPixelVirtual(x, y, col);

      error -= deltay;

      if(error < 0)
      {
         y += ystep;
         error += deltax;
      }
   }
}

tuple<vector<Vec3>, list<vector<size_t>>> mesh;
static uint8_t edges[256*2];
static uint8_t num_edges=0;
static int16_t vertices[256*3];
static uint8_t num_vertices=0;

static uint8_t logsign_vertices[256*3*3];

#define ROTATION_UNITS 1024
static uint16_t log_table[4096];
static uint16_t exp_table[4096];
static uint16_t logsin_table[ROTATION_UNITS];
static uint16_t logcos_table[ROTATION_UNITS];
static uint8_t signsin_table[ROTATION_UNITS];
static uint8_t signcos_table[ROTATION_UNITS];

#define SCALE_ANIM_FRAMES 64
static uint16_t logscale_table[SCALE_ANIM_FRAMES];

void drawIngameScreen(uint16_t* pixels, renderstate_t* rs, const gamestate_t* gs)
{
  const uint32_t frame_start = SDL_GetTicks();
  const float time = (float)((double)(frame_start) / 1000.0);

  clipAndPlotBox(pixels, 0, 0, FRAME_WIDTH, FRAME_HEIGHT, RGB888toRGB565(50, 50, 50));

  clearVirtualFramebuffer();

  vector<Vec2> projected_vertices(num_vertices);

  const double theta = time;
  const double phi = time / 3.0;

  const uint16_t theta_i = (int)(theta / (M_PI * 2.0) * (double)ROTATION_UNITS) % ROTATION_UNITS;
  const uint16_t phi_i   = (int)(phi   / (M_PI * 2.0) * (double)ROTATION_UNITS) % ROTATION_UNITS;

  const uint16_t log_cos_theta = logcos_table[theta_i];
  const uint16_t log_sin_theta = logsin_table[theta_i];
  const uint16_t log_cos_phi = logcos_table[phi_i];
  const uint16_t log_sin_phi = logsin_table[phi_i];

  const uint16_t log_sin_theta_cos_phi = log_sin_theta + log_cos_phi;
  const uint16_t log_sin_theta_sin_phi = log_sin_theta + log_sin_phi;
  const uint16_t log_cos_theta_cos_phi = log_cos_theta + log_cos_phi;
  const uint16_t log_cos_theta_sin_phi = log_cos_theta + log_sin_phi;

  const int16_t sign_cos_theta = signcos_table[theta_i] ? -1 : 1;
  const int16_t sign_sin_theta = signsin_table[theta_i] ? -1 : 1;
  const int16_t sign_cos_phi = signcos_table[phi_i] ? -1 : 1;
  const int16_t sign_sin_phi = signsin_table[phi_i] ? -1 : 1;

  const int16_t sign_sin_theta_cos_phi = sign_sin_theta * sign_cos_phi;
  const int16_t sign_sin_theta_sin_phi = sign_sin_theta * sign_sin_phi;
  const int16_t sign_cos_theta_cos_phi = sign_cos_theta * sign_cos_phi;
  const int16_t sign_cos_theta_sin_phi = sign_cos_theta * sign_sin_phi;


  const uint16_t log_animscale = logscale_table[int(time*60.0)%SCALE_ANIM_FRAMES];

  for(uint8_t i = 0; i < num_vertices; ++i)
  {
#if 1
    int16_t x_accum = 0;
    int16_t y_accum = 0;
    int16_t z_accum = 0;

    const int16_t x = vertices[i*3+0];
    const int16_t y = vertices[i*3+1];
    const int16_t z = vertices[i*3+2];

    assert(abs(vertices[i*3+0]) < 4096);
    const uint16_t log_x = MAX(0, log_table[abs(x)] - log_animscale);
    assert(abs(vertices[i*3+1]) < 4096);

    x_accum += +exp_table[MAX(0, log_x - log_cos_theta)] * sign_cos_theta * sign(x);
    z_accum += -exp_table[MAX(0, log_x - log_sin_theta)] * sign_sin_theta * sign(x);

    const uint16_t log_y = MAX(0, log_table[abs(y)] - log_animscale);
    assert(abs(vertices[i*3+2]) < 4096);

    x_accum += -exp_table[MAX(0, log_y - log_sin_theta_sin_phi)] * sign_sin_theta_sin_phi * sign(y);
    y_accum += +exp_table[MAX(0, log_y - log_cos_phi)] * sign_cos_phi * sign(y);
    z_accum += -exp_table[MAX(0, log_y - log_cos_theta_sin_phi)] * sign_cos_theta_sin_phi * sign(y);

    const uint16_t log_z = MAX(0, log_table[abs(z)] - log_animscale);

    x_accum += +exp_table[MAX(0, log_z - log_sin_theta_cos_phi)] * sign_sin_theta_cos_phi * sign(z);
    y_accum += +exp_table[MAX(0, log_z - log_sin_phi)] * sign_sin_phi * sign(z);
    z_accum += +exp_table[MAX(0, log_z - log_cos_theta_cos_phi)] * sign_cos_theta_cos_phi * sign(z);

    z_accum += 4096 * 2;

    assert(abs(x_accum)<4096);
    assert(abs(y_accum)<4096);
    assert(((z_accum)>>2)<4096);

    int16_t proj_x = exp_table[MAX(0, log_table[abs(x_accum)] - log_table[((z_accum)+2)>>2] + 2500)] * sign(x_accum);
    int16_t proj_y = exp_table[MAX(0, log_table[abs(y_accum)] - log_table[((z_accum)+2)>>2] + 2500)] * sign(y_accum);

    projected_vertices[i][0] = (proj_x + 1) / 2 + VIRTUAL_FRAME_WIDTH / 2;
    projected_vertices[i][1] = (proj_y + 1) / 2 + VIRTUAL_FRAME_HEIGHT / 2;

    //projected_vertices[i][0] = x_accum / 8 + VIRTUAL_FRAME_WIDTH / 2;
    //projected_vertices[i][1] = z_accum / 8 + VIRTUAL_FRAME_HEIGHT / 2;
#else
    const double x0 = (double)vertices[i*3+0] / 4096.0;
    const double y0 = (double)vertices[i*3+1] / 4096.0;
    const double z0 = (double)vertices[i*3+2] / 4096.0;

    const double x2 = x0 * cos(theta) + z0 * sin(theta) * cos(phi) - y0 * sin(theta) * sin(phi);
    const double y2 = y0 * cos(phi) + z0 * sin(phi);
    const double z2 = z0 * cos(theta) * cos(phi) - y0 * cos(theta) * sin(phi) - x0 * sin(theta) + 4.0;

    projected_vertices[i][0] = x2 / z2 * 512 + VIRTUAL_FRAME_WIDTH / 2;
    projected_vertices[i][1] = y2 / z2 * 512 + VIRTUAL_FRAME_HEIGHT / 2;
#endif
  }

  for(uint8_t i = 0; i < num_edges; ++i)
  {
    uint8_t j = edges[i*2+0];
    uint8_t k = edges[i*2+1];
    plotLineVirtual(projected_vertices[j][0], projected_vertices[j][1],
                    projected_vertices[k][0], projected_vertices[k][1], 1);
  }

  blitVirtualFramebuffer(pixels);
}

extern bool_t extern_write_frame;

void updateIngameLogic()
{
}

static void generateAllTables()
{
  const double log_scale=4095.0/log(4096.0);

  for(int i=0;i<4096;++i)
  {
    log_table[i]=(i==0)?0:log((double)i+0.5)*log_scale+0.5;
    printf("log_table[%d]=%d\n",i,(int)log_table[i]);
  }

  {
    FILE* out=fopen("log_table.bin","wb");
    fwrite(log_table, 1, sizeof(log_table), out);
    fclose(out);
  }

  for(int i=0;i<4096;++i)
  {
    exp_table[i]=exp(((double)i-0.5)/log_scale)+0.5;
    printf("exp_table[%d]=%d\n",i,(int)exp_table[i]);
  }

  {
    FILE* out=fopen("exp_table.bin","wb");
    fwrite(exp_table, 1, sizeof(exp_table), out);
    fclose(out);
  }

  for(int i=0;i<4096;++i)
  {
    printf("exp_table[log_table[%d]]-%d=%d\n",i,i,(int)exp_table[log_table[i]]-i);
  }

  for(int i=0;i<ROTATION_UNITS;++i)
  {
    double x=sin((double)i/(double)ROTATION_UNITS * M_PI * 2.0);
    logsin_table[i]=MIN(4095.0,-(log(abs(x))*log_scale+0.5));
    signsin_table[i]=sign(x)<0?1:0;
    printf("logsin_table[%d]=%d\n",i,(int)logsin_table[i]);
  }

  for(int i=0;i<ROTATION_UNITS;++i)
  {
    double x=cos((double)i/(double)ROTATION_UNITS * M_PI * 2.0);
    logcos_table[i]=MIN(4095.0,-(log(abs(x))*log_scale+0.5));
    signcos_table[i]=sign(x)<0?1:0;
    printf("logcos_table[%d]=%d\n",i,(int)logcos_table[i]);
  }

  {
    FILE* out=fopen("logsin_table.bin","wb");
    fwrite(logsin_table, 1, sizeof(logsin_table), out);
    fclose(out);
  }

  {
    FILE* out=fopen("logcos_table.bin","wb");
    fwrite(logcos_table, 1, sizeof(logcos_table), out);
    fclose(out);
  }

  {
    FILE* out=fopen("signsin_table.bin","wb");
    fwrite(signsin_table, 1, sizeof(signsin_table), out);
    fclose(out);
  }

  {
    FILE* out=fopen("signcos_table.bin","wb");
    fwrite(signcos_table, 1, sizeof(signcos_table), out);
    fclose(out);
  }

  for(int i=0;i<SCALE_ANIM_FRAMES;++i)
  {
    double x=(double)i/(double)SCALE_ANIM_FRAMES;
    logscale_table[i]=MIN(4095.0,-(log(abs(x))*log_scale+0.5));
  }

  {
    FILE* out=fopen("logscale_table.bin","wb");
    fwrite(logscale_table, 1, sizeof(logscale_table), out);
    fclose(out);
  }
}

static void convertModel(const char* off_filename,const string& name)
{
  auto& [mesh_vertices, faces] = mesh = loadOFF(off_filename, cout);

  num_vertices=mesh_vertices.size();

  double min_x = +8192.0*16.0;
  double min_y = +8192.0*16.0;
  double min_z = +8192.0*16.0;

  double max_x = -8192.0*16.0;
  double max_y = -8192.0*16.0;
  double max_z = -8192.0*16.0;

  for(size_t i = 0; i < mesh_vertices.size(); ++i)
  {
    min_x = min(min_x, (double)mesh_vertices[i][0]);
    min_y = min(min_y, (double)mesh_vertices[i][1]);
    min_z = min(min_z, (double)mesh_vertices[i][2]);

    max_x = max(max_x, (double)mesh_vertices[i][0]);
    max_y = max(max_y, (double)mesh_vertices[i][1]);
    max_z = max(max_z, (double)mesh_vertices[i][2]);
  }

  double size_x = max_x - min_x;
  double size_y = max_y - min_y;
  double size_z = max_z - min_z;

  double mid_x = (max_x + min_x) / 2.0;
  double mid_y = (max_y + min_y) / 2.0;
  double mid_z = (max_z + min_z) / 2.0;

  const double scale = 4095.0 / max(size_x, max(size_y, size_z)) * 0.6;

  for(size_t i = 0; i < mesh_vertices.size(); ++i)
  {
    vertices[i*3+0]=(mesh_vertices[i][0]-mid_x)*scale;
    vertices[i*3+1]=(mesh_vertices[i][1]-mid_y)*scale;
    vertices[i*3+2]=(mesh_vertices[i][2]-mid_z)*scale;
    printf("vertex %3d = %5d, %5d, %5d\n",i,vertices[i*3+0],vertices[i*3+1],vertices[i*3+2]);

    const int16_t x = vertices[i*3+0];
    const int16_t y = vertices[i*3+1];
    const int16_t z = vertices[i*3+2];

    assert(abs(vertices[i*3+0]) < 4096);
    const uint16_t log_x = log_table[abs(x)];
    assert(abs(vertices[i*3+1]) < 4096);
    const uint16_t log_y = log_table[abs(y)];
    assert(abs(vertices[i*3+2]) < 4096);
    const uint16_t log_z = log_table[abs(z)];

    logsign_vertices[i*3*3+0] = log_x & 0xff;
    logsign_vertices[i*3*3+1] = log_x >> 8;
    logsign_vertices[i*3*3+2] = x < 0 ? 1 : 0;

    logsign_vertices[i*3*3+3] = log_y & 0xff;
    logsign_vertices[i*3*3+4] = log_y >> 8;
    logsign_vertices[i*3*3+5] = y < 0 ? 1 : 0;

    logsign_vertices[i*3*3+6] = log_z & 0xff;
    logsign_vertices[i*3*3+7] = log_z >> 8;
    logsign_vertices[i*3*3+8] = z < 0 ? 1 : 0;
  }

  {
    FILE* out=fopen((name+"_logsign_vertices.bin").c_str(),"wb");
    fwrite(logsign_vertices, 1, num_vertices*3*3, out);
    fclose(out);
  }

  set<pair<uint8_t, uint8_t>> edge_set;

  for(const auto& fi : faces)
    for(size_t i = 0; i < fi.size(); ++i)
      edge_set.insert(pair(MIN(fi[i],fi[(i+1)%fi.size()]),MAX(fi[i],fi[(i+1)%fi.size()])));

  printf("edge_set.size() = %d\n",(int)edge_set.size());

  num_edges=0;

  for(auto [a, b] : edge_set)
  {
    edges[num_edges*2+0]=a;
    edges[num_edges*2+1]=b;
    ++num_edges;
  }

  {
    FILE* out=fopen((name+"_edges.bin").c_str(),"wb");
    fwrite(edges, 1, num_edges*2, out);
    fclose(out);
  }
}

static void init()
{
  generateAllTables();

  //auto& [mesh_vertices, faces] = mesh = loadOFF("Dodecahedron.off", cout);
  //auto& [mesh_vertices, faces] = mesh = loadOFF("Models/house.off", cout);
  //auto& [mesh_vertices, faces] = mesh = loadOFF("Numbers/0.off", cout);
  //auto& [mesh_vertices, faces] = mesh = loadOFF("Numbers/1.off", cout);
  //auto& [mesh_vertices, faces] = mesh = loadOFF("Numbers/2.off", cout);
  //auto& [mesh_vertices, faces] = mesh = loadOFF("Numbers/3.off", cout);
  //auto& [mesh_vertices, faces] = mesh = loadOFF("Numbers/4.off", cout);
  //auto& [mesh_vertices, faces] = mesh = loadOFF("Geometry/truncated_cube.off", cout);
  //auto& [mesh_vertices, faces] = mesh = loadOFF("Geometry/cone.off", cout);

  convertModel("Models/house.off","house");
  convertModel("Dodecahedron.off","doe");
  convertModel("Geometry/cone.off","cone");
  convertModel("Numbers/2.off","two");

/*
  // Convert the test image
  {
    static const int testimage_vram_location_x=0;
    static const int testimage_vram_location_y=1024;
    //static const int testimage_vram_location_y=0;

    map<array<uint8_t,3>,int> palette;
    int num_colours=0;

    FILE* in=fopen("miku.rgba","rb");
    int image_w=0,image_h=0;
    uchar* image=loadFromRGBAFile(in, &image_w, &image_h);
    uint8_t* image_2bpp=(uint8_t*)malloc(image_w*image_h);
    for(int y=0;y<image_h;++y)
      for(int x=0;x<image_w;++x)
      {
        array<uint8_t,3> colour = { image[(x+y*image_w)*4+0],
                                    image[(x+y*image_w)*4+1],
                                    image[(x+y*image_w)*4+2] };

        if(palette.find(colour)!=palette.end())
          image_2bpp[x+y*image_w]=palette[colour];
        else
        {
          assert(num_colours<4);
          palette[colour]=num_colours;
          image_2bpp[x+y*image_w]=num_colours;
          ++num_colours;
        }
      }

    uint8_t palette_555[4*3] = {};
    for(auto& [k, v] : palette)
    {
      palette_555[v*3+0] = (k[0]*31+128)/255;
      palette_555[v*3+1] = (k[1]*31+128)/255;
      palette_555[v*3+2] = (k[2]*31+128)/255;
    }
    {
      FILE* out=fopen("palette.bin","wb");
      fwrite(palette_555, 1, sizeof(palette_555), out);
      fclose(out);
    }

    uint8_t* image_2bpp_packed=(uint8_t*)malloc((image_w*image_h+3)/4);
    memset(image_2bpp_packed,0,(image_w*image_h+3)/4);
    for(int y=0;y<image_h;++y)
      for(int x=0;x<image_w;++x)
      {
        const int ofs=x+y*image_w;
        image_2bpp_packed[ofs/4]|=image_2bpp[x+y*image_w]<<((3-(ofs&3))*2);
      }
    FILE* out=fopen("miku_2bpp.bin","wb");
    fwrite(image_2bpp_packed, 1, (image_w*image_h+3)/4, out);
    fclose(out);

    {
      FILE* out=fopen("constants.asm","w");
      fprintf(out,"TESTIMAGE_VRAM_LOCATION_X:   equ %d\n",testimage_vram_location_x);
      fprintf(out,"TESTIMAGE_VRAM_LOCATION_Y:   equ %d\n",testimage_vram_location_y);
      fprintf(out,"TESTIMAGE_VRAM_SIZE_X:       equ %d\n",image_w);
      fprintf(out,"TESTIMAGE_VRAM_SIZE_Y:       equ %d\n",image_h);
      fprintf(out,"TESTIMAGE_VRAM_LENGTH:       equ (TESTIMAGE_VRAM_SIZE_X*TESTIMAGE_VRAM_SIZE_Y+3)/4\n");
      fclose(out);
    }
  }
*/

  {
    uint8_t palette_555[4*3] = {
          0, 0, 0,
          0, 0, 0,
          31, 31, 31,
          31, 31, 31,
      };
    {
      FILE* out=fopen("palette.bin","wb");
      fwrite(palette_555, 1, sizeof(palette_555), out);
      fclose(out);
    }
  }
}


int main(int argc, char* argv[])
{
  printf("Compiled on " __DATE__ " at " __TIME__ ".\n");

#ifdef __GNUC__
# ifdef __GNUC_MINOR__
#   ifdef __GNUC_PATCHLEVEL__
  printf("Compiled with GCC %d.%d.%d\n", __GNUC__, __GNUC_MINOR__, __GNUC_PATCHLEVEL__);
#   endif
# endif
#endif

  printf("Compiled against SDL version %d.%d.%d\n", SDL_MAJOR_VERSION, SDL_MINOR_VERSION, SDL_PATCHLEVEL);

  const SDL_version* sdl_version = SDL_Linked_Version();

  if(sdl_version)
    printf("Running dynamically-linked SDL version %d.%d.%d\n", sdl_version->major, sdl_version->minor, sdl_version->patch);
  else
    printf("SDL_Linked_Version() returned NULL.\n");

  if(SDL_Init(SDL_INIT_TIMER | SDL_INIT_VIDEO))
      return -1;

  SDL_WM_SetCaption("", NULL);
  
  atexit(SDL_Quit);
  
  SDL_Surface* surface = SDL_SetVideoMode(FRAME_WIDTH, FRAME_HEIGHT, 16, SDL_HWSURFACE | SDL_DOUBLEBUF | SDL_ASYNCBLIT | SDL_ANYFORMAT);

  if(!surface)
      return -1;

  init();

  sysInitialise();

  initRenderstate(&g_rs);

  g_pending_enginemode = ENGINE_MODE_INGAME;

  while(keep_going == 1)
  {
      if(g_pending_enginemode != g_enginemode)
      {
          transitionEngineMode(g_enginemode, g_pending_enginemode);
      }
    
      SDL_Event event;

      while(SDL_PollEvent(&event))
      {
          if(event.type == SDL_QUIT)
          {
              keep_going = 0;
          }
      }

      frame_time = SDL_GetTicks() - g_gs.start_level_time;

      while(frame_time >= g_gs.next_frame)
      {
          if(!sysUpdateInput())
            return 0;
          
          switch(g_enginemode)
          {
            case ENGINE_MODE_TRACKSELECTION:
              break;
            case ENGINE_MODE_INGAME:
              updateIngameLogic();
              break;
          }

          g_gs.next_frame += 15;
      }
      
      SDL_LockSurface(surface);

      uint16_t* pixels = (uint16_t*)surface->pixels;
      
      switch(g_enginemode)
      {
        case ENGINE_MODE_TRACKSELECTION:
          break;
        case ENGINE_MODE_INGAME:
          drawIngameScreen(pixels, &g_rs, &g_gs);
          break;
      }

      SDL_UnlockSurface(surface);

      SDL_UpdateRect(surface, 0, 0, 0, 0);
      SDL_Flip(surface);

      SDL_Delay(5);

  }

  return 0;
}
