; BALL GOLF

; TODO:
;       * Create an ending screen
;       * Create some more levels and arrange the existing ones in difficulty order
;       * Fix bug whereby ball will sometimes stop in mid-air or on a ceiling. (It still happens... or does it?)

ALIGN: MACRO ?boundary
      ds ?boundary - 1 - ($ + ?boundary - 1) % ?boundary
      ENDM

ALIGN_VIRTUAL: MACRO ?boundary
      ds virtual (?boundary - 1 - ($ + ?boundary - 1) % ?boundary)
      ENDM

MAKE_SPRITE: MACRO ?x, ?y, ?p
      db ?y, ?x, ?p, #00
      ENDM

MAKE_HMMV: MACRO ?dx, ?dy, ?nx, ?ny, ?cr
      db #00,#00                ; R#32, R#33
      db #00,#00                ; R#34, R#35
      db ?dx&#FF,(?dx>>8)&1     ; R#36, R#37
      db ?dy&#FF,(?dy>>8)&3     ; R#38, R#39
      db ?nx&#FF,(?nx>>8)&1     ; R#40, R#41
      db ?ny&#FF,(?ny>>8)&3     ; R#42, R#43
      db ?cr                    ; R#44
      db %00000000              ; R#45
      db %11000000              ; R#46
      ENDM

MAKE_HMMC: MACRO ?dx, ?dy, ?nx, ?ny
      db #00,#00                ; R#32, R#33
      db #00,#00                ; R#34, R#35
      db ?dx&#FF,(?dx>>8)&1     ; R#36, R#37
      db ?dy&#FF,(?dy>>8)&3     ; R#38, R#39
      db ?nx&#FF,(?nx>>8)&1     ; R#40, R#41
      db ?ny&#FF,(?ny>>8)&3     ; R#42, R#43
      db %00000000              ; R#44 (placeholder)
      db %00000000              ; R#45
      db %11110000              ; R#46
      ENDM

MAKE_LMMC: MACRO ?dx, ?dy, ?nx, ?ny, ?op
      db #00,#00                ; R#32, R#33
      db #00,#00                ; R#34, R#35
      db ?dx&#FF,(?dx>>8)&1     ; R#36, R#37
      db ?dy&#FF,(?dy>>8)&3     ; R#38, R#39
      db ?nx&#FF,(?nx>>8)&1     ; R#40, R#41
      db ?ny&#FF,(?ny>>8)&3     ; R#42, R#43
      db %00000000              ; R#44 (placeholder)
      db %00000000              ; R#45
      db %10110000|?op          ; R#46
      ENDM

MAKE_HMMM: MACRO ?sx, ?sy, ?dx, ?dy, ?nx, ?ny
      db ?sx&#FF,(?sx>>8)&1     ; R#32, R#33
      db ?sy&#FF,(?sy>>8)&3     ; R#34, R#35
      db ?dx&#FF,(?dx>>8)&1     ; R#36, R#37
      db ?dy&#FF,(?dy>>8)&3     ; R#38, R#39
      db ?nx&#FF,(?nx>>8)&1     ; R#40, R#41
      db ?ny&#FF,(?ny>>8)&3     ; R#42, R#43
      db #00                    ; R#44
      db %00000000              ; R#45
      db %11010000              ; R#46
      ENDM

MAKE_LMMM: MACRO ?sx, ?sy, ?dx, ?dy, ?nx, ?ny, ?op
      db ?sx&#FF,(?sx>>8)&1     ; R#32, R#33
      db ?sy&#FF,(?sy>>8)&3     ; R#34, R#35
      db ?dx&#FF,(?dx>>8)&1     ; R#36, R#37
      db ?dy&#FF,(?dy>>8)&3     ; R#38, R#39
      db ?nx&#FF,(?nx>>8)&1     ; R#40, R#41
      db ?ny&#FF,(?ny>>8)&3     ; R#42, R#43
      db #00                    ; R#44
      db %00000000              ; R#45
      db %10010000|?op          ; R#46
      ENDM

MAKE_YMMM: MACRO ?sy, ?dx, ?dy, ?ny, ?dix
      db #00,#00                ; R#32, R#33
      db ?sy&#FF,(?sy>>8)&3     ; R#34, R#35
      db ?dx&#FF,(?dx>>8)&1     ; R#36, R#37
      db ?dy&#FF,(?dy>>8)&3     ; R#38, R#39
      db #00,#00                ; R#40, R#41
      db ?ny&#FF,(?ny>>8)&3     ; R#42, R#43
      db #00                    ; R#44
      db (?dix&1)<<2            ; R#45
      db %11100000              ; R#46
      ENDM

MAKE_LINE: MACRO ?x, ?y, ?long, ?short, ?cr, ?diy, ?dix, ?maj, ?op
      db #00,#00                  ; R#32, R#33
      db #00,#00                  ; R#34, R#35
      db ?x&#FF,(?x>>8)&1         ; R#36, R#37
      db ?y&#FF,(?y>>8)&3         ; R#38, R#39
      db ?long&#FF,(?long>>8)&1   ; R#40, R#41
      db ?short&#FF,(?short>>8)&3 ; R#42, R#43
      db ?cr                      ; R#44
      db ?maj|(?dix<<2)|(?diy<<3) ; R#45
      db %01110000|?op            ; R#46
      ENDM


MAKE_PALETTE_ENTRY: MACRO ?r, ?g, ?b
      dw ?b|(?r<<4)|(?g<<8)
      ENDM

PREPARE_LDIRVM_FIXED: MACRO ?destination
      ; Set address counter A16 to A14
      ld a,(?destination>>14)&#FF
      out (VDPCTL),a
      ld a,128+14 ; R#14
      out (VDPCTL),a
      ; Set the address counter A7 to A0
      ld a,?destination&#FF
      out (VDPCTL),a
      ; Set the address counter A13 to A8 and operation mode
      ld a,((?destination>>8)&#3F)|64
      out (VDPCTL),a
      ENDM

LDIRVM_FIXED: MACRO ?source, ?destination, ?length
      PREPARE_LDIRVM_FIXED ?destination
      ld c,VDPDAT
      ld hl,?source
      ld b,?length&#FF
      ld d,(?length>>8)&#FF
      inc d
    ldirvm_fixed_loop:
      otir
      dec d
      jr nz,ldirvm_fixed_loop
      ENDM

WAIT_FOR_HBLANK: MACRO
      ; Poll the HR bit in S#2 to wait for HBLANK
      in a,(VDPCTL)
      and 1<<5
      jr z,$-2-2
      ENDM

WAIT_FOR_NOT_HBLANK: MACRO
      ; Poll the HR bit in S#2 to wait for scan outside of HBLANK
      in a,(VDPCTL)
      and 1<<5
      jr nz,$-2-2
      ENDM

; BIOS ROM
CGTABL: equ #0004

; BIOS Routines
RSLREG: equ #0138
WSLREG: equ #013B
ENASLT: equ #0024

; System Variables
FORCLR: equ #F3E9
BAKCLR: equ #F3EA
BDRCLR: equ #F3EB
TXTNAM: equ #F3B3
TXTCGP: equ #F3B7
HKEY:   equ #FD9A
CLIKSW: equ #F3DB
T32NAM: equ #F3BD
T32COL: equ #F3BF
T32CGP: equ #F3C1
T32ATR: equ #F3C3
T32PAT: equ #F3C5
GRPNAM: equ #F3C7
GRPCOL: equ #F3C9
GRPCGP: equ #F3CB
GRPATR: equ #F3CD
GRPPAT: equ #F3CF

; VRAM
PATTBL: equ #0000
NAMTBL: equ #0800
SATRBL: equ #1400
COLTBL: equ #1480
SPATBL: equ #1800

; Ports
VDPDAT: equ #98 ; Port #0
VDPCTL: equ #99 ; Port #1
VDPPAL: equ #9A ; Port #2
VDPREG: equ #9B ; Port #3
SLOTSL: equ #A8
KEYROW: equ #A9
KEYSEL: equ #AA
PSGREG: equ #A0
PSGWRI: equ #A1
PSGREA: equ #A2
OPLLRG: equ #7C
OPLLDT: equ #7D

; PSG Registers
PSG_REG_TONE_0: equ 0
PSG_REG_TONE_1: equ 1
PSG_REG_TONE_2: equ 2
PSG_REG_TONE_3: equ 3
PSG_REG_TONE_4: equ 4
PSG_REG_TONE_5: equ 5
PSG_REG_NOISE:  equ 6
PSG_REG_MIXER:  equ 7 ; Important note: bit 6 must be 0, and bit 7 must be 1.
PSG_REG_AMP_0:  equ 8
PSG_REG_AMP_1:  equ 9
PSG_REG_AMP_2:  equ 10
PSG_REG_ENV_0:  equ 11
PSG_REG_ENV_1:  equ 12
PSG_REG_ENV_2:  equ 13

; Constants
DEBUG_ENABLED:       equ 0
COLOUR_BLACK         equ 1
COLOUR_MEDIUM_GREEN  equ 2
COLOUR_LIGHT_GREEN   equ 3
COLOUR_DARK_BLUE     equ 4
COLOUR_LIGHT_BLUE    equ 5
COLOUR_DARK_RED      equ 6
COLOUR_CYAN          equ 7
COLOUR_MEDIUM_RED    equ 8
COLOUR_LIGHT_RED     equ 9
COLOUR_DARK_YELLOW   equ 10
COLOUR_LIGHT_YELLOW  equ 11
COLOUR_DARK_GREEN    equ 12
COLOUR_MAGENTA       equ 13
COLOUR_GREY          equ 14
COLOUR_WHITE         equ 15

; VRAM Layout
BASE_PATTERN_LAYOUT:        equ #0000
BASE_PATTERN_LAYOUT_PAGE_1_G4: equ #8000
BASE_PATTERN_LAYOUT_PAGE_1_G7: equ #10000
BASE_SPRITE_PATTERNS:       equ #7000+#8000
BASE_SPRITE_COLOURS:        equ #7800+#8000
BASE_SPRITE_ATTRIBUTES:     equ #7A00+#8000

; V9938 Logic Op Codes
LOGIC_OP_IMP:  equ %0000
LOGIC_OP_AND:  equ %0001
LOGIC_OP_OR:   equ %0010
LOGIC_OP_XOR:  equ %0011
LOGIC_OP_NOT:  equ %0100
LOGIC_OP_TIMP: equ %1000
LOGIC_OP_TAND: equ %1001
LOGIC_OP_TOR:  equ %1010
LOGIC_OP_TXOR: equ %1011
LOGIC_OP_TNOT: equ %1100

; Flag Sprite Patterns
;
;     A sprite pattern is 8x8, stored in 8 bytes.
;     In this game, sprites are 16x16 so they use 4 patterns, which is 32 bytes.
;     In G4, 32 bytes is 64 pixels.
;     So each row in G4 stores 128 bytes = 16 patterns = 4 sprites
;     To store a whole set of patterns requires 16 rows of pixels = 2048 bytes
;     Sprite patterns VRAM address is 0xF000 = Y position 480 pixels
;
FLAG_FRAME_COUNT:             equ 32 ; Must be a power of two.

FLAG_CURRENT_PATTERN_INDEX: equ 16*3
FLAG_CURRENT_PATTERN_COUNT: equ 4*2
FLAG_CURRENT_PATTERN_X_POS: equ 0
FLAG_CURRENT_PATTERN_Y_POS: equ 480+(FLAG_CURRENT_PATTERN_INDEX/16)

FLAG_STORED_PATTERN_COUNT:    equ FLAG_FRAME_COUNT*FLAG_CURRENT_PATTERN_COUNT
FLAG_STORED_PATTERN_X_POS:    equ 0
FLAG_STORED_PATTERN_Y_POS:    equ 256+32
FLAG_STORED_PATTERN_Y_EXTENT: equ (FLAG_STORED_PATTERN_COUNT/(4*2))
FLAG_STORED_PATTERN_X_EXTENT: equ (8*FLAG_CURRENT_PATTERN_COUNT)*2

; Flag pole
FLAG_POLE_METATILE_INDEX_0:   equ 246
FLAG_POLE_METATILE_INDEX_1:   equ FLAG_POLE_METATILE_INDEX_0+1

TEXT_OPAQUE_COLOUR:     equ %0110
TEXT_SHADOW_COLOUR:     equ %0010
TEXT_BACKGROUND_COLOUR: equ %0100
TEXT_SHADOW_X_OFFSET:   equ 1
TEXT_SHADOW_Y_OFFSET:   equ 2
TEXT_STATUSBAR_HEIGHT:  equ 17
TEXT_Y_POS:             equ 192-TEXT_STATUSBAR_HEIGHT
TEXT_X_POS:             equ 4
TEXT_STROKE_COUNTER_X:  equ TEXT_X_POS+29*8

BLANK_SPRITE_PATTERN:          equ (blank_patterns-sprite_patterns_start)/8
BATTER_OUTLINE_SPRITE_PATTERN: equ (batter_outline_patterns-sprite_patterns_start)/8

NUM_BATTER_ANGLES:        equ 17
NUM_BATTER_SWING_LEVELS:  equ 64

FONT_Y_POS: equ 128

BGM_INSTRUMENT: equ 2

BALL_SFX_FLAG_BOUNCE_BIT: equ 0
BALL_SFX_FLAG_HIT_BIT:    equ 1
BALL_SFX_FLAG_SWING_BIT:  equ 2

NUMBER_OF_LEVELS: equ 15

TEXT_QUOTE_CHARACTER:     equ 62
TEXT_FULLSTOP_CHARACTER:  equ 63
TEXT_TERMINAL_CHARACTER:  equ 128

  include "string_defs.asm"

; VDP Mode Register Flags
; R#0
VDP_DG:   equ (1 << 6)
VDP_IE2:  equ (1 << 5)
VDP_IE1:  equ (1 << 4)
VDP_M5:   equ (1 << 3)
VDP_M4:   equ (1 << 2)
VDP_M3:   equ (1 << 1)
; R#1
VDP_BL:   equ (1 << 6)
VDP_IE0:  equ (1 << 5)
VDP_M1:   equ (1 << 4)
VDP_M2:   equ (1 << 3)
VDP_SI:   equ (1 << 1)
VDP_MAG:  equ (1 << 0)
; R#8
VDP_MS:   equ (1 << 7)
VDP_LP:   equ (1 << 6)
VDP_TP:   equ (1 << 5)
VDP_CB:   equ (1 << 4)
VDP_VR:   equ (1 << 3)
VDP_SPD:  equ (1 << 1)
VDP_BW:   equ (1 << 0)
; R#9
VDP_LN:   equ (1 << 7)
VDP_S1:   equ (1 << 5)
VDP_S0:   equ (1 << 4)
VDP_IL:   equ (1 << 3)
VDP_E0:   equ (1 << 2)
VDP_NT:   equ (1 << 1)
VDP_DC:   equ (1 << 0)

; 16Kbyte ROM, used mostly for music routines
  org #0000

  ds 32
  jp $

music_addresses:
  include "music_addresses.asm"

music:
  incbin "music.bin"

music_fnumber_table:
  incbin "fnumber_table.bin"

GENERATE_MUSIC_DECODE_ROUTINE: MACRO ?I, ?S
        ld hl,music_bitstream_bit      ; 11
        ld a,?S                        ; 8
        add a,(hl)                     ; 8
        ld (hl),a                      ; 8
        ld hl,(music_bitstream_addr)   ; 17
        ld a,(hl)                      ; 8
        and ((1<<?S)-1)<<?I            ; 8
        IF ?I<4
          REPT ?I
            rrca                       ; 5
          ENDM
        ELSE
          REPT 8-?I
            rlca                       ; 5
          ENDM
        ENDIF
        IF ?I=(8-?S)
          inc hl                       ; 7
          ld (music_bitstream_addr),hl ; 17
        ENDIF
        IF ?I>(8-?S)
          ; Extra bits required = (?I+?S)-8
          ; Destination bit offset = 8-?I
          inc hl                       ; 7
          ld (music_bitstream_addr),hl ; 17
          ld b,a                       ; 5
          ld a,(hl)                    ; 8
          and (1<<((?I+?S)-8))-1       ; 8
          IF (8-?I)<4
            REPT (8-?I)
              rlca                     ; 5
            ENDM
          ELSE
            REPT 8-(8-?I)
              rrca                     ; 5
            ENDM
          ENDIF
          or b                         ; 5
        ENDIF
        ret                            ; 11
      ENDM

GENERATE_MUSIC_DECODE_ROUTINE_SIZE_1: MACRO ?I
        ld hl,music_bitstream_bit      ; 11
        inc (hl)                       ; 12
        ld hl,(music_bitstream_addr)   ; 17
        ld a,(hl)                      ; 8
        IF ?I=7
          inc hl                       ; 7
          ld (music_bitstream_addr),hl ; 17
        ENDIF
        and 1<<?I                      ; 8
        ret                            ; 11
      ENDM

music_decode_field_bit0_size7:
    GENERATE_MUSIC_DECODE_ROUTINE 0, 7
music_decode_field_bit1_size7:
    GENERATE_MUSIC_DECODE_ROUTINE 1, 7
music_decode_field_bit2_size7:
    GENERATE_MUSIC_DECODE_ROUTINE 2, 7
music_decode_field_bit3_size7:
    GENERATE_MUSIC_DECODE_ROUTINE 3, 7
music_decode_field_bit4_size7:
    GENERATE_MUSIC_DECODE_ROUTINE 4, 7
music_decode_field_bit5_size7:
    GENERATE_MUSIC_DECODE_ROUTINE 5, 7
music_decode_field_bit6_size7:
    GENERATE_MUSIC_DECODE_ROUTINE 6, 7
music_decode_field_bit7_size7:
    GENERATE_MUSIC_DECODE_ROUTINE 7, 7

music_decode_field_bit0_size6:
    GENERATE_MUSIC_DECODE_ROUTINE 0, 6
music_decode_field_bit1_size6:
    GENERATE_MUSIC_DECODE_ROUTINE 1, 6
music_decode_field_bit2_size6:
    GENERATE_MUSIC_DECODE_ROUTINE 2, 6
music_decode_field_bit3_size6:
    GENERATE_MUSIC_DECODE_ROUTINE 3, 6
music_decode_field_bit4_size6:
    GENERATE_MUSIC_DECODE_ROUTINE 4, 6
music_decode_field_bit5_size6:
    GENERATE_MUSIC_DECODE_ROUTINE 5, 6
music_decode_field_bit6_size6:
    GENERATE_MUSIC_DECODE_ROUTINE 6, 6
music_decode_field_bit7_size6:
    GENERATE_MUSIC_DECODE_ROUTINE 7, 6

music_decode_field_bit0_size4:
    GENERATE_MUSIC_DECODE_ROUTINE 0, 4
music_decode_field_bit1_size4:
    GENERATE_MUSIC_DECODE_ROUTINE 1, 4
music_decode_field_bit2_size4:
    GENERATE_MUSIC_DECODE_ROUTINE 2, 4
music_decode_field_bit3_size4:
    GENERATE_MUSIC_DECODE_ROUTINE 3, 4
music_decode_field_bit4_size4:
    GENERATE_MUSIC_DECODE_ROUTINE 4, 4
music_decode_field_bit5_size4:
    GENERATE_MUSIC_DECODE_ROUTINE 5, 4
music_decode_field_bit6_size4:
    GENERATE_MUSIC_DECODE_ROUTINE 6, 4
music_decode_field_bit7_size4:
    GENERATE_MUSIC_DECODE_ROUTINE 7, 4

music_decode_field_bit0_size3:
    GENERATE_MUSIC_DECODE_ROUTINE 0, 3
music_decode_field_bit1_size3:
    GENERATE_MUSIC_DECODE_ROUTINE 1, 3
music_decode_field_bit2_size3:
    GENERATE_MUSIC_DECODE_ROUTINE 2, 3
music_decode_field_bit3_size3:
    GENERATE_MUSIC_DECODE_ROUTINE 3, 3
music_decode_field_bit4_size3:
    GENERATE_MUSIC_DECODE_ROUTINE 4, 3
music_decode_field_bit5_size3:
    GENERATE_MUSIC_DECODE_ROUTINE 5, 3
music_decode_field_bit6_size3:
    GENERATE_MUSIC_DECODE_ROUTINE 6, 3
music_decode_field_bit7_size3:
    GENERATE_MUSIC_DECODE_ROUTINE 7, 3

music_decode_field_bit0_size1:
    GENERATE_MUSIC_DECODE_ROUTINE_SIZE_1 0
music_decode_field_bit1_size1:
    GENERATE_MUSIC_DECODE_ROUTINE_SIZE_1 1
music_decode_field_bit2_size1:
    GENERATE_MUSIC_DECODE_ROUTINE_SIZE_1 2
music_decode_field_bit3_size1:
    GENERATE_MUSIC_DECODE_ROUTINE_SIZE_1 3
music_decode_field_bit4_size1:
    GENERATE_MUSIC_DECODE_ROUTINE_SIZE_1 4
music_decode_field_bit5_size1:
    GENERATE_MUSIC_DECODE_ROUTINE_SIZE_1 5
music_decode_field_bit6_size1:
    GENERATE_MUSIC_DECODE_ROUTINE_SIZE_1 6
music_decode_field_bit7_size1:
    GENERATE_MUSIC_DECODE_ROUTINE_SIZE_1 7

music_decode_field_addr_7:
  dw music_decode_field_bit0_size7
  dw music_decode_field_bit1_size7
  dw music_decode_field_bit2_size7
  dw music_decode_field_bit3_size7
  dw music_decode_field_bit4_size7
  dw music_decode_field_bit5_size7
  dw music_decode_field_bit6_size7
  dw music_decode_field_bit7_size7

music_decode_field_addr_6:
  dw music_decode_field_bit0_size6
  dw music_decode_field_bit1_size6
  dw music_decode_field_bit2_size6
  dw music_decode_field_bit3_size6
  dw music_decode_field_bit4_size6
  dw music_decode_field_bit5_size6
  dw music_decode_field_bit6_size6
  dw music_decode_field_bit7_size6

music_decode_field_addr_4:
  dw music_decode_field_bit0_size4
  dw music_decode_field_bit1_size4
  dw music_decode_field_bit2_size4
  dw music_decode_field_bit3_size4
  dw music_decode_field_bit4_size4
  dw music_decode_field_bit5_size4
  dw music_decode_field_bit6_size4
  dw music_decode_field_bit7_size4

music_decode_field_addr_3:
  dw music_decode_field_bit0_size3
  dw music_decode_field_bit1_size3
  dw music_decode_field_bit2_size3
  dw music_decode_field_bit3_size3
  dw music_decode_field_bit4_size3
  dw music_decode_field_bit5_size3
  dw music_decode_field_bit6_size3
  dw music_decode_field_bit7_size3

music_decode_field_addr_1:
  dw music_decode_field_bit0_size1
  dw music_decode_field_bit1_size1
  dw music_decode_field_bit2_size1
  dw music_decode_field_bit3_size1
  dw music_decode_field_bit4_size1
  dw music_decode_field_bit5_size1
  dw music_decode_field_bit6_size1
  dw music_decode_field_bit7_size1

CALL_MUSIC_DECODE_ROUTINE: MACRO ?base
        ld hl,music_bitstream_bit
        ld a,(hl)
        and 7
        rlca
        ld hl,?base
        ld b,0
        ld c,a
        add hl,bc
        ld a,(hl)
        inc hl
        ld h,(hl)
        ld l,a
        ld bc,$+3+1+1
        push bc
        jp (hl)
      ENDM

silence_opll:
  ; Set F-number MSB, Note on flag, and octave to zero
  REPT 6,?I
  ld bc,#2000+(?I<<8)
  call write_opll_register
  ENDM
  ret

init_music:
  ld (music_playback_start_addr),hl
  ld (music_bitstream_addr),hl
  ld a,16
  ld (music_playback_framedelta),a
  xor a
  ld (music_bitstream_bit),a
  ld (music_playback_status),a
  ret

music_playback_update:
  ld a,(music_playback_status)
  bit 0,a
  ret nz

  ld hl,music_playback_framedelta
  dec (hl)
  ret nz

  ; Channel mask
  CALL_MUSIC_DECODE_ROUTINE music_decode_field_addr_6
  ld c,a
  ld b,6
music_playback_update_loop:
  rr c
  ld a,b
  exx
  jp nc,music_playback_update_loop_skipchannel

  ld d,a ; D is channel index
  dec d

  ; Note on/off
  CALL_MUSIC_DECODE_ROUTINE music_decode_field_addr_1
  jr z,music_playback_update_noteoff

  ; Vel
  CALL_MUSIC_DECODE_ROUTINE music_decode_field_addr_3
  ld c,a
  ; Set instrument + velocity
  ld a,#30
  or d
  out (OPLLRG),a
  REPT 0
    ex (sp),hl
  ENDM
  ld a,c
  or BGM_INSTRUMENT<<4
  out (OPLLDT),a
  REPT 0
    ex (sp),hl
  ENDM

  ; Key
  CALL_MUSIC_DECODE_ROUTINE music_decode_field_addr_7
  ; Use lookup table to get values for F-number etc. registers
  rlca
  ld b,0
  ld c,a
  ld hl,music_fnumber_table
  add hl,bc
  ; Set F-number LSBs
  ld a,#10
  or d
  out (OPLLRG),a
  REPT 0
    ex (sp),hl
  ENDM
  ld a,(hl)
  out (OPLLDT),a
  REPT 4
    ex (sp),hl
  ENDM
  ; Set F-number MSB, Note on flag, and octave
  ld a,#20
  or d
  out (OPLLRG),a
  REPT 0
    ex (sp),hl
  ENDM
  inc hl
  ld a,(hl)
  out (OPLLDT),a
  REPT 0
    ex (sp),hl
  ENDM

  jr music_playback_update_loop_skipchannel
music_playback_update_noteoff:
  ; Set F-number LSBs to zero
  ld a,#10
  or d
  out (OPLLRG),a
  REPT 0
    ex (sp),hl
  ENDM
  xor a
  out (OPLLDT),a
  REPT 4
    ex (sp),hl
  ENDM
  ; Set F-number MSB, Note on flag, and octave to zero
  ld a,#20
  or d
  out (OPLLRG),a
  REPT 0
    ex (sp),hl
  ENDM
  xor a
  out (OPLLDT),a
  REPT 0
    ex (sp),hl
  ENDM

music_playback_update_loop_skipchannel:
  exx
  dec b
  jp nz,music_playback_update_loop
  
  ; Frame delta
  CALL_MUSIC_DECODE_ROUTINE music_decode_field_addr_4
  ld (music_playback_framedelta),a
  and a
  call z,music_playback_rewind
  ret

music_playback_stop:
  ld a,#01
  ld (music_playback_status),a
  ret

music_playback_rewind:
  ld hl,(music_playback_start_addr)
  call init_music
  ld a,120
  ld (music_playback_framedelta),a
  ret

APRLOPLL:
  db "APRLOPLL"
OPLL:
  db "OPLL"
check_for_fmpac_clone_string:
  ld hl,#4018
  ld de,APRLOPLL
  ld b,8
  jr check_for_fmpac_string_loop
check_for_fmpac_string:
  ld hl,#401C
  ld de,OPLL
  ld b,4
check_for_fmpac_string_loop:
  ld a,(de)
  cp (hl)
  ret nz
  inc hl
  inc de
  djnz check_for_fmpac_string_loop
  ld a,#00
  and a ; Reset the zero flag
  ret
search_for_fmpac_2:
  in a,(SLOTSL)
  ld c,a
  ; Check primary slots at #4000-#8000
  ld b,4
search_for_fmpac_loop:
  ld a,b
  dec a
  REPT 2
    rlca
  ENDM
  ld d,a
  ld a,c
  ;    33221100
  and %11110011
  or  d
  out (SLOTSL),a
  push bc
  call check_for_fmpac_clone_string
  pop bc
  ret z ; Clone cartridges do not require initialisation.
  push bc
  call check_for_fmpac_string
  pop bc
  jr z,init_fmpac
  djnz search_for_fmpac_loop
  ret

search_for_fmpac:
  call search_for_fmpac_2
  ; Restore this ROM's pages
  ld a,c
  out (SLOTSL),a
  ret

init_fmpac:
  ; A "real" FM-PAC has been found, so initialise it.
  ld hl,#7FF6
  set 0,(hl)
  ret

write_opll_register:
  ld a,b
  out (OPLLRG),a
  ld a,c
  out (OPLLDT),a
  REPT 2
    ex (sp),hl
  ENDM
  ret

unpack_font_G4:
  ld hl,font_graphic
  ld de,font_unpack_temp
  ld bc,1024
unpack_font_G4_loop:
  ; Unpack one byte into four bytes
  ld a,(hl)

  ; Temporary unpacked bytes are stored as DE'HL'
  exx

  ld hl,#0000
  ld de,#0000

  ; Bit pattern for opaque      pixel: %0110 = 6
  ; Bit pattern for transparent pixel: %0000 = 0
  ; Bit pattern for background:        %0100 = 4

  bit 7,a
  jr z,$+2+4
  set 4*1+1,l
  set 4*1+2,l
  bit 6,a
  jr z,$+2+4
  set 4*0+1,l
  set 4*0+2,l
  bit 5,a
  jr z,$+2+4
  set 4*1+1,h
  set 4*1+2,h
  bit 4,a
  jr z,$+2+4
  set 4*0+1,h
  set 4*0+2,h
  bit 3,a
  jr z,$+2+4
  set 4*1+1,e
  set 4*1+2,e
  bit 2,a
  jr z,$+2+4
  set 4*0+1,e
  set 4*0+2,e
  bit 1,a
  jr z,$+2+4
  set 4*1+1,d
  set 4*1+2,d
  bit 0,a
  jr z,$+2+4
  set 4*0+1,d
  set 4*0+2,d

  ld a,l
  exx
  ld (de),a
  inc de
  exx
  ld a,h
  exx
  ld (de),a
  inc de
  exx
  ld a,e
  exx
  ld (de),a
  inc de
  exx
  ld a,d
  exx
  ld (de),a
  inc de

  inc hl
  dec bc
  ld a,b
  or c
  jr nz,unpack_font_G4_loop
  ; Upload
  ld hl,hmmc_font
  ld de,font_unpack_temp
  ld b,(256*32/2)/16
  call execute_hmmc_x16
  ret

unpack_font_G7:
  ld de,font_graphic
  ld hl,font_unpack_temp
  ld bc,1024
unpack_font_G7_loop:
  ; Unpack one byte into eight bytes
  ld a,(de)
  REPT 8
    rlca
    ld (hl),#00
    jr nc,$+2+2
    ld (hl),#FF
    inc hl
  ENDM
  inc de
  dec bc
  ld a,b
  or c
  jr nz,unpack_font_G7_loop
  ; Upload in two parts
  ld hl,hmmc_font_part1
  ld de,font_unpack_temp
  ld b,#00
  call execute_hmmc_x16
  ld hl,hmmc_font_part2
  ld de,font_unpack_temp+256*16
  ld b,#00
  call execute_hmmc_x16
  ret

wall_graphics:
  incbin "wall_graphics_4bpp.bin"

flag_graphics:
  incbin "flag_graphic.bin"

font_graphic:
  incbin "font.bin"

flagpole_graphics: ; Arranged in metatile form
  incbin "flagpole_graphics.bin"

collision_tiles:
  include "collision_tiles.asm"

batter_table:
  incbin "batter_table.bin"

  ds #4000 - $

; 32Kbyte ROM, where the main game code is
  org #4000

  db "AB"
  dw INIT
  db 0,0,0,0,0,0,0,0,0,0,0,0

INIT:
  di

  ld sp,#F000

  ; Switch pages 0 and 2 to the (primary) slot in which this
  ; cartridge is inserted. It is assumed that page 1 is already mapped.
  call RSLREG
  ld b,a
  ;    33221100
  and %00001100
  ld c,a
  REPT 4
    rlca
  ENDM
  or c
  REPT 2
    rrca
  ENDM
  ld c,a
  ld a,b
  ;    33221100
  and %11001100
  or c
  ; Now, since WSLREG is mapped into page 0 currently it can't be called.
  ; So the slot select register port is used directly instead.
  out (SLOTSL),a

  ; Page 3 is kept as RAM, making 16Kbyte of total RAM available.
  ; WARNING: After this point, the BIOS is not available at all.

  ; Search for and initialise the FM-PAC if it was found.
  call search_for_fmpac

  ; Reset
  ld b,#40
opll_reset_loop:
  ld a,b
  dec a
  out (OPLLRG),a
  REPT 2
    ex (sp),hl
  ENDM
  xor a
  out (OPLLDT),a
  REPT 4
    ex (sp),hl
  ENDM
  djnz opll_reset_loop

  ; Prerequisites for rhythm mode

  ld bc,#0E20
  call write_opll_register
  ld bc,#1620
  call write_opll_register
  ld bc,#1750
  call write_opll_register
  ld bc,#18C0
  call write_opll_register
  ld bc,#2605
  call write_opll_register
  ld bc,#2705
  call write_opll_register
  ld bc,#2801
  call write_opll_register

  call init_audio

  ; Initialise RAM
  ld hl,RAM_START
  ld (hl),0
  ld de,RAM_START+1
  ld bc,RAM_END-RAM_START-1
  ldir

  ; Switch to interrupt mode 2 (custom interrupt handler)
  ld a,#BE
  ld i,a
  im 2

  ; *******************************************************************************
  ; Title screen
begin_titlescreen:
  call wait_for_vblank
  di

  ; Set GRAPHIC7 mode and blank the screen
  ; M5 M4 M3 M2 M1
  ;  1  1  1  0  0
  ld a,VDP_M3|VDP_M4|VDP_M5
  out (VDPCTL),a
  ld a,#80
  out (VDPCTL),a
  ld a,0
  out (VDPCTL),a
  ld a,#81
  out (VDPCTL),a
  ld a,VDP_SPD|VDP_VR
  out (VDPCTL),a
  ld a,#88
  out (VDPCTL),a
  ld a,0
  out (VDPCTL),a
  ld a,#89
  out (VDPCTL),a

  ld a,%00000000
  out (VDPCTL),a
  ld a,128+45
  out (VDPCTL),a

  ld a,#00
  call set_border_colour

  call silence_opll

  xor a
  ld (current_level_index),a

  ; Set pattern layout base address
  ; Set to page 1, which should be blank while the titlescreen is being prepared.
  ld a,(BASE_PATTERN_LAYOUT_PAGE_1_G7>>11)|%00011111
  out (VDPCTL),a
  ld a,#82
  out (VDPCTL),a
  ; Sprite attributes
  ld a,((BASE_SPRITE_ATTRIBUTES>>7)&#FF)|7
  out (VDPCTL),a
  ld a,#85
  out (VDPCTL),a
  ld a,BASE_SPRITE_ATTRIBUTES>>15
  out (VDPCTL),a
  ld a,#8B
  out (VDPCTL),a
  ; Sprite patterns
  ld a,BASE_SPRITE_PATTERNS>>11
  out (VDPCTL),a
  ld a,#86
  out (VDPCTL),a

  ; Clear the framebuffer
  call wait_for_vdp_ready
  ld a,32
  out (VDPCTL),a
  ld a,#91
  out (VDPCTL),a
  ld hl,hmmv_clear_framebuffer
  ld c,VDPREG
  call outi_15

  ; Clear the maskbuffer
  call wait_for_vdp_ready
  ld a,32
  out (VDPCTL),a
  ld a,#91
  out (VDPCTL),a
  ld hl,hmmv_clear_maskbuffer
  ld c,VDPREG
  call outi_15

  ; Save the current level index, since this clears all 8kb of the used RAM.
  ld a,(current_level_index)
  push af

  ; Unpack font and copy it to VRAM
  call unpack_font_G7

  ; Reset the area of RAM that was used for temporary font unpacking
  ld hl,font_unpack_temp
  ld (hl),0
  ld de,font_unpack_temp+1
  ld bc,font_unpack_temp_end-font_unpack_temp-1
  ldir

  pop af
  ld (current_level_index),a

  ; Upload brush shadow graphic
  ld hl,hmmc_brush_shadow
  ld de,brush_shadow
  ld b,(12*6)/4
  call execute_hmmc_x4

  ; Upload brush mask
  ld hl,hmmc_brush_mask
  ld de,brush_mask_inverted
  ld b,(12*6)/4
  call execute_hmmc_x4

  ld hl,lmmm_brush
  ld de,lmmm_brush_ram
  ld bc,16
  ldir

  ld hl,lmmm_brush_mask
  ld de,lmmm_brush_mask_ram
  ld bc,16
  ldir

  ld hl,lmmm_text
  ld de,lmmm_text_ram
  ld bc,16
  ldir

  ld hl,shadow_composite_commands_start
  ld de,shadow_composite_commands_ram
  ld bc,shadow_composite_commands_end-shadow_composite_commands_start
  ldir

  ld hl,titlescreen_fadeout_commands_start
  ld de,titlescreen_fadeout_commands_ram
  ld bc,titlescreen_fadeout_commands_end-titlescreen_fadeout_commands_start
  ldir

  ld hl,brush_points_start
  ld (brush_points_pointer),hl

  ; Draw some text
  ld hl,COPYRIGHT_STRING_ADDR
  ld b,128-COPYRIGHT_STRING_LENGTH*4
  call blit_text_string_G7

  ; Initialise music playback
  ld hl,(music_addresses+2)
  call init_music

  ld bc,#0000
  ; Set up for P#0
  ld a,#00
  out (VDPCTL),a
  ld a,#90
  out (VDPCTL),a

  ld a,-32
  ld (scroll_position),a

  xor a
  ld (frame_counter+0),a
  ld (frame_counter+1),a

  ld bc,#0000
  ; Set up for P#0
  ld a,#00
  out (VDPCTL),a
  ld a,#90
  out (VDPCTL),a

  ; Enable screen display
  ld a,VDP_M3|VDP_M4|VDP_M5|VDP_IE1
  out (VDPCTL),a
  ld a,#80
  out (VDPCTL),a
  ld a,VDP_IE0|VDP_BL
  out (VDPCTL),a
  ld a,#81
  out (VDPCTL),a

  ; The titlescreen loop
frameloop_titlescreen:
  ld bc,#0000
  ; Set up for P#0
  ld a,#00
  out (VDPCTL),a
  ld a,#90
  out (VDPCTL),a
  call wait_for_vblank

  di

  ld hl,frame_counter
  inc (hl)
  ld a,(hl)
  and 1
  jr nz,$+2+3+1
  ld hl,frame_counter+1
  inc (hl)

  ; Set pattern layout base address
  ld a,(BASE_PATTERN_LAYOUT>>11)|%00011111
  out (VDPCTL),a
  ld a,#82
  out (VDPCTL),a

  ; Write R#19 (interrupt line register)
  ld a,128+8
  out (VDPCTL),a
  ld a,128+19
  out (VDPCTL),a

  ; Set scroll registers
  ; NOTE: Scroll position affects sprite position!!
  ld a,0
  out (VDPCTL),a
  ld a,128+23
  out (VDPCTL),a

  call music_playback_update

  IF DEBUG_ENABLED
    ld a,#10
    call set_border_colour
  ENDIF

  ; Increment brush colour
  ; Blue
  ld a,(brush_colour_index+0)
  ld hl,brush_sine_table
  ld b,0
  ld c,a
  add hl,bc
  add a,4
  ld (brush_colour_index+0),a
  ld a,(hl)
  ld (brush_colour+0),a
  ; Red
  ld a,(brush_colour_index+1)
  ld hl,brush_sine_table
  ld b,0
  ld c,a
  add hl,bc
  add a,2
  ld (brush_colour_index+1),a
  ld a,(hl)
  ld (brush_colour+1),a
  ; Green
  ld a,(brush_colour_index+2)
  ld hl,brush_sine_table
  ld b,0
  ld c,a
  add hl,bc
  add a,3
  ld (brush_colour_index+2),a
  ld a,(hl)
  ld (brush_colour+2),a

  ; Upload brush graphic
  ld de,brush_mask
  exx
  ld hl,hmmc_brush
  ld de,dither_pattern
  ld b,(12*6)/4
  push bc
  call wait_for_vdp_ready
  ld a,32
  out (VDPCTL),a
  ld a,#91
  out (VDPCTL),a
  ld c,VDPREG
  REPT 12
    outi
  ENDM
  ; First byte of graphic must be loaded already before execution.
  call get_brush_pixel
  out (c),a
  inc hl
  outi
  outi
  ld c,#AC
  REPT 3
    call get_brush_pixel
    ex af,af'
    call wait_for_vdp_transfer_ready
    ex af,af'
    out (VDPCTL),a
    ld a,c
    out (VDPCTL),a
  ENDM
  pop bc
  dec b
  ret z
  ld c,#AC
brush_tr_loop:
  REPT 4
    call get_brush_pixel
    ex af,af'
    call wait_for_vdp_transfer_ready
    ex af,af'
    out (VDPCTL),a
    ld a,c
    out (VDPCTL),a
  ENDM
  djnz brush_tr_loop


  ; Paint at the next brush point
  ld hl,(brush_points_pointer)
  ld b,(hl)
  inc hl
  ld c,(hl)
  ld a,b
  or c
  jp z,end_brush_loop
  inc hl
  ld (brush_points_pointer),hl

  ld ix,lmmm_brush_ram
  ld (ix+4),b ; DX L
  ld (ix+6),c ; DY L
  ld ix,lmmm_brush_mask_ram
  ld (ix+4),b ; DX L
  ld (ix+6),c ; DY L

  ; Offset position for shadow
  REPT 5
    inc b
    inc c
  ENDM

  ld ix,shadow_composite_commands_ram+shadow_composite_command_1-shadow_composite_commands_start
  ld (ix+0),b ; SX L
  ld (ix+2),c ; SY L
  ld ix,shadow_composite_commands_ram+shadow_composite_command_3-shadow_composite_commands_start
  ld (ix+0),b ; SX L
  ld (ix+2),c ; SY L
  ld ix,shadow_composite_commands_ram+shadow_composite_command_4-shadow_composite_commands_start
  ld (ix+0),b ; SX L
  ld (ix+2),c ; SY L
  ld ix,shadow_composite_commands_ram+shadow_composite_command_5-shadow_composite_commands_start
  ld (ix+4),b ; DX L
  ld (ix+6),c ; DY L
  ld ix,shadow_composite_commands_ram+shadow_composite_command_6-shadow_composite_commands_start
  ld (ix+4),b ; DX L
  ld (ix+6),c ; DY L

  ; Blit the brush graphic
  ld a,32
  out (VDPCTL),a
  ld a,#91
  out (VDPCTL),a
  call wait_for_vdp_ready
  ld hl,lmmm_brush_ram
  ld c,VDPREG
  call outi_15

  IF DEBUG_ENABLED
    ld a,%00100101
    call set_border_colour
  ENDIF

  ; Blit the brush mask
  ld a,32
  out (VDPCTL),a
  ld a,#91
  out (VDPCTL),a
  call wait_for_vdp_ready
  ld hl,lmmm_brush_mask_ram
  ld c,VDPREG
  call outi_15

  ; Exeute the shadow compositing commands
  ld hl,shadow_composite_commands_ram
  ld d,6
shadow_composite_loop:
  ld a,32
  out (VDPCTL),a
  ld a,#91
  out (VDPCTL),a
  call wait_for_vdp_ready
  ld c,VDPREG
  call outi_15
  dec d
  jr nz,shadow_composite_loop

end_brush_loop:

  IF DEBUG_ENABLED
    ld a,#ee
    call set_border_colour
  ENDIF

  ld bc,#0000
  ; Set up for P#0
  ld a,#00
  out (VDPCTL),a
  ld a,#90
  out (VDPCTL),a
  ei
  halt
  di
  ; Write R#19 (interrupt line register)
  ; This is done now to prevent annoying interaction with R#23.
  ld a,0
  out (VDPCTL),a
  ld a,128+19
  out (VDPCTL),a
  ; Set scroll registers
  ; NOTE: Scroll position affects sprite position!!

  ld hl,frame_counter+1
  ld a,150
  cp (hl)
  ld hl,scroll_position
  ld a,(hl)
  jr nc,titlescreen_skip_scroll_increment
  cp 0
  jr z,$+2+2
  ; Scroll in the copyright message.
  inc (hl)
  inc (hl)
titlescreen_skip_scroll_increment:
  out (VDPCTL),a
  ld a,128+23
  out (VDPCTL),a

  ld hl,scroll_position
  ld a,(hl)
  cp 0
  jr nz,titlescreen_skip_keyboard_inputs
  ; Handle inputs
  in a,(KEYSEL)
  and #F0
  or 8
  out (KEYSEL),a
  in a,(KEYROW)
  bit 0,a ; Space
  jp z,fade_out_title_screen

  in a,(KEYSEL)
  and #F0
  or 6
  out (KEYSEL),a
  in a,(KEYROW)
  bit 5,a ; F1
  jp z,begin_lasereffect
titlescreen_skip_keyboard_inputs:

  jp frameloop_titlescreen

; BD        13, 16    Ch. 7
; SD        17        Ch. 
; TOM       15        Ch. 9
; TOP-CYM   18        Ch. 9
; HH        14        Ch. 8

trigger_rhythm_0: ; BD
  push bc
  push af

  ; Rhythm Off
  ld b,#0E
  ld a,(opll_rhythm_cache)
  and #FF-(1<<4)
  or #20
  ld c,a
  call write_opll_register

  ; Rhythm On
  ld b,#0E
  ld a,(opll_rhythm_cache)
  or 1<<4
  ld (opll_rhythm_cache),a
  or #20
  ld c,a
  call write_opll_register

  pop bc
  pop af
  ret

trigger_rhythm_1: ; SD
  push bc
  push af

  ; Rhythm Off
  ld b,#0E
  ld a,(opll_rhythm_cache)
  and #FF-(1<<3)
  or #20
  ld c,a
  call write_opll_register

  ; Rhythm On
  ld b,#0E
  ld a,(opll_rhythm_cache)
  or 1<<3
  ld (opll_rhythm_cache),a
  or #20
  ld c,a
  call write_opll_register

  pop bc
  pop af
  ret

trigger_rhythm_2: ; TOM
  push bc
  push af

  ; Rhythm Off
  ld b,#0E
  ld a,(opll_rhythm_cache)
  and #FF-(1<<2)
  or #20
  ld c,a
  call write_opll_register

  ; Rhythm On
  ld b,#0E
  ld a,(opll_rhythm_cache)
  or 1<<2
  ld (opll_rhythm_cache),a
  or #20
  ld c,a
  call write_opll_register
  pop bc
  pop af
  ret

trigger_rhythm_3: ; TOP-CYM
  push bc
  push af

  ; Rhythm Off
  ld b,#0E
  ld a,(opll_rhythm_cache)
  and #FF-(1<<1)
  or #20
  ld c,a
  call write_opll_register

  ; Rhythm On
  ld b,#0E
  ld a,(opll_rhythm_cache)
  or 1<<1
  ld (opll_rhythm_cache),a
  or #20
  ld c,a
  call write_opll_register

  pop bc
  pop af
  ret

trigger_rhythm_4: ; HH
  push bc
  push af

  ; Rhythm Off
  ld b,#0E
  ld a,(opll_rhythm_cache)
  and #FF-(1<<0)
  or #20
  ld c,a
  call write_opll_register

  ; Rhythm On
  ld b,#0E
  ld a,(opll_rhythm_cache)
  or 1<<0
  ld (opll_rhythm_cache),a
  or #20
  ld c,a
  call write_opll_register

  pop bc
  pop af
  ret

  ; Format for G7 is GGGRRRBB
get_brush_pixel:
  push bc
  ; Sine table is in 3.5 fixed point format
  ; Blue channel is 2 bits
  ld a,(de)
  inc de
  ld b,a
  ld a,(brush_colour+0)
  add a,b
  REPT 2
    rlca
  ENDM
  and %00000011
  ld c,a
  ; Red channel is 3 bits
  ld a,(brush_colour+1)
  add a,b
  REPT 3
    rrca
  ENDM
  and %00011100
  or c
  ld c,a
  ; Green channel is 3 bits
  ld a,(brush_colour+2)
  add a,b
  and %11100000
  or c
  ; Apply the mask
  exx
  ld c,a
  ld a,(de)
  inc de
  and c
  exx
  pop bc
  ret

fade_out_title_screen:
  di
  ; Stay in G7, but disable the horizontal scan interrupt.
  ld a,VDP_M3|VDP_M4|VDP_M5
  out (VDPCTL),a
  ld a,#80
  out (VDPCTL),a
  xor a
  ld (frame_counter),a

  ; Set scroll registers
  ; NOTE: Scroll position affects sprite position!!
  ld a,0
  out (VDPCTL),a
  ld a,128+23
  out (VDPCTL),a

fade_out_title_screen_loop:
  ld bc,#0000
  ; Set up for P#0
  ld a,#00
  out (VDPCTL),a
  ld a,#90
  out (VDPCTL),a
  call wait_for_vblank

  di

  IF DEBUG_ENABLED
    ld a,%10000000
    call set_border_colour
  ENDIF

  ; Exeute the shadow compositing commands
  ld hl,titlescreen_fadeout_commands_ram
  ld d,(titlescreen_fadeout_commands_end-titlescreen_fadeout_commands_start)/15
titlescreen_fadeout_loop:
  ld a,32
  out (VDPCTL),a
  ld a,#91
  out (VDPCTL),a
  call wait_for_vdp_ready
  ld c,VDPREG
  call outi_15
  dec d
  jr nz,titlescreen_fadeout_loop

  ; +4 offset to X coordinate
  ld hl,titlescreen_fadeout_commands_ram+4
  ld b,4
titlescreen_fadeout_update_loop1:
  ; Increment the X coordinate
  inc (hl)
  ld de,15
  add hl,de
  djnz titlescreen_fadeout_update_loop1
  ; +2 offset to Y coordinate
  REPT 2
    inc hl
  ENDM
  ld b,3
titlescreen_fadeout_update_loop2:
  ; Decrement the Y coordinate
  dec (hl)
  ld de,15
  add hl,de
  djnz titlescreen_fadeout_update_loop2

  IF DEBUG_ENABLED
    ld a,%00100101
    call set_border_colour
  ENDIF

  ; Exit loop if the fadeout has been completed
  ld hl,frame_counter
  inc (hl)
  ld a,64
  cp (hl)
  jp z,begin_game

  jp fade_out_title_screen_loop


  ; *******************************************************************************
  ; Intermission screen
begin_intermission:
  call wait_for_vblank
  di

  ; Note that the intermission screen must not touch any other page than page 0!

  ; Set palette registers
  ld a,#00
  out (VDPCTL),a
  ld a,#90
  out (VDPCTL),a
  ld hl,black_palette
  ld c,VDPPAL
  call outi_32

  ; Set GRAPHIC4 mode and disable display
  ; M5 M4 M3 M2 M1
  ;  0  1  1  0  0
  ld a,VDP_SI
  out (VDPCTL),a
  ld a,#81
  out (VDPCTL),a
  ld a,VDP_M3|VDP_M4
  out (VDPCTL),a
  ld a,#80
  out (VDPCTL),a
  ld a,VDP_SPD|VDP_VR|VDP_TP
  out (VDPCTL),a
  ld a,#88
  out (VDPCTL),a
  ld a,0
  out (VDPCTL),a
  ld a,#89
  out (VDPCTL),a

  ld a,%00000000
  out (VDPCTL),a
  ld a,128+45
  out (VDPCTL),a

  call silence_opll

  ld a,0
  call set_border_colour

  ; Set scroll registers
  ; NOTE: Scroll position affects sprite position!!
  ld a,0
  out (VDPCTL),a
  ld a,128+23
  out (VDPCTL),a

  ; Pattern layout
  ld a,(BASE_PATTERN_LAYOUT>>10)|%00011111
  out (VDPCTL),a
  ld a,#82
  out (VDPCTL),a
  ; Sprite attributes
  ld a,((BASE_SPRITE_ATTRIBUTES>>7)&#FF)|7
  out (VDPCTL),a
  ld a,#85
  out (VDPCTL),a
  ld a,BASE_SPRITE_ATTRIBUTES>>15
  out (VDPCTL),a
  ld a,#8B
  out (VDPCTL),a
  ; Sprite patterns
  ld a,BASE_SPRITE_PATTERNS>>11
  out (VDPCTL),a
  ld a,#86
  out (VDPCTL),a

  ; Clear the framebuffer
  call wait_for_vdp_ready
  ld a,32
  out (VDPCTL),a
  ld a,#91
  out (VDPCTL),a
  ld hl,hmmv_clear_intermission
  ld c,VDPREG
  call outi_15

  ; Save the current level index, since this clears all 8kb of the used RAM.
  ld a,(current_level_index)
  push af
  ; Save the stroke counter
  ld hl,(stroke_counter_digits)
  push hl
  ; Save the intermission text draw routine address
  ld hl,(draw_intermission_text_address)
  push hl

  ; Unpack font and copy it to VRAM
  call unpack_font_G4_intermission

  ; Bit pattern for opaque      pixel: %1100 = 12
  ; Bit pattern for transparent pixel: %0000 =  0
  ; Bit pattern for background:        %1000 =  8
  ; Bit pattern for shadow:            %0100 =  4

  ; Set up for P#0
  ld a,#00
  out (VDPCTL),a
  ld a,#90
  out (VDPCTL),a

  ; Reset the area of RAM that was used for temporary font unpacking
  ld a,(current_level_index)
  ld hl,font_unpack_temp
  ld (hl),0
  ld de,font_unpack_temp+1
  ld bc,font_unpack_temp_end-font_unpack_temp-1
  ldir

  ; Restore the intermission text draw routine address
  pop hl
  ld (draw_intermission_text_address),hl

  ; Restore the stroke counter
  pop hl
  ; Decrement the stroke counter, since the number of strokes actually taken
  ; is one less than what the counter is showing.
  ld a,h
  sub 1
  jr nc,$+2+2+1
  ld a,9
  dec l
  ld h,a
  ld (stroke_counter_digits),hl

  ; Restore the current level index, since this clears all 8kb of the used RAM.
  pop af
  ld (current_level_index),a

  ; Set up the vertical stripes for the "water surface"
  ; Texture pattern:
  ;
  ;   0+8=8 6+8=E
  ;   4+8=C 2+8=A
  ;
  ld hl,stripes_temp
  ld b,160/2
  ld (hl),#8E
  inc hl
  ld (hl),#CA
  inc hl
  djnz $-1-2-1-2
  ld hl,hmmc_intermission_stripes
  ld de,stripes_temp
  ld b,(2*160/2)/16
  call execute_hmmc_x16

  ld hl,lmmm_intermission_stripes
  ld de,lmmm_intermission_stripes_ram
  ld bc,16
  ldir
  ld b,127
intermission_stripes_clone_loop:
  ld d,b
  ld ix,lmmm_intermission_stripes_ram
  ld a,b
  add a,a
  ld (ix+4),a
  call wait_for_vdp_ready
  ld a,32
  out (VDPCTL),a
  ld a,#91
  out (VDPCTL),a
  ld hl,lmmm_intermission_stripes_ram
  ld c,VDPREG
  call outi_15
  ld b,d
  djnz intermission_stripes_clone_loop

  ld hl,lmmm_text
  ld de,lmmm_text_ram
  ld bc,16
  ldir

  ld hl,lmmm_text_xor
  ld de,lmmm_text_xor_ram
  ld bc,16
  ldir

  ld hl,lmmm_text_or
  ld de,lmmm_text_or_ram
  ld bc,16
  ldir

  ; Clear the upper half of the screen to #F, because this half of the screen
  ; will be reflected into the lower half to mask out the shadow parts of the mirror image.
  call wait_for_vdp_ready
  ld a,32
  out (VDPCTL),a
  ld a,#91
  out (VDPCTL),a
  ld hl,hmmv_clear_intermission_upper_mask
  ld c,VDPREG
  call outi_15

  ; Draw some text, for shadow-masking the water surface only.
  ld hl,(draw_intermission_text_address)
  ld de,$+3+1+1
  push de
  jp (hl)

  ; Reflect the upper half of the screen through the horizon and mask.
  ld hl,lmmm_intermission_reflect
  ld de,lmmm_intermission_reflect_ram
  ld bc,16
  ldir

  ld ix,lmmm_intermission_reflect_ram
  ld (ix+14),%10010000|LOGIC_OP_AND ; Opcode
  ld b,96
  ld e,96
intermission_reflect_loop_2:
  ld d,b
  ld (ix+2),b ; SY L
  ld (ix+6),e ; DY L
  call wait_for_vdp_ready
  ld a,32
  out (VDPCTL),a
  ld a,#91
  out (VDPCTL),a
  ld hl,lmmm_intermission_reflect_ram
  ld c,VDPREG
  call outi_15
  ld b,d
  inc e
  djnz intermission_reflect_loop_2

  ; Clear the upper half of the screen to #0
  call wait_for_vdp_ready
  ld a,32
  out (VDPCTL),a
  ld a,#91
  out (VDPCTL),a
  ld hl,hmmv_clear_intermission_upper_zero
  ld c,VDPREG
  call outi_15

  ; Draw some text which will be textured
  ld hl,(draw_intermission_text_address)
  ld de,$+3+1+1
  push de
  jp (hl)

  ; Set up the vertical stripes for the text
  ; Texture pattern:
  ;
  ;   1+8=9 7+8=F
  ;   5+8=D 3+8=B
  ;
  ld hl,stripes_temp
  ld b,96/2
  ld (hl),#9F
  inc hl
  ld (hl),#DB
  inc hl
  djnz $-1-2-1-2
  ld hl,hmmc_intermission_stripes_2
  ld de,stripes_temp
  ld b,(2*96/2)/16
  call execute_hmmc_x16

  ; The stripes for the text are applied with AND mask, since they
  ; need to be "textured" over the letters.
  ld hl,lmmm_intermission_stripes_2
  ld de,lmmm_intermission_stripes_ram
  ld bc,16
  ldir
  ld b,127
intermission_stripes_clone_loop_2:
  ld d,b
  ld ix,lmmm_intermission_stripes_ram
  ld a,b
  add a,a
  ld (ix+4),a
  call wait_for_vdp_ready
  ld a,32
  out (VDPCTL),a
  ld a,#91
  out (VDPCTL),a
  ld hl,lmmm_intermission_stripes_ram
  ld c,VDPREG
  call outi_15
  ld b,d
  djnz intermission_stripes_clone_loop_2

  ; Reflect the framebuffer through the horizon
  ld hl,lmmm_intermission_reflect
  ld de,lmmm_intermission_reflect_ram
  ld bc,16
  ldir

  ld ix,lmmm_intermission_reflect_ram
  ld b,96
  ld e,96
intermission_reflect_loop:
  ld d,b
  ld (ix+2),b ; SY L
  ld (ix+6),e ; DY L
  call wait_for_vdp_ready
  ld a,32
  out (VDPCTL),a
  ld a,#91
  out (VDPCTL),a
  ld hl,lmmm_intermission_reflect_ram
  ld c,VDPREG
  call outi_15
  ld b,d
  inc e
  djnz intermission_reflect_loop

  ; Clear the upper half of the screen to #1 (the actual background)
  call wait_for_vdp_ready
  ld a,32
  out (VDPCTL),a
  ld a,#91
  out (VDPCTL),a
  ld hl,hmmv_clear_intermission_upper_one
  ld c,VDPREG
  call outi_15

  ; Draw some text (non-textured version)
  ld hl,(draw_intermission_text_address)
  ld de,$+3+1+1
  push de
  jp (hl)

  ; Set palette registers
  ld a,#00
  out (VDPCTL),a
  ld a,#90
  out (VDPCTL),a
  ld hl,greyscale_ramp_palette
  ld c,VDPPAL
  call outi_32

  ; Set up for P#1
  ld a,#01
  out (VDPCTL),a
  ld a,#90
  out (VDPCTL),a
  ld c,VDPPAL
  ld hl,(intermission_gradient+0) ; Background gradient colour stored in HL
  ; Write P#1
  out (c),l ; 14
  out (c),h ; 14

  ld bc,#0000
  ; Set up for P#0
  ld a,#00
  out (VDPCTL),a
  ld a,#90
  out (VDPCTL),a

  ld a,VDP_M3|VDP_M4|VDP_IE1
  out (VDPCTL),a
  ld a,#80
  out (VDPCTL),a
  ; Enable the display
  ld a,VDP_SI|VDP_IE0|VDP_BL
  out (VDPCTL),a
  ld a,#81
  out (VDPCTL),a

  ; The intermission loop
frameloop_intermission:
  ld bc,#0000
  ; Set up for P#0
  ld a,#00
  out (VDPCTL),a
  ld a,#90
  out (VDPCTL),a
  call wait_for_vblank

  di

  ; Handle inputs
  in a,(KEYSEL)
  and #F0
  or 8
  out (KEYSEL),a
  in a,(KEYROW)
  bit 0,a ; Space
  ret z ; The intermission screen is a subroutine!

  ld a,VDP_M3|VDP_M4|VDP_IE1
  out (VDPCTL),a
  ld a,#80
  out (VDPCTL),a

  ; Write R#19 (interrupt line register)
  ld a,96
  out (VDPCTL),a
  ld a,128+19
  out (VDPCTL),a

  ; Set scroll registers
  ; NOTE: Scroll position affects sprite position!!
  ld a,0
  out (VDPCTL),a
  ld a,128+23
  out (VDPCTL),a

  ; Set up for P#15
  ld a,#0F
  out (VDPCTL),a
  ld a,#90
  out (VDPCTL),a
  ld c,VDPPAL
  ld hl,(intermission_text_colour)
  ; Write P#15
  out (c),l ; 14
  out (c),h ; 14

  ;ld a,6
  ;call set_border_colour

  ; Fill in the wavetable
  ld d,sine_table>>8
  ld hl,intermission_wavetable
  ld b,96/2
  ld c,0
intermission_wavetable_loop:
  ld a,(frame_counter)
  add a,c
  REPT 3
    sla a
  ENDM
  ld e,a
  ld a,(de)
  REPT 0
    sra a
  ENDM
  push hl
  push de
  ld l,c
  inc l
  inc l
  REPT 2
    sra l
  ENDM
  call multiply_8x8
  ld a,d
  pop de
  pop hl
  ;sub c
  and #FE
  ld (hl),a
  inc c
  inc hl
  djnz intermission_wavetable_loop

  ;ld a,0
  ;call set_border_colour


  ld bc,#0000
  ; Set up for P#0
  ld a,#00
  out (VDPCTL),a
  ld a,#90
  out (VDPCTL),a
  ei
  ; Wait for the split
  halt
  di
  ; Write R#19 (interrupt line register)
  ; This is done now to prevent annoying interaction with R#23.
  ld a,0
  out (VDPCTL),a
  ld a,128+19
  out (VDPCTL),a

  ld a,VDP_M3|VDP_M4
  out (VDPCTL),a
  ld a,#80
  out (VDPCTL),a

  ; Dither pattern sequence
  ;
  ; 0 3 1 2   Palette index
  ;
  ; A A A A   
  ; B A A A   0
  ; B A B A   1
  ; B A B B   2
  ; B B B B   3
  ;

  ld a,#02
  out (VDPCTL),a
  ld a,#8F
  out (VDPCTL),a ; Select S#2

  ld c,VDPPAL
  ld de,intermission_wavetable

  REPT 12,?I
  ; Set up for P#8
  ld a,#08
  out (VDPCTL),a
  ld a,#90
  out (VDPCTL),a
  ;     B | (R << 4) | (G << 8)
  exx
  ld c,VDPPAL
  ld hl,(intermission_gradient_2+?I*2) ; Text gradient colour stored in HL'
  exx
  ld hl,(intermission_gradient+?I*2) ; Background gradient colour stored in HL
  ld b,4
intermission_palette_loop_0:
  WAIT_FOR_NOT_HBLANK
  WAIT_FOR_HBLANK
  WAIT_FOR_NOT_HBLANK
  ; Set scroll registers
  ; NOTE: Scroll position affects sprite position!!
  ld a,(de)
  inc de
  out (VDPCTL),a
  ld a,128+23
  out (VDPCTL),a
  REPT 17
    nop
  ENDM
  ; Write P#8+b*2+0
  out (c),l ; 14
  out (c),h ; 14
  ; Write P#8+b*2+1
  exx
  out (c),l ; 14
  out (c),h ; 14
  exx
  djnz intermission_palette_loop_0
  ENDM

  ld hl,frame_counter
  inc (hl)
  ;inc (hl)

  jp frameloop_intermission

intermission_gradient:
  REPT 12,?I
    MAKE_PALETTE_ENTRY ((((11-?I)*7)*2)/3)/11, ((((11-?I)*7)*2)/3)/11, ((11-?I)*7)/11
  ENDM
intermission_gradient_2:
  REPT 12,?I
    MAKE_PALETTE_ENTRY ((((11-?I)*7)*3)/3)/11, ((((11-?I)*7)*3)/3)/11, ((11-?I)*7)/11
  ENDM
intermission_text_colour:
  MAKE_PALETTE_ENTRY 7, 7, 7

draw_intermission_text_welldone:

  ld ix,lmmm_text_ram
  ld (ix+6),116-96 ; DY L
  ld hl,WELLDONE1_STRING_ADDR
  ld b,128-WELLDONE1_STRING_LENGTH*4
  call blit_text_string_G7

  ; Fill in the stroke count digits
  ld hl,WELLDONE2_STRING_ADDR
  ld de,temp_string_buffer
  ld bc,WELLDONE2_STRING_LENGTH+1 ; Includes the terminator
  ldir
  ld hl,stroke_counter_digits
  ld de,temp_string_buffer+WELLDONE2_DIGIT_POS
  ld bc,2
  ldir

  ; If the stoke count is 1, then make it say "stroke" instead of "strokes".
  ld b,128-WELLDONE2_STRING_LENGTH*4
  ld a,(stroke_counter_digits+0)
  cp 0
  jr nz,draw_intermission_text_welldone_isplural
  ld a,(stroke_counter_digits+1)
  cp 1
  jr nz,draw_intermission_text_welldone_isplural
  ld hl,temp_string_buffer+WELLDONE2_END_POS
  ld (hl),TEXT_FULLSTOP_CHARACTER
  inc hl
  ld (hl),TEXT_TERMINAL_CHARACTER
  ld b,128-WELLDONE2_STRING_LENGTH*4+4
draw_intermission_text_welldone_isplural:

  ld ix,lmmm_text_ram
  ld (ix+6),116-96+24 ; DY L
  ld hl,temp_string_buffer
  call blit_text_string_G7

  ld ix,lmmm_text_ram
  ld (ix+6),116-96+24+24 ; DY L
  ld hl,WELLDONE3_STRING_ADDR
  ld b,128-WELLDONE3_STRING_LENGTH*4
  call blit_text_string_G7

  ret


draw_intermission_text_congrats:

  ld ix,lmmm_text_ram
  ld (ix+6),116-96 ; DY L
  ld hl,CONGRATS1_STRING_ADDR
  ld b,128-CONGRATS1_STRING_LENGTH*4
  call blit_text_string_G7

  ld ix,lmmm_text_ram
  ld (ix+6),116-96+24 ; DY L
  ld hl,CONGRATS2_STRING_ADDR
  ld b,128-CONGRATS2_STRING_LENGTH*4
  call blit_text_string_G7

  ld ix,lmmm_text_ram
  ld (ix+6),116-96+24+24 ; DY L
  ld hl,CONGRATS3_STRING_ADDR
  ld b,128-CONGRATS3_STRING_LENGTH*4
  call blit_text_string_G7

  ret

  ; *******************************************************************************
  ; Laser effect screen (also the password entry screen)
begin_lasereffect:
  call wait_for_vblank
  di

  ; Set palette registers
  ld a,#00
  out (VDPCTL),a
  ld a,#90
  out (VDPCTL),a
  ld hl,black_palette
  ld c,VDPPAL
  call outi_32

  ; Set GRAPHIC4 mode and disable display
  ; M5 M4 M3 M2 M1
  ;  0  1  1  0  0
  ld a,VDP_SI
  out (VDPCTL),a
  ld a,#81
  out (VDPCTL),a
  ld a,VDP_M3|VDP_M4
  out (VDPCTL),a
  ld a,#80
  out (VDPCTL),a
  ld a,VDP_VR|VDP_TP
  out (VDPCTL),a
  ld a,#88
  out (VDPCTL),a
  ld a,0
  out (VDPCTL),a
  ld a,#89
  out (VDPCTL),a

  ld a,%00000000
  out (VDPCTL),a
  ld a,128+45
  out (VDPCTL),a

  call silence_opll

  ld a,#00
  call set_border_colour

  ; Set scroll registers
  ; NOTE: Scroll position affects sprite position!!
  ld a,0
  out (VDPCTL),a
  ld a,128+23
  out (VDPCTL),a

  ; Pattern layout
  ld a,(BASE_PATTERN_LAYOUT>>10)|%00011111
  out (VDPCTL),a
  ld a,#82
  out (VDPCTL),a
  ; Sprite attributes
  ld a,((BASE_SPRITE_ATTRIBUTES>>7)&#FF)|7
  out (VDPCTL),a
  ld a,#85
  out (VDPCTL),a
  ld a,BASE_SPRITE_ATTRIBUTES>>15
  out (VDPCTL),a
  ld a,#8B
  out (VDPCTL),a
  ; Sprite patterns
  ld a,BASE_SPRITE_PATTERNS>>11
  out (VDPCTL),a
  ld a,#86
  out (VDPCTL),a

  ; Clear the framebuffer
  call wait_for_vdp_ready
  ld a,32
  out (VDPCTL),a
  ld a,#91
  out (VDPCTL),a
  ld hl,hmmv_clear_lasereffect
  ld c,VDPREG
  call outi_15

  ; Copy command structures to RAM
  ld de,line_laser_commands_ram
  ld hl,line_laser_commands_start
  ld bc,line_laser_commands_end-line_laser_commands_start
  ldir

  ; Upload sprite colours
  ld hl,lasereffect_sprite_colours
  ld (hl),15
  ld de,lasereffect_sprite_colours+1
  ld bc,32*16-1
  ldir
  LDIRVM_FIXED lasereffect_sprite_colours, BASE_SPRITE_COLOURS, 32*16
  ; Set up sprites for text
  LDIRVM_FIXED lasereffect_text_sprites_start, BASE_SPRITE_ATTRIBUTES, lasereffect_text_sprites_end-lasereffect_text_sprites_start

  ; Draw text patterns
  PREPARE_LDIRVM_FIXED BASE_SPRITE_PATTERNS
  ld de,ENTER_PASSWORD_STRING_ADDR
  ld b,ENTER_PASSWORD_STRING_LENGTH
lasereffect_draw_text_pattern_loop:
  ld a,(de)
  call out_spritetext_character
  inc de
  dec b
  jp nz,lasereffect_draw_text_pattern_loop

  ; Initialise password buffer
  xor a
  ld (password_length),a
  ld hl,password_type_buffer
  ld (password_type_addr),hl
  ld a,TEXT_QUOTE_CHARACTER
  ld (hl),a
  ld de,password_type_buffer+1
  ld bc,32
  ldir

  ; Initialise password display
  PREPARE_LDIRVM_FIXED BASE_SPRITE_PATTERNS+16*4*8
  ld b,16
init_password_display_loop:
  ld a,TEXT_FULLSTOP_CHARACTER
  call out_spritetext_character
  djnz init_password_display_loop

  ; Prepare for drawing password text characters
  PREPARE_LDIRVM_FIXED BASE_SPRITE_PATTERNS+16*4*8

  ; Initialise music playback
  ld hl,(music_addresses+4)
  call init_music

  ; Set palette registers
  ld a,#00
  out (VDPCTL),a
  ld a,#90
  out (VDPCTL),a
  ld hl,laser_palette
  ld c,VDPPAL
  call outi_32

  ld bc,#0000
  ; Set up for P#0
  ld a,#00
  out (VDPCTL),a
  ld a,#90
  out (VDPCTL),a

  ; Enable the display
  ld a,VDP_SI|VDP_IE0|VDP_BL
  out (VDPCTL),a
  ld a,#81
  out (VDPCTL),a

  ; The lasereffect loop
frameloop_lasereffect:
  ld bc,#0000
  ; Set up for P#0
  ld a,#00
  out (VDPCTL),a
  ld a,#90
  out (VDPCTL),a
  call wait_for_vblank

  di

  ;ld a,8
  ;call set_border_colour

  call music_playback_update

  ld hl,frame_counter
  inc (hl)

  ; Calculate address of cube parameters for the current frame.
  ld iy,cube
  ld a,(frame_counter)
  and 3
  ; Multiply A by 12=8+4
  REPT 2
    rla
  ENDM
  ld b,a
  rla
  add a,b
  ld b,0
  ld c,a
  add iy,bc

  ; Draw a cube, four frames behind
  ld a,(frame_counter)
  and 3
  ld hl,laser_table_1
  ld b,0
  ld c,a
  add hl,bc
  ld a,(hl)
  cpl
  and 15
  ld (iy+(cube_clr-cube)),a ; CLR
  ld a,%01110000|LOGIC_OP_AND ; Opcode
  ld (iy+(cube_op-cube)),a

  call draw_cube_linesonly

  ; Draw a cube
  ld a,(frame_counter)
  and 3
  ld hl,laser_table_1
  ld b,0
  ld c,a
  add hl,bc
  ld a,(hl)
  ld (iy+(cube_clr-cube)),a ; CLR
  ld a,%01110000|LOGIC_OP_OR ; Opcode
  ld (iy+(cube_op-cube)),a
  ld a,(frame_counter)

  call draw_cube

  call wait_for_vdp_ready

  ; Handle inputs
  in a,(KEYSEL)
  and #F0
  or 7
  out (KEYSEL),a
  in a,(KEYROW)
  bit 7,a ; Return
  jr nz,$+2+1+3+3+3
  scf
  call z,check_password
  jp nc,begin_game
  jp c,begin_titlescreen

  ; Scan the keyboard
  in a,(KEYSEL)
  and #F0
  or 6
  out (KEYSEL),a
  in a,(KEYROW)
  bit 0,a ; Shift
  call z,scan_keyboard_shift
  call nz,scan_keyboard_noshift

  ;ld a,0
  ;call set_border_colour

  jp frameloop_lasereffect
laser_table_1:
  REPT 4,?I
    db 1<<?I
  ENDM
draw_cube:
  ld c,a ; Theta
  ld b,a ; Phi

  ld l,b
  ld a,(127*3)/2
  call multiply_8x8
  ld b,d

  ld h,sine_table>>8

  ; Cos(phi)
  ld l,b
  ld a,(hl)
  ld (iy+(cube_cp-cube)),a

  ; Sin(phi)
  ld a,b
  add a,64
  ld l,a
  ld a,(hl)
  ld (iy+(cube_sp-cube)),a

  ; Cos(theta)
  ld l,c
  ld b,(hl) ; cube_ct

  ; Sin(theta)
  ld a,c
  add a,64
  ld l,a
  ld c,(hl) ; cube_st

  ; Cos(theta)*Cos(phi)
  ld l,(iy+(cube_cp-cube))   ; 21
  ld a,b                ; 5   cube_ct
  call multiply_8x8
  ; The previous value of L is now in H.
  ld (iy+(cube_ctcp-cube)),d      ; 21

  ; Sin(theta)*Cos(phi)
  ld l,h                ; 5   cube_cp
  ld a,c                ; 5   cube_st
  call multiply_8x8
  ; Negate stcp
  ld a,d
  neg
  ld (iy+(cube_stcp-cube)),a      ; 21

  ; Cos(theta)*Sin(phi)
  ld l,(iy+(cube_sp-cube))   ; 21
  ld a,b                ; 5   cube_ct
  call multiply_8x8
  ; The previous value of L is now in H.
  ; Negate ctsp
  ld a,d
  neg
  ld (iy+(cube_ctsp-cube)),a      ; 21

  ; Sin(theta)*Sin(phi)
  ld l,h                ; 5   cube_sp
  ld a,c                ; 5   cube_st
  call multiply_8x8
  ld (iy+(cube_stsp-cube)),d      ; 21

  ; Axis 0 = { ct, -stcp, +stsp }
  ; Axis 1 = { st, +ctcp, -ctsp }
  ; Axis 2 = {  0,    sp,    cp }

  ; Determine sign of axis_0.z
  bit 7,(iy+(cube_stsp-cube))
  jr nz,$+2+1+2+1+3+2+3
  ld a,b ; Negate axis_0.x
  neg
  ld b,a
  ld a,(iy+(cube_stcp-cube)) ; Negate axis_0.y
  neg
  ld (iy+(cube_stcp-cube)),a

  ; Determine sign of axis_1.z
  bit 7,(iy+(cube_ctsp-cube))
  jr nz,$+2+1+2+1+3+2+3
  ld a,c ; Negate axis_1.x
  neg
  ld c,a
  ld a,(iy+(cube_ctcp-cube)) ; Negate axis_1.y
  neg
  ld (iy+(cube_ctcp-cube)),a

  ; Determine sign of axis_2.z
  bit 7,(iy+(cube_cp-cube))
  jr nz,$+2+3+2+3
  ld a,(iy+(cube_sp-cube)) ; Negate axis_2.y
  neg
  ld (iy+(cube_sp-cube)),a

  ld (iy+(cube_ct-cube)),b
  ld (iy+(cube_st-cube)),c

  ; Apply scale transformation to axis vectors
  ld b,8
  ld d,iyh
  ld e,iyl
  ld h,cube_scale_lut>>8
cube_scale_loop:
  ld a,(de)
  ld l,a
  ld a,(hl)
  ld (de),a
  inc de
  djnz cube_scale_loop

  ; Calculate the "front" corner of the cube
  ld b,(iy+(cube_ct-cube)) ; axis_0.x
  ld a,(iy+(cube_st-cube)) ; axis_1.x
  add a,b
  neg
  sra a ; Divide by 2
  add a,128 ; Center X
  ld (iy+(cube_corner_x-cube)),a
  ld b,(iy+(cube_stcp-cube)) ; axis_0.y
  ld c,(iy+(cube_ctcp-cube)) ; axis_1.y
  ld a,(iy+(cube_sp-cube)) ; axis_2.y
  add a,b
  add a,c
  neg
  sra a ; Divide by 2
  add a,96 ; Center X
  ld (iy+(cube_corner_y-cube)),a

draw_cube_linesonly:

  ; === Axis 0 ===

  ; Determine octant for axis 0
  ld c,0 ; C will contain octant index
  ld a,(iy+(cube_stcp-cube))
  ld d,a
  rla
  rl c
  ld a,(iy+(cube_ct-cube))
  ld e,a
  rla
  rl c

  ; Get absolute value of axis_0.x
  bit 7,d
  jr z,$+2+1+2+1
  ld a,d
  neg
  ld d,a
  ; Get absolute value of axis_0.y
  bit 7,e
  jr z,$+2+1+2+1
  ld a,e
  neg
  ld e,a

  ld a,e
  sub d
  rl c


  ; Load address of matching command in IX
  REPT 4
    sla c
  ENDM
  ld ix,line_laser_commands_ram
  ld b,0
  add ix,bc

  ; Load common parameters
  ld (ix+8),d  ; Maj
  ld (ix+10),e ; Min
  bit 4,c
  jr nz,$+2+3+3
  ld (ix+8),e  ; Maj
  ld (ix+10),d ; Min

  ld a,(iy+(cube_clr-cube))
  ld (ix+12),a ; CLR
  ld a,(iy+(cube_op-cube))
  ld (ix+14),a ; Opcode

  ; Edges parallel to axis 0

  ; Edge 00

  ld a,(iy+(cube_corner_x-cube))
  ld (ix+4),a ; DX L
  ld a,(iy+(cube_corner_y-cube))
  ld (ix+6),a ; DY L

  ld a,32
  out (VDPCTL),a
  ld a,#91
  out (VDPCTL),a
  ld d,ixh
  ld e,ixl
  ex de,hl
  call wait_for_vdp_ready
  ld c,VDPREG
  call outi_15

  ; Edge 01

  ld a,(iy+(cube_st-cube)) ; axis_1.x
  add a,(iy+(cube_corner_x-cube))
  ld (ix+4),a ; DX L
  ld a,(iy+(cube_ctcp-cube)) ; axis_1.y
  add a,(iy+(cube_corner_y-cube))
  ld (ix+6),a ; DY L

  ld a,32
  out (VDPCTL),a
  ld a,#91
  out (VDPCTL),a
  ld d,ixh
  ld e,ixl
  ex de,hl
  call wait_for_vdp_ready
  ld c,VDPREG
  call outi_15

  ; Edge 02

  ld a,(iy+(cube_corner_x-cube))
  ld (ix+4),a ; DX L
  ld a,(iy+(cube_sp-cube)) ; axis_2.y
  add a,(iy+(cube_corner_y-cube))
  ld (ix+6),a ; DY L

  ld a,32
  out (VDPCTL),a
  ld a,#91
  out (VDPCTL),a
  ld d,ixh
  ld e,ixl
  ex de,hl
  call wait_for_vdp_ready
  ld c,VDPREG
  call outi_15

  ; === Axis 1 ===

  ; Determine octant for axis 1
  ld c,0 ; B will contain octant index
  ld a,(iy+(cube_ctcp-cube)) ; axis_1.y
  ld d,a
  rla
  rl c
  ld a,(iy+(cube_st-cube)) ; axis_1.x
  ld e,a
  rla
  rl c

  ; Get absolute value of axis_1.x
  bit 7,d
  jr z,$+2+1+2+1
  ld a,d
  neg
  ld d,a
  ; Get absolute value of axis_1.y
  bit 7,e
  jr z,$+2+1+2+1
  ld a,e
  neg
  ld e,a

  ld a,e
  sub d
  rl c

  ; Load address of matching command in IX
  REPT 4
    sla c
  ENDM
  ld ix,line_laser_commands_ram
  ld b,0
  add ix,bc

  ; Load common parameters
  ld (ix+8),d  ; Maj
  ld (ix+10),e ; Min
  bit 4,c
  jr nz,$+2+3+3
  ld (ix+8),e  ; Maj
  ld (ix+10),d ; Min

  ld a,(iy+(cube_clr-cube))
  ld (ix+12),a ; CLR
  ld a,(iy+(cube_op-cube))
  ld (ix+14),a ; Opcode

  ; Edges parallel to axis 1

  ; Edge 10

  ld a,(iy+(cube_corner_x-cube))
  ld (ix+4),a ; DX L
  ld a,(iy+(cube_corner_y-cube))
  ld (ix+6),a ; DY L

  ld a,32
  out (VDPCTL),a
  ld a,#91
  out (VDPCTL),a
  ld d,ixh
  ld e,ixl
  ex de,hl
  call wait_for_vdp_ready
  ld c,VDPREG
  call outi_15

  ; Edge 11

  ld a,(iy+(cube_ct-cube)) ; axis_0.x
  add a,(iy+(cube_corner_x-cube))
  ld (ix+4),a ; DX L
  ld a,(iy+(cube_stcp-cube)) ; axis_0.y
  add a,(iy+(cube_corner_y-cube))
  ld (ix+6),a ; DY L

  ld a,32
  out (VDPCTL),a
  ld a,#91
  out (VDPCTL),a
  ld d,ixh
  ld e,ixl
  ex de,hl
  call wait_for_vdp_ready
  ld c,VDPREG
  call outi_15

  ; Edge 12

  ld a,(iy+(cube_corner_x-cube))
  ld (ix+4),a ; DX L
  ld a,(iy+(cube_sp-cube)) ; axis_2.y
  add a,(iy+(cube_corner_y-cube))
  ld (ix+6),a ; DY L

  ld a,32
  out (VDPCTL),a
  ld a,#91
  out (VDPCTL),a
  ld d,ixh
  ld e,ixl
  ex de,hl
  call wait_for_vdp_ready
  ld c,VDPREG
  call outi_15


  ; === Axis 2 ===

  ; Determine octant for axis 2
  ld c,0 ; B will contain octant index
  ld a,(iy+(cube_sp-cube)) ; axis_2.y
  ld d,a
  rla
  rl c
  ld a,0 ; axis_2.x
  ld e,a
  rla
  rl c

  ; Get absolute value of axis_2.x
  bit 7,d
  jr z,$+2+1+2+1
  ld a,d
  neg
  ld d,a
  ; Get absolute value of axis_2.y
  bit 7,e
  jr z,$+2+1+2+1
  ld a,e
  neg
  ld e,a

  ld a,e
  sub d
  rl c

  ; Load address of matching command in IX
  REPT 4
    sla c
  ENDM
  ld ix,line_laser_commands_ram
  ld b,0
  add ix,bc

  ; Load common parameters
  ld (ix+8),d  ; Maj
  ld (ix+10),e ; Min
  bit 4,c
  jr nz,$+2+3+3
  ld (ix+8),e  ; Maj
  ld (ix+10),d ; Min

  ld a,(iy+(cube_clr-cube))
  ld (ix+12),a ; CLR
  ld a,(iy+(cube_op-cube))
  ld (ix+14),a ; Opcode

  ; Edges parallel to axis 2

  ; Edge 20

  ld a,(iy+(cube_corner_x-cube))
  ld (ix+4),a ; DX L
  ld a,(iy+(cube_corner_y-cube))
  ld (ix+6),a ; DY L

  ld a,32
  out (VDPCTL),a
  ld a,#91
  out (VDPCTL),a
  ld d,ixh
  ld e,ixl
  ex de,hl
  call wait_for_vdp_ready
  ld c,VDPREG
  call outi_15

  ; Edge 21

  ld a,(iy+(cube_ct-cube)) ; axis_0.x
  add a,(iy+(cube_corner_x)-cube)
  ld (ix+4),a ; DX L
  ld a,(iy+(cube_stcp-cube)) ; axis_0.y
  add a,(iy+(cube_corner_y-cube))
  ld (ix+6),a ; DY L

  ld a,32
  out (VDPCTL),a
  ld a,#91
  out (VDPCTL),a
  ld d,ixh
  ld e,ixl
  ex de,hl
  call wait_for_vdp_ready
  ld c,VDPREG
  call outi_15

  ; Edge 22

  ld a,(iy+(cube_st-cube)) ; axis_1.x
  add a,(iy+(cube_corner_x-cube))
  ld (ix+4),a ; DX L
  ld a,(iy+(cube_ctcp-cube)) ; axis_1.y
  add a,(iy+(cube_corner_y-cube))
  ld (ix+6),a ; DY L

  ld a,32
  out (VDPCTL),a
  ld a,#91
  out (VDPCTL),a
  ld d,ixh
  ld e,ixl
  ex de,hl
  call wait_for_vdp_ready
  ld c,VDPREG
  call outi_15

  ret

check_password:
  ld hl,current_level_index
  ld (hl),0
  ld b,NUMBER_OF_LEVELS
check_password_loop:
  ld hl,string_offsets+2
  ld a,(current_level_index)
  rlca
  ld d,0
  ld e,a
  add hl,de
  ld e,(hl)
  inc hl
  ld d,(hl)
  REPT 4
    inc de
  ENDM
  call check_single_password
  ret nc
  ld hl,current_level_index
  inc (hl)
  djnz check_password_loop
  ld hl,current_level_index
  ld (hl),0
  scf
  ret

check_single_password:
  ld hl,password_type_buffer
check_single_password_loop:
  ld a,(de)
  cp (hl)
  scf
  ret nz
  cp TEXT_QUOTE_CHARACTER
  jr z,check_single_password_end
  inc de
  inc hl
  jr check_single_password_loop
check_single_password_end:
  scf
  ccf
  ret

scan_keyboard_sample_row:
  push bc
  ld hl,keyboard_states
  ld b,0
  ld c,a
  add hl,bc
  in a,(KEYSEL)
  and #F0
  or c
  out (KEYSEL),a
  in a,(KEYROW)
  xor #FF
  ld b,a
  xor (hl)
  and b
  ld (hl),b
  pop bc
  ret

out_spritetext_character:
  push bc
  bit 6,a
  jp nz,out_spritetext_character_space
  ld hl,font_graphic
  bit 5,a
  jr z,$+2+3+1
  ld bc,32*16
  add hl,bc
  and 31
  ld b,0
  ld c,a
  add hl,bc
  REPT 16
    ld a,(hl)
    out (VDPDAT),a
    ld bc,32
    add hl,bc
  ENDM
  pop bc
  ret
out_spritetext_character_space:
  xor a
  REPT 16
    out (VDPDAT),a
    nop
    nop
  ENDM
  pop bc
  ret

; 0123456789ABCDEFGHIJKLMNOPQRSTUV
; WXYZabcdefghijklmnopqrstuvwxyz".
scan_keyboard_shift:
  push af
  ld d,10

  ; A, B
  ld a,2 ; Row 2
  call scan_keyboard_sample_row
  REPT 6
    rrca
  ENDM
  rrca          ; 1B
  jr c,type_key ; 2B
  inc d         ; 1B
  rrca          ; 1B
  jr c,type_key ; 2B
  inc d         ; 1B

  ; C, D, E, F, G, H, I, J
  ld a,3 ; Row 3
  call scan_keyboard_sample_row
  ld b,8
  rrca          ; 1B
  jr c,type_key ; 2B
  inc d         ; 1B
  djnz $-1-2-1

  ; K, L, M, N, O, P, Q, R
  ld a,4 ; Row 4
  call scan_keyboard_sample_row
  ld b,8
  rrca          ; 1B
  jr c,type_key ; 2B
  inc d         ; 1B
  djnz $-1-2-1

  ; S, T, U, V, W, X, Y, Z
  ld a,5 ; Row 5
  call scan_keyboard_sample_row
  ld b,8
  rrca          ; 1B
  jr c,type_key ; 2B
  inc d         ; 1B
  djnz $-1-2-1

  pop af
  ret

type_key:
  ld hl,password_length
  ld a,(hl)
  cp 16
  ret z
  ld hl,(password_type_addr)
  ld (hl),d
  inc hl
  ld (password_type_addr),hl
  ; Draw a password text character
  ld a,d
  call out_spritetext_character
  ld hl,password_length
  inc (hl)
  ret

; 0123456789ABCDEFGHIJKLMNOPQRSTUV
; WXYZabcdefghijklmnopqrstuvwxyz".
scan_keyboard_noshift:
  push af

  ld d,0

  ; 0, 1, 2, 3, 4, 5, 6, 7
  ld a,0 ; Row 0
  call scan_keyboard_sample_row
  ld b,8
  rrca          ; 1B
  jr c,type_key ; 2B
  inc d         ; 1B
  djnz $-1-2-1

  ; 8, 9
  ld a,1 ; Row 1
  call scan_keyboard_sample_row
  rrca          ; 1B
  jr c,type_key ; 2B
  inc d         ; 1B
  rrca          ; 1B
  jr c,type_key ; 2B
  inc d         ; 1B

  ld d,36

  ; A, B
  ld a,2 ; Row 2
  call scan_keyboard_sample_row
  REPT 6
    rrca
  ENDM
  rrca          ; 1B
  jr c,type_key ; 2B
  inc d         ; 1B
  rrca          ; 1B
  jr c,type_key ; 2B
  inc d         ; 1B

  ; C, D, E, F, G, H, I, J
  ld a,3 ; Row 3
  call scan_keyboard_sample_row
  ld b,8
  rrca          ; 1B
  jr c,type_key ; 2B
  inc d         ; 1B
  djnz $-1-2-1

  ; K, L, M, N, O, P, Q, R
  ld a,4 ; Row 4
  call scan_keyboard_sample_row
  ld b,8
  rrca          ; 1B
  jr c,type_key ; 2B
  inc d         ; 1B
  djnz $-1-2-1

  ; S, T, U, V, W, X, Y, Z
  ld a,5 ; Row 5
  call scan_keyboard_sample_row
  ld b,8
  rrca          ; 1B
  jr c,type_key ; 2B
  inc d         ; 1B
  djnz $-1-2-1

  ; Space bar
  ld a,8
  call scan_keyboard_sample_row
  rrca
  ld d,64
  jr c,type_key

  pop af
  ret


; Multiply L by A and put result in DE.
multiply_8x8:
  ld h,#FF
  bit 7,l
  jr nz,$+2+2
  ld h,#00
  ld de,#0000
  REPT 8,?I
    add hl,hl         ; 12
    rra               ; 5
    IF ?I=7
      ret nc          ; 12/6
      ex de,hl        ; 5
      and a           ; 5
      sbc hl,de       ; 12
      ex de,hl        ; 5
    ELSE
      jr nc,$+2+1+1+1 ; 13/8
      ex de,hl        ; 5
      add hl,de       ; 12
      ex de,hl        ; 5
    ENDIF
  ENDM
  ret

  ; *******************************************************************************
  ; In-game
begin_game:
  ld bc,#0000
  ld a,#00
  out (VDPCTL),a
  ld a,#90
  out (VDPCTL),a
  call wait_for_vblank
  di

  call silence_opll

  ; Set palette registers
  ld a,#00
  out (VDPCTL),a
  ld a,#90
  out (VDPCTL),a
  ld hl,black_palette
  ld c,VDPPAL
  call outi_32

  ; Set GRAPHIC4 mode and blank the screen
  ; M5 M4 M3 M2 M1
  ;  0  1  1  0  0
  ld a,VDP_SI
  out (VDPCTL),a
  ld a,#81
  out (VDPCTL),a
  ld a,VDP_M3|VDP_M4
  out (VDPCTL),a
  ld a,#80
  out (VDPCTL),a
  ld a,VDP_VR|VDP_TP
  out (VDPCTL),a
  ld a,#88
  out (VDPCTL),a
  ld a,0
  out (VDPCTL),a
  ld a,#89
  out (VDPCTL),a

  ld a,%00000000
  out (VDPCTL),a
  ld a,128+45
  out (VDPCTL),a

  ld a,#02
  call set_border_colour

  ; Pattern layout
  ld a,(BASE_PATTERN_LAYOUT>>10)|%00011111
  out (VDPCTL),a
  ld a,#82
  out (VDPCTL),a
  ; Sprite attributes
  ld a,((BASE_SPRITE_ATTRIBUTES>>7)&#FF)|7
  out (VDPCTL),a
  ld a,#85
  out (VDPCTL),a
  ld a,BASE_SPRITE_ATTRIBUTES>>15
  out (VDPCTL),a
  ld a,#8B
  out (VDPCTL),a
  ; Sprite patterns
  ld a,BASE_SPRITE_PATTERNS>>11
  out (VDPCTL),a
  ld a,#86
  out (VDPCTL),a

  ; Clear the framebuffer
  call wait_for_vdp_ready
  ld a,32
  out (VDPCTL),a
  ld a,#91
  out (VDPCTL),a
  ld hl,hmmv_clear_ingame_vram
  ld c,VDPREG
  call outi_15

  call init_ingame

  ; Prepare the pre-composited digits for use in the stroke counter
  call wait_for_vdp_ready
  ld a,32
  out (VDPCTL),a
  ld a,#91
  out (VDPCTL),a
  ld hl,hmmv_clear_stroke_counter_digits
  ld c,VDPREG
  call outi_15
  ; Draw the stroke counter
  ; Note that the counter digit ordinals and the digits themselves match up.
  ld a,128 ; String terminator symbol
  ld (stroke_counter_digits+10),a
  ld hl,stroke_counter_digits
  ld b,10
  xor a
  ld (hl),a
  inc a
  inc hl
  djnz $-1-1-1

  ; Adjust the DY register for these commands.
  ld ix,lmmm_text_xor_ram
  ld (ix+6),FONT_Y_POS-16+TEXT_SHADOW_Y_OFFSET
  ld ix,lmmm_text_or_ram
  ld (ix+6),FONT_Y_POS-16

  ld hl,stroke_counter_digits
  ld b,0
  call blit_text_string_G4
  xor a
  ld (stroke_counter_digits+0),a
  ld (stroke_counter_digits+1),a
  ld a,128 ; String terminator symbol
  ld (stroke_counter_digits+2),a

  ; Restore the DY register for these commands.
  ld ix,lmmm_text_xor_ram
  ld (ix+6),TEXT_Y_POS+TEXT_SHADOW_Y_OFFSET
  ld ix,lmmm_text_or_ram
  ld (ix+6),TEXT_Y_POS


  ; Upload wall graphics
  ld hl,hmmc_wall_graphics
  ld de,wall_graphics
  ld b,(15*8*8/2)/16
  call execute_hmmc_x16

  ; Upload flag sprite patterns
  ld hl,hmmc_flag_patterns
  ld de,flag_graphics
  ld b,(FLAG_STORED_PATTERN_COUNT*8)/16
  call execute_hmmc_x16

  ; Upload flagpole graphic
  ld hl,hmmc_flagpole_graphic
  ld de,flagpole_graphics
  ld b,(128*8/2)/16
  call execute_hmmc_x16

  ; Blit the metatiles, first page
  ; Copy wall tile blit commands to RAM
  ld hl,hmmm_wall_graphics_1
  ld de,hmmm_wall_graphics_ram
  ld bc,15*16
  ldir
  ld hl,metatiles
  ld bc,#0000
metatiles_y_loop:
metatiles_x_loop:
  ld a,(hl)
  inc hl
  REPT 4
    rlca
  ENDM
  ld d,0
  ld e,a
  ld ix,hmmm_wall_graphics_ram
  add ix,de
  ld (ix+4),b ; DX L
  ld (ix+6),c ; DY L
  ; Blit
  call wait_for_vdp_ready
  ld a,32
  out (VDPCTL),a
  ld a,#91
  out (VDPCTL),a
  REPT 15,?I
    ld a,(ix+?I)
    out (VDPREG),a
  ENDM
  ld a,8
  add a,b
  ld b,a
  jr nz,metatiles_x_loop
  ld a,8
  add a,c
  ld c,a
  jr nz,metatiles_y_loop

  ; Blit the metatiles, second page
  ; Copy wall tile blit commands to RAM
  ld hl,hmmm_wall_graphics_2
  ld de,hmmm_wall_graphics_ram
  ld bc,15*16
  ldir
  ld hl,metatiles+1024
  ld bc,#0000
metatiles_y_loop2:
metatiles_x_loop2:
  ld a,(hl)
  inc hl
  REPT 4
    rlca
  ENDM
  ld d,0
  ld e,a
  ld ix,hmmm_wall_graphics_ram
  add ix,de
  ld (ix+4),b ; DX L
  ld (ix+6),c ; DY L
  ; Blit
  call wait_for_vdp_ready
  ld a,32
  out (VDPCTL),a
  ld a,#91
  out (VDPCTL),a
  REPT 15,?I
    ld a,(ix+?I)
    out (VDPREG),a
  ENDM
  ld a,8
  add a,b
  ld b,a
  jr nz,metatiles_x_loop2
  ld a,8
  add a,c
  ld c,a
  jr nz,metatiles_y_loop2

  ; Upload sprite colours
  LDIRVM_FIXED sprite_colours, BASE_SPRITE_COLOURS, sprite_colours_end-sprite_colours
  ; Upload sprite attributes
  LDIRVM_FIXED sprite_attributes, BASE_SPRITE_ATTRIBUTES, 128
  ; Upload sprite patterns
  LDIRVM_FIXED sprite_patterns_start, BASE_SPRITE_PATTERNS, sprite_patterns_end-sprite_patterns_start

  ; Write R#19 (interrupt line register)
  ld a,128
  out (VDPCTL),a
  ld a,128+19
  out (VDPCTL),a

  ld bc,#0000
  ; Set up for P#0
  ld a,#00
  out (VDPCTL),a
  ld a,#90
  out (VDPCTL),a

  ; Enable screen display
  ld a,VDP_M3|VDP_M4|VDP_IE1
  out (VDPCTL),a
  ld a,#80
  out (VDPCTL),a
  ld a,VDP_SI|VDP_IE0|VDP_BL
  out (VDPCTL),a
  ld a,#81
  out (VDPCTL),a

  ; Initialise handler and transition addresses
  ld hl,frame_handler_nop
  ld (frame_handler_address),hl
  ld hl,transition_to_next_level
  ld (transition_address),hl

  ; Switch to interrupt mode 2 (custom interrupt handler)
  ld a,#BE
  ld i,a
  im 2

  ; The main game loop
frameloop:

  ;ld a,#06
  ;call set_border_colour

  ld bc,#0000
  ; Set up for P#0
  ld a,#00
  out (VDPCTL),a
  ld a,#90
  out (VDPCTL),a
  call wait_for_vblank

  ;ld a,#02
  ;call set_border_colour


  IF DEBUG_ENABLED
    ; Set the default border colour
    ld a,#02
    call set_border_colour
  ENDIF

  ; This is a hack...
  ld hl,blit_stroke_counter_flag
  ld (hl),0

  ; Perform an indirect call for frame logic (and rendering)
  ld hl,(frame_handler_address)
  ld de,$+3+1+1
  push de
  jp (hl)

  ld hl,frame_counter
  inc (hl)

  ; Perform an indirect call for game state transition
  ld hl,(transition_address)
  ld de,$+3+1+1
  push de
  jp (hl)

  jr frameloop

init_ingame:
  di

  ; Set palette registers
  ld a,#00
  out (VDPCTL),a
  ld a,#90
  out (VDPCTL),a
  ld hl,black_palette
  ld c,VDPPAL
  call outi_32

  ; Set GRAPHIC4 mode and blank the screen
  ; M5 M4 M3 M2 M1
  ;  0  1  1  0  0
  ld a,VDP_SI
  out (VDPCTL),a
  ld a,#81
  out (VDPCTL),a
  ld a,VDP_M3|VDP_M4
  out (VDPCTL),a
  ld a,#80
  out (VDPCTL),a
  ld a,VDP_VR|VDP_TP
  out (VDPCTL),a
  ld a,#88
  out (VDPCTL),a
  ld a,0
  out (VDPCTL),a
  ld a,#89
  out (VDPCTL),a

  ld a,#02
  call set_border_colour

  ; Pattern layout
  ld a,(BASE_PATTERN_LAYOUT>>10)|%00011111
  out (VDPCTL),a
  ld a,#82
  out (VDPCTL),a
  ; Sprite attributes
  ld a,((BASE_SPRITE_ATTRIBUTES>>7)&#FF)|7
  out (VDPCTL),a
  ld a,#85
  out (VDPCTL),a
  ld a,BASE_SPRITE_ATTRIBUTES>>15
  out (VDPCTL),a
  ld a,#8B
  out (VDPCTL),a
  ; Sprite patterns
  ld a,BASE_SPRITE_PATTERNS>>11
  out (VDPCTL),a
  ld a,#86
  out (VDPCTL),a

  ; Save the current level index, since this clears all 8kb of the used RAM.
  ld a,(current_level_index)
  push af

  ; Unpack font and copy it to VRAM
  call unpack_font_G4

  ; Reset the area of RAM that was used for temporary font unpacking
  ld a,(current_level_index)
  ld hl,font_unpack_temp
  ld (hl),0
  ld de,font_unpack_temp+1
  ld bc,font_unpack_temp_end-font_unpack_temp-1
  ldir

  pop af
  ld (current_level_index),a

  ld hl,hmmm_32x8_metatile
  ld de,hmmm_32x8_metatile_ram
  ld bc,16
  ldir

  ld hl,hmmm_64x8_metatile_flagpole
  ld de,hmmm_64x8_metatile_flagpole_ram
  ld bc,16
  ldir

  ld hl,lmmm_text_xor
  ld de,lmmm_text_xor_ram
  ld bc,16
  ldir

  ld hl,lmmm_text_or
  ld de,lmmm_text_or_ram
  ld bc,16
  ldir

  ld hl,hmmm_stroke_counter_digit
  ld de,hmmm_stroke_counter_digit_ram
  ld bc,16
  ldir

  ld hl,sprite_attributes_init
  ld de,sprite_attributes
  ld bc,128
  ldir

  ; Initialise music playback
  ld hl,(music_addresses+0)
  call init_music

  ret


scroll_down:
  push af
  ld hl,(scroll_position)
  ld bc,3
  add hl,bc
  ld bc,1023-192
  and a
  ld d,h
  ld e,l
  sbc hl,bc
  ld h,d
  ld l,e
  jr c,$+2+3
  ld hl,1023-192
  ld (scroll_position),hl
  xor a ; Down
  ld (scroll_direction),a
  pop af
  ret

scroll_up:
  push af
  ld hl,(scroll_position)
  ld bc,3
  and a
  sbc hl,bc
  jr nc,$+2+3
  ld hl,0
  ld (scroll_position),hl
  ld a,1 ; Up
  ld (scroll_direction),a
  pop af
  ret

fixup_sprite_y_position:
  cp #D8
  ret nz
  ld a,#D7
  ret

do_rendering:
  ; Set scroll registers
  ; NOTE: Scroll position affects sprite position!!
  ld hl,scroll_position+0
  ld a,(hl)
  out (VDPCTL),a
  ld a,128+23
  out (VDPCTL),a

  ; Set pattern layout base address
  ld a,(BASE_PATTERN_LAYOUT>>10)|%00011111
  out (VDPCTL),a
  ld a,#82
  out (VDPCTL),a

  ; Sprites
  ; First populate all of the sprites and then blank out the ones which are out of view.
  ld hl,sprite_attributes
  ; Define the four sprites which will make up the ball
  ; Sprite #00
  ; Bitplane 1
  ld a,(ball_py_current+1)
  sub 8
  call fixup_sprite_y_position
  ld (hl),a ; Y
  inc l
  ld a,(ball_px_current+1)
  sub 8
  ld (hl),a ; X
  inc l
  ld a,4*0
  ld (hl),a ; Pattern
  inc l
  xor a
  ld (hl),a ; Reserved
  inc l
  ; Sprite #01
  ; Bitplane 2
  ld a,(ball_py_current+1)
  sub 8
  call fixup_sprite_y_position
  ld (hl),a ; Y
  inc l
  ld a,(ball_px_current+1)
  sub 8
  ld (hl),a ; X
  inc l
  ld a,4*1
  ld (hl),a ; Pattern
  inc l
  xor a
  ld (hl),a ; Reserved
  inc l
  ; Sprite #02
  ; Bitplane 3
  ld a,(ball_py_current+1)
  sub 8
  call fixup_sprite_y_position
  ld (hl),a ; Y
  inc l
  ld a,(ball_px_current+1)
  sub 8
  ld (hl),a ; X
  inc l
  ld a,4*2
  ld (hl),a ; Pattern
  inc l
  xor a
  ld (hl),a ; Reserved
  inc l
  ; Sprite #03
  ; Bitplane 4
  ld a,(ball_py_current+1)
  sub 8
  call fixup_sprite_y_position
  ld (hl),a ; Y
  inc l
  ld a,(ball_px_current+1)
  sub 8
  ld (hl),a ; X
  inc l
  ld a,4*3
  ld (hl),a ; Pattern
  inc l
  xor a
  ld (hl),a ; Reserved
  inc l
  ; Sprite #04
  ; Batter sprite
  ld a,(batter_py+1)
  sub 8
  call fixup_sprite_y_position
  ld (hl),a ; Y
  inc l
  ld a,(batter_px+1)
  sub 8
  ld (hl),a ; X
  inc l
  ld a,(batter_pattern)
  ld (hl),a ; Pattern
  inc l
  xor a
  ld (hl),a ; Reserved
  inc l
  ; Sprite #05
  ; Batter outline sprite
  ld a,(batter_py+1)
  sub 8
  call fixup_sprite_y_position
  ld (hl),a ; Y
  inc l
  ld a,(batter_px+1)
  sub 8
  ld (hl),a ; X
  inc l
  ld a,(batter_pattern)
  add a,4
  ld (hl),a ; Pattern
  inc l
  xor a
  ld (hl),a ; Reserved
  inc l
  ; Define the 2 sprites which will make up the flag
  ; Sprite #06
  ; White parts
  ld a,(flag_py+0)
  call fixup_sprite_y_position
  ld (hl),a ; Y
  inc l
  ld a,(flag_px+0)
  ld (hl),a ; X
  inc l
  ld a,FLAG_CURRENT_PATTERN_INDEX+0
  ld (hl),a ; Pattern
  inc l
  xor a
  ld (hl),a ; Reserved
  inc l
  ; Sprite #07
  ; Black parts
  ld a,(flag_py+0)
  call fixup_sprite_y_position
  ld (hl),a ; Y
  inc l
  ld a,(flag_px+0)
  ld (hl),a ; X
  inc l
  ld a,FLAG_CURRENT_PATTERN_INDEX+4
  ld (hl),a ; Pattern
  inc l
  xor a
  ld (hl),a ; Reserved
  inc l

  ; Sprite #08
  ld (hl),#D8 ; Y (disabled)

  ; Blank out those sprites which are not in view
  ld ix,sprite_attributes
  ; Check whether or not ball is in visible region
  ld hl,(scroll_position)
  ex de,hl
  ld hl,(ball_py_current+1)
  and a ; Reset carry flag
  sbc hl,de
  ld de,16
  add hl,de
  ld a,h
  cp #00
  ld hl,sprite_attributes
  jr z,skip_ball_sprite
  ld (ix+#00*4+2),BLANK_SPRITE_PATTERN
  ld (ix+#01*4+2),BLANK_SPRITE_PATTERN
  ld (ix+#02*4+2),BLANK_SPRITE_PATTERN
  ld (ix+#03*4+2),BLANK_SPRITE_PATTERN
  ld (ix+#04*4+2),BLANK_SPRITE_PATTERN
  ld (ix+#05*4+2),BLANK_SPRITE_PATTERN
skip_ball_sprite:

  ; Check whether or not flag is in visible region
  ld hl,(scroll_position)
  ex de,hl
  ld hl,(flag_py)
  and a ; Reset carry flag
  sbc hl,de
  ld de,16
  add hl,de
  ld a,h
  cp #00
  jr z,skip_flag_sprite
  ld (ix+#06*4+2),BLANK_SPRITE_PATTERN
  ld (ix+#07*4+2),BLANK_SPRITE_PATTERN
skip_flag_sprite:

  ; Upload sprite attributes
  LDIRVM_FIXED sprite_attributes, BASE_SPRITE_ATTRIBUTES, 4*9

  ; Leftmost metatile of scrolling front
  ld hl,(scroll_position)
  ld a,l
  and #F0
  sra h
  rra
  ld l,a
  ld bc,metatile_map+(#C0/16)*8
  ld a,(scroll_direction)
  and 1
  jr z,$+2+3
  ld bc,metatile_map-8
  add hl,bc
  ld d,(hl)

  ld bc,#00C0
  jr z,$+2+3
  ld bc,#00F0
  ld a,(scroll_position+0)
  and #F8
  add a,c
  ld c,a
  ld a,d
  ld ix,hmmm_32x8_metatile_ram
  call blit_32x8_metatile

  ; Set up for P#0
  ld a,#00
  out (VDPCTL),a
  ld a,#90
  out (VDPCTL),a

  ; Offset the rasterbar palette base index
  ld a,(palette_fade_counter+1)
  ; Clamp to [0,7]
  cp 8
  jr nz,$+2+2
  ld a,7
  cp -1
  jr nz,$+2+2
  ld a,0
  ld h,0
  ld l,a
  ld de,#0000
  ; Multiply HL by 28=4+8+16
  REPT 2
    add hl,hl
  ENDM
  ex de,hl
  add hl,de
  ex de,hl
  add hl,hl
  ex de,hl
  add hl,de
  ex de,hl
  add hl,hl
  ex de,hl
  add hl,de
  ld bc,rasterbar_palette
  add hl,bc

  ; Write P#0
  ld a,(hl)
  out (VDPPAL),a
  inc hl
  ld a,(hl)
  out (VDPPAL),a
  inc hl
  ; Write P#1
  ld a,(hl)
  out (VDPPAL),a
  inc hl
  ld a,(hl)
  out (VDPPAL),a
  inc hl

  ; Write R#19 (interrupt line register)
  ld a,16-1
  ld de,scroll_position+0
  ex de,hl
  add a,(hl)
  ex de,hl
  out (VDPCTL),a
  ld a,128+19
  out (VDPCTL),a

  ; Save HL
  push hl

  ; Update music playback
  call music_playback_update
  ; Play SFX
  ld hl,ball_bounce_sfx_flags
  bit BALL_SFX_FLAG_BOUNCE_BIT,(hl)
  call nz,trigger_rhythm_0
  bit BALL_SFX_FLAG_HIT_BIT,(hl)
  call nz,trigger_rhythm_3
  bit BALL_SFX_FLAG_SWING_BIT,(hl)
  call nz,trigger_rhythm_1
  ld (hl),#00

  ld hl,blit_stroke_counter_flag
  ld a,1
  cp (hl)
  ; This section is conditionally skipped to free up cycles.
  jr nz,skip_blitting_stroke_counter

  ; Draw the stroke counter.
  ; First digit
  ld ix,hmmm_stroke_counter_digit_ram
  ld a,(stroke_counter_digits+0)
  REPT 3
    rlca
  ENDM
  ld (ix+0),a ; SX L
  ld a,TEXT_STROKE_COUNTER_X
  ld (ix+4),a ; DX L
  call wait_for_vdp_ready
  ld a,32
  out (VDPCTL),a
  ld a,#91
  out (VDPCTL),a
  ld hl,hmmm_stroke_counter_digit_ram
  ld c,VDPREG
  call outi_15
  ; Second digit
  ld ix,hmmm_stroke_counter_digit_ram
  ld a,(stroke_counter_digits+1)
  REPT 3
    rlca
  ENDM
  ld (ix+0),a ; SX L
  ld a,TEXT_STROKE_COUNTER_X+8
  ld (ix+4),a ; DX L
  call wait_for_vdp_ready
  ld a,32
  out (VDPCTL),a
  ld a,#91
  out (VDPCTL),a
  ld hl,hmmm_stroke_counter_digit_ram
  ld c,VDPREG
  call outi_15

skip_blitting_stroke_counter:
  ld hl,blit_stroke_counter_flag
  ld (hl),0
  pop hl

  ; Enable horizontal retrace interrupt
  ld a,%00010110
  out (VDPCTL),a
  ld a,#80
  out (VDPCTL),a

  ; Turn sprites on
  ld a,%00101000
  out (VDPCTL),a
  ld a,#88
  out (VDPCTL),a

  ; Blit the scrolling front
  REPT 10, ?I
    IF ?I&1
      ; Set up for P#0
      ld a,#00
      out (VDPCTL),a
      ld a,#90
      out (VDPCTL),a
    ELSE
      ; Set up for P#1
      ld a,#01
      out (VDPCTL),a
      ld a,#90
      out (VDPCTL),a
    ENDIF

    IF ?I>0 && ?I<8
      ; Blit a single metatile
      push hl
      ld hl,(scroll_position)
      ld a,l
      and #F0
      sra h
      rra
      or ?I
      ld l,a

      ld bc,metatile_map+(#C0/16)*8
      ld a,(scroll_direction)
      and 1
      jr z,$+2+3
      ld bc,metatile_map-8

      add hl,bc
      ld d,(hl)

      ld bc,#00C0+#2000*?I
      jr z,$+2+3
      ld bc,#00F0+#2000*?I
      ld a,(scroll_position+0)
      and #F8
      add a,c
      ld c,a
      ld a,d
      ld ix,hmmm_32x8_metatile_ram
      call blit_32x8_metatile
      pop hl
    ENDIF

    IF ?I=8
      ; Update flag sprite graphics
      push hl
      ld a,(frame_counter)
      and FLAG_FRAME_COUNT-1
      ld h,0
      ld l,a
      ; Multiply by 16
      REPT 4
        add hl,hl
      ENDM
      ld bc,flag_pattern_ymmms_start
      add hl,bc
      ; Blit the flag sprite patterns
      call wait_for_vdp_ready
      ld a,32
      out (VDPCTL),a
      ld a,#91
      out (VDPCTL),a
      ld c,VDPREG
      call outi_15
      pop hl
    ENDIF

    ld b,(hl)
    inc hl
    ld c,(hl)
    inc hl

    IF DEBUG_ENABLED
      ld a,#06
      call set_border_colour
    ENDIF

    ; Wait for the next rasterbar transition point
    halt

    IF DEBUG_ENABLED
      ld a,#04
      call set_border_colour
    ENDIF

    IF ?I<9
      ; Write R#19 (interrupt line register)
      ; For the next rasterbar transition point
      ld a,(?I+2)*16-1
      ld de,scroll_position+0
      ex de,hl
      add a,(hl)
      ex de,hl
      out (VDPCTL),a
      ld a,128+19
      out (VDPCTL),a
    ENDIF
  ENDM

  ; Write R#19 (interrupt line register)
  ; For the statusbar
  ld a,TEXT_Y_POS-4
  ld de,scroll_position+0
  ex de,hl
  add a,(hl)
  ex de,hl
  out (VDPCTL),a
  ld a,128+19
  out (VDPCTL),a

  ; Set up statusbar split
  ; Set up for P#0
  ld a,#00
  out (VDPCTL),a
  ld a,#90
  out (VDPCTL),a
  ld hl,%00100010 | (#81 << 8)
  ; Wait for statusbar split (B and C should be unchanged from previous halt)
  halt
  ld c,VDPCTL
  ; Deal with an annoying interaction between registers #23 and #19.
  out (c),l
  out (c),h
  ld hl,%00000110 | (#80 << 8)
  out (c),l
  out (c),h
  ; Turn sprites off
  ld hl,%00101010 | (#88 << 8)
  out (c),l
  out (c),h
  ; Set scroll register to zero
  ld hl,0 | ((128+23) << 8)
  out (c),l
  out (c),h
  ; Set pattern layout base address to base of page 1
  ld hl,(BASE_PATTERN_LAYOUT_PAGE_1_G4>>10)|%00011111 | (#82 << 8)
  out (c),l
  out (c),h

  ; Re-enable display
  ld hl,%01100010 | (#81 << 8)
  out (c),l
  out (c),h

  ret

do_physics:
  ld hl,ball_px_current
  ld de,ball_px_stopped_previous ; This needs to be renamed..
  ld bc,6
  ldir

  ld b,5
do_physics_loop:
  push bc
  call verlet_iterate_ball
  call calc_ball_velocity
  call check_collision
  pop bc
  djnz do_physics_loop

  ld a,(ball_px_current+1)
  ld b,a
  ld a,(ball_px_stopped_previous+1)
  xor b
  jr nz,reset_ball_stopped_counter

  ld a,(ball_py_current+2)
  ld b,a
  ld a,(ball_py_stopped_previous+2)
  xor b
  jr nz,reset_ball_stopped_counter
  ld a,(ball_py_current+1)
  ld b,a
  ld a,(ball_py_stopped_previous+1)
  xor b
  jr nz,reset_ball_stopped_counter

  ; Update the ball stopped counter
  ld hl,ball_stopped_counter
  inc (hl)
  ld a,(hl)
  cp 30
  call nc,check_ball_stopped_state
  ret
reset_ball_stopped_counter:
  ; Reset
  xor a
  ld (ball_stopped_counter),a
  ret

check_ball_stopped_state:
  ld hl,transition_inflight_to_swingready
  ld (transition_address),hl
  ; Check to see if the ball has stopped at the flag (i.e. goal reached)
  ld hl,(ball_px_current+1) ; X
  ; Divide HL by 16
  REPT 4
    sra h
    rr l
  ENDM
  ld a,l
  ld hl,goal_map_position+0
  cp (hl)
  ret nz
  ld hl,(ball_py_current+1) ; Y
  ; Divide HL by 16
  REPT 4
    sra h
    rr l
  ENDM
  ld a,l
  ld hl,goal_map_position+1
  cp (hl)
  call z,skip_to_next_level
  ret

frame_handler_swingready:
  xor a
  ld (batter_swing_level),a

  ; Set the batter's pattern
  ld a,(batter_patterns-sprite_patterns_start)/8
  ld (batter_pattern),a

  ld hl,blit_stroke_counter_flag
  ld (hl),1

  ; Render the game
  call do_rendering
  ; Update the scroll position
  call update_scroll_position
  ; Handle keys
  in a,(KEYSEL)
  and #F0
  or 6 ; Select row 6
  out (KEYSEL),a
  in a,(KEYROW)
  bit 5,a ; F1
  call z,skip_to_next_level
  bit 6,a ; F2
  jp z,begin_titlescreen
  bit 7,a ; F3
  call z,restart_current_level
  bit 0,a ; Shift
  jr nz,$+2+3+3
  ld hl,transition_swingready_to_observing
  ld (transition_address),hl
  bit 1,a ; Ctrl
  ld hl,input_leftright_gate
  call nz,reset_input_leftright_gate
  ; Handle inputs
  in a,(KEYSEL)
  and #F0
  or 8 ; Select row 8
  out (KEYSEL),a
  in a,(KEYROW)
  ld b,a
  ld a,(hl)
  and a
  jr nz,frame_handler_swingready_skip_leftright
  bit 7,b ; Right
  call z,right_pressed
  bit 4,b ; Left
  call z,left_pressed
  bit 0,b ; Space
  jr nz,$+2+3+3
  ld hl,transition_swingready_to_swinging
  ld (transition_address),hl
frame_handler_swingready_skip_leftright:
  bit 7,b ; Right
  ret z
  bit 4,b ; Left
  call nz,reset_input_leftright_gate
  ret
reset_input_leftright_gate:
  ld hl,input_leftright_gate
  ld (hl),0
  ret
transition_swingready_to_swinging:
  ; Reset the accumulated velocity
  ld hl,accumulated_swing_velocity_x
  ld (hl),0
  ld de,accumulated_swing_velocity_x+1
  ld bc,5
  ldir

  ; Set pending addresses
  ld hl,frame_handler_swinging
  ld (frame_handler_address),hl
  ld hl,transition_nop
  ld (transition_address),hl
  ret
transition_swingready_to_observing:
  ; Set pending addresses
  ld hl,frame_handler_observing
  ld (frame_handler_address),hl
  ld hl,transition_nop
  ld (transition_address),hl
  ret

frame_handler_observing:
  xor a
  ld (batter_swing_level),a

  ; Set the batter's pattern
  ld a,BLANK_SPRITE_PATTERN
  ld (batter_pattern),a

  ; Render the game
  call do_rendering
  ; Handle keys
  in a,(KEYSEL)
  and #F0
  or 6 ; Select row 6
  out (KEYSEL),a
  in a,(KEYROW)
  bit 0,a ; Shift
  jr z,$+2+3+3
  ld hl,transition_observing_to_swingready
  ld (transition_address),hl
  ; Handle inputs
  in a,(KEYSEL)
  and #F0
  or 8 ; Select row 8
  out (KEYSEL),a
  in a,(KEYROW)
  bit 5,a ; Up
  call z,scroll_up
  bit 6,a ; Down
  call z,scroll_down
  IF DEBUG_ENABLED
    ld a,#02
    call set_border_colour
  ENDIF
  ret
transition_observing_to_swingready:
  ; Set pending addresses
  ld hl,frame_handler_swingready
  ld (frame_handler_address),hl
  ld hl,transition_nop
  ld (transition_address),hl
  ret

skip_to_next_level:
  ld hl,#0700
  ld (palette_fade_counter),hl

  ; Increment the level index
  ld hl,current_level_index
  inc (hl)

  ; Set pending addresses
  ld hl,frame_handler_levelend
  ld (frame_handler_address),hl
  ld hl,transition_nop
  ld (transition_address),hl
  ret

restart_current_level:
  ld hl,#0700
  ld (palette_fade_counter),hl

  ; Set pending addresses
  ld hl,frame_handler_levelrestart
  ld (frame_handler_address),hl
  ld hl,transition_nop
  ld (transition_address),hl
  ret

outi_15:
  REPT 15
    outi
  ENDM
  ret

outi_32:
  REPT 32
    outi
  ENDM
  ret

transition_to_next_level:
  di
  ; Disable screen display
  ld a,%00100010
  out (VDPCTL),a
  ld a,#81
  out (VDPCTL),a

  ; Load the level maps
  ld a,(current_level_index)
  call load_level

  xor a
  ld (scroll_position+0),a
  ld (scroll_position+1),a
  ld (scroll_direction),a
  ld (palette_fade_counter+0),a
  ld (palette_fade_counter+1),a

  ; Construct the flagpole metatiles
  ld a,(goal_graphic_metatiles+0)
  ld bc,(((FLAG_POLE_METATILE_INDEX_0 - 128) / 4) << 3) | (((FLAG_POLE_METATILE_INDEX_0 - 128) & 3) << (8 + 6))
  ld ix,hmmm_64x8_metatile_flagpole_ram
  call blit_64x8_metatile

  ld a,(goal_graphic_metatiles+1)
  ld bc,(((FLAG_POLE_METATILE_INDEX_1 - 128) / 4) << 3) | (((FLAG_POLE_METATILE_INDEX_1 - 128) & 3) << (8 + 6))
  ld ix,hmmm_64x8_metatile_flagpole_ram
  call blit_64x8_metatile

  ; Composite the flagpole graphics into the flagpole metatiles
  ; Check for even/odd
  ld a,(goal_map_position+0)
  and 1
  ld hl,hmmm_flagpole_graphic_odd
  jr nz,$+2+3
  ld hl,hmmm_flagpole_graphic_even
  ; Blit
  call wait_for_vdp_ready
  ld a,32
  out (VDPCTL),a
  ld a,#91
  out (VDPCTL),a
  ld c,VDPREG
  call outi_15

  ; Construct level graphics from metatiles
  ld hl,metatile_map
  ld bc,#0000
  ld ix,hmmm_32x8_metatile_ram
level_y_loop:
level_x_loop:
  ld a,(hl)
  ld d,a
  inc hl
  ; First row
  and 3
  REPT 2
    rrca
  ENDM
  ld (ix+0),a ; SX L
  ld a,d
  and #FC
  sla a
  ld (ix+2),a ; SY L
  ld a,d
  rlca
  and 1
  or 2
  ld (ix+3),a ; SY H
  ld (ix+4),b ; DX L
  ld (ix+6),c ; DY L
  ; Blit
  call wait_for_vdp_ready
  ld a,32
  out (VDPCTL),a
  ld a,#91
  out (VDPCTL),a
  REPT 15,?I
    ld a,(ix+?I)
    out (VDPREG),a
  ENDM
  ; Second row
  ld a,d
  and 3
  REPT 2
    rrca
  ENDM
  or 32
  ld (ix+0),a ; SX L
  ld a,d
  and #FC
  sla a
  ld (ix+2),a ; SY L
  ld a,d
  rlca
  and 1
  or 2
  ld (ix+3),a ; SY H
  ld (ix+4),b ; DX L
  ld a,8
  add a,c
  ld c,a
  ld (ix+6),c ; DY L
  add a,-8
  ld c,a
  ; Blit
  call wait_for_vdp_ready
  ld a,32
  out (VDPCTL),a
  ld a,#91
  out (VDPCTL),a
  REPT 15,?I
    ld a,(ix+?I)
    out (VDPREG),a
  ENDM
  ; Step
  ld a,32
  add a,b
  ld b,a
  jp nz,level_x_loop
  ld a,16
  add a,c
  ld c,a
  jp nz,level_y_loop

  ; Clear the statusbar
  call wait_for_vdp_ready
  ld a,32
  out (VDPCTL),a
  ld a,#91
  out (VDPCTL),a
  ld hl,hmmv_clear_statusbar
  ld c,VDPREG
  call outi_15

  ; Draw the default HUD text into the statusbar
  ld hl,strings
  ld b,TEXT_X_POS
  call blit_text_string_G4

  ; Get the level name string
  ld a,(current_level_index)
  inc a
  ld h,0
  ld l,a
  add hl,hl
  ld bc,string_offsets
  add hl,bc
  ld e,(hl)
  inc hl
  ld d,(hl)
  ex de,hl
  ; Draw the name of the level into the statusbar
  ld b,TEXT_X_POS
  call blit_text_string_G4

  ; Set pattern layout base address
  ld a,(BASE_PATTERN_LAYOUT>>10)|%00011111
  out (VDPCTL),a
  ld a,#82
  out (VDPCTL),a

  ; Set black palette
  ld a,#00
  out (VDPCTL),a
  ld a,#90
  out (VDPCTL),a
  ld hl,black_palette
  ld c,VDPPAL
  call outi_32

  ; It seems that an interrupt can be triggered while (or just after) re-enabling
  ; the display, so set BC to #0000 to ensure that it has a benign effect.
  ld bc,#0000
  ; Set up for P#0, for the above reason.
  ld a,#00
  out (VDPCTL),a
  ld a,#90
  out (VDPCTL),a

  ; Enable screen display
  ld a,%01100010
  out (VDPCTL),a
  ld a,#81
  out (VDPCTL),a

  ; Set pending addresses
  ld hl,frame_handler_levelbegin
  ld (frame_handler_address),hl
  ld hl,transition_nop
  ld (transition_address),hl
  ret

accumulate_swing_velocity:
  ; Accumulate velocity
  ld hl,batter_table_addresses
  ld a,(batter_angle)
  sla a
  ld b,0
  ld c,a
  add hl,bc
  ld e,(hl)
  inc hl
  ld d,(hl)
  ex de,hl
  ; Offset to velo_x and velo_y
  ld bc,12
  add hl,bc
  ld e,(hl)
  inc hl
  ld d,(hl)
  inc hl
  ld b,(hl)
  inc hl
  ; Add velo_x
  push hl
  ld hl,(accumulated_swing_velocity_x)
  ld a,(accumulated_swing_velocity_x+2)
  add hl,de
  adc a,b
  ld (accumulated_swing_velocity_x),hl
  ld (accumulated_swing_velocity_x+2),a
  pop hl
  ld e,(hl)
  inc hl
  ld d,(hl)
  inc hl
  ld b,(hl)
  ; Add velo_y
  ld hl,(accumulated_swing_velocity_y)
  ld a,(accumulated_swing_velocity_y+2)
  add hl,de
  adc a,b
  ld (accumulated_swing_velocity_y),hl
  ld (accumulated_swing_velocity_y+2),a
  ret

frame_handler_swinging:
  ld a,(batter_swing_level)
  inc a
  ld (batter_swing_level),a
  cp NUM_BATTER_SWING_LEVELS
  jr nz,$+2+3+3
  ld hl,transition_swinging_to_hitting
  ld (transition_address),hl

  call accumulate_swing_velocity

  ; Set the batter's pattern
  ld a,(batter_patterns-sprite_patterns_start)/8
  ld (batter_pattern),a

  ; Move the batter away from the ball
  ld hl,batter_table_addresses
  ld a,(batter_angle)
  sla a
  ld b,0
  ld c,a
  add hl,bc
  ld e,(hl)
  inc hl
  ld d,(hl)
  ex de,hl
  ; Offset to move_x
  ld bc,6
  add hl,bc
  ld e,(hl)
  inc hl
  ld d,(hl)
  inc hl
  ld b,(hl)
  ld hl,(batter_px)
  ld a,(batter_px+2)
  add hl,de
  adc a,b
  ld (batter_px),hl
  ld (batter_px+2),a

  ld hl,batter_table_addresses
  ld a,(batter_angle)
  sla a
  ld b,0
  ld c,a
  add hl,bc
  ld e,(hl)
  inc hl
  ld d,(hl)
  ex de,hl
  ; Offset to move_y
  ld bc,9
  add hl,bc
  ld e,(hl)
  inc hl
  ld d,(hl)
  inc hl
  ld b,(hl)
  ld hl,(batter_py)
  ld a,(batter_py+2)
  add hl,de
  adc a,b
  ld (batter_py),hl
  ld (batter_py+2),a

  ; Queue SFX
  ld a,(batter_swing_level)
  dec a
  and 3
  jr nz,$+2+3+2+3
  ld a,(ball_bounce_sfx_flags)
  or 1<<BALL_SFX_FLAG_SWING_BIT
  ld (ball_bounce_sfx_flags),a

  ; Render the game
  call do_rendering
  ; Update the scroll position
  call update_scroll_position

  ; Handle inputs
  in a,(KEYSEL)
  and #F0
  or 8
  out (KEYSEL),a
  in a,(KEYROW)
  bit 0,a ; Space
  jr z,$+2+3+3
  ld hl,transition_swinging_to_hitting
  ld (transition_address),hl
  IF DEBUG_ENABLED
    ld a,#02
    call set_border_colour
  ENDIF
  ret
transition_swinging_to_hitting:
  ; Set pending addresses
  ld hl,frame_handler_hitting
  ld (frame_handler_address),hl
  ld hl,transition_nop
  ld (transition_address),hl
  ret

apply_fade_level_to_palette:
  ; Apply fade to palette and set palette registers
  ld de,palette_ram
  ld hl,default_palette
  ld c,16
palette_fade_apply_loop:
  ; Blue
  ld a,(hl)
  and %00000111
  ld b,a
  ld a,(palette_fade_counter+1)
  and 7
  REPT 3
    rlca
  ENDM
  or b
  exx
  ld hl,palette_ramp_table
  ld b,0
  ld c,a
  add hl,bc
  ld a,(hl)
  exx
  ld (de),a
  ; Red
  ld a,(hl)
  and %01110000
  rrca
  ld b,a
  ld a,(palette_fade_counter+1)
  and 7
  or b
  exx
  ld hl,palette_ramp_table
  ld b,0
  ld c,a
  add hl,bc
  ld a,(hl)
  exx
  REPT 4
    rlca
  ENDM
  ex de,hl
  or (hl)
  ld (hl),a
  ex de,hl
  ; Green
  inc hl
  inc de
  ld a,(hl)
  and %00000111
  ld b,a
  ld a,(palette_fade_counter+1)
  and 7
  REPT 3
    rlca
  ENDM
  or b
  exx
  ld hl,palette_ramp_table
  ld b,0
  ld c,a
  add hl,bc
  ld a,(hl)
  exx
  ld (de),a
  inc hl
  inc de
  dec c
  jr nz,palette_fade_apply_loop
  ret

frame_handler_levelbegin:

  ld a,BLANK_SPRITE_PATTERN
  ld (batter_pattern),a

  call apply_fade_level_to_palette

  ld a,#00
  out (VDPCTL),a
  ld a,#90
  out (VDPCTL),a
  ld hl,palette_ram
  ld c,VDPPAL
  call outi_32

  ld hl,(palette_fade_counter)
  ld bc,64
  add hl,bc
  ld (palette_fade_counter),hl

  ld a,(palette_fade_counter+1)
  cp 8
  jr nz,$+2+3+3
  ld hl,transition_levelbegin_to_inflight
  ld (transition_address),hl

  ; Render the game
  call do_rendering
  ret
transition_levelbegin_to_inflight:
  ; Reset counter
  xor a
  ld (ball_stopped_counter),a
  ld (stroke_counter_digits+0),a
  ld a,1
  ld (stroke_counter_digits+1),a

  ld hl,#0700
  ld (palette_fade_counter),hl

  ; Set pending addresses
  ld hl,frame_handler_inflight
  ld (frame_handler_address),hl
  ld hl,transition_nop
  ld (transition_address),hl
  ret

increment_stroke_counter_digits:
  ld hl,stroke_counter_digits+1
  inc (hl)
  ld a,(hl)
  cp 10
  ret nz
  ld (hl),0
  dec hl
  inc (hl)
  ld a,(hl)
  cp 10
  ret nz
  ld (hl),9
  ret

frame_handler_levelend:

  ld a,BLANK_SPRITE_PATTERN
  ld (batter_pattern),a

  call apply_fade_level_to_palette

  ld a,#00
  out (VDPCTL),a
  ld a,#90
  out (VDPCTL),a
  ld hl,palette_ram
  ld c,VDPPAL
  call outi_32

  ld hl,(palette_fade_counter)
  ld bc,64
  and a ; Reset the carry flag
  sbc hl,bc
  ld (palette_fade_counter),hl

  ld a,(palette_fade_counter+1)
  cp #FF
  jr nz,$+2+3+3
  ld hl,transition_levelend_to_welldone
  ld (transition_address),hl

  ; Render the game
  call do_rendering
  ret
transition_levelend_to_welldone:
  ; Reset counter
  xor a
  ld (ball_stopped_counter),a

  ld hl,#0000
  ld (palette_fade_counter),hl

  ; Note that this is special, because an execution mode is being used
  ; like a subroutine.
  ld hl,draw_intermission_text_welldone
  ld (draw_intermission_text_address),hl
  call begin_intermission

  ; If this was the last level, then go to the congratulations screen.
  ld a,(current_level_index)
  cp 15
  jr nz,transition_levelend_to_welldone_notfinal
  ld hl,draw_intermission_text_congrats
  ld (draw_intermission_text_address),hl
  call begin_intermission
  jp begin_titlescreen
transition_levelend_to_welldone_notfinal:

  call init_ingame

  ld bc,#0000
  ; Set up for P#0
  ld a,#00
  out (VDPCTL),a
  ld a,#90
  out (VDPCTL),a
  ; Enable screen display
  ld a,VDP_M3|VDP_M4|VDP_IE1
  out (VDPCTL),a
  ld a,#80
  out (VDPCTL),a
  ld a,VDP_SI|VDP_IE0|VDP_BL
  out (VDPCTL),a
  ld a,#81
  out (VDPCTL),a

  ; Set pending addresses
  ld hl,frame_handler_nop
  ld (frame_handler_address),hl
  ld hl,transition_levelend_to_levelbegin
  ld (transition_address),hl
  ret
transition_levelend_to_levelbegin:
  ; Reset counter
  xor a
  ld (ball_stopped_counter),a

  ld hl,#0000
  ld (palette_fade_counter),hl

  ; Set pending addresses
  ld hl,frame_handler_nop
  ld (frame_handler_address),hl
  ld hl,transition_to_next_level
  ld (transition_address),hl
  ret

frame_handler_levelrestart:

  ld a,BLANK_SPRITE_PATTERN
  ld (batter_pattern),a

  call apply_fade_level_to_palette

  ld a,#00
  out (VDPCTL),a
  ld a,#90
  out (VDPCTL),a
  ld hl,palette_ram
  ld c,VDPPAL
  call outi_32

  ld hl,(palette_fade_counter)
  ld bc,64
  and a ; Reset the carry flag
  sbc hl,bc
  ld (palette_fade_counter),hl

  ld a,(palette_fade_counter+1)
  cp #FF
  jr nz,$+2+3+3
  ld hl,transition_levelend_to_levelbegin
  ld (transition_address),hl

  ; Render the game
  call do_rendering
  ret

frame_handler_hitting:
  ld a,(batter_swing_level)
  REPT 4
    dec a
  ENDM
  ld (batter_swing_level),a
  and 128
  jr z,$+2+3+3
  ld hl,transition_hitting_to_inflight
  ld (transition_address),hl

  ; Move the batter towards the ball
  ld hl,batter_table_addresses
  ld a,(batter_angle)
  sla a
  ld b,0
  ld c,a
  add hl,bc
  ld e,(hl)
  inc hl
  ld d,(hl)
  ex de,hl
  ; Offset to move_x
  ld bc,6
  add hl,bc
  ld e,(hl)
  inc hl
  ld d,(hl)
  inc hl
  ld b,(hl)

  ; Multiply by 4
  REPT 2
    sla e
    rl d
    rl b
  ENDM

  ld hl,(batter_px)
  ld a,(batter_px+2)
  and a
  sbc hl,de
  sbc a,b
  ld (batter_px),hl
  ld (batter_px+2),a

  ld hl,batter_table_addresses
  ld a,(batter_angle)
  sla a
  ld b,0
  ld c,a
  add hl,bc
  ld e,(hl)
  inc hl
  ld d,(hl)
  ex de,hl
  ; Offset to move_y
  ld bc,9
  add hl,bc
  ld e,(hl)
  inc hl
  ld d,(hl)
  inc hl
  ld b,(hl)

  ; Multiply by 4
  REPT 2
    sla e
    rl d
    rl b
  ENDM

  ld hl,(batter_py)
  ld a,(batter_py+2)
  and a
  sbc hl,de
  sbc a,b
  ld (batter_py),hl
  ld (batter_py+2),a

  ; Render the game
  call do_rendering
  ; Update the scroll position
  call update_scroll_position
  ret
transition_hitting_to_inflight:
  ; Reset counter
  xor a
  ld (ball_stopped_counter),a

  ; Increment the stroke counter if is it less than 99
  ld a,(stroke_counter_digits+1)
  ld b,a
  ld a,(stroke_counter_digits+0)
  ; Multiply A by 10=8+2
  rlca
  ld c,a
  REPT 2
    rlca
  ENDM
  add a,c
  add a,b
  cp 99
  call nz,increment_stroke_counter_digits

  ; Apply force to ball (divided by two, rounded to nearest)
  ld hl,(accumulated_swing_velocity_x)
  inc hl
  sra h
  rr l
  ld (ball_vx),hl

  ld hl,(accumulated_swing_velocity_y)
  inc hl
  sra h
  rr l
  ld (ball_vy),hl

  call set_ball_prevpos_from_vel

  ; Queue SFX
  ld a,(ball_bounce_sfx_flags)
  or 1<<BALL_SFX_FLAG_HIT_BIT
  ld (ball_bounce_sfx_flags),a

  ; Set pending addresses
  ld hl,frame_handler_inflight
  ld (frame_handler_address),hl
  ld hl,transition_nop
  ld (transition_address),hl
  ret

update_scroll_position:
  ; Determine scroll delta
  IF 1
    ld hl,(scroll_position)
    ex de,hl
    ld hl,(ball_py_current+1)
    and a ; Reset carry flag
    sbc hl,de
    ld a,h
    cp #FF
    jr z,scroll_delta_up
    cp #FE
    jr z,scroll_delta_up
    cp #FD
    jr z,scroll_delta_up
    cp #FC
    jr z,scroll_delta_up
    cp #01
    jr z,scroll_delta_down
    cp #02
    jr z,scroll_delta_down
    cp #03
    jr z,scroll_delta_down
    cp #04
    jr z,scroll_delta_down
    ld a,l
    cp 128+16
    jr nc,scroll_delta_down
    cp 64+16
    jr c,scroll_delta_up
    jr scroll_delta_none

  scroll_delta_down:
    call scroll_down
    jr scroll_delta_none

  scroll_delta_down2:
    REPT 2
      call scroll_down
    ENDM
    jr scroll_delta_none

  scroll_delta_up:
    call scroll_up
    jr scroll_delta_none

  scroll_delta_up2:
    REPT 2
      call scroll_up
    ENDM
    jr scroll_delta_none

  scroll_delta_none:
  ELSE
    ld hl,(ball_py_current+1)
    ld bc,-192/2
    add hl,bc
    ld (scroll_position),hl
  ENDIF
  ret

frame_handler_inflight:
  ; Set the batter's pattern
  ld a,BLANK_SPRITE_PATTERN
  ld (batter_pattern),a
  ; Refresh the simulation
  call do_physics
  ; Render the game
  call do_rendering
  ; Update the scroll position
  call update_scroll_position
  ; Handle inputs
  in a,(KEYSEL)
  and #F0
  or 8
  out (KEYSEL),a
  in a,(KEYROW)
  bit 7,a ; Right
  ;call z,right_pressed2
  bit 4,a ; Left
  ;call z,left_pressed2
  ret
transition_inflight_to_swingready:
  ld a,NUM_BATTER_ANGLES/2
  ld (batter_angle),a

  call position_batter_around_ball

  ; Set pending addresses
  ld hl,frame_handler_swingready
  ld (frame_handler_address),hl
  ld hl,transition_nop
  ld (transition_address),hl
  ret

;  A = Metatile index
; BC = Destination XY
blit_32x8_metatile:
  ; Blit one 32x8 metatile row
  ld d,a
  and 3
  REPT 2
    rrca
  ENDM
  ld e,a
  ld a,c
  and 8
  REPT 2
    rlca
  ENDM
  add a,e
  ld (ix+0),a ; SX L
  ld a,d
  and #FC
  sla a
  ld (ix+2),a ; SY L
  ld a,d
  rlca
  and 1
  or 2
  ld (ix+3),a ; SY H
  ld (ix+4),b ; DX L
  ld (ix+6),c ; DY L
  ; Blit
  IF DEBUG_ENABLED
    ld a,#0F
    call set_border_colour
  ENDIF
  call wait_for_vdp_ready
  IF DEBUG_ENABLED
    ld a,#0A
    call set_border_colour
  ENDIF
  ld a,32
  out (VDPCTL),a
  ld a,#91
  out (VDPCTL),a
  ld d,ixh
  ld e,ixl
  ex de,hl
  ld c,VDPREG
  jp outi_15

;  A = Metatile index
; BC = Destination XY
blit_64x8_metatile:
  ; Blit one 64x8 metatile row
  ld d,a
  and 3
  REPT 2
    rrca
  ENDM
  ld (ix+0),a ; SX L
  ld a,d
  and #FC
  sla a
  ld (ix+2),a ; SY L
  ld a,d
  rlca
  and 1
  or 2
  ld (ix+3),a ; SY H
  ld (ix+4),b ; DX L
  ld (ix+6),c ; DY L
  ; Blit
  call wait_for_vdp_ready
  ld a,32
  out (VDPCTL),a
  ld a,#91
  out (VDPCTL),a
  ld d,ixh
  ld e,ixl
  ex de,hl
  ld c,VDPREG
  jp outi_15

set_border_colour:
  out (VDPCTL),a
  ld a,#87
  out (VDPCTL),a
  ret

wait_for_vblank:
  xor a
  ld (frame_advance),a
  ei
  ld a,(frame_advance)
  and a
  jr z,$-4
  ret

wait_for_vdp_ready:
  ld a,#02
  out (VDPCTL),a
  ld a,#8F
  out (VDPCTL),a
  in a,(VDPCTL)
  rra
  jr c,wait_for_vdp_ready
  ret

wait_for_vdp_transfer_ready:
  ld a,#02
  out (VDPCTL),a
  ld a,#8F
  out (VDPCTL),a
  in a,(VDPCTL)
  rla
  jr nc,$-1-2
  ret

; HL = Command address
; DE = Data address
;  B = Data Length / 4
execute_hmmc_x4:
  push bc
  call wait_for_vdp_ready
  ld a,32
  out (VDPCTL),a
  ld a,#91
  out (VDPCTL),a
  ld c,VDPREG
  REPT 12
    outi
  ENDM
  ; First byte of graphic must be loaded already before execution.
  ex de,hl
  outi
  ex de,hl
  inc hl
  outi
  outi
  ld c,#AC
  REPT 3
    call wait_for_vdp_transfer_ready
    ld a,(de)
    inc de
    out (VDPCTL),a
    ld a,c
    out (VDPCTL),a
  ENDM
  pop bc
  dec b
  ret z
  ld c,#AC
execute_hmmc_x4_tr_loop:
  REPT 4
    call wait_for_vdp_transfer_ready
    ld a,(de)
    inc de
    out (VDPCTL),a
    ld a,c
    out (VDPCTL),a
  ENDM
  djnz execute_hmmc_x4_tr_loop
  ret

; HL = Command address
; DE = Data address
;  B = Data Length / 16
execute_hmmc_x16:
  push bc
  call wait_for_vdp_ready
  ld a,32
  out (VDPCTL),a
  ld a,#91
  out (VDPCTL),a
  ld c,VDPREG
  REPT 12
    outi
  ENDM
  ; First byte of graphic must be loaded already before execution.
  ex de,hl
  outi
  ex de,hl
  inc hl
  outi
  outi
  ld c,#AC
  REPT 15
    call wait_for_vdp_transfer_ready
    ld a,(de)
    inc de
    out (VDPCTL),a
    ld a,c
    out (VDPCTL),a
  ENDM
  pop bc
  dec b
  ret z
  ld c,#AC
execute_hmmc_x16_tr_loop:
  REPT 16
    call wait_for_vdp_transfer_ready
    ld a,(de)
    inc de
    out (VDPCTL),a
    ld a,c
    out (VDPCTL),a
  ENDM
  dec b
  jp nz,execute_hmmc_x16_tr_loop
  ret

; HL = Source address in RAM
; DE = Destination address in VRAM
; BC = Data length
ldirvm:
  ; Set address counter A16 to A14
  ret

init_audio:
  ; Initialise the PSG
  REPT 14,?I
    ld a,?I
    out (PSGREG),a
    IF ?I=7
      xor %10111111
      out (PSGWRI),a
    ELSE
      xor a
      out (PSGWRI),a
    ENDIF
  ENDM
  ret


frame_handler_nop:
  ret
transition_nop:
  ret

position_batter_around_ball:
  ; Position the batter
  ld hl,batter_table_addresses
  ld a,(batter_angle)
  sla a
  ld b,0
  ld c,a
  add hl,bc
  ld e,(hl)
  inc hl
  ld d,(hl)
  ex de,hl

  ld e,(hl)
  inc hl
  ld d,(hl)
  inc hl
  ld b,(hl)
  ld hl,(ball_px_current)
  add hl,de
  ld (batter_px),hl

  ld hl,batter_table_addresses
  ld a,(batter_angle)
  sla a
  ld b,0
  ld c,a
  add hl,bc
  ld e,(hl)
  inc hl
  ld d,(hl)
  ex de,hl
  REPT 3
    inc hl
  ENDM

  ld e,(hl)
  inc hl
  ld d,(hl)
  inc hl
  ld b,(hl)
  ld hl,(ball_py_current)
  ld a,(ball_py_current+2)
  add hl,de
  adc a,b
  ld (batter_py),hl
  ld (batter_py+2),a
  ret

left_pressed:
  push af
  push bc
  ld a,(batter_angle)
  cp NUM_BATTER_ANGLES-1
  jr z,$+2+1+3
  inc a
  ld (batter_angle),a
  call position_batter_around_ball
  ld hl,input_leftright_gate
  ld (hl),1
  pop bc
  pop af
  ret

right_pressed:
  push af
  push bc
  ld a,(batter_angle)
  cp 0
  jr z,$+2+1+3
  dec a
  ld (batter_angle),a
  call position_batter_around_ball
  ld hl,input_leftright_gate
  ld (hl),1
  pop bc
  pop af
  ret

; HL = X, DE = Y in pixel coordinates
init_ball:
  xor a
  ld (ball_px_current+0),a
  ld (ball_py_current+0),a
  ld (ball_px_current+1),hl
  ex de,hl
  ld (ball_py_current+1),hl
set_ball_prevpos_from_vel:
  ld hl,(ball_vx)
  ex de,hl
  ld hl,(ball_px_current)
  and a
  sbc hl,de
  ld (ball_px_previous),hl

  ld hl,(ball_vy)
  ex de,hl
  ld hl,(ball_py_current)
  ld a,(ball_py_current+2)
  and a
  sbc hl,de
  ld (ball_py_previous),hl
  ld (ball_py_previous+2),a
  ret

calc_ball_velocity:
  ld hl,(ball_px_current)
  ex de,hl
  ld hl,(ball_px_next)
  and a
  sbc hl,de
  ld (ball_vx),hl

  ld hl,(ball_py_current)
  ex de,hl
  ld hl,(ball_py_next)
  and a
  sbc hl,de
  ld (ball_vy),hl
  ret

verlet_iterate_ball:
  ld hl,(ball_px_previous)
  ex de,hl
  ; Some of the following lines are commented out because the
  ; X position of the ball cannot exceed 255.
  ld hl,(ball_px_current)
  ; Multiply by 2
  add hl,hl
  and a ; Clear the carry flag
  sbc hl,de
  ld (ball_px_next),hl

  ld hl,(ball_py_previous)
  ex de,hl
  ld a,(ball_py_previous+2)
  ld b,a
  ld hl,(ball_py_current)
  ld a,(ball_py_current+2)
  ; Multiply by 2
  add hl,hl
  adc a,a
  and a ; Clear the carry flag
  sbc hl,de
  sbc a,b
  ; Add 1 for acceleration
  ld bc,1
  add hl,bc
  adc a,0
  ld (ball_py_next),hl
  ld (ball_py_next+2),a
  ret

step_ball:
  ld hl,(ball_px_current)
  ld (ball_px_previous),hl

  ld hl,(ball_py_current)
  ld (ball_py_previous),hl
  ld a,(ball_py_current+2)
  ld (ball_py_previous+2),a

  ld hl,(ball_px_next)
  ld (ball_px_current),hl

  ld hl,(ball_py_next)
  ld (ball_py_current),hl
  ld a,(ball_py_next+2)
  ld (ball_py_current+2),a
  ret

check_collision:
  ld a,(ball_px_next+1)
  REPT 3
    rrca
  ENDM
  and 31 ; Map is 32 tiles wide and 128 tiles tall
  ld c,a
  ld b,0
  ld hl,(ball_py_next+1)
  ld a,l
  and 255-7
  ld l,a
  REPT 2
    add hl,hl
  ENDM
  add hl,bc
  ld bc,collision_map
  add hl,bc
  ld l,(hl) ; Pre-multiplied by 16
  ld h,0
  REPT 3 ; Multiply by 8 to get the required stride of 16 x 8 = 128
    add hl,hl
  ENDM
  ld bc,collision_tiles
  add hl,bc
  ld a,(ball_px_next+1) ; Now get intra-tile coordinates
  and 7
  ld b,a
  ld a,(ball_py_next+1)
  and 7
  REPT 3
    rlca
  ENDM
  add a,b
  add a,a
  ld c,a
  ld b,0
  add hl,bc
  ; Get the jump destination address
  ld e,(hl)
  inc hl
  ld d,(hl)
  ex de,hl
  ; Make the indirect jump
  jp (hl)

bounce_ball_x:
  call absorb_bounce_x
  ld hl,(ball_vx)
  call check_for_bounce_sfx
  ; Invert vx
  ld a,l
  cpl
  ld l,a
  ld a,h
  cpl
  ld h,a
  inc hl
  ld (ball_vx),hl
  jp set_ball_prevpos_from_vel

bounce_ball_y:
  call absorb_bounce_y
  ld hl,(ball_vy)
  call check_for_bounce_sfx
  ; Invert vy
  ld a,l
  cpl
  ld l,a
  ld a,h
  cpl
  ld h,a
  inc hl
  ld (ball_vy),hl
  jp set_ball_prevpos_from_vel

check_for_bounce_sfx:
  ; Check the magnitude of vx or vy to see if this bounce should make a sound.
  ld d,h
  ld e,l
  bit 7,d
  jr z,check_for_bounce_sfx_noinvert_de
  ; Invert DE
  ld a,e
  cpl
  ld e,a
  ld a,d
  cpl
  ld d,a
  inc de
check_for_bounce_sfx_noinvert_de:
  ld a,d
  inc a
  cp 1
  ret c
  ld a,e
  cp 16
  ret c
  ld a,(ball_bounce_sfx_flags)
  or 1<<BALL_SFX_FLAG_BOUNCE_BIT
  ld (ball_bounce_sfx_flags),a
  ret

; 01
; 23

collision_tile_diag_0:
  ; Compare subpixel coordinates
  ld a,(ball_px_next)
  cpl
  ld b,a
  ld a,(ball_py_next)
  cp b
  jr nc,bounce_ball_xy
  jp step_ball

collision_tile_diag_1:
  ; Compare subpixel coordinates
  ld a,(ball_px_next)
  ld b,a
  ld a,(ball_py_next)
  cp b
  jr nc,bounce_ball_yx
  jp step_ball

collision_tile_diag_2:
  ; Compare subpixel coordinates
  ld a,(ball_px_next)
  ld b,a
  ld a,(ball_py_next)
  cp b
  jr c,bounce_ball_yx
  jp step_ball

collision_tile_diag_3:
  ; Compare subpixel coordinates
  ld a,(ball_px_next)
  cpl
  ld b,a
  ld a,(ball_py_next)
  cp b
  jr c,bounce_ball_xy
  jp step_ball

bounce_ball_xy: ; Reflect in (+1,+1) vector
  call absorb_bounce_y

  ld hl,(ball_vx)
  call check_for_bounce_sfx
  ld hl,(ball_vy)
  call check_for_bounce_sfx

  ld hl,(ball_vy)
  ex de,hl
  ld hl,(ball_vx)
  add hl,de
  ex de,hl
  ld hl,(ball_vx)
  ; r=v-n2n.v
  ; rx=vx-1/sqrt(2)*2/sqrt(2)*(vx+vy)
  ; rx=vx-(vx+vy)
  ; rx=-vy
  ; ry=vy-(vx+vy)
  ; ry=-vx
  and a
  sbc hl,de
  ld (ball_vx),hl
  ;   nvy = ball_v[1] - dn;
  ld hl,(ball_vy)
  and a
  sbc hl,de
  ld (ball_vy),hl
  jp set_ball_prevpos_from_vel

bounce_ball_yx: ; Reflect in (+1,-1) vector
  call absorb_bounce_y

  ld hl,(ball_vx)
  call check_for_bounce_sfx
  ld hl,(ball_vy)
  call check_for_bounce_sfx

  ld hl,(ball_vy)
  ex de,hl
  ld hl,(ball_vx)
  and a
  sbc hl,de
  ; Now dn is in AHL
  ; nvx = ball_v[0] - (dn * 3) / 2;
  ex de,hl
  ld hl,(ball_vx)
  and a
  sbc hl,de
  ld (ball_vx),hl
  ;   nvy = ball_v[1] + (dn * 3) / 2;
  ld hl,(ball_vy)
  add hl,de
  ld (ball_vy),hl
  jp set_ball_prevpos_from_vel

absorb_bounce_x:
  ; Divide vx by two
  ld hl,(ball_vx)
  inc hl
  sra h
  rr l
  ld (ball_vx),hl
  ; Multiply vy by 11/16 = (1+2+8)/16 = 0.6875
  ld hl,(ball_vy)
  ld d,h
  ld e,l
  ; Multiply by 11=1+2+8
  add hl,hl
  ex de,hl
  add hl,de
  ex de,hl
  add hl,hl
  add hl,hl
  ex de,hl
  add hl,de
  ; Add 8, so that it rounds to nearest.
  ld bc,8
  add hl,bc
  ; Divide by 16
  REPT 4
    sra h
    rr l
  ENDM
  ld (ball_vy),hl
  ret

absorb_bounce_y:
  ; Multiply vx by 11/16 = (1+2+8)/16 = 0.6875
  ld hl,(ball_vx)
  ld d,h
  ld e,l
  ; Multiply by 11=1+2+8
  add hl,hl
  ex de,hl
  add hl,de
  ex de,hl
  add hl,hl
  add hl,hl
  ex de,hl
  add hl,de
  ; Add 8, so that it rounds to nearest.
  ld bc,8
  add hl,bc
  ; Divide by 16
  REPT 4
    sra h
    rr l
  ENDM
  ld (ball_vx),hl
  ; Divide vy by two and increase
  ld hl,(ball_vy)
  ld bc,1+8
  add hl,bc
  sra h
  rr l
  ld (ball_vy),hl
  ret

; HL = Compressed data address
; DE = Destination address
;  B = High byte of end of destination buffer
decompress_rle:
  ld a,b
  exx
  ld b,a
  exx
decompress_rle_loop:
  ld a,(hl)
  inc hl
  ld c,(hl)
  inc hl
  push hl
  dec c
  ld (de),a
  ld h,d
  ld l,e
  inc de
  jr z,$+6
  ld b,0
  ldir
  pop hl
  exx
  ld a,b
  exx
  cp d
  jr nz,decompress_rle_loop
  ret

brush_sine_table:
  incbin "brush_sine_table.bin"

  ALIGN 256
cube_scale_lut:
  REPT 128,?I
    db (?I*9)/10
  ENDM
  REPT 128,?I
    db (127-((127-?I)*9)/10)-128
  ENDM

  ALIGN 256
sine_table:
  incbin "sine_table.bin"

brush_points_start:
  incbin "brush_points.bin"
  db #00, #00
brush_points_end:

batter_table_addresses:
  REPT 33, ?I
    dw batter_table+?I*18
  ENDM

sprite_patterns_start:
ball_patterns:
  incbin "ball_graphic_planar.bin"

batter_patterns:
  db %00000000
  db %00000000
  db %00000000
  db %00000000
  db %00000000
  db %00000000
  db %00000011
  db %00000011

  db %00000011
  db %00000011
  db %00000000
  db %00000000
  db %00000000
  db %00000000
  db %00000000
  db %00000000

  db %00000000
  db %00000000
  db %00000000
  db %00000000
  db %00000000
  db %00000000
  db %11000000
  db %11000000

  db %11000000
  db %11000000
  db %00000000
  db %00000000
  db %00000000
  db %00000000
  db %00000000
  db %00000000

batter_outline_patterns:
  db %00000000
  db %00000000
  db %00000000
  db %00000000
  db %00000000
  db %00000011
  db %00000111
  db %00000111

  db %00000111
  db %00000111
  db %00000011
  db %00000000
  db %00000000
  db %00000000
  db %00000000
  db %00000000

  db %00000000
  db %00000000
  db %00000000
  db %00000000
  db %00000000
  db %11000000
  db %11100000
  db %11100000

  db %11100000
  db %11100000
  db %11000000
  db %00000000
  db %00000000
  db %00000000
  db %00000000
  db %00000000

blank_patterns:
  REPT 8*4*2
    db 0
  ENDM
sprite_patterns_end:

sprite_colours:
  ; Ball (constructed using logical OR from CC bit trick)
  REPT 4,?I
    REPT 16
      IF ?I>0
        db (1<<?I)|(1<<6)
      ELSE
        db (1<<?I)
      ENDIF
    ENDM
  ENDM
  ; Batter
  REPT 16
    db 6
  ENDM
  ; Batter Outline
  REPT 16
    db 2
  ENDM
  ; Flag
  REPT 16 ; White parts
    db 6
  ENDM
  REPT 16 ; Black parts
    db 2
  ENDM
sprite_colours_end:

hmmv_clear_ingame_vram:
  MAKE_HMMV 0, 0, 256, 1024, 0
  db 0 ; Force 16B alignment

hmmv_clear_lasereffect:
  MAKE_HMMV 0, 0, 256, 256, %0000
  db 0 ; Force 16B alignment

hmmv_clear_intermission:
  MAKE_HMMV 0, 0, 256, 256, #00
  db 0 ; Force 16B alignment

hmmv_clear_intermission_upper_mask
  MAKE_HMMV 0, 0, 256, 96, #FF
  db 0 ; Force 16B alignment

hmmv_clear_intermission_upper_zero
  MAKE_HMMV 0, 0, 256, 96, #00
  db 0 ; Force 16B alignment

hmmv_clear_intermission_upper_one
  MAKE_HMMV 0, 0, 256, 100, #11
  db 0 ; Force 16B alignment

hmmv_clear_framebuffer:
  MAKE_HMMV 0, 0, 256, 256, %01001001
  db 0 ; Force 16B alignment

hmmv_clear_maskbuffer:
  MAKE_HMMV 0, 256, 256, 256, %11111111
  db 0 ; Force 16B alignment

hmmc_intermission_stripes:
  MAKE_HMMC 0, 96+5, 2, 160
  db 0 ; Force 16B alignment

hmmc_intermission_stripes_2:
  MAKE_HMMC 0, 0, 2, 96
  db 0 ; Force 16B alignment

lmmm_intermission_stripes:
  MAKE_LMMM 0, 96, 0, 96, 2, 160, LOGIC_OP_IMP ; Placeholder DX L
  db 0 ; Force 16B alignment

lmmm_intermission_stripes_2:
  MAKE_LMMM 0, 0, 0, 0, 2, 96, LOGIC_OP_AND ; Placeholder DX L
  db 0 ; Force 16B alignment

lmmm_intermission_reflect
  MAKE_LMMM 4, 0, 0, 0, 256-4, 1, LOGIC_OP_TIMP ; Placeholder SY L, DY L
  db 0 ; Force 16B alignment

hmmc_flag_patterns:
  MAKE_HMMC 0, FLAG_STORED_PATTERN_Y_POS, FLAG_STORED_PATTERN_X_EXTENT, FLAG_STORED_PATTERN_Y_EXTENT
  db 0 ; Force 16B alignment

hmmc_flagpole_graphic:
  MAKE_HMMC 0, 256+8, 128, 8
  db 0 ; Force 16B alignment

hmmc_font:
  MAKE_HMMC 0, 256+FONT_Y_POS, 256, 32
  db 0 ; Force 16B alignment

hmmc_font_part1:
  MAKE_HMMC 0, 256+FONT_Y_POS, 256, 16
  db 0 ; Force 16B alignment

hmmc_font_part2:
  MAKE_HMMC 0, 256+FONT_Y_POS+16, 256, 16
  db 0 ; Force 16B alignment


; For drawing text with a drop shadow, XOR is used for the shadow
; part and OR is used for the foreground part. This allows the same source graphic
; to be used for both background and foreground.
lmmm_text_xor:
  MAKE_LMMM 0, 256, 0, 256+TEXT_Y_POS+TEXT_SHADOW_Y_OFFSET, 8, 16, LOGIC_OP_XOR ; Placeholder SX L, SY L, DX L
  db 0 ; Force 16B alignment
lmmm_text_or:
  MAKE_LMMM 0, 256, 0, 256+TEXT_Y_POS, 8, 16, LOGIC_OP_OR ; Placeholder SX L, SY L, DX L
  db 0 ; Force 16B alignment
lmmm_text:
  MAKE_LMMM 0, 256, 0, 192-16*2, 8, 16, LOGIC_OP_TIMP ; Placeholder SX L, SY L, DX L
  db 0 ; Force 16B alignment

hmmm_stroke_counter_digit:
  MAKE_HMMM 0, 256+FONT_Y_POS-16, 0, 256+TEXT_Y_POS, 8, 16 ; Placeholder SX L, DX L
  db 0 ; Force 16B alignment

hmmv_clear_stroke_counter:
  MAKE_HMMV TEXT_STROKE_COUNTER_X, TEXT_Y_POS-2+256, 16+TEXT_SHADOW_X_OFFSET+1, TEXT_STATUSBAR_HEIGHT+2, TEXT_BACKGROUND_COLOUR | (TEXT_BACKGROUND_COLOUR << 4)
  db 0 ; Force 16B alignment

hmmv_clear_stroke_counter_digits:
  MAKE_HMMV 0, 256+FONT_Y_POS-16, 256, 16, TEXT_BACKGROUND_COLOUR | (TEXT_BACKGROUND_COLOUR << 4)
  db 0 ; Force 16B alignment

hmmv_clear_statusbar:
  MAKE_HMMV 0, TEXT_Y_POS-2+256, 256, TEXT_STATUSBAR_HEIGHT+2, TEXT_BACKGROUND_COLOUR | (TEXT_BACKGROUND_COLOUR << 4)
  db 0 ; Force 16B alignment

; Commands for compositing flagpole graphic into flagpole metatiles
hmmm_flagpole_graphic_even:
  MAKE_LMMM 0, 256+8, (FLAG_POLE_METATILE_INDEX_0 & 3) * 64, 512 + (FLAG_POLE_METATILE_INDEX_0 / 4) * 8, 128, 8, LOGIC_OP_TIMP
  db 0 ; Force 16B alignment
hmmm_flagpole_graphic_odd:
  MAKE_LMMM 0, 256+8, (FLAG_POLE_METATILE_INDEX_0 & 3) * 64 + 16, 512 + (FLAG_POLE_METATILE_INDEX_0 / 4) * 8, 128, 8, LOGIC_OP_TIMP
  db 0 ; Force 16B alignment

flag_pattern_ymmms_start:
  REPT FLAG_FRAME_COUNT,?I
    MAKE_YMMM FLAG_STORED_PATTERN_Y_POS+?I, FLAG_STORED_PATTERN_X_EXTENT-1, FLAG_CURRENT_PATTERN_Y_POS, 1, 1
    db 0 ; Force 16B alignment  
  ENDM
flag_pattern_ymmms_end:

; Brush shadow compositing sequence:
; A = Brush graphic
; B = Framebuffer region copy
; C = Maskbuffer region copy
; D = Shadow graphic
; E = Brush mask

shadow_composite_commands_start:
shadow_composite_command_1: ; HMMM Framebuffer to B
  MAKE_HMMM 0, 0, 16*1, 192, 12, 6 ; Placeholder SX L, SY L
shadow_composite_command_2: ; LMMM Shadow graphic to B with TIMP
  MAKE_LMMM 16*3, 192, 16*1, 192, 12, 6, LOGIC_OP_TIMP
shadow_composite_command_3: ; LMMM Maskbuffer to B with AND
  MAKE_LMMM 0, 256, 16*1, 192, 12, 6, LOGIC_OP_AND ; Placeholder SX L, SY L
shadow_composite_command_4: ; LMMM Maskbuffer to C with NOT
  MAKE_LMMM 0, 256, 16*2, 192, 12, 6, LOGIC_OP_NOT ; Placeholder SX L, SY L
shadow_composite_command_5: ; LMMM C to Framebuffer with AND
  MAKE_LMMM 16*2, 192, 0, 0, 12, 6, LOGIC_OP_AND ; Placeholder DX L, DY L
shadow_composite_command_6: ; LMMM B to Framebuffer with OR
  MAKE_LMMM 16*1, 192, 0, 0, 12, 6, LOGIC_OP_OR ; Placeholder DX L, DY L
shadow_composite_commands_end:

hmmc_brush:
  MAKE_HMMC 0, 192, 12, 6
  db 0 ; Force 16B alignment

hmmc_brush_shadow:
  MAKE_HMMC 16*3, 192, 12, 6
  db 0 ; Force 16B alignment

hmmc_brush_mask:
  MAKE_HMMC 16*4, 192, 12, 6
  db 0 ; Force 16B alignment

lmmm_brush:
  MAKE_LMMM 0, 192, 0, 0, 12, 6, LOGIC_OP_TIMP ; Placeholder DX L, DY L
  db 0 ; Force 16B alignment

lmmm_brush_mask:
  MAKE_LMMM 16*4, 192, 0, 256, 12, 6, LOGIC_OP_AND ; Placeholder DX L, DY L
  db 0 ; Force 16B alignment


; Laser effect line commands
;
;  octants:
;
;     13
;    0  2
;    4  6
;     57
;
line_laser_commands_start:
line_laser_q6: ; DIX=Right, DIY=Down, X-Major
  MAKE_LINE 0, 0, 0, 0, %0000, 0, 0, 0, LOGIC_OP_OR
  db 0 ; Force 16B alignment
line_laser_q7: ; DIX=Right, DIY=Down, Y-Major
  MAKE_LINE 0, 0, 0, 0, %0000, 0, 0, 1, LOGIC_OP_OR
  db 0 ; Force 16B alignment
line_laser_q4: ; DIX=Left,  DIY=Down, X-Major
  MAKE_LINE 0, 0, 0, 0, %0000, 0, 1, 0, LOGIC_OP_OR
  db 0 ; Force 16B alignment
line_laser_q5: ; DIX=Left,  DIY=Down, Y-Major
  MAKE_LINE 0, 0, 0, 0, %0000, 0, 1, 1, LOGIC_OP_OR
  db 0 ; Force 16B alignment
line_laser_q2: ; DIX=Right, DIY=Up,   X-Major
  MAKE_LINE 0, 0, 0, 0, %0000, 1, 0, 0, LOGIC_OP_OR
  db 0 ; Force 16B alignment
line_laser_q3: ; DIX=Right, DIY=Up,   Y-Major
  MAKE_LINE 0, 0, 0, 0, %0000, 1, 0, 1, LOGIC_OP_OR
  db 0 ; Force 16B alignment
line_laser_q0: ; DIX=Left,  DIY=Up,   X-Major
  MAKE_LINE 0, 0, 0, 0, %0000, 1, 1, 0, LOGIC_OP_OR
  db 0 ; Force 16B alignment
line_laser_q1: ; DIX=Left,  DIY=Up,   Y-Major
  MAKE_LINE 0, 0, 0, 0, %0000, 1, 1, 1, LOGIC_OP_OR
  db 0 ; Force 16B alignment
line_laser_commands_end:

line_laser_command_test:
  MAKE_LINE 128,128, 32, 8, %1111, 0, 0, 0, LOGIC_OP_IMP

titlescreen_fadeout_commands_start:
  REPT 4, ?I
    MAKE_LINE ?I*256/4, 0, 256, 256, %00000000, 0, 0, 0, LOGIC_OP_IMP
  ENDM
  REPT 3, ?I
    MAKE_LINE 0, (?I+1)*256/4, 256, 256, %00000000, 0, 0, 0, LOGIC_OP_IMP
  ENDM
titlescreen_fadeout_commands_end:

metatiles:
  incbin "metatiles.bin"

unpack_font_G4_intermission:
  ld hl,font_graphic
  ld de,font_unpack_temp
  ld bc,1024
unpack_font_G4_intermission_loop:
  ; Unpack one byte into four bytes
  ld a,(hl)

  ; Temporary unpacked bytes are stored as DE'HL'
  exx

  ld hl,#0000
  ld de,#0000

  bit 7,a
  jr z,$+2+4+4
  set 4*1+0,l
  set 4*1+1,l
  set 4*1+2,l
  set 4*1+3,l
  bit 6,a
  jr z,$+2+4+4
  set 4*0+0,l
  set 4*0+1,l
  set 4*0+2,l
  set 4*0+3,l
  bit 5,a
  jr z,$+2+4+4
  set 4*1+0,h
  set 4*1+1,h
  set 4*1+2,h
  set 4*1+3,h
  bit 4,a
  jr z,$+2+4+4
  set 4*0+0,h
  set 4*0+1,h
  set 4*0+2,h
  set 4*0+3,h
  bit 3,a
  jr z,$+2+4+4
  set 4*1+0,e
  set 4*1+1,e
  set 4*1+2,e
  set 4*1+3,e
  bit 2,a
  jr z,$+2+4+4
  set 4*0+0,e
  set 4*0+1,e
  set 4*0+2,e
  set 4*0+3,e
  bit 1,a
  jr z,$+2+4+4
  set 4*1+0,d
  set 4*1+1,d
  set 4*1+2,d
  set 4*1+3,d
  bit 0,a
  jr z,$+2+4+4
  set 4*0+0,d
  set 4*0+1,d
  set 4*0+2,d
  set 4*0+3,d

  ld a,l
  exx
  ld (de),a
  inc de
  exx
  ld a,h
  exx
  ld (de),a
  inc de
  exx
  ld a,e
  exx
  ld (de),a
  inc de
  exx
  ld a,d
  exx
  ld (de),a
  inc de

  inc hl
  dec bc
  ld a,b
  or c
  jp nz,unpack_font_G4_intermission_loop
  ; Upload
  ld hl,hmmc_font
  ld de,font_unpack_temp
  ld b,(256*32/2)/16
  call execute_hmmc_x16
  ret

  ds #8000 - $
level_data_compressed: ; Must be located at #8000.
  incbin "level_data.bin"

default_palette:
  incbin "ball_graphic_palette.bin"

hmmc_wall_graphics:
  MAKE_HMMC 0, 256, 15*8, 8
  db 0 ; Force 16B alignment

hmmm_wall_graphics_1: ; First page
  REPT 15,?I
    MAKE_HMMM ?I*8, 256, 0, 512, 32, 8
    db 0 ; Force 16B alignment
  ENDM

hmmm_wall_graphics_2: ; Second page
  REPT 15,?I
    MAKE_HMMM ?I*8, 256, 0, 768, 32, 8
    db 0 ; Force 16B alignment
  ENDM

hmmm_32x8_metatile:
  MAKE_HMMM 0, 512, 0, 0, 32, 8
  db 0 ; Force 16B alignment

hmmm_64x8_metatile_flagpole: ; Destination is always on page 3
  MAKE_HMMM 0, 512, 0, 256*3, 64, 8
  db 0 ; Force 16B alignment

laser_palette:
  REPT 15, ?I
    MAKE_PALETTE_ENTRY 0, ((((?I)&1)+(((?I)>>1)&1)+(((?I)>>2)&1)+(((?I)>>3)&1))*7)/4, 0
  ENDM
  MAKE_PALETTE_ENTRY 7,7,7

palette_ramp_table:
  REPT 8, ?Y
    REPT 8, ?X
      db (?X*?Y+3)/7
    ENDM
  ENDM

rasterbar_palette:
  incbin "palette.bin"

dither_pattern:
  incbin "dither_pattern.bin"

brush_mask:
  incbin "titlescreen_brush_mask.bin"

brush_shadow:
  incbin "titlescreen_brush_shadow.bin"

brush_mask_inverted:
  incbin "titlescreen_brush_mask_inverted.bin"

lasereffect_text_sprites_start:
  REPT (ENTER_PASSWORD_STRING_LENGTH+1)/2,?I
    MAKE_SPRITE 128-ENTER_PASSWORD_STRING_LENGTH*4+?I*16,96-8-16,?I*4
  ENDM
  REPT 8,?I
    MAKE_SPRITE 128-16*4+?I*16,96+16-8,16*4+?I*4
  ENDM
  db #D8
lasereffect_text_sprites_end:

strings:
  incbin "strings.bin"
string_offsets:
  include "string_offsets.asm"

  db "Copyright (c) 2024 Edd Biddulph"

; A = Character symbol
; B = Caret position in pixels
blit_text_character:
  push hl
  push bc
  ld c,a
  and 31
  REPT 3
    rlca
  ENDM
  ld (ix+0),a ; SX L
  ld a,FONT_Y_POS
  bit 5,c
  jr z,$+2+2
  ld a,FONT_Y_POS+16
  ld (ix+2),a ; SY L
  ld (ix+4),b ; DX L
  ; Blit
  ld d,ixh
  ld e,ixl
  call wait_for_vdp_ready
  ld a,32
  out (VDPCTL),a
  ld a,#91
  out (VDPCTL),a
  ex de,hl
  ld c,VDPREG
  call outi_15
  pop bc
  pop hl
  ret

blit_text_character_shadow:
  REPT TEXT_SHADOW_X_OFFSET
    inc b
  ENDM
  call blit_text_character
  REPT TEXT_SHADOW_X_OFFSET
    dec b
  ENDM
  ret

; HL = String to blit
;  B = Start caret position in pixels
blit_text_string_G4:
  bit 7,(hl)
  ret nz
  ld a,(hl)
  bit 6,a
  ; Shadow
  push af
  ld ix,lmmm_text_xor_ram
  call z,blit_text_character_shadow
  pop af
  ; Foreground
  ld ix,lmmm_text_or_ram
  call z,blit_text_character
  ld a,8
  add a,b
  ld b,a
  inc hl
  jr blit_text_string_G4

; HL = String to blit
;  B = Start caret position in pixels
blit_text_string_G7:
  ld ix,lmmm_text_ram
  bit 7,(hl)
  ret nz
  ld a,(hl)
  bit 6,a
  ; Shadow
  push af
  REPT TEXT_SHADOW_Y_OFFSET
   inc (ix+6) ; DY
  ENDM
  pop af
  push af
  ld (ix+14),%10010000|LOGIC_OP_TNOT ; Opcode
  call z,blit_text_character_shadow
  pop af
  push af
  REPT TEXT_SHADOW_Y_OFFSET
   dec (ix+6) ; DY
  ENDM
  pop af
  ; Foreground
  ld (ix+14),%10010000|LOGIC_OP_TIMP ; Opcode
  call z,blit_text_character
  ld a,8
  add a,b
  ld b,a
  inc hl
  jr blit_text_string_G7

; A = Level index
load_level:
  ld hl,level_data_compressed
  ld b,0
  ld c,a
  sla c
  add hl,bc
  ; Load address of compressed data section for this level
  ld e,(hl)
  inc hl
  ld d,(hl)
  ex de,hl
  ; Load player start X and Y
  ld b,(hl) ; X
  inc hl
  ld c,(hl) ; Y
  inc hl

  push hl

  ; Multiply Y by 16 and add 8
  ld h,0
  ld l,c
  REPT 4
    add hl,hl
  ENDM
  ld de,8
  add hl,de
  ex de,hl

  ; Multiply X by 16 and add 8
  ld h,0
  ld l,b
  REPT 4
    add hl,hl
  ENDM
  ld bc,8
  add hl,bc

  xor a
  ld (ball_vx+0),a
  ld (ball_vx+1),a
  ld (ball_vy+0),a
  ld (ball_vy+1),a
  call init_ball

  pop hl

  ; Load goal X and Y
  ld e,(hl) ; X
  inc hl
  ld d,(hl) ; Y
  inc hl
  ex de,hl

  ld (goal_map_position),hl

  ; Set flag sprite position
  ld a,(goal_map_position+0) ; Map X
  ld h,0
  ld l,a
  ; Mutiply HL by 16
  REPT 4
    add hl,hl
  ENDM
  ld bc,8
  add hl,bc
  ld (flag_px),hl
  ld a,(goal_map_position+1) ; Map y
  ld h,0
  ld l,a
  ; Mutiply HL by 16
  REPT 4
    add hl,hl
  ENDM
  ld bc,-14
  add hl,bc
  ld (flag_py),hl

  ex de,hl

  ; Load goal graphic metatile indices
  ld e,(hl) ; X
  inc hl
  ld d,(hl) ; Y
  inc hl
  ex de,hl
  ld (goal_graphic_metatiles),hl
  ex de,hl

  ; Load the collision map
  ld de,collision_map
  ld b,(collision_map+4096)>>8
  call decompress_rle
  ; Load the graphic metatile map
  ld de,metatile_map
  ld b,(metatile_map+512)>>8
  call decompress_rle
  ret

handle_vblank:
  ld a,1
  ld (frame_advance),a
  ei
  reti

  ds #BE00 - $

  REPT #101
    db #BF
  ENDM

sprite_attributes_init:
  REPT 32
    ;    Y,   X,   P,   *
    db #D8, #D8, #D8, #D8
  ENDM

greyscale_ramp_palette:
  REPT 16, ?I
    MAKE_PALETTE_ENTRY ?I/2, ?I/2, ?I/2
  ENDM

  ds #BFBF - $
interrupt_handler:
  ; Write P#0/1
  ld a,b          ; 4
  out (VDPPAL),a  ; 11
  ld a,c
  out (VDPPAL),a

  ld a,#01
  out (VDPCTL),a
  ld a,15 + 128
  out (VDPCTL),a ; Select S#1
  in a,(VDPCTL)

  ld a,#00
  out (VDPCTL),a
  ld a,15 + 128
  out (VDPCTL),a ; Select S#0
  in a,(VDPCTL)
  rla
  jp c,handle_vblank

  ei
  reti

  ds #C000 - $

; 16Kbyte RAM
  org #C000
RAM_START:
current_level_index:
  ds virtual 1
sprite_attributes:
  ds virtual 4*32
frame_advance:
  ds virtual 1
frame_counter:
  ds virtual 1
ball_px_next:
  ds virtual 3
ball_py_next:
  ds virtual 3
ball_px_current:
  ds virtual 3
ball_py_current:
  ds virtual 3
ball_px_previous:
  ds virtual 3
ball_py_previous:
  ds virtual 3
ball_vx:
  ds virtual 3
ball_vy:
  ds virtual 3
ball_px_stopped_previous:
  ds virtual 3
ball_py_stopped_previous:
  ds virtual 3
hmmm_32x8_metatile_ram:
  ds virtual 16
hmmm_64x8_metatile_flagpole_ram:
  ds virtual 16
lmmm_brush_ram:
  ds virtual 16
lmmm_brush_mask_ram:
  ds virtual 16
scroll_position:
  ds virtual 2
scroll_direction:
  ds virtual 1
hmmm_wall_graphics_ram:
  ds virtual 16*16
batter_px:
  ds virtual 3
batter_py:
  ds virtual 3
batter_angle:
  ds virtual 1
batter_pattern:
  ds virtual 1
batter_swing_level:
  ds virtual 1
accumulated_swing_velocity_x:
  ds virtual 3
accumulated_swing_velocity_y:
  ds virtual 3
frame_handler_address:
  ds virtual 2
transition_address:
  ds virtual 2
ball_stopped_counter:
  ds virtual 1
brush_points_pointer:
  ds virtual 2
brush_colour_index:
  ds virtual 3
brush_colour:
  ds virtual 3
shadow_composite_commands_ram:
  ds virtual shadow_composite_commands_end-shadow_composite_commands_start
titlescreen_fadeout_commands_ram:
  ds virtual titlescreen_fadeout_commands_end-titlescreen_fadeout_commands_start
  ds virtual 15
palette_ram:
  ds virtual 16*2
palette_fade_counter:
  ds virtual 2
  ALIGN_VIRTUAL 256
metatile_map:
  ds virtual 512
collision_map:
  ds virtual 4096
goal_map_position:
  ds virtual 2
goal_graphic_metatiles:
  ds virtual 2
flag_px:
  ds virtual 2
flag_py:
  ds virtual 2
lmmm_text_xor_ram:
  ds virtual 16
lmmm_text_or_ram:
  ds virtual 16
lmmm_text_or_laser_ram:
  ds virtual 16
hmmm_stroke_counter_digit_ram:
  ds virtual 16
lmmm_text_ram:
  ds virtual 16
input_leftright_gate:
  ds virtual 1
stroke_counter_digits:
  ds virtual 11
blit_stroke_counter_flag:
  ds virtual 1
  ALIGN_VIRTUAL 256
line_laser_commands_ram:
  ds virtual line_laser_commands_end-line_laser_commands_start
cube:
cube_ct:
  ds virtual 1
cube_st:
  ds virtual 1
cube_sp:
  ds virtual 1
cube_ctcp:
  ds virtual 1
cube_stcp:
  ds virtual 1
cube_stsp:
  ds virtual 1
cube_ctsp:
  ds virtual 1
cube_cp:
  ds virtual 1
cube_corner_x:
  ds virtual 1
cube_corner_y:
  ds virtual 1
cube_clr:
  ds virtual 1
cube_op:
  ds virtual 1
cube_end: ; Size = 12 bytes
cube_history:
  ds virtual (cube_end-cube)*3
opll_test_keystate:
  ds virtual 1
music_bitstream_bit:
  ds virtual 1
music_bitstream_addr:
  ds virtual 2
music_playback_status:
  ds virtual 1
music_playback_framedelta:
  ds virtual 1
ball_bounce_sfx_flags:
  ds virtual 1
opll_rhythm_cache:
  ds virtual 1
lasereffect_sprite_colours:
  ds virtual 512
keyboard_states:
  ds virtual 11
password_type_addr:
  ds virtual 2
password_type_buffer:
  ds virtual 33
password_length:
  ds virtual 1
music_playback_start_addr:
  ds virtual 2
black_palette:
  ds virtual 16
lmmm_intermission_stripes_ram:
lmmm_intermission_reflect_ram:
intermission_wavetable:
stripes_temp:
  ds virtual 512
temp_string_buffer:
  ds virtual 32
draw_intermission_text_address:
  ds virtual 2
RAM_END:
  ds virtual #E000 - $

  org #C000
font_unpack_temp: ; Overlay
  ds virtual 8192
font_unpack_temp_end: