
#include <stddef.h>
#include <stdlib.h>
#include <stdio.h>
#include <assert.h>
#include <math.h>
#include <string.h>

#include "render.h"
#include "defs.h"
#include "font.h"
#include "mem.h"
#include "input.h"

#define STB_IMAGE_IMPLEMENTATION
#include "stb_image.h"

bool_t extern_write_frame=FALSE;

static int sign(float x)
{
  return (x < 0.0) ? -1 : +1;
}

static uint8_t virtual_framebuffer[VIRTUAL_FRAME_WIDTH*VIRTUAL_FRAME_HEIGHT];

#define NUM_ANGLE_UNITS 1024
#define ANGLE_UNIT_FOV  ((NUM_ANGLE_UNITS)/4)
#define UNITS_PER_CELL  64
#define MAP_SIZE        64
#define MAX_COORDINATE  (UNITS_PER_CELL*MAP_SIZE)

#define VIEWPORT_LEFT   0
#define VIEWPORT_RIGHT  VIRTUAL_FRAME_WIDTH
#define VIEWPORT_TOP    0
#define VIEWPORT_BOTTOM (VIRTUAL_FRAME_HEIGHT/2)

struct Ramp
{
  uint8_t* pixels;
  uint8_t* pixels_2bpp;
  int w,h;
  int size_2bpp;
  bool_t packed;
  int scaling_axis;
  int direction;

  uint16_t atlas_x, atlas_y;

  uint16_t vram_atlas_x, vram_atlas_y; // These ones take DIX and DIY into account.
  uint16_t gen_u, gen_dudx, gen_dudy;
  uint8_t gen_w, gen_h;
  uint16_t gen_dSdD;
  uint32_t gen_dSdD_highprec;
  bool is_template;

  uint16_t dydx;
};

static const int num_ramp_angles=64+32;
static Ramp ramps[num_ramp_angles*2];
static const int num_ramps=sizeof(ramps)/sizeof(ramps[0]);
static int ramp_remap[num_ramps];
static int ramp_unremap[num_ramps];


// ********    ROM    ********
static uint8_t polar_table[128*128];
static uint8_t ray_intersection_test_table[NUM_ANGLE_UNITS];
static uint8_t x_angle_table[ANGLE_UNIT_FOV/2];
static const int height_over_dist_table_size=256;
static uint16_t height_over_dist_table[height_over_dist_table_size];
static int16_t logtan_table[NUM_ANGLE_UNITS/4];
static int16_t logsin_table[NUM_ANGLE_UNITS/2];
static int16_t log_table[4096];
static uint16_t exp_table[4096];
static int8_t rampindex_table[2048];
static uint8_t one_over_cosine_minus_one_table[NUM_ANGLE_UNITS/8+1];
static uint8_t arctanlog_table[2048+1];
static uint16_t log_partial_sum_table[4096];
static int16_t log_bob_table[256];
static uint8_t camera_move_sinus_table[NUM_ANGLE_UNITS/4];
static uint16_t ramp_slopes[num_ramp_angles];

// ********    RAM    ********
static const uint8_t ramp_index_cache_maxcount=0;//8;
static uint8_t ramp_index_cache_count;
static uint8_t ramp_index_cache[ramp_index_cache_maxcount*2]; // key, value

static uint8_t column_heights[128];

static uint32_t cycle_count;

static int log_bob=0;

static double log_sum_f(double x)
{
  assert(1.0+std::exp(x)>0.0);
  return std::log(1.0+std::exp(x));
}

static double log_sum_g(double x)
{
  assert(1.0+std::exp(2.0*x)>0.0);
  return std::log(1.0+std::exp(2.0*x))/2.0;
}

static const int wall_height = MAX_COORDINATE;
static const double tangent_max=std::ceil(std::tan(double(NUM_ANGLE_UNITS/4-1)*M_PI*2.0/double(NUM_ANGLE_UNITS)));

static void generateAllTables()
{
  for(int i=0;i<NUM_ANGLE_UNITS/4;++i)
  {
    double t=double(i)*M_PI*2.0/double(NUM_ANGLE_UNITS);
    int64_t j=std::sin(t)*8.0*2.0;
    assert(j<256);
    assert(j>=0);
    camera_move_sinus_table[i]=j;
  }

  for(int i=0;i<256;++i)
  {
    double bob_phase=double(i)/256.0*M_PI*2.0;
    int64_t j=std::log((0.8+0.4*(std::cos(bob_phase)+1.0)/2.0))/std::log(tangent_max)*2047.0+0.5;
    assert(j>-128);
    assert(j<+128);
    log_bob_table[i]=j;
  }

  for(int i=0;i<NUM_ANGLE_UNITS/8+1;++i)
  {
    int64_t j=32.0/cos(double(i)*M_PI*2.0/double(NUM_ANGLE_UNITS))-32.0+0.5;
    assert(j>=0);
    assert(j<16);
    one_over_cosine_minus_one_table[i]=j;
  }

  printf("tangent_max=%f\n",(float)tangent_max);

  for(int i=0;i<=2048;++i)
  {
    double x=std::floor(std::atan(std::pow(double(tangent_max), double((i+1024)*2-4096)/2047.0))*double(NUM_ANGLE_UNITS)/(M_PI*2.0));
    assert(x>=0.0);
    assert(x<256.0);
    arctanlog_table[i]=(uint8_t)x;
    printf("arctanlog_table[%d]=%f\n",i,(float)x);
  }

  for(int i=0;i<4096;++i)
  {
    double e=std::pow(double(tangent_max), double(i*2-4096)/2047.0*2.0);
    double x=std::floor(0.5*std::log(1.0+e)/std::log(tangent_max)*2047.0+0.5);
    assert(x>-4096);
    assert(x<+4096);
    log_partial_sum_table[i]=(uint16_t)x;
    printf("log_partial_sum_table[%d]=%f\n",i,(float)x);
  }

  for(int i=0;i<NUM_ANGLE_UNITS/4;++i)
  {
    int64_t j=(i==0)?-2047:std::log(std::tan(double(i)*M_PI*2.0/double(NUM_ANGLE_UNITS)))/std::log(tangent_max)*2047.0+0.5;
    assert(j>-2048);
    assert(j<+2048);
    logtan_table[i]=j;
    //printf("logtan_table[%d] = %d\n",i,(int)logtan_table[i]);
  }

  printf("correction: %f\n",(float)(std::log(32.0*1.0/128.0)/std::log(tangent_max)*2047.0));

  for(int i=0;i<4096;++i)
  {
    int64_t j=(i==0)?0:std::log(double(i))/std::log(tangent_max)*2047.0+0.5;
    assert(j>=0);
    assert(j<4096);
    log_table[i]=j;
    //printf("log_table[%d] = %d\n",i,(int)log_table[i]);
  }

  for(int i=0;i<4096;++i)
  {
    int64_t j=std::min(std::pow(double(tangent_max), double(i)/2047.0)+0.5, 4095.0);
    assert(j>=0);
    assert(j<4096);
    exp_table[i] = j;
    printf("exp_table[%d] = %d\n",i,(int)exp_table[i]);
  }

  // double slope=double(wall_height)*std::sin(theta)/adjacent/128.0;
  // double ramp_angle=std::atan(slope);

  static const double log_sinemin=std::log(double(wall_height)/32.0*std::sin(1.0*M_PI*2.0/double(NUM_ANGLE_UNITS)))/std::log(tangent_max)*2047.0;
  static const double log_sinemax=std::log(double(wall_height)/32.0*std::sin(double(NUM_ANGLE_UNITS/4-1)*M_PI*2.0/double(NUM_ANGLE_UNITS)))/std::log(tangent_max)*2047.0;

  printf("log_sinemin=%f\n",(float)log_sinemin);
  printf("log_sinemax=%f\n",(float)log_sinemax);

  for(int i=0;i<NUM_ANGLE_UNITS/2;++i)
  {
    //printf("c=%f\n",(float)double(wall_height)*std::sin(double(i)*M_PI*2.0/double(NUM_ANGLE_UNITS)));
    int64_t j=/*std::ceil(-log_sinemin)+*/std::log(double(wall_height)*std::abs(std::sin(double(i)*M_PI*2.0/double(NUM_ANGLE_UNITS))))/std::log(tangent_max)*2047.0+0.0;
    if(i==0||i==512)
      j=1024;
    assert(j>=0);
    assert(j<4096);
    logsin_table[i]=j;
    printf("logsin_table[%d] = %d\n",i,(int)logsin_table[i]);
  }

  for(int i=0;i<2048;++i)
  {
    double slope=std::pow(tangent_max, (double(i*2)/*-std::ceil(-log_sinemin)*/)/2047.0)/128.0;//*(i>4096?+1:-1);
    //printf("slope=%f\n",(float)slope);
    int64_t j=std::atan(slope)*double(NUM_ANGLE_UNITS)/(M_PI*2.0)*double(num_ramp_angles-1)*4.0/double(NUM_ANGLE_UNITS)+0.5;
    assert(j>=0);
    assert(j<num_ramp_angles);
    rampindex_table[i] = j;
    //printf("rampindex_table[%d] = %d\n",i,(int)rampindex_table[i]);
  }

  // rampindex_table[log_sin-log_adjacent]

  for(int i=0;i<height_over_dist_table_size;++i)
  {
    double distance=double(i)*sqrt(127.0*127.0*2.0)/double(height_over_dist_table_size-1);
    //printf("distance=%f\n",distance);
    fflush(stdout);
    int64_t j=(i==0) ? 4096.0 : (4095.0/distance+0.5);
    assert(j>=0);
    //assert(j<=4097);
    j=std::min(j,(int64_t)4096);
    height_over_dist_table[i]=j;
  }

  for(int i=0;i<128;++i)
  {
    for(int j=0;j<128;++j)
    {
      if(i>j)
        continue;

      const int64_t opp=i,adj=j;
      const int64_t height_over_dist_index = sqrt(opp*opp+adj*adj)/sqrt(127.0*127.0*2.0)*double(height_over_dist_table_size-1)+0.5;
      assert(height_over_dist_index < height_over_dist_table_size);
      assert(height_over_dist_index >= 0);

      int64_t angle = 0;

      if(i==j)
      {
        assert(angle>=0);
        assert(angle<NUM_ANGLE_UNITS/8);
        assert(angle<256);
        assert(height_over_dist_index<256);

        //uint16_t data = (uint16_t)angle | (height_over_dist_index << 7);

        polar_table[i+j*128]=angle;
        polar_table[j+i*128]=height_over_dist_index;
      }
      else
      {
        angle = std::min(atan((double)opp / (double)adj) / (M_PI / 4.0) * (double)NUM_ANGLE_UNITS / 8.0+0.5,(double)NUM_ANGLE_UNITS/8-1);

        assert(angle>=0);
        assert(angle<NUM_ANGLE_UNITS/8);
        assert(angle<256);
        assert(height_over_dist_index<256);

        //uint16_t data = (uint16_t)angle | (height_over_dist_index << 7);

        polar_table[i+j*128]=angle;
        polar_table[j+i*128]=height_over_dist_index;
      }
      //printf("%02d ", (int)angle);

    }
    //printf("\n");
  }

  printf("tangent_max=%f\n",tangent_max);

  for(int i=0;i<ANGLE_UNIT_FOV/2;++i)
  {
    int64_t j=(i==0) ? VIRTUAL_FRAME_WIDTH / 2 : VIRTUAL_FRAME_WIDTH / 2 - tan((double)i / ((double)ANGLE_UNIT_FOV / 2.0) * M_PI / 4.0) * (double)VIRTUAL_FRAME_WIDTH / 2.0+0.5;
    assert(j>=-VIRTUAL_FRAME_WIDTH/2);
    assert(j<=+VIRTUAL_FRAME_WIDTH/2);
    x_angle_table[i] = j;
    //printf("x_angle_table[%d] = %d\n",i,(int)x_angle_table[i]);
    fflush(stdout);
  }

  memset(ray_intersection_test_table,0,sizeof(ray_intersection_test_table));
  for(int angle=0;angle<NUM_ANGLE_UNITS;++angle)
  {
    // Orientation 0
    if(angle<=NUM_ANGLE_UNITS/2)
      ray_intersection_test_table[angle]|=(1<<0);

    // Orientation 1
    if(angle>=(NUM_ANGLE_UNITS*1)/4 && angle<=(NUM_ANGLE_UNITS*3)/4)
      ray_intersection_test_table[angle]|=(1<<1);

    // Orientation 2
    if(angle>=NUM_ANGLE_UNITS/2 || angle==0)
      ray_intersection_test_table[angle]|=(1<<2);

    // Orientation 3
    if(angle<=(NUM_ANGLE_UNITS*1)/4 || angle>=(NUM_ANGLE_UNITS*3)/4)
      ray_intersection_test_table[angle]|=(1<<3);
  }

  //printf("sizeof(polar_table) = %d\n",(int)sizeof(polar_table));
  printf("sizeof(x_angle_table) = %d\n",(int)sizeof(x_angle_table));
  printf("sizeof(ray_intersection_test_table) = %d\n",(int)sizeof(ray_intersection_test_table));
  //printf("sizeof(height_over_dist_table) = %d\n",(int)sizeof(height_over_dist_table));
  printf("sizeof(logtan_table) = %d\n",(int)sizeof(logtan_table));
  printf("sizeof(logsin_table) = %d\n",(int)sizeof(logsin_table));
  printf("sizeof(log_table) = %d\n",(int)sizeof(log_table));
  printf("sizeof(exp_table) = %d\n",(int)sizeof(exp_table));
  printf("sizeof(rampindex_table) = %d\n",(int)sizeof(rampindex_table));
  printf("sizeof(arctanlog_table) = %d\n",(int)sizeof(arctanlog_table));
  printf("sizeof(log_partial_sum_table) = %d\n",(int)sizeof(log_partial_sum_table));
  printf("sizeof(log_bob_table) = %d\n",sizeof(log_bob_table));
  printf("sizeof(camera_move_sinus_table) = %d\n",sizeof(camera_move_sinus_table));

/*
  uint32_t total_size=sizeof(polar_table)+
                      sizeof(x_angle_table)+
                      sizeof(ray_intersection_test_table)+
                      sizeof(height_over_dist_table)+
      sizeof(logtan_table)+sizeof(log_table)+sizeof(exp_table)+
      sizeof(rampindex_table)+sizeof(logsin_table);
*/

  uint32_t total_size=sizeof(x_angle_table)+
                      sizeof(ray_intersection_test_table)+
                      sizeof(logtan_table)+
                      sizeof(log_table)+
                      sizeof(exp_table)+
                      sizeof(rampindex_table)*2+
                      sizeof(logsin_table)+
                      sizeof(arctanlog_table)+
                      sizeof(log_partial_sum_table)+
                      sizeof(log_bob_table)+
                      sizeof(camera_move_sinus_table);

  printf("total_size=%u\n",total_size);

  {
    FILE* out=fopen("x_angle_table.bin","wb");
    fwrite(x_angle_table, 1, sizeof(x_angle_table), out);
    fclose(out);
  }

  {
    FILE* out=fopen("ray_intersection_test_table.bin","wb");
    fwrite(ray_intersection_test_table, 1, sizeof(ray_intersection_test_table), out);
    fclose(out);
  }

  {
    FILE* out=fopen("logtan_table.bin","wb");
    fwrite(logtan_table, 1, sizeof(logtan_table), out);
    fclose(out);
  }

  {
    FILE* out=fopen("log_table.bin","wb");
    fwrite(log_table, 1, sizeof(log_table), out);
    fclose(out);
  }

  {
    FILE* out=fopen("exp_table.bin","wb");
    fwrite(exp_table, 1, sizeof(exp_table), out);
    fclose(out);
  }

  {
    FILE* out=fopen("logsin_table.bin","wb");
    fwrite(logsin_table, 1, sizeof(logsin_table), out);
    fclose(out);
  }

  {
    FILE* out=fopen("arctanlog_table.bin","wb");
    fwrite(arctanlog_table, 1, sizeof(arctanlog_table), out);
    fclose(out);
  }

  {
    FILE* out=fopen("log_partial_sum_table.bin","wb");
    fwrite(log_partial_sum_table, 1, sizeof(log_partial_sum_table), out);
    fclose(out);
  }

  {
    FILE* out=fopen("log_bob_table.bin","wb");
    fwrite(log_bob_table, 1, sizeof(log_bob_table), out);
    fclose(out);
  }

  {
    FILE* out=fopen("camera_move_sinus_table.bin","wb");
    fwrite(camera_move_sinus_table, 1, sizeof(camera_move_sinus_table), out);
    fclose(out);
  }

  {
    FILE* out=fopen("rampdesc_addr_table_L2R.asm","w");
    for(int i=0;i<1024;++i)
    {
      fprintf(out,"  dw ramp_descriptors+%d*RAMP_DESCRIPTOR_SIZE\n",
            ramp_unremap[rampindex_table[i*2]]);
    }
    fclose(out);
  }

  {
    FILE* out=fopen("rampdesc_addr_table_R2L.asm","w");
    for(int i=0;i<1024;++i)
    {
      fprintf(out,"  dw ramp_descriptors+%d*RAMP_DESCRIPTOR_SIZE\n",
            ramp_unremap[rampindex_table[i*2]+num_ramp_angles]);
    }
    fclose(out);
  }

  {
    for(int i=0;i<num_ramp_angles;++i)
    {
      ramp_slopes[i]=ramps[i].dydx;
    }
    FILE* out=fopen("ramp_slopes.bin","wb");
    fwrite(&ramp_slopes[0], 1, sizeof(ramp_slopes), out);
    fclose(out);
  }

  {
    FILE* out=fopen("ramp_slope_table_L2R.asm","w");
    for(int i=0;i<1024;++i)
    {
      fprintf(out,"  dw %d\n",
            +ramp_slopes[rampindex_table[i*2]]);
    }
    fclose(out);
  }

  {
    FILE* out=fopen("ramp_slope_table_R2L.asm","w");
    for(int i=0;i<1024;++i)
    {
      fprintf(out,"  dw %d\n",
            -ramp_slopes[rampindex_table[i*2]]);
    }
    fclose(out);
  }

}

static int num_intersect_ray_with_wall;
static int num_octant_angle;
static int num_project_post;
static int num_cheap_rays;
static int num_reused_points;

static uint16_t octantAngle(uint16_t opp, uint16_t adj)
{
  ++num_octant_angle;

  assert(opp<4096);
  assert(adj<4096);

/*
  This routine returns the byte offset from polar table base address for the given
  opposite and adjacent coordinates.

  This operation should be about 200 cycles or less.

  4096 = 128 * 32

  HL = opp
  DE = adj

  Result is in HL

    ; Put upper 5 bits of 12-bit number into high byte
    sla l   ; 10c
    rl h    ; 10c
    sla e   ; 10c
    rl d    ; 10c
    ; E is now in [0,256)
    ; L is now in [0,256)

    ; OR the high bytes together to determine the required shift [0,5] which is determined
    ; from the value [0,32) in the high byte.
    ld a,h  ;  5c
    or d    ;  5c

  Shift 0:
    ; Indices are in the upper 7 bits of L and E.
    srl e   ; 10c 2b
    ; E is now in [0,128)
    ; L is now in [0,256)

    ; Shift one bit from E into L
    srl e   ; 10c 2b
    rr l    ; 10c 2b
    ld h,e  ;  5c 1b
    ret     ; 11c 1b

  Shift 1:
    ; Indices are in the upper 6 bits of L and E and lower 1 bits of H and D.
    srl d   ; 10c 2b
    rr e    ; 10c 2b
    srl e   ; 10c 2b
    ; E is now in [0,128)

    sra h   ; 10c 2b
    rr l    ; 10c 2b
    ; L is now in [0,256)

    ; Shift one bit from E into L
    srl e   ; 10c 2b
    rr l    ; 10c 2b
    ld h,e  ;  5c 1b
    ret     ; 11c 1b

  Shift 2:
    ; Indices are in the upper 5 bits of L and E and lower 2 bits of H and D.
    REPT 2
      srl d   ; 10c 2b
      rr e    ; 10c 2b
    ENDM
    srl e   ; 10c 2b
    ; E is now in [0,128)

    REPT 2
      srl h   ; 10c 2b
      rr l    ; 10c 2b
    ENDM
    ; L is now in [0,256)

    ; Shift one bit from E into L
    srl e   ; 10c 2b
    rr l    ; 10c 2b
    ld h,e  ;  5c 1b
    ret     ; 11c 1b

  Shift 3:
    ; Indices are in the upper 4 bits of L and E and lower 3 bits of H and D.
    REPT 3
      srl d   ; 10c 2b
      rr e    ; 10c 2b
    ENDM
    srl e   ; 10c 2b
    ; E is now in [0,128)

    REPT 3
      srl h   ; 10c 2b
      rr l    ; 10c 2b
    ENDM
    ; L is now in [0,256)

    ; Shift one bit from E into L
    srl e   ; 10c 2b
    rr l    ; 10c 2b
    ld h,e  ;  5c 1b
    ret     ; 11c 1b

  Shift 4:
    ; Indices are in the upper 3 bits of L and E and lower 4 bits of H and D.
    REPT 3
      sla e   ; 10c 2b
      rl d    ; 10c 2b
    ENDM
    ; D is now in [0,128)

    REPT 4
      sla l   ; 10c 2b
      rl h    ; 10c 2b
    ENDM
    ; H is now in [0,256)

    ld l,h  ;  5c 1b
    ; Shift one bit from D into L
    srl d   ; 10c 2b
    rr l    ; 10c 2b
    ld h,d  ;  5c 1b
    ret     ; 11c 1b

  Shift 5:
    ; Indices are in the upper 2 bits of L and E and lower 5 bits of H and D.
    REPT 2
      sla e   ; 10c 2b
      rl d    ; 10c 2b
    ENDM
    ; D is now in [0,128)

    REPT 3
      sla l   ; 10c 2b
      rl h    ; 10c 2b
    ENDM
    ; H is now in [0,256)

    ld l,h  ;  5c 1b
    ; Shift one bit from D into L
    srl d   ; 10c 2b
    rr l    ; 10c 2b
    ld h,d  ;  5c 1b
    ret     ; 11c 1b
*/

  assert(opp > 0 || adj > 0);
  assert(adj >= opp);
  int shift_count=0;
  while(opp > 127 || adj > 127)
  {
    opp >>= 1;
    adj >>= 1;
    ++shift_count;
  }
  assert(shift_count<=5);

  if(opp==adj)
    return NUM_ANGLE_UNITS/8;

  cycle_count += 200;

  assert((opp+adj*128)<16384);

  return polar_table[opp+adj*128];
}

static uint16_t getAngleForPoint(int16_t x, int16_t y)
{
  if(x == 0 && y == 0)
    assert(false);

/*
  if(y == 0 && x > 0)
    return 0;
  if(x == 0 && y > 0)
    return NUM_ANGLE_UNITS / 4;
  if(y == 0 && x < 0)
    return NUM_ANGLE_UNITS / 2;
  if(x == 0 && y < 0)
    return (NUM_ANGLE_UNITS * 3) / 4;
*/

  const uint16_t absx = labs(x), absy = labs(y);
  uint16_t res=0xFFFF;
  if(y > 0)
  {
    // Birant 1
    if(x > 0)
    {
      // Quadrant 1
      if(absx > absy)
      {
        // Octant 1
        res = octantAngle(absy, absx);
      }
      else
      {
        // Octant 2
        res = NUM_ANGLE_UNITS / 8 + (NUM_ANGLE_UNITS / 8 - octantAngle(absx, absy));
      }
    }
    else
    {
      // Quadrant 2
      if(absx < absy)
      {
        // Octant 3
        res = NUM_ANGLE_UNITS / 4 + octantAngle(absx, absy);
      }
      else
      {
        // Octant 4
        res = (NUM_ANGLE_UNITS * 3) / 8 + (NUM_ANGLE_UNITS / 8 - octantAngle(absy, absx));
      }
    }
  }
  else
  {
    // Birant 2
    if(x < 0)
    {
      // Quadrant 3
      if(absx > absy)
      {
        // Octant 5
        res = NUM_ANGLE_UNITS / 2 + octantAngle(absy, absx);
      }
      else
      {
        // Octant 6
        res = (NUM_ANGLE_UNITS * 5) / 8 + (NUM_ANGLE_UNITS / 8 - octantAngle(absx, absy));
      }
    }
    else
    {
      // Quadrant 4
      if(absx < absy)
      {
        // Octant 7
        res = (NUM_ANGLE_UNITS * 3) / 4 + octantAngle(absx, absy);
      }
      else
      {
        // Octant 8
        res = (NUM_ANGLE_UNITS * 7) / 8 + (NUM_ANGLE_UNITS / 8 - octantAngle(absy, absx));
      }
    }
  }

  assert(res!=0xFFFF);

  //assert(res < NUM_ANGLE_UNITS);

  res = res & (NUM_ANGLE_UNITS - 1);
  return res;
  assert(false);
}

static void testAngleTable()
{
  int max_error = 0;
  for(int y = -MAX_COORDINATE; y <= +MAX_COORDINATE; ++y)
  {
    for(int x = -MAX_COORDINATE; x <= +MAX_COORDINATE; ++x)
    {
      if(x==0&&y==0)
        continue;

      int angle_unit = getAngleForPoint(x, y);

      assert(angle_unit>=0);
      assert(angle_unit<NUM_ANGLE_UNITS);

      double angle = atan2((double)y, (double)x);
      if(angle < 0.0)
        angle += 2.0 * M_PI;
      angle *= (double)NUM_ANGLE_UNITS / (M_PI * 2.0);
      //printf("(%5d, %5d) -> %5d (%5d)\n", x, y, angle_unit, (int)angle);
      int error = labs(angle_unit - (int)angle);
      if(error > max_error)
        max_error = error;
    }
  }
  printf("testAngleTable: max_error = %d\n", max_error);
}

static void blitVirtualFramebuffer(uint16_t* pixel_buffer)
{
  static const uint16_t palette[4] = { 0x0000, RGB888toRGB565(255,128,0),
                       RGB888toRGB565(155,155,155), RGB888toRGB565(255,255,255) };
  for(int y=0;y<VIRTUAL_FRAME_HEIGHT;++y)
    for(int x=0;x<VIRTUAL_FRAME_WIDTH;++x)
    {
      for(int v=0;v<VIRTUAL_PIXEL_HEIGHT*FRAME_ZOOM_FACTOR;++v)
        for(int u=0;u<VIRTUAL_PIXEL_WIDTH*FRAME_ZOOM_FACTOR;++u)
        {
          plotPixel(pixel_buffer, x * VIRTUAL_PIXEL_WIDTH*FRAME_ZOOM_FACTOR + u, y * VIRTUAL_PIXEL_HEIGHT*FRAME_ZOOM_FACTOR + v, palette[virtual_framebuffer[x+y*VIRTUAL_FRAME_WIDTH]&3]);
        }
    }
}

static void clearVirtualFramebuffer()
{
  for(int y=0;y<VIRTUAL_FRAME_HEIGHT;++y)
    for(int x=0;x<VIRTUAL_FRAME_WIDTH;++x)
    {
      virtual_framebuffer[x+y*VIRTUAL_FRAME_WIDTH] = 0;
    }
}

static void plotPixelVirtual(uint32_t x, uint32_t y, const uint8_t colour)
{
    if(x < VIRTUAL_FRAME_WIDTH && y < VIRTUAL_FRAME_HEIGHT)
        virtual_framebuffer[x + y * VIRTUAL_FRAME_WIDTH] = colour;
}

static void plotLineVirtual(int x0, int y0, int x1, int y1, const uint8_t col)
{
   const bool_t steep = labs(y1 - y0) > labs(x1 - x0);

   if(steep)
   {
      SWAP(int, x0, y0);
      SWAP(int, x1, y1);
   }

   if(x0 > x1)
   {
      SWAP(int, x0, x1);
      SWAP(int, y0, y1);
   }

   const int deltax = x1 - x0;
   const int deltay = labs(y1 - y0);
   int error = deltax / 2;
   int y = y0;
   const int ystep = (y0 < y1) ? 1 : -1;

   for(int x = x0; x <= x1; ++x)
   {
      if(steep)
         plotPixelVirtual(y, x, col);
      else
         plotPixelVirtual(x, y, col);

      error -= deltay;

      if(error < 0)
      {
         y += ystep;
         error += deltax;
      }
   }
}

#if 0
static const char map_geometry[64*64+1]=
  "................................................................"
  "................................................................"
  "................................................................"
  "................................................................"
  "................................................................"
  "................................................................"
  "................................................................"
  "................................................................"
  "................................................................"
  "................................................................"
  "................................................................"
  "................................................................"
  "................................................................"
  "................................................................"
  "................................................................"
  "................................................................"
  "................................................................"
  "................................................................"
  "................................................................"
  "........XXXXXXXXX..............................................."
  "........X.......X..............................................."
  "........X.@.....X..............................................."
  "........X.......X....XXXXX.XXX.................................."
  "........XXXXXX..X....X...XXX.X.................................."
  "............XX..XXXXXX.......X.................................."
  "............X................X.................................."
  "............XX..XXXXXX...XXXXX.................................."
  ".............X..X....X...X......................................"
  "...........XXX..XX...XXXXX......................................"
  "...........X.....X.............................................."
  "...........XXX..XX.............................................."
  ".............XXXX..............................................."
  "................................................................"
  "................................................................"
  "................................................................"
  "................................................................"
  "................................................................"
  "................................................................"
  "................................................................"
  "................................................................"
  "................................................................"
  "................................................................"
  "................................................................"
  "................................................................"
  "................................................................"
  "................................................................"
  "................................................................"
  "................................................................"
  "................................................................"
  "................................................................"
  "................................................................"
  "................................................................"
  "................................................................"
  "................................................................"
  "................................................................"
  "................................................................"
  "................................................................"
  "................................................................"
  "................................................................"
  "................................................................"
  "................................................................"
  "................................................................"
  "................................................................"
  "................................................................";
#else
static char map_geometry[64*64+1];
#endif

static char getMapChar(int x, int y)
{
  if(x<0 || y<0 || x>31 || y>31)
    return 'X';
  return map_geometry[x+y*32];
}

struct Wall
{
  double x0, y0, x1, y1;
  int orientation;
  int portal_room;
  int room;
  uint8_t ramp_index_cache_key;
};

static const int max_walls = 1<<20;
static Wall walls[max_walls];
static int num_walls = 0;

struct Room
{
  int x0,y0,x1,y1;

  int first_wall;

  int first_wall_q[4];
  int num_walls;

  // Wall references, one for each direction quadrant.
  // Sorted so that the first visible wall is draw from the right side
  // of the screen.
  int first_wall_ref[4];

  bool_t contains_orange;
  int orange_x,orange_y;
};

static const int max_rooms = 1<<20;
static Room rooms[max_rooms];
static int num_rooms = 0;

static const int max_wall_references=max_rooms*32;
static int wall_references[max_wall_references];
static int num_wall_references = 0;

static void calculateWall(int i)
{
  assert(i < num_walls);
  const int o = walls[i].orientation;
  int key;
  switch(o)
  {
    case 0:
      assert(walls[i].y0==walls[i].y1);
      key=(walls[i].y0*2)/UNITS_PER_CELL;
      assert(key>=0);
      assert(key<128);
      walls[i].ramp_index_cache_key = key | 0;
      break;
    case 1:
      assert(walls[i].x0==walls[i].x1);
      key=(walls[i].x0*2)/UNITS_PER_CELL;
      assert(key>=0);
      assert(key<128);
      walls[i].ramp_index_cache_key = key | 128;
      break;
    case 2:
      assert(walls[i].y0==walls[i].y1);
      key=(walls[i].y0*2)/UNITS_PER_CELL;
      assert(key>=0);
      assert(key<128);
      walls[i].ramp_index_cache_key = key | 0;
      break;
    case 3:
      assert(walls[i].x0==walls[i].x1);
      key=(walls[i].x0*2)/UNITS_PER_CELL;
      assert(key>=0);
      assert(key<128);
      walls[i].ramp_index_cache_key = key | 128;
      break;
  }
}

static int cell_visited[64*64];

struct Tile
{
  int first_room, num_rooms;
  int collision_index;
  int first_wall, num_walls;
  int arrangement_index;

  int cell_rooms[4*4];

  int entrances[4];
};

static const char *const tile_collision_strings[16] = {
    "XXXX" // 0000
    "XXXX"
    "XXXX"
    "XXXX",

    "XXaa" // 0001
    "XXaa"
    "XXXX"
    "XXXX",

    "XXaa" // 0010
    "XXaa"
    "XXXX"
    "XXXX",

    "XXaa" // 0011
    "XXaa"
    "XXXX"
    "XXXX",

    "XXaa" // 0100 (unused)
    "XXaa"
    "XXaa"
    "XXaa",

    "XXaa" // 0101
    "XXaa"
    "XXaa"
    "XXaa",

    "XXaa" // 0110
    "XXaa"
    "XXaa"
    "XXaa",

    "XXaa" // 0111
    "XXaa"
    "XXaa"
    "XXaa",

    "aaaa" // 1000 (unused)
    "aaaa"
    "XXXX"
    "XXXX",

    "aaaa" // 1001
    "aaaa"
    "XXXX"
    "XXXX",

    "aaaa" // 1010
    "aaaa"
    "XXXX"
    "XXXX",

    "aaaa" // 1011
    "aaaa"
    "XXXX"
    "XXXX",

    "aaaa" // 1100
    "aaaa"
    "XXbb"
    "XXbb",

    "aaaa" // 1101
    "aaaa"
    "XXbb"
    "XXbb",

    "aaaa" // 1110
    "aaaa"
    "XXbb"
    "XXbb",

    "aaaa" // 1111
    "aaaa"
    "XXbb"
    "XXbb",
  };

/*
static const char *const tile_collision_strings[16] = {
    "XXXX" // 0000
    "XXXX"
    "XXXX"
    "XXXX",

    "XXaa" // 0001
    "XXaa"
    "XXbX"
    "XXXX",

    "XXaa" // 0010
    "Xbaa"
    "XXXX"
    "XXXX",

    "XXaa" // 0011
    "XXaa"
    "XXXX"
    "XXXX",

    "XXaa" // 0100 (unused)
    "XXaa"
    "XXaa"
    "XXaa",

    "XXaa" // 0101
    "XXaa"
    "XXaa"
    "XXaa",

    "XXaa" // 0110
    "XXaa"
    "XXaa"
    "XXaa",

    "XXaa" // 0111
    "XXaa"
    "XXaa"
    "XXaa",

    "aaaa" // 1000 (unused)
    "aaaa"
    "XXXX"
    "XXXX",

    "aaaa" // 1001
    "aaaa"
    "XXXX"
    "XXXX",

    "aaaa" // 1010
    "aaaa"
    "XXXX"
    "XXXX",

    "aaaa" // 1011
    "aaaa"
    "XXXX"
    "XXXX",

    "aaaa" // 1100
    "aaaa"
    "XXbb"
    "XXbb",

    "aaaa" // 1101
    "aaaa"
    "XXbb"
    "XXbb",

    "aaaa" // 1110
    "aaaa"
    "XXbb"
    "XXbb",

    "aaaa" // 1111
    "aaaa"
    "XXbb"
    "XXbb",
  };
*/

/*
static const char *const tile_collision_strings[16] = {
    "XXXX" // 0000
    "XXXX"
    "XXXX"
    "XXXX",

    "X  X" // 0001
    "X  X"
    "X  X"
    "XXXX",

    "XXXX" // 0010
    "X   "
    "X   "
    "XXXX",

    "X  X" // 0011
    "X   "
    "X   "
    "XXXX",

    "XXXX" // 0100 (unused)
    "X  X"
    "X  X"
    "X  X",

    "X  X" // 0101
    "X  X"
    "X  X"
    "X  X",

    "XXXX" // 0110
    "X   "
    "X   "
    "X  X",

    "X  X" // 0111
    "XX  "
    "XX  "
    "X  X",

    "XXXX" // 1000 (unused)
    "   X"
    "   X"
    "XXXX",

    "X  X" // 1001
    "   X"
    "   X"
    "XXXX",

    "XXXX" // 1010
    "    "
    "    "
    "XXXX",

    "X  X" // 1011
    "    "
    " XX "
    "XXXX",

    "XXXX" // 1100
    "   X"
    "   X"
    "X  X",

    "X  X" // 1101
    "  XX"
    "  XX"
    "X  X",

    "XXXX" // 1110
    " XX "
    "    "
    "X  X",

    "X  X" // 1111
    "    "
    "    "
    "X  X",
  };
*/

static const char *const tile_arrangement_strings[16] = {
    "XXXXX" // 0000
    "XXXXX"
    "XXXXX"
    "XXXXX"
    "XXXXX",

    "XX XX" // 0001
    "X   X"
    "X   X"
    "X   X"
    "XXXXX",

    "XXXXX" // 0010
    "X   X"
    "X    "
    "X   X"
    "XXXXX",

    "X   X" // 0011
    "X    "
    "X    "
    "X    "
    "XXXXX",

    "XXXXX" // 0100 (unused)
    "X   X"
    "X   X"
    "X   X"
    "X   X",

    "X   X" // 0101
    "X   X"
    "X   X"
    "X   X"
    "X   X",

    "XXXXX" // 0110
    "     "
    "     "
    "     "
    "XXXXX",

    "   X" // 0111
    "    "
    "    "
    "    ",

    "XXXX" // 1000 (unused)
    "   X"
    "   X"
    "   X",

    "   X" // 1001
    "   X"
    "   X"
    "   X",

    "XXXX" // 1010
    "    "
    "    "
    "    ",

    "   X" // 1011
    "    "
    "    "
    "    ",

    "XXXX" // 1100
    "   X"
    "   X"
    "   X",

    "   X" // 1101
    "   X"
    "   X"
    "   X",

    "XXXX" // 1110
    "    "
    "    "
    "    ",

    "   X" // 1111
    "    "
    "    "
    "    ",
  };


static int maze[16*16];
static Tile tiles[16];

struct xorshift32_state {
    uint32_t a;
};

/* The state must be initialized to non-zero */
uint32_t xorshift32(struct xorshift32_state *state)
{
	/* Algorithm "xor" from p. 4 of Marsaglia, "Xorshift RNGs" */
	uint32_t x = state->a;
	x ^= x << 13;
	x ^= x >> 17;
	x ^= x << 5;
	return state->a = x;
}


static void generateMaze()
{
  {
    xorshift32_state state;
    state.a = 0xC234BEDD;
    // xorshift test
    printf("xorshift test:\n");
    for(int i=0;i<8;++i)
    {
      uint32_t x=xorshift32(&state);
      for(int j=0;j<32;++j)
      {
        printf("%c",(x&(1<<j))?'1':'0');
      }
    }
    printf("\n");
  }

  srand(12345);
  memset(maze,0,sizeof(maze));
  for(uint32_t y=0;y<8;++y)
    for(uint32_t x=0;x<8;++x)
    {
      if(rand()<RAND_MAX/2 && y>0)
      {
        // Open north
        maze[x+y*8]|=1;
        maze[x+((y-1)&7)*8]|=4;
      }
      else if(x<7)
      {
        // Open east
        maze[x+y*8]|=2;
        maze[((x+1)&7)+y*8]|=8;
      }
    }

/*
  // Expand hallways
  for(uint32_t y=0;y<16;++y)
    for(uint32_t x=0;x<16;++x)
    {
      if((maze[x+y*16]&15)==0b1010 && (maze[((x+1)&15)+y*16]&15)==0b1010)
      {
        maze[((x+0)&15)+y*16]|=16;
      }
      if((maze[x+y*16]&15)==0b0101 && (maze[x+((y-1)&15)*16]&15)==0b0101)
      {
        maze[x+((y+0)&15)*16]|=32;
      }
    }


  // Expand dead-ends
  for(uint32_t y=0;y<16;++y)
    for(uint32_t x=0;x<16;++x)
    {
      if((maze[x+y*16]&15)==0b0010)
      {
        maze[((x+0)&15)+y*16]|=16;
      }
      if((maze[x+y*16]&15)==0b0001)
      {
        maze[((x+0)&15)+y*16]|=32;
      }
    }
*/

#if 1
  memset(map_geometry,'X',sizeof(map_geometry));
  map_geometry[sizeof(map_geometry)-1]=0;

  for(int y=0;y<8;++y)
    for(int x=0;x<8;++x)
    {
      int i=maze[x+y*8]&15;
      assert(i>=0);
      assert(i<16);
/*
      if(maze[x+y*16]&32)
      {
        for(int v=0;v<4;++v)
          for(int u=0;u<4;++u)
          {
            if(tile_collision_strings[4][u+v*4]!=' ')
              map_geometry[x*4+u+(y*4+v)*64]='X';
          }
      }
      else if(maze[x+y*16]&16)
      {
        for(int v=0;v<4;++v)
          for(int u=0;u<4;++u)
          {
            if(tile_collision_strings[5][u+v*4]!=' ')
              map_geometry[x*4+u+(y*4+v)*64]='X';
          }
      }
      else*/
      {
        for(int v=0;v<4;++v)
          for(int u=0;u<4;++u)
          {
            map_geometry[x*4+u+(y*4+v)*32]=tile_collision_strings[tiles[i].collision_index][u+v*4];
          }
      }
    }

  //map_geometry[17+17*32]='@';
#endif
}

static uint16_t tile_cell_rooms[16*4*4];

static char getMapCharForTile(const char* map,uint32_t x,uint32_t y)
{
  if(x>3||y>3)
    return 'X';
  return map[x+y*4];
}

static void makeRoomsForTile(Tile& tile)
{
  memset(cell_visited,0,sizeof(cell_visited));

  const char* map=tile_collision_strings[tile.collision_index];

  tile.first_room=num_rooms;
  tile.num_rooms=0;

  tile.first_wall=num_walls;
  tile.num_walls=0;

  // Determine rooms
  for(int y=0;y<4;++y)
    for(int x=0;x<4;++x)
    {
      if(cell_visited[x+y*4]==0 && getMapCharForTile(map,x,y)!='X')
      {
        assert(num_rooms<max_rooms);
        Room* r=&rooms[num_rooms++];
        r->x0=x;
        r->y0=y;
        r->x1=x+1;
        r->y1=y+1;

        const char room_name=getMapCharForTile(map,x, y);
        assert(room_name=='a'||room_name=='b');

        tile.num_rooms++;

        bool grow_x,grow_y;

        do
        {
          grow_x=true;
          grow_y=true;

          for(int i=r->y0;i<r->y1;++i)
          {
            if(getMapCharForTile(map,r->x1,i)!=room_name||(r->x1>3)||cell_visited[r->x1+i*4]!=0)
            {
              grow_x=false;
              break;
            }
          }

          if(grow_x)
            ++r->x1;

          for(int i=r->x0;i<r->x1;++i)
          {
            if(getMapCharForTile(map,i,r->y1)!=room_name||r->y1>3||cell_visited[i+r->y1*4]!=0)
            {
              grow_y=false;
              break;
            }
          }

          if(grow_y)
            ++r->y1;
        }
        while(grow_x||grow_y);

        for(int y2=r->y0;y2<r->y1;++y2)
          for(int x2=r->x0;x2<r->x1;++x2)
          {
            assert(cell_visited[x2+y2*4]==0);
            cell_visited[x2+y2*4]=num_rooms;
          }
      }
    }

  for(int y=0;y<4;++y)
    for(int x=0;x<4;++x)
    {
      tile.cell_rooms[x+y*4]=cell_visited[x+y*4]-1;
    }

  printf("tile.first_room=%d\n",(int)tile.first_room);
  printf("tile.num_rooms=%d\n",(int)tile.num_rooms);

  assert(tile.num_rooms==1||tile.num_rooms==2);

  if(tile.num_rooms==1)
  {
    tile.entrances[0]=tile.first_room;
    tile.entrances[1]=tile.first_room;
    tile.entrances[2]=tile.first_room;
    tile.entrances[3]=tile.first_room;
  }
  else if(tile.num_rooms==2)
  {
    tile.entrances[0]=tile.first_room;
    tile.entrances[1]=tile.first_room;
    tile.entrances[2]=tile.first_room+1;
    tile.entrances[3]=tile.first_room;
  }

  // Walls must be directed clockwise.
  for(int k=tile.first_room;k<tile.first_room+tile.num_rooms;++k)
  {
    Room* r=&rooms[k];

    r->first_wall=num_walls;

    for(int i=r->x0;i<r->x1;++i)
    {
      if(getMapCharForTile(map,i,r->y0-1)=='X')
      {
        assert(num_walls < max_walls);
        int j = num_walls++;
        tile.num_walls++;
        walls[j].x0 = (i+0) * UNITS_PER_CELL;
        walls[j].y0 = r->y0 * UNITS_PER_CELL;
        walls[j].x1 = (i+1) * UNITS_PER_CELL;
        walls[j].y1 = r->y0 * UNITS_PER_CELL;
        walls[j].orientation = 0;
        walls[j].room = k;
        walls[j].portal_room = -1;
        calculateWall(j);
        continue;
      }
      const int ori=cell_visited[i+(r->y0-1)*4]-1;
      if(ori!=k)
      {
        assert(ori>=0);
        assert(num_walls < max_walls);
        int j = num_walls++;
        tile.num_walls++;
        walls[j].x0 = (i+0) * UNITS_PER_CELL;
        walls[j].y0 = r->y0 * UNITS_PER_CELL;
        walls[j].x1 = (i+1) * UNITS_PER_CELL;
        walls[j].y1 = r->y0 * UNITS_PER_CELL;
        walls[j].orientation = 0;
        walls[j].room = k;
        walls[j].portal_room = ori;
        calculateWall(j);
      }
    }

    for(int i=r->y0;i<r->y1;++i)
    {
      if(getMapCharForTile(map,r->x1,i)=='X')
      {
        assert(num_walls < max_walls);
        int j = num_walls++;
        tile.num_walls++;
        walls[j].x0 = r->x1 * UNITS_PER_CELL;
        walls[j].y0 = (i+0) * UNITS_PER_CELL;
        walls[j].x1 = r->x1 * UNITS_PER_CELL;
        walls[j].y1 = (i+1) * UNITS_PER_CELL;
        walls[j].orientation = 1;
        walls[j].room = k;
        walls[j].portal_room = -1;
        calculateWall(j);
        continue;
      }
      const int ori=cell_visited[r->x1+i*4]-1;
      if(ori!=k)
      {
        assert(ori>=0);
        assert(num_walls < max_walls);
        int j = num_walls++;
        tile.num_walls++;
        walls[j].x0 = r->x1 * UNITS_PER_CELL;
        walls[j].y0 = (i+0) * UNITS_PER_CELL;
        walls[j].x1 = r->x1 * UNITS_PER_CELL;
        walls[j].y1 = (i+1) * UNITS_PER_CELL;
        walls[j].orientation = 1;
        walls[j].room = k;
        walls[j].portal_room = ori;
        calculateWall(j);
      }
    }

    for(int i=r->x1-1;i>=r->x0;--i)
    {
      if(getMapCharForTile(map,i,r->y1)=='X')
      {
        assert(num_walls < max_walls);
        int j = num_walls++;
        tile.num_walls++;
        walls[j].x0 = (i+1) * UNITS_PER_CELL;
        walls[j].y0 = r->y1 * UNITS_PER_CELL;
        walls[j].x1 = (i+0) * UNITS_PER_CELL;
        walls[j].y1 = r->y1 * UNITS_PER_CELL;
        walls[j].orientation = 2;
        walls[j].room = k;
        walls[j].portal_room = -1;
        calculateWall(j);
        continue;
      }
      const int ori=cell_visited[i+r->y1*4]-1;
      if(ori!=k)
      {
        assert(ori>=0);
        assert(num_walls < max_walls);
        int j = num_walls++;
        tile.num_walls++;
        walls[j].x0 = (i+1) * UNITS_PER_CELL;
        walls[j].y0 = r->y1 * UNITS_PER_CELL;
        walls[j].x1 = (i+0) * UNITS_PER_CELL;
        walls[j].y1 = r->y1 * UNITS_PER_CELL;
        walls[j].orientation = 2;
        walls[j].room = k;
        walls[j].portal_room = ori;
        calculateWall(j);
      }
    }

    for(int i=r->y1-1;i>=r->y0;--i)
    {
      if(getMapCharForTile(map,r->x0-1,i)=='X')
      {
        assert(num_walls < max_walls);
        int j = num_walls++;
        tile.num_walls++;
        walls[j].x0 = r->x0 * UNITS_PER_CELL;
        walls[j].y0 = (i+1) * UNITS_PER_CELL;
        walls[j].x1 = r->x0 * UNITS_PER_CELL;
        walls[j].y1 = (i+0) * UNITS_PER_CELL;
        walls[j].orientation = 3;
        walls[j].room = k;
        walls[j].portal_room = -1;
        calculateWall(j);
        continue;
      }
      const int ori=cell_visited[(r->x0-1)+i*4]-1;
      if(ori!=k)
      {
        assert(ori>=0);
        assert(num_walls < max_walls);
        int j = num_walls++;
        tile.num_walls++;
        walls[j].x0 = r->x0 * UNITS_PER_CELL;
        walls[j].y0 = (i+1) * UNITS_PER_CELL;
        walls[j].x1 = r->x0 * UNITS_PER_CELL;
        walls[j].y1 = (i+0) * UNITS_PER_CELL;
        walls[j].orientation = 3;
        walls[j].room = k;
        walls[j].portal_room = ori;
        calculateWall(j);
      }
    }

    r->num_walls=num_walls-r->first_wall;
  }

  // Assign superportals
  for(int i=tile.first_wall;i<tile.first_wall+tile.num_walls;++i)
  {
    if(walls[i].portal_room<0)
    {
      if(walls[i].orientation==0 && walls[i].y0==0 && (tile.collision_index&1) && walls[i].x0>=2*UNITS_PER_CELL)
      {
        walls[i].portal_room=-2;
      }
      else if(walls[i].orientation==1 && walls[i].x0==4*UNITS_PER_CELL && (tile.collision_index&2) && walls[i].y0<2*UNITS_PER_CELL)
      {
        walls[i].portal_room=-3;
      }
      else if(walls[i].orientation==2 && walls[i].y0==4*UNITS_PER_CELL && (tile.collision_index&4))
      {
        walls[i].portal_room=-4;
      }
      else if(walls[i].orientation==3 && walls[i].x0==0 && (tile.collision_index&8))
      {
        walls[i].portal_room=-5;
      }
    }
  }


#if 1
  // Merge walls
  for(int i=tile.first_wall;i<tile.first_wall+tile.num_walls;++i)
  {
    for(int j=i+1;j<tile.first_wall+tile.num_walls;++j)
    {
      if( walls[i].orientation != walls[j].orientation ||
          walls[i].portal_room != walls[j].portal_room ||
          walls[i].room != walls[j].room)
        continue;
      if((walls[i].x0 == walls[j].x1 && walls[i].y0 == walls[j].y1) ||
         (walls[i].x1 == walls[j].x0 && walls[i].y1 == walls[j].y0) ||
         (walls[i].x0 == walls[j].x0 && walls[i].y0 == walls[j].y0) ||
         (walls[i].x1 == walls[j].x1 && walls[i].y1 == walls[j].y1))
      {
        if(walls[i].orientation == 0)
        {
          const int temp=walls[i].x0;
          walls[i].x0 = fmin(fmin(temp, walls[j].x1), fmin(walls[i].x1, walls[j].x0));
          walls[i].x1 = fmax(fmax(temp, walls[j].x1), fmax(walls[i].x1, walls[j].x0));
          assert(walls[i].y0 == walls[i].y1);
        }
        else if(walls[i].orientation == 1)
        {
          const int temp=walls[i].y0;
          walls[i].y0 = fmin(fmin(temp, walls[j].y1), fmin(walls[i].y1, walls[j].y0));
          walls[i].y1 = fmax(fmax(temp, walls[j].y1), fmax(walls[i].y1, walls[j].y0));
          assert(walls[i].x0 == walls[i].x1);
        }
        else if(walls[i].orientation == 2)
        {
          const int temp=walls[i].x0;
          walls[i].x0 = fmax(fmax(temp, walls[j].x1), fmax(walls[i].x1, walls[j].x0));
          walls[i].x1 = fmin(fmin(temp, walls[j].x1), fmin(walls[i].x1, walls[j].x0));
          assert(walls[i].y0 == walls[i].y1);
        }
        else if(walls[i].orientation == 3)
        {
          const int temp=walls[i].y0;
          walls[i].y0 = fmax(fmax(temp, walls[j].y1), fmax(walls[i].y1, walls[j].y0));
          walls[i].y1 = fmin(fmin(temp, walls[j].y1), fmin(walls[i].y1, walls[j].y0));
          assert(walls[i].x0 == walls[i].x1);
        }

        for(int k=tile.first_room;k<tile.first_room+tile.num_rooms;++k)
        {
          assert(rooms[k].num_walls>0);
          if(j>=rooms[k].first_wall && j<(rooms[k].first_wall+rooms[k].num_walls))
          {
            rooms[k].num_walls--;
            ++k;
            for(;k<tile.first_room+tile.num_rooms;++k)
              rooms[k].first_wall--;
            break;
          }
        }

        tile.num_walls--;
        num_walls--;
        for(int k=j;k<tile.first_wall+tile.num_walls;++k)
          walls[k] = walls[k+1];
        i--;
        break;
      }
    }
  }
#endif


  for(int k=tile.first_room;k<tile.first_room+tile.num_rooms;++k)
  {
    // Add (almost) another cycle for easy quadrant indexing
    assert(rooms[k].num_walls>0);
    const int n=rooms[k].num_walls;
    const int j_start=rooms[k].first_wall;
    const int j_end=j_start+n-1;
    for(int j=j_start;j<j_end;++j)
    {
      // Insert a copy of this wall
      int target=j+n;
      for(int i=num_walls;i>target;--i)
        walls[i]=walls[i-1];

      walls[target]=walls[j];
      num_walls++;
    }
    for(int j=k+1;j<tile.first_room+tile.num_rooms;++j)
      rooms[j].first_wall+=n-1;
  }

  // Determine the 4 quadrant orderings for each room.

  for(int k=tile.first_room;k<tile.first_room+tile.num_rooms;++k)
  {
    static const int quadrant_directions[4][2] = {
        { +1, +1 }, { -1, +1 }, { -1, -1 }, { +1, -1 }
      };
    Room *const r=&rooms[k];
    for(int j=0;j<4;++j)
    {
      int min_d=(1<<29),min_k=-1;
      for(int k=0;k<r->num_walls;++k)
      {
        const int i=r->first_wall+k;
        int d=walls[i].x0*quadrant_directions[j][0]+walls[i].y0*quadrant_directions[j][1];
        if(d<min_d)
        {
          min_d=d;
          min_k=k;
        }
      }
      assert(min_k>=0);

      rooms[k].first_wall_q[j]=r->first_wall+min_k;
    }
  }

  printf("tile.first_wall=%d\n",(int)tile.first_wall);
  printf("tile.num_walls=%d\n",(int)tile.num_walls);
}


static void generateTiles()
{
  num_rooms=0;
  memset(rooms,0,sizeof(rooms));
  memset(tiles,0,sizeof(tiles));
  memset(walls, 0, sizeof(walls));

  int dups=0;

  for(int i=0;i<16;++i)
  {
    printf("** tile %d **\n",i);

    tiles[i].collision_index=i;
    tiles[i].first_room=num_rooms;
    if(i==0)
    {
      tiles[i].num_rooms=0; // Inaccessible
      continue;
    }

/*
    makeRoomsForTile(tiles[i]);

    for(int j=0;j<i;++j)
    {
      bool dup=true;
      if(tiles[i].num_rooms!=tiles[j].num_rooms ||
         tiles[i].num_walls!=tiles[j].num_walls)
      {
        dup=false;
      }
      for(int k=0;k<tiles[i].num_walls;++k)
      {
        if(walls[tiles[i].first_wall+k].orientation!=walls[tiles[j].first_wall+k].orientation ||
           walls[tiles[i].first_wall+k].x0!=walls[tiles[j].first_wall+k].x0 ||
           walls[tiles[i].first_wall+k].x1!=walls[tiles[j].first_wall+k].x1 ||
           walls[tiles[i].first_wall+k].y0!=walls[tiles[j].first_wall+k].y0 ||
           walls[tiles[i].first_wall+k].y1!=walls[tiles[j].first_wall+k].y1
          )
        {
          dup=false;
        }
        if((walls[tiles[i].first_wall+k].portal_room<0 ||
            walls[tiles[j].first_wall+k].portal_room<0) && walls[tiles[i].first_wall+k].portal_room!=walls[tiles[j].first_wall+k].portal_room)
        {
          dup=false;
        }
      }
      if(dup)
      {
        ++dups;

        for(int k=tiles[i].first_room;k<tiles[i].first_room+tiles[i].num_rooms;++k)
          memcpy(&rooms[k],&rooms[k+tiles[i].num_rooms],sizeof(Room));

        num_rooms-=tiles[i].num_rooms;

        for(int k=tiles[i].first_wall;k<tiles[i].first_wall+tiles[i].num_walls;++k)
          memcpy(&walls[k],&walls[k+tiles[i].num_walls],sizeof(Wall));

        num_walls-=tiles[i].num_walls;

        memcpy(&tiles[i],&tiles[j],sizeof(Tile));
      }

    }
*/
  }

  printf("dups = %d\n",dups);
}

static void makeRooms(camera_t* camera)
{
  num_walls=0;
  memset(walls, 0, sizeof(walls));

  camera->px = 17*UNITS_PER_CELL+UNITS_PER_CELL/2;
  camera->py = 17*UNITS_PER_CELL+UNITS_PER_CELL/2;
  camera->angle = 0;

/*
  for(int y=0;y<32;++y)
    for(int x=0;x<32;++x)
    {
      char c=getMapChar(x, y);
      if(c=='@')
      {
        camera->px = x*UNITS_PER_CELL+UNITS_PER_CELL/2;
        camera->py = y*UNITS_PER_CELL+UNITS_PER_CELL/2;
        camera->angle = 0;
      }
    }
*/

  num_rooms=0;
  memset(cell_visited,0,sizeof(cell_visited));
  memset(rooms,0,sizeof(rooms));

  for(int y=0;y<32;++y)
    for(int x=0;x<32;++x)
    {
      if(cell_visited[x+y*32]==0 && getMapChar(x, y)!='X')
      {
        assert(num_rooms<max_rooms);
        Room* r=&rooms[num_rooms++];
        r->x0=x;
        r->y0=y;
        r->x1=x+1;
        r->y1=y+1;

        const char room_name=getMapChar(x, y);
        //assert(room_name=='a'||room_name=='b');

        bool grow_x,grow_y;

        do
        {
          grow_x=true;
          grow_y=true;

          for(int i=r->y0;i<r->y1;++i)
          {
            if(getMapChar(r->x1,i)=='X'||(r->x1>31)||cell_visited[r->x1+i*32]!=0)
            //if(getMapChar(r->x1,i)!=room_name||(r->x1>31)||cell_visited[r->x1+i*32]!=0)
            {
              grow_x=false;
              break;
            }
          }

          if(grow_x)
            ++r->x1;

          for(int i=r->x0;i<r->x1;++i)
          {
            if(getMapChar(i,r->y1)=='X'||r->y1>31||cell_visited[i+r->y1*32]!=0)
            //if(getMapChar(i,r->y1)!=room_name||r->y1>31||cell_visited[i+r->y1*32]!=0)
            {
              grow_y=false;
              break;
            }
          }

          if(grow_y)
            ++r->y1;
        }
        while(grow_x||grow_y);

        for(int y2=r->y0;y2<r->y1;++y2)
          for(int x2=r->x0;x2<r->x1;++x2)
          {
            assert(cell_visited[x2+y2*32]==0);
            cell_visited[x2+y2*32]=num_rooms;
          }

        //return;
      }
    }

  // Assign oranges to rooms
  for(int y=0;y<32;++y)
    for(int x=0;x<32;++x)
    {
      const int room_index=cell_visited[x+y*32]-1;
      if(room_index<0)
        continue;
      char c=getMapChar(x, y);
      if(c=='O')
      {
        assert(!rooms[room_index].contains_orange);
        rooms[room_index].orange_x = x*UNITS_PER_CELL+UNITS_PER_CELL/2;
        rooms[room_index].orange_y = y*UNITS_PER_CELL+UNITS_PER_CELL/2;
        rooms[room_index].contains_orange = TRUE;
      }
    }

  // Walls must be directed clockwise.

  for(int k=0;k<num_rooms;++k)
  {
    Room* r=&rooms[k];

    r->first_wall=num_walls;

    //if(r->y0>0)
      for(int i=r->x0;i<r->x1;++i)
      {
        const int ori=cell_visited[i+(r->y0-1)*32]-1;
        if(getMapChar(i,r->y0-1)=='X')
        {
          assert(num_walls < max_walls);
          int j = num_walls++;
          walls[j].x0 = (i+0) * UNITS_PER_CELL;
          walls[j].y0 = r->y0 * UNITS_PER_CELL;
          walls[j].x1 = (i+1) * UNITS_PER_CELL;
          walls[j].y1 = r->y0 * UNITS_PER_CELL;
          walls[j].orientation = 0;
          walls[j].room = k;
          walls[j].portal_room = -1;
          calculateWall(j);
        }
        else if(ori!=k)
        {
          assert(ori>=0);
          assert(num_walls < max_walls);
          int j = num_walls++;
          walls[j].x0 = (i+0) * UNITS_PER_CELL;
          walls[j].y0 = r->y0 * UNITS_PER_CELL;
          walls[j].x1 = (i+1) * UNITS_PER_CELL;
          walls[j].y1 = r->y0 * UNITS_PER_CELL;
          walls[j].orientation = 0;
          walls[j].room = k;
          walls[j].portal_room = ori;
          calculateWall(j);
        }
      }

    //if(r->x1<32)
      for(int i=r->y0;i<r->y1;++i)
      {
        const int ori=cell_visited[r->x1+i*32]-1;
        if(getMapChar(r->x1,i)=='X')
        {
          assert(num_walls < max_walls);
          int j = num_walls++;
          walls[j].x0 = r->x1 * UNITS_PER_CELL;
          walls[j].y0 = (i+0) * UNITS_PER_CELL;
          walls[j].x1 = r->x1 * UNITS_PER_CELL;
          walls[j].y1 = (i+1) * UNITS_PER_CELL;
          walls[j].orientation = 1;
          walls[j].room = k;
          walls[j].portal_room = -1;
          calculateWall(j);
        }
        else if(ori!=k)
        {
          assert(ori>=0);
          assert(num_walls < max_walls);
          int j = num_walls++;
          walls[j].x0 = r->x1 * UNITS_PER_CELL;
          walls[j].y0 = (i+0) * UNITS_PER_CELL;
          walls[j].x1 = r->x1 * UNITS_PER_CELL;
          walls[j].y1 = (i+1) * UNITS_PER_CELL;
          walls[j].orientation = 1;
          walls[j].room = k;
          walls[j].portal_room = ori;
          calculateWall(j);
        }
      }

    //if(r->y1<32)
      for(int i=r->x1-1;i>=r->x0;--i)
      {
        const int ori=cell_visited[i+r->y1*32]-1;
        if(getMapChar(i,r->y1)=='X')
        {
          assert(num_walls < max_walls);
          int j = num_walls++;
          walls[j].x0 = (i+1) * UNITS_PER_CELL;
          walls[j].y0 = r->y1 * UNITS_PER_CELL;
          walls[j].x1 = (i+0) * UNITS_PER_CELL;
          walls[j].y1 = r->y1 * UNITS_PER_CELL;
          walls[j].orientation = 2;
          walls[j].room = k;
          walls[j].portal_room = -1;
          calculateWall(j);
        }
        else if(ori!=k)
        {
          assert(ori>=0);
          assert(num_walls < max_walls);
          int j = num_walls++;
          walls[j].x0 = (i+1) * UNITS_PER_CELL;
          walls[j].y0 = r->y1 * UNITS_PER_CELL;
          walls[j].x1 = (i+0) * UNITS_PER_CELL;
          walls[j].y1 = r->y1 * UNITS_PER_CELL;
          walls[j].orientation = 2;
          walls[j].room = k;
          walls[j].portal_room = ori;
          calculateWall(j);
        }
      }

    //if(r->x0>0)
      for(int i=r->y1-1;i>=r->y0;--i)
      {
        const int ori=cell_visited[(r->x0-1)+i*32]-1;
        if(getMapChar(r->x0-1,i)=='X')
        {
          assert(num_walls < max_walls);
          int j = num_walls++;
          walls[j].x0 = r->x0 * UNITS_PER_CELL;
          walls[j].y0 = (i+1) * UNITS_PER_CELL;
          walls[j].x1 = r->x0 * UNITS_PER_CELL;
          walls[j].y1 = (i+0) * UNITS_PER_CELL;
          walls[j].orientation = 3;
          walls[j].room = k;
          walls[j].portal_room = -1;
          calculateWall(j);
        }
        else if(ori!=k)
        {
          assert(ori>=0);
          assert(num_walls < max_walls);
          int j = num_walls++;
          walls[j].x0 = r->x0 * UNITS_PER_CELL;
          walls[j].y0 = (i+1) * UNITS_PER_CELL;
          walls[j].x1 = r->x0 * UNITS_PER_CELL;
          walls[j].y1 = (i+0) * UNITS_PER_CELL;
          walls[j].orientation = 3;
          walls[j].room = k;
          walls[j].portal_room = ori;
          calculateWall(j);
        }
      }

    r->num_walls=num_walls-r->first_wall;
  }

#if 1
  for(int i=0;i<num_walls;++i)
  {
    for(int j=i+1;j<num_walls;++j)
    {
      if( walls[i].orientation != walls[j].orientation ||
          walls[i].portal_room != walls[j].portal_room ||
          walls[i].room != walls[j].room)
        continue;
      if((walls[i].x0 == walls[j].x1 && walls[i].y0 == walls[j].y1) ||
         (walls[i].x1 == walls[j].x0 && walls[i].y1 == walls[j].y0) ||
         (walls[i].x0 == walls[j].x0 && walls[i].y0 == walls[j].y0) ||
         (walls[i].x1 == walls[j].x1 && walls[i].y1 == walls[j].y1))
      {
        if(walls[i].orientation == 0)
        {
          const int temp=walls[i].x0;
          walls[i].x0 = fmin(fmin(temp, walls[j].x1), fmin(walls[i].x1, walls[j].x0));
          walls[i].x1 = fmax(fmax(temp, walls[j].x1), fmax(walls[i].x1, walls[j].x0));
          assert(walls[i].y0 == walls[i].y1);
        }
        else if(walls[i].orientation == 1)
        {
          const int temp=walls[i].y0;
          walls[i].y0 = fmin(fmin(temp, walls[j].y1), fmin(walls[i].y1, walls[j].y0));
          walls[i].y1 = fmax(fmax(temp, walls[j].y1), fmax(walls[i].y1, walls[j].y0));
          assert(walls[i].x0 == walls[i].x1);
        }
        else if(walls[i].orientation == 2)
        {
          const int temp=walls[i].x0;
          walls[i].x0 = fmax(fmax(temp, walls[j].x1), fmax(walls[i].x1, walls[j].x0));
          walls[i].x1 = fmin(fmin(temp, walls[j].x1), fmin(walls[i].x1, walls[j].x0));
          assert(walls[i].y0 == walls[i].y1);
        }
        else if(walls[i].orientation == 3)
        {
          const int temp=walls[i].y0;
          walls[i].y0 = fmax(fmax(temp, walls[j].y1), fmax(walls[i].y1, walls[j].y0));
          walls[i].y1 = fmin(fmin(temp, walls[j].y1), fmin(walls[i].y1, walls[j].y0));
          assert(walls[i].x0 == walls[i].x1);
        }

        for(int k=0;k<num_rooms;++k)
        {
          assert(rooms[k].num_walls>0);
          if(j>=rooms[k].first_wall && j<(rooms[k].first_wall+rooms[k].num_walls))
          {
            rooms[k].num_walls--;
            ++k;
            for(;k<num_rooms;++k)
              rooms[k].first_wall--;
            break;
          }
        }

        num_walls--;
        for(int k=j;k<num_walls;++k)
          walls[k] = walls[k+1];
        i--;
        break;
      }
    }
  }
#endif

  printf("num_rooms=%d\n",num_rooms);


  for(int k=0;k<num_rooms;++k)
  {
    // Add (almost) another cycle for easy quadrant indexing
    assert(rooms[k].num_walls>0);
    const int n=rooms[k].num_walls;
    const int j_start=rooms[k].first_wall;
    const int j_end=j_start+n-1;
    for(int j=j_start;j<j_end;++j)
    {
      // Insert a copy of this wall
      int target=j+n;
      for(int i=num_walls;i>target;--i)
        walls[i]=walls[i-1];

      walls[target]=walls[j];
      num_walls++;
    }
    for(int j=k+1;j<num_rooms;++j)
      rooms[j].first_wall+=n-1;
  }

  // Determine the 4 quadrant orderings for each room.

  for(int k=0;k<num_rooms;++k)
  {
    static const int quadrant_directions[4][2] = {
        { +1, +1 }, { -1, +1 }, { -1, -1 }, { +1, -1 }
      };
    Room *const r=&rooms[k];
    for(int j=0;j<4;++j)
    {
      int min_d=(1<<29),min_k=-1;
      for(int k=0;k<r->num_walls;++k)
      {
        const int i=r->first_wall+k;
        int d=walls[i].x0*quadrant_directions[j][0]+walls[i].y0*quadrant_directions[j][1];
        if(d<min_d)
        {
          min_d=d;
          min_k=k;
        }
      }
      assert(min_k>=0);

      rooms[k].first_wall_q[j]=r->first_wall+min_k;
    }
  }

  // Determine the 4 quadrant orderings for each room.

  memset(wall_references,0,sizeof(wall_references));
  num_wall_references = 0;

  for(int room_index=0;room_index<num_rooms;++room_index)
  {
    static const int quadrant_directions[4][2] = {
        { +1, +1 }, { -1, +1 }, { -1, -1 }, { +1, -1 }
      };
    Room *const r=&rooms[room_index];
    for(int j=0;j<4;++j)
    {
      int min_d=(1<<29),min_k=-1;
      for(int k=0;k<r->num_walls;++k)
      {
        const int i=r->first_wall+k;
        int d=walls[i].x0*quadrant_directions[j][0]+walls[i].y0*quadrant_directions[j][1];
        if(d<min_d)
        {
          min_d=d;
          min_k=k;
        }
      }
      assert(min_k>=0);
      assert(num_wall_references<max_wall_references);
      r->first_wall_ref[j]=num_wall_references;
      for(int k=0;k<r->num_walls;++k)
      {
        wall_references[r->first_wall_ref[j]+k]=r->first_wall+((k+min_k)%r->num_walls);
      }
      num_wall_references+=r->num_walls;
    }
  }

  for(int i=0;i<num_walls;++i)
  {
    printf("walls[%d].x0=%d\n",i,(int)walls[i].x0);
  }

  printf("makeRooms: num_walls = %d\n",(int)num_walls);
  printf("makeRooms: num_rooms = %d\n",(int)num_rooms);
}

static void debugDrawWalls(uint16_t* pixel_buffer)
{
  for(int i=0;i<num_walls;++i)
  {
    const uint64_t colour=walls[i].portal_room==-1 ? RGB888toRGB565(255,0,255) : RGB888toRGB565(0,255,255);
    plotLine(pixel_buffer, walls[i].x0, walls[i].y0, walls[i].x1, walls[i].y1, colour);
    clipAndPlotBox(pixel_buffer, walls[i].x0-2, walls[i].y0-2, 4,4, colour);
    clipAndPlotBox(pixel_buffer, walls[i].x1-2, walls[i].y1-2, 4,4, colour);
  }
}

static void debugDrawRooms(uint16_t* pixel_buffer)
{
  for(int i=0;i<num_rooms;++i)
  {
    uint64_t colour=RGB888toRGB565(100+(i*16)&127,100+((i+1)*4)&127,100+((i+2)*16)&127);
    clipAndPlotBox(pixel_buffer, rooms[i].x0*16, rooms[i].y0*16,
      rooms[i].x1*16-rooms[i].x0*16,rooms[i].y1*16-rooms[i].y0*16, colour);
  }
}

static uint32_t walls_drawn=0,rooms_traversed=0,portals_entered=0;

struct Line
{
  int16_t x0,y0,x1,y1;
  uint16_t colour;
};

static const int frame_width=VIRTUAL_FRAME_WIDTH;
static const int frame_height=VIRTUAL_FRAME_HEIGHT/2;

static int ramp_comp(const void* a,const void* b)
{
  int i=*(int*)a;
  int j=*(int*)b;
  const Ramp* ra=&ramps[i];
  const Ramp* rb=&ramps[j];

  if(ra->w==frame_width && ra->h==frame_height)
    return +1;

  if(rb->w==frame_width && rb->h==frame_height)
    return -1;

  if(ra->scaling_axis > rb->scaling_axis)
    return +1;
  if(ra->scaling_axis < rb->scaling_axis)
    return -1;

  if(ra->direction > rb->direction)
    return +1;
  if(ra->direction < rb->direction)
    return -1;

  return 0;
}

// https://media.xiph.org/svt/SGIIMAGESPEC

static unsigned char getbyte(FILE *inf)
{
  unsigned char buf[1];
  fread(buf,1,1,inf);
  return buf[0];
}

static unsigned short getshort(FILE *inf)
{
  unsigned char buf[2];
  fread(buf,2,1,inf);
  return (buf[0]<<8)+(buf[1]<<0);
}

static unsigned long getlong(FILE *inf)
{
  unsigned char buf[4];
  fread(buf,4,1,inf);
  return (buf[0]<<24)+(buf[1]<<16)+(buf[2]<<8)+(buf[3]<<0);
}

uchar* loadFromRGBAFile(FILE *const src, int* out_w, int* out_h)
{
  short magic = getshort(src);
  assert(magic == 474);
  char storage = getbyte(src);
  char bpc = getbyte(src);
  short dimension = getshort(src);
  short xsize = getshort(src);
  short ysize = getshort(src);
  short zsize = getshort(src);
  long pixmin = getlong(src);
  long pixmax = getlong(src);
  long dummy = getlong(src);
  char name[80] = { 0 };
  fread(name, 1, 80, src);
  long colormap = getlong(src);
  char dummy2[404];
  fread(dummy2, 1, 404, src);
  printf("magic = %u\n", magic);
  printf("storage = %u\n", storage);
  printf("bpc = %u\n", bpc);
  printf("dimension = %u\n", dimension);
  printf("xsize = %u\n", xsize);
  printf("ysize = %u\n", ysize);
  printf("zsize = %u\n", zsize);
  printf("pixmin = %u\n", pixmin);
  printf("pixmax = %u\n", pixmax);
  printf("name = '%s'\n", name);
  printf("colormap = %u\n", colormap);
  uchar* pixels = (uchar*)malloc(xsize * ysize * zsize);
  fread(pixels, 1, xsize * ysize * zsize, src);
  *out_w = xsize;
  *out_h = ysize;
  uchar* pixels2 = (uchar*)malloc(xsize * ysize * zsize);
  // re-interleave
  for(int y = 0; y < ysize; ++y)
  {
    for(int x = 0; x < xsize; ++x)
    {
      for(int z = 0; z < zsize; ++z)
      {
        pixels2[z + (x + (ysize-1-y) * xsize) * zsize] = pixels[(x + y * xsize) + z * xsize * ysize];
      }
    }
  }
  free(pixels);
  return pixels2;
}

static void putbyte(FILE *outf, unsigned char x)
{
  unsigned char buf[1] = { x };
  fwrite(buf,1,1,outf);
}

static void putshort(FILE *outf, unsigned short x)
{
  unsigned char buf[2] = { (unsigned char)(x >> 8), (unsigned char)(x & 0xff) };
  fwrite(buf,2,1,outf);
}

static void putlong(FILE *outf, unsigned long x)
{
  unsigned char buf[4] = { (unsigned char)(x >> 24),
                           (unsigned char)((x >> 16) & 0xff),
                           (unsigned char)((x >> 8) & 0xff),
                           (unsigned char)(x & 0xff)  };
  fwrite(buf,4,1,outf);
}

static void writeRGBAFile(FILE *const out, int w, int h, unsigned char* pixels)
{
    putshort(out, 474); // magic
    putbyte (out,   0); // storage
    putbyte (out,   1); // bpc
    putshort(out,   3); // dimensions
    putshort(out,   w); // xsize
    putshort(out,   h); // ysize
    putshort(out,   3); // zsize
    putlong (out,   0); // pixmin
    putlong (out,   0xffffff); // pixmax
    putlong (out,   0); // dummy
    char name[80] = { 0 };
    fwrite(name, 1, 80, out);
    putlong (out,   0); // colormap ID
    char dummy2[404];
    memset(dummy2, 0, sizeof(dummy2));
    fwrite(dummy2, 1, 404, out);

    unsigned char* pixels2 = (unsigned char*)malloc(w * h * 3);
    // re-interleave
    for(int y = 0; y < h; ++y)
    {
        for(int x = 0; x < w; ++x)
        {
            for(int z = 0; z < 3; ++z)
            {
                pixels2[(x + y * w) + z * w * h] = pixels[z + (x + (h-1-y) * w) * 3];
            }
        }
    }

    fwrite(pixels2, 1, w * h * 3, out);
    
    free(pixels2);
}

static void processFont()
{
  printf("Loading \"anuvverbubbla_8x8.rgba\"\n");
  FILE* in=fopen("anuvverbubbla_8x8.rgba","rb");
  int source_w=0,source_h=0;
  uchar* source_pixels=loadFromRGBAFile(in, &source_w, &source_h);

  //                           0123456789ABCDEF0123456789ABCDEF
  static const char charset[]="V9990 3D DEMO BY EDD BIDDULPH   "
                              ;

  static const int num_chars=sizeof(charset)-1;

  int target_w=256, target_h=8;

  uchar* target_pixels_2bpp=(uchar*)malloc(target_w/4*target_h);
  memset(target_pixels_2bpp,0,target_w/4*target_h);

  uchar* target_pixels=(uchar*)malloc(target_w*target_h*3);
  memset(target_pixels,0,target_w*target_h*3);

  int palette[4][3] = { { 255,   0, 255 },
                        {   0,   0,   0 },
                        { 120, 120, 120 },
                        { 255, 255, 255 },
                      };

  for(int i=0;i<num_chars;++i)
  {
    const int j=(int)charset[i]-32;
    assert(j>=0);
    assert(j<77);
    for(int y=0;y<8;++y)
      for(int x=0;x<8;++x)
      {
        const uchar *const p=&source_pixels[(j*8+x+y*source_w)*4+0];
        int k=0; // Transparent

        if(p[1]>128)
          k=2; // Inner
        else if(p[1]>64)
          k=2; // Shading
        else if(p[3]>64)
          k=3; // Outer

        const int ofs=((i*8)&255)+x+((i/32)*8+y)*target_w;

        target_pixels_2bpp[ofs/4]|=k<<((3-(ofs&3))*2);

        target_pixels[ofs*3+0]=palette[k][0];
        target_pixels[ofs*3+1]=palette[k][1];
        target_pixels[ofs*3+2]=palette[k][2];
      }
  }

  {
    FILE* out=fopen("font.rgba","wb");
    writeRGBAFile(out, target_w, target_h, target_pixels);
    fclose(out);
  }

  {
    FILE* out=fopen("font.bin","wb");
    fwrite(target_pixels_2bpp, 1, target_w/4*target_h, out);
    fclose(out);
  }

  {
    FILE* out=fopen("font_chars.asm","w");
    for(int i=0;i<num_chars;++i)
    {
      fprintf(out,"  dw FONT_VRAM_LOCATION_X+%3d, FONT_VRAM_LOCATION_Y+%3d ; '%c'\n",
          (i*8)%target_w, (i/(target_w/8))*8, charset[i]);
    }
    fclose(out);
  }
}

static void generateRamps()
{
  memset(ramps,0,sizeof(ramps));

  uint32_t num_pixels=0;
  uint32_t num_bytes=0;

  int num_full_frame=0;
  int num_x_scaled=0;
  int num_y_scaled=0;

  int template_ramp_0=-1;
  int template_ramp_1=-1;

  int num_x_scaled_0=0;
  int num_x_scaled_1=0;
  int num_y_scaled_0=0;
  int num_y_scaled_1=0;

  for(int j=0;j<2;++j)
  {
    for(int i=0;i<num_ramp_angles;++i)
    {
      const double th=M_PI/2.0*double(i)/double(num_ramp_angles);
      const double tx=tan(M_PI/2.0-th)*double(frame_height);
      const double ty=tan(th)*double(frame_width);
      const double fw=std::ceil(std::min(double(frame_width),tx+1.0/128.0));
      const double fh=std::ceil(std::min(double(frame_height),ty+1.0/128.0));
      const int w=(int)fw;
      const int h=(int)fh;
      assert(w>0.0);
      assert(h>0.0);
      assert(w<=frame_width);
      assert(h<=frame_height);
      assert(w<=256);
      assert(h<=256);
      num_pixels+=int(w)*int(h);

      Ramp *r=&ramps[j*num_ramp_angles+i];

      // frame_width=VIRTUAL_FRAME_WIDTH=256
      // frame_height=VIRTUAL_FRAME_HEIGHT/2=192/2=96;
      const double dydx=fh/fw*256.0;
      assert(dydx>=0.0);
      assert(dydx<65535.0);
      r->dydx=dydx;

      printf("r->dydx = %u\n",(uint32_t)r->dydx);

      r->pixels=(uint8_t*)malloc(w*h);
      r->size_2bpp=(w*h*2+7)/8;
      r->pixels_2bpp=(uint8_t*)malloc(r->size_2bpp);
      r->w=w;
      r->h=h;

      r->is_template=false;

      assert(w==frame_width || h==frame_height);
      if(w==frame_width && h==frame_height)
      {
        ++num_full_frame;
        if(j==0)
        {
          assert(template_ramp_0==-1);
          template_ramp_0=j*num_ramp_angles+i;
          r->is_template=true;
        }
        else if(j==1)
        {
          assert(template_ramp_1==-1);
          template_ramp_1=j*num_ramp_angles+i;
          r->is_template=true;
        }
      }

      r->gen_w=w-1;
      r->gen_h=h-1;

      r->scaling_axis=(w==frame_width)?1:0;
      r->direction=j;

      if(j==0)
        if(r->scaling_axis==0)
          ++num_x_scaled_0;
        else
          ++num_y_scaled_0;
      else
        if(r->scaling_axis==0)
          ++num_x_scaled_1;
        else
          ++num_y_scaled_1;


      if(r->scaling_axis==0)
        ++num_x_scaled;
      else
        ++num_y_scaled;

      if(r->scaling_axis==0)
      {
        double scale=double(frame_width)/double(w)*256.0;
        assert(scale>=0.0);
        assert(scale<65536.0);
        r->gen_dSdD=scale;

        double scale_highprec=double(frame_width)/double(w)*16777216.0+0.5;
        assert(scale_highprec>=0.0);
        assert(scale_highprec<4294967296.0);
        r->gen_dSdD_highprec=scale_highprec;
      }
      else
      {
        double scale=double(frame_height)/double(h)*256.0;
        assert(scale>=0.0);
        assert(scale<65536.0);
        r->gen_dSdD=scale;

        double scale_highprec=double(frame_height)/double(h)*16777216.0+0.5;
        assert(scale_highprec>=0.0);
        assert(scale_highprec<4294967296.0);
        r->gen_dSdD_highprec=scale_highprec;
      }

      printf("r->gen_dSdD = %u\n",(uint32_t)r->gen_dSdD);
      printf("r->gen_dSdD_highprec = %u\n",(uint32_t)r->gen_dSdD_highprec);

      printf("ramp w=%d, h=%d\n",w,h);

      memset(r->pixels_2bpp,0,r->size_2bpp);
      memset(r->pixels,0,w*h);

      num_bytes+=r->size_2bpp;

      if(j)
      {
        r->gen_u = 32768;
        r->gen_dudx = +32767.0 / double(w-1);
        r->gen_dudy = -32767.0 / double(h-1);
      }
      else
      {
        r->gen_u = 32768;
        r->gen_dudx = +32767.0 / double(w-1);
        r->gen_dudy = +32767.0 / double(h-1);
      }

      for(int y=0;y<h;++y)
        for(int x=0;x<w;++x)
        {
          uint16_t u = r->gen_u + r->gen_dudx * x + r->gen_dudy * y;
          if(u < 32768)
          {
            int offset_2bpp=(x+y*w)*2;
            assert(offset_2bpp>=0);
            assert(offset_2bpp/8<r->size_2bpp);
            r->pixels_2bpp[offset_2bpp/8] |= 3 << (offset_2bpp&7);
            r->pixels[x+y*w]=0xFF;
          }
        }

#if 0
      for(int y=0;y<h;++y)
        for(int x=0;x<w;++x)
        {
          double v=j ? double(h)*double(x)/double(w) : (h-double(h)*double(x)/double(w));
          if(y>=v)
          {
            int offset_2bpp=(x+y*w)*2;
            assert(offset_2bpp>=0);
            assert(offset_2bpp/8<r->size_2bpp);
            r->pixels_2bpp[offset_2bpp/8] |= 3 << (offset_2bpp&7);
            r->pixels[x+y*w]=0xFF;
          }
          else
          {
            r->pixels[x+y*w]=0x00;
          }
        }
#endif

    }
  }

  assert(num_full_frame==2);

  printf("num_x_scaled=%d\n",num_x_scaled);
  printf("num_y_scaled=%d\n",num_y_scaled);
  printf("num_full_frame=%d\n",num_full_frame);
  printf("num_pixels=%u\n",num_pixels);
  printf("num_bytes=%u\n",num_bytes);

  // Pack the ramp images into an atlas
  {
    int packing_y[2] = { 256+221, 0 };
    int packing_x[2] = { 0, 256 };

    uint32_t num_unpacked=num_ramps;
    uint32_t num_unpacked_2=0;
    const int thin_w=33;

    int num_between=0;
    int between_index=-1;

    for(int i=0;i<num_ramps;++i)
      if(ramps[i].h==43)
      {
        ++num_between;
        between_index=i;
      }

    assert(num_between>0);
    assert(between_index>=0);

    printf("num_between=%d\n",num_between);

    {
      ramps[between_index].atlas_x=0;
      ramps[between_index].atlas_y=212;
      ramps[between_index].packed=TRUE;
      --num_unpacked;
    }

    for(int i=0;i<num_ramps;++i)
      if(ramps[i].w==VIRTUAL_FRAME_WIDTH && ramps[i].h<thin_w)
        num_unpacked_2++;

    for(int i=0;i<num_ramps;++i)
    {
      if(!ramps[i].packed)
      {
        if(ramps[i].w==VIRTUAL_FRAME_WIDTH && ramps[i].h>=thin_w)
        {
          if(4096-packing_y[0]>=ramps[i].h)
          {
            ramps[i].atlas_x=packing_x[0];
            ramps[i].atlas_y=packing_y[0];
            ramps[i].packed=TRUE;
            --num_unpacked;
            packing_y[0]+=ramps[i].h;
          }
          else if(4096-packing_y[1]>=ramps[i].h)
          {
            ramps[i].atlas_x=packing_x[1];
            ramps[i].atlas_y=packing_y[1];
            ramps[i].packed=TRUE;
            --num_unpacked;
            packing_y[1]+=ramps[i].h;
          }
          else
            assert(false);
        }
      }
    }

    while(num_unpacked>num_unpacked_2)
    {
      int j=-1;
      int m=0;
      int side=-1;
      for(int i=0;i<num_ramps;++i)
      {
        if(!ramps[i].packed && ramps[i].h>=thin_w)
        {
          if(packing_x[0]+ramps[i].w<=256 &&
             packing_y[0]+ramps[i].h<=4096 && ramps[i].w>m)
          {
            //assert(VIRTUAL_FRAME_HEIGHT/2==ramps[i].h);
            m=ramps[i].w;
            j=i;
            side=0;
          }
          if(packing_x[1]+ramps[i].w<=512 &&
             packing_y[1]+ramps[i].h<=4096 && ramps[i].w>m)
          {
            //assert(VIRTUAL_FRAME_HEIGHT/2==ramps[i].h);
            m=ramps[i].w;
            j=i;
            side=1;
          }
        }
      }
      if(j>=0)
      {
        if(side==0)
        {
          ramps[j].atlas_x=packing_x[0];
          ramps[j].atlas_y=packing_y[0];
          ramps[j].packed=TRUE;
          --num_unpacked;
          packing_x[0]+=ramps[j].w;
        }
        else
        {
          ramps[j].atlas_x=packing_x[1];
          ramps[j].atlas_y=packing_y[1];
          ramps[j].packed=TRUE;
          --num_unpacked;
          packing_x[1]+=ramps[j].w;
        }
      }
      else
      {
        packing_x[0]=0;
        packing_y[0]+=VIRTUAL_FRAME_HEIGHT/2;
        packing_x[1]=256;
        packing_y[1]+=VIRTUAL_FRAME_HEIGHT/2;
      }
    }

    if(packing_x[0]!=0)
    {
      packing_x[0]=0;
      packing_y[0]+=VIRTUAL_FRAME_HEIGHT/2;
    }

    if(packing_x[1]!=256)
    {
      packing_x[1]=256;
      packing_y[1]+=VIRTUAL_FRAME_HEIGHT/2;
    }

    for(int i=0;i<num_ramps;++i)
    {
      if(!ramps[i].packed)
      {
        if(ramps[i].w==VIRTUAL_FRAME_WIDTH && ramps[i].h<thin_w)
        {
          if(4096-packing_y[0]>=ramps[i].h)
          {
            ramps[i].atlas_x=packing_x[0];
            ramps[i].atlas_y=packing_y[0];
            ramps[i].packed=TRUE;
            --num_unpacked;
            packing_y[0]+=ramps[i].h;
          }
          else if(4096-packing_y[1]>=ramps[i].h)
          {
            ramps[i].atlas_x=packing_x[1];
            ramps[i].atlas_y=packing_y[1];
            ramps[i].packed=TRUE;
            --num_unpacked;
            packing_y[1]+=ramps[i].h;
          }
          else
            assert(false);
        }
      }
    }

    uint16_t atlas_w=0,atlas_h=0;
    for(int i=0;i<num_ramps;++i)
    {
      assert(ramps[i].packed);
      atlas_w=std::max(atlas_w,uint16_t(ramps[i].atlas_x+ramps[i].w));
      atlas_h=std::max(atlas_h,uint16_t(ramps[i].atlas_y+ramps[i].h));
    }

    printf("atlas_size=%dx%d=%dB\n",(int)atlas_w,(int)atlas_h,((int)atlas_w*(int)atlas_h*2+7)/8);
  }

  printf("num_ramps=%d\n",num_ramps);

  for(int i=0;i<num_ramps;++i)
    ramp_remap[i]=i;

  qsort(&ramp_remap[0], num_ramps, sizeof(ramp_remap[0]), &ramp_comp);

  for(int i=0;i<num_x_scaled;++i)
    assert(ramps[ramp_remap[i]].scaling_axis==0);
  for(int i=0;i<num_y_scaled;++i)
    assert(ramps[ramp_remap[num_x_scaled+i]].scaling_axis==1);

  assert(num_x_scaled+num_y_scaled==num_ramps);

  for(int i=0;i<num_ramps;++i)
    ramp_unremap[ramp_remap[i]]=i;

  assert(num_x_scaled_0+num_x_scaled_1+num_y_scaled_0+num_y_scaled_1==num_ramps);
  assert(num_x_scaled_0+num_x_scaled_1==num_x_scaled);
  assert(num_y_scaled_0+num_y_scaled_1==num_y_scaled);

  for(int i=0;i<num_x_scaled_0;++i)
    assert(ramps[ramp_remap[i]].scaling_axis==0 && ramps[ramp_remap[i]].direction==0);
  for(int i=0;i<num_x_scaled_1;++i)
  {
    int j=i+num_x_scaled_0;
    assert(ramps[ramp_remap[j]].scaling_axis==0 && ramps[ramp_remap[j]].direction==1);
  }
  for(int i=0;i<num_y_scaled_0-1;++i)
  {
    int j=i+num_x_scaled_0+num_x_scaled_1;
    assert(ramps[ramp_remap[j]].scaling_axis==1);
    assert(ramps[ramp_remap[j]].direction==0);
  }
  for(int i=0;i<num_y_scaled_1-1;++i)
  {
    int j=i+num_x_scaled_0+num_x_scaled_1+num_y_scaled_0-1;
    assert(ramps[ramp_remap[j]].scaling_axis==1 && ramps[ramp_remap[j]].direction==1);
  }

  for(int i=0;i<num_ramps;++i)
  {
    // DIY=1
    ramps[i].vram_atlas_y=ramps[i].atlas_y+ramps[i].gen_h-1;
    // DIX=1, DIX=0
    if(ramps[i].direction==0)
      ramps[i].vram_atlas_x=ramps[i].atlas_x+ramps[i].gen_w-1;
    else
      ramps[i].vram_atlas_x=ramps[i].atlas_x;
  }

  {
    FILE* out=fopen("ramps.asm","w");
    fprintf(out,"NUM_RAMPS:               equ %d\n",num_ramps);
    fprintf(out,"NUM_SCALED_RAMPS:        equ %d\n",num_ramps-num_full_frame);
    fprintf(out,"NUM_X_SCALED_RAMPS_0:    equ %d\n",num_x_scaled_0);
    fprintf(out,"NUM_X_SCALED_RAMPS_1:    equ %d\n",num_x_scaled_1);
    fprintf(out,"NUM_Y_SCALED_RAMPS_0:    equ %d\n",num_y_scaled_0-1);
    fprintf(out,"NUM_Y_SCALED_RAMPS_1:    equ %d\n",num_y_scaled_1-1);
    fprintf(out,"UNSCALED_RAMP_0_INDEX:   equ %d\n",ramp_unremap[template_ramp_0]);
    fprintf(out,"UNSCALED_RAMP_1_INDEX:   equ %d\n",ramp_unremap[template_ramp_1]);
    fprintf(out,"UNSCALED_RAMP_0_ATLAS_X: equ %d\n",ramps[template_ramp_0].atlas_x);
    fprintf(out,"UNSCALED_RAMP_0_ATLAS_Y: equ %d\n",ramps[template_ramp_0].atlas_y);
    fprintf(out,"UNSCALED_RAMP_1_ATLAS_X: equ %d\n",ramps[template_ramp_1].atlas_x);
    fprintf(out,"UNSCALED_RAMP_1_ATLAS_Y: equ %d\n",ramps[template_ramp_1].atlas_y);
    fprintf(out,"\n\n");
    for(int i=0;i<num_ramps;++i)
    {
      const Ramp *const r=&ramps[ramp_remap[i]];

/*
      fprintf(out,"  MAKE_RAMP %5d, %5d, %5d, %5d, #%04X, #%04X, #%04X, #%04X ; %3d, %d, %d\n",
                        r->atlas_x, r->atlas_y, r->gen_w, r->gen_h,
                        r->gen_u, r->gen_dudx, r->gen_dudy, r->gen_dSdD,
          ramp_remap[i],r->scaling_axis,r->direction);
*/

      if(r->is_template)
      {
        fprintf(out,"  MAKE_RAMP %5d, %5d, %5d, %5d, #%04X, #%04X, #%04X; %3d, %d, %d\n",
                          r->atlas_x, r->atlas_y, r->gen_w, r->gen_h,
                          r->gen_u, r->gen_dudx, r->gen_dudy,
            ramp_remap[i],r->scaling_axis,r->direction);
      }
      else
      {
        fprintf(out,"  MAKE_RAMP %5d, %5d, %5d, %5d, #%04X, #%04X, #%04X; %3d, %d, %d\n",
                          r->atlas_x, r->atlas_y, r->gen_w, r->gen_h,
                          0, r->gen_dSdD_highprec&0xFFFF, r->gen_dSdD_highprec>>16,
            ramp_remap[i],r->scaling_axis,r->direction);
      }
    }
    fclose(out);
  }

  {
    const int w=512,h=4096;
    unsigned char* pixels=(unsigned char*)malloc(w*h*3);
    memset(pixels,0,w*h*3);
    FILE* out=fopen("atlas.rgba","wb");

    for(int i=0;i<num_ramps;++i)
    {
      const Ramp *const r=&ramps[ramp_remap[i]];
      for(int y=r->atlas_y;y<r->atlas_y+r->h;++y)
        for(int x=r->atlas_x;x<r->atlas_x+r->w;++x)
        {
          uint8_t v=r->pixels[(x-r->atlas_x)+(y-r->atlas_y)*r->w];

          if(x==r->atlas_x || y==r->atlas_y || x==(r->atlas_x+r->w-1) || y==(r->atlas_y+r->h-1))
          {
            pixels[(x+y*w)*3+0]=255;
            pixels[(x+y*w)*3+1]=v;
            pixels[(x+y*w)*3+2]=100;
          }
          else
          {
            pixels[(x+y*w)*3+0]=255/2;
            pixels[(x+y*w)*3+1]=v;
            pixels[(x+y*w)*3+2]=100/2;
          }
        }
    }

    writeRGBAFile(out, w, h, pixels);
    fclose(out);
    free(pixels);
  }
}

static const uint32_t vram_size=512*1024;
static const uint32_t ramp_vram_size=512*1024-16384*4;
static const uint32_t max_frame_commands=32;

// Byte Move to XY from Linear
static const uint8_t Command_BMXL = 0b10000000;
// Logical Move to Memory from VDP
static const uint8_t Command_LMMV = 0b00100000;
// Logical Move to Memory from Memory
static const uint8_t Command_LMMM = 0b01000000;

struct Command
{
  uint8_t SX_L;    // R#32
  uint8_t SX_H;    // R#33
  uint8_t SY_L;    // R#34
  uint8_t SY_H;    // R#35
  uint8_t DX_L;    // R#36
  uint8_t DX_H;    // R#37
  uint8_t DY_L;    // R#38
  uint8_t DY_H;    // R#39
  uint8_t NX_L;    // R#40
  uint8_t NX_H;    // R#41
  uint8_t NY_L;    // R#42
  uint8_t NY_H;    // R#43
  uint8_t ARG;     // R#44
  uint8_t LOP;     // R#45
  uint8_t WM_L;    // R#46
  uint8_t WM_H;    // R#47
  uint8_t FC_L;    // R#48
  uint8_t FC_H;    // R#49
  uint8_t BC_L;    // R#50
  uint8_t BC_H;    // R#51
  uint8_t OP_CODE; // R#52
};

struct CapturedFrame
{
  int num_commands;
  Command commands[max_frame_commands];
  uint32_t base_vram_address;
  uint32_t vram_data_size;
  uint8_t vram_data[ramp_vram_size];
};

static CapturedFrame captured_frame;

static void addLMMV(uint32_t dx,uint32_t dy,uint32_t nx,uint32_t ny,uint32_t arg,uint32_t lop)
{
  cycle_count += 21*18; // outi

  Command *c = &captured_frame.commands[captured_frame.num_commands];

  c->DX_L = (dx >> 0) & 0xFF;
  c->DX_H = (dx >> 8) & 0xFF;

  c->DY_L = (dy >> 0) & 0xFF;
  c->DY_H = (dy >> 8) & 0xFF;

  c->NX_L = (nx >> 0) & 0xFF;
  c->NX_H = (nx >> 8) & 0xFF;

  c->NY_L = (ny >> 0) & 0xFF;
  c->NY_H = (ny >> 8) & 0xFF;

  c->ARG = arg;
  c->LOP = lop | (1 << 4); // TP is always set

  c->WM_L = c->WM_H = 0xFF;

  c->FC_L = 0b01010101;
  c->FC_H = 0b01010101;

  c->OP_CODE = Command_LMMV;

  ++captured_frame.num_commands;
}

static void addBMXL(uint32_t vram_address,uint32_t dx,uint32_t dy,uint32_t nx,uint32_t ny,uint32_t arg,uint32_t lop)
{
  Command *c = &captured_frame.commands[captured_frame.num_commands];

  c->OP_CODE = Command_BMXL;

  ++captured_frame.num_commands;
}

static void addLMMM(uint32_t sx,uint32_t sy,uint32_t dx,uint32_t dy,uint32_t nx,uint32_t ny,uint32_t arg,uint32_t lop)
{
  cycle_count += 21*18; // outi

  Command *c = &captured_frame.commands[captured_frame.num_commands];

  c->SX_L = (sx >> 0) & 0xFF;
  c->SX_H = (sx >> 8) & 0xFF;

  c->SY_L = (sy >> 0) & 0xFF;
  c->SY_H = (sy >> 8) & 0xFF;

  c->DX_L = (dx >> 0) & 0xFF;
  c->DX_H = (dx >> 8) & 0xFF;

  c->DY_L = (dy >> 0) & 0xFF;
  c->DY_H = (dy >> 8) & 0xFF;

  c->NX_L = (nx >> 0) & 0xFF;
  c->NX_H = (nx >> 8) & 0xFF;

  c->NY_L = (ny >> 0) & 0xFF;
  c->NY_H = (ny >> 8) & 0xFF;

  c->ARG = arg;
  c->LOP = lop | (1 << 4); // TP is always set

  c->WM_L = c->WM_H = 0xFF;

  c->FC_L = 0b01010101;
  c->FC_H = 0b01010101;

  c->OP_CODE = Command_LMMM;

  ++captured_frame.num_commands;
}

static int16_t doWallConeIntersectionLookup(int16_t angle, int log_adj)
{
  ++num_intersect_ray_with_wall;
/*
  DE = angle
  HL = adj

  ; Total 220 cycles

  ; Multiply DE by 2
  add hl,hl           ; 12c
  ex de,hl            ;  5c
  ; Multiply HL by 2
  add hl,hl           ; 12c
  ; HL and DE are now swapped.
  ld bc,logtan_table  ; 11c
  add hl,bc           ; 12c
  ld c,(hl)           ;  8c
  inc hl              ;  7c
  ld b,(hl)           ;  8c
  ld h,b              ;  5c
  ld l,c              ;  5c
  ; log_tangent is now in HL
  ex de,hl            ;  5c
  ld bc,log_table     ; 11c
  add hl,bc           ; 12c
  ld c,(hl)           ;  8c
  inc hl              ;  7c
  ld b,(hl)           ;  8c
  ld h,b              ;  5c
  ld l,c              ;  5c
  ; log_tangent is now in DE
  ; log_adj is now in HL
  add hl,de           ; 12c
  ; log_product=max(0,log_tangent+log_adj)
  ld bc,exp_table     ; 11c
  add hl,bc           ; 12c
  ld e,(hl)           ;  8c
  inc hl              ;  7c
  ld d,(hl)           ;  8c
  ex de,hl            ;  5c
  ret                 ; 11c
  ; Result is in HL
*/

  //return tan(double(angle)*M_PI*2.0/double(NUM_ANGLE_UNITS))*double(adj);

  assert(labs(angle)<NUM_ANGLE_UNITS/4);

  assert(angle>-NUM_ANGLE_UNITS/4);
  assert(angle<+NUM_ANGLE_UNITS/4);

  cycle_count += 220;

  // This version uses log tables

  int log_tangent=logtan_table[labs(angle)]; // 16-bit

  assert(log_tangent>-2048);
  assert(log_tangent<+2048);

  int log_product=std::max(0,std::min(4095,log_tangent+log_adj)); // 16-bit

  assert(log_product>=0);
  assert(log_product<4096);
  int product=exp_table[log_product]; // 16-bit

  assert(product>=0);
  assert(product<4096);

  int16_t res=0;
  if(angle==0)
  {
    res = 0;
  }
  else if(angle>0)
  {
    res = +product;
  }
  else if(angle<0)
  {
    res = -product;
  }
  return res;
}

static int16_t calculateOppCoarseRD(int16_t angle, uint16_t adj)
{
  //return tan(double(angle)*2.0*M_PI/double(NUM_ANGLE_UNITS))*adj;

  assert(angle>-NUM_ANGLE_UNITS/4);
  assert(angle<+NUM_ANGLE_UNITS/4);
  assert(adj>0);
  assert(adj<MAX_COORDINATE);

  assert(labs(angle)<NUM_ANGLE_UNITS/4);
  assert(adj<4096);

  int coarse_angle=(int)std::floor(double(angle)/16.0);

  coarse_angle=std::max(-15,std::min(+15,coarse_angle));

  assert(coarse_angle>-16);
  assert(coarse_angle<+16);

  int product=std::floor(tan(double(coarse_angle)*2.0*M_PI/double(64))*double(adj)/256.0);

  //assert(product>=0);
  //assert(product<65536);

  return product*256;
}

static int16_t calculateOppCoarseRU(int16_t angle, uint16_t adj)
{
  //return tan(double(angle)*2.0*M_PI/double(NUM_ANGLE_UNITS))*adj;

  assert(angle>-NUM_ANGLE_UNITS/4);
  assert(angle<+NUM_ANGLE_UNITS/4);
  assert(adj>0);
  assert(adj<MAX_COORDINATE);

  assert(labs(angle)<NUM_ANGLE_UNITS/4);
  assert(adj<4096);

  int coarse_angle=(int)std::ceil(double(angle)/16.0);

  coarse_angle=std::max(-15,std::min(+15,coarse_angle));

  assert(coarse_angle>-16);
  assert(coarse_angle<+16);

  int product=std::ceil(tan(double(coarse_angle)*2.0*M_PI/double(64))*double(adj)/256.0);

  //assert(product>=0);
  //assert(product<65536);

  return product*256;
}

static const int max_lines_to_draw=2048;
static Line lines_to_draw[max_lines_to_draw];
static int num_lines_to_draw=0;

static char renderlog_string[1024*32];

static void projectPost(int16_t x,int16_t y,bool clipped,uint16_t clipped_angle,uint16_t camera_angle,uint8_t* fx,uint16_t* fy,uint16_t* portal_angle, int16_t log_opposite, int16_t log_adjacent, int16_t sopp, int16_t sadj, int orientation)
{
#if 0
  // Rotate
  uint16_t angle = clipped ? clipped_angle : getAngleForPoint(x, y);
  assert(angle >= 0);
  assert(angle < NUM_ANGLE_UNITS);
#else

  if((orientation&1)==1)
  {
    std::swap(sopp,sadj);
    std::swap(log_opposite,log_adjacent);
  }

  const int i=(log_adjacent-log_opposite+4096)/2;
  assert(i>=0);
  assert(i<4096);
  uint16_t angle=arctanlog_table[std::max(1024,std::min(3072,i))-1024];
  assert(angle>=0);
  assert(angle<NUM_ANGLE_UNITS/4);

  if(sopp<0)
  {
    if(sadj<0)
      angle=NUM_ANGLE_UNITS/2+angle; // Reflected twice <=> Rotated
    else
      angle=NUM_ANGLE_UNITS/2-angle;
  }
  else
  {
    if(sadj<0)
      angle=(NUM_ANGLE_UNITS-angle)&(NUM_ANGLE_UNITS-1);
  }

  if(clipped)
    angle=clipped_angle;

  assert(angle>=0);
  assert(angle<NUM_ANGLE_UNITS);

#endif

  *portal_angle=angle;
  angle = (angle - camera_angle) & (NUM_ANGLE_UNITS-1);

  uint16_t abs_angle = angle;
  if(angle >= NUM_ANGLE_UNITS / 2)
    abs_angle = NUM_ANGLE_UNITS - 1 - angle;

  //abs_angle=std::min(abs_angle,uint16_t(ANGLE_UNIT_FOV / 2 - 1));

  assert(abs_angle >= 0);
  assert(abs_angle <= ANGLE_UNIT_FOV / 2);

  // X projection
  *fx = x_angle_table[abs_angle];

  cycle_count += 11+12+12+8+7+8; // generic 16-bit table lookup

  if(angle >= NUM_ANGLE_UNITS / 2)
    *fx = VIRTUAL_FRAME_WIDTH - 1 - *fx;

  // Y projection

  assert(x!=0 || y!=0);
  assert(abs_angle <= NUM_ANGLE_UNITS/8);

#if 0
  double log_c=std::log(4096.0);
  double log_x=std::log((double)labs(x));
  double log_y=std::log((double)labs(y));

  double log_cos=std::log(std::cos(double(abs_angle)*2.0*M_PI/double(NUM_ANGLE_UNITS)));
  double log_sin=std::log(std::sin(double(abs_angle)*2.0*M_PI/double(NUM_ANGLE_UNITS)));
  double log_tan=std::log(std::tan(double(abs_angle)*2.0*M_PI/double(NUM_ANGLE_UNITS)));

  double log_f=(log_c-log_cos)-log_x-log_sum_g(log_y-log_x);
  //double log_f=(log_c+log_sin)-log_tan-log_x-log_sum_g(log_y-log_x);

  *fy = (uint16_t)std::exp(log_f);
#else

  static const int log_c=std::log(4096.0)/std::log(double(tangent_max))*2047.0;

  {
    static bool x=true;
    if(x)
      printf("log_c=%d\n",log_c);
    x=false;
  }

  const int log_cos=logsin_table[NUM_ANGLE_UNITS/4-abs_angle];

  assert(i>=0);
  assert(i<4096);
  const int log_distsq=log_opposite+log_partial_sum_table[i];

  //const int log_f=log_c*2-log_cos-log_distsq;

  const int log_f=log_c*2-log_cos-log_opposite-log_partial_sum_table[i]+log_bob;

  assert(log_f>=0);
  assert(log_f<+4096);

  *fy = exp_table[log_f]; // x2 for bobbing

#endif

/*
  uint16_t opp=std::labs(x);
  uint16_t adj=std::labs(y);
  int shift=0;

#if 1
  while(opp > 127 || adj > 127)
  {
    opp >>= 1;
    adj >>= 1;
    ++shift;
  }

  assert(opp!=0 || adj!=0);
  assert(abs_angle <= NUM_ANGLE_UNITS/8);

  if(opp>adj)
    std::swap(opp,adj);

  assert(opp<128);
  assert(adj<128);
  const uint8_t height_over_dist_index = polar_table[adj+opp*128];
  assert(height_over_dist_index<height_over_dist_table_size);
  const uint16_t height_over_dist = height_over_dist_table[height_over_dist_index];
#else
  //double distance=sqrt(opp*opp+adj*adj);
  assert(std::abs((double)opp)>0.0);
  assert(std::abs((double)adj)>0.0);
  double log_opp_sq=std::log(std::abs((double)opp))*2.0;
  double log_adj_sq=std::log(std::abs((double)adj))*2.0;
  double log_distance=(log_opp_sq+log_sum_f(log_adj_sq-log_opp_sq))/2.0;
  //double distance=sqrt(opp*opp+adj*adj);
  double distance=std::exp(log_distance);
  int64_t j=(4095.0/distance+0.5);
  assert(j>=0);
  const uint16_t height_over_dist=std::min(j,(int64_t)4096);
#endif

  assert(height_over_dist<4096);

  ++num_project_post;

  // This version relies on the observation that 2 > 1/cos(abs_angle) >= 1

  int one_over_cosine_minus_one  = one_over_cosine_minus_one_table[abs_angle];
  assert(one_over_cosine_minus_one>=0);
  assert(one_over_cosine_minus_one<16); // 4 bits

  assert(int(height_over_dist) * int(one_over_cosine_minus_one) < 65536);

  cycle_count += 11+12+12+8+7+8; // generic 16-bit table lookup
  cycle_count += 24*5; // multiplication

  // Multiplication of 4-bit number with 11-bit number can be done with a basic
  // shift-add loop, max. 4 iterations.

  int fy2 = height_over_dist + ((height_over_dist * one_over_cosine_minus_one + 16) >> 5);

  assert(fy2>=0);
  assert(fy2<=4096);

  *fy = fy2 >> shift;
*/
}

static void drawRoomThroughPortal(uint16_t room_index,uint16_t camera_x,uint16_t camera_y,uint16_t camera_angle, uint16_t a0,uint16_t a1,int portal_depth,uint8_t viewx0,uint8_t viewx1,uint16_t from_room_index, uint16_t portal_fy0, int16_t portal_fy1, int16_t portal_x0, int16_t portal_y0, int16_t portal_x1, int16_t portal_y1)
{
  assert(camera_angle<NUM_ANGLE_UNITS);

  assert(room_index>=0);
  assert(room_index<num_rooms);

  assert(camera_x<MAX_COORDINATE);
  assert(camera_y<MAX_COORDINATE);

  assert(a0>=0);
  assert(a1>=0);
  assert(a0<NUM_ANGLE_UNITS);
  assert(a1<NUM_ANGLE_UNITS);

  const uint8_t intersectionTestFlags0=ray_intersection_test_table[a0];
  const uint8_t intersectionTestFlags1=ray_intersection_test_table[a1];

  const uint8_t intersectionTestFlagsBoth=intersectionTestFlags0&intersectionTestFlags1;

  uint8_t prev_proj_x=-1;
  uint16_t prev_proj_y=-1;
  uint16_t prev_portal_angle=0xFFFF;

  bool has_clipped_a0=false;
  bool has_drawn_an_unculled_wall=false;

  int prev_ramp_orientation=-1;
  uint8_t ramp_index = 0xFF;

/*
  printf("  ; ********* Room Start *********\n");
  printf("  ld iy,portal_stack\n");

  printf("  ld hl,%d\n",(int)camera_angle);
  printf("  ld (iy+STACK_CAMERA_ANGLE+0),l\n");
  printf("  ld (iy+STACK_CAMERA_ANGLE+1),h\n");

  printf("  ld hl,%d\n",(int)a0);
  printf("  ld (iy+STACK_A0+0),l\n");
  printf("  ld (iy+STACK_A0+1),h\n");

  printf("  ld hl,%d\n",(int)a1);
  printf("  ld (iy+STACK_A1+0),l\n");
  printf("  ld (iy+STACK_A1+1),h\n");

  printf("  ld hl,%d\n",-(int)camera_x); // Negated
  printf("  ld (iy+STACK_CAMERA_X+0),l\n");
  printf("  ld (iy+STACK_CAMERA_X+1),h\n");

  printf("  ld hl,%d\n",-(int)camera_y); // Negated
  printf("  ld (iy+STACK_CAMERA_Y+0),l\n");
  printf("  ld (iy+STACK_CAMERA_Y+1),h\n");
*/

  for(int k=0;k<rooms[room_index].num_walls;++k)
  {
    const int sort_quadrant=(camera_angle*4)/NUM_ANGLE_UNITS;
    assert(sort_quadrant>=0);
    assert(sort_quadrant<4);
    const int i=wall_references[rooms[room_index].first_wall_ref[sort_quadrant]+k];
    const int j=wall_references[rooms[room_index].first_wall_ref[sort_quadrant]+((k+1)%rooms[room_index].num_walls)];
    const int n=wall_references[rooms[room_index].first_wall_ref[sort_quadrant]+((k+rooms[room_index].num_walls-1)%rooms[room_index].num_walls)];

    assert(walls[i].room==room_index);

    // Check connectivity
    assert(walls[i].x1 == walls[j].x0);
    assert(walls[i].y1 == walls[j].y0);
    assert(walls[n].x1 == walls[i].x0);
    assert(walls[n].y1 == walls[i].y0);

    int16_t x0=walls[i].x0-camera_x; // Range: [-MAX_COORDINATE+1, +MAX_COORDINATE-1]
    int16_t y0=walls[i].y0-camera_y; // Range: [-MAX_COORDINATE+1, +MAX_COORDINATE-1]
    int16_t x1=walls[i].x1-camera_x; // Range: [-MAX_COORDINATE+1, +MAX_COORDINATE-1]
    int16_t y1=walls[i].y1-camera_y; // Range: [-MAX_COORDINATE+1, +MAX_COORDINATE-1]

/*
    // Handle special case when the camera is sitting directly on top of
    // a portal.
    if(walls[i].portal_room>-1 && walls[i].portal_room!=from_room_index && std::min(x0,x1)<=0 && std::min(y0,y1)<=0 && std::max(x0,x1)>=0 && std::max(y0,y1)>=0)
    {
      ++portals_entered;

      // Recurse into portal
      drawRoomThroughPortal(walls[i].portal_room, camera_x, camera_y, camera_angle, a0, a1, portal_depth+1, viewx0, viewx1, room_index, portal_fy0, portal_fy1, portal_x0, portal_y0, portal_x1, portal_y1);

      prev_portal_angle=0xFFFF;
      continue;
    }
*/

    const uint8_t wall_colour=(walls[i].orientation&1)?3:2;
    static uint8_t shade_lut[4] = {
        0b00010011, 0b00010011, 0b00011100, 0b00011111
      };

/*
    if(walls[i].portal_room<0)
    {

      printf("  ; Wall %d\n",(int)walls_drawn);
      printf("  ld ix,wall_temp\n");

      printf("  ld hl,%d\n",(int)walls[i].x0);
      printf("  ld (ix+WALL_X0+0),l\n");
      printf("  ld (ix+WALL_X0+1),h\n");

      printf("  ld hl,%d\n",(int)walls[i].x1);
      printf("  ld (ix+WALL_X1+0),l\n");
      printf("  ld (ix+WALL_X1+1),h\n");

      printf("  ld hl,%d\n",(int)walls[i].y0);
      printf("  ld (ix+WALL_Y0+0),l\n");
      printf("  ld (ix+WALL_Y0+1),h\n");

      printf("  ld hl,%d\n",(int)walls[i].y1);
      printf("  ld (ix+WALL_Y1+0),l\n");
      printf("  ld (ix+WALL_Y1+1),h\n");

      switch(walls[i].orientation)
      {
        case 0: printf("  ld hl,draw_wall_orientation_0\n"); break;
        case 1: printf("  ld hl,draw_wall_orientation_1\n"); break;
        case 2: printf("  ld hl,draw_wall_orientation_2\n"); break;
        case 3: printf("  ld hl,draw_wall_orientation_3\n"); break;
      }
      printf("  ld (ix+WALL_ROUTINE+0),l\n");
      printf("  ld (ix+WALL_ROUTINE+1),h\n");

      if(walls[i].orientation&1)
      {
        printf("  ld (ix+WALL_SHADE_UPPER), SHADE_UPPER_WALL_B\n");
        printf("  ld (ix+WALL_SHADE_MIDDLE),SHADE_MIDDLE_WALL_B\n");
        printf("  ld (ix+WALL_SHADE_LOWER), SHADE_LOWER_WALL_B\n");
      }
      else
      {
        printf("  ld (ix+WALL_SHADE_UPPER), SHADE_UPPER_WALL_A\n");
        printf("  ld (ix+WALL_SHADE_MIDDLE),SHADE_MIDDLE_WALL_A\n");
        printf("  ld (ix+WALL_SHADE_LOWER), SHADE_LOWER_WALL_A\n");
      }

      printf("  ld (ix+WALL_SHADE),%d\n",shade_lut[wall_colour]);
      printf("  call draw_wall\n");
    }
*/
    // Check ranges
    assert(x0>-MAX_COORDINATE);
    assert(x0<+MAX_COORDINATE);
    assert(y0>-MAX_COORDINATE);
    assert(y0<+MAX_COORDINATE);
    assert(x1>-MAX_COORDINATE);
    assert(x1<+MAX_COORDINATE);
    assert(y1>-MAX_COORDINATE);
    assert(y1<+MAX_COORDINATE);

    assert(walls[i].orientation>=0 && walls[i].orientation<=3);

    switch(walls[i].orientation)
    {
      case 0: assert(x1>x0); assert(y0==y1); break;
      case 1: assert(y1>y0); assert(x0==x1); break;
      case 2: assert(x1<x0); assert(y0==y1); break;
      case 3: assert(y1<y0); assert(x0==x1); break;
    }

    // Backface culling
    if(walls[i].orientation==0 && y0>=0)
      continue;
    else if(walls[i].orientation==1 && x0<=0)
      continue;
    else if(walls[i].orientation==2 && y0<=0)
      continue;
    else if(walls[i].orientation==3 && x0>=0)
      continue;

    // Frustum culling
    if(intersectionTestFlagsBoth & (1 << walls[i].orientation))
      continue;

    bool clipped_a0=false,clipped_a1=false;
    int log_adjacent;
    uint16_t adj;
    int16_t sadj;

    switch(walls[i].orientation)
    {
      case 0: sadj=y0; break;
      case 1: sadj=x0; break;
      case 2: sadj=y0; break;
      case 3: sadj=x0; break;
      default: assert(false); break;
    }

    adj=labs(sadj);

    assert(adj>=0 && adj<MAX_COORDINATE);
    assert(adj>0);
    assert(adj<MAX_COORDINATE);
    assert(adj<4096);
    log_adjacent=log_table[adj];                // 16-bit
    assert(log_adjacent>=0);
    assert(log_adjacent<4096);

    switch(walls[i].orientation)
    {
      case 0:
        {
          if((!has_drawn_an_unculled_wall && !has_clipped_a0) && !(intersectionTestFlags0 & (1<<0)))
          {
            const int16_t local_angle=+(a0-(NUM_ANGLE_UNITS*3)/4);

            if(x0!=portal_x0 || y0!=portal_y0)
            {
              const int16_t opp=doWallConeIntersectionLookup(local_angle, log_adjacent);
              if(opp>=x0&&opp<x1)
              {
                clipped_a0=true;
                has_clipped_a0=true;
                x0=opp;
              }
              else if(opp>=x1)
                continue;
            }
          }
          if(!(intersectionTestFlags1 & (1<<0)))
          {
            const int16_t local_angle=+(a1-(NUM_ANGLE_UNITS*3)/4);

            if(x1!=portal_x1 || y1!=portal_y1)
            {
              const int16_t opp=doWallConeIntersectionLookup(local_angle, log_adjacent);
              if(opp>x0&&opp<=x1)
              {
                clipped_a1=true;
                x1=opp;
              }
              else if(opp<=x0)
                continue;
            }
          }
        }
        break;

      case 1:
        {
          if((!has_drawn_an_unculled_wall && !has_clipped_a0) && !(intersectionTestFlags0 & (1<<1)))
          {
            int ta0=a0;
            if(ta0>NUM_ANGLE_UNITS/2)
              ta0-=NUM_ANGLE_UNITS;

            if(x0!=portal_x0 || y0!=portal_y0)
            {
              const int16_t opp=doWallConeIntersectionLookup(+ta0, log_adjacent);
              if(opp>=y0&&opp<y1)
              {
                clipped_a0=true;
                has_clipped_a0=true;
                y0=opp;
              }
              else if(opp>=y1)
                continue;
            }
          }

          if(!(intersectionTestFlags1 & (1<<1)))
          {
            int ta1=a1;
            if(ta1>NUM_ANGLE_UNITS/2)
              ta1-=NUM_ANGLE_UNITS;

            if(x1!=portal_x1 || y1!=portal_y1)
            {
              const int16_t opp=doWallConeIntersectionLookup(+ta1, log_adjacent);
              if(opp>y0&&opp<=y1)
              {
                clipped_a1=true;
                y1=opp;
              }
              else if(opp<=y0)
                continue;
            }
          }
        }
        break;

      case 2:
        {
          if((!has_drawn_an_unculled_wall && !has_clipped_a0) && !(intersectionTestFlags0 & (1<<2)))
          {
            const int16_t local_angle=(NUM_ANGLE_UNITS*1)/4-a0;

            if(x0!=portal_x0 || y0!=portal_y0)
            {
              const int16_t opp=doWallConeIntersectionLookup(local_angle, log_adjacent);
              if(opp>x1&&opp<=x0)
              {
                clipped_a0=true;
                has_clipped_a0=true;
                x0=opp;
              }
              else if(opp<=x1)
                continue;
            }
          }
          if(!(intersectionTestFlags1 & (1<<2)))
          {
            const int16_t local_angle=(NUM_ANGLE_UNITS*1)/4-a1;

            if(x1!=portal_x1 || y1!=portal_y1)
            {
              const int16_t opp=doWallConeIntersectionLookup(local_angle, log_adjacent);
              if(opp>=x1&&opp<x0)
              {
                clipped_a1=true;
                x1=opp;
              }
              else if(opp>=x0)
                continue;
            }
          }
        }
        break;

      case 3:
        {
          if((!has_drawn_an_unculled_wall && !has_clipped_a0) && !(intersectionTestFlags0 & (1<<3)))
          {
            const int16_t local_angle=-(a0-(NUM_ANGLE_UNITS*1)/2);

            if(x0!=portal_x0 || y0!=portal_y0)
            {
              const int16_t opp=doWallConeIntersectionLookup(local_angle, log_adjacent);
              if(opp>y1&&opp<=y0)
              {
                clipped_a0=true;
                has_clipped_a0=true;
                y0=opp;
              }
              else if(opp<=y1)
                continue;
            }
          }

          if(!(intersectionTestFlags1 & (1<<3)))
          {
            const int16_t local_angle=-(a1-(NUM_ANGLE_UNITS*1)/2);

            if(x1!=portal_x1 || y1!=portal_y1)
            {
              const int16_t opp=doWallConeIntersectionLookup(local_angle, log_adjacent);
              if(opp>=y1&&opp<y0)
              {
                clipped_a1=true;
                y1=opp;
              }
              else if(opp>=y0)
                continue;
            }
          }
        }
        break;
    }

    if(x0==x1 && y0==y1)
      continue;

    uint16_t portal_angle0,portal_angle1;
    uint8_t fx0,fx1;
    uint16_t fy0,fy1;

    if(walls[i].portal_room<0)
    {
/*
      printf("  ; Wall %d\n",(int)walls_drawn);
      printf("  ld iy,portal_stack\n");

      printf("  ld hl,%d\n",(int)log_adjacent);
      printf("  ld (iy+STACK_LOG_ADJACENT+0),l\n");
      printf("  ld (iy+STACK_LOG_ADJACENT+1),h\n");
      printf("  ld hl,%d\n",(int)sadj);
      printf("  ld (iy+STACK_ADJACENT+0),l\n");
      printf("  ld (iy+STACK_ADJACENT+1),h\n");
      printf("  ld hl,%d\n",(int)camera_angle);
      printf("  ld (iy+STACK_CAMERA_ANGLE+0),l\n");
      printf("  ld (iy+STACK_CAMERA_ANGLE+1),h\n");

      printf("  ld (iy+STACK_ORIENTATION),%d\n",(int)walls[i].orientation);
*/
    }

#if 0
    if(x0==portal_x0 && y0==portal_y0)
    {
      fx0 = viewx0;
      fy0 = portal_fy0;
      portal_angle0 = a0;
      ++num_reused_points;
    }
    else if(clipped_a0 || prev_portal_angle==0xFFFF)
#endif
    {
        int log_opposite;
        uint16_t opp;
        int16_t sopp;

        switch(walls[i].orientation)
        {
          case 0: sopp=x0; break;
          case 1: sopp=y0; break;
          case 2: sopp=x0; break;
          case 3: sopp=y0; break;
          default: assert(false); break;
        }

        opp=labs(sopp);

        assert(opp>=0 && opp<MAX_COORDINATE);
        assert(opp<MAX_COORDINATE);
        assert(opp<4096);
        log_opposite=log_table[opp];                // 16-bit
        assert(log_opposite>=0);
        assert(log_opposite<4096);

        projectPost(x0, y0, clipped_a0, a0, camera_angle, &fx0, &fy0, &portal_angle0, log_opposite, log_adjacent, sopp, sadj, walls[i].orientation);

        if(walls[i].portal_room<0)
        {
/*
          printf("  ld hl,%d\n",(int)log_opposite);
          printf("  ld (iy+STACK_LOG_OPPOSITE+0),l\n");
          printf("  ld (iy+STACK_LOG_OPPOSITE+1),h\n");
          printf("  ld hl,%d\n",(int)sopp);
          printf("  ld (iy+STACK_OPPOSITE+0),l\n");
          printf("  ld (iy+STACK_OPPOSITE+1),h\n");
          printf("  call project_post\n");
          printf("  ld a,(iy+STACK_FX1)\n");
          printf("  ld (iy+STACK_FX0),a\n");
          printf("  ld a,(iy+STACK_FY1)\n");
          printf("  ld (iy+STACK_FY0),a\n");
*/
        }
    }
#if 0
    else
    {
      assert(prev_portal_angle>=0);

      fx0=prev_proj_x;
      fy0=prev_proj_y;
      portal_angle0=prev_portal_angle;
    }
#endif

#if 0
    if(x1==portal_x1 && y0==portal_y1)
    {
      fx1 = viewx1;
      fy1 = portal_fy1;
      portal_angle1 = a1;
      ++num_reused_points;
    }
    else
#endif
    {
      int log_opposite;
      uint16_t opp;
      int16_t sopp;

      switch(walls[i].orientation)
      {
        case 0: sopp=x1; break;
        case 1: sopp=y1; break;
        case 2: sopp=x1; break;
        case 3: sopp=y1; break;
        default: assert(false); break;
      }

      opp=labs(sopp);

      assert(opp>=0 && opp<MAX_COORDINATE);
      assert(opp<MAX_COORDINATE);
      assert(opp<4096);
      log_opposite=log_table[opp];                // 16-bit
      assert(log_opposite>=0);
      assert(log_opposite<4096);

      projectPost(x1, y1, clipped_a1, a1, camera_angle, &fx1, &fy1, &portal_angle1, log_opposite, log_adjacent, sopp, sadj, walls[i].orientation);

      if(walls[i].portal_room<0)
      {
/*
        printf("  ld hl,%d\n",(int)log_opposite);
        printf("  ld (iy+STACK_LOG_OPPOSITE+0),l\n");
        printf("  ld (iy+STACK_LOG_OPPOSITE+1),h\n");
        printf("  ld hl,%d\n",(int)sopp);
        printf("  ld (iy+STACK_OPPOSITE+0),l\n");
        printf("  ld (iy+STACK_OPPOSITE+1),h\n");
        printf("  call project_post\n");
*/
      }
    }

    fx0=std::min(fx0,viewx0);
    fx1=std::max(fx1,viewx1);

    if(fx0<=fx1)
      continue;

    has_drawn_an_unculled_wall=true;

    prev_proj_x=fx1;
    prev_proj_y=fy1;
    prev_portal_angle=portal_angle1;

    assert(fx0<=viewx0);
    assert(fx1>=viewx1);


    if(walls[i].portal_room>-1)
    {
      ++portals_entered;

      assert(portal_angle0!=0xFFFF);
      assert(portal_angle1!=0xFFFF);

      // Recurse into portal
      drawRoomThroughPortal(walls[i].portal_room, camera_x, camera_y, camera_angle, portal_angle0, portal_angle1, portal_depth+1, fx0, fx1, room_index, fy0, fy1, x0, y0, x1, y1);
    }
    else
    {
      // Draw

      ++walls_drawn;

      assert(fx0>=0);
      assert(fx0<VIRTUAL_FRAME_WIDTH);
      assert(fx1>=0);
      assert(fx1<VIRTUAL_FRAME_WIDTH);

/*
      assert(fy0>=0);
      assert(fy0<VIRTUAL_FRAME_HEIGHT);
      assert(fy1>=0);
      assert(fy1<VIRTUAL_FRAME_HEIGHT);
*/

      int relative_angle;

      switch(walls[i].orientation)
      {
        case 0:
          relative_angle=camera_angle-(NUM_ANGLE_UNITS*3)/4;
          break;
        case 1:
          relative_angle=camera_angle;
          break;
        case 2:
          relative_angle=camera_angle-NUM_ANGLE_UNITS/4;
          break;
        case 3:
          relative_angle=camera_angle-NUM_ANGLE_UNITS/2;
          break;
      }

      if(walls[i].orientation!=prev_ramp_orientation)
      {
        const int log_sin=logsin_table[relative_angle&(NUM_ANGLE_UNITS/2-1)];
        const int log_quotient=log_sin-log_adjacent+log_bob;

        assert(log_quotient>=-4096);
        assert(log_quotient<4096);

        ramp_index=rampindex_table[std::max(0,log_quotient)>>1]; // x2 for bobbing
        ramp_index=std::min(ramp_index,uint8_t(num_ramp_angles-1));

        assert(ramp_index >= 0);
        assert(ramp_index < num_ramp_angles);

        prev_ramp_orientation=walls[i].orientation;
      }

      // Set the column heights
      {
        static bool_t init=TRUE;

        //for(int x=fx1/2;x<fx0/2;++x)
        //{
          //column_heights[x]=(fy1+((fy0-fy1)*(x-fx1/2))/(fx0/2-fx1/2));
        //}

        if((relative_angle&(NUM_ANGLE_UNITS-1))<=NUM_ANGLE_UNITS/2)
        {
          const uint16_t dydx=ramp_slopes[ramp_index];
          uint16_t y=fy1*128;
          uint8_t x0=(fx1+1)/2;
          uint8_t x1=(fx0+1)/2;
          for(int x=x0;x<x1;++x)
          {
            assert(y/256<256);
            column_heights[x]=MIN(y/256,48);
            y+=dydx;
          }
          column_heights[x0]=0;
        }
        else
        {
          const uint16_t dydx=ramp_slopes[ramp_index];
          uint16_t y=fy1*128;
          uint8_t x0=(fx1+1)/2;
          uint8_t x1=(fx0+1)/2;
          for(int x=x0;x<x1;++x)
          {
            assert(y/256<256);
            column_heights[x]=MIN(y/256,48);
            y-=dydx;
          }
          column_heights[x0]=0;
/*
          const uint16_t dydx=ramp_slopes[ramp_index];
          uint16_t y=fy0*128;
          uint8_t x0=(fx1+1)/2;
          uint8_t x1=(fx0+1)/2;
          for(int i=0;i<x1-x0;++i)
          {
            assert(y/256<256);
            column_heights[x0+(x1-x0)-1-i]=MIN(y/256,48);
            y+=dydx;
            if(y>65535)
              y=65535;
          }
          column_heights[x0]=0;
*/
        }
      }

/*
      printf("  ld (iy+STACK_SHADE),%d\n",(int)shade_lut[wall_colour]);
      printf("  ld hl,%d\n",(int)relative_angle);
      printf("  ld (iy+STACK_RELATIVE_ANGLE+0),l\n");
      printf("  ld (iy+STACK_RELATIVE_ANGLE+1),h\n");
      printf("  call generate_wall_commands\n");
*/

      //if(fy1<fy0)
      if((relative_angle&(NUM_ANGLE_UNITS-1))<=NUM_ANGLE_UNITS/2)
      {
/*
        printf("  ld (iy+STACK_FX0),%d\n",fx0);
        printf("  ld (iy+STACK_FY0),%d\n",fy0);
        printf("  ld (iy+STACK_FX1),%d\n",fx1);
        printf("  ld (iy+STACK_FY1),%d\n",fy1);
*/

/*

        //printf("ld hl,ramp_descriptors+%d*RAMP_DESCRIPTOR_SIZE\n",ramp_unremap[ramp_index]);
        //printf("ld (iy+STACK_RAMP+0),l\n");
        //printf("ld (iy+STACK_RAMP+1),h\n");
        printf("  ld (iy+STACK_SHADE),%d\n",(int)shade_lut[wall_colour]);
        printf("  ld hl,%d\n",(int)relative_angle);
        printf("  ld (iy+STACK_RELATIVE_ANGLE+0),l\n");
        printf("  ld (iy+STACK_RELATIVE_ANGLE+1),h\n");
        //printf("ld hl,%d\n",log_adjacent);
        //printf("ld (iy+STACK_LOG_ADJACENT+0),l\n");
        //printf("ld (iy+STACK_LOG_ADJACENT+1),h\n");

        //printf("  call calculate_ramp\n");
        //printf("  call generate_wall_commands_L2R\n");

        printf("  call generate_wall_commands\n");
*/

        const Ramp *const r=&ramps[ramp_index];
        assert(r->pixels);

        {
          // Origin (dx,dy) is at the bottom-left corner
          // Note that this origin applies to the source rectangle too.

          // LMMM
          int dx=fx1, dy=std::max(0,VIRTUAL_FRAME_HEIGHT/2-fy1-1);

          assert(dx>=0);
          assert(dy>=0);
          assert(dx<256);
          assert(dy<256);

          int nx=std::min(int(fx0-fx1),r->w), ny=std::min(int(fy0-fy1),r->h);

/*
          assert(nx>=0);
          assert(ny>=0);
          assert(nx<256);
          assert(ny<256);
*/
          if(nx>0 && ny>0)
          {
            addLMMM(r->atlas_x, r->atlas_y+r->h-1, dx, dy, nx, ny,
                    0b1000, (walls[i].orientation&1) ? 0b0001 : 0b1001);

            for(uint16_t y=0;y<ny;++y)
            {
              uint16_t py=dy-y;
              if(py > 128)
                break;
              for(uint8_t x=0;x<nx;++x)
              {
                uint8_t px=dx+x;
                uint8_t sx=x;
                uint16_t sy=r->h-1-y;
                plotPixelVirtual(px, py, r->pixels[sx+sy*r->w] ? wall_colour : 0);
                plotPixelVirtual(px, VIRTUAL_FRAME_HEIGHT-1-py, r->pixels[sx+sy*r->w] ? wall_colour : 0);
              }
            }
          }
        }

        if(int(fx0-fx1)>r->w)
        {
          // Origin (dx,dy) is at the top-left corner
          // LMMV
          int nx=fx0-(fx1+r->w), ny=VIEWPORT_BOTTOM;
          int dx=fx1+r->w,dy=0;

          if(nx>0 && ny>0)
          {
            for(int y=0;y<ny;++y)
              for(int x=0;x<nx;++x)
              {
                plotPixelVirtual(dx+x, dy+y, wall_colour);
                plotPixelVirtual(dx+x, VIRTUAL_FRAME_HEIGHT-1-(dy+y), wall_colour);
              }
          }
        }

        {
          // Origin (dx,dy) is at the top-left corner

          // LMMV
          int dx=fx1,dy=VIRTUAL_FRAME_HEIGHT/2-fy1;

          int nx=int(fx0-fx1), ny=int(VIEWPORT_BOTTOM-dy);

          if(nx>0 && ny>0)
          {
            /*
                BG    = 01
                RAMP  = 01          L11, L10, L01, L00
                C0    = 10    LOP =   0,   X,   X,   1
                C1    = 11    LOP =   1,   X,   X,   1
                C2    = 00    LOP =   0,   X,   X,   0
            */

            addLMMV(dx, dy, nx, ny, 0b0000, (walls[i].orientation&1) ? 0b0001 : 0b1001);

            for(int y=0;y<ny;++y)
              for(int x=0;x<nx;++x)
              {
                plotPixelVirtual(dx+x, dy+y, wall_colour);
                plotPixelVirtual(dx+x, VIRTUAL_FRAME_HEIGHT-1-(dy+y), wall_colour);
              }
          }
        }
      }
      else
      {
        const Ramp *const r=&ramps[ramp_index+num_ramp_angles];
        assert(r->pixels);

        {
          // Origin (dx,dy) is at the bottom-right corner
          // Note that this origin applies to the source rectangle too.

          // LMMM
          int dx=fx0-1,dy=std::max(0,VIRTUAL_FRAME_HEIGHT/2-fy0-1);

          assert(dx>=0);
          assert(dy>=0);
          assert(dx<256);
          assert(dy<256);

          int nx=std::min(int(fx0-fx1),r->w), ny=std::min(int(fy1-fy0),r->h);

/*
          assert(nx>=0);
          assert(ny>=0);
          assert(nx<256);
          assert(ny<256);
*/

          if(nx>0 && ny>0)
          {
            addLMMM(r->atlas_x+r->w-1, r->atlas_y+r->h-1, dx, dy, nx, ny,
                    0b1100, (walls[i].orientation&1) ? 0b0001 : 0b1001);

            for(uint16_t y=0;y<ny;++y)
            {
              uint16_t py=dy-y;
              if(py > 128)
                break;
              for(uint8_t x=0;x<nx;++x)
              {
                uint8_t px=dx-x;
                uint8_t sx=r->w-1-x;
                uint16_t sy=r->h-1-y;
                plotPixelVirtual(px, py, r->pixels[sx+sy*r->w] ? wall_colour : 0);
                plotPixelVirtual(px, VIRTUAL_FRAME_HEIGHT-1-py, r->pixels[sx+sy*r->w] ? wall_colour : 0);
              }
            }
          }
        }

        if(int(fx0-fx1)>r->w)
        {
          // Origin (dx,dy) is at the top-left corner
          // LMMV
          int nx=(fx0-fx1-r->w), ny=VIEWPORT_BOTTOM;
          int dx=fx1,dy=0;

          if(nx>0 && ny>0)
          {
            for(int y=0;y<ny;++y)
              for(int x=0;x<nx;++x)
              {
                plotPixelVirtual(dx+x, dy+y, wall_colour);
                plotPixelVirtual(dx+x, VIRTUAL_FRAME_HEIGHT-1-(dy+y), wall_colour);
              }
          }
        }

        {
          // Origin (dx,dy) is at the top-right corner

          // LMMV
          int dx=fx0-1,dy=VIRTUAL_FRAME_HEIGHT/2-fy0;

          int nx=int(fx0-fx1), ny=int(VIEWPORT_BOTTOM-dy);

          if(nx>0 && ny>0)
          {
            /*
                BG    = 01
                RAMP  = 01          L11, L10, L01, L00
                C0    = 10    LOP =   0,   X,   X,   1
                C1    = 11    LOP =   1,   X,   X,   1
                C2    = 00    LOP =   0,   X,   X,   0
            */

            addLMMV(dx, dy, nx, ny, 0b0100, (walls[i].orientation&1) ? 0b0001 : 0b1001);

            for(int y=0;y<ny;++y)
              for(int x=0;x<nx;++x)
              {
                plotPixelVirtual(dx-x, dy+y, wall_colour);
                plotPixelVirtual(dx-x, VIRTUAL_FRAME_HEIGHT-1-(dy+y), wall_colour);
              }
          }
        }
      }

    }

    int colour=RGB888toRGB565(255,255,255);

    switch(walls[i].orientation)
    {
      case 0: colour=RGB888toRGB565(255,100,100); break;
      case 1: colour=RGB888toRGB565(100,255,100); break;
      case 2: colour=RGB888toRGB565(255,100,255); break;
      case 3: colour=RGB888toRGB565(100,255,255); break;
    }

    static char str[1024];
    sprintf(str, "i=%d, portal_depth=%d\n"
                 "      x0=%d, y0=%d, x1=%d, y1=%d\n"
                 "      portal_x0=%d, portal_y0=%d, portal_x1=%d, portal_y1=%d\n"
        ,i,portal_depth
        ,x0,y0,x1,y1
        ,portal_x0,portal_y0,portal_x1,portal_y1
        );
    strcat(renderlog_string,str);

#if 1
    assert(num_lines_to_draw<max_lines_to_draw);
    lines_to_draw[num_lines_to_draw].x0 = fx0*4;
    lines_to_draw[num_lines_to_draw].y0 = FRAME_HEIGHT/2+fy0*4;
    lines_to_draw[num_lines_to_draw].x1 = fx1*4;
    lines_to_draw[num_lines_to_draw].y1 = FRAME_HEIGHT/2+fy1*4;
    lines_to_draw[num_lines_to_draw].colour = colour;
    ++num_lines_to_draw;

    assert(num_lines_to_draw<max_lines_to_draw);
    lines_to_draw[num_lines_to_draw].x0 = fx0*4;
    lines_to_draw[num_lines_to_draw].y0 = FRAME_HEIGHT/2+fy0*4;
    lines_to_draw[num_lines_to_draw].x1 = fx0*4;
    lines_to_draw[num_lines_to_draw].y1 = FRAME_HEIGHT/2-fy0*4;
    lines_to_draw[num_lines_to_draw].colour = colour;
    ++num_lines_to_draw;

    assert(num_lines_to_draw<max_lines_to_draw);
    lines_to_draw[num_lines_to_draw].x0 = fx0*4;
    lines_to_draw[num_lines_to_draw].y0 = FRAME_HEIGHT/2-fy0*4;
    lines_to_draw[num_lines_to_draw].x1 = fx1*4;
    lines_to_draw[num_lines_to_draw].y1 = FRAME_HEIGHT/2-fy1*4;
    lines_to_draw[num_lines_to_draw].colour = colour;
    ++num_lines_to_draw;

    assert(num_lines_to_draw<max_lines_to_draw);
    lines_to_draw[num_lines_to_draw].x0 = fx1*4;
    lines_to_draw[num_lines_to_draw].y0 = FRAME_HEIGHT/2+fy1*4;
    lines_to_draw[num_lines_to_draw].x1 = fx1*4;
    lines_to_draw[num_lines_to_draw].y1 = FRAME_HEIGHT/2-fy1*4;
    lines_to_draw[num_lines_to_draw].colour = colour;
    ++num_lines_to_draw;

    {
      static const int wall_height = MAX_COORDINATE;
      double adjacent,theta;

      switch(walls[i].orientation)
      {
        case 0:
          theta=double((camera_angle-(NUM_ANGLE_UNITS*3)/4))*M_PI*2.0/double(NUM_ANGLE_UNITS);
          adjacent=labs(y0);
          break;
        case 1:
          theta=double(camera_angle)*M_PI*2.0/double(NUM_ANGLE_UNITS);
          adjacent=labs(x0);
          break;
        case 2:
          theta=double((camera_angle-NUM_ANGLE_UNITS/4))*M_PI*2.0/double(NUM_ANGLE_UNITS);
          adjacent=labs(y0);
          break;
        case 3:
          theta=double((camera_angle-NUM_ANGLE_UNITS/2))*M_PI*2.0/double(NUM_ANGLE_UNITS);
          adjacent=labs(x0);
          break;
      }

      double slope=double(wall_height)*std::sin(theta)/adjacent/128.0;
      double ramp_angle=std::atan(slope);

      assert(num_lines_to_draw<max_lines_to_draw);
      lines_to_draw[num_lines_to_draw].x0 = fx1*4;
      lines_to_draw[num_lines_to_draw].y0 = FRAME_HEIGHT/2+fy1*4;
      lines_to_draw[num_lines_to_draw].x1 = lines_to_draw[num_lines_to_draw].x0+std::cos(ramp_angle)*128.0;
      lines_to_draw[num_lines_to_draw].y1 = lines_to_draw[num_lines_to_draw].y0+std::sin(ramp_angle)*128.0;
      lines_to_draw[num_lines_to_draw].colour = RGB888toRGB565(255,255,255);;
      ++num_lines_to_draw;
    }
#endif

    if(clipped_a1)
      continue;

    // If the wall was clipped by both view edges then there can't be
    // any remaining visible walls in this room.
    if(clipped_a0&&clipped_a1) break;
  }

/*/
  // Draw an orange if necessary
  if(rooms[room_index].contains_orange)
  {
    uint8_t fx;
    uint16_t fy;
    uint16_t portal_angle;
    projectPost(rooms[room_index].orange_x-camera_x, rooms[room_index].orange_y-camera_y, FALSE, 0, camera_angle, &fx, &fy, &portal_angle, log_opposite, log_adjacent);

    int ramp_index = num_ramp_angles/2;

    assert(ramp_index >= 0);
    assert(ramp_index < num_ramp_angles);

    // Right-hand side of orange
    {
      const Ramp *const r=&ramps[ramp_index+num_ramp_angles+1];
      assert(r->pixels);

      // Origin (dx,dy) is at the bottom-right corner
      // Note that this origin applies to the source rectangle too.

      // LMMM
      int nx=std::min(int(fy)/2,r->w);
      int sx=nx/2;
      int ny=std::min(int(fy)/2,r->h);
      int dx=fx+nx-sx,dy=VIRTUAL_FRAME_HEIGHT/2;

      assert(dx>=0);
      assert(dy>=0);
      assert(dx<256);
      assert(dy<256);

      assert(nx>=0);
      assert(ny>=0);
      assert(nx<256);
      assert(ny<256);

      if(nx>0 && ny>0)
      {
        addLMMM(r->atlas_x+r->w-1-sx, r->atlas_y+r->h-1, dx, dy, nx, ny,
                0b1100, 0b1001);

        for(uint16_t y=0;y<ny;++y)
        {
          uint16_t py=dy-y;
          if(py > 128)
            break;
          for(uint8_t x=0;x<nx;++x)
          {
            uint8_t px=dx-x;
            uint8_t tx=r->w-1-x-sx+2;
            uint16_t ty=r->h-1-y;
            plotPixelVirtual(px, py, r->pixels[tx+ty*r->w] ? 1 : 0);
            plotPixelVirtual(px, VIRTUAL_FRAME_HEIGHT-1-py, r->pixels[tx+ty*r->w] ? 1 : 0);
          }
        }
      }
    }

    // Left-hand side of orange
    {
      const Ramp *const r=&ramps[ramp_index];
      assert(r->pixels);

      // Origin (dx,dy) is at the bottom-left corner
      // Note that this origin applies to the source rectangle too.

      // LMMM
      int nx=std::min(int(fy)/2,r->w);
      int sx=nx/2;
      int ny=std::min(int(fy)/2,r->h);
      int dx=std::max(0,fx-nx-sx+1),dy=VIRTUAL_FRAME_HEIGHT/2;

      assert(dx>=0);
      assert(dy>=0);
      assert(dx<256);
      assert(dy<256);

      assert(nx>=0);
      assert(ny>=0);
      assert(nx<256);
      assert(ny<256);

      if(nx>0 && ny>0)
      {
        addLMMM(r->atlas_x+sx, r->atlas_y+r->h-1, dx, dy, nx, ny,
                0b1100, 0b1001);

        for(uint16_t y=0;y<ny;++y)
        {
          uint16_t py=dy-y;
          if(py > 128)
            break;
          for(uint8_t x=0;x<nx;++x)
          {
            uint8_t px=dx+x;
            uint8_t tx=x+sx;
            uint16_t ty=r->h-1-y;
            plotPixelVirtual(px, py, r->pixels[tx+ty*r->w] ? 1 : 0);
            plotPixelVirtual(px, VIRTUAL_FRAME_HEIGHT-1-py, r->pixels[tx+ty*r->w] ? 1 : 0);
          }
        }
      }
    }

  }
*/

}

static void renderMapFromCameraView(double camera_x, double camera_y, double camera_angle)
{
  const int room_index=cell_visited[int(camera_x)/UNITS_PER_CELL+int(camera_y)/UNITS_PER_CELL*32]-1;

  if(room_index<0)
    return;

  //printf("room_index=%d\n",room_index);

  walls_drawn=0;
  portals_entered=0;

  cycle_count=0;

  const uint16_t camera_angle_int=int(camera_angle/(M_PI*2.0)*double(NUM_ANGLE_UNITS))&(NUM_ANGLE_UNITS-1);

  memset(renderlog_string,0,sizeof(renderlog_string));

  ramp_index_cache_count=0;
  num_intersect_ray_with_wall=0;
  num_octant_angle=0;
  num_project_post=0;
  num_cheap_rays=0;
  num_reused_points=0;

  //double bob_phase=double(camera_x+camera_y)/16.0;
  //log_bob=std::log((0.8+0.4*(std::cos(bob_phase)+1.0)/2.0))/std::log(tangent_max)*2047.0+0.5;

  //log_bob=log_bob_table[(int(camera_x+camera_y))&255];
  log_bob=0;

  drawRoomThroughPortal(room_index, camera_x, camera_y, camera_angle_int, (camera_angle_int-ANGLE_UNIT_FOV/2)&(NUM_ANGLE_UNITS-1), (camera_angle_int+ANGLE_UNIT_FOV/2-1)&(NUM_ANGLE_UNITS-1), 0, 255, 0, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF);

}

void renderColumns(uint16_t* pixel_buffer, renderstate_t* rs, camera_t* camera, const float* map, const float* map2)
{

    static bool_t init = TRUE;

    if(init)
    {
        init = FALSE;

#if 1


        generateTiles();

/*
        {
          FILE* out=fopen("tiles.asm","w");
          for(int i=0;i<16;++i)
          {
            fprintf(out,"  ; Tile %2d\n",i);
            for(int y=0;y<4;++y)
            {
              fprintf(out, "  ; ");
              for(int x=0;x<4;++x)
              {
                fprintf(out, "%c",tile_collision_strings[tiles[i].collision_index][x+y*4]);
              }
              fprintf(out,"\n");
            }
            for(int y=0;y<4;++y)
            {
              fprintf(out, "  dw ");
              for(int x=0;x<4;++x)
              {
                if(tiles[i].cell_rooms[x+y*4]<0)
                  fprintf(out, "#FFFF");
                else
                  fprintf(out, "rooms+%d*ROOM_SIZE",tiles[i].cell_rooms[x+y*4]);

                if(x<3)
                  fprintf(out, ", ");
              }
              fprintf(out,"\n");
            }
          }
          fclose(out);
        }

        {
          FILE* out=fopen("tile_entrances.asm","w");
          for(int i=0;i<16;++i)
          {
            fprintf(out,"  ; Tile %2d\n",i);
            fprintf(out,"  dw ");
            fprintf(out, "rooms+%d*ROOM_SIZE, ",tiles[i].entrances[0]);
            fprintf(out, "rooms+%d*ROOM_SIZE, ",tiles[i].entrances[1]);
            fprintf(out, "rooms+%d*ROOM_SIZE, ",tiles[i].entrances[2]);
            fprintf(out, "rooms+%d*ROOM_SIZE",tiles[i].entrances[3]);
            fprintf(out,"\n");
          }
          fclose(out);
        }

        {
          FILE* out=fopen("walls.asm","w");
          for(int i=0;i<num_walls;++i)
          {
            char portal_addr[64];
            
            if(walls[i].portal_room>=0)
              sprintf(portal_addr,"rooms+%d*ROOM_SIZE",walls[i].portal_room);
            else if(walls[i].portal_room==-1)
              sprintf(portal_addr,"#FFFF");
            else if(walls[i].portal_room==-2)
              sprintf(portal_addr,"#FF00");
            else if(walls[i].portal_room==-3)
              sprintf(portal_addr,"#FF01");
            else if(walls[i].portal_room==-4)
              sprintf(portal_addr,"#FF02");
            else if(walls[i].portal_room==-5)
              sprintf(portal_addr,"#FF03");

            fprintf(out,"  MAKE_WALL %4d, %4d, %4d, %4d, %1d, %s\n",
                int(walls[i].x0),int(walls[i].y0),int(walls[i].x1),int(walls[i].y1),
                int(walls[i].orientation), portal_addr);
          }
          fclose(out);
        }

        {
          FILE* out=fopen("rooms.asm","w");
          for(int i=0;i<num_rooms;++i)
          {
            fprintf(out,"  MAKE_ROOM %4d, %4d, %4d, %4d, %4d\n",
                int(rooms[i].num_walls),
                int(rooms[i].first_wall_q[0]),
                int(rooms[i].first_wall_q[1]),
                int(rooms[i].first_wall_q[2]),
                int(rooms[i].first_wall_q[3])
                 );
          }
          fclose(out);
        }
*/

        generateMaze();

/*
        {
          srand(12345);
          FILE* out=fopen("maze.asm","w");
          for(int y=0;y<8;++y)
          {
            fprintf(out,"  db ");
            for(int x=0;x<8;++x)
            {
              int j=maze[x+y*8];
              fprintf(out,"#%02X",j);
              if(x<15)
                fprintf(out,", ");
            }
            fprintf(out,"\n");
          }
          fclose(out);
        }
*/


#endif

        generateRamps();
        generateAllTables();
        processFont();

        //exit(0);

        makeRooms(camera);
        //testAngleTable();

        {
          FILE* out=fopen("maze.asm","w");
          for(int y=0;y<16;++y)
          {
            fprintf(out,"  dw ");
            for(int x=0;x<16;++x)
            {
              const int room_index=cell_visited[x*2+y*2*32]-1;

              if(room_index<0)
                fprintf(out,"#FFFF");
              else
                fprintf(out,"rooms+%d*ROOM_SIZE",room_index);

              if(x<15)
                fprintf(out,", ");
            }
            fprintf(out,"\n");
          }
          fclose(out);
        }

        {
          FILE* out=fopen("walls.asm","w");
          for(int i=0;i<num_walls;++i)
          {
            char portal_addr[64];
            
            if(walls[i].portal_room>=0)
              sprintf(portal_addr,"rooms+%d*ROOM_SIZE",walls[i].portal_room);
            else if(walls[i].portal_room==-1)
              sprintf(portal_addr,"#FFFF");
            else
              sprintf(portal_addr,"xxxx");

            fprintf(out,"  MAKE_WALL %4d, %4d, %4d, %4d, %1d, %s\n",
                int(walls[i].x0),int(walls[i].y0),int(walls[i].x1),int(walls[i].y1),
                int(walls[i].orientation), portal_addr);
          }
          fclose(out);
        }

        {
          FILE* out=fopen("rooms.asm","w");
          for(int i=0;i<num_rooms;++i)
          {
            fprintf(out,"  MAKE_ROOM %4d, %4d, %4d, %4d, %4d\n",
                int(rooms[i].num_walls),
                int(rooms[i].first_wall_q[0]),
                int(rooms[i].first_wall_q[1]),
                int(rooms[i].first_wall_q[2]),
                int(rooms[i].first_wall_q[3])
                 );
          }
          fclose(out);
        }
    }

    const float brush_move = sinf(rs->time);

#if 0
  // Maze display
  {
    clipAndPlotBox(pixel_buffer, 0, 0, FRAME_WIDTH, FRAME_HEIGHT, RGB888toRGB565(50, 50, 50));
    for(int y=0;y<8;++y)
      for(int x=0;x<8;++x)
      {
        int i=maze[x+y*8]&15;
        assert(i>=0);
        assert(i<16);
/*
        if(maze[x+y*16]&32)
        {
          for(int v=0;v<4;++v)
            for(int u=0;u<4;++u)
            {
              if(tile_collision_strings[4][u+v*4]!=' ')
                clipAndPlotBox(pixel_buffer, (x*4+u)*8, (y*4+v)*8, 8, 8, RGB888toRGB565(250, 250, 250));
            }
        }
        else if(maze[x+y*16]&16)
        {
          for(int v=0;v<4;++v)
            for(int u=0;u<4;++u)
            {
              if(tile_collision_strings[5][u+v*4]!=' ')
                clipAndPlotBox(pixel_buffer, (x*4+u)*8, (y*4+v)*8, 8, 8, RGB888toRGB565(250, 250, 250));
            }
        }
        else*/
        {
          for(int v=0;v<4;++v)
            for(int u=0;u<4;++u)
            {
              if(tile_collision_strings[tiles[i].collision_index][u+v*4]=='X')
                clipAndPlotBox(pixel_buffer, (x*4+u)*8, (y*4+v)*8, 8, 8, RGB888toRGB565(250, 250, 250));
            }
        }
      }
    return;
  }
#endif

  num_lines_to_draw=0;

  memset(&captured_frame,0,sizeof(captured_frame));

  captured_frame.base_vram_address=vram_size-ramp_vram_size;


  clearVirtualFramebuffer();
  renderMapFromCameraView(camera->px, camera->py, camera->angle);

#if 0
  {
    static int ramp_index=10;
    ramp_index=(ramp_index+1)%(num_ramp_angles*2);
    const Ramp *const r=&ramps[ramp_index];
    if(r->pixels)
      for(int y=0;y<r->h;++y)
        for(int x=0;x<r->w;++x)
        {
          plotPixelVirtual(x, y, r->pixels[x+y*r->w] ? 0xffff : 0x0000);
        }
  }
#endif

  {
    static bool_t init=TRUE;
    static uint8_t charset[256*8];
    static uint8_t translation_table[256*4*3];

    if(init)
    {
      init=FALSE;

      memset(charset,0,sizeof(charset));

      for(int h0=0;h0<5;++h0)
        for(int h1=0;h1<5;++h1)
          for(int h2=0;h2<5;++h2)
          {
            const int i=(h0*5+h1)*5+h2;
            assert(i<pow(5,3));
            assert(i<125);
            const int j=i^255;
            assert(j>=256-125);

            const int hs[3]={h0,h1,h2};
            uint8_t* ch1=&charset[i*8];
            uint8_t* ch2=&charset[j*8];
            for(int y=0;y<8;++y)
              for(int x=0;x<6;++x)
              {
                if(3-y/2<hs[x/2])
                {
                  ch1[y]|=1<<(7-x);
                  ch2[7-y]|=1<<(7-x);
                }
              }
          }

      {
        FILE* out=fopen("charset.bin","wb");
        fwrite(&charset[0], 1, sizeof(charset), out);
        fclose(out);
      }

      for(int h=0;h<4;++h)
        for(int x=0;x<3;++x)
        {
          for(int i=0;i<125;++i)
          {
            int h2=i%5;
            int h1=(i/5)%5;
            int h0=(i/25)%5;
            if(x==0)
              h0=h+1;
            else if(x==1)
              h1=h+1;
            else if(x==2)
              h2=h+1;

            translation_table[(x*4+3-h)*256+i]=(h0*5+h1)*5+h2;
          }
        }

      {
        FILE* out=fopen("translation_table.bin","wb");
        fwrite(&translation_table[0], 1, sizeof(translation_table), out);
        fclose(out);
      }

      // Hardcoded fullstates
      static const char* s[8] = {
          "000",
          "001",
          "010",
          "011",
          "100",
          "101",
          "110",
          "111",
        };
      for(int i=0;i<8;++i)
      {
        int j=0;
        if(i&1)
          j=translation_table[(0*4+0)*256+j];
        if(i&2)
          j=translation_table[(1*4+0)*256+j];
        if(i&4)
          j=translation_table[(2*4+0)*256+j];
        printf("TRANSLATION_FULLSTATE_%s: equ %3d\n",s[i],j);
      }
    }

    // Simulate drawing of column heights for TMS9918 version

    static uint8_t frame[40*24];

    for(int tx=0;tx<40;++tx)
    {
      int8_t a=column_heights[tx*3+0];
      int8_t b=column_heights[tx*3+1];
      int8_t c=column_heights[tx*3+2];
      int i=0;
      for(int ty=0;ty<12;++ty)
      {
        a-=4;
        b-=4;
        c-=4;

        if(a<0&&a>=-4)
        {
          i=translation_table[(0*4+(column_heights[tx*3+0]&3))*256+i];
        }
        if(b<0&&b>=-4)
        {
          i=translation_table[(1*4+(column_heights[tx*3+1]&3))*256+i];
        }
        if(c<0&&c>=-4)
        {
          i=translation_table[(2*4+(column_heights[tx*3+2]&3))*256+i];
        }

        // Bottom
        frame[tx+(ty+12)*40]=i;

/*
        for(int y=0;y<8;++y)
          for(int x=0;x<6;++x)
          {
            plotPixelVirtual(tx*6+x, (ty+12)*8+y, charset[i*8+y]&(1<<(7-x)) ? 0 : 1);
          }
*/

        // Top
        frame[tx+(11-ty)*40]=i^255;

/*
        for(int y=0;y<8;++y)
          for(int x=0;x<6;++x)
          {
            plotPixelVirtual(tx*6+x, (11-ty)*8+y, charset[(i^255)*8+y]&(1<<(7-x)) ? 0 : 1);
          }
*/

        //plotPixelVirtual(tx*6, VIRTUAL_FRAME_HEIGHT/2 + ty*8, 2);

        if(a<0)
          i=translation_table[(0*4+0)*256+i];
        if(b<0)
          i=translation_table[(1*4+0)*256+i];
        if(c<0)
          i=translation_table[(2*4+0)*256+i];
      }
    }

    for(int ty=0;ty<24;++ty)
      for(int tx=0;tx<40;++tx)
      {
        for(int y=0;y<8;++y)
          for(int x=0;x<6;++x)
          {
            plotPixelVirtual(tx*6+x, ty*8+y, charset[frame[tx+ty*40]*8+y]&(1<<(7-x)) ? 0 : 1);
          }
      }

    {
      FILE* out=fopen("column_heights.bin","wb");
      fwrite(&column_heights[0], 1, sizeof(column_heights), out);
      fclose(out);
    }

    {
      FILE* out=fopen("frame.bin","wb");
      fwrite(&frame[0], 1, sizeof(frame), out);
      fclose(out);
    }

#if 0
    for(int c=0;c<120;++c)
    {
      int h=column_heights[c];
      for(int x=c*2;x<c*2+2;++x)
      {
        for(int y=0;y<h;++y)
        {
          plotPixelVirtual(x, VIRTUAL_FRAME_HEIGHT/2 + y, 1);
          //plotPixelVirtual(x, VIRTUAL_FRAME_HEIGHT/2 - y, 1);
        }
        for(int y=h;y<VIRTUAL_FRAME_HEIGHT/2;++y)
        {
          plotPixelVirtual(x, VIRTUAL_FRAME_HEIGHT/2 + y, 0);
          //plotPixelVirtual(x, VIRTUAL_FRAME_HEIGHT/2 - y, 0);
        }
      }
    }
#endif

  }

  blitVirtualFramebuffer(pixel_buffer);

#if 0
  {
    for(int i=0;i<num_ramps;++i)
    {
      const Ramp *const r=&ramps[i];
      if(r->pixels && r->packed)
        for(int y=0;y<r->h;++y)
          for(int x=0;x<r->w;++x)
          {
            int xofs=0;
            plotPixel(pixel_buffer, xofs+r->atlas_x+x, ((r->atlas_y)+y)/8, r->pixels[x+y*r->w] ? 0xffff : 0x0000);
          }
    }
  }
#endif

#if 1
  for(int i=0;i<num_lines_to_draw;++i)
  {
    plotLine(pixel_buffer, lines_to_draw[i].x0, lines_to_draw[i].y0, lines_to_draw[i].x1, lines_to_draw[i].y1, lines_to_draw[i].colour);
  }
#endif

  {
    static char debug_text[8192];
    memset(debug_text, 0, sizeof(debug_text));
    snprintf(debug_text, sizeof(debug_text),
                     "walls_drawn       = %u\n"
                     "portals_entered   = %u\n"
                     "num_commands      = %u\n"
                     "cycle_count       = %u\n"
                     "milliseconds      = %6.3f\n"
                     "%% of frame budget = %4.1f%%\n"
                     "num_intersect_ray_with_wall = %d\n"
                     "num_octant_angle = %d\n"
                     "num_project_post = %d\n"
                     "num_cheap_rays = %d\n"
                     "num_reused_points = %d\n"
                     ,walls_drawn,portals_entered,captured_frame.num_commands,cycle_count,
                      double(cycle_count)/3500.0,(double(cycle_count)/3500.0)/8.0*100.0,
                      num_intersect_ray_with_wall,num_octant_angle,num_project_post,
                      num_cheap_rays,num_reused_points);

    plotString(pixel_buffer, 0, 128-32*3, debug_text, 1);
  }

  plotString(pixel_buffer, 0, 128+64, renderlog_string, 1);


  //debugDrawRooms(pixel_buffer);
  //debugDrawWalls(pixel_buffer);

  if(extern_write_frame)
  {
    {
      FILE* out=fopen("commands.bin","wb");
      fwrite(&captured_frame.commands[0], sizeof(Command), captured_frame.num_commands, out);
      fclose(out);
    }
    extern_write_frame=FALSE;
  }

}


void captureFrame(uint16_t* pixel_buffer, renderstate_t* rs)
{
  memcpy(rs->captured_frame, pixel_buffer, FRAME_WIDTH * FRAME_HEIGHT * sizeof(uint16_t));
}

static void loadTexture(const char* filename, uint16_t* out)
{
  int w = 32,h = 32,n = 4;
  
  unsigned char *data = stbi_load(filename, &w, &h, &n, 0);
  assert(data != NULL);
  assert(w == TEXTURE_TILE_SIZE && h == TEXTURE_TILE_SIZE);
  assert(n == 4);
  
  
  for(int y = 0; y < h; ++y)
    for(int x = 0; x < w; ++x)
    {
//      out[x + y * w] = RGB888toRGB565(x * 4, y * 4, 100);
      out[x + y * w] = RGB888toRGB565(data[(x + y * w) * n + 0], data[(x + y * w) * n + 1], data[(x + y * w) * n + 2]);
    }
  
  stbi_image_free(data);
}

void initRenderstate(renderstate_t* rs)
{
  memset(rs, 0, sizeof(renderstate_t));
  rs->aspect = (float)FRAME_HEIGHT / (float)FRAME_WIDTH;
  rs->viewport_zoom = 0.95f;
  
  // load textures
  loadTexture("Textures/Wood/WOODTILE.png", rs->tex_woodtile);
  loadTexture("Textures/Wood/WOODB.png", rs->tex_woodb);
  loadTexture("Textures/Wood/DARKWOOD.png", rs->tex_darkwood);
  loadTexture("Textures/Wood/CREAKYWOOD.png", rs->tex_creakywood);
  loadTexture("Textures/Industrial/CROSSCUBE.png", rs->tex_crosscube);
  loadTexture("Textures/Industrial/METALTILE.png", rs->tex_metaltile);
  loadTexture("Textures/Tech/HEXAGONS.png", rs->tex_hexagons);
}

void clipAndPlotBox(uint16_t* pixel_buffer, int32_t x, int32_t y, int32_t w, int32_t h, uint32_t colour)
{
  if(-x > w || -y > h || x >= FRAME_WIDTH || y >= FRAME_HEIGHT)
    return;
  
  if(x < 0)
  {
    w += x;
    x = 0;
  }
  
  if(y < 0)
  {
    h += y;
    y = 0;
  }
  
  if(x + w > FRAME_WIDTH)
    w = FRAME_WIDTH - x;
  
  if(y + h > FRAME_HEIGHT)
    h = FRAME_HEIGHT - y;
  
  for(int32_t v = 0; v < h; ++v)
    for(int32_t u = 0; u < w; ++u)
      pixel_buffer[(x + u) + (y + v) * FRAME_WIDTH] = colour;   
}

void plotPixel(uint16_t* pixel_buffer, uint32_t x, uint32_t y, uint32_t colour)
{
    if(x < FRAME_WIDTH && y < FRAME_HEIGHT)
        pixel_buffer[x + y * FRAME_WIDTH] = colour;
}

// bresenham line rastering
void plotLine(uint16_t* pixel_buffer, int x0, int y0, int x1, int y1, const uint32_t col)
{
   const bool_t steep = labs(y1 - y0) > labs(x1 - x0);

   if(steep)
   {
      SWAP(int, x0, y0);
      SWAP(int, x1, y1);
   }

   if(x0 > x1)
   {
      SWAP(int, x0, x1);
      SWAP(int, y0, y1);
   }

   const int deltax = x1 - x0;
   const int deltay = labs(y1 - y0);
   int error = deltax / 2;
   int y = y0;
   const int ystep = (y0 < y1) ? 1 : -1;

   for(int x = x0; x <= x1; ++x)
   {
      if(steep)
         plotPixel(pixel_buffer, y, x, col);
      else
         plotPixel(pixel_buffer, x, y, col);

      error -= deltay;

      if(error < 0)
      {
         y += ystep;
         error += deltax;
      }
   }
}




void plotChar(uint16_t* pixel_buffer, uint32_t x, uint32_t y, uint32_t c, int scale_factor)
{
    for(uint32_t v = 0; v < 12; ++v)
        for(uint32_t u = 0; u < 6; ++u)
        {
            const uint32_t u2 = u + (c & 31) * 6;
            const uint32_t v2 = v + (c / 32) * 12;
            const uint32_t x2 = x + u * scale_factor;
            const uint32_t y2 = y + v * scale_factor;
            if(u2 < FONT_IMAGE_W && v2 < FONT_IMAGE_H)
            {
                const uint32_t texel_index = (u2) + (FONT_IMAGE_H - 1 - v2) * FONT_IMAGE_W;
                const uint32_t colour = (font_image_data[texel_index / 32U] & (0x80000000U >> (texel_index & 31U))) ? 0xffffffU : 0U;
                for(int i = 0; i <  scale_factor; ++i)
                  for(int j = 0; j < scale_factor; ++j)
                  {
                    const uint32_t x3 = x2 + i;
                    const uint32_t y3 = y2 + j;
                    if(x3 < FRAME_WIDTH && y3 < FRAME_HEIGHT/* && colour > 0*/)
                        pixel_buffer[x3 + y3 * FRAME_WIDTH] = colour;
                  }
            }
        }
}

      
void plotLine3D(uint16_t* pixel_buffer, const float* v0vsp, const float* v1vsp, const uint32_t col, renderstate_t* rs)
{

  
  float line[2][3];

    line[0][0] = v0vsp[0];
    line[0][1] = v0vsp[1];
    line[0][2] = v0vsp[2];

    line[1][0] = v1vsp[0];
    line[1][1] = v1vsp[1];
    line[1][2] = v1vsp[2];
    

  float projected[2][2];
  uint32_t projected_pix[2][2];

  for(int j = 0; j < 2; ++j)
  {
      const float z = 1.0f / line[j][2];
      const float x = line[j][0] * z;
      const float y = line[j][1] * z / -rs->aspect;
      projected[j][0] = x;
      projected[j][1] = y;
  }

  const float linebox[4] = {   MIN(projected[0][0], projected[1][0]), MIN(projected[0][1], projected[1][1]),
                               MAX(projected[0][0], projected[1][0]), MAX(projected[0][1], projected[1][1]) };

  if(linebox[0] < -1.001f || linebox[1] < -1.001f || linebox[2] > +1.001f || linebox[3] > +1.001f)
      return;

  for(int j = 0; j < 2; ++j)
  {
      projected_pix[j][0] = (projected[j][0] * 0.5f + 0.5f) * FRAME_WIDTH;
      projected_pix[j][1] = (projected[j][1] * 0.5f + 0.5f) * FRAME_HEIGHT;
  }


  plotLine(pixel_buffer,  projected_pix[0][0], projected_pix[0][1], projected_pix[1][0], projected_pix[1][1], col);

}

void plotString(uint16_t* pixel_buffer, uint32_t x, uint32_t y, const char* str, int scale_factor)
{
    const uint32_t x2 = x;
    while(str[0])
    {
        if(str[0] == '\n')
        {
            x = x2;
            y += 12 * scale_factor;
        }
        else if(str[0] >= 32)
        {
            plotChar(pixel_buffer, x, y, str[0] - 32, scale_factor);
            x += 6 * scale_factor;
        }
        ++str;
    }
}

