; HALLS AND HEARTS (techdemo)

V9990_WAIT_FOR_COMMAND_READY: MACRO
      in a,(V9P_STATUS)
      rra
      jr c,$-3
      ENDM

V9990_WAIT_FOR_TRANSFER_READY: MACRO
      in a,(V9P_STATUS)
      rla
      jr nc,$-3
      ENDM

V9990_SET_BORDER_COLOUR: MACRO ?c
      push af
      ld a,?c
      call v9990_set_border_colour
      pop af
      ENDM

MAKE_LMMV: MACRO ?dx, ?dy, ?nx, ?ny, ?lop
      db #00,#00                ; R#32, R#33
      db #00,#00                ; R#34, R#35
      db ?dx&#FF,(?dx>>8)&1     ; R#36, R#37
      db ?dy&#FF,(?dy>>8)&3     ; R#38, R#39
      db ?nx&#FF,(?nx>>8)&1     ; R#40, R#41
      db ?ny&#FF,(?ny>>8)&3     ; R#42, R#43
      db %00000000              ; R#44
      db ?lop                   ; R#45
      db V9O_LMMV<<4            ; R#52
      ENDM

; Ramp descriptor
RAMP_DESCRIPTOR_ATLAS_X:  equ  0 ; 2
RAMP_DESCRIPTOR_ATLAS_Y:  equ  2 ; 2
RAMP_DESCRIPTOR_W:        equ  4 ; 1
RAMP_DESCRIPTOR_H:        equ  5 ; 1
RAMP_DESCRIPTOR_U:        equ  6 ; 2
RAMP_DESCRIPTOR_DUDX:     equ  8 ; 2
RAMP_DESCRIPTOR_DUDY:     equ 10 ; 2
RAMP_DESCRIPTOR_LENGTH:   equ 12 ; 2

RAMP_DESCRIPTOR_SIZE:     equ 14

MAKE_RAMP: MACRO ?atlas_x, ?atlas_y, ?w, ?h, ?u, ?dudx, ?dudy
      dw ?atlas_x, ?atlas_y
      db ?w, ?h
      dw ?u
      dw ?dudx, ?dudy ; Also dSdD_highprec
      dw (((?w+1)*(?h+1))*FRAME_BPP+7)/8
      ENDM


; V9990 command structure
COMMAND_SX:     equ  0 ; 2
COMMAND_SY:     equ  2 ; 2
COMMAND_DX:     equ  4 ; 2
COMMAND_DY:     equ  6 ; 2
COMMAND_NX:     equ  8 ; 2
COMMAND_NY:     equ 10 ; 2
COMMAND_ARG:    equ 12 ; 1
COMMAND_LOP:    equ 13 ; 1
COMMAND_OPCODE: equ 14 ; 1

COMMAND_SIZE:   equ 15

; Portal traversal stack frame layout
STACK_FX0:            equ  0 ; 1
STACK_FY0_UPPER:      equ  1 ; 1
STACK_FX1:            equ  2 ; 1
STACK_FY1_UPPER:      equ  3 ; 1
STACK_RAMP_UPPER:     equ  4 ; 2
STACK_FLAGS:          equ  6 ; 1
STACK_LOG_ADJACENT:   equ  7 ; 2
STACK_RELATIVE_ANGLE: equ  9 ; 2
STACK_LOG_OPPOSITE:   equ 11 ; 2
STACK_ORIENTATION:    equ 13 ; 1
STACK_X0:             equ 14 ; 2
STACK_Y0:             equ 16 ; 2
STACK_X1:             equ 18 ; 2
STACK_Y1:             equ 20 ; 2
STACK_OPPOSITE:       equ 22 ; 2
STACK_ADJACENT:       equ 24 ; 2
STACK_A0:             equ 28 ; 2    Pass by value
STACK_A1:             equ 30 ; 2    Pass by value
STACK_ITESTFLAGS0:    equ 32 ; 1
STACK_ITESTFLAGS1:    equ 33 ; 1
STACK_ITESTFLAGS:     equ 34 ; 1
STACK_CAMERA_X:       equ 36 ; 2    Pass by value
STACK_CAMERA_Y:       equ 38 ; 2    Pass by value
STACK_RAMP_LOWER:     equ 40 ; 2
STACK_FY0_LOWER:      equ 42 ; 1
STACK_FY1_LOWER:      equ 43 ; 1

STACK_SIZE:           equ 44

; Wall descriptor
WALL_X1:              equ  0 ; 2
WALL_Y1:              equ  2 ; 2
WALL_ROUTINE:         equ  4 ; 2
WALL_PORTAL_ROOM:     equ  6 ; 2

WALL_SIZE:            equ  8

MAKE_WALL: MACRO ?x0, ?y0, ?x1, ?y1, ?orientation, ?portal_room
      dw ?x1, ?y1
      IF ?orientation=0
        dw draw_wall_orientation_0
      ENDIF
      IF ?orientation=1
        dw draw_wall_orientation_1
      ENDIF
      IF ?orientation=2
        dw draw_wall_orientation_2
      ENDIF
      IF ?orientation=3
        dw draw_wall_orientation_3
      ENDIF
      dw ?portal_room
      ENDM

; Room descriptor

ROOM_NUM_WALLS:       equ  0 ; 1
ROOM_FIRST_WALL_Q0:   equ  1 ; 2
ROOM_FIRST_WALL_Q1:   equ  3 ; 2
ROOM_FIRST_WALL_Q2:   equ  5 ; 2
ROOM_FIRST_WALL_Q3:   equ  7 ; 2

ROOM_SIZE:            equ  9

MAKE_ROOM: MACRO ?num_walls, ?first_wall_q0, ?first_wall_q1, ?first_wall_q2, ?first_wall_q3
      db ?num_walls
      dw ?first_wall_q0, ?first_wall_q1, ?first_wall_q2, ?first_wall_q3
      ENDM

GEN_RAMP_X: MACRO ?unscaled_atlas_x, ?unscaled_atlas_y, ?num_ramps
      ld iy,ramp_gen_lmmm
      ld (iy+COMMAND_SY+0),?unscaled_atlas_y&#FF ; SY L
      ld (iy+COMMAND_SY+1),?unscaled_atlas_y>>8  ; SY H
      ld e,?num_ramps
outer_loop:
      ld a,(ix+RAMP_DESCRIPTOR_ATLAS_X+0)
      ld (iy+COMMAND_DX+0),a  ; DX L
      ld a,(ix+RAMP_DESCRIPTOR_ATLAS_X+1)
      ld (iy+COMMAND_DX+1),a  ; DX H
      ld a,(ix+RAMP_DESCRIPTOR_ATLAS_Y+0)
      ld (iy+COMMAND_DY+0),a  ; DY L
      ld a,(ix+RAMP_DESCRIPTOR_ATLAS_Y+1)
      ld (iy+COMMAND_DY+1),a  ; DY H
      ld (iy+COMMAND_NX+0),1                ; NX L
      ld (iy+COMMAND_NY+0),RAMP_HEIGHT&#FF  ; NY L
      ld (iy+COMMAND_NY+1),RAMP_HEIGHT>>8   ; NY H

      ex af,af'
      ld a,?unscaled_atlas_x>>8         ; Source X H
      ex af,af'

      exx
      ld de,#0000                       ; Source X LL
      ld c,(ix+RAMP_DESCRIPTOR_DUDX+0)  ; Source X LL delta
      ld b,(ix+RAMP_DESCRIPTOR_DUDX+1)
      exx
      ld hl,((?unscaled_atlas_x&#FF)<<8)|#80  ; Source X L
      ld c,(ix+RAMP_DESCRIPTOR_DUDY+0)  ; Source X L delta
      ld b,(ix+RAMP_DESCRIPTOR_DUDY+1)

      ld d,(ix+RAMP_DESCRIPTOR_W) ; Destination width
      inc d
      call ramp_gen_x_inner

      ld bc,RAMP_DESCRIPTOR_SIZE
      add ix,bc
      dec e
      jr nz,outer_loop
      ENDM

GEN_RAMP_Y: MACRO ?unscaled_atlas_x, ?unscaled_atlas_y, ?num_ramps
      ld iy,ramp_gen_lmmm
      ld (iy+COMMAND_SX+0),?unscaled_atlas_x&#FF ; SX L
      ld (iy+COMMAND_SX+1),?unscaled_atlas_x>>8  ; SX H
      ld e,?num_ramps
outer_loop:
      ld a,(ix+RAMP_DESCRIPTOR_ATLAS_X+0)
      ld (iy+COMMAND_DX+0),a  ; DX L
      ld a,(ix+RAMP_DESCRIPTOR_ATLAS_X+1)
      ld (iy+COMMAND_DX+1),a  ; DX H
      ld a,(ix+RAMP_DESCRIPTOR_ATLAS_Y+0)
      ld (iy+COMMAND_DY+0),a  ; DY L
      ld a,(ix+RAMP_DESCRIPTOR_ATLAS_Y+1)
      ld (iy+COMMAND_DY+1),a  ; DY H
      ld (iy+COMMAND_NX+0),RAMP_WIDTH&#FF  ; NX L
      ld (iy+COMMAND_NX+1),RAMP_WIDTH>>8   ; NX H
      ld (iy+COMMAND_NY+0),1               ; NY L

      ex af,af'
      ld a,?unscaled_atlas_y>>8         ; Source Y H
      ex af,af'

      exx
      ld de,#0000                       ; Source Y LL
      ld c,(ix+RAMP_DESCRIPTOR_DUDX+0)  ; Source Y LL delta
      ld b,(ix+RAMP_DESCRIPTOR_DUDX+1)
      exx
      ld hl,((?unscaled_atlas_y&#FF)<<8)|#80  ; Source Y L
      ld c,(ix+RAMP_DESCRIPTOR_DUDY+0)  ; Source Y L delta
      ld b,(ix+RAMP_DESCRIPTOR_DUDY+1)

      ld d,(ix+RAMP_DESCRIPTOR_H) ; Destination height
      inc d
      call ramp_gen_y_inner

      ld bc,RAMP_DESCRIPTOR_SIZE
      add ix,bc
      dec e
      jr nz,outer_loop
      ENDM

CLIP_WALL: MACRO

      IF ENABLE_STAT_COUNTERS
        push af
        ld a,(stat_num_clip_walls)
        inc a
        ld (stat_num_clip_walls),a
        pop af
      ENDIF

      ; Calculate opposite for remapped ray

      ; hl ϵ (-256, +256)

      ; Use A to indicate negative angle
      xor a
      ; Calculate abs(angle)
      bit 7,h
      jr z,$+2+1+1+1+1+1+1+1+2
      ; Negate HL
      ld a,h  ;  5c
      cpl     ;  5c
      ld h,a  ;  5c
      ld a,l  ;  5c
      cpl     ;  5c
      ld l,a  ;  5c
      inc hl  ;  7c
      ld a,1
      ; hl ϵ [0, +256)

      add hl,hl           ; 12c
      ld de,logtan_table  ; 11c
      add hl,de           ; 12c
      ld e,(hl)           ;  8c
      inc hl              ;  7c
      ld d,(hl)           ;  8c
      ex de,hl
      ; log_tangent is now in HL
      ; log_adjacent is kept in BC
      add hl,bc           ; 12c
      ; log_opposite=max(0,log_tangent+log_adjacent)
      bit 7,h
      jr z,$+2+3
      ld hl,#0000
      ld (iy+STACK_LOG_OPPOSITE+0),l
      ld (iy+STACK_LOG_OPPOSITE+1),h

      add hl,hl
      ld de,exp_table     ; 11c
      add hl,de           ; 12c
      ld e,(hl)           ;  8c
      inc hl              ;  7c
      ld d,(hl)           ;  8c
      ex de,hl            ;  5c
      ; hl ϵ [0, +4096)
 
      ; Negate opposite if the angle was negative
      and a
      jr z,$+2+1+1+1+1+1+1+1
      ; Negate HL
      ld a,h  ;  5c
      cpl     ;  5c
      ld h,a  ;  5c
      ld a,l  ;  5c
      cpl     ;  5c
      ld l,a  ;  5c
      inc hl  ;  7c
      ; hl ϵ (-4096, +4096)
      ENDM

DRAW_WALL_ORIENTATION_N: MACRO ?orientation, ?o0, ?o1, ?o2, ?o3
      ld (iy+STACK_ORIENTATION),?orientation

      ; Skip ray 0 if an intersection has been found already
      bit 0,(iy+STACK_FLAGS)
      jr nz,skip_ray_0

      ; Clip the wall to the frustum sides

      ; Note that for a given wall orientation and adjacent, the opposites
      ; of the ray intersection points are constant, so they need to be recalculated only
      ; at orientation changes within a room.

      ; Test for intersection with ray 0
      ; If there is no intersection of ray 0 with the line of this wall, then since there
      ; must be a previous wall which did then the previous wall's endpost can be re-used.
      bit ?orientation,(iy+STACK_ITESTFLAGS0)     ; Ray 0 intersection test flags
      jr nz,skip_ray_0

      ; Calculate opposite for ray 0

      ; Remap ray 0 angle for clipping
      ld l,(iy+STACK_A0+0)         ; 21c   Ray 0 angle
      ld h,(iy+STACK_A0+1)         ; 21c
      IF ?orientation=0
        ld de,-(NUM_ANGLE_UNITS*3)/4
        add hl,de
      ENDIF
      IF ?orientation=1
        bit 1,h
        jr z,$+2+3+1+2
        ld de,NUM_ANGLE_UNITS
        and a
        sbc hl,de
      ENDIF
      IF ?orientation=2
        ex de,hl
        ld hl,+(NUM_ANGLE_UNITS*1)/4
        and a
        sbc hl,de
      ENDIF
      IF ?orientation=3
        ex de,hl
        ld hl,+(NUM_ANGLE_UNITS*1)/2
        and a
        sbc hl,de
      ENDIF
      ; hl ϵ (-256, +256)
      CLIP_WALL
      ; hl ϵ (-4096, +4096)
      ; Opposite for ray 0 is now in HL

      ; Compare computed opposite against wall extents

      ; If opposite >= o3 then discard this wall (return immediately)
      ld c,(iy+?o3+0)
      ld b,(iy+?o3+1)
      ld e,l
      ld d,h
      and a
      sbc hl,bc
      IF ?orientation<2
        ret p
      ELSE
        ret m
      ENDIF

      ld l,e
      ld h,d
      ld (iy+STACK_OPPOSITE+0),l
      ld (iy+STACK_OPPOSITE+1),h
      ; If opposite <= o2 then re-calculate the opposite from o2
      ld c,(iy+?o2+0)
      ld b,(iy+?o2+1)
      and a
      sbc hl,bc
      jr z,use_clipped_opposite_0
      IF ?orientation<2
        jp p,use_clipped_opposite_0
      ELSE
        jp m,use_clipped_opposite_0
      ENDIF

skip_ray_0:

      set 0,(iy+STACK_FLAGS)

      ; Do not rely on previous calculation if this is the very first
      ; vertex, since there is none.
      ld a,(global_portal_traversal_flags)
      bit 0,a
      jr z,use_clipped_opposite_0

      ; Re-use the previously-calculated vertex

      ld a,(iy+STACK_FX1)
      ld (iy+STACK_FX0),a
      ld a,(iy+STACK_FY1_UPPER)
      ld (iy+STACK_FY0_UPPER),a
      ld a,(iy+STACK_FY1_LOWER)
      ld (iy+STACK_FY0_LOWER),a
      ld a,(iy+STACK_SIZE+STACK_A1+0)
      ld (iy+STACK_SIZE+STACK_A0+0),a
      ld a,(iy+STACK_SIZE+STACK_A1+1)
      ld (iy+STACK_SIZE+STACK_A0+1),a

      jr after_ray_0

      ; Else use the clipped opposite as-is
use_clipped_opposite_0:

      set 0,(iy+STACK_FLAGS)

      ; The angle is taken to be equal to A0
      ; Project endpost 0
      call project_post_clipped_a0
      ld a,(iy+STACK_FX1)
      ld (iy+STACK_FX0),a
      ld a,(iy+STACK_FY1_UPPER)
      ld (iy+STACK_FY0_UPPER),a
      ld a,(iy+STACK_FY1_LOWER)
      ld (iy+STACK_FY0_LOWER),a

after_ray_0:

      ld hl,global_portal_traversal_flags
      set 0,(hl)

      ; Skip ray 1 if an intersection with ray 0 has not been found already
      bit 0,(iy+STACK_FLAGS)
      jp z,skip_ray_1

      ; Test for intersection with ray 1
      bit ?orientation,(iy+STACK_ITESTFLAGS1)     ; Ray 1 intersection test flags
      jr nz,skip_ray_1

      ; Calculate opposite for ray 1

      ld c,(iy+STACK_LOG_ADJACENT+0)         ; 21c
      ld b,(iy+STACK_LOG_ADJACENT+1)         ; 21c

      ; Remap ray 1 angle for clipping
      ld l,(iy+STACK_A1+0)         ; 21c   Ray 1 angle
      ld h,(iy+STACK_A1+1)         ; 21c
      IF ?orientation=0
        ld de,-(NUM_ANGLE_UNITS*3)/4
        add hl,de
      ENDIF
      IF ?orientation=1
        bit 1,h
        jr z,$+2+3+1+2
        ld de,NUM_ANGLE_UNITS
        and a
        sbc hl,de
      ENDIF
      IF ?orientation=2
        ex de,hl
        ld hl,+(NUM_ANGLE_UNITS*1)/4
        and a
        sbc hl,de
      ENDIF
      IF ?orientation=3
        ex de,hl
        ld hl,+(NUM_ANGLE_UNITS*1)/2
        and a
        sbc hl,de
      ENDIF
      ; hl ϵ (-256, +256)
      CLIP_WALL
      ; hl ϵ (-4096, +4096)
      ; Opposite for ray 1 is now in HL

      ; Compare computed opposite against wall extents

      ; If opposite <= o2 then discard this wall (return immediately)
      ld e,l
      ld d,h
      ld l,(iy+?o2+0)
      ld h,(iy+?o2+1)
      and a
      sbc hl,de
      IF ?orientation<2
        ret p
      ELSE
        ret m
      ENDIF

      ld l,e
      ld h,d
      ld (iy+STACK_OPPOSITE+0),l
      ld (iy+STACK_OPPOSITE+1),h
      ; If opposite >= o0 and opposite <= o1 then use this opposite, and later return
      ld l,(iy+?o3+0)
      ld h,(iy+?o3+1)
      and a
      sbc hl,de
      jr z,use_clipped_opposite_1
      IF ?orientation<2
        jp p,use_clipped_opposite_1
      ELSE
        jp m,use_clipped_opposite_1
      ENDIF

skip_ray_1:
      ld l,(iy+?o3+0)
      ld h,(iy+?o3+1)
      ld (iy+STACK_OPPOSITE+0),l
      ld (iy+STACK_OPPOSITE+1),h
      ; Calculate abs(opposite)
      ; Negate HL if it is negative.
      bit 7,h
      jr z,$+2+1+1+1+1+1+1+1
      ld a,h
      cpl
      ld h,a
      ld a,l
      cpl
      ld l,a
      inc hl
      ; hl ϵ (0, 4096)
      add hl,hl           ; 12c
      ld bc,log_table     ; 11c
      add hl,bc           ; 12c
      ld c,(hl)
      inc hl
      ld b,(hl)
      ; bc ϵ [0, 4096)
      ld (iy+STACK_LOG_OPPOSITE+0),c         ; 21c
      ld (iy+STACK_LOG_OPPOSITE+1),b         ; 21c

      ; The angle is not known yet, so a full projection is needed
      ; Project endpost 1
      call project_post
      jr after_ray_1

      ; Else use the clipped opposite as-is
use_clipped_opposite_1:

      ; Early-out of the room loop, since this is the last visible wall in the room.
      set 1,(iy+STACK_FLAGS)

      ; The angle is taken to be equal to A1
      ; Project endpost 1
      call project_post_clipped_a1

after_ray_1:

      ; Clamp to the portal bounds
      ld a,(iy-STACK_SIZE+STACK_FX0)
      cp (iy+STACK_FX0)
      jr nc,$+2+3
      ld (iy+STACK_FX0),a

      ld a,(iy-STACK_SIZE+STACK_FX1)
      cp (iy+STACK_FX1)
      jr c,$+2+3
      ld (iy+STACK_FX1),a

      ; Discard if invisibly-thin or flipped.
      ld a,(iy+STACK_FX1)
      cp (iy+STACK_FX0)
      ret nc

      bit 7,(ix+WALL_PORTAL_ROOM+1)
      jp z,enter_portal

      ; Calculate relative camera angle
      ; relative_angle=camera_angle-(NUM_ANGLE_UNITS*3)/4;
      ld hl,(camera_angle)
      IF ?orientation=0
        ld de,-(NUM_ANGLE_UNITS*3)/4
        add hl,de
      ENDIF
      IF ?orientation=1
      ENDIF
      IF ?orientation=2
        ld de,-NUM_ANGLE_UNITS/4
        add hl,de
      ENDIF
      IF ?orientation=3
        ld de,-NUM_ANGLE_UNITS/2
        add hl,de
      ENDIF
      ld (iy+STACK_RELATIVE_ANGLE+0),l
      ld (iy+STACK_RELATIVE_ANGLE+1),h

      ; Cached vertex projection possibilities:
      ;   previous_vertex_in_room - Available after the first visible wall has been drawn
      ;   entered_portal_vertex_1 - Available always except in the first room
      ;   entered_portal_vertex_2 - Available always except in the first room

      ; It should be knowable offline whether or not the two latter cases can be assumed or not.
      ; Since orderings are already found offline, backface culling can be done offline too.

      IF ?orientation=0
        ld hl,SHADE_UPPER_WALL_A|(SHADE_MIDDLE_WALL_A<<8)
        ld (wall_shade+0),hl
        ld a,SHADE_LOWER_WALL_A
        ld (wall_shade+2),a
      ENDIF
      IF ?orientation=1
        ld hl,SHADE_UPPER_WALL_B|(SHADE_MIDDLE_WALL_B<<8)
        ld (wall_shade+0),hl
        ld a,SHADE_LOWER_WALL_B
        ld (wall_shade+2),a
      ENDIF
      IF ?orientation=2
        ld hl,SHADE_UPPER_WALL_A|(SHADE_MIDDLE_WALL_A<<8)
        ld (wall_shade+0),hl
        ld a,SHADE_LOWER_WALL_A
        ld (wall_shade+2),a
      ENDIF
      IF ?orientation=3
        ld hl,SHADE_UPPER_WALL_B|(SHADE_MIDDLE_WALL_B<<8)
        ld (wall_shade+0),hl
        ld a,SHADE_LOWER_WALL_B
        ld (wall_shade+2),a
      ENDIF

      jp generate_wall_commands
      ENDM

PROJECT_POST_Y: MACRO
      ; Y projection

      ; log(cos)
      ld a,d
      cpl
      ld h,0
      ld l,a
      inc hl
      add hl,hl
      ld de,logsin_table
      add hl,de
      ld e,(hl)
      inc hl
      ld d,(hl)

      ; log_partial_sum
      pop hl ; Re-use the index computed earlier.
      ld bc,log_partial_sum_table
      add hl,bc
      ld c,(hl)
      inc hl
      ld b,(hl)

      ; log(c) * 2 = log(wall_height) / log(tangent_max) * 2047 * 2
      ld hl,3342*2
      and a
      sbc hl,de
      and a
      sbc hl,bc

      pop bc ; log_opposite
      and a
      sbc hl,bc

      ; Bobbing
      ld bc,(log_bob)
      ld de,exp_table

      REPT 2,?I
        push hl
        ; Clamp
        ld a,h
        and #F8
        jr z,$+2+3
        ld hl,#0800
        ; Exponentiation
        IF ?I=0
          add hl,bc
        ELSE
          and a
          sbc hl,bc
        ENDIF
        add hl,hl
        add hl,de
        ld a,(hl)
        IF ?I=0
          ld (iy+STACK_FY1_UPPER),a
        ELSE
          ld (iy+STACK_FY1_LOWER),a
        ENDIF
        pop hl
      ENDM
      ENDM

; BIOS ROM
CGTABL: equ #0004

; BIOS Routines
RSLREG: equ #0138
WSLREG: equ #013B
ENASLT: equ #0024

; System Variables
FORCLR: equ #F3E9
BAKCLR: equ #F3EA
BDRCLR: equ #F3EB
TXTNAM: equ #F3B3
TXTCGP: equ #F3B7
HKEY:   equ #FD9A
CLIKSW: equ #F3DB
T32NAM: equ #F3BD
T32COL: equ #F3BF
T32CGP: equ #F3C1
T32ATR: equ #F3C3
T32PAT: equ #F3C5
GRPNAM: equ #F3C7
GRPCOL: equ #F3C9
GRPCGP: equ #F3CB
GRPATR: equ #F3CD
GRPPAT: equ #F3CF

; VRAM
PATTBL: equ #0000
NAMTBL: equ #0800
SATRBL: equ #1400
COLTBL: equ #1480
SPATBL: equ #1800

; Ports
VDPDAT: equ #98 ; Port #0
VDPCTL: equ #99 ; Port #1
VDPPAL: equ #9A ; Port #2
VDPREG: equ #9B ; Port #3
SLOTSL: equ #A8
KEYROW: equ #A9
KEYSEL: equ #AA
PSGREG: equ #A0
PSGWRI: equ #A1
PSGREA: equ #A2
OPLLRG: equ #7C
OPLLDT: equ #7D

; PSG Registers
PSG_REG_TONE_0: equ 0
PSG_REG_TONE_1: equ 1
PSG_REG_TONE_2: equ 2
PSG_REG_TONE_3: equ 3
PSG_REG_TONE_4: equ 4
PSG_REG_TONE_5: equ 5
PSG_REG_NOISE:  equ 6
PSG_REG_MIXER:  equ 7 ; Important note: bit 6 must be 0, and bit 7 must be 1.
PSG_REG_AMP_0:  equ 8
PSG_REG_AMP_1:  equ 9
PSG_REG_AMP_2:  equ 10
PSG_REG_ENV_0:  equ 11
PSG_REG_ENV_1:  equ 12
PSG_REG_ENV_2:  equ 13

; V9990 Ports
V9P_VRAMDAT: equ #60 ; VRAM Data
V9P_PALDATA: equ #61 ; Palette Data
V9P_COMDATA: equ #62 ; Command Data
V9P_REGDATA: equ #63 ; Register Data
V9P_REGSLCT: equ #64 ; Register Select
V9P_STATUS:  equ #65 ; Status
V9P_IFLAGS:  equ #66 ; Interrupt Flags
V9P_SYSCTRL: equ #67 ; System Control

; V9990 Registers
V9R_VRAMWR0: equ  0  ; VRAM Write Address
V9R_VRAMWR1: equ  1
V9R_VRAMWR2: equ  2
V9R_VRAMRD0: equ  3  ; VRAM Read Address
V9R_VRAMRD1: equ  4
V9R_VRAMRD2: equ  5
V9R_SCNMOD0: equ  6  ; Screen Mode
V9R_SCNMOD1: equ  7
V9R_CONTROL: equ  8  ; Control
V9R_INTRPT0: equ  9  ; Interrupt
V9R_INTRPT1: equ 10
V9R_INTRPT2: equ 11
V9R_INTRPT3: equ 12
V9R_PALCTRL: equ 13  ; Palette Control
V9R_PALPNTR: equ 14  ; Palette Pointer
V9R_BDRPCOL: equ 15  ; Backdrop Colour
V9R_DISPADJ: equ 16  ; Display Adjust
V9R_SCROLL0: equ 17  ; Scroll Control
V9R_SCROLL1: equ 18
V9R_SCROLL2: equ 19
V9R_SCROLL3: equ 10
V9R_SCROLL4: equ 21
V9R_SCROLL5: equ 22
V9R_SCROLL6: equ 23
V9R_SCROLL7: equ 24
V9R_SPRBASE: equ 25  ; Sprite Generator Base Address
V9R_LCDCTRL: equ 26  ; LCD Control
V9R_PRICTRL: equ 27  ; Priority Control
V9R_CURPOFS: equ 28  ; Cursor Sprite Palette Offset
V9R_CMNDWR0: equ 32  ; Command (Write)
V9R_CMDLOGT: equ 45  ; Logic + TP
V9R_FONTCOL: equ 48  ; Font Colour
V9R_CMNDOPC: equ 52  ; Command Operation Code
V9R_CMNDRD0: equ 53  ; Command (Read)

; V9990 Opcodes
V9O_STOP:     equ  0
V9O_LMMC:     equ  1
V9O_LMMV:     equ  2
V9O_LMCM:     equ  3
V9O_LMMM:     equ  4
V9O_CMMC:     equ  5
V9O_CMMM:     equ  7
V9O_BMXL:     equ  8
V9O_BMLX:     equ  9
V9O_BMLL:     equ 10
V9O_LINE:     equ 11
V9O_SEARCH:   equ 12
V9O_POINT:    equ 13
V9O_PSET:     equ 14
V9O_ADVANCE:  equ 15

; Constants
FRAME_WIDTH:          equ 256 ; B1 mode
FRAME_HEIGHT:         equ 212
FRAME_BPP:            equ 2
RAMP_WIDTH:           equ FRAME_WIDTH
RAMP_HEIGHT:          equ FRAME_HEIGHT/2
RAMP_PIXEL_PALCOL:    equ %11
IMAGE_SPACE_W:        equ 512
IMAGE_SPACE_H:        equ 4096
MAX_COMMANDS:         equ 128
NUM_ANGLE_UNITS:      equ 1024
ANGLE_UNIT_FOV:       equ NUM_ANGLE_UNITS/4
HORIZON_Y_POSITION:   equ FRAME_HEIGHT/2
HUD_END_Y_POSITION:   equ 16
UNITS_PER_CELL:       equ 64
UNITS_PER_TILE:       equ 256
MAX_PORTAL_DEPTH:     equ 5
PLAYER_BOX_HALF_WIDTH:equ (UNITS_PER_CELL*16)/3

BACKGROUND_COLOUR:    equ %10101010
BORDER_COLOUR:        equ 12

SHADE_UPPER_BG:       equ %00011000
SHADE_UPPER_WALL_A:   equ %00010100
SHADE_UPPER_WALL_B:   equ %00011100

SHADE_MIDDLE_WALL_A:  equ %00000011
SHADE_MIDDLE_WALL_B:  equ %00001111

SHADE_LOWER_BG:       equ %00001000
SHADE_LOWER_WALL_A:   equ %00001001
SHADE_LOWER_WALL_B:   equ %00001011

PALETTE_OFFSET_HUD:           equ 0
PALETTE_OFFSET_ABOVE_HORIZON: equ 4>>2
PALETTE_OFFSET_BELOW_HORIZON: equ 8>>2

FONT_VRAM_LOCATION_X: equ 0
FONT_VRAM_LOCATION_Y: equ 4096-8

FONT_VRAM_SIZE_X: equ 256
FONT_VRAM_SIZE_Y: equ 8

FONT_VRAM_LENGTH: equ FONT_VRAM_SIZE_X*FONT_VRAM_SIZE_Y/4

ENABLE_STAT_COUNTERS: equ 0


  ; 48Kbyte ROM
  org #0000

  ds 16
  jp $

  ; Interrupt Service Routine
  ds #0038 - $
  push af
  in a,(V9P_IFLAGS)
  rrca
  jr c,handle_V_interrupt
  rrca
  jr c,handle_H_interrupt ; Must come before CE to ensure a close-to-horizon split.
  rrca
  jr c,handle_CE_interrupt
  pop af
  ei
  ret

handle_CE_interrupt:
  ld a,%00000100
  out (V9P_IFLAGS),a
  push hl
  ; Check to see if the commandlist is empty
  ld hl,(commandlist_read_length_ptr)
  xor a
  cp (hl)
  jr z,handle_CE_interrupt.empty
  ; Execute a command from the commandlist
  push bc
  dec (hl)
  ld hl,(commandlist_read_ptr)
  ; Put the read head at the start address of the next command
  ld bc,-COMMAND_SIZE
  add hl,bc
  ld (commandlist_read_ptr),hl
  ; Execute a command
  ld a,V9R_CMNDWR0
  out (V9P_REGSLCT),a
  ld c,V9P_REGDATA
  REPT COMMAND_SIZE-1
    outi
  ENDM
  ld a,V9R_CMNDOPC
  out (V9P_REGSLCT),a
  outi ; OP-CODE
  pop bc
  pop hl
  pop af
  ei
  ret
handle_CE_interrupt.empty:
  ld a,1
  ld (last_command_done_flag),a
  pop hl
  pop af
  ei
  ret

handle_H_interrupt:
  ld a,%00000010
  out (V9P_IFLAGS),a
  push hl

  ; Set the palette offset register.
  ld a,V9R_PALCTRL
  out (V9P_REGSLCT),a
  ld hl,(horizontal_splits_ptr)
  ld a,(hl)
  out (V9P_REGDATA),a
  inc hl
  ; Set the split position register.
  ld a,V9R_INTRPT1
  out (V9P_REGSLCT),a
  ld a,(hl)
  out (V9P_REGDATA),a
  inc hl
  ld (horizontal_splits_ptr),hl
  pop hl
  pop af
  ei
  ret

handle_V_interrupt:
  ld a,%00000001
  out (V9P_IFLAGS),a
  ; Increment the frameskip count
  ld a,(frames_to_skip)
  inc a
  ld (frames_to_skip),a

  ; Reset the splits
  ld a,horizontal_splits&#FF
  ld (horizontal_splits_ptr+0),a
  ld a,horizontal_splits>>8
  ld (horizontal_splits_ptr+1),a
  ; Set the palette offset register.
  ld a,V9R_PALCTRL
  out (V9P_REGSLCT),a
  ld a,PALETTE_OFFSET_HUD
  out (V9P_REGDATA),a
  ; Set the split position register.
  ld a,V9R_INTRPT1
  out (V9P_REGSLCT),a
  ld a,HUD_END_Y_POSITION
  out (V9P_REGDATA),a


  ; Check to see if the writing commandlist if full yet or not
  ld a,(commandlist_filled_flag)
  and a
  jp z,handle_V_interrupt.return
  ; It is full.

  ; Check to see if the reading commandlist is finished yet or not
  ld a,(last_command_done_flag)
  and a
  jp z,handle_V_interrupt.return
  ; It is empty.

  IF 0
  ; Horrible WebMSX
  in a,(V9P_STATUS)
  rra
  jp c,handle_V_interrupt.return
  push hl
  ld hl,(commandlist_read_length_ptr)
  xor a
  cp (hl)
  pop hl
  jp nz,handle_V_interrupt.return
  ENDIF

  ; Both conditions hold (all pending commands have executed and there is
  ; a complete commandlist ready to begin execution).

  ; So do a swap and update the game logic.
  push hl
  push bc
  push ix

  call swap_commandlists

  ; Start a command to clear the framebuffer
  ld ix,clear_framebuffer_lmmv_ram
  ld a,(buffer_swap)
  xor 1
  ld (ix+COMMAND_DY+1),a
  ld hl,clear_framebuffer_lmmv_ram
  ld a,V9R_CMNDWR0
  out (V9P_REGSLCT),a
  ld c,V9P_REGDATA
  REPT 14
    outi
  ENDM
  ld a,V9R_CMNDOPC
  out (V9P_REGSLCT),a
  outi

  ld a,(frames_to_skip)
  ld b,a
frameskip_loop:
  push bc

  ; Handle inputs
  in a,(KEYSEL)
  and #F0
  or 8 ; Select row 8
  out (KEYSEL),a
  in a,(KEYROW)
  bit 7,a ; Right
  call z,right_pressed
  bit 6,a ; Down
  call z,down_pressed
  bit 5,a ; Up
  call z,up_pressed
  bit 4,a ; Left
  call z,left_pressed

  ; Apply dynamics
  call move_camera_step

  pop bc
  djnz frameskip_loop

  ; Update the bobbing animation
  ld a,(bob_phase)
  ld l,a
  ld h,0
  add hl,hl
  ld bc,log_bob_table
  add hl,bc
  ld c,(hl)
  inc hl
  ld b,(hl)
  ld (log_bob),bc

  xor a
  ld (frames_to_skip),a

  pop ix
  pop bc
  pop hl

  ; Reset the flags
  xor a
  ld (last_command_done_flag),a
  ld (commandlist_filled_flag),a
handle_V_interrupt.return:
  pop af
  ei
  ret

top:
  ; Initialise RAM
  ld hl,RAM_START
  ld (hl),0
  ld de,RAM_START+1
  ld bc,RAM_END-RAM_START-1
  ldir

  ; Set up the V9990
  call v9990_reset
  call v9990_setmode_B1_BP2_512x4096
  call v9990_set_512k

  ld a,V9R_CONTROL
  out (V9P_REGSLCT),a
  ld a,%01000010
  out (V9P_REGDATA),a

  ; Set some common parameters

  V9990_WAIT_FOR_COMMAND_READY

  ; Set DIY+DIX
  ld a,44
  out (V9P_REGSLCT),a
  ld a,%00000000
  out (V9P_REGDATA),a ; R#44

  ; Set Write Mask
  ld a,46
  out (V9P_REGSLCT),a
  ld a,%11111111
  out (V9P_REGDATA),a ; R#46
  out (V9P_REGDATA),a ; R#47

  ; Set FC - this value will stay constant throughout.
  ld a,V9R_FONTCOL
  out (V9P_REGSLCT),a
  ld a,BACKGROUND_COLOUR
  out (V9P_REGDATA),a ; R#48
  out (V9P_REGDATA),a ; R#49


  ; Clear all of VRAM
  V9990_WAIT_FOR_COMMAND_READY
  ld a,V9R_CMNDWR0+4
  out (V9P_REGSLCT),a
  ; Set DX
  xor a
  out (V9P_REGDATA),a ; R#36
  out (V9P_REGDATA),a ; R#37
  ; Set DY
  out (V9P_REGDATA),a ; R#38
  out (V9P_REGDATA),a ; R#39
  ; Set NX
  out (V9P_REGDATA),a ; R#40
  ld a,IMAGE_SPACE_W>>8 ; 512
  out (V9P_REGDATA),a ; R#41
  ; Set NY
  xor a
  out (V9P_REGDATA),a ; R#42
  ld a,IMAGE_SPACE_H>>8 ; 4096
  out (V9P_REGDATA),a ; R#43

  ; Set Logical Operation, for clearing to zero.
  ld a,V9R_CMDLOGT
  out (V9P_REGSLCT),a
  ld a,%00000000
  out (V9P_REGDATA),a ; R#45

  ; Set OP
  ld a,V9R_CMNDOPC
  out (V9P_REGSLCT),a
  ld a,V9O_LMMV<<4
  out (V9P_REGDATA),a

  V9990_WAIT_FOR_COMMAND_READY

  ; Set Logical Operation, for uploading images.
  ld a,V9R_CMDLOGT
  out (V9P_REGSLCT),a
  ld a,%00001100
  out (V9P_REGDATA),a ; R#45

  ; Upload the font image.
  call upload_font

  ; Generate the two "template" ramp images on CPU.
  ld ix,ramp_descriptors+UNSCALED_RAMP_0_INDEX*RAMP_DESCRIPTOR_SIZE
  call draw_ramp_image
  call upload_ramp_image
  ld ix,ramp_descriptors+UNSCALED_RAMP_1_INDEX*RAMP_DESCRIPTOR_SIZE
  call draw_ramp_image
  call upload_ramp_image

  ; Initialise RAM again
  ld hl,RAM_START
  ld (hl),0
  ld de,RAM_START+1
  ld bc,RAM_END-RAM_START-1
  ldir

  ; Initialise the ramp-generation LMMM command data
  ld hl,ramp_gen_lmmm
  ld (hl),0
  ld de,ramp_gen_lmmm+1
  ld bc,14-1
  ldir

  ld iy,ramp_gen_lmmm
  ld (iy+COMMAND_ARG),%00000000 ; ARG
  ld (iy+COMMAND_LOP),%00011100 ; LOP

  ; Use the V9990 to generate the rest of the ramps by scaling down the templates.
  ld ix,ramp_descriptors
  GEN_RAMP_X UNSCALED_RAMP_0_ATLAS_X, UNSCALED_RAMP_0_ATLAS_Y, NUM_X_SCALED_RAMPS_0
  GEN_RAMP_X UNSCALED_RAMP_1_ATLAS_X, UNSCALED_RAMP_1_ATLAS_Y, NUM_X_SCALED_RAMPS_1
  GEN_RAMP_Y UNSCALED_RAMP_0_ATLAS_X, UNSCALED_RAMP_0_ATLAS_Y, NUM_Y_SCALED_RAMPS_0
  GEN_RAMP_Y UNSCALED_RAMP_1_ATLAS_X, UNSCALED_RAMP_1_ATLAS_Y, NUM_Y_SCALED_RAMPS_1

  ; ----------------------------------------------------

  call init_commandlists

  ld hl,(18*UNITS_PER_CELL+UNITS_PER_CELL/2)*16
  ld (camera_x),hl

  ld hl,(17*UNITS_PER_CELL+UNITS_PER_CELL/2)*16
  ld (camera_y),hl

  ; ----------------------------------------------------

  ; Reset Scroll Registers
  ld a,V9R_SCROLL0
  out (V9P_REGSLCT),a
  ld a,#00
  out (V9P_REGDATA),a ; R#17
  ld a,#00
  out (V9P_REGDATA),a ; R#18
  ld a,#00
  out (V9P_REGDATA),a ; R#19
  ld a,#00
  out (V9P_REGDATA),a ; R#20
  ld a,#00
  out (V9P_REGDATA),a ; R#21
  ld a,#00
  out (V9P_REGDATA),a ; R#22
  ld a,#00
  out (V9P_REGDATA),a ; R#23
  ld a,#00
  out (V9P_REGDATA),a ; R#24

  ; Wait for all commands to be completed
  V9990_WAIT_FOR_COMMAND_READY

  ; Fill the palette
  ld a,V9R_PALPNTR
  out (V9P_REGSLCT),a
  xor a
  out (V9P_REGDATA),a
  ld c,V9P_PALDATA
  ld b,palette_rgb_end-palette_rgb_start
  ld hl,palette_rgb_start
  otir

  V9990_SET_BORDER_COLOUR BORDER_COLOUR

  xor a
  ld (scroll_position+0),a
  ld (scroll_position+1),a
  ld (scroll_position+2),a
  ld (scroll_position+3),a

  ; Set interrupt enables
  ld a,V9R_INTRPT0
  out (V9P_REGSLCT),a
  ld a,%00000111 ; IECE, IEH, IEV
  out (V9P_REGDATA),a

  ld a,V9R_INTRPT1
  out (V9P_REGSLCT),a
  ld a,HUD_END_Y_POSITION ; Interrupt Line (bit 7-0)
  out (V9P_REGDATA),a

  ld a,V9R_INTRPT2
  out (V9P_REGSLCT),a
  ld a,%00000000 ; Interrupt Line (bit 9-8) and IEHM
  out (V9P_REGDATA),a

  ld a,V9R_INTRPT3
  out (V9P_REGSLCT),a
  ld a,%00000000 ; Interrupt Horizontal Position
  out (V9P_REGDATA),a

  ld hl,clear_framebuffer_lmmv
  ld de,clear_framebuffer_lmmv_ram
  ld bc,COMMAND_SIZE
  ldir

  ; Enable the display
  call v9990_enable_display

  ld a,1
  ld (last_command_done_flag),a

  ei

mainloop:

  IF ENABLE_STAT_COUNTERS
    ld hl,stat_counters_start
    ld (hl),0
    ld de,stat_counters_start+1
    ld bc,stat_counters_end-stat_counters_start-1
    ldir
  ENDIF

  call append_hud_commands
  call draw_main_view

  ld hl,commandlist_filled_flag
  ld (hl),1
  xor a

  cp (hl)
  jr nz,$-1

  jr mainloop

; ---------------------- Input Handling and Dynamics ----------------------

; Input angle in BC
; Output sinus in BC
calc_camera_sinus:
  ld l,b
  ld h,0
  add hl,hl
  ld de,calc_camera_sinus.jump_table
  add hl,de
  ld e,(hl)
  inc hl
  ld d,(hl)
  ex de,hl
  jp (hl)
calc_camera_sinus.q0:
  ld l,c
  ld h,0
  ; Load
  ld bc,camera_move_sinus_table
  add hl,bc
  ld b,0
  ld c,(hl)
  ret
calc_camera_sinus.q1:
  ; Reverse
  ld a,c
  cpl
  ld l,a
  ld h,0
  ; Load
  ld bc,camera_move_sinus_table
  add hl,bc
  ld b,0
  ld c,(hl)
  ret
calc_camera_sinus.q2:
  ld l,c
  ld h,0
  ; Load
  ld bc,camera_move_sinus_table
  add hl,bc
  ld b,0
  ld c,(hl)
  ; Negate
  ld a,b
  cpl
  ld b,a
  ld a,c
  cpl
  ld c,a
  inc bc
  ret
calc_camera_sinus.q3:
  ; Reverse
  ld a,c
  cpl
  ld l,a
  ld h,0
  ; Load
  ld bc,camera_move_sinus_table
  add hl,bc
  ld b,0
  ld c,(hl)
  ; Negate
  ld a,b
  cpl
  ld b,a
  ld a,c
  cpl
  ld c,a
  inc bc
  ret
calc_camera_sinus.jump_table:
  dw calc_camera_sinus.q0,calc_camera_sinus.q1,calc_camera_sinus.q2,calc_camera_sinus.q3

; HL = Coordinate X
; DE = Coordinate Y
; Returns result in Z flag
check_position:
  ; Divide by 16*128=2048=256*8
  REPT 3
    sra h
  ENDM
  ld a,15
  and h
  ld h,a
  ex de,hl
  ; Divide by 16*128=2048=256*8
  REPT 3
    sra h
  ENDM
  ld a,15
  and h
  REPT 4
    rlca
  ENDM
  or d
  ld l,a
  ld h,0
  add hl,hl
  ld bc,maze
  add hl,bc
  ; Load the address
  inc hl
  bit 7,(hl)
  ret

check_x_move:
  ; Expand in X by box width by move direction
  ld de,+PLAYER_BOX_HALF_WIDTH
  bit 7,b
  jr z,$+2+3
  ld de,-PLAYER_BOX_HALF_WIDTH
  add hl,de
  push de

  push hl
  ex de,hl
  ; Expand in Y by box width in up direction
  ld hl,(camera_y)
  ld bc,-PLAYER_BOX_HALF_WIDTH
  add hl,bc
  ex de,hl
  call check_position
  jr nz,check_x_move.end
  pop hl

  push hl
  ex de,hl
  ; Expand in Y by box width in down direction
  ld hl,(camera_y)
  ld bc,+PLAYER_BOX_HALF_WIDTH
  add hl,bc
  ex de,hl
  call check_position

check_x_move.end:
  pop hl
  pop de
  push af
  and a
  sbc hl,de ; Undo the expansion
  pop af
  ret

check_y_move:
  ; Expand in Y by box width by move direction
  ld de,+PLAYER_BOX_HALF_WIDTH
  bit 7,b
  jr z,$+2+3
  ld de,-PLAYER_BOX_HALF_WIDTH
  add hl,de
  push de

  push hl
  ex de,hl
  ; Expand in X by box width in up direction
  ld hl,(camera_x)
  ld bc,-PLAYER_BOX_HALF_WIDTH
  add hl,bc
  call check_position
  jr nz,check_y_move.end
  pop hl

  push hl
  ex de,hl
  ; Expand in X by box width in down direction
  ld hl,(camera_x)
  ld bc,+PLAYER_BOX_HALF_WIDTH
  add hl,bc
  call check_position

check_y_move.end:
  pop hl
  pop de
  push af
  and a
  sbc hl,de ; Undo the expansion
  pop af
  ret

up_pressed:
  push af

  ; Cos
  ld bc,(camera_angle)
  ld a,b
  inc a
  and 3
  ld b,a
  call calc_camera_sinus
  ld hl,(camera_vel_x)
  add hl,bc
  ld (camera_vel_x),hl

  ; Sin
  ld bc,(camera_angle)
  call calc_camera_sinus
  ld hl,(camera_vel_y)
  add hl,bc
  ld (camera_vel_y),hl

  pop af
  ret

down_pressed:
  push af

  ; Cos
  ld bc,(camera_angle)
  ld a,b
  inc a
  and 3
  ld b,a
  call calc_camera_sinus
  ; Negate BC
  ld a,c
  cpl
  ld c,a
  ld a,b
  cpl
  ld b,a
  inc bc
  ld hl,(camera_vel_x)
  add hl,bc
  ld (camera_vel_x),hl

  ; Sin
  ld bc,(camera_angle)
  call calc_camera_sinus
  ; Negate BC
  ld a,c
  cpl
  ld c,a
  ld a,b
  cpl
  ld b,a
  inc bc
  ld hl,(camera_vel_y)
  add hl,bc
  ld (camera_vel_y),hl

  pop af
  ret

left_pressed:
  push af
  push hl
  ld hl,(camera_angle_velocity)
  ld bc,+2*4
  add hl,bc
  ld (camera_angle_velocity),hl
  pop hl
  pop af
  ret

right_pressed:
  push af
  push hl
  ld hl,(camera_angle_velocity)
  ld bc,-2*4
  add hl,bc
  ld (camera_angle_velocity),hl
  pop hl
  pop af
  ret

move_camera_step:
  ld de,(camera_angle_velocity)
  ld hl,(camera_angle_highprec)
  add hl,de
  ld (camera_angle_highprec),hl

  REPT 3
    sra h
    rr l
  ENDM
  ld a,(NUM_ANGLE_UNITS-1)>>8
  and h
  ld h,a
  ld (camera_angle),hl

  ld l,e
  ld h,d
  ; Round to inf
  bit 7,h
  ld bc,7
  jr nz,$+2+1
  add hl,bc
  REPT 3
    sra h
    rr l
  ENDM
  ex de,hl
  and a
  sbc hl,de
  ld (camera_angle_velocity),hl

  ld bc,(camera_vel_x)
  ; Round to zero
  bit 7,b
  jr z,$+2+1
  inc bc
  sra b
  rr c
  ld hl,(camera_x)
  add hl,bc
  call check_x_move
  jr nz,$+2+3
  ld (camera_x),hl

  ld bc,(camera_vel_y)
  ; Round to zero
  bit 7,b
  jr z,$+2+1
  inc bc
  sra b
  rr c
  ld hl,(camera_y)
  add hl,bc
  call check_y_move
  jr nz,$+2+3
  ld (camera_y),hl

  ld hl,(camera_vel_x)
  bit 7,h
  ; Abs
  jr z,$+2+1+1+1+1+1+1+1
  ; Negate HL
  ld a,h  ;  5c
  cpl     ;  5c
  ld h,a  ;  5c
  ld a,l  ;  5c
  cpl     ;  5c
  ld l,a  ;  5c
  inc hl  ;  7c
  ex de,hl
  ld hl,(camera_vel_y)
  bit 7,h
  ; Abs
  jr z,$+2+1+1+1+1+1+1+1
  ; Negate HL
  ld a,h  ;  5c
  cpl     ;  5c
  ld h,a  ;  5c
  ld a,l  ;  5c
  cpl     ;  5c
  ld l,a  ;  5c
  inc hl  ;  7c
  add hl,de
  REPT 4
    sra l
  ENDM
  ld b,l
  inc b
  ld hl,bob_phase
  dec (hl)
  inc (hl)
  djnz $-1

  IF 0
  ld hl,(camera_vel_x)
  ; Round to zero
  bit 7,h
  jr z,$+2+1
  inc hl
  sra h
  rr l
  ld (camera_vel_x),hl

  ld hl,(camera_vel_y)
  ; Round to zero
  bit 7,h
  jr z,$+2+1
  inc hl
  sra h
  rr l
  ld (camera_vel_y),hl
  ENDIF

  ld hl,(camera_vel_x)
  ld e,l
  ld d,h
  ; Round to inf
  bit 7,h
  ld bc,7
  jr nz,$+2+1
  add hl,bc
  REPT 3
    sra h
    rr l
  ENDM
  ex de,hl
  and a
  sbc hl,de
  ld (camera_vel_x),hl

  ld hl,(camera_vel_y)
  ld e,l
  ld d,h
  ; Round to inf
  bit 7,h
  ld bc,7
  jr nz,$+2+1
  add hl,bc
  REPT 3
    sra h
    rr l
  ENDM
  ex de,hl
  and a
  sbc hl,de
  ld (camera_vel_y),hl

  ret

; ---------------------- V9990 Common ----------------------

v9990_setmode_B1_BP2_512x4096:
  ;     |    Register #6    |       Register #7       |
  ; MCS DSPM DCKM XIMM CLRM C25M SM1 SM PAL EO IL HSCN
  ;   0   10   00   01   00    0   0  0   0  0  0    0
  ld a,#00 ; MCS = 0
  out (V9P_SYSCTRL),a
  ld a,V9R_SCNMOD0
  out (V9P_REGSLCT),a
  ld a,%10000100
  out (V9P_REGDATA),a
  ld a,%00000000
  out (V9P_REGDATA),a
  ; Palette Control
  ld a,V9R_PALCTRL
  out (V9P_REGSLCT),a
  ld a,%00000000
  out (V9P_REGDATA),a
  ret

v9990_enable_display:
  ld a,V9R_CONTROL
  out (V9P_REGSLCT),a
  ld a,%11000010
  out (V9P_REGDATA),a
  ret

v9990_set_512k:
  ld a,V9R_CONTROL | %11000000
  out (V9P_REGSLCT),a
  in a,(V9P_REGDATA)
  and %11111000
  or  %00000010
  out (V9P_REGDATA),a
  ret

v9990_reset:
  ld a,#02
  out (V9P_SYSCTRL),a
  xor a
  out (V9P_SYSCTRL),a
  ld a,V9R_SCNMOD0
  out (V9P_REGSLCT),a
  ld a,%11000000 ; Stand-by mode
  out (V9P_REGDATA),a
  ret

v9990_set_border_colour:
  push af
  ld b,a
  ld a,V9R_BDRPCOL
  out (V9P_REGSLCT),a
  ld a,b
  out (V9P_REGDATA),a
  pop af
  ret


; ---------------------- Ramp Generation ----------------------

ramp_gen_x_inner:
  ex af,af'
  ld (iy+COMMAND_SX+0),h  ; SX L
  ld (iy+COMMAND_SX+1),a  ; SX H
  exx
  ex de,hl
  add hl,bc
  ex de,hl
  exx
  adc hl,bc
  adc a,0
  ex af,af'
  exx
  ld hl,ramp_gen_lmmm
  ld a,V9R_CMNDWR0
  out (V9P_REGSLCT),a
  ld c,V9P_REGDATA
  V9990_WAIT_FOR_COMMAND_READY
  REPT 14
    outi
  ENDM
  ld a,V9R_CMNDOPC
  out (V9P_REGSLCT),a
  ld a,V9O_LMMM<<4
  out (V9P_REGDATA),a
  exx
  inc (iy+COMMAND_DX+0)   ; DX L
  jr nz,$+2+3
  inc (iy+COMMAND_DX+1)   ; DX H
  dec d
  jr nz,ramp_gen_x_inner
  ret

ramp_gen_y_inner:
  ex af,af'
  ld (iy+COMMAND_SY+0),h  ; SY L
  ld (iy+COMMAND_SY+1),a  ; SY H
  exx
  ex de,hl
  add hl,bc
  ex de,hl
  exx
  adc hl,bc
  adc a,0
  ex af,af'
  exx
  ld hl,ramp_gen_lmmm
  ld a,V9R_CMNDWR0
  out (V9P_REGSLCT),a
  ld c,V9P_REGDATA
  V9990_WAIT_FOR_COMMAND_READY
  REPT 14
    outi
  ENDM
  ld a,V9R_CMNDOPC
  out (V9P_REGSLCT),a
  ld a,V9O_LMMM<<4
  out (V9P_REGDATA),a
  exx
  inc (iy+COMMAND_DY+0)   ; DY L
  jr nz,$+2+3
  inc (iy+COMMAND_DY+1)   ; DY H
  dec d
  jr nz,ramp_gen_y_inner
  ret

draw_ramp_image:
  exx
  ld l,(ix+RAMP_DESCRIPTOR_U+0) ; U
  ld h,(ix+RAMP_DESCRIPTOR_U+1)
  ld c,(ix+RAMP_DESCRIPTOR_DUDX+0) ; DUDX
  ld b,(ix+RAMP_DESCRIPTOR_DUDX+1)
  ld e,(ix+RAMP_DESCRIPTOR_DUDY+0) ; DUDY
  ld d,(ix+RAMP_DESCRIPTOR_DUDY+1)
  exx
  ld hl,ramp_image_buffer
  ld (hl),0
  ld d,8/FRAME_BPP
  ld e,RAMP_PIXEL_PALCOL
  ld c,(ix+RAMP_DESCRIPTOR_W) ; W
  ld b,(ix+RAMP_DESCRIPTOR_H) ; H
  inc b
draw_ramp_image.y:
  push bc
  exx
  push hl
  exx
  ld b,c ; W
  inc b
draw_ramp_image.x:
  REPT FRAME_BPP
    sla (hl)
  ENDM
  exx
  bit 7,h
  add hl,bc ; DUDX
  exx
  jr nz,$+2+1+1+1
  ld a,e
  or (hl)
  ld (hl),a
  dec d
  jr nz,$+2+2+1
  ld d,8/FRAME_BPP
  inc hl
  djnz draw_ramp_image.x
  exx
  pop hl
  add hl,de ; DUDY
  exx
  pop bc
  djnz draw_ramp_image.y
  ret

upload_ramp_image:
  ; Upload to VRAM
  V9990_WAIT_FOR_COMMAND_READY
  ld a,V9R_CMNDWR0+4
  out (V9P_REGSLCT),a
  ; Set DX
  ld a,(ix+RAMP_DESCRIPTOR_ATLAS_X+0)
  out (V9P_REGDATA),a ; R#36 DX L
  ld a,(ix+RAMP_DESCRIPTOR_ATLAS_X+1)
  out (V9P_REGDATA),a ; R#37 DX H
  ; Set DY
  ld a,(ix+RAMP_DESCRIPTOR_ATLAS_Y+0)
  out (V9P_REGDATA),a ; R#38 DY L
  ld a,(ix+RAMP_DESCRIPTOR_ATLAS_Y+1)
  out (V9P_REGDATA),a ; R#39 DY H
  ; Set NX
  ld c,(ix+RAMP_DESCRIPTOR_W) ; W
  ld b,0
  inc bc
  ld a,c
  out (V9P_REGDATA),a ; R#40 NX L
  ld a,b
  out (V9P_REGDATA),a ; R#41 NX H
  ; Set NY
  ld c,(ix+RAMP_DESCRIPTOR_H) ; H
  ld b,0
  inc bc
  ld a,c
  out (V9P_REGDATA),a ; R#42 NY L
  ld a,b
  out (V9P_REGDATA),a ; R#43 NY H
  ; Set OP
  ld a,V9R_CMNDOPC
  out (V9P_REGSLCT),a
  ld a,V9O_LMMC << 4
  out (V9P_REGDATA),a
  V9990_WAIT_FOR_TRANSFER_READY
  ld e,(ix+RAMP_DESCRIPTOR_LENGTH+0) ; num_bytes
  ld d,(ix+RAMP_DESCRIPTOR_LENGTH+1)
  ld hl,ramp_image_buffer
  ; "Unrolled" loop using OTIR
  ld c,V9P_COMDATA
  inc d
  ld b,e
  otir
  dec d
  jr nz,$-1-2
  ret

; ---------------------- Font and Strings ----------------------

upload_font:
  ; Upload to VRAM
  V9990_WAIT_FOR_COMMAND_READY
  ld a,V9R_CMNDWR0+4
  out (V9P_REGSLCT),a
  ; Set DX
  ld a,FONT_VRAM_LOCATION_X&#FF
  out (V9P_REGDATA),a ; R#36 DX L
  ld a,FONT_VRAM_LOCATION_X>>8
  out (V9P_REGDATA),a ; R#37 DX H
  ; Set DY
  ld a,FONT_VRAM_LOCATION_Y&#FF
  out (V9P_REGDATA),a ; R#38 DY L
  ld a,FONT_VRAM_LOCATION_Y>>8
  out (V9P_REGDATA),a ; R#39 DY H
  ; Set NX
  ld a,FONT_VRAM_SIZE_X&#FF
  out (V9P_REGDATA),a ; R#40 NX L
  ld a,FONT_VRAM_SIZE_X>>8
  out (V9P_REGDATA),a ; R#41 NX H
  ; Set NY
  ld a,FONT_VRAM_SIZE_Y&#FF
  out (V9P_REGDATA),a ; R#42 NY L
  ld a,FONT_VRAM_SIZE_Y>>8
  out (V9P_REGDATA),a ; R#43 NY H
  ; Set OP
  ld a,V9R_CMNDOPC
  out (V9P_REGSLCT),a
  ld a,V9O_LMMC << 4
  out (V9P_REGDATA),a
  V9990_WAIT_FOR_TRANSFER_READY
  ld de,FONT_VRAM_LENGTH ; num_bytes
  ld hl,font
  ; "Unrolled" loop using OTIR
  ld c,V9P_COMDATA
  inc d
  ld b,e
  otir
  dec d
  jr nz,$-1-2
  ret

; ---------------------- Portal Renderer ----------------------

; Expects:
;   STACK_FX0
;   STACK_FX1
;   STACK_FY0_LOWER
;   STACK_FY1_LOWER
;   STACK_FY0_UPPER
;   STACK_FY1_UPPER
;   STACK_SIZE+STACK_A0
;   STACK_SIZE+STACK_A1
enter_portal:
  ; Load the address of the first wall in the next room into IX
  ld l,(ix+WALL_PORTAL_ROOM+0)
  ld h,(ix+WALL_PORTAL_ROOM+1)
  exx
  ld l,(iy+STACK_CAMERA_X+0)
  ld h,(iy+STACK_CAMERA_X+1)
  ld e,(iy+STACK_CAMERA_Y+0)
  ld d,(iy+STACK_CAMERA_Y+1)
  exx

  ; If the wall is a portal:
  push iy
  push ix

  ld b,(hl) ; ROOM_NUM_WALLS
  inc hl

  ; Load the first wall index based on view direction quadrant
  ld a,(camera_angle+1)
  and 3
  rlca
  ld e,a
  ld d,0
  add hl,de

  ld a,(hl) ; ROOM_FIRST_WALL
  inc hl
  ld h,(hl)
  ld l,a
  REPT 3 ; WALL_SIZE
    add hl,hl
  ENDM
  ex de,hl
  ld ix,walls
  add ix,de

  ; Push the portal stack
  ld de,STACK_SIZE
  add iy,de
  exx
  ; Pass camera X by value
  ld (iy+STACK_CAMERA_X+0),l
  ld (iy+STACK_CAMERA_X+1),h
  ; Pass camera Y by value
  ld (iy+STACK_CAMERA_Y+0),e
  ld (iy+STACK_CAMERA_Y+1),d
  exx
  ; A0 and A1 are already set.
  call draw_room_through_portal
  pop ix
  pop iy
  ret

;
; Stack frame layout:
;
;   16  Camera X (Must be on stack because it can change when traversing across tiles)
;   16  Camera Y
;   16  Ray 0 angle
;   16  Ray 1 angle
;   16  Current wall address
;   16  Camera-relative wall X0
;   16  Camera-relative wall Y0
;   16  Camera-relative wall X1
;   16  Camera-relative wall Y1
;   16  log_adjacent
;    8  Ray 0 intersection test flags
;    8  Ray 1 intersection test flags
;    8  ramp_index
;    8  Projected X0 (These should be placed near the end for negative IY offsets)
;   16  Projected Y0
;    8  Projected X1
;   16  Projected Y1
;    8  Vertex 0 angle
;    8  Vertex 1 angle
;   16  Ray 0 opposite
;   16  Ray 1 opposite
;
; Some stuff can be stored on the "real" stack too, of course.
;

draw_main_view:

  xor a
  ld (global_portal_traversal_flags),a

  ld iy,portal_stack_dummy
  ld (iy+STACK_FX0),FRAME_WIDTH-1
  ld (iy+STACK_FX1),0

  ld iy,portal_stack

  ld hl,(camera_angle)
  ld bc,-ANGLE_UNIT_FOV/2
  add hl,bc
  ld a,(NUM_ANGLE_UNITS-1)>>8
  and h
  ld h,a
  ld (iy+STACK_A0+0),l
  ld (iy+STACK_A0+1),h

  ld hl,(camera_angle)
  ld bc,+ANGLE_UNIT_FOV/2-1
  add hl,bc
  ld a,(NUM_ANGLE_UNITS-1)>>8
  and h
  ld h,a
  ld (iy+STACK_A1+0),l
  ld (iy+STACK_A1+1),h

  ld hl,(camera_y)
  ; Divide by 16
  REPT 4
    sra h
    rr l
  ENDM
  ld a,4095>>8
  and h
  ld h,a
  ; Negate
  ld a,l
  cpl
  ld l,a
  ld a,h
  cpl
  ld h,a
  inc hl
  ld (iy+STACK_CAMERA_Y+0),l
  ld (iy+STACK_CAMERA_Y+1),h

  ld hl,(camera_x)
  ; Divide by 16
  REPT 4
    sra h
    rr l
  ENDM
  ld a,4095>>8
  and h
  ld h,a
  ; Negate
  ld a,l
  cpl
  ld l,a
  ld a,h
  cpl
  ld h,a
  inc hl
  ld (iy+STACK_CAMERA_X+0),l
  ld (iy+STACK_CAMERA_X+1),h

  ; Determine the starting room
  ; Divide by 16*128=2048=256*8
  ld hl,(camera_x)
  REPT 3
    sra h
  ENDM
  ld a,15
  and h
  ld h,a
  ex de,hl
  ld hl,(camera_y)
  ; Divide by 16*128=2048=256*8
  REPT 3
    sra h
  ENDM
  ld a,15
  and h
  REPT 4
    rlca
  ENDM
  or d
  ld l,a
  ld h,0
  add hl,hl
  ld bc,maze
  add hl,bc
  ; Load the address
  ld e,(hl)
  inc hl
  ld d,(hl)

  bit 7,d
  ret nz

  ex de,hl

  ld b,(hl) ; ROOM_NUM_WALLS
  inc hl

  ; Load the first wall index based on view direction quadrant
  ld a,(camera_angle+1)
  and 3
  rlca
  ld e,a
  ld d,0
  add hl,de

  ld a,(hl) ; ROOM_FIRST_WALL
  inc hl
  ld h,(hl)
  ld l,a
  REPT 3 ; WALL_SIZE
    add hl,hl
  ENDM
  ex de,hl
  ld ix,walls
  add ix,de

  call draw_room_through_portal

  ret

draw_room_through_portal:

  ld hl,portal_depth
  ld a,MAX_PORTAL_DEPTH
  cp (hl)
  ret z
  inc (hl)

  push bc

  ; Get ray intersection test flags
  ld bc,ray_intersection_test_table ; 11c

  ld l,(iy+STACK_A0+0)              ; 21c
  ld h,(iy+STACK_A0+1)              ; 21c
  add hl,bc                         ; 12c
  ld a,(hl)                         ;  8c
  ld (iy+STACK_ITESTFLAGS0),a       ; 21c

  ld l,(iy+STACK_A1+0)              ; 21c
  ld h,(iy+STACK_A1+1)              ; 21c
  add hl,bc                         ; 12c
  ld d,(hl)                         ;  8c
  ld (iy+STACK_ITESTFLAGS1),d       ; 21c

  ; Get bitwise AND of intersection test flags
  and d                             ;  5c
  ld (iy+STACK_ITESTFLAGS),a        ; 21c

  pop bc

  ; Calculate location in cameraspace of first vertex

  push ix

  ld l,b
  dec l
  ld h,0
  REPT 3
    add hl,hl
  ENDM
  ex de,hl
  add ix,de

  ld e,(iy+STACK_CAMERA_X+0)  ; 21c   Camera X (negated)
  ld d,(iy+STACK_CAMERA_X+1)  ; 21c

  ; Calculate x0
  ld l,(ix+WALL_X1+0)         ; 21c
  ld h,(ix+WALL_X1+1)         ; 21c
  add hl,de                   ; 12c
  ld (iy+STACK_X0+0),l        ; 21c
  ld (iy+STACK_X0+1),h        ; 21c

  ld e,(iy+STACK_CAMERA_Y+0)  ; 21c   Camera Y (negated)
  ld d,(iy+STACK_CAMERA_Y+1)  ; 21c

  ; Calculate y0
  ld l,(ix+WALL_Y1+0)         ; 21c
  ld h,(ix+WALL_Y1+1)         ; 21c
  add hl,de                   ; 12c
  ld (iy+STACK_Y0+0),l        ; 21c
  ld (iy+STACK_Y0+1),h        ; 21c

  pop ix

  ; Pre-load FX1 and FY1 from the outer portal
  ld a,(iy-STACK_SIZE+STACK_FX0)
  ld (iy+STACK_FX1),a
  ld a,(iy-STACK_SIZE+STACK_FY0_LOWER)
  ld (iy+STACK_FY1_LOWER),a
  ld a,(iy-STACK_SIZE+STACK_FY0_UPPER)
  ld (iy+STACK_FY1_UPPER),a

  ; Pre-load the inner A1 too
  ld a,(iy+STACK_A1+0)
  ld (iy+STACK_SIZE+STACK_A1+0),a
  ld a,(iy+STACK_A1+1)
  ld (iy+STACK_SIZE+STACK_A1+1),a

  ; Clear flags
  ld (iy+STACK_FLAGS),0

  ; Loop through the walls in this room
draw_room_through_portal.loop:
  push bc
  push ix

  ld e,(iy+STACK_CAMERA_X+0)  ; 21c   Camera X (negated)
  ld d,(iy+STACK_CAMERA_X+1)  ; 21c

  ; Calculate x1
  ld l,(ix+WALL_X1+0)         ; 21c
  ld h,(ix+WALL_X1+1)         ; 21c
  add hl,de                   ; 12c
  ld (iy+STACK_X1+0),l        ; 21c
  ld (iy+STACK_X1+1),h        ; 21c

  ld e,(iy+STACK_CAMERA_Y+0)  ; 21c   Camera Y (negated)
  ld d,(iy+STACK_CAMERA_Y+1)  ; 21c

  ; Calculate y1
  ld l,(ix+WALL_Y1+0)         ; 21c
  ld h,(ix+WALL_Y1+1)         ; 21c
  add hl,de                   ; 12c
  ld (iy+STACK_Y1+0),l        ; 21c
  ld (iy+STACK_Y1+1),h        ; 21c

  ; Draw a wall
  call draw_wall

  ; Copy x1 to x0
  ld l,(iy+STACK_X1+0)        ; 21c
  ld h,(iy+STACK_X1+1)        ; 21c
  ld (iy+STACK_X0+0),l        ; 21c
  ld (iy+STACK_X0+1),h        ; 21c

  ld a,(iy+STACK_FX1)
  ld (iy+STACK_FX0),a

  ; Copy y1 to y0
  ld l,(iy+STACK_Y1+0)        ; 21c
  ld h,(iy+STACK_Y1+1)        ; 21c
  ld (iy+STACK_Y0+0),l        ; 21c
  ld (iy+STACK_Y0+1),h        ; 21c

  ld a,(iy+STACK_FY1_LOWER)
  ld (iy+STACK_FY0_LOWER),a
  ld a,(iy+STACK_FY1_UPPER)
  ld (iy+STACK_FY0_UPPER),a

  pop ix
  pop bc

  bit 1,(iy+STACK_FLAGS)
  jr nz,draw_room_through_portal.early_out

  ld de,WALL_SIZE
  add ix,de
  djnz draw_room_through_portal.loop

draw_room_through_portal.early_out:
  ld hl,portal_depth
  dec (hl)

  ret             ; 11c

; IY = Portal traversal stack frame
; IX = Wall data base address
draw_wall:

  IF ENABLE_STAT_COUNTERS
    push af
    ld a,(stat_num_walls_drawn)
    inc a
    ld (stat_num_walls_drawn),a
    pop af
  ENDIF

  ; Call the routine corresponding to this wall's orientation
  ; The address of the routine is pre-stored in the wall data.
  ld l,(ix+WALL_ROUTINE+0)     ; 21c
  ld h,(ix+WALL_ROUTINE+1)     ; 21c
  jp (hl)         ;  5c

draw_wall_orientation_0_specialcase:
  bit 7,(ix+WALL_PORTAL_ROOM+1)
  ret nz

  ; Check that the camera is really inside the wall
  bit 7,(iy+STACK_X0+1)
  ret z
  bit 7,(iy+STACK_X1+1)
  ret nz

  ld a,(iy+STACK_A0+0)
  ld (iy+STACK_SIZE+STACK_A0+0),a
  ld a,(iy+STACK_A0+1)
  ld (iy+STACK_SIZE+STACK_A0+1),a

  ld a,(iy+STACK_A1+0)
  ld (iy+STACK_SIZE+STACK_A1+0),a
  ld a,(iy+STACK_A1+1)
  ld (iy+STACK_SIZE+STACK_A1+1),a

  ld (iy+STACK_FX0),FRAME_WIDTH-1
  ld (iy+STACK_FX1),0

  jp enter_portal

draw_wall_orientation_0:

  ; Apply frustum culling
  bit 0,(iy+STACK_ITESTFLAGS)
  ret nz

  ; Apply backface culling
  ld h,(iy+STACK_Y1+1)        ; 21c
  ld l,(iy+STACK_Y1+0)        ; 21c

  ; If the camera is exactly on this wall's line then go to
  ; the special case handling.
  ld a,h
  or l
  jr z,draw_wall_orientation_0_specialcase

  bit 7,h
  ret z

  ; Lookup log_adjacent
  ; hl ϵ (-4096, 0)
  ld (iy+STACK_ADJACENT+0),l  ; 21c
  ld (iy+STACK_ADJACENT+1),h  ; 21c
  ; Calculate abs(adjacent)
  ; Negate HL since it is known to be negative.
  ld a,h
  cpl
  ld h,a
  ld a,l
  cpl
  ld l,a
  inc hl
  ; hl ϵ (0, 4096)
  add hl,hl           ; 12c
  ld bc,log_table     ; 11c
  add hl,bc           ; 12c
  ld c,(hl)
  inc hl
  ld b,(hl)
  ; bc ϵ [0, 4096)
  ld (iy+STACK_LOG_ADJACENT+0),c         ; 21c
  ld (iy+STACK_LOG_ADJACENT+1),b         ; 21c

  DRAW_WALL_ORIENTATION_N 0,STACK_X0,STACK_X1,STACK_X0,STACK_X1

draw_wall_orientation_1:

  ; Apply frustum culling
  bit 1,(iy+STACK_ITESTFLAGS)
  ret nz

  ; Apply backface culling
  ld h,(iy+STACK_X1+1)        ; 21c
  bit 7,h
  ret nz

  ld l,(iy+STACK_X1+0)        ; 21c

  ; Lookup log_adjacent
  ; hl ϵ (-4096, 0)
  ld (iy+STACK_ADJACENT+0),l        ; 21c
  ld (iy+STACK_ADJACENT+1),h        ; 21c
  ; HL is known to be non-negative already.
  ; hl ϵ (0, 4096)
  add hl,hl           ; 12c
  ld bc,log_table     ; 11c
  add hl,bc           ; 12c
  ld c,(hl)
  inc hl
  ld b,(hl)
  ; bc ϵ [0, 4096)
  ld (iy+STACK_LOG_ADJACENT+0),c         ; 21c
  ld (iy+STACK_LOG_ADJACENT+1),b         ; 21c

  DRAW_WALL_ORIENTATION_N 1,STACK_Y0,STACK_Y1,STACK_Y0,STACK_Y1

draw_wall_orientation_2:

  ; Apply frustum culling
  bit 2,(iy+STACK_ITESTFLAGS)
  ret nz

  ; Apply backface culling
  ld h,(iy+STACK_Y1+1)        ; 21c
  bit 7,h
  ret nz

  ld l,(iy+STACK_Y1+0)        ; 21c

  ; Lookup log_adjacent
  ; hl ϵ [0, 4096)
  ld (iy+STACK_ADJACENT+0),l        ; 21c
  ld (iy+STACK_ADJACENT+1),h        ; 21c
  ; HL is known to be non-negative already.
  ; hl ϵ [0, 4096)
  add hl,hl           ; 12c
  ld bc,log_table     ; 11c
  add hl,bc           ; 12c
  ld c,(hl)
  inc hl
  ld b,(hl)
  ; bc ϵ [0, 4096)
  ld (iy+STACK_LOG_ADJACENT+0),c         ; 21c
  ld (iy+STACK_LOG_ADJACENT+1),b         ; 21c

  DRAW_WALL_ORIENTATION_N 2,STACK_X1,STACK_X0,STACK_X0,STACK_X1

draw_wall_orientation_3_specialcase:
  bit 7,(ix+WALL_PORTAL_ROOM+1)
  ret nz
  
  ; Check that the camera is really inside the wall
  bit 7,(iy+STACK_Y1+1)
  ret z
  bit 7,(iy+STACK_Y0+1)
  ret nz

  ld a,(iy+STACK_A0+0)
  ld (iy+STACK_SIZE+STACK_A0+0),a
  ld a,(iy+STACK_A0+1)
  ld (iy+STACK_SIZE+STACK_A0+1),a

  ld a,(iy+STACK_A1+0)
  ld (iy+STACK_SIZE+STACK_A1+0),a
  ld a,(iy+STACK_A1+1)
  ld (iy+STACK_SIZE+STACK_A1+1),a

  ld (iy+STACK_FX0),FRAME_WIDTH-1
  ld (iy+STACK_FX1),0

  jp enter_portal

draw_wall_orientation_3:

  ; Apply frustum culling
  bit 3,(iy+STACK_ITESTFLAGS)
  ret nz

  ; Apply backface culling
  ld h,(iy+STACK_X1+1)        ; 21c
  ld l,(iy+STACK_X1+0)        ; 21c

  ; If the camera is exactly on this wall's line then go to
  ; the special case handling.
  ld a,h
  or l
  jr z,draw_wall_orientation_3_specialcase

  bit 7,h
  ret z

  ; Lookup log_adjacent
  ; hl ϵ (-4096, 0)
  ld (iy+STACK_ADJACENT+0),l        ; 21c
  ld (iy+STACK_ADJACENT+1),h        ; 21c
  ; Calculate abs(adjacent)
  ; Negate HL since it is known to be negative.
  ld a,h
  cpl
  ld h,a
  ld a,l
  cpl
  ld l,a
  inc hl
  ; hl ϵ (0, 4096)
  add hl,hl           ; 12c
  ld bc,log_table     ; 11c
  add hl,bc           ; 12c
  ld c,(hl)
  inc hl
  ld b,(hl)
  ; bc ϵ [0, 4096)
  ld (iy+STACK_LOG_ADJACENT+0),c         ; 21c
  ld (iy+STACK_LOG_ADJACENT+1),b         ; 21c

  DRAW_WALL_ORIENTATION_N 3,STACK_Y1,STACK_Y0,STACK_Y0,STACK_Y1



; Reads:
;       STACK_LOG_ADJACENT
;       STACK_ADJACENT
;       STACK_ORIENTATION
;       STACK_LOG_OPPOSITE
;       STACK_OPPOSITE
; Writes:
;       STACK_FX1
;       STACK_FY1_UPPER
;       STACK_FY1_LOWER
project_post:

  IF ENABLE_STAT_COUNTERS
    push af
    ld a,(stat_num_project_posts)
    inc a
    ld (stat_num_project_posts),a
    pop af
  ENDIF

  ; X projection

  bit 0,(iy+STACK_ORIENTATION)
  jr nz,project_post.swap
  ; Lookup arctanlog_table
  ld c,(iy+STACK_LOG_OPPOSITE+0)
  ld b,(iy+STACK_LOG_OPPOSITE+1)
  ld l,(iy+STACK_LOG_ADJACENT+0)
  ld h,(iy+STACK_LOG_ADJACENT+1)
  ld e,(iy+STACK_OPPOSITE+1)
  ld d,(iy+STACK_ADJACENT+1)
  jr project_post.after_swap
project_post.swap:
  ; Swapped
  ld c,(iy+STACK_LOG_ADJACENT+0)
  ld b,(iy+STACK_LOG_ADJACENT+1)
  ld l,(iy+STACK_LOG_OPPOSITE+0)
  ld h,(iy+STACK_LOG_OPPOSITE+1)
  ld e,(iy+STACK_ADJACENT+1)
  ld d,(iy+STACK_OPPOSITE+1)
project_post.after_swap:

  push bc ; log_opposite will be used later.

  ; i=(log_adjacent-log_opposite+4096)/2
  and a
  sbc hl,bc
  ld bc,4096
  add hl,bc
  ld a,#FE
  and l
  ld l,a
  push hl ; This index will be re-used.
  ; Divide HL by two
  srl h
  rr l

  ; hl ϵ [0, +4096)
  ; Apply bounds restrictions
  ld a,h
  rrca
  xor h
  bit 2,a
  jr nz,project_post.clamp_end
  bit 2,h
  jr nz,project_post.upper_clamp
  ; Lower clamp
  ld hl,#0400
  jr project_post.clamp_end
project_post.upper_clamp:
  ; Upper clamp
  ld hl,#0C00
project_post.clamp_end:
  ; hl ϵ [1024, +3072]

  ld bc,-1024
  add hl,bc
  ld bc,arctanlog_table
  add hl,bc
  ; Load angle
  ld c,(hl)
  ld b,0
  ; bc ϵ [0, +256)

  ; Map the angle according to quadrant.
  bit 7,e
  jr z,project_post.opposite_nonneg
  ; If opposite < 0:
  bit 7,d
  jr z,project_post.adjacent_nonneg
  ;   If adjacent < 0:
  ;     angle=NUM_ANGLE_UNITS/2+angle
  ld hl,NUM_ANGLE_UNITS/2
  add hl,bc
  jr project_post.after_quadrants
  ;   Else
  ;     angle=NUM_ANGLE_UNITS/2-angle
project_post.adjacent_nonneg:
  ld hl,NUM_ANGLE_UNITS/2
  and a
  sbc hl,bc
  jr project_post.after_quadrants
  ;   Endif
  ; Else
project_post.opposite_nonneg:
  bit 7,d
  jr z,project_post.adjacent_nonneg2
  ;   If adjacent < 0:
  ;     angle=(NUM_ANGLE_UNITS-angle)&(NUM_ANGLE_UNITS-1)
  ld hl,NUM_ANGLE_UNITS
  and a
  sbc hl,bc
  jr project_post.after_quadrants
  ;   Else
project_post.adjacent_nonneg2:
  ld h,b
  ld l,c
  ;   Endif
  ; Endif
project_post.after_quadrants:

  ld a,(NUM_ANGLE_UNITS-1)>>8
  and h
  ld h,a

  ; hl ϵ [0, +1024)
  ; Store the angle in the next stack frame in case it is needed for
  ; portal traversal.
  ld (iy+STACK_SIZE+STACK_A1+0),l
  ld (iy+STACK_SIZE+STACK_A1+1),h

  ; angle = (angle - camera_angle) & (NUM_ANGLE_UNITS-1)
  ld bc,(camera_angle)
  and a
  sbc hl,bc
  ld a,(NUM_ANGLE_UNITS-1)>>8
  and h
  ld h,a

  ;if(angle > NUM_ANGLE_UNITS / 2)
  ;  abs_angle = NUM_ANGLE_UNITS - 1 - angle;
  bit 1,h
  jr z,project_post.abs_angle_no_remap
  ex de,hl
  ld hl,NUM_ANGLE_UNITS-1
  and a
  sbc hl,de
  ; hl ϵ [0, +256)
  ld d,l
  ; Lookup x_angle_table
  ld bc,x_angle_table
  add hl,bc
  ld a,(hl)
  cpl
  ld (iy+STACK_FX1),a
  jr project_post.after_abs_angle_remap
project_post.abs_angle_no_remap:
  ; hl ϵ [0, +256)
  ld d,l
  ; Lookup x_angle_table
  ld bc,x_angle_table
  add hl,bc
  ld a,(hl)
  ld (iy+STACK_FX1),a
project_post.after_abs_angle_remap:

  ; Y projection

  ; log(cos)
  ld a,d
  cpl
  ld h,0
  ld l,a
  inc hl
  add hl,hl
  ld de,logsin_table
  add hl,de
  ld e,(hl)
  inc hl
  ld d,(hl)

  ; log_partial_sum
  pop hl ; Re-use the index computed earlier.
  ld bc,log_partial_sum_table
  add hl,bc
  ld c,(hl)
  inc hl
  ld b,(hl)

  ; log(c) * 2 = log(wall_height) / log(tangent_max) * 2047 * 2
  ld hl,3342*2
  and a
  sbc hl,de
  and a
  sbc hl,bc

  pop bc ; log_opposite
  and a
  sbc hl,bc

  ; Bobbing
  ld bc,(log_bob)
  ld de,exp_table

  REPT 2,?I
    push hl
    ; Clamp
    ld a,h
    and #F8
    jr z,$+2+3
    ld hl,#0800
    ; Exponentiation
    IF ?I=0
      add hl,bc
    ELSE
      and a
      sbc hl,bc
    ENDIF
    add hl,hl
    add hl,de
    ld a,(hl)
    IF ?I=0
      ld (iy+STACK_FY1_UPPER),a
    ELSE
      ld (iy+STACK_FY1_LOWER),a
    ENDIF
    pop hl
  ENDM

  ret

project_post_clipped_a0:

  ; X projection

  bit 0,(iy+STACK_ORIENTATION)
  jr nz,project_post_clipped_a0.swap
  ld c,(iy+STACK_LOG_OPPOSITE+0)
  ld b,(iy+STACK_LOG_OPPOSITE+1)
  ld l,(iy+STACK_LOG_ADJACENT+0)
  ld h,(iy+STACK_LOG_ADJACENT+1)
  ld e,(iy+STACK_OPPOSITE+1)
  ld d,(iy+STACK_ADJACENT+1)
  jr project_post_clipped_a0.after_swap
project_post_clipped_a0.swap:
  ; Swapped
  ld c,(iy+STACK_LOG_ADJACENT+0)
  ld b,(iy+STACK_LOG_ADJACENT+1)
  ld l,(iy+STACK_LOG_OPPOSITE+0)
  ld h,(iy+STACK_LOG_OPPOSITE+1)
  ld e,(iy+STACK_ADJACENT+1)
  ld d,(iy+STACK_OPPOSITE+1)
project_post_clipped_a0.after_swap:

  push bc ; log_opposite will be used later.

  ; i=(log_adjacent-log_opposite+4096)/2
  and a
  sbc hl,bc
  ld bc,4096
  add hl,bc
  ld a,#FE
  and l
  ld l,a
  push hl ; This index will be re-used.

  ; Load angle
  ld l,(iy+STACK_A0+0)
  ld h,(iy+STACK_A0+1)

  ; hl ϵ [0, +1024)
  ; Store the angle in the next stack frame in case it is needed for
  ; portal traversal.
  ld (iy+STACK_SIZE+STACK_A0+0),l
  ld (iy+STACK_SIZE+STACK_A0+1),h

  ; angle = (angle - camera_angle) & (NUM_ANGLE_UNITS-1)
  ld bc,(camera_angle)
  and a
  sbc hl,bc
  ld a,(NUM_ANGLE_UNITS-1)>>8
  and h
  ld h,a

  ;if(angle > NUM_ANGLE_UNITS / 2)
  ;  abs_angle = NUM_ANGLE_UNITS - 1 - angle;
  bit 1,h
  jr z,project_post_clipped_a0.abs_angle_no_remap
  ex de,hl
  ld hl,NUM_ANGLE_UNITS-1
  and a
  sbc hl,de
  ; hl ϵ [0, +256)
  ld d,l
  jr project_post_clipped_a0.after_abs_angle_remap
project_post_clipped_a0.abs_angle_no_remap:
  ; hl ϵ [0, +256)
  ld d,l
project_post_clipped_a0.after_abs_angle_remap:

  PROJECT_POST_Y

  ret

project_post_clipped_a1:

  ; X projection

  bit 0,(iy+STACK_ORIENTATION)
  jr nz,project_post_clipped_a1.swap
  ld c,(iy+STACK_LOG_OPPOSITE+0)
  ld b,(iy+STACK_LOG_OPPOSITE+1)
  ld l,(iy+STACK_LOG_ADJACENT+0)
  ld h,(iy+STACK_LOG_ADJACENT+1)
  ld e,(iy+STACK_OPPOSITE+1)
  ld d,(iy+STACK_ADJACENT+1)
  jr project_post_clipped_a1.after_swap
project_post_clipped_a1.swap:
  ; Swapped
  ld c,(iy+STACK_LOG_ADJACENT+0)
  ld b,(iy+STACK_LOG_ADJACENT+1)
  ld l,(iy+STACK_LOG_OPPOSITE+0)
  ld h,(iy+STACK_LOG_OPPOSITE+1)
  ld e,(iy+STACK_ADJACENT+1)
  ld d,(iy+STACK_OPPOSITE+1)
project_post_clipped_a1.after_swap:

  push bc ; log_opposite will be used later.

  ; i=(log_adjacent-log_opposite+4096)/2
  and a
  sbc hl,bc
  ld bc,4096
  add hl,bc
  ld a,#FE
  and l
  ld l,a
  push hl ; This index will be re-used.

  ; Load angle
  ld l,(iy+STACK_A1+0)
  ld h,(iy+STACK_A1+1)

  ; hl ϵ [0, +1024)
  ; Store the angle in the next stack frame in case it is needed for
  ; portal traversal.
  ld (iy+STACK_SIZE+STACK_A1+0),l
  ld (iy+STACK_SIZE+STACK_A1+1),h

  ; Use FX1 from the outer portal
  ld a,(iy-STACK_SIZE+STACK_FX1)
  ld (iy+STACK_FX1),a

  ; angle = (angle - camera_angle) & (NUM_ANGLE_UNITS-1)
  ld bc,(camera_angle)
  and a
  sbc hl,bc
  ld a,(NUM_ANGLE_UNITS-1)>>8
  and h
  ld h,a

  ;if(angle > NUM_ANGLE_UNITS / 2)
  ;  abs_angle = NUM_ANGLE_UNITS - 1 - angle;
  bit 1,h
  jr z,project_post_clipped_a1.abs_angle_no_remap
  ex de,hl
  ld hl,NUM_ANGLE_UNITS-1
  and a
  sbc hl,de
  ; hl ϵ [0, +256)
  ld d,l
  jr project_post_clipped_a1.after_abs_angle_remap
project_post_clipped_a1.abs_angle_no_remap:
  ; hl ϵ [0, +256)
  ld d,l
project_post_clipped_a1.after_abs_angle_remap:

  PROJECT_POST_Y

  ret

; ---------------------- Command lists ----------------------

init_commandlists:
  ; Clear the commandlists
  ld hl,commandlists_start
  ld (hl),0
  ld de,commandlists_start+1
  ld bc,commandlists_end-commandlists_start-1
  ldir

  ; Initialise the commandlist pointers.
  ld hl,commandlist_1+1
  ld (commandlist_read_ptr),hl
  ld hl,commandlist_1+0
  ld (commandlist_read_length_ptr),hl
  ld (hl),0

  ld hl,commandlist_0+1
  ld (commandlist_write_ptr),hl
  ld hl,commandlist_0+0
  ld (commandlist_write_length_ptr),hl
  ld (hl),0

  ; Add 256 to DY in all of the commands in one commandlist.
  ld b,MAX_COMMANDS
  ld ix,commandlist_1+1
  ld de,COMMAND_SIZE
init_commandlists.loop:
  ld (ix+COMMAND_DY+1),1
  add ix,de
  djnz init_commandlists.loop

  xor a
  ld (buffer_swap),a

  ret

swap_commandlists:
  push hl
  push de
  ; Swap the read/write heads.
  ld de,(commandlist_read_ptr)
  ld hl,(commandlist_write_ptr)
  ld (commandlist_write_ptr),de
  ld (commandlist_read_ptr),hl
  ld de,(commandlist_read_length_ptr)
  ld hl,(commandlist_write_length_ptr)
  ld (commandlist_write_length_ptr),de
  ld (commandlist_read_length_ptr),hl
  ; "Swap" the framebuffer. There is no need to disable interrupts here,
  ; because there is a guarantee that no commands are still being executed.
  ; Set Scroll Registers
  ld hl,buffer_swap
  ld a,V9R_SCROLL1
  out (V9P_REGSLCT),a
  ld a,(hl)
  xor 1
  out (V9P_REGDATA),a ; R#18 SCAY (bit 12-8)
  ld (hl),a
  pop de
  pop hl
  ret

execute_commandlist:
  ; Execute a commandlist
  ld hl,(commandlist_read_length_ptr)
execute_commandlist_loop:
  exx
  ld hl,(commandlist_read_ptr)
  ; Put the read head at the start address of the next command
  ld bc,-COMMAND_SIZE
  add hl,bc
  ld (commandlist_read_ptr),hl
  ; Execute a command
  ld a,V9R_CMNDWR0
  out (V9P_REGSLCT),a
  ld c,V9P_REGDATA
  V9990_WAIT_FOR_COMMAND_READY
  REPT COMMAND_SIZE-1
    outi
  ENDM
  ld a,V9R_CMNDOPC
  out (V9P_REGSLCT),a
  outi ; OP-CODE
  exx
  dec (hl)
  jr nz,execute_commandlist_loop
  ret

generate_wall_commands:
  ld a,(iy+STACK_FY0_UPPER)
  cp FRAME_HEIGHT/2-1
  jr c,$+2+4
  ld (iy+STACK_FY0_UPPER),FRAME_HEIGHT/2-1

  ld a,(iy+STACK_FY0_LOWER)
  cp FRAME_HEIGHT/2-1
  jr c,$+2+4
  ld (iy+STACK_FY0_LOWER),FRAME_HEIGHT/2-1

  ld a,(iy+STACK_FY1_UPPER)
  cp FRAME_HEIGHT/2-1
  jr c,$+2+4
  ld (iy+STACK_FY1_UPPER),FRAME_HEIGHT/2-1

  ld a,(iy+STACK_FY1_LOWER)
  cp FRAME_HEIGHT/2-1
  jr c,$+2+4
  ld (iy+STACK_FY1_LOWER),FRAME_HEIGHT/2-1


  bit 1,(iy+STACK_RELATIVE_ANGLE+1)
  jr nz,generate_wall_commands_R2L
  jp generate_wall_commands_L2R

; Determining LOP value from wall colour:
;
;                SD  SD  SD  SD
;               L11 L10 L01 L00
;    Colour 0     0   0   0   0   LOP = %00010000
;    Colour 1     0   0   1   1   LOP = %00010011
;    Colour 2     1   1   0   0   LOP = %00011100
;    Colour 3     1   1   1   1   LOP = %00011111


; Right-to-left
generate_wall_commands_R2L:
  push ix
  ld ix,(commandlist_write_ptr)

  ; Look up the ramp descriptor address
  ld l,(iy+STACK_RELATIVE_ANGLE+0)
  ld h,(iy+STACK_RELATIVE_ANGLE+1)
  ; hl ϵ [-65536, +65535)
  ld a,(NUM_ANGLE_UNITS/2-1)>>8
  and h
  ld h,a
  ; hl ϵ [0, +512)
  ; Multiply HL by two
  add hl,hl
  ld bc,logsin_table ; Range is [0, +512)
  add hl,bc
  ld e,(hl)
  inc hl
  ld d,(hl)
  ld l,(iy+STACK_LOG_ADJACENT+0)
  ld h,(iy+STACK_LOG_ADJACENT+1)
  ex de,hl
  and a
  sbc hl,de

  push hl

  ; Append an LMMM command to draw the upper
  ; part of the wall.
  ld a,(iy+STACK_FY0_UPPER)
  cp FRAME_HEIGHT/2
  call c,append_lmmm_command_for_wall_R2L_upper

  ; If fy0_lower + fy0_upper > 0 then append an LMMV command to fill in the lower
  ; part of the wall.
  ld a,(iy+STACK_FY0_UPPER)
  add a,(iy+STACK_FY0_LOWER)
  and a
  call nz,append_lmmv_command_for_wall_R2L_middle

  pop hl

  ; Append an LMMM command to draw the lower
  ; part of the wall.
  ld a,(iy+STACK_FY0_LOWER)
  cp FRAME_HEIGHT/2
  call c,append_lmmm_command_for_wall_R2L_lower

  ld (commandlist_write_ptr),ix
  pop ix
  ret


; Right-to-left
append_lmmm_command_for_wall_R2L_upper:
  ; Append an LMMM command to fill in the upper-right
  ; part of the wall.

  ld a,(iy+STACK_FY0_UPPER)
  cp (iy+STACK_FY1_UPPER)
  ret nc

  ; Get the ramp descriptor address
  ld de,rampdesc_addr_table_R2L ; Range is [0, +1024)
  ; log(quotient) = log(sin) - log(adjacent)
  ; Bobbing
  ld bc,(log_bob)
  add hl,bc
  ; hl ϵ [-4096, +4096)
  ; Clamp HL to zero
  bit 7,h
  jr z,$+2+3
  ld hl,#0000
  ; hl ϵ [0, +4096)
  ; Divide HL by two
  srl h
  rr l
  ; hl ϵ [0, +2048)
  ; Align HL to words
  ld a,#FE    ;  8c
  and l       ;  5c
  ld l,a      ;  5c
  add hl,de
  ; Load the ramp descriptor address from the table
  ld c,(hl)
  inc hl
  ld b,(hl)

  ld hl,#0000
  ld (wall_upper_rectangular_part_width),hl

  ld a,(iy+STACK_FX0)
  ld l,(iy+STACK_FX1)
  sub l
  ld l,a
  ld h,0

  ld (ix+COMMAND_NX+0),l   ; NX L
  ld (ix+COMMAND_NX+1),h   ; NX H

  push iy
  ld iyl,c
  ld iyh,b

  ; Load and/or clip NX
  ld c,(iy+RAMP_DESCRIPTOR_W)
  ld b,0
  inc bc
  and a
  sbc hl,bc
  jr c,append_lmmm_command_for_wall_R2L_upper.xclipped
  ld (ix+COMMAND_NX+0),c   ; NX L
  ld (ix+COMMAND_NX+1),b   ; NX H
  ; Store (positive) width of rectangular upper part
  ld (wall_upper_rectangular_part_width),hl
append_lmmm_command_for_wall_R2L_upper.xclipped:

  ; SX = ATLAS_X + W
  ld l,(iy+RAMP_DESCRIPTOR_ATLAS_X+0)
  ld h,(iy+RAMP_DESCRIPTOR_ATLAS_X+1)
  ld c,(iy+RAMP_DESCRIPTOR_W)
  ld b,0
  add hl,bc
  ld (ix+COMMAND_SX+0),l
  ld (ix+COMMAND_SX+1),h

  ; SY = ATLAS_Y + H
  ld l,(iy+RAMP_DESCRIPTOR_ATLAS_Y+0)
  ld h,(iy+RAMP_DESCRIPTOR_ATLAS_Y+1)
  ld c,(iy+RAMP_DESCRIPTOR_H)
  ld b,0
  add hl,bc
  ld (ix+COMMAND_SY+0),l
  ld (ix+COMMAND_SY+1),h

  ; Load and/or clip NY
  ld c,(iy+RAMP_DESCRIPTOR_H)
  ld b,0
  inc bc

  pop iy

  ld a,(iy+STACK_FY1_UPPER)
  ld l,(iy+STACK_FY0_UPPER)
  sub l
  ld l,a
  ld h,0

  ld e,l
  ld d,h
  and a
  sbc hl,bc
  jr nc,append_lmmm_command_for_wall_R2L_upper.skip_yclip
  ld c,e
  ld b,d
append_lmmm_command_for_wall_R2L_upper.skip_yclip:
  ld (ix+COMMAND_NY+0),c   ; NY L
  ld (ix+COMMAND_NY+1),b   ; NY H

  ; DX = fx0 - 1
  ld a,(iy+STACK_FX0)
  dec a
  ld (ix+COMMAND_DX+0),a   ; DX L
  ; DY = FRAME_HEIGHT/2 - fy0
  ld a,FRAME_HEIGHT/2
  ld b,(iy+STACK_FY0_UPPER)
  sub b
  ld (ix+COMMAND_DY+0),a   ; DY L

  ld (ix+COMMAND_ARG),%00001100  ; ARG, DIY=1, DIX=1

  ld a,(wall_shade+0)
  ld (ix+COMMAND_LOP),a       ; LOP
  ld (ix+COMMAND_OPCODE),V9O_LMMM<<4  ; OP-CODE

  ld bc,COMMAND_SIZE
  add ix,bc
  ld hl,(commandlist_write_length_ptr)
  inc (hl)

  IF ENABLE_STAT_COUNTERS
    push af
    ld a,(stat_num_commands_written)
    inc a
    ld (stat_num_commands_written),a
    pop af
  ENDIF

  ld hl,(wall_upper_rectangular_part_width)
  ld a,h
  or l
  jr nz,append_lmmv_command_for_wall_R2L_upper_rectangular

  ret

append_lmmv_command_for_wall_R2L_upper_rectangular:
  ; LMMV

  ; DX = fx1
  ld a,(iy+STACK_FX1)
  ld (ix+COMMAND_DX+0),a   ; DX L

  ; DY = 0
  ld (ix+COMMAND_DY+0),0   ; DY L

  ; NX = wall_upper_rectangular_part_width
  ld hl,(wall_upper_rectangular_part_width)
  ld (ix+COMMAND_NX+0),l   ; NX L
  ld (ix+COMMAND_NX+1),h   ; NX H

  ; NY = FRAME_HEIGHT/2
  ld (ix+COMMAND_NY+0),FRAME_HEIGHT/2   ; NY L
  ld (ix+COMMAND_NY+1),0   ; NY H

  ld (ix+COMMAND_ARG),%00000000  ; ARG, DIY=0, DIX=0

  ld a,(wall_shade+1)
  ld (ix+COMMAND_LOP),a       ; LOP
  ld (ix+COMMAND_OPCODE),V9O_LMMV<<4  ; OP-CODE

  ld bc,COMMAND_SIZE
  add ix,bc
  ld hl,(commandlist_write_length_ptr)
  inc (hl)

  IF ENABLE_STAT_COUNTERS
    push af
    ld a,(stat_num_commands_written)
    inc a
    ld (stat_num_commands_written),a
    pop af
  ENDIF

  ret


; Right-to-left
append_lmmv_command_for_wall_R2L_middle:

  ; LMMV

  ; DX = fx0 - 1
  ld a,(iy+STACK_FX0)
  dec a
  ld (ix+COMMAND_DX+0),a   ; DX L

  ; DY = FRAME_HEIGHT/2 + fy0_lower
  ld a,(iy+STACK_FY0_LOWER)
  add a,FRAME_HEIGHT/2
  ld (ix+COMMAND_DY+0),a   ; DY L

  ; NX = fx0 - fx1
  ld a,(iy+STACK_FX0)
  ld b,(iy+STACK_FX1)
  sub b
  ld (ix+COMMAND_NX+0),a   ; NX L

  ; NY = fy0_upper + fy0_lower (should be approximately constant)
  ld a,(iy+STACK_FY0_UPPER)
  add a,(iy+STACK_FY0_LOWER)
  ld (ix+COMMAND_NY+0),a   ; NY L

  ld (ix+COMMAND_ARG),%00001100  ; ARG, DIY=1, DIX=1

  ld a,(wall_shade+1)
  ld (ix+COMMAND_LOP),a       ; LOP
  ld (ix+COMMAND_OPCODE),V9O_LMMV<<4  ; OP-CODE

  ld bc,COMMAND_SIZE
  add ix,bc
  ld hl,(commandlist_write_length_ptr)
  inc (hl)

  IF ENABLE_STAT_COUNTERS
    push af
    ld a,(stat_num_commands_written)
    inc a
    ld (stat_num_commands_written),a
    pop af
  ENDIF

  ret


; Right-to-left
append_lmmm_command_for_wall_R2L_lower:
  ; Append an LMMM command to fill in the lower-right
  ; part of the wall.

  ld a,(iy+STACK_FY0_LOWER)
  cp (iy+STACK_FY1_LOWER)
  ret nc

  ; Get the ramp descriptor address
  ld de,rampdesc_addr_table_L2R ; Range is [0, +1024)
  ; log(quotient) = log(sin) - log(adjacent)

  ; Bobbing
  ld bc,(log_bob)
  and a
  sbc hl,bc
  ; hl ϵ [-4096, +4096)
  ; Clamp HL to zero
  bit 7,h
  jr z,$+2+3
  ld hl,#0000
  ; hl ϵ [0, +4096)
  ; Divide HL by two
  srl h
  rr l
  ; hl ϵ [0, +2048)
  ; Align HL to words
  ld a,#FE    ;  8c
  and l       ;  5c
  ld l,a      ;  5c
  add hl,de
  ; Load the ramp descriptor address from the table
  ld c,(hl)
  inc hl
  ld b,(hl)

  ld hl,#0000
  ld (wall_lower_rectangular_part_width),hl

  ld a,(iy+STACK_FX0)
  ld l,(iy+STACK_FX1)
  sub l
  ld l,a
  ld h,0

  ld (ix+COMMAND_NX+0),l   ; NX L
  ld (ix+COMMAND_NX+1),h   ; NX H

  push iy
  ld iyl,c
  ld iyh,b

  ; Load and/or clip NX
  ld c,(iy+RAMP_DESCRIPTOR_W)
  ld b,0
  inc bc
  and a
  sbc hl,bc
  jr c,append_lmmm_command_for_wall_R2L_lower.xclipped
  ld (ix+COMMAND_NX+0),c   ; NX L
  ld (ix+COMMAND_NX+1),b   ; NX H
  ; Store (positive) width of rectangular upper part
  ld (wall_lower_rectangular_part_width),hl
append_lmmm_command_for_wall_R2L_lower.xclipped:

  ; SX = ATLAS_X + W
  ld l,(iy+RAMP_DESCRIPTOR_ATLAS_X+0)
  ld h,(iy+RAMP_DESCRIPTOR_ATLAS_X+1)
  ld c,(iy+RAMP_DESCRIPTOR_W)
  ld b,0
  add hl,bc
  ld (ix+COMMAND_SX+0),l
  ld (ix+COMMAND_SX+1),h

  ; SY = ATLAS_Y
  ld l,(iy+RAMP_DESCRIPTOR_ATLAS_Y+0)
  ld h,(iy+RAMP_DESCRIPTOR_ATLAS_Y+1)
  ld (ix+COMMAND_SY+0),l
  ld (ix+COMMAND_SY+1),h

  ; Load and/or clip NY
  ld c,(iy+RAMP_DESCRIPTOR_H)
  ld b,0
  inc bc

  pop iy

  ld a,(iy+STACK_FY1_LOWER)
  ld l,(iy+STACK_FY0_LOWER)
  sub l
  ld l,a
  ld h,0

  ld e,l
  ld d,h
  and a
  sbc hl,bc
  jr nc,append_lmmm_command_for_wall_R2L_lower.skip_yclip
  ld c,e
  ld b,d
append_lmmm_command_for_wall_R2L_lower.skip_yclip:
  ld (ix+COMMAND_NY+0),c   ; NY L
  ld (ix+COMMAND_NY+1),b   ; NY H

  ; DX = fx0 - 1
  ld a,(iy+STACK_FX0)
  dec a
  ld (ix+COMMAND_DX+0),a   ; DX L
  ; DY = FRAME_HEIGHT/2 + fy0_lower
  ld a,FRAME_HEIGHT/2
  add a,(iy+STACK_FY0_LOWER)
  ld (ix+COMMAND_DY+0),a   ; DY L

  ld (ix+COMMAND_ARG),%00000100  ; ARG, DIY=0, DIX=1

  ld a,(wall_shade+2)
  ld (ix+COMMAND_LOP),a       ; LOP
  ld (ix+COMMAND_OPCODE),V9O_LMMM<<4  ; OP-CODE

  ld bc,COMMAND_SIZE
  add ix,bc
  ld hl,(commandlist_write_length_ptr)
  inc (hl)

  IF ENABLE_STAT_COUNTERS
    push af
    ld a,(stat_num_commands_written)
    inc a
    ld (stat_num_commands_written),a
    pop af
  ENDIF

  ld hl,(wall_lower_rectangular_part_width)
  ld a,h
  or l
  jr nz,append_lmmv_command_for_wall_R2L_lower_rectangular

  ret

append_lmmv_command_for_wall_R2L_lower_rectangular:
  ; LMMV

  ; DX = fx1
  ld a,(iy+STACK_FX1)
  ld (ix+COMMAND_DX+0),a   ; DX L

  ; DY = FRAME_HEIGHT - 1
  ld (ix+COMMAND_DY+0),FRAME_HEIGHT-1   ; DY L

  ; NX = wall_lower_rectangular_part_width
  ld hl,(wall_lower_rectangular_part_width)
  ld (ix+COMMAND_NX+0),l   ; NX L
  ld (ix+COMMAND_NX+1),h   ; NX H

  ; NY = FRAME_HEIGHT/2
  ld (ix+COMMAND_NY+0),FRAME_HEIGHT/2   ; NY L
  ld (ix+COMMAND_NY+1),0   ; NY H

  ld (ix+COMMAND_ARG),%00001000  ; ARG, DIY=1, DIX=0

  ld a,(wall_shade+1)
  ld (ix+COMMAND_LOP),a       ; LOP
  ld (ix+COMMAND_OPCODE),V9O_LMMV<<4  ; OP-CODE

  ld bc,COMMAND_SIZE
  add ix,bc
  ld hl,(commandlist_write_length_ptr)
  inc (hl)

  IF ENABLE_STAT_COUNTERS
    push af
    ld a,(stat_num_commands_written)
    inc a
    ld (stat_num_commands_written),a
    pop af
  ENDIF

  ret

append_hud_commands:
  push ix
  ld ix,(commandlist_write_ptr)

  ld hl,FONT_VRAM_LOCATION_X
  ld (ix+COMMAND_SX+0),l   ; SX L
  ld (ix+COMMAND_SX+1),h   ; SX H

  ld hl,FONT_VRAM_LOCATION_Y
  ld (ix+COMMAND_SY+0),l   ; SY L
  ld (ix+COMMAND_SY+1),h   ; SY H

  ld hl,FONT_VRAM_SIZE_X
  ld (ix+COMMAND_NX+0),l   ; NX L
  ld (ix+COMMAND_NX+1),h   ; NX H

  ld hl,FONT_VRAM_SIZE_Y
  ld (ix+COMMAND_NY+0),l   ; NY L
  ld (ix+COMMAND_NY+1),h   ; NY H

  ld a,3*8/2
  ld (ix+COMMAND_DX+0),a   ; DX L

  ld a,4
  ld (ix+COMMAND_DY+0),a   ; DY L

  ld (ix+COMMAND_ARG),%00000000  ; ARG, DIY=0, DIX=0
  ld (ix+COMMAND_LOP),%00010011       ; LOP - Invert
  ld (ix+COMMAND_OPCODE),V9O_LMMM<<4  ; OP-CODE

  ld bc,COMMAND_SIZE
  add ix,bc
  ld hl,(commandlist_write_length_ptr)
  inc (hl)

  IF ENABLE_STAT_COUNTERS
    push af
    ld a,(stat_num_commands_written)
    inc a
    ld (stat_num_commands_written),a
    pop af
  ENDIF

  ld (commandlist_write_ptr),ix
  pop ix
  ret

; Left-to-right
generate_wall_commands_L2R:
  push ix
  ld ix,(commandlist_write_ptr)

  ; Look up the ramp descriptor address
  ld l,(iy+STACK_RELATIVE_ANGLE+0)
  ld h,(iy+STACK_RELATIVE_ANGLE+1)
  ; hl ϵ [-65536, +65535)
  ld a,(NUM_ANGLE_UNITS/2-1)>>8
  and h
  ld h,a
  ; hl ϵ [0, +512)
  ; Multiply HL by two
  add hl,hl
  ld bc,logsin_table ; Range is [0, +512)
  add hl,bc
  ld e,(hl)
  inc hl
  ld d,(hl)
  ld l,(iy+STACK_LOG_ADJACENT+0)
  ld h,(iy+STACK_LOG_ADJACENT+1)
  ex de,hl
  and a
  sbc hl,de

  push hl

  ; If fy1 < FRAME_HEIGHT/2 then append an LMMM command to draw the upper
  ; part of the wall.
  ld a,(iy+STACK_FY1_UPPER)
  cp FRAME_HEIGHT/2
  call c,append_lmmm_command_for_wall_L2R_upper

  ; If fy1 > 0 then append an LMMV command to fill in the lower
  ; part of the wall.
  ld a,(iy+STACK_FY1_UPPER)
  and a
  call nz,append_lmmv_command_for_wall_L2R_middle

  pop hl

  ; Append an LMMM command to draw the lower
  ; part of the wall.
  ld a,(iy+STACK_FY1_LOWER)
  cp FRAME_HEIGHT/2
  call c,append_lmmm_command_for_wall_L2R_lower

  ld (commandlist_write_ptr),ix
  pop ix
  ret


; Left-to-right
append_lmmm_command_for_wall_L2R_upper:
  ; Append an LMMM command to fill in the upper-left
  ; part of the wall.

  ld a,(iy+STACK_FY1_UPPER)
  cp (iy+STACK_FY0_UPPER)
  ret nc

  ; Get the ramp descriptor address
  ld de,rampdesc_addr_table_L2R ; Range is [0, +1024)
  ; log(quotient) = log(sin) - log(adjacent)
  ; Bobbing
  ld bc,(log_bob)
  add hl,bc
  ; hl ϵ [-4096, +4096)
  ; Clamp HL to zero
  bit 7,h
  jr z,$+2+3
  ld hl,#0000
  ; hl ϵ [0, +4096)
  ; Divide HL by two
  srl h
  rr l
  ; hl ϵ [0, +2048)
  ; Align HL to words
  ld a,#FE    ;  8c
  and l       ;  5c
  ld l,a      ;  5c
  add hl,de
  ; Load the ramp descriptor address from the table
  ld c,(hl)
  inc hl
  ld b,(hl)

  ld hl,#0000
  ld (wall_upper_rectangular_part_width),hl

  ld a,(iy+STACK_FX0)
  ld l,(iy+STACK_FX1)
  sub l
  ld l,a
  ld h,0

  ld (ix+COMMAND_NX+0),l   ; NX L
  ld (ix+COMMAND_NX+1),h   ; NX H

  push iy
  ld iyl,c
  ld iyh,b

  ; Load and/or clip NX
  ld c,(iy+RAMP_DESCRIPTOR_W)
  ld b,0
  inc bc
  and a
  sbc hl,bc
  jr c,append_lmmm_command_for_wall_L2R_upper.xclipped
  ld (ix+COMMAND_NX+0),c   ; NX L
  ld (ix+COMMAND_NX+1),b   ; NX H
  ; Store (positive) width of rectangular upper part
  ld (wall_upper_rectangular_part_width),hl
append_lmmm_command_for_wall_L2R_upper.xclipped:

  ; Copy SX from the ramp descriptor
  ld a,(iy+RAMP_DESCRIPTOR_ATLAS_X+0)
  ld (ix+COMMAND_SX+0),a
  ld a,(iy+RAMP_DESCRIPTOR_ATLAS_X+1)
  ld (ix+COMMAND_SX+1),a

  ; SY = ATLAS_Y + H
  ld l,(iy+RAMP_DESCRIPTOR_ATLAS_Y+0)
  ld h,(iy+RAMP_DESCRIPTOR_ATLAS_Y+1)
  ld c,(iy+RAMP_DESCRIPTOR_H)
  ld b,0
  add hl,bc
  ld (ix+COMMAND_SY+0),l
  ld (ix+COMMAND_SY+1),h

  ; Load and/or clip NY
  ld c,(iy+RAMP_DESCRIPTOR_H)
  ld b,0
  inc bc

  pop iy

  ld a,(iy+STACK_FY0_UPPER)
  ld l,(iy+STACK_FY1_UPPER)
  sub l
  ld l,a
  ld h,0

  ld e,l
  ld d,h
  and a
  sbc hl,bc
  jr nc,append_lmmm_command_for_wall_L2R_upper.skip_yclip
  ld c,e
  ld b,d
append_lmmm_command_for_wall_L2R_upper.skip_yclip:
  ld (ix+COMMAND_NY+0),c   ; NY L
  ld (ix+COMMAND_NY+1),b   ; NY H

  ; DX = fx1
  ld a,(iy+STACK_FX1)
  ld (ix+COMMAND_DX+0),a   ; DX L
  ; DY = FRAME_HEIGHT/2 - fy1
  ld a,FRAME_HEIGHT/2
  ld b,(iy+STACK_FY1_UPPER)
  sub b
  ld (ix+COMMAND_DY+0),a   ; DY L

  ld (ix+COMMAND_ARG),%00001000  ; ARG, DIY=1, DIX=0

  ld a,(wall_shade+0)
  ld (ix+COMMAND_LOP),a       ; LOP
  ld (ix+COMMAND_OPCODE),V9O_LMMM<<4  ; OP-CODE

  ld bc,COMMAND_SIZE
  add ix,bc
  ld hl,(commandlist_write_length_ptr)
  inc (hl)

  IF ENABLE_STAT_COUNTERS
    push af
    ld a,(stat_num_commands_written)
    inc a
    ld (stat_num_commands_written),a
    pop af
  ENDIF

  ld hl,(wall_upper_rectangular_part_width)
  ld a,h
  or l
  jr nz,append_lmmv_command_for_wall_L2R_upper_rectangular

  ret

append_lmmv_command_for_wall_L2R_upper_rectangular:
  ; LMMV

  ; DX = fx0 - 1
  ld a,(iy+STACK_FX0)
  dec a
  ld (ix+COMMAND_DX+0),a   ; DX L

  ; DY = 0
  ld (ix+COMMAND_DY+0),0   ; DY L

  ; NX = wall_upper_rectangular_part_width
  ld hl,(wall_upper_rectangular_part_width)
  ld (ix+COMMAND_NX+0),l   ; NX L
  ld (ix+COMMAND_NX+1),h   ; NX H

  ; NY = FRAME_HEIGHT/2
  ld (ix+COMMAND_NY+0),FRAME_HEIGHT/2   ; NY L
  ld (ix+COMMAND_NY+1),0   ; NY H

  ld (ix+COMMAND_ARG),%00000100  ; ARG, DIY=0, DIX=1

  ld a,(wall_shade+1)
  ld (ix+COMMAND_LOP),a       ; LOP
  ld (ix+COMMAND_OPCODE),V9O_LMMV<<4  ; OP-CODE

  ld bc,COMMAND_SIZE
  add ix,bc
  ld hl,(commandlist_write_length_ptr)
  inc (hl)

  IF ENABLE_STAT_COUNTERS
    push af
    ld a,(stat_num_commands_written)
    inc a
    ld (stat_num_commands_written),a
    pop af
  ENDIF

  ret

; Left-to-right
append_lmmv_command_for_wall_L2R_middle:

  ; LMMV

  ; DX = fx1
  ld a,(iy+STACK_FX1)
  ld (ix+COMMAND_DX+0),a   ; DX L

  ; DY = FRAME_HEIGHT/2 + fy1_lower
  ld a,(iy+STACK_FY1_LOWER)
  add a,FRAME_HEIGHT/2
  ld (ix+COMMAND_DY+0),a   ; DY L

  ; NX = fx0 - fx1
  ld a,(iy+STACK_FX0)
  ld b,(iy+STACK_FX1)
  sub b
  ld (ix+COMMAND_NX+0),a   ; NX L

  ; NY = fy1_upper + fy1_lower (should be approximately constant)
  ld a,(iy+STACK_FY1_UPPER)
  add a,(iy+STACK_FY1_LOWER)
  ld (ix+COMMAND_NY+0),a   ; NY L

  ld (ix+COMMAND_ARG),%00001000  ; ARG, DIY=1, DIX=0

  ld a,(wall_shade+1)
  ld (ix+COMMAND_LOP),a       ; LOP
  ld (ix+COMMAND_OPCODE),V9O_LMMV<<4  ; OP-CODE

  ld bc,COMMAND_SIZE
  add ix,bc
  ld hl,(commandlist_write_length_ptr)
  inc (hl)

  IF ENABLE_STAT_COUNTERS
    push af
    ld a,(stat_num_commands_written)
    inc a
    ld (stat_num_commands_written),a
    pop af
  ENDIF

  ret

; Left-to-right
append_lmmm_command_for_wall_L2R_lower:
  ; Append an LMMM command to fill in the upper-left
  ; part of the wall.

  ld a,(iy+STACK_FY1_LOWER)
  cp (iy+STACK_FY0_LOWER)
  ret nc

  ; Get the ramp descriptor address
  ld de,rampdesc_addr_table_R2L ; Range is [0, +1024)
  ; log(quotient) = log(sin) - log(adjacent)
  ; Bobbing
  ld bc,(log_bob)
  and a
  sbc hl,bc
  ; hl ϵ [-4096, +4096)
  ; Clamp HL to zero
  bit 7,h
  jr z,$+2+3
  ld hl,#0000
  ; hl ϵ [0, +4096)
  ; Divide HL by two
  srl h
  rr l
  ; hl ϵ [0, +2048)
  ; Align HL to words
  ld a,#FE    ;  8c
  and l       ;  5c
  ld l,a      ;  5c
  add hl,de
  ; Load the ramp descriptor address from the table
  ld c,(hl)
  inc hl
  ld b,(hl)

  ld hl,#0000
  ld (wall_lower_rectangular_part_width),hl

  ld a,(iy+STACK_FX0)
  ld l,(iy+STACK_FX1)
  sub l
  ld l,a
  ld h,0

  ld (ix+COMMAND_NX+0),l   ; NX L
  ld (ix+COMMAND_NX+1),h   ; NX H

  push iy
  ld iyl,c
  ld iyh,b

  ; Load and/or clip NX
  ld c,(iy+RAMP_DESCRIPTOR_W)
  ld b,0
  inc bc
  and a
  sbc hl,bc
  jr c,append_lmmm_command_for_wall_L2R_lower.xclipped
  ld (ix+COMMAND_NX+0),c   ; NX L
  ld (ix+COMMAND_NX+1),b   ; NX H
  ; Store (positive) width of rectangular upper part
  ld (wall_lower_rectangular_part_width),hl
append_lmmm_command_for_wall_L2R_lower.xclipped:

  ; Copy SX from the ramp descriptor
  ld a,(iy+RAMP_DESCRIPTOR_ATLAS_X+0)
  ld (ix+COMMAND_SX+0),a
  ld a,(iy+RAMP_DESCRIPTOR_ATLAS_X+1)
  ld (ix+COMMAND_SX+1),a

  ; SY = ATLAS_Y
  ld l,(iy+RAMP_DESCRIPTOR_ATLAS_Y+0)
  ld h,(iy+RAMP_DESCRIPTOR_ATLAS_Y+1)
  ld (ix+COMMAND_SY+0),l
  ld (ix+COMMAND_SY+1),h

  ; Load and/or clip NY
  ld c,(iy+RAMP_DESCRIPTOR_H)
  ld b,0
  inc bc

  pop iy

  ld a,(iy+STACK_FY0_LOWER)
  ld l,(iy+STACK_FY1_LOWER)
  sub l
  ld l,a
  ld h,0

  ld e,l
  ld d,h
  and a
  sbc hl,bc
  jr nc,append_lmmm_command_for_wall_L2R_lower.skip_yclip
  ld c,e
  ld b,d
append_lmmm_command_for_wall_L2R_lower.skip_yclip:
  ld (ix+COMMAND_NY+0),c   ; NY L
  ld (ix+COMMAND_NY+1),b   ; NY H

  ; DX = fx1
  ld a,(iy+STACK_FX1)
  ld (ix+COMMAND_DX+0),a   ; DX L
  ; DY = FRAME_HEIGHT/2 + fy1_lower
  ld a,FRAME_HEIGHT/2
  add a,(iy+STACK_FY1_LOWER)
  ld (ix+COMMAND_DY+0),a   ; DY L

  ld (ix+COMMAND_ARG),%00000000  ; ARG, DIY=0, DIX=0

  ld a,(wall_shade+2)
  ld (ix+COMMAND_LOP),a       ; LOP
  ld (ix+COMMAND_OPCODE),V9O_LMMM<<4  ; OP-CODE

  ld bc,COMMAND_SIZE
  add ix,bc
  ld hl,(commandlist_write_length_ptr)
  inc (hl)

  IF ENABLE_STAT_COUNTERS
    push af
    ld a,(stat_num_commands_written)
    inc a
    ld (stat_num_commands_written),a
    pop af
  ENDIF

  ld hl,(wall_lower_rectangular_part_width)
  ld a,h
  or l
  jr nz,append_lmmv_command_for_wall_L2R_lower_rectangular

  ret

append_lmmv_command_for_wall_L2R_lower_rectangular:
  ; LMMV

  ; DX = fx0 - 1
  ld a,(iy+STACK_FX0)
  dec a
  ld (ix+COMMAND_DX+0),a   ; DX L

  ; DY = FRAME_HEIGHT - 1
  ld (ix+COMMAND_DY+0),FRAME_HEIGHT-1   ; DY L

  ; NX = wall_lower_rectangular_part_width
  ld hl,(wall_lower_rectangular_part_width)
  ld (ix+COMMAND_NX+0),l   ; NX L
  ld (ix+COMMAND_NX+1),h   ; NX H

  ; NY = FRAME_HEIGHT/2
  ld (ix+COMMAND_NY+0),FRAME_HEIGHT/2   ; NY L
  ld (ix+COMMAND_NY+1),0   ; NY H

  ld (ix+COMMAND_ARG),%00001100  ; ARG, DIY=1, DIX=1

  ld a,(wall_shade+1)
  ld (ix+COMMAND_LOP),a       ; LOP
  ld (ix+COMMAND_OPCODE),V9O_LMMV<<4  ; OP-CODE

  ld bc,COMMAND_SIZE
  add ix,bc
  ld hl,(commandlist_write_length_ptr)
  inc (hl)

  IF ENABLE_STAT_COUNTERS
    push af
    ld a,(stat_num_commands_written)
    inc a
    ld (stat_num_commands_written),a
    pop af
  ENDIF

  ret

; ---------------------- Interrupt handlers ----------------------

; ---------------------- Misc. lookup tables ----------------------

log_partial_sum_table:
  incbin "log_partial_sum_table.bin"
palette_rgb_start:
  ; Offset  0
  db (31*0)/3, (31*0)/3, (31*0)/3 ; Text Outline
  db (31*3)/3, (31*3)/3, (31*3)/3 ; Wall A
  db (31*2)/3, (31*2)/3, (31*3)/3 ; Background (Blue)
  db (31*1)/3, (31*1)/3, (31*1)/3 ; Wall B
  ; Offset  4
  db (31*3)/3, (31*0)/3, (31*0)/3 ; Pickup Item
  db (31*3)/3, (31*3)/3, (31*3)/3 ; Wall A
  db (31*2)/3, (31*2)/3, (31*3)/3 ; Background (Blue)
  db (31*1)/3, (31*1)/3, (31*1)/3 ; Wall B
  ; Offset  8
  db (31*3)/3, (31*0)/3, (31*0)/3 ; Pickup Item
  db (31*3)/3, (31*3)/3, (31*3)/3 ; Wall A
  db (31*1)/3, (31*2)/3, (31*1)/3 ; Background (Green)
  db (31*1)/3, (31*1)/3, (31*1)/3 ; Wall B
  ; Offset 12
  db 0, 0, 0 ; Border
palette_rgb_end:
maze:
  include "maze.asm"

  ds #4000 - $

; ---------------------- MSX ROM header for BIOS ----------------------

  org #4000

  db "AB"
  dw INIT
  db 0,0,0,0,0,0,0,0,0,0,0,0

INIT:
  di
  im 1 ; Interrupt mode 1 - RST #38
  ld sp,#F000

  ; Turn off the TMS9918 / V9938 so that we don't get useless interrupts from it.
  ld a,%00000000
  out (VDPCTL),a
  ld a,#81
  out (VDPCTL),a

  ; Switch pages 0 and 2 to the (primary) slot in which this
  ; cartridge is inserted. It is assumed that page 1 is already mapped.
  call RSLREG
  ld b,a
  ;    33221100
  and %00001100
  ld c,a
  REPT 4
    rlca
  ENDM
  or c
  REPT 2
    rrca
  ENDM
  ld c,a
  ld a,b
  ;    33221100
  and %11001100
  or c
  ; Now, since WSLREG is mapped into page 0 currently it can't be called.
  ; So the slot select register port is used directly instead.
  out (SLOTSL),a

  ; Page 3 is kept as RAM, making 16Kbyte of total RAM available.
  ; WARNING: After this point, the BIOS is not available at all.
  jp top

  db "Copyright (c) Edd Biddulph 2024"

; ---------------------- Tables ----------------------

; Note: Rooms and walls must come before #8000 due to how the flags work.
rooms:
  include "rooms.asm"
walls:
  include "walls.asm"
horizontal_splits:
  db PALETTE_OFFSET_ABOVE_HORIZON, HORIZON_Y_POSITION
  db PALETTE_OFFSET_BELOW_HORIZON, 0
ramp_descriptors:
  include "ramps.asm"
x_angle_table:
  incbin "x_angle_table.bin"
ray_intersection_test_table:
  incbin "ray_intersection_test_table.bin"
logtan_table:
  incbin "logtan_table.bin"
log_table:
  incbin "log_table.bin"
exp_table:
  incbin "exp_table.bin"
logsin_table:
  incbin "logsin_table.bin"
arctanlog_table:
  incbin "arctanlog_table.bin"
rampdesc_addr_table_L2R:
  include "rampdesc_addr_table_L2R.asm"
rampdesc_addr_table_R2L:
  include "rampdesc_addr_table_R2L.asm"
log_bob_table:
  incbin "log_bob_table.bin"
camera_move_sinus_table:
  incbin "camera_move_sinus_table.bin"
font:
  incbin "font.bin"

clear_framebuffer_lmmv:
  MAKE_LMMV 0, 0, FRAME_WIDTH, FRAME_HEIGHT, %00001100

  ds #C000 - $

; ---------------------- Variables ----------------------

; 16Kbyte RAM
  org #C000
ramp_image_buffer:
  ds virtual (RAMP_WIDTH*RAMP_HEIGHT*FRAME_BPP+7)/8 ; 8192B

  org #C000
RAM_START:
commandlists_start:
commandlist_0:
  ds virtual MAX_COMMANDS*COMMAND_SIZE+1
commandlist_1:
  ds virtual MAX_COMMANDS*COMMAND_SIZE+1
commandlists_end:
bob_phase:
  ds virtual 1
log_bob:
  ds virtual 2
scroll_position:
  ds virtual 4
commandlist_read_ptr:
  ds virtual 2
commandlist_read_length_ptr:
  ds virtual 2
commandlist_write_ptr:
  ds virtual 2
commandlist_write_length_ptr:
  ds virtual 2
ramp_gen_lmmm:
  ds virtual COMMAND_SIZE
clear_framebuffer_lmmv_ram:
  ds virtual COMMAND_SIZE
portal_stack_dummy:
  ds virtual STACK_SIZE
portal_stack:
  ds virtual STACK_SIZE*(MAX_PORTAL_DEPTH+1)
hblank_counter:
  ds virtual 1
wall_temp:
  ds virtual WALL_SIZE
buffer_swap:
  ds virtual 1
camera_angle:
  ds virtual 2
camera_angle_highprec:
  ds virtual 2
camera_angle_velocity:
  ds virtual 2
camera_x: ; 12.4 fixed
  ds virtual 2
camera_y: ; 12.4 fixed
  ds virtual 2
camera_vel_x:
  ds virtual 2
camera_vel_y:
  ds virtual 2
wall_shade:
  ds virtual 3
last_command_done_flag:
  ds virtual 1
horizontal_splits_ptr:
  ds virtual 2
commandlist_filled_flag:
  ds virtual 1
portal_depth:
  ds virtual 1
frames_to_skip:
  ds virtual 1
rng_padding:
  ds virtual 1
rng_state:
  ds virtual 5
stat_counters_start:
stat_num_commands_written:
  ds virtual 1
stat_num_project_posts:
  ds virtual 1
stat_num_clip_walls:
  ds virtual 1
stat_num_walls_drawn:
  ds virtual 1
wall_upper_rectangular_part_width:
  ds virtual 2
wall_lower_rectangular_part_width:
  ds virtual 2
global_portal_traversal_flags:
  ds virtual 1
stat_counters_end:
RAM_END:
  ds virtual #E000 - $

