#include <SDL.h>
#include <math.h>
#include <assert.h>

#include "font.h"
#include "defs.h"
#include "mem.h"
#include "render.h"
#include "game.h"
#include "input.h"


#define ENGINE_MODE_NULL            0
#define ENGINE_MODE_TITLE           1
#define ENGINE_MODE_TRACKSELECTION  2
#define ENGINE_MODE_INGAME          3


static hunk_marker_t mem_start;
static hunk_marker_t mem_frame;
static hunk_marker_t mem_level;
static hunk_marker_t mem_objmesh;
static hunk_marker_t mem_objdraw;


static bool_t game_paused = FALSE;
static int keep_going = 1;

static renderstate_t g_rs;
static player_t g_player;
static gamestate_t g_gs;

static camera_t g_camera;
static float g_map[16 * 16] = {
4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4,
4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 4,
4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 4,
4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 4,
4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 4,
4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 4,
4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 4,
4, 0, 0, 0, 4, 0, 0, 0, 0, 4, 0, 0, 0, 0, 0, 4,
4, 0, 0, 0, 0, 0, 0, 0, 0, 4, 0, 0, 0, 0, 0, 4,
4, 4, 0, 0, 0, 0, 4, 4, 4, 4, 4, 4, 4, 0, 0, 4,
4, 0, 0, 4, 0, 0, 0, 0, 0, 4, 0, 0, 0, 0, 0, 4,
4, 0, 4, 4, 0, 0, 0, 0, 0, 4, 0, 0, 0, 0, 0, 4,
4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 4,
4, 0, 4, 0, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 4,
4, 4, 4, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 4,
4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, };
static float g_map2[16 * 16] = {
4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4,
4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4,
4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4,
4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4,
4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4,
4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4,
4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4,
4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4,
4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4,
4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4,
4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4,
4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4,
4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4,
4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4,
4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4,
4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, };
static float g_looka = 0.0f;

static int num_obj_drawn = 0;

uint32_t frame_time = 0;


int g_pending_enginemode = ENGINE_MODE_NULL;
int g_enginemode = ENGINE_MODE_NULL;

static int sign(float x)
{
  return (x < 0.0) ? -1 : +1;
}

float sampleCollisionH0(float cx, float cy, float boxSize)
{
   float h0 = -1000;
   for(int i = -1; i < 2; i += 1)
   {
      for(int j = -1; j < 2; j += 1)
      {
         int cx2 = MAX(0, MIN(15, cx + boxSize * i));
         int cy2 = MAX(0, MIN(15, cy + boxSize * j));
         h0 = MAX(h0, g_map[cx2 + cy2 * 16] + boxSize);
      }
   }
   return h0;
}

float sampleCollisionH1(float cx, float cy, float boxSize)
{
   float h1 = +1000;
   for(int i = -1; i < 2; i += 1)
   {
      for(int j = -1; j < 2; j += 1)
      {
         int cx2 = MAX(0, MIN(15, cx + boxSize * i));
         int cy2 = MAX(0, MIN(15, cy + boxSize * j));
         h1 = MIN(h1, g_map2[cx2 + cy2 * 16] - boxSize);
      }
   }
   return h1;
}

void traceRay(float ox, float oy, float oz, float rx, float ry, float rz, float maxt, float boxSize, float* res_t, int* res_side)
{
   if(rx == 0.0f && ry == 0.0f && rz == 0.0f)
   {
     *res_t = 0;
     *res_side = -1;
     return;
   }
   
   int cx = (ox / boxSize), cy = (oy / boxSize);
   float intersection;
   float h0, h1, h, t;
   float t0 = 0.0f, t1 = 0.0f;
   int side = -1;
   
   if(rx == 0.0f && ry == 0.0f)
   {
      h0 = sampleCollisionH0(cx * boxSize, cy * boxSize, boxSize);
      h1 = sampleCollisionH1(cx * boxSize, cy * boxSize, boxSize);
      
      float mt = 1e4;
      
      if(rz < 0.0f)
      {
        t = (h0 - oz) / rz;

        if(t >= 0.0f && t < mt)
        {
          side = 1;
          mt = t;
        }
      }
      else
      {

        t = (h1 - oz) / rz;

        if(t >= 0.0f && t < mt)
        {
          side = 2;
          mt = t;
        }
      }

      if(mt < 1e4)
      {
        if(mt >= maxt)
        {
          *res_t = maxt;
          *res_side = -1;
          return;
        }

        *res_t = mt;
        *res_side = side;
        return;
      }
      
     *res_t = maxt;
     *res_side = -1;
     return;
   }
   
   float sx = (rx != 0.0f) ? boxSize / rx : 0, sy = (ry != 0.0f) ? boxSize / ry : 0;
   float ex = ((cx + ((rx > 0) ? 1 : 0)) - ox / boxSize) * sx;
   float ey = ((cy + ((ry > 0) ? 1 : 0)) - oy / boxSize) * sy;
   int ix = sign(rx), iy = sign(ry);

   sx = fabsf(sx);
   sy = fabsf(sy);

   float ph0 = sampleCollisionH0(cx * boxSize, cy * boxSize, boxSize);
   float ph1 = sampleCollisionH1(cx * boxSize, cy * boxSize, boxSize);
      
   for(int step = 0; step < 32; step += 1)
   {
      h0 = sampleCollisionH0(cx * boxSize, cy * boxSize, boxSize);
      h1 = sampleCollisionH1(cx * boxSize, cy * boxSize, boxSize);
      
       t1 = (rx == 0.0f) ? ey : (ry == 0.0f) ? ex : MIN(ex, ey);
       
       float mt = 1e4;

       if(step > 0)
       {
         float z = oz + rz * t0;

         if(z <= h0 && z >= ph0)
           mt = MIN(mt, t0);
          
         if(z <= ph1 && z >= h1)
           mt = MIN(mt, t0);
       }

       if(rz != 0)
       {
          if(rz < 0.0f)
          {
              t = (h0 - oz) / rz;
              
              if(t <= t1 && t >= 0.0f && t < mt)
              {
                side = 1;
                mt = t;
              }
          }
          else
          {
              
              t = (h1 - oz) / rz;
              
              if(t <= t1 && t >= 0.0f && t < mt)
              {
                side = 2;
                mt = t;
              }
          }
       }

       if(mt < 1e4)
       {
         if(mt >= maxt)
         {
           *res_t = maxt;
           *res_side = -1;
           return;
         }
            
         *res_t = mt;
         *res_side = side;
         return;
       }
       
       t0 = t1;
       
       if(t0 >= maxt)
       {
         *res_t = maxt;
         *res_side = -1;
         return;
       }
       
       side = (ex < ey) ? 3 : 4;
       side = (rx == 0.0f) ? 4 : (ry == 0.0f) ? 3 : side;
       
       if(rx == 0.0f)
       {
        cy += iy;
        ey += sy;
       }
       else if(ry == 0.0f)
       {
        cx += ix;
        ex += sx;
       }
       else
       {
        if (ex < ey)
        {
          cx += ix;
          ex += sx;
        }
        else
        {
          cy += iy;
          ey += sy;
        }
       }
        
        ph0 = h0;
        ph1 = h1;
   }
   
   *res_t = maxt;
   *res_side = -1;
}

  static float velx = 0.0f, vely = 0.0f, velz = 0.0f;
 
static void moveCamera(float dx, float dy, float dz)
{
  float t = 0.0f;
  int side = -1;

  float remaining_t = 1.0f;
  int max_slides = 8;

  do
  {
    traceRay(g_camera.px, g_camera.py, g_camera.pz, dx, dy, dz, remaining_t, 0.25f, &t, &side);

    g_camera.px += dx * MAX(0.0f, t - 0.001f);
    g_camera.py += dy * MAX(0.0f, t - 0.001f);
    g_camera.pz += dz * MAX(0.0f, t - 0.001f);
    
    remaining_t -= t;
    
    switch(side)
    {
      case 1:
      case 2:
        if(dz != 0.0f)
          g_camera.pz += +0.001f * sign(-dz);
        dz = 0.0f;
        velz = 0.0f;
        break;
      case 3:
        if(dx != 0.0f)
          g_camera.px += +0.001f * sign(-dx);
        dx = 0.0f;
        velx *= 0.5f;
        break;
      case 4:
        if(dy != 0.0f)
          g_camera.py += +0.001f * sign(-dy);
        dy = 0.0f;
        vely *= 0.5f;
        break;
    }
    
    --max_slides;
    
    if(max_slides == 0)
      break;

  } while(remaining_t > 0.0f && (dx != 0.0f || dy != 0.0f || dz != 0.0f));

}


void transitionEngineMode(int old_mode, int new_mode)
{
  assert(old_mode != new_mode);
  assert(new_mode != ENGINE_MODE_NULL);
  
  switch(old_mode)
  {
    case ENGINE_MODE_NULL:
      break;
    
    case ENGINE_MODE_TRACKSELECTION:
      break;
      
    case ENGINE_MODE_INGAME:
      break;
      
    default:
      assert(0);
      break;
  }
  
  switch(new_mode)
  {
    case ENGINE_MODE_TRACKSELECTION:
      break;
      
    case ENGINE_MODE_INGAME:
      resetLevel(&g_player, &g_gs, SDL_GetTicks());
      break;
      
    default:
      assert(0);
      break;
  }
  
  g_enginemode = new_mode;
}


static void quitLevel()
{
  keep_going = 0;
}


      
typedef struct
{
  const char* label;
} menu_item_t;

typedef struct
{
  menu_item_t* items;
  uint16_t current_index;
  uint16_t num_items;
} menu_t;

static menu_item_t pause_menu_items[] = { { "CONTINUE" }, { "RETRY" }, { "QUIT" } };
static menu_t pause_menu = (menu_t){ pause_menu_items, 0, sizeof(pause_menu_items) / sizeof(menu_item_t) };

void drawPauseMenu(uint16_t *const pixels)
{
  const uint16_t menubox_width = 100;
  const uint16_t menubox_height = pause_menu.num_items * HUD_FONT_H;
  const uint16_t menubox_left = FRAME_WIDTH / 2 - menubox_width / 2;
  const uint16_t menubox_top = FRAME_HEIGHT / 2 - menubox_height / 2 - 30;
  
  for(uint16_t y = menubox_top; y < menubox_top + menubox_height; ++y)
  {
    for(uint16_t x = menubox_left; x < menubox_left + menubox_width; ++x)
    {
      pixels[x + y * FRAME_WIDTH] = (pixels[x + y * FRAME_WIDTH] >> 1) & 0b0111101111101111;
    }
  }
  
  const uint16_t selectionbox_width = menubox_width;
  const uint16_t selectionbox_height = HUD_FONT_H;
  const uint16_t selectionbox_left = menubox_left;
  const uint16_t selectionbox_top = menubox_top + HUD_FONT_H * pause_menu.current_index;
  
  for(uint16_t y = selectionbox_top; y < selectionbox_top + selectionbox_height; ++y)
  {
    for(uint16_t x = selectionbox_left; x < selectionbox_left + selectionbox_width; ++x)
    {
      pixels[x + y * FRAME_WIDTH] = ((pixels[x + y * FRAME_WIDTH] >> 1) & 0b0111101111101111) + ((RGB888toRGB565(255, 100, 100) >> 1) & 0b0111101111101111);
    }
  }
  
  for(int i = 0; i < pause_menu.num_items; ++i)
  {
    plotString(pixels, FRAME_WIDTH / 2 - (strlen(pause_menu.items[i].label) * HUD_FONT_W) / 2, menubox_top + i * HUD_FONT_H, pause_menu.items[i].label, 1);
  }
}

void updateMenu(menu_t* menu)
{
  if(sys_inputs[SYS_INPUT_MENU_UP] != sys_inputs_previous[SYS_INPUT_MENU_UP])
  {
    if(sys_inputs[SYS_INPUT_MENU_UP])
    {
      menu->current_index = (menu->current_index + menu->num_items - 1) % menu->num_items;
    }
  }
  
  if(sys_inputs[SYS_INPUT_MENU_DOWN] != sys_inputs_previous[SYS_INPUT_MENU_DOWN])
  {
    if(sys_inputs[SYS_INPUT_MENU_DOWN])
    {
      menu->current_index = (menu->current_index + 1u) % menu->num_items;
    }
  }
}

void updatePauseMenu()
{
  updateMenu(&pause_menu);
  
  if(sys_inputs[SYS_INPUT_MENU_CHOOSE])
  {
    if(pause_menu.current_index == 1)
      resetLevel(&g_player, &g_gs, SDL_GetTicks());
    else if(pause_menu.current_index == 2)
      quitLevel();
    game_paused = FALSE;
  }
}


void drawHUD(uint16_t *const pixels, renderstate_t* rs, const gamestate_t* gs, const player_t* player)
{

}


void drawObjects(uint16_t *const pixels, const float time, renderstate_t* rs, gamestate_t* gs, player_t* player)
{
  hunk_marker_t mem_drawobjects;
  hunkMark(&mem_drawobjects);

  for(int idx = 0; idx < gs->num_mines; ++idx)
    if(gs->mines[idx].active)
    {
      clipAndPlotBox(pixels, gs->mines[idx].posx, gs->mines[idx].posy, PLAYER_SIZE, PLAYER_SIZE, RGB888toRGB565(255, 255, 255));
    }
    
  for(int idx = 0; idx < gs->num_bullets; ++idx)
    if(gs->bullets[idx].active)
    {
      clipAndPlotBox(pixels, gs->bullets[idx].posx, gs->bullets[idx].posy, PLAYER_SIZE, PLAYER_SIZE, RGB888toRGB565(255, 0, 0));
    }
    
    
  hunkFree(&mem_drawobjects);
}

void drawIngameScreen(uint16_t* pixels, renderstate_t* rs, const gamestate_t* gs)
{
  const float time = (float)((double)(gs->play_time) / 1000.0);
  const uint32_t frame_start = SDL_GetTicks();

  //clipAndPlotBox(pixels, 0, 0, FRAME_WIDTH, FRAME_HEIGHT, RGB888toRGB565(50, 50, 50));

  float lookx = cosf(g_looka);
  float looky = sinf(g_looka);
  g_camera.matrix[0] = lookx;
  g_camera.matrix[1] = looky;
  g_camera.matrix[2] = -looky;
  g_camera.matrix[3] = lookx;
  g_camera.angle = g_looka;
 
  g_rs.time = (float)((double)(frame_start) / 1000.0);;

  renderColumns(pixels, &g_rs, &g_camera, g_map, g_map2);


  const uint32_t hud_start = SDL_GetTicks();
  drawHUD(pixels, &g_rs, &g_gs, &g_player);

  if(game_paused)
   drawPauseMenu(pixels);

  const uint32_t hud_end = SDL_GetTicks();


  const uint32_t frame_end = SDL_GetTicks();

  //if(rs->debug_display_enabled)
  {
    static char debug_text[8192];
    memset(debug_text, 0, sizeof(debug_text));
    snprintf(debug_text, sizeof(debug_text),
                     "hud_ms            = %u\n"
                     "total_ms          = %u\n"
                     "num_obj_drawn     = %u\n"
                     ,hud_end - hud_start, frame_end - frame_start, num_obj_drawn);

    plotString(pixels, 0, 0, debug_text, 1);
  }

}

extern bool_t extern_write_frame;

void updateIngameLogic()
{
/*
  if(sys_inputs[SYS_INPUT_PAUSE] != sys_inputs_previous[SYS_INPUT_PAUSE])
  {
    if(sys_inputs[SYS_INPUT_PAUSE])
    {
      game_paused ^= sys_inputs[SYS_INPUT_PAUSE];
      if(game_paused)
      {
        pause_menu.current_index = 0;
      }
    }
  }
  
  static int prev_drop_state = 0;
  if(sys_inputs[SYS_INPUT_DROP] != prev_drop_state)
    g_rs.debug_display_enabled ^= (int)sys_inputs[SYS_INPUT_DROP];
  prev_drop_state = sys_inputs[SYS_INPUT_DROP];
  
  if(game_paused)
    updatePauseMenu();
  */

  if(sys_inputs[SYS_INPUT_PAUSE] != sys_inputs_previous[SYS_INPUT_PAUSE])
  {
    if(sys_inputs[SYS_INPUT_PAUSE])
    {
      extern_write_frame=TRUE;
    }
  }

  const float turn_speed = 1.0f/64.0f;
  const float move_speed = .01f * 16.0f*8.0f;
   
  if(!game_paused)
  {
    if(sys_inputs[SYS_INPUT_LEFT])
      g_looka += turn_speed;
      
    if(sys_inputs[SYS_INPUT_RIGHT])
      g_looka -= turn_speed;
    

    if(sys_inputs[SYS_INPUT_UP])
    {
      velx += cos(g_looka) * move_speed;
      vely += sin(g_looka) * move_speed;
    }
      
    if(sys_inputs[SYS_INPUT_DOWN])
    {
      velx += cos(g_looka) * -move_speed;
      vely += sin(g_looka) * -move_speed;
    }
    
    
    //moveCamera(velx, vely, velz);

    g_camera.px += velx;
    g_camera.py += vely;

    velx *= 0.75f;
    vely *= 0.75f;

    
  }
}
     


int main(int argc, char* argv[])
{
  printf("Compiled on " __DATE__ " at " __TIME__ ".\n");

#ifdef __GNUC__
# ifdef __GNUC_MINOR__
#   ifdef __GNUC_PATCHLEVEL__
  printf("Compiled with GCC %d.%d.%d\n", __GNUC__, __GNUC_MINOR__, __GNUC_PATCHLEVEL__);
#   endif
# endif
#endif

  printf("Compiled against SDL version %d.%d.%d\n", SDL_MAJOR_VERSION, SDL_MINOR_VERSION, SDL_PATCHLEVEL);

  const SDL_version* sdl_version = SDL_Linked_Version();

  if(sdl_version)
    printf("Running dynamically-linked SDL version %d.%d.%d\n", sdl_version->major, sdl_version->minor, sdl_version->patch);
  else
    printf("SDL_Linked_Version() returned NULL.\n");


  hunkInit();
  hunkMark(&mem_start);

  if(SDL_Init(SDL_INIT_TIMER | SDL_INIT_VIDEO))
      return -1;

  SDL_WM_SetCaption("", NULL);
  
  atexit(SDL_Quit);
  
  SDL_Surface* surface = SDL_SetVideoMode(FRAME_WIDTH, FRAME_HEIGHT, 16, SDL_HWSURFACE | SDL_DOUBLEBUF | SDL_ASYNCBLIT | SDL_ANYFORMAT);

  if(!surface)
      return -1;
    
  sysInitialise();

  g_camera.matrix[0] = 1;
  g_camera.matrix[1] = 0;
  g_camera.matrix[2] = 0;
  g_camera.matrix[3] = 1;
  g_camera.horizontalFOV = 3.14159265358979323f * 100. / 180.;
  g_looka = 0;
  g_camera.px = 0.0f;
  g_camera.py = 0.0f;
  g_camera.pz = 0.5f;

  initRenderstate(&g_rs);

  g_pending_enginemode = ENGINE_MODE_INGAME;

  while(keep_going == 1)
  {
      if(g_pending_enginemode != g_enginemode)
      {
          transitionEngineMode(g_enginemode, g_pending_enginemode);
      }
    
      SDL_Event event;

      while(SDL_PollEvent(&event))
      {
          if(event.type == SDL_QUIT)
          {
              keep_going = 0;
          }
      }

      frame_time = SDL_GetTicks() - g_gs.start_level_time;

      while(frame_time >= g_gs.next_frame)
      {
          if(!sysUpdateInput())
            return 0;
          
          switch(g_enginemode)
          {
            case ENGINE_MODE_TRACKSELECTION:
              break;
            case ENGINE_MODE_INGAME:
              updateIngameLogic();
              break;
          }

          g_gs.next_frame += 15;
      }
      
      hunkMark(&mem_frame);
      
      SDL_LockSurface(surface);

      uint16_t* pixels = (uint16_t*)surface->pixels;
      
      switch(g_enginemode)
      {
        case ENGINE_MODE_TRACKSELECTION:
          break;
        case ENGINE_MODE_INGAME:
          drawIngameScreen(pixels, &g_rs, &g_gs);
          break;
      }

      SDL_UnlockSurface(surface);

      SDL_UpdateRect(surface, 0, 0, 0, 0);
      SDL_Flip(surface);

      SDL_Delay(10);

      hunkFree(&mem_frame);
  }


  printf("mem_hunk_highwater = %u\n", mem_hunk_highwater);
  
  return 0;
}
