; HALLS AND HEARTS (techdemo)
; TMS9918 Version

ALIGN: MACRO ?boundary
       ds ?boundary - 1 - ($ + ?boundary - 1) % ?boundary
       ENDM

ALIGN_VIRTUAL: MACRO ?boundary
       ds virtual (?boundary - 1 - ($ + ?boundary - 1) % ?boundary)
       ENDM

GENERATE_SCAN: MACRO ?r1, ?r2, ?r3, ?i1, ?i2, ?i3, ?t1, ?t2
      ld b,(hl) ; A
      inc hl
      ld c,(hl) ; B
      inc hl
      ld d,(hl) ; C
      inc hl

      ld a,?r1
      ld (scan_heights+2),a  ; 14C
      ld a,?r2
      ld (scan_heights+1),a  ; 14C
      ld a,?r3
      ld (scan_heights+0),a  ; 14C

      REPT 2
        srl b
        srl c
        srl d
      ENDM

      ld a,12
      sub ?r1
      push af ; Last

      ld a,?r1
      sub ?r2
      push af

      ld a,?r2
      sub ?r3
      push af

      ld a,?r3

      exx

      ld de,40
    loop3:
      dec a       ;  5C
      jp m,loop3_out ; 13C
      ld (hl),d   ;  7C  D = 0, since DE = 40
      add hl,de   ; 11C
      jr loop3    ; 13C
    loop3_out:

      ; Apply translation
      ld a,(scan_heights+0)  ; 14C
      and 3
      or ?i1*4
      ld b,a
      ld c,0
      ex de,hl
      ld hl,translation_table
      add hl,bc
      ld c,(hl)
      ex de,hl

      ; Apply translation, part 2
      ld b,?t1

      ; Restore
      pop af
      ld de,40
    loop2:
      dec a       ;  5C
      jp m,loop2_out ; 13C
      ld (hl),c   ;  7C
      add hl,de   ; 11C
      ; Commit part 2
      ld c,b      ;  5C
      jr loop2    ; 13C
    loop2_out:

      ; Apply translation
      ld a,(scan_heights+1)  ; 14C
      and 3
      or ?i2*4
      ld b,a
      ex de,hl
      ld hl,translation_table
      add hl,bc
      ld c,(hl)
      ex de,hl

      ; Apply translation, part 2
      ld b,?t2

      ; Restore
      pop af
      ld de,40
    loop1:
      dec a       ;  5C
      jp m,loop1_out ; 13C
      ld (hl),c   ;  7C
      add hl,de   ; 11C
      ; Commit part 2
      ld c,b      ;  5C
      jr loop1    ; 13C
    loop1_out:

      ; Apply translation
      ld a,(scan_heights+2)  ; 14C
      and 3
      or ?i3*4
      ld b,a
      ex de,hl
      ld hl,translation_table
      add hl,bc
      ld c,(hl)
      ex de,hl

      ; Apply translation, part 2
      ld b,TRANSLATION_FULLSTATE_111

      ; Restore Last
      pop af
      ld de,40
    loopLast:
      dec a       ;  5C
      jp m,loopLast_out ; 13C
      ld (hl),c   ;  7C
      add hl,de   ; 11C
      ; Commit part 2
      ld c,b      ;  5C
      jr loopLast ; 13C
    loopLast_out:

      ; Rewind back to the top of the column, while filling in
      ; a mirror image for the top half of the display.
      ld e,l
      ld d,h
      ex de,hl
      ld bc,-40*24
      add hl,bc
      ex de,hl
      ld bc,-40
    REPT 12
      add hl,bc
      ld a,(hl)
      ex de,hl
      xor 255
      ld (hl),a
      and a
      sbc hl,bc
      ex de,hl
    ENDM

      ; Step to the top of the next column
      inc hl

      exx

      ret
      ENDM

; Portal traversal stack frame layout
STACK_FX0:            equ  0 ; 1
STACK_FY0:            equ  1 ; 1
STACK_FX1:            equ  2 ; 1
STACK_FY1:            equ  3 ; 1
STACK_RAMP_UPPER:     equ  4 ; 2
STACK_FLAGS:          equ  6 ; 1
STACK_LOG_ADJACENT:   equ  7 ; 2
STACK_RELATIVE_ANGLE: equ  9 ; 2
STACK_LOG_OPPOSITE:   equ 11 ; 2
STACK_ORIENTATION:    equ 13 ; 1
STACK_X0:             equ 14 ; 2
STACK_Y0:             equ 16 ; 2
STACK_X1:             equ 18 ; 2
STACK_Y1:             equ 20 ; 2
STACK_OPPOSITE:       equ 22 ; 2
STACK_ADJACENT:       equ 24 ; 2
STACK_A0:             equ 28 ; 2    Pass by value
STACK_A1:             equ 30 ; 2    Pass by value
STACK_ITESTFLAGS0:    equ 32 ; 1
STACK_ITESTFLAGS1:    equ 33 ; 1
STACK_ITESTFLAGS:     equ 34 ; 1
STACK_CAMERA_X:       equ 36 ; 2    Pass by value
STACK_CAMERA_Y:       equ 38 ; 2    Pass by value
STACK_RAMP_LOWER:     equ 40 ; 2

STACK_SIZE:           equ 42

; Wall descriptor
WALL_X1:              equ  0 ; 2
WALL_Y1:              equ  2 ; 2
WALL_ROUTINE:         equ  4 ; 2
WALL_PORTAL_ROOM:     equ  6 ; 2

WALL_SIZE:            equ  8

MAKE_WALL: MACRO ?x0, ?y0, ?x1, ?y1, ?orientation, ?portal_room
      dw ?x1, ?y1
      IF ?orientation=0
        dw draw_wall_orientation_0
      ENDIF
      IF ?orientation=1
        dw draw_wall_orientation_1
      ENDIF
      IF ?orientation=2
        dw draw_wall_orientation_2
      ENDIF
      IF ?orientation=3
        dw draw_wall_orientation_3
      ENDIF
      dw ?portal_room
      ENDM

; Room descriptor

ROOM_NUM_WALLS:       equ  0 ; 1
ROOM_FIRST_WALL_Q0:   equ  1 ; 2
ROOM_FIRST_WALL_Q1:   equ  3 ; 2
ROOM_FIRST_WALL_Q2:   equ  5 ; 2
ROOM_FIRST_WALL_Q3:   equ  7 ; 2

ROOM_SIZE:            equ  9

MAKE_ROOM: MACRO ?num_walls, ?first_wall_q0, ?first_wall_q1, ?first_wall_q2, ?first_wall_q3
      db ?num_walls
      dw ?first_wall_q0, ?first_wall_q1, ?first_wall_q2, ?first_wall_q3
      ENDM

CLIP_WALL: MACRO

      IF ENABLE_STAT_COUNTERS
        push af
        ld a,(stat_num_clip_walls)
        inc a
        ld (stat_num_clip_walls),a
        pop af
      ENDIF

      ; Calculate opposite for remapped ray

      ; hl ϵ (-256, +256)

      ; Use A to indicate negative angle
      xor a
      ; Calculate abs(angle)
      bit 7,h
      jr z,$+2+1+1+1+1+1+1+1+2
      ; Negate HL
      ld a,h  ;  5c
      cpl     ;  5c
      ld h,a  ;  5c
      ld a,l  ;  5c
      cpl     ;  5c
      ld l,a  ;  5c
      inc hl  ;  7c
      ld a,1
      ; hl ϵ [0, +256)

      add hl,hl           ; 12c
      ld de,logtan_table  ; 11c
      add hl,de           ; 12c
      ld e,(hl)           ;  8c
      inc hl              ;  7c
      ld d,(hl)           ;  8c
      ex de,hl
      ; log_tangent is now in HL
      ; log_adjacent is kept in BC
      add hl,bc           ; 12c
      ; log_opposite=max(0,log_tangent+log_adjacent)
      bit 7,h
      jr z,$+2+3
      ld hl,#0000
      ld (iy+STACK_LOG_OPPOSITE+0),l
      ld (iy+STACK_LOG_OPPOSITE+1),h

      add hl,hl
      ld de,exp_table     ; 11c
      add hl,de           ; 12c
      ld e,(hl)           ;  8c
      inc hl              ;  7c
      ld d,(hl)           ;  8c
      ex de,hl            ;  5c
      ; hl ϵ [0, +4096)
 
      ; Negate opposite if the angle was negative
      and a
      jr z,$+2+1+1+1+1+1+1+1
      ; Negate HL
      ld a,h  ;  5c
      cpl     ;  5c
      ld h,a  ;  5c
      ld a,l  ;  5c
      cpl     ;  5c
      ld l,a  ;  5c
      inc hl  ;  7c
      ; hl ϵ (-4096, +4096)
      ENDM

DRAW_WALL_ORIENTATION_N: MACRO ?orientation, ?o0, ?o1, ?o2, ?o3
      ld (iy+STACK_ORIENTATION),?orientation

      ; Skip ray 0 if an intersection has been found already
      bit 0,(iy+STACK_FLAGS)
      jr nz,skip_ray_0

      ; Clip the wall to the frustum sides

      ; Note that for a given wall orientation and adjacent, the opposites
      ; of the ray intersection points are constant, so they need to be recalculated only
      ; at orientation changes within a room.

      ; Test for intersection with ray 0
      ; If there is no intersection of ray 0 with the line of this wall, then since there
      ; must be a previous wall which did then the previous wall's endpost can be re-used.
      bit ?orientation,(iy+STACK_ITESTFLAGS0)     ; Ray 0 intersection test flags
      jr nz,skip_ray_0

      ; Calculate opposite for ray 0

      ; Remap ray 0 angle for clipping
      ld l,(iy+STACK_A0+0)         ; 21c   Ray 0 angle
      ld h,(iy+STACK_A0+1)         ; 21c
      IF ?orientation=0
        ld de,-(NUM_ANGLE_UNITS*3)/4
        add hl,de
      ENDIF
      IF ?orientation=1
        bit 1,h
        jr z,$+2+3+1+2
        ld de,NUM_ANGLE_UNITS
        and a
        sbc hl,de
      ENDIF
      IF ?orientation=2
        ex de,hl
        ld hl,+(NUM_ANGLE_UNITS*1)/4
        and a
        sbc hl,de
      ENDIF
      IF ?orientation=3
        ex de,hl
        ld hl,+(NUM_ANGLE_UNITS*1)/2
        and a
        sbc hl,de
      ENDIF
      ; hl ϵ (-256, +256)
      CLIP_WALL
      ; hl ϵ (-4096, +4096)
      ; Opposite for ray 0 is now in HL

      ; Compare computed opposite against wall extents

      ; If opposite >= o3 then discard this wall (return immediately)
      ld c,(iy+?o3+0)
      ld b,(iy+?o3+1)
      ld e,l
      ld d,h
      and a
      sbc hl,bc
      IF ?orientation<2
        ret p
      ELSE
        ret m
      ENDIF

      ld l,e
      ld h,d
      ld (iy+STACK_OPPOSITE+0),l
      ld (iy+STACK_OPPOSITE+1),h
      ; If opposite <= o2 then re-calculate the opposite from o2
      ld c,(iy+?o2+0)
      ld b,(iy+?o2+1)
      and a
      sbc hl,bc
      jr z,use_clipped_opposite_0
      IF ?orientation<2
        jp p,use_clipped_opposite_0
      ELSE
        jp m,use_clipped_opposite_0
      ENDIF

skip_ray_0:

      set 0,(iy+STACK_FLAGS)

      ; Do not rely on previous calculation if this is the very first
      ; vertex, since there is none.
      ld a,(global_portal_traversal_flags)
      bit 0,a
      jr z,use_clipped_opposite_0

      ; Re-use the previously-calculated vertex

      ld a,(iy+STACK_FX1)
      ld (iy+STACK_FX0),a
      ld a,(iy+STACK_FY1)
      ld (iy+STACK_FY0),a
      ld a,(iy+STACK_SIZE+STACK_A1+0)
      ld (iy+STACK_SIZE+STACK_A0+0),a
      ld a,(iy+STACK_SIZE+STACK_A1+1)
      ld (iy+STACK_SIZE+STACK_A0+1),a

      jr after_ray_0

      ; Else use the clipped opposite as-is
use_clipped_opposite_0:

      set 0,(iy+STACK_FLAGS)

      ; The angle is taken to be equal to A0
      ; Project endpost 0
      call project_post_clipped_a0
      ld a,(iy+STACK_FX1)
      ld (iy+STACK_FX0),a
      ld a,(iy+STACK_FY1)
      ld (iy+STACK_FY0),a

after_ray_0:

      ld hl,global_portal_traversal_flags
      set 0,(hl)

      ; Skip ray 1 if an intersection with ray 0 has not been found already
      bit 0,(iy+STACK_FLAGS)
      jp z,skip_ray_1

      ; Test for intersection with ray 1
      bit ?orientation,(iy+STACK_ITESTFLAGS1)     ; Ray 1 intersection test flags
      jr nz,skip_ray_1

      ; Calculate opposite for ray 1

      ld c,(iy+STACK_LOG_ADJACENT+0)         ; 21c
      ld b,(iy+STACK_LOG_ADJACENT+1)         ; 21c

      ; Remap ray 1 angle for clipping
      ld l,(iy+STACK_A1+0)         ; 21c   Ray 1 angle
      ld h,(iy+STACK_A1+1)         ; 21c
      IF ?orientation=0
        ld de,-(NUM_ANGLE_UNITS*3)/4
        add hl,de
      ENDIF
      IF ?orientation=1
        bit 1,h
        jr z,$+2+3+1+2
        ld de,NUM_ANGLE_UNITS
        and a
        sbc hl,de
      ENDIF
      IF ?orientation=2
        ex de,hl
        ld hl,+(NUM_ANGLE_UNITS*1)/4
        and a
        sbc hl,de
      ENDIF
      IF ?orientation=3
        ex de,hl
        ld hl,+(NUM_ANGLE_UNITS*1)/2
        and a
        sbc hl,de
      ENDIF
      ; hl ϵ (-256, +256)
      CLIP_WALL
      ; hl ϵ (-4096, +4096)
      ; Opposite for ray 1 is now in HL

      ; Compare computed opposite against wall extents

      ; If opposite <= o2 then discard this wall (return immediately)
      ld e,l
      ld d,h
      ld l,(iy+?o2+0)
      ld h,(iy+?o2+1)
      and a
      sbc hl,de
      IF ?orientation<2
        ret p
      ELSE
        ret m
      ENDIF

      ld l,e
      ld h,d
      ld (iy+STACK_OPPOSITE+0),l
      ld (iy+STACK_OPPOSITE+1),h
      ; If opposite >= o0 and opposite <= o1 then use this opposite, and later return
      ld l,(iy+?o3+0)
      ld h,(iy+?o3+1)
      and a
      sbc hl,de
      jr z,use_clipped_opposite_1
      IF ?orientation<2
        jp p,use_clipped_opposite_1
      ELSE
        jp m,use_clipped_opposite_1
      ENDIF

skip_ray_1:
      ld l,(iy+?o3+0)
      ld h,(iy+?o3+1)
      ld (iy+STACK_OPPOSITE+0),l
      ld (iy+STACK_OPPOSITE+1),h
      ; Calculate abs(opposite)
      ; Negate HL if it is negative.
      bit 7,h
      jr z,$+2+1+1+1+1+1+1+1
      ld a,h
      cpl
      ld h,a
      ld a,l
      cpl
      ld l,a
      inc hl
      ; hl ϵ (0, 4096)
      add hl,hl           ; 12c
      ld bc,log_table     ; 11c
      add hl,bc           ; 12c
      ld c,(hl)
      inc hl
      ld b,(hl)
      ; bc ϵ [0, 4096)
      ld (iy+STACK_LOG_OPPOSITE+0),c         ; 21c
      ld (iy+STACK_LOG_OPPOSITE+1),b         ; 21c

      ; The angle is not known yet, so a full projection is needed
      ; Project endpost 1
      call project_post
      jr after_ray_1

      ; Else use the clipped opposite as-is
use_clipped_opposite_1:

      ; Early-out of the room loop, since this is the last visible wall in the room.
      set 1,(iy+STACK_FLAGS)

      ; The angle is taken to be equal to A1
      ; Project endpost 1
      call project_post_clipped_a1

after_ray_1:

      ; Clamp to the portal bounds
      ld a,(iy-STACK_SIZE+STACK_FX0)
      cp (iy+STACK_FX0)
      jr nc,$+2+3
      ld (iy+STACK_FX0),a

      ld a,(iy-STACK_SIZE+STACK_FX1)
      cp (iy+STACK_FX1)
      jr c,$+2+3
      ld (iy+STACK_FX1),a

      ; Discard if invisibly-thin or flipped.
      ld a,(iy+STACK_FX1)
      cp (iy+STACK_FX0)
      ret nc

      bit 7,(ix+WALL_PORTAL_ROOM+1)
      jp z,enter_portal

      ; Calculate relative camera angle
      ; relative_angle=camera_angle-(NUM_ANGLE_UNITS*3)/4;
      ld hl,(camera_angle)
      IF ?orientation=0
        ld de,-(NUM_ANGLE_UNITS*3)/4
        add hl,de
      ENDIF
      IF ?orientation=1
      ENDIF
      IF ?orientation=2
        ld de,-NUM_ANGLE_UNITS/4
        add hl,de
      ENDIF
      IF ?orientation=3
        ld de,-NUM_ANGLE_UNITS/2
        add hl,de
      ENDIF
      ld (iy+STACK_RELATIVE_ANGLE+0),l
      ld (iy+STACK_RELATIVE_ANGLE+1),h

      ; Cached vertex projection possibilities:
      ;   previous_vertex_in_room - Available after the first visible wall has been drawn
      ;   entered_portal_vertex_1 - Available always except in the first room
      ;   entered_portal_vertex_2 - Available always except in the first room

      ; It should be knowable offline whether or not the two latter cases can be assumed or not.
      ; Since orderings are already found offline, backface culling can be done offline too.

      jp generate_wall
      ENDM

PROJECT_POST_Y: MACRO
      ; Y projection

      ; log(cos)
      ld a,d
      cpl
      ld h,0
      ld l,a
      inc hl
      add hl,hl
      ld de,logsin_table
      add hl,de
      ld e,(hl)
      inc hl
      ld d,(hl)

      ; log_partial_sum
      pop hl ; Re-use the index computed earlier.
      ld bc,log_partial_sum_table
      add hl,bc
      ld c,(hl)
      inc hl
      ld b,(hl)

      ; log(c) * 2 = log(wall_height) / log(tangent_max) * 2047 * 2
      ld hl,3342*2
      and a
      sbc hl,de
      and a
      sbc hl,bc

      pop bc ; log_opposite
      and a
      sbc hl,bc

      ; Bobbing
      ld de,exp_table

      REPT 2,?I
        push hl
        ; Clamp
        ld a,h
        and #F8
        jr z,$+2+3
        ld hl,#0800
        ; Exponentiation
        add hl,hl
        add hl,de
        ld a,(hl)
        IF ?I=0
          ld (iy+STACK_FY1),a
        ELSE
          ld (iy+STACK_FY1),a
        ENDIF
        pop hl
      ENDM
      ENDM

; BIOS ROM
CGTABL: equ #0004

; BIOS Routines
RSLREG: equ #0138
WSLREG: equ #013B
ENASLT: equ #0024
TOTEXT: equ #00D2
CHGCLR: equ #0062
GTSTCK: equ #00D5
SETWRT: equ #0053
CHGMOD: equ #005f
INITXT: equ #006C
LDIRVM: equ #005C
INIT32: equ #006F
INIGRP: equ #0072
FILVRM: equ #0056

; System Variables
FORCLR: equ #F3E9
BAKCLR: equ #F3EA
BDRCLR: equ #F3EB
TXTNAM: equ #F3B3
TXTCGP: equ #F3B7
HKEY:   equ #FD9A
CLIKSW: equ #F3DB
T32NAM: equ #F3BD
T32COL: equ #F3BF
T32CGP: equ #F3C1
T32ATR: equ #F3C3
T32PAT: equ #F3C5
GRPNAM: equ #F3C7
GRPCOL: equ #F3C9
GRPCGP: equ #F3CB
GRPATR: equ #F3CD
GRPPAT: equ #F3CF

; VRAM
PATTBL:  equ #0000
NAMTBL:  equ #0800
NAMTBL1: equ #0400 ; In-game screen 1
NAMTBL2: equ #0C00 ; In-game screen 2

; Ports
VDPDAT: equ #98 ; TMS9918 Port #0
VDPCTL: equ #99 ; TMS9918 Port #1
SLOTSL: equ #A8
KEYROW: equ #A9
KEYSEL: equ #AA
PSGREG: equ #A0
PSGWRI: equ #A1
PSGREA: equ #A2
OPLLRG: equ #7C
OPLLDT: equ #7D

; PSG Registers
PSG_REG_TONE_0: equ 0
PSG_REG_TONE_1: equ 1
PSG_REG_TONE_2: equ 2
PSG_REG_TONE_3: equ 3
PSG_REG_TONE_4: equ 4
PSG_REG_TONE_5: equ 5
PSG_REG_NOISE:  equ 6
PSG_REG_MIXER:  equ 7 ; Important note: bit 6 must be 0, and bit 7 must be 1.
PSG_REG_AMP_0:  equ 8
PSG_REG_AMP_1:  equ 9
PSG_REG_AMP_2:  equ 10
PSG_REG_ENV_0:  equ 11
PSG_REG_ENV_1:  equ 12
PSG_REG_ENV_2:  equ 13

; Constants
FRAME_WIDTH:          equ 40*6 ; Text mode / 2
FRAME_HEIGHT:         equ 24*8
NUM_ANGLE_UNITS:      equ 1024
ANGLE_UNIT_FOV:       equ NUM_ANGLE_UNITS/4
UNITS_PER_CELL:       equ 64
UNITS_PER_TILE:       equ 256
MAX_PORTAL_DEPTH:     equ 5
PLAYER_BOX_HALF_WIDTH:equ (UNITS_PER_CELL*16)/3

COLOUR_BLACK         equ 1
COLOUR_MEDIUM_GREEN  equ 2
COLOUR_LIGHT_GREEN   equ 3
COLOUR_DARK_BLUE     equ 4
COLOUR_LIGHT_BLUE    equ 5
COLOUR_DARK_RED      equ 6
COLOUR_CYAN          equ 7
COLOUR_MEDIUM_RED    equ 8
COLOUR_LIGHT_RED     equ 9
COLOUR_DARK_YELLOW   equ 10
COLOUR_LIGHT_YELLOW  equ 11
COLOUR_DARK_GREEN    equ 12
COLOUR_MAGENTA       equ 13
COLOUR_GREY          equ 14
COLOUR_WHITE         equ 15

TRANSLATION_FULLSTATE_000: equ   0
TRANSLATION_FULLSTATE_001: equ 100
TRANSLATION_FULLSTATE_010: equ  20
TRANSLATION_FULLSTATE_011: equ 120
TRANSLATION_FULLSTATE_100: equ   4
TRANSLATION_FULLSTATE_101: equ 104
TRANSLATION_FULLSTATE_110: equ  24
TRANSLATION_FULLSTATE_111: equ 124

ENABLE_STAT_COUNTERS: equ 0

  ; 48Kbyte ROM
  org #0000

  ds 16
  jp $

  ; Interrupt Service Routine
  ds #0038 - $
interrupt_handler:
  push af
  in a,(VDPCTL)
  ld a,1
  ld (frame_advance),a
  ; Increment the frameskip count
  ld a,(frames_to_skip)
  inc a
  ld (frames_to_skip),a
  pop af
  ei
  ret

update_game_logic:
  ; Update the game logic.

  ld a,(frames_to_skip)
  ld b,a
frameskip_loop:
  push bc

  ; Handle inputs
  in a,(KEYSEL)
  and #F0
  or 8 ; Select row 8
  out (KEYSEL),a
  in a,(KEYROW)
  bit 7,a ; Right
  call z,right_pressed
  bit 6,a ; Down
  call z,down_pressed
  bit 5,a ; Up
  call z,up_pressed
  bit 4,a ; Left
  call z,left_pressed

  ; Apply dynamics
  call move_camera_step

  pop bc
  djnz frameskip_loop

  xor a
  ld (frames_to_skip),a

  ret

top:
  ; Initialise RAM
  ld hl,RAM_START
  ld (hl),0
  ld de,RAM_START+1
  ld bc,RAM_END-RAM_START-1
  ldir

  ;ld de,nametable_buffer
  ;ld hl,test_frame
  ;ld bc,40*24
  ;ldir

  ; ----------------------------------------------------

  ld hl,(18*UNITS_PER_CELL+UNITS_PER_CELL/2)*16
  ld (camera_x),hl

  ld hl,(17*UNITS_PER_CELL+UNITS_PER_CELL/2)*16
  ld (camera_y),hl

  ; ----------------------------------------------------

  ei

mainloop:

  call wait_for_vblank

  ; Set nametable address for write
  ld hl,NAMTBL
  ld a,l
  out (VDPCTL),a
  ld a,h
  and 63
  or 64
  out (VDPCTL),a

  ld hl,nametable_buffer
  ld c,VDPDAT
  REPT 40*24
    outi ; 18
    ; This NOP is required on MSX2 due to V9938 longer access time in text mode.
    nop  ;  5
  ENDM

  IF ENABLE_STAT_COUNTERS
    ld hl,stat_counters_start
    ld (hl),0
    ld de,stat_counters_start+1
    ld bc,stat_counters_end-stat_counters_start-1
    ldir
  ENDIF

  call update_game_logic

  ld hl,column_heights+119
  ld (column_height_addr),hl

  call draw_main_view

  ; Scan the columns
  call scan_columns

  jp mainloop

wait_for_vblank:
  xor a
  ld (frame_advance),a
  ld a,(frame_advance)
  and a
  jr z,$-4
  ret

set_border_colour:
  and #0F
  or #F0
  out (VDPCTL),a
  ld a,#87
  out (VDPCTL),a
  ret

; ---------------------- Column Scanning Routines ----------------------

scan_columns:

  exx
  ld hl,nametable_buffer+12*40
  exx
  ld hl,column_heights

  ld b,40
scan_columns.loop:
  push bc
  call start_column_scan
  pop bc
  djnz scan_columns.loop

  ret

start_column_scan:
  ld a,(hl)
  inc hl
  ld b,(hl)
  inc hl
  ld c,(hl)
  REPT 2
    dec hl
  ENDM

  cp b
  jr c,start_column_scan.l1
  ; a>=b
  cp c
  jr nc,start_column_scan.l4
  ; c>a && a>=b
  ; order = CAB
  jp scan_order_CAB
start_column_scan.l4:
  ; a>=c
  ld a,b
  cp c
  jr nc,start_column_scan.l5
  ; a>=c && c>b
  ; order = ACB
  jp scan_order_ACB
start_column_scan.l5:
  ; a>=b && b>=c
  ; order = ABC
  jp scan_order_ABC
start_column_scan.l1:
  ; b>a
  cp c
  jr c,start_column_scan.l2
  ; b>a && a>=c
  ; order = BAC
  jp scan_order_BAC
start_column_scan.l2:
  ; c>a
  ld a,c
  cp b
  jr c,start_column_scan.l3
  ; c>=b && b>a
  ; order = CBA
  jp scan_order_CBA
start_column_scan.l3:
  ; b>c && c>a
  ; order = BCA
  jp scan_order_BCA

; A = reg b, B = reg c, C = reg d

scan_order_CAB: ; C>=A>=B
  GENERATE_SCAN d, b, c, 1, 0, 2, TRANSLATION_FULLSTATE_010, TRANSLATION_FULLSTATE_011
scan_order_ACB: ; A>=C>=B
  GENERATE_SCAN b, d, c, 1, 2, 0, TRANSLATION_FULLSTATE_010, TRANSLATION_FULLSTATE_110
scan_order_ABC: ; A>=B>=C
  GENERATE_SCAN b, c, d, 2, 1, 0, TRANSLATION_FULLSTATE_100, TRANSLATION_FULLSTATE_110
scan_order_BAC: ; B>=A>=C
  GENERATE_SCAN c, b, d, 2, 0, 1, TRANSLATION_FULLSTATE_100, TRANSLATION_FULLSTATE_101
scan_order_CBA: ; C>=B>=A
  GENERATE_SCAN d, c, b, 0, 1, 2, TRANSLATION_FULLSTATE_001, TRANSLATION_FULLSTATE_011
scan_order_BCA: ; B>=C>=A
  GENERATE_SCAN c, d, b, 0, 2, 1, TRANSLATION_FULLSTATE_001, TRANSLATION_FULLSTATE_101

; ---------------------- Input Handling and Dynamics ----------------------

; Input angle in BC
; Output sinus in BC
calc_camera_sinus:
  ld l,b
  ld h,0
  add hl,hl
  ld de,calc_camera_sinus.jump_table
  add hl,de
  ld e,(hl)
  inc hl
  ld d,(hl)
  ex de,hl
  jp (hl)
calc_camera_sinus.q0:
  ld l,c
  ld h,0
  ; Load
  ld bc,camera_move_sinus_table
  add hl,bc
  ld b,0
  ld c,(hl)
  ret
calc_camera_sinus.q1:
  ; Reverse
  ld a,c
  cpl
  ld l,a
  ld h,0
  ; Load
  ld bc,camera_move_sinus_table
  add hl,bc
  ld b,0
  ld c,(hl)
  ret
calc_camera_sinus.q2:
  ld l,c
  ld h,0
  ; Load
  ld bc,camera_move_sinus_table
  add hl,bc
  ld b,0
  ld c,(hl)
  ; Negate
  ld a,b
  cpl
  ld b,a
  ld a,c
  cpl
  ld c,a
  inc bc
  ret
calc_camera_sinus.q3:
  ; Reverse
  ld a,c
  cpl
  ld l,a
  ld h,0
  ; Load
  ld bc,camera_move_sinus_table
  add hl,bc
  ld b,0
  ld c,(hl)
  ; Negate
  ld a,b
  cpl
  ld b,a
  ld a,c
  cpl
  ld c,a
  inc bc
  ret
calc_camera_sinus.jump_table:
  dw calc_camera_sinus.q0,calc_camera_sinus.q1,calc_camera_sinus.q2,calc_camera_sinus.q3

; HL = Coordinate X
; DE = Coordinate Y
; Returns result in Z flag
check_position:
  ; Divide by 16*128=2048=256*8
  REPT 3
    sra h
  ENDM
  ld a,15
  and h
  ld h,a
  ex de,hl
  ; Divide by 16*128=2048=256*8
  REPT 3
    sra h
  ENDM
  ld a,15
  and h
  REPT 4
    rlca
  ENDM
  or d
  ld l,a
  ld h,0
  add hl,hl
  ld bc,maze
  add hl,bc
  ; Load the address
  inc hl
  bit 7,(hl)
  ret

check_x_move:
  ; Expand in X by box width by move direction
  ld de,+PLAYER_BOX_HALF_WIDTH
  bit 7,b
  jr z,$+2+3
  ld de,-PLAYER_BOX_HALF_WIDTH
  add hl,de
  push de

  push hl
  ex de,hl
  ; Expand in Y by box width in up direction
  ld hl,(camera_y)
  ld bc,-PLAYER_BOX_HALF_WIDTH
  add hl,bc
  ex de,hl
  call check_position
  jr nz,check_x_move.end
  pop hl

  push hl
  ex de,hl
  ; Expand in Y by box width in down direction
  ld hl,(camera_y)
  ld bc,+PLAYER_BOX_HALF_WIDTH
  add hl,bc
  ex de,hl
  call check_position

check_x_move.end:
  pop hl
  pop de
  push af
  and a
  sbc hl,de ; Undo the expansion
  pop af
  ret

check_y_move:
  ; Expand in Y by box width by move direction
  ld de,+PLAYER_BOX_HALF_WIDTH
  bit 7,b
  jr z,$+2+3
  ld de,-PLAYER_BOX_HALF_WIDTH
  add hl,de
  push de

  push hl
  ex de,hl
  ; Expand in X by box width in up direction
  ld hl,(camera_x)
  ld bc,-PLAYER_BOX_HALF_WIDTH
  add hl,bc
  call check_position
  jr nz,check_y_move.end
  pop hl

  push hl
  ex de,hl
  ; Expand in X by box width in down direction
  ld hl,(camera_x)
  ld bc,+PLAYER_BOX_HALF_WIDTH
  add hl,bc
  call check_position

check_y_move.end:
  pop hl
  pop de
  push af
  and a
  sbc hl,de ; Undo the expansion
  pop af
  ret

up_pressed:
  push af

  ; Cos
  ld bc,(camera_angle)
  ld a,b
  inc a
  and 3
  ld b,a
  call calc_camera_sinus
  ld hl,(camera_vel_x)
  add hl,bc
  ld (camera_vel_x),hl

  ; Sin
  ld bc,(camera_angle)
  call calc_camera_sinus
  ld hl,(camera_vel_y)
  add hl,bc
  ld (camera_vel_y),hl

  pop af
  ret

down_pressed:
  push af

  ; Cos
  ld bc,(camera_angle)
  ld a,b
  inc a
  and 3
  ld b,a
  call calc_camera_sinus
  ; Negate BC
  ld a,c
  cpl
  ld c,a
  ld a,b
  cpl
  ld b,a
  inc bc
  ld hl,(camera_vel_x)
  add hl,bc
  ld (camera_vel_x),hl

  ; Sin
  ld bc,(camera_angle)
  call calc_camera_sinus
  ; Negate BC
  ld a,c
  cpl
  ld c,a
  ld a,b
  cpl
  ld b,a
  inc bc
  ld hl,(camera_vel_y)
  add hl,bc
  ld (camera_vel_y),hl

  pop af
  ret

left_pressed:
  push af
  push hl
  ld hl,(camera_angle_velocity)
  ld bc,+2*4
  add hl,bc
  ld (camera_angle_velocity),hl
  pop hl
  pop af
  ret

right_pressed:
  push af
  push hl
  ld hl,(camera_angle_velocity)
  ld bc,-2*4
  add hl,bc
  ld (camera_angle_velocity),hl
  pop hl
  pop af
  ret

move_camera_step:
  ld de,(camera_angle_velocity)
  ld hl,(camera_angle_highprec)
  add hl,de
  ld (camera_angle_highprec),hl

  REPT 3
    sra h
    rr l
  ENDM
  ld a,(NUM_ANGLE_UNITS-1)>>8
  and h
  ld h,a
  ld (camera_angle),hl

  ld l,e
  ld h,d
  ; Round to inf
  bit 7,h
  ld bc,7
  jr nz,$+2+1
  add hl,bc
  REPT 3
    sra h
    rr l
  ENDM
  ex de,hl
  and a
  sbc hl,de
  ld (camera_angle_velocity),hl

  ld bc,(camera_vel_x)
  ; Round to zero
  bit 7,b
  jr z,$+2+1
  inc bc
  sra b
  rr c
  ld hl,(camera_x)
  add hl,bc
  call check_x_move
  jr nz,$+2+3
  ld (camera_x),hl

  ld bc,(camera_vel_y)
  ; Round to zero
  bit 7,b
  jr z,$+2+1
  inc bc
  sra b
  rr c
  ld hl,(camera_y)
  add hl,bc
  call check_y_move
  jr nz,$+2+3
  ld (camera_y),hl

  ld hl,(camera_vel_x)
  ld e,l
  ld d,h
  ; Round to inf
  bit 7,h
  ld bc,7
  jr nz,$+2+1
  add hl,bc
  REPT 3
    sra h
    rr l
  ENDM
  ex de,hl
  and a
  sbc hl,de
  ld (camera_vel_x),hl

  ld hl,(camera_vel_y)
  ld e,l
  ld d,h
  ; Round to inf
  bit 7,h
  ld bc,7
  jr nz,$+2+1
  add hl,bc
  REPT 3
    sra h
    rr l
  ENDM
  ex de,hl
  and a
  sbc hl,de
  ld (camera_vel_y),hl

  ret

; ---------------------- Portal Renderer ----------------------

; Expects:
;   STACK_FX0
;   STACK_FX1
;   STACK_FY0
;   STACK_FY1
;   STACK_SIZE+STACK_A0
;   STACK_SIZE+STACK_A1
enter_portal:
  ; Load the address of the first wall in the next room into IX
  ld l,(ix+WALL_PORTAL_ROOM+0)
  ld h,(ix+WALL_PORTAL_ROOM+1)
  exx
  ld l,(iy+STACK_CAMERA_X+0)
  ld h,(iy+STACK_CAMERA_X+1)
  ld e,(iy+STACK_CAMERA_Y+0)
  ld d,(iy+STACK_CAMERA_Y+1)
  exx

  ; If the wall is a portal:
  push iy
  push ix

  ld b,(hl) ; ROOM_NUM_WALLS
  inc hl

  ; Load the first wall index based on view direction quadrant
  ld a,(camera_angle+1)
  and 3
  rlca
  ld e,a
  ld d,0
  add hl,de

  ld a,(hl) ; ROOM_FIRST_WALL
  inc hl
  ld h,(hl)
  ld l,a
  REPT 3 ; WALL_SIZE
    add hl,hl
  ENDM
  ex de,hl
  ld ix,walls
  add ix,de

  ; Push the portal stack
  ld de,STACK_SIZE
  add iy,de
  exx
  ; Pass camera X by value
  ld (iy+STACK_CAMERA_X+0),l
  ld (iy+STACK_CAMERA_X+1),h
  ; Pass camera Y by value
  ld (iy+STACK_CAMERA_Y+0),e
  ld (iy+STACK_CAMERA_Y+1),d
  exx
  ; A0 and A1 are already set.
  call draw_room_through_portal
  pop ix
  pop iy
  ret

;
; Stack frame layout:
;
;   16  Camera X (Must be on stack because it can change when traversing across tiles)
;   16  Camera Y
;   16  Ray 0 angle
;   16  Ray 1 angle
;   16  Current wall address
;   16  Camera-relative wall X0
;   16  Camera-relative wall Y0
;   16  Camera-relative wall X1
;   16  Camera-relative wall Y1
;   16  log_adjacent
;    8  Ray 0 intersection test flags
;    8  Ray 1 intersection test flags
;    8  ramp_index
;    8  Projected X0 (These should be placed near the end for negative IY offsets)
;   16  Projected Y0
;    8  Projected X1
;   16  Projected Y1
;    8  Vertex 0 angle
;    8  Vertex 1 angle
;   16  Ray 0 opposite
;   16  Ray 1 opposite
;
; Some stuff can be stored on the "real" stack too, of course.
;

draw_main_view:

  xor a
  ld (global_portal_traversal_flags),a

  ld iy,portal_stack_dummy
  ld (iy+STACK_FX0),FRAME_WIDTH-1
  ld (iy+STACK_FX1),0

  ld iy,portal_stack

  ld hl,(camera_angle)
  ld bc,-ANGLE_UNIT_FOV/2
  add hl,bc
  ld a,(NUM_ANGLE_UNITS-1)>>8
  and h
  ld h,a
  ld (iy+STACK_A0+0),l
  ld (iy+STACK_A0+1),h

  ld hl,(camera_angle)
  ld bc,+ANGLE_UNIT_FOV/2-1
  add hl,bc
  ld a,(NUM_ANGLE_UNITS-1)>>8
  and h
  ld h,a
  ld (iy+STACK_A1+0),l
  ld (iy+STACK_A1+1),h

  ld hl,(camera_y)
  ; Divide by 16
  REPT 4
    sra h
    rr l
  ENDM
  ld a,4095>>8
  and h
  ld h,a
  ; Negate
  ld a,l
  cpl
  ld l,a
  ld a,h
  cpl
  ld h,a
  inc hl
  ld (iy+STACK_CAMERA_Y+0),l
  ld (iy+STACK_CAMERA_Y+1),h

  ld hl,(camera_x)
  ; Divide by 16
  REPT 4
    sra h
    rr l
  ENDM
  ld a,4095>>8
  and h
  ld h,a
  ; Negate
  ld a,l
  cpl
  ld l,a
  ld a,h
  cpl
  ld h,a
  inc hl
  ld (iy+STACK_CAMERA_X+0),l
  ld (iy+STACK_CAMERA_X+1),h

  ; Determine the starting room
  ; Divide by 16*128=2048=256*8
  ld hl,(camera_x)
  REPT 3
    sra h
  ENDM
  ld a,15
  and h
  ld h,a
  ex de,hl
  ld hl,(camera_y)
  ; Divide by 16*128=2048=256*8
  REPT 3
    sra h
  ENDM
  ld a,15
  and h
  REPT 4
    rlca
  ENDM
  or d
  ld l,a
  ld h,0
  add hl,hl
  ld bc,maze
  add hl,bc
  ; Load the address
  ld e,(hl)
  inc hl
  ld d,(hl)

  bit 7,d
  ret nz

  ex de,hl

  ld b,(hl) ; ROOM_NUM_WALLS
  inc hl

  ; Load the first wall index based on view direction quadrant
  ld a,(camera_angle+1)
  and 3
  rlca
  ld e,a
  ld d,0
  add hl,de

  ld a,(hl) ; ROOM_FIRST_WALL
  inc hl
  ld h,(hl)
  ld l,a
  REPT 3 ; WALL_SIZE
    add hl,hl
  ENDM
  ex de,hl
  ld ix,walls
  add ix,de

  call draw_room_through_portal

  ret

draw_room_through_portal:

  ld hl,portal_depth
  ld a,MAX_PORTAL_DEPTH
  cp (hl)
  ret z
  inc (hl)

  push bc

  ; Get ray intersection test flags
  ld bc,ray_intersection_test_table ; 11c

  ld l,(iy+STACK_A0+0)              ; 21c
  ld h,(iy+STACK_A0+1)              ; 21c
  add hl,bc                         ; 12c
  ld a,(hl)                         ;  8c
  ld (iy+STACK_ITESTFLAGS0),a       ; 21c

  ld l,(iy+STACK_A1+0)              ; 21c
  ld h,(iy+STACK_A1+1)              ; 21c
  add hl,bc                         ; 12c
  ld d,(hl)                         ;  8c
  ld (iy+STACK_ITESTFLAGS1),d       ; 21c

  ; Get bitwise AND of intersection test flags
  and d                             ;  5c
  ld (iy+STACK_ITESTFLAGS),a        ; 21c

  pop bc

  ; Calculate location in cameraspace of first vertex

  push ix

  ld l,b
  dec l
  ld h,0
  REPT 3
    add hl,hl
  ENDM
  ex de,hl
  add ix,de

  ld e,(iy+STACK_CAMERA_X+0)  ; 21c   Camera X (negated)
  ld d,(iy+STACK_CAMERA_X+1)  ; 21c

  ; Calculate x0
  ld l,(ix+WALL_X1+0)         ; 21c
  ld h,(ix+WALL_X1+1)         ; 21c
  add hl,de                   ; 12c
  ld (iy+STACK_X0+0),l        ; 21c
  ld (iy+STACK_X0+1),h        ; 21c

  ld e,(iy+STACK_CAMERA_Y+0)  ; 21c   Camera Y (negated)
  ld d,(iy+STACK_CAMERA_Y+1)  ; 21c

  ; Calculate y0
  ld l,(ix+WALL_Y1+0)         ; 21c
  ld h,(ix+WALL_Y1+1)         ; 21c
  add hl,de                   ; 12c
  ld (iy+STACK_Y0+0),l        ; 21c
  ld (iy+STACK_Y0+1),h        ; 21c

  pop ix

  ; Pre-load FX1 and FY1 from the outer portal
  ld a,(iy-STACK_SIZE+STACK_FX0)
  ld (iy+STACK_FX1),a
  ld a,(iy-STACK_SIZE+STACK_FY0)
  ld (iy+STACK_FY1),a

  ; Pre-load the inner A1 too
  ld a,(iy+STACK_A1+0)
  ld (iy+STACK_SIZE+STACK_A1+0),a
  ld a,(iy+STACK_A1+1)
  ld (iy+STACK_SIZE+STACK_A1+1),a

  ; Clear flags
  ld (iy+STACK_FLAGS),0

  ; Loop through the walls in this room
draw_room_through_portal.loop:
  push bc
  push ix

  ld e,(iy+STACK_CAMERA_X+0)  ; 21c   Camera X (negated)
  ld d,(iy+STACK_CAMERA_X+1)  ; 21c

  ; Calculate x1
  ld l,(ix+WALL_X1+0)         ; 21c
  ld h,(ix+WALL_X1+1)         ; 21c
  add hl,de                   ; 12c
  ld (iy+STACK_X1+0),l        ; 21c
  ld (iy+STACK_X1+1),h        ; 21c

  ld e,(iy+STACK_CAMERA_Y+0)  ; 21c   Camera Y (negated)
  ld d,(iy+STACK_CAMERA_Y+1)  ; 21c

  ; Calculate y1
  ld l,(ix+WALL_Y1+0)         ; 21c
  ld h,(ix+WALL_Y1+1)         ; 21c
  add hl,de                   ; 12c
  ld (iy+STACK_Y1+0),l        ; 21c
  ld (iy+STACK_Y1+1),h        ; 21c

  ; Draw a wall
  call draw_wall

  ; Copy x1 to x0
  ld l,(iy+STACK_X1+0)        ; 21c
  ld h,(iy+STACK_X1+1)        ; 21c
  ld (iy+STACK_X0+0),l        ; 21c
  ld (iy+STACK_X0+1),h        ; 21c

  ld a,(iy+STACK_FX1)
  ld (iy+STACK_FX0),a

  ; Copy y1 to y0
  ld l,(iy+STACK_Y1+0)        ; 21c
  ld h,(iy+STACK_Y1+1)        ; 21c
  ld (iy+STACK_Y0+0),l        ; 21c
  ld (iy+STACK_Y0+1),h        ; 21c

  ld a,(iy+STACK_FY1)
  ld (iy+STACK_FY0),a

  pop ix
  pop bc

  bit 1,(iy+STACK_FLAGS)
  jr nz,draw_room_through_portal.early_out

  ld de,WALL_SIZE
  add ix,de
  djnz draw_room_through_portal.loop

draw_room_through_portal.early_out:
  ld hl,portal_depth
  dec (hl)

  ret             ; 11c

; IY = Portal traversal stack frame
; IX = Wall data base address
draw_wall:

  IF ENABLE_STAT_COUNTERS
    push af
    ld a,(stat_num_walls_drawn)
    inc a
    ld (stat_num_walls_drawn),a
    pop af
  ENDIF

  ; Call the routine corresponding to this wall's orientation
  ; The address of the routine is pre-stored in the wall data.
  ld l,(ix+WALL_ROUTINE+0)     ; 21c
  ld h,(ix+WALL_ROUTINE+1)     ; 21c
  jp (hl)         ;  5c

draw_wall_orientation_0_specialcase:
  bit 7,(ix+WALL_PORTAL_ROOM+1)
  ret nz

  ; Check that the camera is really inside the wall
  bit 7,(iy+STACK_X0+1)
  ret z
  bit 7,(iy+STACK_X1+1)
  ret nz

  ld a,(iy+STACK_A0+0)
  ld (iy+STACK_SIZE+STACK_A0+0),a
  ld a,(iy+STACK_A0+1)
  ld (iy+STACK_SIZE+STACK_A0+1),a

  ld a,(iy+STACK_A1+0)
  ld (iy+STACK_SIZE+STACK_A1+0),a
  ld a,(iy+STACK_A1+1)
  ld (iy+STACK_SIZE+STACK_A1+1),a

  ld (iy+STACK_FX0),FRAME_WIDTH-1
  ld (iy+STACK_FX1),0

  jp enter_portal

draw_wall_orientation_0:

  ; Apply frustum culling
  bit 0,(iy+STACK_ITESTFLAGS)
  ret nz

  ; Apply backface culling
  ld h,(iy+STACK_Y1+1)        ; 21c
  ld l,(iy+STACK_Y1+0)        ; 21c

  ; If the camera is exactly on this wall's line then go to
  ; the special case handling.
  ld a,h
  or l
  jr z,draw_wall_orientation_0_specialcase

  bit 7,h
  ret z

  ; Lookup log_adjacent
  ; hl ϵ (-4096, 0)
  ld (iy+STACK_ADJACENT+0),l  ; 21c
  ld (iy+STACK_ADJACENT+1),h  ; 21c
  ; Calculate abs(adjacent)
  ; Negate HL since it is known to be negative.
  ld a,h
  cpl
  ld h,a
  ld a,l
  cpl
  ld l,a
  inc hl
  ; hl ϵ (0, 4096)
  add hl,hl           ; 12c
  ld bc,log_table     ; 11c
  add hl,bc           ; 12c
  ld c,(hl)
  inc hl
  ld b,(hl)
  ; bc ϵ [0, 4096)
  ld (iy+STACK_LOG_ADJACENT+0),c         ; 21c
  ld (iy+STACK_LOG_ADJACENT+1),b         ; 21c

  DRAW_WALL_ORIENTATION_N 0,STACK_X0,STACK_X1,STACK_X0,STACK_X1

draw_wall_orientation_1:

  ; Apply frustum culling
  bit 1,(iy+STACK_ITESTFLAGS)
  ret nz

  ; Apply backface culling
  ld h,(iy+STACK_X1+1)        ; 21c
  bit 7,h
  ret nz

  ld l,(iy+STACK_X1+0)        ; 21c

  ; Lookup log_adjacent
  ; hl ϵ (-4096, 0)
  ld (iy+STACK_ADJACENT+0),l        ; 21c
  ld (iy+STACK_ADJACENT+1),h        ; 21c
  ; HL is known to be non-negative already.
  ; hl ϵ (0, 4096)
  add hl,hl           ; 12c
  ld bc,log_table     ; 11c
  add hl,bc           ; 12c
  ld c,(hl)
  inc hl
  ld b,(hl)
  ; bc ϵ [0, 4096)
  ld (iy+STACK_LOG_ADJACENT+0),c         ; 21c
  ld (iy+STACK_LOG_ADJACENT+1),b         ; 21c

  DRAW_WALL_ORIENTATION_N 1,STACK_Y0,STACK_Y1,STACK_Y0,STACK_Y1

draw_wall_orientation_2:

  ; Apply frustum culling
  bit 2,(iy+STACK_ITESTFLAGS)
  ret nz

  ; Apply backface culling
  ld h,(iy+STACK_Y1+1)        ; 21c
  bit 7,h
  ret nz

  ld l,(iy+STACK_Y1+0)        ; 21c

  ; Lookup log_adjacent
  ; hl ϵ [0, 4096)
  ld (iy+STACK_ADJACENT+0),l        ; 21c
  ld (iy+STACK_ADJACENT+1),h        ; 21c
  ; HL is known to be non-negative already.
  ; hl ϵ [0, 4096)
  add hl,hl           ; 12c
  ld bc,log_table     ; 11c
  add hl,bc           ; 12c
  ld c,(hl)
  inc hl
  ld b,(hl)
  ; bc ϵ [0, 4096)
  ld (iy+STACK_LOG_ADJACENT+0),c         ; 21c
  ld (iy+STACK_LOG_ADJACENT+1),b         ; 21c

  DRAW_WALL_ORIENTATION_N 2,STACK_X1,STACK_X0,STACK_X0,STACK_X1

draw_wall_orientation_3_specialcase:
  bit 7,(ix+WALL_PORTAL_ROOM+1)
  ret nz
  
  ; Check that the camera is really inside the wall
  bit 7,(iy+STACK_Y1+1)
  ret z
  bit 7,(iy+STACK_Y0+1)
  ret nz

  ld a,(iy+STACK_A0+0)
  ld (iy+STACK_SIZE+STACK_A0+0),a
  ld a,(iy+STACK_A0+1)
  ld (iy+STACK_SIZE+STACK_A0+1),a

  ld a,(iy+STACK_A1+0)
  ld (iy+STACK_SIZE+STACK_A1+0),a
  ld a,(iy+STACK_A1+1)
  ld (iy+STACK_SIZE+STACK_A1+1),a

  ld (iy+STACK_FX0),FRAME_WIDTH-1
  ld (iy+STACK_FX1),0

  jp enter_portal

draw_wall_orientation_3:

  ; Apply frustum culling
  bit 3,(iy+STACK_ITESTFLAGS)
  ret nz

  ; Apply backface culling
  ld h,(iy+STACK_X1+1)        ; 21c
  ld l,(iy+STACK_X1+0)        ; 21c

  ; If the camera is exactly on this wall's line then go to
  ; the special case handling.
  ld a,h
  or l
  jr z,draw_wall_orientation_3_specialcase

  bit 7,h
  ret z

  ; Lookup log_adjacent
  ; hl ϵ (-4096, 0)
  ld (iy+STACK_ADJACENT+0),l        ; 21c
  ld (iy+STACK_ADJACENT+1),h        ; 21c
  ; Calculate abs(adjacent)
  ; Negate HL since it is known to be negative.
  ld a,h
  cpl
  ld h,a
  ld a,l
  cpl
  ld l,a
  inc hl
  ; hl ϵ (0, 4096)
  add hl,hl           ; 12c
  ld bc,log_table     ; 11c
  add hl,bc           ; 12c
  ld c,(hl)
  inc hl
  ld b,(hl)
  ; bc ϵ [0, 4096)
  ld (iy+STACK_LOG_ADJACENT+0),c         ; 21c
  ld (iy+STACK_LOG_ADJACENT+1),b         ; 21c

  DRAW_WALL_ORIENTATION_N 3,STACK_Y1,STACK_Y0,STACK_Y0,STACK_Y1



; ---------------------- Misc. lookup tables ----------------------

log_partial_sum_table:
  incbin "log_partial_sum_table.bin"

  ds #4000 - $

; ---------------------- MSX ROM header for BIOS ----------------------

  org #4000

  db "AB"
  dw INIT
  db 0,0,0,0,0,0,0,0,0,0,0,0

INIT:
  di
  im 1 ; Interrupt mode 1 - RST #38
  ld sp,#F000

  xor a
  ld (CLIKSW),a

  ; Clear all of VRAM
  ld hl,#0000
  ld bc,#4000
  xor a
  call FILVRM

  ; Switch to Text mode
  ld hl,NAMTBL
  ld (TXTNAM),hl
  ld hl,PATTBL
  ld (TXTCGP),hl
  call INITXT

  ; Set foreground and background colour VDP register
  ; Set blank nametable base address
  ld a,(COLOUR_DARK_BLUE<<4)|COLOUR_WHITE
  out (VDPCTL),a
  ld a,#87
  out (VDPCTL),a

  ; Copy the charset patterns to VRAM
  ld de,PATTBL
  ld hl,charset
  ld bc,256*8
  call LDIRVM

  di

  ; Switch pages 0 and 2 to the (primary) slot in which this
  ; cartridge is inserted. It is assumed that page 1 is already mapped.
  call RSLREG
  ld b,a
  ;    33221100
  and %00001100
  ld c,a
  REPT 4
    rlca
  ENDM
  or c
  REPT 2
    rrca
  ENDM
  ld c,a
  ld a,b
  ;    33221100
  and %11001100
  or c
  ; Now, since WSLREG is mapped into page 0 currently it can't be called.
  ; So the slot select register port is used directly instead.
  out (SLOTSL),a

  ; Page 3 is kept as RAM, making 16Kbyte of total RAM available.
  ; WARNING: After this point, the BIOS is not available at all.
  jp top

  db "Copyright (c) Edd Biddulph 2025"

generate_wall:
  ; Look up the ramp descriptor address
  ld l,(iy+STACK_RELATIVE_ANGLE+0)
  ld h,(iy+STACK_RELATIVE_ANGLE+1)
  ; hl ϵ [-65536, +65535)
  ld a,(NUM_ANGLE_UNITS/2-1)>>8
  and h
  ld h,a
  ; hl ϵ [0, +512)
  ; Multiply HL by two
  add hl,hl
  ld bc,logsin_table ; Range is [0, +512)
  add hl,bc
  ld e,(hl)
  inc hl
  ld d,(hl)
  ld l,(iy+STACK_LOG_ADJACENT+0)
  ld h,(iy+STACK_LOG_ADJACENT+1)
  ex de,hl
  and a
  sbc hl,de

  ; Get the ramp slope address
  ld de,ramp_slope_table_L2R ; Range is [0, +1024)
  ; log(quotient) = log(sin) - log(adjacent)
  ; hl ϵ [-4096, +4096)
  ; Clamp HL to zero
  bit 7,h
  jr z,$+2+3
  ld hl,#0000
  ; hl ϵ [0, +4096)
  ; Divide HL by two
  srl h
  rr l
  ; hl ϵ [0, +2048)
  ; Align HL to words
  ld a,#FE    ;  8c
  and l       ;  5c
  ld l,a      ;  5c
  add hl,de
  ; Load the ramp slope from the table
  ld c,(hl)
  inc hl
  ld b,(hl)

  ; Set the column height for each wall column, left-to-right.

  ; Calculate wall width
  ld d,(iy+STACK_FX1)
  ld a,(iy+STACK_FX0)
  sub d
  ; Divide width by 2 (rounded to nearest)
  inc a
  srl a
  ret z

  bit 1,(iy+STACK_RELATIVE_ANGLE+1)
  jr nz,generate_wall_R2L
  jp generate_wall_L2R


; Right-to-left
generate_wall_R2L:
  ; Walls are consistently drawn right-to-left, so all we need is
  ; a single pointer that progresses right-to-left across the column
  ; heights array.
  ld de,(column_height_addr)
  ld h,(iy+STACK_FY0)
  ld l,0
  ; Divide Y position by 2
  srl h
  rr l

  ; Set one "empty" column to delineate the walls a bit better
  ex de,hl
  ld (hl),0
  dec hl
  ex de,hl
  add hl,bc
  dec a
  jr z,generate_wall_R2L.end

generate_wall_R2L.loop:
  ex af,af'
  ld a,h
  cp 48
  jr c,$+2+2
  ld a,48
  ld (de),a
  dec de
  add hl,bc
  ex af,af'
  dec a
  jr nz,generate_wall_R2L.loop

generate_wall_R2L.end:
  ld (column_height_addr),de

  ret

; Left-to-right
generate_wall_L2R:

  ; Walls are consistently drawn right-to-left, so all we need is
  ; a single pointer that progresses right-to-left across the column
  ; heights array.
  ld de,(column_height_addr)
  ld h,0
  ld l,a
  and a
  ex de,hl
  ; Set one "empty" column to delineate the walls a bit better
  ld (hl),0
  sbc hl,de
  ex de,hl
  ld (column_height_addr),de
  inc de
  ld h,(iy+STACK_FY1)
  ld l,0
  ; Divide Y position by 2
  srl h
  rr l
  ; Account for the "empty" column
  dec a
  ret z
generate_wall_L2R.loop:
  ex af,af'
  ld a,h
  cp 48
  jr c,$+2+2
  ld a,48
  ld (de),a
  inc de
  add hl,bc
  ex af,af'
  dec a
  jr nz,generate_wall_L2R.loop

  ret

; Reads:
;       STACK_LOG_ADJACENT
;       STACK_ADJACENT
;       STACK_ORIENTATION
;       STACK_LOG_OPPOSITE
;       STACK_OPPOSITE
; Writes:
;       STACK_FX1
;       STACK_FY1
project_post:

  IF ENABLE_STAT_COUNTERS
    push af
    ld a,(stat_num_project_posts)
    inc a
    ld (stat_num_project_posts),a
    pop af
  ENDIF

  ; X projection

  bit 0,(iy+STACK_ORIENTATION)
  jr nz,project_post.swap
  ; Lookup arctanlog_table
  ld c,(iy+STACK_LOG_OPPOSITE+0)
  ld b,(iy+STACK_LOG_OPPOSITE+1)
  ld l,(iy+STACK_LOG_ADJACENT+0)
  ld h,(iy+STACK_LOG_ADJACENT+1)
  ld e,(iy+STACK_OPPOSITE+1)
  ld d,(iy+STACK_ADJACENT+1)
  jr project_post.after_swap
project_post.swap:
  ; Swapped
  ld c,(iy+STACK_LOG_ADJACENT+0)
  ld b,(iy+STACK_LOG_ADJACENT+1)
  ld l,(iy+STACK_LOG_OPPOSITE+0)
  ld h,(iy+STACK_LOG_OPPOSITE+1)
  ld e,(iy+STACK_ADJACENT+1)
  ld d,(iy+STACK_OPPOSITE+1)
project_post.after_swap:

  push bc ; log_opposite will be used later.

  ; i=(log_adjacent-log_opposite+4096)/2
  and a
  sbc hl,bc
  ld bc,4096
  add hl,bc
  ld a,#FE
  and l
  ld l,a
  push hl ; This index will be re-used.
  ; Divide HL by two
  srl h
  rr l

  ; hl ϵ [0, +4096)
  ; Apply bounds restrictions
  ld a,h
  rrca
  xor h
  bit 2,a
  jr nz,project_post.clamp_end
  bit 2,h
  jr nz,project_post.upper_clamp
  ; Lower clamp
  ld hl,#0400
  jr project_post.clamp_end
project_post.upper_clamp:
  ; Upper clamp
  ld hl,#0C00
project_post.clamp_end:
  ; hl ϵ [1024, +3072]

  ld bc,-1024
  add hl,bc
  ld bc,arctanlog_table
  add hl,bc
  ; Load angle
  ld c,(hl)
  ld b,0
  ; bc ϵ [0, +256)

  ; Map the angle according to quadrant.
  bit 7,e
  jr z,project_post.opposite_nonneg
  ; If opposite < 0:
  bit 7,d
  jr z,project_post.adjacent_nonneg
  ;   If adjacent < 0:
  ;     angle=NUM_ANGLE_UNITS/2+angle
  ld hl,NUM_ANGLE_UNITS/2
  add hl,bc
  jr project_post.after_quadrants
  ;   Else
  ;     angle=NUM_ANGLE_UNITS/2-angle
project_post.adjacent_nonneg:
  ld hl,NUM_ANGLE_UNITS/2
  and a
  sbc hl,bc
  jr project_post.after_quadrants
  ;   Endif
  ; Else
project_post.opposite_nonneg:
  bit 7,d
  jr z,project_post.adjacent_nonneg2
  ;   If adjacent < 0:
  ;     angle=(NUM_ANGLE_UNITS-angle)&(NUM_ANGLE_UNITS-1)
  ld hl,NUM_ANGLE_UNITS
  and a
  sbc hl,bc
  jr project_post.after_quadrants
  ;   Else
project_post.adjacent_nonneg2:
  ld h,b
  ld l,c
  ;   Endif
  ; Endif
project_post.after_quadrants:

  ld a,(NUM_ANGLE_UNITS-1)>>8
  and h
  ld h,a

  ; hl ϵ [0, +1024)
  ; Store the angle in the next stack frame in case it is needed for
  ; portal traversal.
  ld (iy+STACK_SIZE+STACK_A1+0),l
  ld (iy+STACK_SIZE+STACK_A1+1),h

  ; angle = (angle - camera_angle) & (NUM_ANGLE_UNITS-1)
  ld bc,(camera_angle)
  and a
  sbc hl,bc
  ld a,(NUM_ANGLE_UNITS-1)>>8
  and h
  ld h,a

  ;if(angle > NUM_ANGLE_UNITS / 2)
  ;  abs_angle = NUM_ANGLE_UNITS - 1 - angle;
  bit 1,h
  jr z,project_post.abs_angle_no_remap
  ex de,hl
  ld hl,NUM_ANGLE_UNITS-1
  and a
  sbc hl,de
  ; hl ϵ [0, +256)
  ld d,l
  ; Lookup x_angle_table
  ld bc,x_angle_table
  add hl,bc
  ld a,(hl)
  cpl
  ld (iy+STACK_FX1),a
  jr project_post.after_abs_angle_remap
project_post.abs_angle_no_remap:
  ; hl ϵ [0, +256)
  ld d,l
  ; Lookup x_angle_table
  ld bc,x_angle_table
  add hl,bc
  ld a,(hl)
  ld (iy+STACK_FX1),a
project_post.after_abs_angle_remap:

  ; Y projection

  ; log(cos)
  ld a,d
  cpl
  ld h,0
  ld l,a
  inc hl
  add hl,hl
  ld de,logsin_table
  add hl,de
  ld e,(hl)
  inc hl
  ld d,(hl)

  ; log_partial_sum
  pop hl ; Re-use the index computed earlier.
  ld bc,log_partial_sum_table
  add hl,bc
  ld c,(hl)
  inc hl
  ld b,(hl)

  ; log(c) * 2 = log(wall_height) / log(tangent_max) * 2047 * 2
  ld hl,3342*2
  and a
  sbc hl,de
  and a
  sbc hl,bc

  pop bc ; log_opposite
  and a
  sbc hl,bc

  ld de,exp_table

  push hl
  ; Clamp
  ld a,h
  and #F8
  jr z,$+2+3
  ld hl,#0800
  ; Exponentiation
  add hl,hl
  add hl,de
  ld a,(hl)
  ld (iy+STACK_FY1),a
  pop hl

  ret

project_post_clipped_a0:

  ; X projection

  bit 0,(iy+STACK_ORIENTATION)
  jr nz,project_post_clipped_a0.swap
  ld c,(iy+STACK_LOG_OPPOSITE+0)
  ld b,(iy+STACK_LOG_OPPOSITE+1)
  ld l,(iy+STACK_LOG_ADJACENT+0)
  ld h,(iy+STACK_LOG_ADJACENT+1)
  ld e,(iy+STACK_OPPOSITE+1)
  ld d,(iy+STACK_ADJACENT+1)
  jr project_post_clipped_a0.after_swap
project_post_clipped_a0.swap:
  ; Swapped
  ld c,(iy+STACK_LOG_ADJACENT+0)
  ld b,(iy+STACK_LOG_ADJACENT+1)
  ld l,(iy+STACK_LOG_OPPOSITE+0)
  ld h,(iy+STACK_LOG_OPPOSITE+1)
  ld e,(iy+STACK_ADJACENT+1)
  ld d,(iy+STACK_OPPOSITE+1)
project_post_clipped_a0.after_swap:

  push bc ; log_opposite will be used later.

  ; i=(log_adjacent-log_opposite+4096)/2
  and a
  sbc hl,bc
  ld bc,4096
  add hl,bc
  ld a,#FE
  and l
  ld l,a
  push hl ; This index will be re-used.

  ; Load angle
  ld l,(iy+STACK_A0+0)
  ld h,(iy+STACK_A0+1)

  ; hl ϵ [0, +1024)
  ; Store the angle in the next stack frame in case it is needed for
  ; portal traversal.
  ld (iy+STACK_SIZE+STACK_A0+0),l
  ld (iy+STACK_SIZE+STACK_A0+1),h

  ; angle = (angle - camera_angle) & (NUM_ANGLE_UNITS-1)
  ld bc,(camera_angle)
  and a
  sbc hl,bc
  ld a,(NUM_ANGLE_UNITS-1)>>8
  and h
  ld h,a

  ;if(angle > NUM_ANGLE_UNITS / 2)
  ;  abs_angle = NUM_ANGLE_UNITS - 1 - angle;
  bit 1,h
  jr z,project_post_clipped_a0.abs_angle_no_remap
  ex de,hl
  ld hl,NUM_ANGLE_UNITS-1
  and a
  sbc hl,de
  ; hl ϵ [0, +256)
  ld d,l
  jr project_post_clipped_a0.after_abs_angle_remap
project_post_clipped_a0.abs_angle_no_remap:
  ; hl ϵ [0, +256)
  ld d,l
project_post_clipped_a0.after_abs_angle_remap:

  PROJECT_POST_Y

  ret

project_post_clipped_a1:

  ; X projection

  bit 0,(iy+STACK_ORIENTATION)
  jr nz,project_post_clipped_a1.swap
  ld c,(iy+STACK_LOG_OPPOSITE+0)
  ld b,(iy+STACK_LOG_OPPOSITE+1)
  ld l,(iy+STACK_LOG_ADJACENT+0)
  ld h,(iy+STACK_LOG_ADJACENT+1)
  ld e,(iy+STACK_OPPOSITE+1)
  ld d,(iy+STACK_ADJACENT+1)
  jr project_post_clipped_a1.after_swap
project_post_clipped_a1.swap:
  ; Swapped
  ld c,(iy+STACK_LOG_ADJACENT+0)
  ld b,(iy+STACK_LOG_ADJACENT+1)
  ld l,(iy+STACK_LOG_OPPOSITE+0)
  ld h,(iy+STACK_LOG_OPPOSITE+1)
  ld e,(iy+STACK_ADJACENT+1)
  ld d,(iy+STACK_OPPOSITE+1)
project_post_clipped_a1.after_swap:

  push bc ; log_opposite will be used later.

  ; i=(log_adjacent-log_opposite+4096)/2
  and a
  sbc hl,bc
  ld bc,4096
  add hl,bc
  ld a,#FE
  and l
  ld l,a
  push hl ; This index will be re-used.

  ; Load angle
  ld l,(iy+STACK_A1+0)
  ld h,(iy+STACK_A1+1)

  ; hl ϵ [0, +1024)
  ; Store the angle in the next stack frame in case it is needed for
  ; portal traversal.
  ld (iy+STACK_SIZE+STACK_A1+0),l
  ld (iy+STACK_SIZE+STACK_A1+1),h

  ; Use FX1 from the outer portal
  ld a,(iy-STACK_SIZE+STACK_FX1)
  ld (iy+STACK_FX1),a

  ; angle = (angle - camera_angle) & (NUM_ANGLE_UNITS-1)
  ld bc,(camera_angle)
  and a
  sbc hl,bc
  ld a,(NUM_ANGLE_UNITS-1)>>8
  and h
  ld h,a

  ;if(angle > NUM_ANGLE_UNITS / 2)
  ;  abs_angle = NUM_ANGLE_UNITS - 1 - angle;
  bit 1,h
  jr z,project_post_clipped_a1.abs_angle_no_remap
  ex de,hl
  ld hl,NUM_ANGLE_UNITS-1
  and a
  sbc hl,de
  ; hl ϵ [0, +256)
  ld d,l
  jr project_post_clipped_a1.after_abs_angle_remap
project_post_clipped_a1.abs_angle_no_remap:
  ; hl ϵ [0, +256)
  ld d,l
project_post_clipped_a1.after_abs_angle_remap:

  PROJECT_POST_Y

  ret

; ---------------------- Tables ----------------------

; Note: Rooms and walls must come before #8000 due to how the flags work.
rooms:
  include "rooms.asm"
walls:
  include "walls.asm"
charset:
  incbin "charset.bin"
x_angle_table:
  incbin "x_angle_table.bin"
ray_intersection_test_table:
  incbin "ray_intersection_test_table.bin"
logtan_table:
  incbin "logtan_table.bin"
log_table:
  incbin "log_table.bin"
exp_table:
  incbin "exp_table.bin"
logsin_table:
  incbin "logsin_table.bin"
arctanlog_table:
  incbin "arctanlog_table.bin"
camera_move_sinus_table:
  incbin "camera_move_sinus_table.bin"
translation_table:
  incbin "translation_table.bin"
vram_name_table_addresses:
  dw NAMTBL1,NAMTBL2
maze:
  include "maze.asm"
ramp_slope_table_L2R:
  include "ramp_slope_table_L2R.asm"

  IF (charset + 2048) > #8000
    ERROR (charset + 2048) > #8000
  ENDIF

  ds #C000 - $

; ---------------------- Variables ----------------------

; 16Kbyte RAM
  org #C000
RAM_START:
portal_stack_dummy:
  ds virtual STACK_SIZE
portal_stack:
  ds virtual STACK_SIZE*(MAX_PORTAL_DEPTH+1)
wall_temp:
  ds virtual WALL_SIZE
camera_angle:
  ds virtual 2
camera_angle_highprec:
  ds virtual 2
camera_angle_velocity:
  ds virtual 2
camera_x: ; 12.4 fixed
  ds virtual 2
camera_y: ; 12.4 fixed
  ds virtual 2
camera_vel_x:
  ds virtual 2
camera_vel_y:
  ds virtual 2
portal_depth:
  ds virtual 1
frames_to_skip:
  ds virtual 1
rng_padding:
  ds virtual 1
rng_state:
  ds virtual 5
frame_advance:
  ds virtual 1
frame_counter:
  ds virtual 1
scan_heights:
  ds virtual 3
pending_vram_name_table_address:
  ds virtual 2
nametable_buffer:
  ds virtual 40*24
  ALIGN_VIRTUAL 256
column_heights:
  ds virtual 128
column_height_addr:
  ds virtual 2
stat_counters_start:
stat_num_project_posts:
  ds virtual 1
stat_num_clip_walls:
  ds virtual 1
stat_num_walls_drawn:
  ds virtual 1
wall_upper_rectangular_part_width:
  ds virtual 2
wall_lower_rectangular_part_width:
  ds virtual 2
global_portal_traversal_flags:
  ds virtual 1
stat_counters_end:
RAM_END:
  ds virtual #E000 - $

