; FRUIT MAZES

ALIGN: MACRO ?boundary
       ds ?boundary - 1 - ($ + ?boundary - 1) % ?boundary
       ENDM

ALIGN_VIRTUAL: MACRO ?boundary
       ds virtual (?boundary - 1 - ($ + ?boundary - 1) % ?boundary)
       ENDM

CONVERT_SUBPIXEL_TO_PIXEL: MACRO
        REPT SUBPIXEL_SHIFT
          sra h
          rr l
        ENDM
        ENDM

; BIOS ROM
CGTABL: equ #0004

; BIOS Routines
TOTEXT: equ #00D2
CHGCLR: equ #0062
GTSTCK: equ #00D5
SETWRT: equ #0053
CHGMOD: equ #005f
INITXT: equ #006C
LDIRVM: equ #005C
INIT32: equ #006F
INIGRP: equ #0072
FILVRM: equ #0056

; System Variables
FORCLR: equ #F3E9
BAKCLR: equ #F3EA
BDRCLR: equ #F3EB
TXTNAM: equ #F3B3
TXTCGP: equ #F3B7
HKEY:   equ #FD9A
CLIKSW: equ #F3DB
T32NAM: equ #F3BD
T32COL: equ #F3BF
T32CGP: equ #F3C1
T32ATR: equ #F3C3
T32PAT: equ #F3C5
GRPNAM: equ #F3C7
GRPCOL: equ #F3C9
GRPCGP: equ #F3CB
GRPATR: equ #F3CD
GRPPAT: equ #F3CF

; VRAM
PATTBL:  equ #0000
NAMTBL1: equ #0400 ; In-game screen 1
NAMTBL2: equ #0C00 ; In-game screen 2
SATRBL:  equ #1400
COLTBL:  equ #1480
SPATBL:  equ #1800
NAMTBL3: equ #2400 ; Title screen
NAMTBL4: equ #2C00 ; Blank screen (for loading wait)
NAMTBL5: equ #3400 ; End screen

; Ports
VDPDAT: equ #98
VDPCTL: equ #99
KEYSEL: equ #AA
KEYROW: equ #A9
PSGREG: equ #A0
PSGWRI: equ #A1
PSGREA: equ #A2

; PSG Registers
PSG_REG_TONE_0: equ 0
PSG_REG_TONE_1: equ 1
PSG_REG_TONE_2: equ 2
PSG_REG_TONE_3: equ 3
PSG_REG_TONE_4: equ 4
PSG_REG_TONE_5: equ 5
PSG_REG_NOISE:  equ 6
PSG_REG_MIXER:  equ 7 ; Important note: bit 6 must be 0, and bit 7 must be 1.
PSG_REG_AMP_0:  equ 8
PSG_REG_AMP_1:  equ 9
PSG_REG_AMP_2:  equ 10
PSG_REG_ENV_0:  equ 11
PSG_REG_ENV_1:  equ 12
PSG_REG_ENV_2:  equ 13


; Constants
POS_ACC              equ +2
NEG_ACC              equ -2
VIEWPORT_WIDTH       equ 24
VIEWPORT_HEIGHT      equ 24
CAMERA_SHIFT         equ 3
SUBPIXEL_SHIFT       equ 4
MAX_FRUITS           equ 32
FRUIT_BYTES          equ 8
NAMES_BASE           equ (128*8+32*24)/8
HUD_STAGE_X          equ 5
HUD_STAGE_Y          equ 2
HUD_FRUIT_X          equ 5
HUD_FRUIT_Y          equ 5
HUD_TIME_X           equ 6
HUD_TIME_Y           equ 8
HUD_WIDTH            equ 32-VIEWPORT_WIDTH
HUD_HEIGHT           equ 24
HUD_MSG_WIDTH        equ HUD_WIDTH
HUD_MSG_HEIGHT       equ 12
COLOUR_BLACK         equ 1
COLOUR_MEDIUM_GREEN  equ 2
COLOUR_LIGHT_GREEN   equ 3
COLOUR_DARK_BLUE     equ 4
COLOUR_LIGHT_BLUE    equ 5
COLOUR_DARK_RED      equ 6
COLOUR_CYAN          equ 7
COLOUR_MEDIUM_RED    equ 8
COLOUR_LIGHT_RED     equ 9
COLOUR_DARK_YELLOW   equ 10
COLOUR_LIGHT_YELLOW  equ 11
COLOUR_DARK_GREEN    equ 12
COLOUR_MAGENTA       equ 13
COLOUR_GREY          equ 14
COLOUR_WHITE         equ 15
FRUIT_SPRITE_NAME    equ 128
FRUIT_SPRITE_COLOUR  equ COLOUR_LIGHT_RED
LEAF_SPRITE_NAME     equ 128+8
LEAF_SPRITE_COLOUR   equ COLOUR_LIGHT_GREEN
HAT_SPRITE_NAME      equ 128+8+8
HAT_SPRITE_COLOUR    equ COLOUR_MAGENTA
UFO_SPRITE_NAME      equ 128+8+8+1
UFO_SPRITE_COLOUR    equ COLOUR_LIGHT_YELLOW
PARTICLE_SPRITE_NAME equ 128+8+8+1+4
PARTICLE_SPRITE_COLOUR equ COLOUR_CYAN
NUM_PARTICLES        equ 8
MAX_PARTICLE_STEPS   equ 80
DEBUG_ENABLED:       equ 0

; 16Kbyte ROM
  org #4000

  db "AB"
  dw INIT
  db 0,0,0,0,0,0,0,0,0,0,0,0

INIT:
  di
  xor a
  ld (CLIKSW),a

  ; Switch to Graphics 1 mode
  ld hl,NAMTBL1
  ld (T32NAM),hl
  ld hl,PATTBL
  ld (T32CGP),hl
  ld hl,COLTBL
  ld (T32COL),hl
  ld hl,SATRBL
  ld (T32ATR),hl
  ld hl,SPATBL
  ld (T32PAT),hl
  call INIT32

  ; Clear all of VRAM
  ld hl,#0000
  ld bc,#4000
  xor a
  call FILVRM

  IF 1
  ; Decompress patterns into temporary storage in RAM
  ld hl,patterns_compressed
  ld b,#A0
  call decompress_rle
  ; Copy patterns to VRAM
  REPT 8,?I
    ld de,256*8*?I
    ld hl,rle_output+128*8*?I
    ld bc,128*8
    call LDIRVM
  ENDM
  ELSE
  ; Copy patterns to VRAM
  REPT 8,?I
    ld de,256*8*?I
    ld hl,patterns+128*8*?I
    ld bc,128*8
    call LDIRVM
  ENDM
  ENDIF

  ; Copy sprite patterns to VRAM
  ld de,SPATBL+8*FRUIT_SPRITE_NAME
  ld hl,fruit_patterns
  ld bc,8*8
  call LDIRVM
  ld de,SPATBL+8*LEAF_SPRITE_NAME
  ld hl,leaf_patterns
  ld bc,8*8
  call LDIRVM
  ld de,SPATBL+8*HAT_SPRITE_NAME
  ld hl,hat_pattern
  ld bc,8
  call LDIRVM
  ld de,SPATBL+8*UFO_SPRITE_NAME
  ld hl,ufo_patterns
  ld bc,8*4
  call LDIRVM
  ld de,SPATBL+8*PARTICLE_SPRITE_NAME
  ld hl,particle_patterns
  ld bc,8*4
  call LDIRVM

  ; Initialise RAM
  ld hl,RAM_START
  ld (hl),0
  ld de,RAM_START+1
  ld bc,RAM_END-RAM_START-1
  ldir

  ; Generate code
  call generate_outi_unrolled_loop
  call generate_staging_loop

  ; Copy BIOS charset to VRAM
  ld hl,char_indices
  ld de,char_buffer
bios_charset_copy_loop:
  ld a,(hl)
  inc hl
  ld b,0
  ld c,a
  ; Multiply BC by 8
  REPT 3
    sla c
    rl b
  ENDM
  push hl
  push de
  ld hl,(CGTABL)
  add hl,bc
  ld bc,8
  ldir
  pop de
  pop hl
  ld a,(char_indices_end-char_indices-1)*8
  cp e
  ; Add 8 to DE
  REPT 8
    inc de
  ENDM
  jr nz,bios_charset_copy_loop
  REPT 8,?I
    ld de,256*8*?I+NAMES_BASE*8
    ld hl,char_buffer
    ld bc,(char_indices_end-char_indices)*8
    call LDIRVM
  ENDM

  ; Fill colour table in VRAM
  ld de,COLTBL
  ld hl,default_colour_table
  ld bc,32
  call LDIRVM

  ; Fill nametables with blank pattern
  ld hl,NAMTBL1
  ld bc,32*24
  ld a,0
  call FILVRM
  ld hl,NAMTBL2
  ld bc,32*24
  ld a,0
  call FILVRM

  ;  Copy titlescreen nametable to VRAM
  ld hl,titlescreen_nametable_compressed
  ld b,(rle_output>>8)+3
  call decompress_rle
  ld de,NAMTBL3
  ld hl,rle_output
  ld bc,32*24
  call LDIRVM

  ; Initialise blank nametable
  ld hl,NAMTBL4
  ld bc,32*24
  ld a,0
  call FILVRM

  ;  Copy endscreen nametable to VRAM
  ld hl,endscreen_nametable_compressed
  ld b,(rle_output>>8)+3
  call decompress_rle
  ld de,NAMTBL5
  ld hl,rle_output
  ld bc,32*24
  call LDIRVM

  ; Initialise sprites
  ld hl,SATRBL
  ld bc,128
  ld a,#D0
  call FILVRM
  ld hl,sprite_buffer
  ld (hl),#D0
  ld de,sprite_buffer+1
  ld bc,127
  ldir

  ; Initialise HUD
  ld hl,hud_bg
  ld de,hud_buffer
  ld bc,HUD_WIDTH*HUD_HEIGHT
  ldir

  ; Initialise handler and transition addresses
  ld hl,frame_handler_titlescreen
  ld (frame_handler_address),hl
  ld hl,transition_nop
  ld (transition_address),hl

  ld hl,NAMTBL4
  ld (pending_vram_name_table_address),hl

  ld a,COLOUR_DARK_BLUE
  call set_border_colour

  call init_audio

  ; Switch to interrupt mode 2 (custom interrupt handler)
  ld a,#7E
  ld i,a
  im 2

  ; The main game loop
frameloop:
  call wait_for_vblank
  di

  ; Perform an indirect call for frame logic (and rendering)
  ld hl,(frame_handler_address)
  ld de,$+3+1+1
  push de
  jp (hl)

  ld hl,frame_counter
  inc (hl)

  IF DEBUG_ENABLED
  ld a,COLOUR_MEDIUM_GREEN
  call set_border_colour
  ENDIF

  ; Perform an indirect call for game state transition
  ld hl,(transition_address)
  ld de,$+3+1+1
  push de
  jp (hl)

  jr frameloop

wait_for_vblank:
  xor a
  ld (frame_advance),a
  ei
  ld a,(frame_advance)
  and a
  jr z,$-4
  ret

init_audio:
  ; Initialise the PSG
  REPT 14,?I
    ld a,?I
    out (PSGREG),a
    IF ?I=7
      xor %10111111
      out (PSGWRI),a
    ELSE
      xor a
      out (PSGWRI),a
    ENDIF
  ENDM
  ret

transition_nop:
  ret

init_particles:
  ld b,NUM_PARTICLES
  ld hl,particles
init_particles_loop:
  push bc
  ld bc,4<<SUBPIXEL_SHIFT
  ld de,(player_pos_x)
  ex de,hl
  add hl,bc
  ex de,hl
  ld (hl),e
  inc hl
  ld (hl),d
  inc hl
  ld de,(player_pos_y)
  ex de,hl
  add hl,bc
  ex de,hl
  ld (hl),e
  inc hl
  ld (hl),d
  inc hl
  pop bc
  djnz init_particles_loop
  ret

step_particles:
  ld hl,particle_step_counter
  ld a,MAX_PARTICLE_STEPS
  cp (hl)
  ret z
  inc (hl)
  REPT NUM_PARTICLES,?I
    ; X
    ld hl,(particles+?I*4+0)
    ld bc,(particle_vectors+?I*4+0)
    add hl,bc
    ld (particles+?I*4+0),hl
    ; Y
    ld hl,(particles+?I*4+2)
    ld bc,(particle_vectors+?I*4+2)
    add hl,bc
    ld (particles+?I*4+2),hl
  ENDM
  ret

transition_ingame_to_gameover:
  call init_audio
  call init_particles
  ; Clear the sprite buffer
  ld hl,sprite_buffer
  ld (hl),#D0
  ld de,sprite_buffer+1
  ld bc,127
  ldir
  ; Set pending addresses
  ld hl,frame_handler_gameover
  ld (frame_handler_address),hl
  ld hl,transition_nop
  ld (transition_address),hl
  ld a,128
  ld (explosion_sound_counter),a
  ret

transition_ingame_to_timeover:
  call init_audio
  call init_particles
  ; Clear the sprite buffer
  ld hl,sprite_buffer
  ld (hl),#D0
  ld de,sprite_buffer+1
  ld bc,127
  ldir
  ; Set pending addresses
  ld hl,frame_handler_timeover
  ld (frame_handler_address),hl
  ld hl,transition_nop
  ld (transition_address),hl
  ld a,128
  ld (explosion_sound_counter),a
  ret

copy_colour_table_to_vram:
  ; Write colour table to VRAM
  ld a,COLTBL&#FF         ; 8
  out (VDPCTL),a          ; 11
  ld a,((COLTBL>>8)&63)|64; 8
  out (VDPCTL),a          ; 11
  ld hl,colour_table ; 11
  ld c,VDPDAT             ; 8
  ld b,32
  otir
  ret

transition_gameover_to_ingame:
  call init_audio
  call wait_for_vblank
  di

  ; "Disable" sprites by terminating at the first sprite
  ld a,SATRBL&#FF         ; 8
  out (VDPCTL),a          ; 11
  ld a,((SATRBL>>8)&63)|64; 8
  out (VDPCTL),a          ; 11
  ld a,#D0
  out (VDPDAT),a

  ; Set blank nametable base address
  ld a,NAMTBL4>>10
  out (VDPCTL),a
  ld a,#82
  out (VDPCTL),a

  call begin_game

  call wait_for_vblank
  di

  ; Set the pending nametable address to the blank
  ; nametable, so that the first frame is blanked.
  ; Otherwise, the game will display garbage for one frame.
  ld hl,NAMTBL4
  ld (pending_vram_name_table_address),hl

  ld hl,sprite_buffer
  ld (hl),#D0

  ; Set pending addresses
  ld hl,frame_handler_ingame
  ld (frame_handler_address),hl
  ld hl,transition_nop
  ld (transition_address),hl
  ret

transition_timeover_to_ingame:
  jp transition_gameover_to_ingame

transition_timeover_to_titlescreen:
  jp transition_gameover_to_titlescreen

transition_ingame_to_stageclear:
  call init_audio
  ld hl,#0000
  ld (player_vel_x),hl
  ld (player_vel_y),hl

  ; Set pending addresses
  ld hl,frame_handler_stageclear
  ld (frame_handler_address),hl
  ld hl,transition_nop
  ld (transition_address),hl
  ld a,128
  ld (explosion_sound_counter),a
  ret

transition_stageclear_to_endscreen:
  call init_audio
  ld hl,#0000
  ld (player_vel_x),hl
  ld (player_vel_y),hl

  ; Set pending addresses
  ld hl,frame_handler_endscreen
  ld (frame_handler_address),hl
  ld hl,transition_nop
  ld (transition_address),hl
  ret

transition_stageclear_to_ingame:
  call init_audio
  ; If the last stage has just been cleared,
  ; the transition instead to the endscreen.
  ld a,(stage_index)
  cp 11
  jp z,transition_stageclear_to_endscreen

  call wait_for_vblank
  di

  ; "Disable" sprites by terminating at the first sprite
  ld a,SATRBL&#FF         ; 8
  out (VDPCTL),a          ; 11
  ld a,((SATRBL>>8)&63)|64; 8
  out (VDPCTL),a          ; 11
  ld a,#D0
  out (VDPDAT),a

  ; Set blank nametable base address
  ld a,NAMTBL4>>10
  out (VDPCTL),a
  ld a,#82
  out (VDPCTL),a

  call skip_stage

  call wait_for_vblank
  di

  ; Set the pending nametable address to the blank
  ; nametable, so that the first frame is blanked.
  ; Otherwise, the game will display garbage for one frame.
  ld hl,NAMTBL4
  ld (pending_vram_name_table_address),hl

  ld hl,sprite_buffer
  ld (hl),#D0

  ; Set pending addresses
  ld hl,frame_handler_ingame
  ld (frame_handler_address),hl
  ld hl,transition_nop
  ld (transition_address),hl
  ret

transition_titlescreen_to_ingame:
  call init_audio
  call wait_for_vblank
  di

  ; "Disable" sprites by terminating at the first sprite
  ld a,SATRBL&#FF         ; 8
  out (VDPCTL),a          ; 11
  ld a,((SATRBL>>8)&63)|64; 8
  out (VDPCTL),a          ; 11
  ld a,#D0
  out (VDPDAT),a

  ; Set blank nametable base address
  ld a,NAMTBL4>>10
  out (VDPCTL),a
  ld a,#82
  out (VDPCTL),a

  call begin_game

  call wait_for_vblank
  di

  ; Set the pending nametable address to the blank
  ; nametable, so that the first frame is blanked.
  ; Otherwise, the game will display garbage for one frame.
  ld hl,NAMTBL4
  ld (pending_vram_name_table_address),hl

  ld hl,sprite_buffer
  ld (hl),#D0

  ; Set pending addresses
  ld hl,frame_handler_ingame
  ld (frame_handler_address),hl
  ld hl,transition_nop
  ld (transition_address),hl
  ret

transition_gameover_to_titlescreen:
  call init_audio
  ; Set pending addresses
  ld hl,frame_handler_titlescreen
  ld (frame_handler_address),hl
  ld hl,transition_nop
  ld (transition_address),hl
  ret

frame_handler_titlescreen:

  ; Position some sprites
  ; Write colour table to VRAM
  ld a,SATRBL&#FF         ; 8
  out (VDPCTL),a          ; 11
  ld a,((SATRBL>>8)&63)|64; 8
  out (VDPCTL),a          ; 11

  ; Ghostbones
  ld a,#0B*8
  out (VDPDAT),a ; Y
  ld a,#0F*8
  out (VDPDAT),a ; X
  ld a,(frame_counter)
  sra a
  and 3
  add a,UFO_SPRITE_NAME
  out (VDPDAT),a ; Pattern
  ld a,UFO_SPRITE_COLOUR
  out (VDPDAT),a ; Colour

  ; Ghostbones' hat
  ld a,#0A*8
  out (VDPDAT),a ; Y
  ld a,#0F*8
  out (VDPDAT),a ; X
  ld a,HAT_SPRITE_NAME
  out (VDPDAT),a ; Pattern
  ld a,HAT_SPRITE_COLOUR
  out (VDPDAT),a ; Colour

  ; Fruit
  ld a,#0D*8
  out (VDPDAT),a ; Y
  ld a,#0F*8
  out (VDPDAT),a ; X
  ld a,FRUIT_SPRITE_NAME
  out (VDPDAT),a ; Pattern
  ld a,FRUIT_SPRITE_COLOUR
  out (VDPDAT),a ; Colour

  ; Leaf
  ld a,#0C*8
  out (VDPDAT),a ; Y
  ld a,#0F*8
  out (VDPDAT),a ; X
  ld a,LEAF_SPRITE_NAME
  out (VDPDAT),a ; Pattern
  ld a,LEAF_SPRITE_COLOUR
  out (VDPDAT),a ; Colour

  ; Terminate the sprite list
  ld a,#D0
  out (VDPDAT),a

  ; Set patterngenerator base address to scroll bricks
  ld a,(frame_counter)
  REPT 3
    sra a
  ENDM
  and 7
  ld b,0
  ld c,a
  ld hl,vram_pattern_table_addresses
  add hl,bc
  ld a,(hl)
  out (VDPCTL),a
  ld a,#84
  out (VDPCTL),a

  ; Set nametable base address to show titlescreen
  ld a,NAMTBL3>>10
  out (VDPCTL),a
  ld a,#82
  out (VDPCTL),a

  ; Make the blinking "PRESS SPACE" message
  ; PRESS
  ld a,(NAMTBL3+24+2+20*32)&#FF
  out (VDPCTL),a
  ld a,(((NAMTBL3+24+2+20*32)>>8)&63)|64
  out (VDPCTL),a
  ld b,5
  ld hl,hud_msg_gameover+2+8*8
  ld a,(frame_counter)
  and 8
  jr nz,$+2+3
  ld hl,hud_msg_gameover+2+7*8
  ld c,VDPDAT
  otir
  ; SPACE
  ld a,(NAMTBL3+24+2+21*32)&#FF
  out (VDPCTL),a
  ld a,(((NAMTBL3+24+2+21*32)>>8)&63)|64
  out (VDPCTL),a
  ld b,5
  ld hl,hud_msg_gameover+2+9*8
  ld a,(frame_counter)
  and 8
  jr nz,$+2+3
  ld hl,hud_msg_gameover+2+7*8
  ld c,VDPDAT
  otir

  ; Handle inputs
  in a,(KEYSEL)
  and #F0
  or 8
  out (KEYSEL),a
  in a,(KEYROW)
  xor #FF
  ld b,a
  ld hl,key_rows_state+8
  xor (hl)
  and b
  bit 0,a ; Spacebar
  jr z,$+2+3+3
  ld hl,transition_titlescreen_to_ingame
  ld (transition_address),hl
  ld hl,key_rows_state+8
  ld (hl),b

  ret


frame_handler_endscreen:

  ; Position some sprites
  ; Write colour table to VRAM
  ld a,SATRBL&#FF         ; 8
  out (VDPCTL),a          ; 11
  ld a,((SATRBL>>8)&63)|64; 8
  out (VDPCTL),a          ; 11

  ; Terminate the sprite list
  ld a,#D0
  out (VDPDAT),a

  ; Set nametable base address to show endscreen
  ld a,NAMTBL5>>10
  out (VDPCTL),a
  ld a,#82
  out (VDPCTL),a

  ; Make the blinking "PRESS SPACE" message
  ; PRESS
  ld a,(NAMTBL5+24+2+20*32)&#FF
  out (VDPCTL),a
  ld a,(((NAMTBL5+24+2+20*32)>>8)&63)|64
  out (VDPCTL),a
  ld b,5
  ld hl,hud_msg_gameover+2+8*8
  ld a,(frame_counter)
  and 8
  jr nz,$+2+3
  ld hl,hud_msg_gameover+2+7*8
  ld c,VDPDAT
  otir
  ; SPACE
  ld a,(NAMTBL5+24+2+21*32)&#FF
  out (VDPCTL),a
  ld a,(((NAMTBL5+24+2+21*32)>>8)&63)|64
  out (VDPCTL),a
  ld b,5
  ld hl,hud_msg_gameover+2+9*8
  ld a,(frame_counter)
  and 8
  jr nz,$+2+3
  ld hl,hud_msg_gameover+2+7*8
  ld c,VDPDAT
  otir

  ; Handle inputs
  in a,(KEYSEL)
  and #F0
  or 8
  out (KEYSEL),a
  in a,(KEYROW)
  xor #FF
  ld b,a
  ld hl,key_rows_state+8
  xor (hl)
  and b
  bit 0,a ; Spacebar
  jr z,$+2+3+3
  ld hl,transition_gameover_to_titlescreen
  ld (transition_address),hl
  ld hl,key_rows_state+8
  ld (hl),b
  ret

frame_handler_stageclear:
  ; Render the game
  call do_rendering
  ; Handle inputs
  in a,(KEYSEL)
  and #F0
  or 8
  out (KEYSEL),a
  in a,(KEYROW)
  xor #FF
  ld b,a
  ld hl,key_rows_state+8
  xor (hl)
  and b
  bit 0,a ; Spacebar
  jr z,$+2+3+3
  ld hl,transition_stageclear_to_ingame
  ld (transition_address),hl
  ld hl,key_rows_state+8
  ld (hl),b

  IF DEBUG_ENABLED
  ld a,COLOUR_MAGENTA
  call set_border_colour
  ENDIF

  call handle_camera_dynamics

  IF DEBUG_ENABLED
  ld a,COLOUR_CYAN
  call set_border_colour
  ENDIF

  call build_sprites_from_player_and_fruits

  IF DEBUG_ENABLED
  ld a,COLOUR_GREY
  call set_border_colour
  ENDIF

  ; Update HUD message in buffer
  ld a,(frame_counter)
  and 8
  call z,print_hud_nextstage_msg
  call nz,clear_hud_message

  ; "Stage clear" sound
  ld hl,explosion_sound_counter
  xor a
  or (hl)
  ret z
  dec (hl)

  ld a,(hl)
  and 15
  ld l,a
  ld h,0
  REPT 4
    add hl,hl
  ENDM
  ex de,hl
  ld hl,#01AC-100
  add hl,de

  ld a,PSG_REG_TONE_0
  out (PSGREG),a
  ld a,l
  out (PSGWRI),a
  ld a,PSG_REG_TONE_1
  out (PSGREG),a
  ld a,h
  out (PSGWRI),a

  ld a,PSG_REG_AMP_0
  out (PSGREG),a
  ld a,(explosion_sound_counter)
  REPT 3
    srl a
  ENDM
  and 15
  out (PSGWRI),a

  ld a,PSG_REG_MIXER
  out (PSGREG),a
  ld a,%10111110
  out (PSGWRI),a
  ret

frame_handler_gameover:
  ; Render the game
  call do_rendering
  ; Handle inputs
  in a,(KEYSEL)
  and #F0
  or 8
  out (KEYSEL),a
  in a,(KEYROW)
  xor #FF
  ld b,a
  ld hl,key_rows_state+8
  xor (hl)
  and b
  bit 0,a ; Spacebar
  jr z,$+2+3+3
  ld hl,transition_gameover_to_titlescreen
  ld (transition_address),hl
  ld hl,key_rows_state+8
  ld (hl),b

  IF DEBUG_ENABLED
  ld a,COLOUR_MAGENTA
  call set_border_colour
  ENDIF

  call handle_camera_dynamics
  call step_particles

  IF DEBUG_ENABLED
  ld a,COLOUR_CYAN
  call set_border_colour
  ENDIF

  call build_sprites_from_particles

  IF DEBUG_ENABLED
  ld a,COLOUR_GREY
  call set_border_colour
  ENDIF

  ; Update HUD message in buffer
  ld a,(frame_counter)
  and 8
  call z,print_hud_gameover_msg
  call nz,clear_hud_message

  ; Explosion sound
  ld hl,explosion_sound_counter
  xor a
  or (hl)
  ret z
  dec (hl)
  ld a,PSG_REG_NOISE
  out (PSGREG),a
  ld a,(explosion_sound_counter)
  cpl
  sla a
  and 31
  out (PSGWRI),a

  ld a,PSG_REG_AMP_0
  out (PSGREG),a
  ld a,(explosion_sound_counter)
  REPT 3
    srl a
  ENDM
  and 15
  out (PSGWRI),a

  ld a,PSG_REG_MIXER
  out (PSGREG),a
  ld a,%10110111
  out (PSGWRI),a
  ret

frame_handler_timeover:
  ; Render the game
  call do_rendering
  ; Handle inputs
  in a,(KEYSEL)
  and #F0
  or 8
  out (KEYSEL),a
  in a,(KEYROW)
  xor #FF
  ld b,a
  ld hl,key_rows_state+8
  xor (hl)
  and b
  bit 0,a ; Spacebar
  jr z,$+2+3+3
  ld hl,transition_timeover_to_titlescreen
  ld (transition_address),hl
  ld hl,key_rows_state+8
  ld (hl),b

  IF DEBUG_ENABLED
  ld a,COLOUR_MAGENTA
  call set_border_colour
  ENDIF

  call handle_camera_dynamics
  call step_particles

  IF DEBUG_ENABLED
  ld a,COLOUR_CYAN
  call set_border_colour
  ENDIF

  call build_sprites_from_particles

  IF DEBUG_ENABLED
  ld a,COLOUR_GREY
  call set_border_colour
  ENDIF

  ; Update HUD message in buffer
  ld a,(frame_counter)
  and 8
  call z,print_hud_timeover_msg
  call nz,clear_hud_message

  ; Explosion sound
  ld hl,explosion_sound_counter
  xor a
  or (hl)
  ret z
  dec (hl)
  ld a,PSG_REG_NOISE
  out (PSGREG),a
  ld a,(explosion_sound_counter)
  cpl
  sla a
  and 31
  out (PSGWRI),a

  ld a,PSG_REG_AMP_0
  out (PSGREG),a
  ld a,(explosion_sound_counter)
  REPT 3
    srl a
  ENDM
  and 15
  out (PSGWRI),a

  ld a,PSG_REG_MIXER
  out (PSGREG),a
  ld a,%10110111
  out (PSGWRI),a
  ret

frame_handler_ingame:
  ; Render the game
  call do_rendering
  ; Handle inputs
  in a,(KEYSEL)
  and #F0
  or 8
  out (KEYSEL),a
  in a,(KEYROW)
  bit 7,a ; Right
  call z,right_pressed
  bit 6,a ; Down
  call z,down_pressed
  bit 5,a ; Up
  call z,up_pressed
  bit 4,a ; Left
  call z,left_pressed

  IF DEBUG_ENABLED
  ; Handle debug keys
  in a,(KEYSEL)
  and #F0
  or 4
  out (KEYSEL),a
  in a,(KEYROW)
  xor #FF
  ld b,a
  ld hl,key_rows_state+4
  xor (hl)
  and b
  bit 6,a ; Q
  jr z,$+2+3+3
  ld hl,transition_ingame_to_stageclear
  ld (transition_address),hl
  ld hl,key_rows_state+4
  ld (hl),b
  ENDIF

  IF DEBUG_ENABLED
  ld a,COLOUR_MAGENTA
  call set_border_colour
  ENDIF

  call handle_player_dynamics
  call handle_camera_dynamics

  IF DEBUG_ENABLED
  ld a,COLOUR_CYAN
  call set_border_colour
  ENDIF

  call build_sprites_from_player_and_fruits
  call update_stage_timer

  ; Fruit pickup sound
  ld hl,pickup_sound_counter
  xor a
  or (hl)
  jp z,init_audio
  dec (hl)

  ld l,(hl)
  ld h,0
  REPT 3
    add hl,hl
  ENDM
  ex de,hl
  ld hl,#01AC ; Middle C
  add hl,de

  ld a,PSG_REG_TONE_0
  out (PSGREG),a
  ld a,l
  out (PSGWRI),a
  ld a,PSG_REG_TONE_1
  out (PSGREG),a
  ld a,h
  out (PSGWRI),a

  ld a,PSG_REG_AMP_0
  out (PSGREG),a
  ld a,#0C
  out (PSGWRI),a

  ld a,PSG_REG_MIXER
  out (PSGREG),a
  ld a,%10111110
  out (PSGWRI),a
  ret

update_stage_timer:
  ; Check for timer increment
  ld hl,jiffy_counter
  inc (hl)
  ld a,(hl)
  cp 60
  ret nz
  ld (hl),0
  ; Increment the timer in the HUD
  ld hl,hud_buffer+HUD_TIME_Y*HUD_WIDTH+HUD_TIME_X
  ; Seconds
  inc (hl)
  ld a,NAMES_BASE+10
  cp (hl)
  ret nz
  ld (hl),NAMES_BASE+0
  dec hl
  ; 10s of seconds
  inc (hl)
  ld a,NAMES_BASE+6
  cp (hl)
  ret nz
  ld (hl),NAMES_BASE+0
  dec hl
  dec hl ; Skip the colon too
  ; Minutes
  inc (hl)
  ld a,NAMES_BASE+10
  cp (hl)
  ret nz
  ld (hl),NAMES_BASE+0
  dec hl
  ; 10s of minutes
  inc (hl)
  ld a,NAMES_BASE+1
  cp (hl)
  ret nz
  ; The player took more than 10 minutes to
  ; finish the stage. Make that an instance Game Over.
  ld hl,transition_ingame_to_timeover
  ld (transition_address),hl
  ret

clear_particle_sprites:
  ld hl,sprite_buffer
  ld (hl),#D0
  ret

build_sprites_from_particles:
  ld hl,particle_step_counter
  ld a,MAX_PARTICLE_STEPS
  cp (hl)
  jp z,clear_particle_sprites

  ld hl,sprite_buffer
  ld (sprite_buffer_tail),hl

  ld bc,sprite_buffer
  exx
  ld b,NUM_PARTICLES
  exx

  ld ix,particles

build_sprites_from_particles_loop:
  ld hl,(scroll_position+0) ; Scroll Y
  ex de,hl
  ld l,(ix+2) ; Particle Y
  ld h,(ix+3)
  CONVERT_SUBPIXEL_TO_PIXEL
  or a ; Clear the carry flag
  sbc hl,de
  ld a,l
  add a,-4-1
  ld (bc),a ; Sprite Y
  inc bc

  ld hl,(scroll_position+2) ; Scroll X
  ex de,hl
  ld l,(ix+0) ; Particle X
  ld h,(ix+1)
  CONVERT_SUBPIXEL_TO_PIXEL
  or a ; Clear the carry flag
  sbc hl,de
  ld a,l
  add a,-4-1
  ld (bc),a ; Sprite X
  inc bc

  ld a,(frame_counter)
  sra a
  and 3
  add a,PARTICLE_SPRITE_NAME
  ld (bc),a ; Sprite Pattern
  inc bc
  ld a,PARTICLE_SPRITE_COLOUR
  ld (bc),a ; Sprite Colour
  inc bc

  ld de,4
  add ix,de
  exx
  dec b
  exx
  jr nz,build_sprites_from_particles_loop

  ret

build_sprites_from_player_and_fruits:
  ; Update sprites
  ld hl,sprite_buffer
  ld (sprite_buffer_tail),hl

  ; Player sprite
  ; This code is somewhat simplified by the assumption
  ; that the player will never go offscreen.
  ld de,sprite_buffer
  ld hl,(scroll_position+0) ; Scroll Y
  ld b,h
  ld c,l
  ld hl,(player_pos_y) ; Player Y
  CONVERT_SUBPIXEL_TO_PIXEL
  or a ; Clear the carry flag
  sbc hl,bc
  ld a,l
  dec a ; Account for TMS9918 oddity
  ld (de),a ; Sprite Y
  push af
  inc de
  ld hl,(scroll_position+2) ; Scroll X
  ld b,h
  ld c,l
  ld hl,(player_pos_x) ; Player X
  CONVERT_SUBPIXEL_TO_PIXEL
  or a ; Clear the carry flag
  sbc hl,bc
  ld a,l
  ld (de),a ; Sprite X
  ld b,a
  inc de
  ld a,(frame_counter)
  sra a
  and 3
  add a,UFO_SPRITE_NAME
  ld (de),a ; Sprite Pattern
  inc de
  ld a,UFO_SPRITE_COLOUR
  ld (de),a ; Sprite Colour
  inc de
  pop af
  ; Add the hat sprite too
  ex de,hl
  sub 8
  ld (hl),a ; Sprite Y
  inc hl
  ld a,b
  ld (hl),a ; Sprite X
  inc hl
  ld (hl),HAT_SPRITE_NAME ; Sprite Pattern
  inc hl
  ld (hl),HAT_SPRITE_COLOUR ; Sprite Colour
  inc hl
  ex de,hl

  ld hl,sprite_buffer_tail
  ld (hl),e
  inc hl
  ld (hl),d
  inc hl


  ; Fruit sprites
  ; also detect fruit pickups
  ld a,(num_fruits)
  and a
  jp z,no_sprites
  ld de,fruits
  ld bc,FRUIT_BYTES
  exx
  ld hl,(sprite_buffer_tail)
  exx
  ex af,af'
  ld a,(num_fruits)
  ex af,af'
make_fruit_sprites_loop:
  push bc
  push de

  ld ixh,d
  ld ixl,e

  ld a,(de) ; Fruit flags
  inc de
  cp 1
  jp nz,skip_fruit_sprite

  ld hl,(scroll_position+0) ; Scroll Y
  ld b,h
  ld c,l
  ld a,(de) ; 7  ; Fruit Y low
  ld l,a    ; 4
  inc de    ; 4
  ld a,(de) ; Fruit Y high
  ld h,a
  inc de
  CONVERT_SUBPIXEL_TO_PIXEL
  or a ; Clear the carry flag
  sbc hl,bc
  ld bc,8 ; Allow "rolling-on" of sprite
  add hl,bc
  ld a,h
  cp #00
  jp nz,skip_fruit_sprite
  ld a,l
  cp #C0+12
  jp nc,skip_fruit_sprite
  exx
  ld b,a
  exx

  ld hl,(scroll_position+2) ; Scroll X
  ld b,h
  ld c,l
  ld a,(de) ; 7  ; Fruit X low
  ld l,a    ; 4
  inc de    ; 4
  ld a,(de) ; Fruit X high
  ld h,a
  inc de
  CONVERT_SUBPIXEL_TO_PIXEL
  or a ; Clear the carry flag
  sbc hl,bc
  ld bc,32 ; Account for EC
  add hl,bc
  ld a,h
  cp #00
  jp nz,skip_fruit_sprite
  ld a,l
  cp #C0+32
  jp nc,skip_fruit_sprite

  exx
  ; Apple part
  ld c,a
  ld a,b
  dec a ; Account for TMS9918 oddity
  sub 8 ; Account for "rolling-on" of sprite
  ld (hl),a ; Sprite Y
  inc hl
  ld a,c
  ld (hl),a ; Sprite X
  inc hl
  ; Calculate pattern offset for fake rolloff on right side
  ld (hl),FRUIT_SPRITE_NAME ; Sprite Pattern
  cp 192+32-8 ; +32 to account for EC
  jr c,$+2+2+2+1
  and 7
  add a,FRUIT_SPRITE_NAME
  ld (hl),a ; Sprite Pattern
  ld d,(hl)
  inc hl
  ld (hl),FRUIT_SPRITE_COLOUR+#80 ; Sprite Colour + EC bit
  inc hl
  ; Leaf part
  ld a,b
  sub 8+1+8 ; +1 to Account for TMS9918 oddity
  ld (hl),a ; Sprite Y
  inc hl
  ld a,c
  ld (hl),a ; Sprite X
  inc hl
  ld a,d
  add a,LEAF_SPRITE_NAME-FRUIT_SPRITE_NAME
  ld (hl),a ; Sprite Pattern
  inc hl
  ld (hl),LEAF_SPRITE_COLOUR+#80 ; Sprite Colour + EC bit
  inc hl
  exx

  ; Detect pickup
  ; Y
  ld hl,(player_pos_y)
  ex de,hl
  ld l,(ix+1)
  ld h,(ix+2)
  ld bc,4<<SUBPIXEL_SHIFT
  add hl,bc
  or a
  sbc hl,de
  bit 7,l
  jr nz,skip_fruit_sprite
  xor a
  cp h
  jr nz,skip_fruit_sprite
  ; X
  ld hl,(player_pos_x)
  ex de,hl
  ld l,(ix+3)
  ld h,(ix+4)
  ld bc,4<<SUBPIXEL_SHIFT
  add hl,bc
  or a
  sbc hl,de
  bit 7,l
  jr nz,skip_fruit_sprite
  xor a
  cp h
  jr nz,skip_fruit_sprite

  ; Fruit was picked up.
  ; Mask out the fruit
  xor a
  ld (ix+0),a

  ; Trigger pickup sound
  ld a,16
  ld (pickup_sound_counter),a

  ; Decrement the counter in the HUD
  ld hl,hud_buffer+HUD_FRUIT_Y*HUD_WIDTH+HUD_FRUIT_X+1
  dec (hl)
  ld a,NAMES_BASE-1
  cp (hl)
  jr nz,$+2+2+2
  ld (hl),NAMES_BASE+9
  dec hl
  dec (hl)

  ; Check to see if all fruit have been picked up.
  ld hl,hud_buffer+HUD_FRUIT_Y*HUD_WIDTH+HUD_FRUIT_X
  ld a,(hl)
  sub NAMES_BASE+0
  ld b,a
  inc hl
  ld a,(hl)
  sub NAMES_BASE+0
  or b
  jr nz,$+2+3+3
  ld hl,transition_ingame_to_stageclear
  ld (transition_address),hl

skip_fruit_sprite:
  pop de
  pop bc
  ex de,hl
  add hl,bc
  ex de,hl
  ex af,af'
  ld h,a
  dec a
  ex af,af'
  dec h
  jp nz,make_fruit_sprites_loop
  exx
  ld (sprite_buffer_tail),hl
  exx
no_sprites:

  ld hl,(sprite_buffer_tail)
  ld (hl),#D0 ; Terminate the list
  ret

handle_player_dynamics:
  ; Perform collision detection
  ld a,NAMES_BASE+0
  ld (player_pattern),a

  ld bc,1<<SUBPIXEL_SHIFT
  ld hl,(player_pos_x)
  add hl,bc ; Inset
  ex de,hl
  ld hl,(player_pos_y)
  add hl,bc ; Inset
  call check_box
  jr z,$+2+3+3
  ld hl,transition_ingame_to_gameover
  ld (transition_address),hl

  ; Update player motion dynamics
  ld hl,(player_vel_y)
  ex de,hl
  ld hl,(player_pos_y)
  add hl,de
  ld (player_pos_y),hl

  ld hl,(player_vel_x)
  ex de,hl
  ld hl,(player_pos_x)
  add hl,de
  ld (player_pos_x),hl
  ret

handle_camera_dynamics:
  ; Update the camera
  ld hl,(camera_y)
  ex de,hl
  ld hl,(player_pos_y)
  or a
  sbc hl,de
  REPT CAMERA_SHIFT
    sra h
    rr l
  ENDM
  ld b,h
  ld c,l
  ld hl,(camera_y)
  add hl,bc
  ld (camera_y),hl

  ld hl,(camera_x)
  ex de,hl
  ld hl,(player_pos_x)
  or a
  sbc hl,de
  REPT CAMERA_SHIFT
    sra h
    rr l
  ENDM
  ld b,h
  ld c,l
  ld hl,(camera_x)
  add hl,bc
  ld (camera_x),hl
  ret

do_rendering:
  IF DEBUG_ENABLED
  ld a,COLOUR_MEDIUM_RED
  call set_border_colour
  ENDIF

  ; Write sprites to VRAM
  ld a,SATRBL&#FF         ; 8
  out (VDPCTL),a          ; 11
  ld a,((SATRBL>>8)&63)|64; 8
  out (VDPCTL),a          ; 11
  ld hl,sprite_buffer ; 11
  ld c,VDPDAT             ; 8
  call outi_unrolled_loop

  IF DEBUG_ENABLED
  ld a,COLOUR_DARK_BLUE
  call set_border_colour
  ENDIF

  ; Set patterngenerator base address
  ld hl,(scroll_position+0)
  ld a,l
  and 7
  ld b,0
  ld c,a
  ld hl,vram_pattern_table_addresses
  add hl,bc
  ld a,(hl)
  out (VDPCTL),a
  ld a,#84
  out (VDPCTL),a

  ; Set scroll position according to camera
  ld hl,(camera_y)
  CONVERT_SUBPIXEL_TO_PIXEL
  ld bc,-(VIEWPORT_WIDTH*8)/2
  add hl,bc
  ld (scroll_position+0),hl
  ld hl,(camera_x)
  CONVERT_SUBPIXEL_TO_PIXEL
  ld bc,-(VIEWPORT_HEIGHT*8)/2
  add hl,bc
  ld (scroll_position+2),hl

  ; Set nametable base address
  ld a,(pending_vram_name_table_address+1)
  REPT 2
    srl a
  ENDM
  out (VDPCTL),a
  ld a,#82
  out (VDPCTL),a

  ; Fill map address array Y
  ld hl,(scroll_position+0)
  ; Multiply HL by 64/8=8
  REPT 3
    sla l
    rl h
  ENDM
  ld a,l
  and 255-63
  ld l,a
  ld bc,64
  ld de,map_addresses_y
  REPT 24
    ld a,l
    ld (de),a
    inc de
    ld a,h
    ; Wrap at 64*64=4096=256*16
    and 15
    add a,map_64x64>>8
    ld (de),a
    inc de
    add hl,bc
  ENDM

  ; Fill map address array X
  ld hl,(scroll_position+2)
  ld a,l
  and 7
  REPT 4
    add a,a
  ENDM
  ; Save pixel X offset (in upper nibble)
  ld e,a
  ; Divide HL by 8
  REPT 3
    sra h
    rr l
  ENDM
  ld a,l
  ld hl,map_addresses_x
  REPT 32
    and 63
    ld (hl),a
    inc hl
    inc a
  ENDM

  IF DEBUG_ENABLED
  ld a,COLOUR_LIGHT_YELLOW
  call set_border_colour
  ENDIF

  ; Fill nametable staging buffer
  ld hl,vram_name_table_addresses
  ld a,(frame_counter)
  and 1
  sla a
  ld b,0
  ld c,a
  add hl,bc
  ; This address will be used for display during the next frame.
  ld a,(hl)
  ld (pending_vram_name_table_address+0),a
  out (VDPCTL),a
  inc hl
  ld a,(hl)
  ld (pending_vram_name_table_address+1),a
  and 63
  or 64
  out (VDPCTL),a

  ld b,e
  ld de,hud_buffer
  jp staging_loop

print_hud_gameover_msg:
  push af
  ld hl,hud_msg_gameover
  ld de,hud_buffer+12*HUD_WIDTH
  ld bc,HUD_MSG_WIDTH*HUD_MSG_HEIGHT
  ldir
  pop af
  ret

print_hud_timeover_msg:
  push af
  ld hl,hud_msg_timeover
  ld de,hud_buffer+12*HUD_WIDTH
  ld bc,HUD_MSG_WIDTH*HUD_MSG_HEIGHT
  ldir
  pop af
  ret

print_hud_nextstage_msg:
  push af
  ld hl,hud_msg_nextstage
  ld de,hud_buffer+12*HUD_WIDTH
  ld bc,HUD_MSG_WIDTH*HUD_MSG_HEIGHT
  ldir
  pop af
  ret

clear_hud_message:
  push af
  ld hl,hud_bg+12*HUD_WIDTH
  ld de,hud_buffer+12*HUD_WIDTH
  ld bc,HUD_MSG_WIDTH*HUD_MSG_HEIGHT
  ldir
  pop af
  ret

clear_game_variables:
  ; Clear all game variables
  ld hl,game_variables
  ld (hl),0
  ld de,game_variables+1
  ld bc,game_variables_end-game_variables-1
  ldir
  ret

begin_game:
  call clear_game_variables

  ; Initialise HUD
  ld hl,hud_bg
  ld de,hud_buffer
  ld bc,HUD_WIDTH*HUD_HEIGHT
  ldir

  call load_stage
  ret

skip_stage:
  ; Clear all game variables except stage_index
  ld a,(stage_index)
  call clear_game_variables
  ld (stage_index),a

  ; Initialise HUD
  ld hl,hud_bg
  ld de,hud_buffer
  ld bc,HUD_WIDTH*HUD_HEIGHT
  ldir

  ld hl,stage_index
  inc (hl)
  call load_stage
  ret

load_stage:
  ld bc,map_addresses
  ld h,0
  ld a,(stage_index)
  ld l,a
  add hl,hl
  add hl,bc
  ld e,(hl)
  inc hl
  ld d,(hl)
  inc hl
  ld h,d
  ld l,e
  push hl
  call expand_1bpp_map
  call fill_map_64x64
  ld hl,fruits
  ld (fruits_tail),hl
  pop hl
  call find_fruits
  ; Initialise player and camera state
  ld hl,-(4<<SUBPIXEL_SHIFT)
  ld (player_pos_x),hl
  ld (camera_x),hl
  ld hl,28<<SUBPIXEL_SHIFT
  ld (player_pos_y),hl
  ld (camera_y),hl
  ld a,NAMES_BASE+0
  ld (player_pattern),a
  ; Update HUD
  ld a,(stage_index)
  inc a
  and 15
  ld b,0
  ld c,a
  ld hl,hud_mod10_table
  add hl,bc
  ld a,(hl)
  ld c,a
  ld hl,hud_buffer+HUD_STAGE_Y*8+HUD_STAGE_X
  ld b,NAMES_BASE+0
  ld a,(stage_index)
  inc a
  cp 10
  jr c,$+4
  ld b,NAMES_BASE+1
  ld (hl),b
  inc hl
  ld (hl),c
  inc hl
  ret
hud_mod10_table:
  REPT 16,?I
    db NAMES_BASE+(?I % 10)
  ENDM
  ret

; DE = Subpixel X coordinate
; HL = Subpixel Y coordinate
check_pos:
  ld a,d
  ld c,e
  ; Shift top bit of E into D
  sla c
  rla
  and 63
  ld c,a
  ; C = --XXXXXX
  ld a,h
  and 31
  ld b,a
  ld a,l
  ; Shift HL right
  sra b
  rra
  and 255-63
  ; B = ----YYYY
  ; A = YY------
  or c
  ld c,a
  ; C = YYXXXXXX
  ld a,b
  add a,map_64x64>>8
  ld b,a
  ld a,(bc)
  bit 0,a
  ret

; DE = Subpixel X coordinate
; HL = Subpixel Y coordinate
;      (Top-left corner of box to check)
check_box:
  ; Top-left
  call check_pos
  jr z,$+3
  ret

  ; Bottom-left
  push hl
  ld bc,5<<SUBPIXEL_SHIFT
  add hl,bc
  call check_pos
  pop hl
  jr z,$+3
  ret

  ; Top-right
  push de
  push hl
  ld h,d
  ld l,e
  ld bc,5<<SUBPIXEL_SHIFT
  add hl,bc
  ld d,h
  ld e,l
  pop hl
  call check_pos
  pop de
  jr z,$+3
  ret

  ; Bottom-right
  push de
  push hl
  ld h,d
  ld l,e
  ld bc,5<<SUBPIXEL_SHIFT
  add hl,bc
  ld d,h
  ld e,l
  pop hl
  add hl,bc
  call check_pos
  pop de
  ret

up_pressed:
  ld bc,NEG_ACC
  ld hl,(player_vel_y)
  add hl,bc
  ld (player_vel_y),hl
  ret

down_pressed:
  ld bc,POS_ACC
  ld hl,(player_vel_y)
  add hl,bc
  ld (player_vel_y),hl
  ret

left_pressed:
  ld bc,NEG_ACC
  ld hl,(player_vel_x)
  add hl,bc
  ld (player_vel_x),hl
  ret

right_pressed:
  ld bc,POS_ACC
  ld hl,(player_vel_x)
  add hl,bc
  ld (player_vel_x),hl
  ret

outi_unrolled_loop_iteration:
  outi ; ED A3
generate_outi_unrolled_loop:
  ld de,outi_unrolled_loop
  ld bc,128*2
loop:
  ld hl,outi_unrolled_loop_iteration
  ldi
  ldi
  jp pe,loop
  ld a,#C9 ; ret
  ld (de),a
  ret

staging_iteration:
  ;ld hl,(map_addresses_y+?Y*2)
  ld c,l
  ; Emit map on-the-fly
  REPT VIEWPORT_WIDTH,?X
    ld a,(map_addresses_x+?X)
    or c
    ld l,a
    ld a,(hl)
    or b
    out (VDPDAT),a
  ENDM
  ; Emit HUD from buffer
  REPT 32-VIEWPORT_WIDTH
    ld a,(de)       ; 7
    inc de          ; 6
    out (VDPDAT),a  ; 11
  ENDM
staging_iteration_end:

generate_staging_loop:
  ld de,staging_loop
  ex af,af'
  ld a,0
loop2:
  ex af,af'
  ld a,#2A ; ld hl,(nn)
  ld (de),a
  inc de
  ld hl,map_addresses_y
  ld b,0
  ex af,af'
  ld c,a
  ex af,af'
  sla c
  add hl,bc
  ld a,l
  ld (de),a ; address low
  inc de
  ld a,h
  ld (de),a ; address high
  inc de
  ld hl,staging_iteration
  ld bc,staging_iteration_end-staging_iteration
  ldir
  ex af,af'
  inc a
  cp VIEWPORT_HEIGHT
  jr nz,loop2
  ex af,af'
  ld a,#C9 ; ret
  ld (de),a
  ret

add_fruit:
  push hl
  push af
  push bc
  push de
  ld ix,(fruits_tail)
  xor a
  ; Subpixel X = Fruit X x 4x8x16 = 512
  REPT 1
    sla d
  ENDM
  ld l,a
  ld h,d
  ld bc,-4<<SUBPIXEL_SHIFT ; Position so spawnpoint is at center
  add hl,bc
  ld (ix+3),l
  ld (ix+4),h
  ; Subpixel Y = Fruit Y x 4x8x16 = 512
  REPT 1
    sla e
  ENDM
  ld l,a
  ld h,e
  ld bc,-4<<SUBPIXEL_SHIFT ; Position so spawnpoint is at center
  add hl,bc
  ld (ix+1),l
  ld (ix+2),h
  ld a,1 ; Flags
  ld (ix+0),a
  ld bc,FRUIT_BYTES
  add ix,bc
  ld (fruits_tail),ix
  ld a,(num_fruits)
  inc a
  ld (num_fruits),a

  ; Increment the counter in the HUD
  ld hl,hud_buffer+HUD_FRUIT_Y*HUD_WIDTH+HUD_FRUIT_X+1
  inc (hl)
  ld a,NAMES_BASE+10
  cp (hl)
  jr nz,$+2+2+2
  ld (hl),NAMES_BASE+0
  dec hl
  inc (hl)

  pop de
  pop bc
  pop af
  pop hl
  ret

find_fruits:
  xor a
  ld (num_fruits),a
  ld c,8
  ld e,0
find_fruits_loop_y:
  ld b,8
  ld d,0
find_fruits_loop_x:
  ; Count the number of open passages
  push de
  xor a
  ; Top
  call lookup_1bpp_map
  jr nz,$+3
  inc a
  ; Right
  inc d
  inc e
  call lookup_1bpp_map
  jr nz,$+3
  inc a
  ; Left
  dec d
  dec d
  call lookup_1bpp_map
  jr nz,$+3
  inc a
  ; Bottom
  inc d
  inc e
  call lookup_1bpp_map
  jr nz,$+3
  inc a
  ; If there is only one open passage, then add a fruit here.
  dec e
  cp 1
  call z,add_fruit
  pop de
  inc d
  inc d
  djnz find_fruits_loop_x
  inc e
  inc e
  dec c
  jr nz,find_fruits_loop_y
  ret

; Use map combined with metatiles to create a flattened map
fill_map_64x64:
  ld de,#0000
fill_map_64x64_loop:
  ; Construct map address from coarse (16x16) coordinates
  ; Those being the upper 4 bits of the 6-bit flattened coordinates.
  ld a,e
  REPT 2
    sra a
  ENDM
  and 15
  ld h,d
  REPT 4
    sla h
  ENDM
  or h
  ld h,map>>8
  ld l,a
  ; Save the metatile index
  ld c,(hl)
  ; Use coordinates constructed from lower 2 bits.
  ld a,e
  ;    YY----XX
  and %11000011
  ld b,0
  rla
  rl b
  rla
  rl b
  sla b
  sla b
  sra a
  sra a
  and 3 ; ------AA
  or b  ; ----BBAA
  or c  ; CCCCBBAA
  ; Perform lookup on metatile
  ld h,metatiles>>8
  ld l,a
  ; Switch to the tighter mazes after stage 6
  ld a,(stage_index)
  cp 6
  jr c,$+2+1
  inc h
  ld a,(hl)

  ld hl,map_64x64
  add hl,de
  ld (hl),a

  inc de
  ld a,d
  cp 4096>>8 ; 64x64=4096
  jr nz,fill_map_64x64_loop
  ret

; HL = Base address of map
; DE = Map cell XY coordinates
lookup_1bpp_map:
  push bc
  push hl
  push de
  ex af,af'
  ; Apply wrap to X
  ld a,d
  and 15
  ld d,a
  ; Apply wrap to Y
  ld a,e
  and 15
  ld e,a

  sla e
  ld a,d
  and 8
  REPT 3
    sra a
  ENDM

  ld b,0
  ld c,a
  add hl,bc
  ld c,e
  add hl,bc
  ld a,d
  and 7
  ld d,(hl)

  ld hl,bit_lookup_table
  ld c,a
  add hl,bc
  ld a,(hl)

  and d
  ex af,af'
  ld b,a
  ex af,af'
  ld a,b

  pop de
  pop hl
  pop bc
  ret
bit_lookup_table:
  REPT 8,?I
    db 1<<?I
  ENDM

expand_1bpp_map:
  push hl
  push af
  push de
  push bc
  ld (expand_1bpp_map_temp),hl
  ; Initialise
  ld hl,map
  ld (hl),0
  ld d,h
  ld e,l
  inc de
  ld bc,256-1
  ldir
  ld hl,map
  ld e,0
  ld c,16
expand_1bpp_map_loop_y:
  ld d,0
  ld b,16
expand_1bpp_map_loop_x:
  xor a
  push hl
  ld hl,(expand_1bpp_map_temp)
  call lookup_1bpp_map
  jr z,$+4
  or 1<<4
  inc d
  call lookup_1bpp_map
  jr z,$+4
  or 1<<5
  dec d
  inc e
  call lookup_1bpp_map
  jr z,$+4
  or 1<<6
  inc d
  call lookup_1bpp_map
  jr z,$+4
  or 1<<7
  dec e
  pop hl
  ld (hl),a
  inc hl
  djnz expand_1bpp_map_loop_x
  inc e
  dec c
  jr nz,expand_1bpp_map_loop_y
  pop bc
  pop de
  pop af
  pop hl
  ret

; Patterns are RLE-compressed
; HL = Compressed data address
;  B = Output length / 256
decompress_rle:
  ld de,rle_output
  ld a,b
  exx
  ld b,a
  exx
decompress_rle_loop:
  ld a,(hl)
  inc hl
  ld c,(hl)
  inc hl
  push hl
  dec c
  ld (de),a
  ld h,d
  ld l,e
  inc de
  jr z,$+6
  ld b,0
  ldir
  pop hl
  exx
  ld a,b
  exx
  cp d
  jr nz,decompress_rle_loop
  ret

char_indices:
  incbin "hud_chars.bin"
char_indices_end:

hud_bg:
  incbin "hud.bin"
hud_msg_gameover:
  incbin "hud_msg_gameover.bin"
hud_msg_nextstage:
  incbin "hud_msg_nextstage.bin"
hud_msg_timeover:
  incbin "hud_msg_timeover.bin"

  db "Copyright (c) 2024 Edd Biddulph"

leaf_patterns:
  REPT 8,?I
  db %00000000 & (255-((1<<?I)-1))
  db %00000000 & (255-((1<<?I)-1))
  db %00000000 & (255-((1<<?I)-1))
  db %00000000 & (255-((1<<?I)-1))
  db %01100000 & (255-((1<<?I)-1))
  db %01110000 & (255-((1<<?I)-1))
  db %00110000 & (255-((1<<?I)-1))
  db %00010000 & (255-((1<<?I)-1))
  ENDM
fruit_patterns:
  REPT 8,?I
  db %01101100 & (255-((1<<?I)-1))
  db %11111110 & (255-((1<<?I)-1))
  db %11111110 & (255-((1<<?I)-1))
  db %11111110 & (255-((1<<?I)-1))
  db %11111110 & (255-((1<<?I)-1))
  db %01111100 & (255-((1<<?I)-1))
  db %00101000 & (255-((1<<?I)-1))
  db %00000000 & (255-((1<<?I)-1))
  ENDM
hat_pattern:
  db %00000000
  db %00000000
  db %00000000
  db %00000000
  db %00000000
  db %00011100
  db %00111110
  db %11111110
ufo_patterns:
  db %01111110 ; 1
  db %11111111
  db %11111111
  db %11111111
  db %11011011
  db %11011011
  db %11111111
  db %11001100
  db %01111110 ; 2
  db %11111111
  db %11111111
  db %11111111
  db %11011011
  db %11011011
  db %11111111
  db %10011001
  db %01111110 ; 3
  db %11111111
  db %11111111
  db %11111111
  db %11011011
  db %11011011
  db %11111111
  db %00110011
  db %01111110 ; 4
  db %11111111
  db %11111111
  db %11111111
  db %11011011
  db %11011011
  db %11111111
  db %01100110
particle_patterns:
  db %00000000 ; 1
  db %00000000
  db %00000000
  db %00011000
  db %00011000
  db %00000000
  db %00000000
  db %00000000
  db %00000000 ; 2
  db %00000000
  db %00011000
  db %00111100
  db %00111100
  db %00011000
  db %00000000
  db %00000000
  db %00000000 ; 3
  db %00011000
  db %00111100
  db %01111110
  db %01111110
  db %00111100
  db %00011000
  db %00000000
  db %00000000 ; 4
  db %00000000
  db %00011000
  db %00111100
  db %00111100
  db %00011000
  db %00000000
  db %00000000

vram_pattern_table_addresses:
  REPT 8,?N
    db (PATTBL+?N*#0800)>>11
  ENDM
vram_name_table_addresses:
  dw NAMTBL1,NAMTBL2
patterns_compressed:
  incbin "tiles_compressed.bin"
  ALIGN #100
;patterns:
;  incbin "tiles.bin"
;  ALIGN #100
map_1bpp:
  incbin "map_1bpp.bin"
  ALIGN #100
metatiles:
  incbin "metatiles.bin"
particle_vectors:
  incbin "particle_vectors.bin"
;titlescreen_nametable:
;  incbin "titlescreen.bin"
titlescreen_nametable_compressed:
  incbin "titlescreen_compressed.bin"
;endscreen_nametable:
;  incbin "endscreen.bin"
endscreen_nametable_compressed:
  incbin "endscreen_compressed.bin"
default_colour_table:
  REPT 16
    db (COLOUR_MEDIUM_GREEN<<4) | 1
  ENDM
  REPT 16
    db #F1
  ENDM
blank_colour_table:
  REPT 16
    db #11
  ENDM
  REPT 16
    db #F1
  ENDM
map_addresses:
  REPT 12,?I
    dw map_1bpp+?I*32
  ENDM

set_border_colour:
  and #0F
  or #F0
  out (VDPCTL),a
  ld a,#87
  out (VDPCTL),a
  ret

  ds #7E00 - $

  REPT #101
    db #7F
  ENDM

  ds #7F7F - $
interrupt_handler:
  push af
  in a,(VDPCTL)
  ld a,1
  ld (frame_advance),a
  pop af
  ;ei
  reti

  ds #8000 - $

; 16Kbyte RAM
  org #8000
RAM_START:
  ALIGN_VIRTUAL #100
map:
  ds virtual 256 ; 16x16
  ALIGN_VIRTUAL #100
map_64x64:
  ds virtual 4096 ; 64x64
  ALIGN_VIRTUAL #100
hud_buffer:
  ds virtual (32-VIEWPORT_WIDTH)*24
  ALIGN_VIRTUAL #100
char_buffer:
  ds virtual (char_indices_end-char_indices)*8
padding:
  ds virtual 6
outi_unrolled_loop:
  ds virtual 128*2+1
staging_loop:
  ds virtual (staging_iteration_end-staging_iteration+3)*VIEWPORT_HEIGHT+1
map_addresses_x:
  ds virtual 32
map_addresses_y:
  ds virtual 24*2
debug_keys_state:
  ds virtual 1
frame_handler_address:
  ds virtual 2
transition_address:
  ds virtual 2
key_rows_state:
  ds virtual 16

game_variables:
frame_advance:
  ds virtual 1
frame_counter:
  ds virtual 1
scroll_position:
  ds virtual 4
player_pos_x:
  ds virtual 2
player_pos_y:
  ds virtual 2
player_vel_x:
  ds virtual 2
player_vel_y:
  ds virtual 2
player_acc_x:
  ds virtual 2
player_acc_y:
  ds virtual 2
player_pattern:
  ds virtual 1
sprite_pos_x:
  ds virtual 2
sprite_pos_y:
  ds virtual 2
sprite_pattern:
  ds virtual 1
camera_x:
  ds virtual 2
camera_y:
  ds virtual 2
expand_1bpp_map_temp:
  ds virtual 4
sprite_buffer:
  ds virtual 128
sprite_buffer_tail:
  ds virtual 2
particles: ; Overlaid with fruits, size = 8x4=32
fruits:
  ds virtual MAX_FRUITS*FRUIT_BYTES
fruits_tail:
  ds virtual 2
num_fruits:
  ds virtual 1
stage_index:
  ds virtual 1
colour_table:
  ds virtual 32
jiffy_counter:
  ds virtual 1
particle_step_counter:
  ds virtual 1
explosion_sound_counter:
  ds virtual 1
pickup_sound_counter:
  ds virtual 1
pending_vram_name_table_address:
  ds virtual 2
game_variables_end:

RAM_END:
  ds virtual #c000 - $

; Working area for pattern decompression, overlaid
  org #8000
rle_output:
  ds virtual #2000

; 1 pixel    = 16x16 subpixels
; 1 tile     = 8x8 pixels
; 1 metatile = 4x4 tiles
; 1 map      = 16x16 metatiles
;
; 1 metatile = (4*8*16)^2    = 512^2 subpixels
; 1 map      = (16*4*8*16)^2 = 8192^2 subpixels
