#include <SDL.h>
#include <math.h>
#include <assert.h>

#include "font.h"
#include "defs.h"
#include "mem.h"
#include "render.h"
#include "game.h"
#include "input.h"


#define ENGINE_MODE_NULL            0
#define ENGINE_MODE_TITLE           1
#define ENGINE_MODE_TRACKSELECTION  2
#define ENGINE_MODE_INGAME          3


static hunk_marker_t mem_start;
static hunk_marker_t mem_frame;
static hunk_marker_t mem_level;
static hunk_marker_t mem_objmesh;
static hunk_marker_t mem_objdraw;


static bool_t game_paused = FALSE;
static int keep_going = 1;

static renderstate_t g_rs;
static player_t g_player;
static gamestate_t g_gs;

static camera_t g_camera;
static float g_looka = 0.0f;

static int num_obj_drawn = 0;

uint32_t frame_time = 0;


int g_pending_enginemode = ENGINE_MODE_NULL;
int g_enginemode = ENGINE_MODE_NULL;

static int sign(float x)
{
  return (x < 0.0) ? -1 : +1;
}


void transitionEngineMode(int old_mode, int new_mode)
{
  assert(old_mode != new_mode);
  assert(new_mode != ENGINE_MODE_NULL);
  
  switch(old_mode)
  {
    case ENGINE_MODE_NULL:
      break;
    
    case ENGINE_MODE_TRACKSELECTION:
      break;
      
    case ENGINE_MODE_INGAME:
      break;
      
    default:
      assert(0);
      break;
  }
  
  switch(new_mode)
  {
    case ENGINE_MODE_TRACKSELECTION:
      break;
      
    case ENGINE_MODE_INGAME:
      resetLevel(&g_player, &g_gs, SDL_GetTicks());
      break;
      
    default:
      assert(0);
      break;
  }
  
  g_enginemode = new_mode;
}


static void quitLevel()
{
  keep_going = 0;
}


      
typedef struct
{
  const char* label;
} menu_item_t;

typedef struct
{
  menu_item_t* items;
  uint16_t current_index;
  uint16_t num_items;
} menu_t;

static menu_item_t pause_menu_items[] = { { "CONTINUE" }, { "RETRY" }, { "QUIT" } };
static menu_t pause_menu = (menu_t){ pause_menu_items, 0, sizeof(pause_menu_items) / sizeof(menu_item_t) };

void drawPauseMenu(uint16_t *const pixels)
{
  const uint16_t menubox_width = 100;
  const uint16_t menubox_height = pause_menu.num_items * HUD_FONT_H;
  const uint16_t menubox_left = FRAME_WIDTH / 2 - menubox_width / 2;
  const uint16_t menubox_top = FRAME_HEIGHT / 2 - menubox_height / 2 - 30;
  
  for(uint16_t y = menubox_top; y < menubox_top + menubox_height; ++y)
  {
    for(uint16_t x = menubox_left; x < menubox_left + menubox_width; ++x)
    {
      pixels[x + y * FRAME_WIDTH] = (pixels[x + y * FRAME_WIDTH] >> 1) & 0b0111101111101111;
    }
  }
  
  const uint16_t selectionbox_width = menubox_width;
  const uint16_t selectionbox_height = HUD_FONT_H;
  const uint16_t selectionbox_left = menubox_left;
  const uint16_t selectionbox_top = menubox_top + HUD_FONT_H * pause_menu.current_index;
  
  for(uint16_t y = selectionbox_top; y < selectionbox_top + selectionbox_height; ++y)
  {
    for(uint16_t x = selectionbox_left; x < selectionbox_left + selectionbox_width; ++x)
    {
      pixels[x + y * FRAME_WIDTH] = ((pixels[x + y * FRAME_WIDTH] >> 1) & 0b0111101111101111) + ((RGB888toRGB565(255, 100, 100) >> 1) & 0b0111101111101111);
    }
  }
  
  for(int i = 0; i < pause_menu.num_items; ++i)
  {
    plotString(pixels, FRAME_WIDTH / 2 - (strlen(pause_menu.items[i].label) * HUD_FONT_W) / 2, menubox_top + i * HUD_FONT_H, pause_menu.items[i].label, 1);
  }
}

void updateMenu(menu_t* menu)
{
  if(sys_inputs[SYS_INPUT_MENU_UP] != sys_inputs_previous[SYS_INPUT_MENU_UP])
  {
    if(sys_inputs[SYS_INPUT_MENU_UP])
    {
      menu->current_index = (menu->current_index + menu->num_items - 1) % menu->num_items;
    }
  }
  
  if(sys_inputs[SYS_INPUT_MENU_DOWN] != sys_inputs_previous[SYS_INPUT_MENU_DOWN])
  {
    if(sys_inputs[SYS_INPUT_MENU_DOWN])
    {
      menu->current_index = (menu->current_index + 1u) % menu->num_items;
    }
  }
}

void updatePauseMenu()
{
  updateMenu(&pause_menu);
  
  if(sys_inputs[SYS_INPUT_MENU_CHOOSE])
  {
    if(pause_menu.current_index == 1)
      resetLevel(&g_player, &g_gs, SDL_GetTicks());
    else if(pause_menu.current_index == 2)
      quitLevel();
    game_paused = FALSE;
  }
}


void drawHUD(uint16_t *const pixels, renderstate_t* rs, const gamestate_t* gs, const player_t* player)
{

}


void drawObjects(uint16_t *const pixels, const float time, renderstate_t* rs, gamestate_t* gs, player_t* player)
{
  hunk_marker_t mem_drawobjects;
  hunkMark(&mem_drawobjects);

  for(int idx = 0; idx < gs->num_mines; ++idx)
    if(gs->mines[idx].active)
    {
      clipAndPlotBox(pixels, gs->mines[idx].posx, gs->mines[idx].posy, PLAYER_SIZE, PLAYER_SIZE, RGB888toRGB565(255, 255, 255));
    }
    
  for(int idx = 0; idx < gs->num_bullets; ++idx)
    if(gs->bullets[idx].active)
    {
      clipAndPlotBox(pixels, gs->bullets[idx].posx, gs->bullets[idx].posy, PLAYER_SIZE, PLAYER_SIZE, RGB888toRGB565(255, 0, 0));
    }
    
    
  hunkFree(&mem_drawobjects);
}


#define RADIX_BITS         8
#define RADIX_MULTIPLIER   (1 << RADIX_BITS)

static int32_t player_x = 150 * RADIX_MULTIPLIER;
static int32_t player_y = 50 * RADIX_MULTIPLIER;

static int32_t camera_x = 0;
static int32_t camera_y = 0;

static int32_t player_vel_x = 0;
static int32_t player_vel_y = 0;

static const int32_t s = 16;
static const int32_t ps[2] = { 16, 32 };

void drawIngameScreen(uint16_t* pixels, renderstate_t* rs, const gamestate_t* gs)
{
  const float time = (float)((double)(gs->play_time) / 1000.0);
  const uint32_t frame_start = SDL_GetTicks();

  clipAndPlotBox(pixels, 0, 0, FRAME_WIDTH, FRAME_HEIGHT, RGB888toRGB565(50, 50, 50));


  const uint32_t hud_start = SDL_GetTicks();
  drawHUD(pixels, &g_rs, &g_gs, &g_player);

  if(game_paused)
   drawPauseMenu(pixels);

  const uint32_t hud_end = SDL_GetTicks();

  camera_x = player_x - 100 * RADIX_MULTIPLIER;
  camera_y = player_y - 100 * RADIX_MULTIPLIER;

  clipAndPlotBox(pixels, (player_x - camera_x) / RADIX_MULTIPLIER, (player_y - camera_y) / RADIX_MULTIPLIER, ps[0], ps[1], RGB888toRGB565(255, 0, 0));


  const uint32_t frame_end = SDL_GetTicks();

  if(rs->debug_display_enabled)
  {
    static char debug_text[8192];
    memset(debug_text, 0, sizeof(debug_text));
    snprintf(debug_text, sizeof(debug_text),
                     "hud_ms            = %u\n"
                     "total_ms          = %u\n"
                     "num_obj_drawn     = %u\n"
                     ,hud_end - hud_start, frame_end - frame_start, num_obj_drawn);

    plotString(pixels, 0, 0, debug_text, 1);
  }

}

void updateIngameLogic()
{

}
     


int main(int argc, char* argv[])
{
  printf("Compiled on " __DATE__ " at " __TIME__ ".\n");

#ifdef __GNUC__
# ifdef __GNUC_MINOR__
#   ifdef __GNUC_PATCHLEVEL__
  printf("Compiled with GCC %d.%d.%d\n", __GNUC__, __GNUC_MINOR__, __GNUC_PATCHLEVEL__);
#   endif
# endif
#endif

  printf("Compiled against SDL version %d.%d.%d\n", SDL_MAJOR_VERSION, SDL_MINOR_VERSION, SDL_PATCHLEVEL);

  const SDL_version* sdl_version = SDL_Linked_Version();

  if(sdl_version)
    printf("Running dynamically-linked SDL version %d.%d.%d\n", sdl_version->major, sdl_version->minor, sdl_version->patch);
  else
    printf("SDL_Linked_Version() returned NULL.\n");


  hunkInit();
  hunkMark(&mem_start);

  if(SDL_Init(SDL_INIT_TIMER | SDL_INIT_VIDEO))
      return -1;

  SDL_WM_SetCaption("", NULL);
  
  atexit(SDL_Quit);
  
  SDL_Surface* surface = SDL_SetVideoMode(FRAME_WIDTH, FRAME_HEIGHT, 16, SDL_HWSURFACE | SDL_DOUBLEBUF | SDL_ASYNCBLIT | SDL_ANYFORMAT);

  if(!surface)
      return -1;
    
  sysInitialise();

  g_camera.matrix[0] = 1;
  g_camera.matrix[1] = 0;
  g_camera.matrix[2] = 0;
  g_camera.matrix[3] = 1;
  g_camera.horizontalFOV = 3.14159265358979323f * 100. / 180.;
  g_looka = 0;
  g_camera.px = 3.5;
  g_camera.py = 6.5;
  g_camera.pz = .5;

  initRenderstate(&g_rs);

  g_pending_enginemode = ENGINE_MODE_INGAME;

  while(keep_going == 1)
  {
      if(g_pending_enginemode != g_enginemode)
      {
          transitionEngineMode(g_enginemode, g_pending_enginemode);
      }
    
      SDL_Event event;

      while(SDL_PollEvent(&event))
      {
          if(event.type == SDL_QUIT)
          {
              keep_going = 0;
          }
      }

      frame_time = SDL_GetTicks() - g_gs.start_level_time;

      if(frame_time >= g_gs.next_frame)
      {
          if(!sysUpdateInput())
            break;
          
          switch(g_enginemode)
          {
            case ENGINE_MODE_TRACKSELECTION:
              break;
            case ENGINE_MODE_INGAME:
              updateIngameLogic();
              break;
          }

          g_gs.next_frame += 15;
      }
      
      hunkMark(&mem_frame);
      
      SDL_LockSurface(surface);

      uint16_t* pixels = (uint16_t*)surface->pixels;
      
      switch(g_enginemode)
      {
        case ENGINE_MODE_TRACKSELECTION:
          break;
        case ENGINE_MODE_INGAME:
          drawIngameScreen(pixels, &g_rs, &g_gs);
          break;
      }

      SDL_UnlockSurface(surface);

      SDL_UpdateRect(surface, 0, 0, 0, 0);
      SDL_Flip(surface);

      hunkFree(&mem_frame);
  }


  printf("mem_hunk_highwater = %u\n", mem_hunk_highwater);
  
  return 0;
}
