#ifndef __RENDER_H__
#define __RENDER_H__

#include <stdint.h>

#include "defs.h"


typedef struct
{
  float px, py, pz;
  float horizontalFOV;
  float matrix[4];
  float angle;
} camera_t;

typedef struct
{
  float aspect;
  float viewport_zoom;
  float viewport[4];
  bool_t debug_display_enabled;
  uint16_t captured_frame[FRAME_WIDTH * FRAME_HEIGHT];
  bool_t frame_captured;
  float time;

  uint16_t tex_woodtile[TEXTURE_TILE_SIZE * TEXTURE_TILE_SIZE];
  uint16_t tex_woodb[TEXTURE_TILE_SIZE * TEXTURE_TILE_SIZE];
  uint16_t tex_darkwood[TEXTURE_TILE_SIZE * TEXTURE_TILE_SIZE];
  uint16_t tex_creakywood[TEXTURE_TILE_SIZE * TEXTURE_TILE_SIZE];
  uint16_t tex_crosscube[TEXTURE_TILE_SIZE * TEXTURE_TILE_SIZE];
  uint16_t tex_metaltile[TEXTURE_TILE_SIZE * TEXTURE_TILE_SIZE];
  uint16_t tex_hexagons[TEXTURE_TILE_SIZE * TEXTURE_TILE_SIZE];

} renderstate_t;

void renderColumns(uint16_t* pixel_buffer, renderstate_t* rs, camera_t* camera, const float* map, const float* map2);

void captureFrame(uint16_t* pixel_buffer, renderstate_t* rs);

void initRenderstate(renderstate_t* rs);
void plotString(uint16_t* pixel_buffer, uint32_t x, uint32_t y, const char* str, int scale_factor);
void plotLine3D(uint16_t* pixel_buffer, const float* v0vsp, const float* v1vsp, const uint32_t col, renderstate_t* rs);
void plotChar(uint16_t* pixel_buffer, uint32_t x, uint32_t y, uint32_t c, int scale_factor);
void plotLine(uint16_t* pixel_buffer, int x0, int y0, int x1, int y1, const uint32_t col);
void plotPixel(uint16_t* pixel_buffer, uint32_t x, uint32_t y, uint32_t colour);
void clipAndPlotBox(uint16_t* pixel_buffer, int32_t x, int32_t y, int32_t w, int32_t h, uint32_t colour);

static inline uint16_t RGB888toRGB565(uint8_t r, uint8_t g, uint8_t b)
{
  return (b >> 3u) | ((g & 0xfcu) << 3u) | ((r & 0xf8u) << 8u);
}

static inline uint32_t RGB565toRGB888(uint16_t rgb)
{
  uint8_t r = (rgb >> 11u) & 0x1fu;
  uint8_t g = (rgb >> 5u) & 0x3fu;
  uint8_t b = rgb & 0x1fu;
  return (r << 27) | (g << 18) | (b << 3);
}

// The least-significant bit of the green channel is reserved for alpha blending.
// So the result is effectively 15bpp, but using the 16bpp format.
static inline uint16_t RGB888toRGB555(uint8_t r, uint8_t g, uint8_t b)
{
  return (b >> 3u) | ((g & 0xf8u) << 3u) | ((r & 0xf8u) << 8u);
}

static inline uint16_t RGBA565mult(uint16_t c0, uint16_t c1)
{
  uint16_t r = (((c0 >> 11u) & 0x1fu) * ((c1 >> 11u) & 0x1fu)) >> 5u;
  uint16_t g = (((c0 >> 5u) & 0x3fu) * ((c1 >> 5u) & 0x3fu)) >> 6u;
  uint16_t b = ((c0 & 0x1fu) * (c1 & 0x1fu)) >> 5u;
  return (r << 11u) | (g << 5u) | b;
}

static inline uint16_t RGBA565blend(uint16_t c0, uint16_t c1)
{
  return ((c0 >> 1) & 0b0111101111101111) + ((c1 >> 1) & 0b0111101111101111);
}


#endif // __RENDER_H__