
#include <stddef.h>
#include <stdlib.h>
#include <stdio.h>
#include <assert.h>
#include <math.h>
#include <string.h>

#include "render.h"
#include "defs.h"
#include "font.h"
#include "mem.h"
#include "input.h"

#define STB_IMAGE_IMPLEMENTATION
#include "stb_image.h"

bool_t extern_write_frame=FALSE;

static int sign(float x)
{
  return (x < 0.0) ? -1 : +1;
}

void initRenderstate(renderstate_t* rs)
{
  memset(rs, 0, sizeof(renderstate_t));
  rs->aspect = (float)FRAME_HEIGHT / (float)FRAME_WIDTH;
  rs->viewport_zoom = 0.95f;
}

void clipAndPlotBox(uint16_t* pixel_buffer, int32_t x, int32_t y, int32_t w, int32_t h, uint32_t colour)
{
  if(-x > w || -y > h || x >= FRAME_WIDTH || y >= FRAME_HEIGHT)
    return;
  
  if(x < 0)
  {
    w += x;
    x = 0;
  }
  
  if(y < 0)
  {
    h += y;
    y = 0;
  }
  
  if(x + w > FRAME_WIDTH)
    w = FRAME_WIDTH - x;
  
  if(y + h > FRAME_HEIGHT)
    h = FRAME_HEIGHT - y;
  
  for(int32_t v = 0; v < h; ++v)
    for(int32_t u = 0; u < w; ++u)
      pixel_buffer[(x + u) + (y + v) * FRAME_WIDTH] = colour;   
}

void plotPixel(uint16_t* pixel_buffer, uint32_t x, uint32_t y, uint32_t colour)
{
    if(x < FRAME_WIDTH && y < FRAME_HEIGHT)
        pixel_buffer[x + y * FRAME_WIDTH] = colour;
}

// bresenham line rastering
void plotLine(uint16_t* pixel_buffer, int x0, int y0, int x1, int y1, const uint32_t col)
{
   const bool_t steep = labs(y1 - y0) > labs(x1 - x0);

   if(steep)
   {
      SWAP(int, x0, y0);
      SWAP(int, x1, y1);
   }

   if(x0 > x1)
   {
      SWAP(int, x0, x1);
      SWAP(int, y0, y1);
   }

   const int deltax = x1 - x0;
   const int deltay = labs(y1 - y0);
   int error = deltax / 2;
   int y = y0;
   const int ystep = (y0 < y1) ? 1 : -1;

   for(int x = x0; x <= x1; ++x)
   {
      if(steep)
         plotPixel(pixel_buffer, y, x, col);
      else
         plotPixel(pixel_buffer, x, y, col);

      error -= deltay;

      if(error < 0)
      {
         y += ystep;
         error += deltax;
      }
   }
}

void plotChar(uint16_t* pixel_buffer, uint32_t x, uint32_t y, uint32_t c, int scale_factor)
{
    for(uint32_t v = 0; v < 12; ++v)
        for(uint32_t u = 0; u < 6; ++u)
        {
            const uint32_t u2 = u + (c & 31) * 6;
            const uint32_t v2 = v + (c / 32) * 12;
            const uint32_t x2 = x + u * scale_factor;
            const uint32_t y2 = y + v * scale_factor;
            if(u2 < FONT_IMAGE_W && v2 < FONT_IMAGE_H)
            {
                const uint32_t texel_index = (u2) + (FONT_IMAGE_H - 1 - v2) * FONT_IMAGE_W;
                const uint32_t colour = (font_image_data[texel_index / 32U] & (0x80000000U >> (texel_index & 31U))) ? 0xffffffU : 0U;
                for(int i = 0; i <  scale_factor; ++i)
                  for(int j = 0; j < scale_factor; ++j)
                  {
                    const uint32_t x3 = x2 + i;
                    const uint32_t y3 = y2 + j;
                    if(x3 < FRAME_WIDTH && y3 < FRAME_HEIGHT/* && colour > 0*/)
                        pixel_buffer[x3 + y3 * FRAME_WIDTH] = colour;
                  }
            }
        }
}

      
void plotLine3D(uint16_t* pixel_buffer, const float* v0vsp, const float* v1vsp, const uint32_t col, renderstate_t* rs)
{

  
  float line[2][3];

    line[0][0] = v0vsp[0];
    line[0][1] = v0vsp[1];
    line[0][2] = v0vsp[2];

    line[1][0] = v1vsp[0];
    line[1][1] = v1vsp[1];
    line[1][2] = v1vsp[2];
    

  float projected[2][2];
  uint32_t projected_pix[2][2];

  for(int j = 0; j < 2; ++j)
  {
      const float z = 1.0f / line[j][2];
      const float x = line[j][0] * z;
      const float y = line[j][1] * z / -rs->aspect;
      projected[j][0] = x;
      projected[j][1] = y;
  }

  const float linebox[4] = {   MIN(projected[0][0], projected[1][0]), MIN(projected[0][1], projected[1][1]),
                               MAX(projected[0][0], projected[1][0]), MAX(projected[0][1], projected[1][1]) };

  if(linebox[0] < -1.001f || linebox[1] < -1.001f || linebox[2] > +1.001f || linebox[3] > +1.001f)
      return;

  for(int j = 0; j < 2; ++j)
  {
      projected_pix[j][0] = (projected[j][0] * 0.5f + 0.5f) * FRAME_WIDTH;
      projected_pix[j][1] = (projected[j][1] * 0.5f + 0.5f) * FRAME_HEIGHT;
  }


  plotLine(pixel_buffer,  projected_pix[0][0], projected_pix[0][1], projected_pix[1][0], projected_pix[1][1], col);

}

void plotString(uint16_t* pixel_buffer, uint32_t x, uint32_t y, const char* str, int scale_factor)
{
    const uint32_t x2 = x;
    while(str[0])
    {
        if(str[0] == '\n')
        {
            x = x2;
            y += 12 * scale_factor;
        }
        else if(str[0] >= 32)
        {
            plotChar(pixel_buffer, x, y, str[0] - 32, scale_factor);
            x += 6 * scale_factor;
        }
        ++str;
    }
}

