// CYBER-TRAVERSER
// Cyber Coaster

#include <SDL.h>
#include <math.h>
#include <assert.h>

#include <set>
#include <string>

#include "font.h"
#include "defs.h"
#include "mem.h"
#include "render.h"
#include "game.h"
#include "input.h"

#include "6502.h"

#define STB_RECT_PACK_IMPLEMENTATION
#include "stb_rect_pack.h"

#define ENGINE_MODE_NULL            0
#define ENGINE_MODE_TITLE           1
#define ENGINE_MODE_TRACKSELECTION  2
#define ENGINE_MODE_INGAME          3

static bool_t game_paused = FALSE;
static int keep_going = 1;

static renderstate_t g_rs;
static player_t g_player;
static gamestate_t g_gs;

static int num_obj_drawn = 0;

uint32_t frame_time = 0;

int g_pending_enginemode = ENGINE_MODE_NULL;
int g_enginemode = ENGINE_MODE_NULL;

static int sign(float x)
{
  return (x < 0.0) ? -1 : +1;
}

struct xorshift32_state {
    uint32_t a;
};

/* The state must be initialized to non-zero */
uint32_t xorshift32(struct xorshift32_state *state)
{
	/* Algorithm "xor" from p. 4 of Marsaglia, "Xorshift RNGs" */
	uint32_t x = state->a;
	x ^= x << 13;
	x ^= x >> 17;
	x ^= x << 5;
	return state->a = x;
}


// https://media.xiph.org/svt/SGIIMAGESPEC

static unsigned char getbyte(FILE *inf)
{
  unsigned char buf[1];
  fread(buf,1,1,inf);
  return buf[0];
}

static unsigned short getshort(FILE *inf)
{
  unsigned char buf[2];
  fread(buf,2,1,inf);
  return (buf[0]<<8)+(buf[1]<<0);
}

static unsigned long getlong(FILE *inf)
{
  unsigned char buf[4];
  fread(buf,4,1,inf);
  return (buf[0]<<24)+(buf[1]<<16)+(buf[2]<<8)+(buf[3]<<0);
}

uchar* loadFromRGBAFile(FILE *const src, int* out_w, int* out_h)
{
  short magic = getshort(src);
  assert(magic == 474);
  char storage = getbyte(src);
  char bpc = getbyte(src);
  short dimension = getshort(src);
  short xsize = getshort(src);
  short ysize = getshort(src);
  short zsize = getshort(src);
  long pixmin = getlong(src);
  long pixmax = getlong(src);
  long dummy = getlong(src);
  char name[80] = { 0 };
  fread(name, 1, 80, src);
  long colormap = getlong(src);
  char dummy2[404];
  fread(dummy2, 1, 404, src);
  printf("magic = %u\n", magic);
  printf("storage = %u\n", storage);
  printf("bpc = %u\n", bpc);
  printf("dimension = %u\n", dimension);
  printf("xsize = %u\n", xsize);
  printf("ysize = %u\n", ysize);
  printf("zsize = %u\n", zsize);
  printf("pixmin = %u\n", pixmin);
  printf("pixmax = %u\n", pixmax);
  printf("name = '%s'\n", name);
  printf("colormap = %u\n", colormap);
  uchar* pixels = (uchar*)malloc(xsize * ysize * zsize);
  fread(pixels, 1, xsize * ysize * zsize, src);
  *out_w = xsize;
  *out_h = ysize;
  uchar* pixels2 = (uchar*)malloc(xsize * ysize * zsize);
  // re-interleave
  for(int y = 0; y < ysize; ++y)
  {
    for(int x = 0; x < xsize; ++x)
    {
      for(int z = 0; z < zsize; ++z)
      {
        pixels2[z + (x + (ysize-1-y) * xsize) * zsize] = pixels[(x + y * xsize) + z * xsize * ysize];
      }
    }
  }
  free(pixels);
  return pixels2;
}

static void putbyte(FILE *outf, unsigned char x)
{
  unsigned char buf[1] = { x };
  fwrite(buf,1,1,outf);
}

static void putshort(FILE *outf, unsigned short x)
{
  unsigned char buf[2] = { (unsigned char)(x >> 8), (unsigned char)(x & 0xff) };
  fwrite(buf,2,1,outf);
}

static void putlong(FILE *outf, unsigned long x)
{
  unsigned char buf[4] = { (unsigned char)(x >> 24),
                           (unsigned char)((x >> 16) & 0xff),
                           (unsigned char)((x >> 8) & 0xff),
                           (unsigned char)(x & 0xff)  };
  fwrite(buf,4,1,outf);
}

static void writeRGBAFile(FILE *const out, int w, int h, unsigned char* pixels)
{
    putshort(out, 474); // magic
    putbyte (out,   0); // storage
    putbyte (out,   1); // bpc
    putshort(out,   3); // dimensions
    putshort(out,   w); // xsize
    putshort(out,   h); // ysize
    putshort(out,   3); // zsize
    putlong (out,   0); // pixmin
    putlong (out,   0xffffff); // pixmax
    putlong (out,   0); // dummy
    char name[80] = { 0 };
    fwrite(name, 1, 80, out);
    putlong (out,   0); // colormap ID
    char dummy2[404];
    memset(dummy2, 0, sizeof(dummy2));
    fwrite(dummy2, 1, 404, out);

    unsigned char* pixels2 = (unsigned char*)malloc(w * h * 3);
    // re-interleave
    for(int y = 0; y < h; ++y)
    {
        for(int x = 0; x < w; ++x)
        {
            for(int z = 0; z < 3; ++z)
            {
                pixels2[(x + y * w) + z * w * h] = pixels[z + (x + (h-1-y) * w) * 3];
            }
        }
    }

    fwrite(pixels2, 1, w * h * 3, out);
    
    free(pixels2);
}

static uint8_t virtual_framebuffer[VIRTUAL_FRAME_WIDTH*VIRTUAL_FRAME_HEIGHT];
static uint8_t palette[0x40*3];

static void blitVirtualFramebuffer(uint16_t* pixel_buffer)
{
  for(int y=0;y<VIRTUAL_FRAME_HEIGHT;++y)
    for(int x=0;x<VIRTUAL_FRAME_WIDTH;++x)
    {
      for(int v=0;v<VIRTUAL_PIXEL_HEIGHT*FRAME_ZOOM_FACTOR;++v)
        for(int u=0;u<VIRTUAL_PIXEL_WIDTH*FRAME_ZOOM_FACTOR;++u)
        {
          uint8_t r=palette[virtual_framebuffer[x+y*VIRTUAL_FRAME_WIDTH]*3+0];
          uint8_t g=palette[virtual_framebuffer[x+y*VIRTUAL_FRAME_WIDTH]*3+1];
          uint8_t b=palette[virtual_framebuffer[x+y*VIRTUAL_FRAME_WIDTH]*3+2];
          plotPixel(pixel_buffer, x * VIRTUAL_PIXEL_WIDTH*FRAME_ZOOM_FACTOR + u, y * VIRTUAL_PIXEL_HEIGHT*FRAME_ZOOM_FACTOR + v, RGB888toRGB565(r,g,b));
        }
    }
}

static void clearVirtualFramebuffer()
{
  for(int y=0;y<VIRTUAL_FRAME_HEIGHT;++y)
    for(int x=0;x<VIRTUAL_FRAME_WIDTH;++x)
    {
      virtual_framebuffer[x+y*VIRTUAL_FRAME_WIDTH] = 0;
    }
}

static void plotPixelVirtual(uint32_t x, uint32_t y, const uint8_t colour)
{
    if(x < VIRTUAL_FRAME_WIDTH && y < VIRTUAL_FRAME_HEIGHT)
        virtual_framebuffer[x + y * VIRTUAL_FRAME_WIDTH] = colour;
}

static void plotLineVirtual(int x0, int y0, int x1, int y1, const uint8_t col)
{
   const bool_t steep = labs(y1 - y0) > labs(x1 - x0);

   if(steep)
   {
      SWAP(int, x0, y0);
      SWAP(int, x1, y1);
   }

   if(x0 > x1)
   {
      SWAP(int, x0, x1);
      SWAP(int, y0, y1);
   }

   const int deltax = x1 - x0;
   const int deltay = labs(y1 - y0);
   int error = deltax / 2;
   int y = y0;
   const int ystep = (y0 < y1) ? 1 : -1;

   for(int x = x0; x <= x1; ++x)
   {
      if(steep)
         plotPixelVirtual(y, x, col);
      else
         plotPixelVirtual(x, y, col);

      error -= deltay;

      if(error < 0)
      {
         y += ystep;
         error += deltax;
      }
   }
}

static uint8_t RAM_6502[2048];  // 0x0000-0x0800
static uint8_t ROM_6502[32768]; // 0x8000-0xFFFF
static M6502 m6502;

static bool m6502_done=false;

static zuint8 read_6502(void *context, zuint16 address)
{
  if(address>=0x8000)
    return ROM_6502[address-0x8000];
  else if(address<0x800)
    return RAM_6502[address];
  assert(false);
  return 0;
}

static void write_6502(void *context, zuint16 address, zuint8 value)
{
  if(address==0x07FF)
    m6502_done=true;
  if(address<0x0800)
    RAM_6502[address] = value;
}

static zusize total_cycles=0;

static void do6502()
{
  memset(&m6502,0,sizeof(m6502));
  m6502.read=&read_6502;
  m6502.write=&write_6502;
  m6502.state.pc=0x8000;
  m6502_done=false;
  total_cycles=0;
  while(!m6502_done)
  {
    zusize cycles=m6502_run(&m6502, 1);
    total_cycles+=cycles;
  }
  //printf("total_cycles=%d\n",(int)total_cycles);
}

static int8_t g_cam_x = 0;
static int8_t g_cam_y = 0;
static uint16_t g_cam_sub_z = 0;
static uint16_t g_cam_speed = 0;
static uint8_t g_rect_offset = 0;
static const uint16_t top_speed=4*256;
static uint8_t g_time_remaining=10;
static int frame_count = 0;
static int g_lap_count = 0;
static int g_laps = 3;

#define MODE_INGAME   0
#define MODE_TIMEOVER 1
#define MODE_CRASHED  2
#define MODE_WON      3

static int g_game_mode = MODE_INGAME;

static void tweenRectangle( double t,double* current,double t0,double t1,
                            double x,double y,double w,double h)
{
  double next[4]={x,y,w,h};
  double f=MAX(0.0,MIN(1.0,(t-t0)/(t1-t0)));
  f=3.0*pow(f,2.0)-2.0*pow(f,3.0);
  if(f>=1.0)
    for(int i=0;i<4;++i)
      current[i]=next[i];
  else if(f>=0.0)
    for(int i=0;i<4;++i)
      current[i]=next[i]*f+current[i]*(1.0-f);
}

static double frand()
{
  return double(rand())/double(RAND_MAX);
}

static void drawBigTextInNametable(uint8_t* nametable,const char* s,int x,int y)
{
  int l=strlen(s);
  for(int i=0;i<l;++i)
  {
    nametable[x+(y+0)*32+i]=(s[i]-32)*2+0+64;
    nametable[x+(y+1)*32+i]=(s[i]-32)*2+1+64;
  }
}

void drawIngameScreen(uint16_t* pixels, renderstate_t* rs, const gamestate_t* gs)
{
  const uint32_t frame_start = SDL_GetTicks();
  //const float time = 1.2f;
  const float time = float(frame_start)/1000.0f * 1.0f;

  static const int pattern_bank_size=256*8*8;
  static const int num_banks=4;

  static uint8_t patterns[pattern_bank_size*num_banks];
  static int vector_to_pattern_right[9*9*9*9];
  static int vector_to_pattern_left[9*9*9*9];
  static int pattern_to_vector[256];

  static const int num_title_screen_graphic_frames=12;
  static const int flatfont_height=5;

  static uint8_t title_screen_graphic[128*8];
  static uint8_t title_screen_graphic_scaled[128*flatfont_height*num_title_screen_graphic_frames];

  // In the ROM:
  static uint8_t strip_splat_table[256*8];
  static uint8_t pattern_banks[256*16*num_banks];
  static uint8_t sprites_timeover[256];
  static uint8_t sprites_crashed[256];
  static uint8_t sprites_won[256];
  static uint8_t sprites_paused[256];
  static uint8_t hud_nametable[4*32];
  static uint8_t stars_nametables[2048];

  static bool_t init=TRUE;

  if(init)
  {
    init=FALSE;

    memset(patterns,0,sizeof(patterns));
    memset(vector_to_pattern_right,0,sizeof(vector_to_pattern_right));
    memset(vector_to_pattern_left,0,sizeof(vector_to_pattern_left));
    memset(pattern_to_vector,0,sizeof(pattern_to_vector));
    memset(strip_splat_table,0,sizeof(strip_splat_table));
    memset(pattern_banks,0,sizeof(pattern_banks));

    static uint8_t remap[256];

    for(int i=0;i<256;++i)
      remap[i]=i;
    remap[0]=0xFF;
    remap[0xFF]=0;

    {
      FILE* in=fopen("2C02G_wiki.pal","rb");
      fread(palette,1,sizeof(palette),in);
      fclose(in);
    }

    int count=0;
    int count_left=0;
    int count_right=0;

    // Left side
    for(int a=0;a<=8;++a)
      for(int b=0;b<=8;++b)
        for(int c=0;c<=8;++c)
        {
          if(a+b+c==8)
          {
            assert(count<256);
            assert(count_left<45);

            vector_to_pattern_left[a*9*9*9 + b*9*9 + c*9] = remap[count];
            pattern_to_vector[remap[count]] = a*9*9*9 + b*9*9 + c*9;

            int bank=0;

            for(int y=0;y<8;++y)
              for(int x=0;x<8;++x)
              {
                patterns[bank*pattern_bank_size+(remap[count]*8+y)*8+x]=(
                                x < a ? 3 :
                          x < (a + b) ? 2 :
                                        1);
              }

            ++count;
            ++count_left;
          }
        }

    // Right side
    for(int a=0;a<=8;++a)
      for(int b=0;b<=8;++b)
        for(int c=0;c<=8;++c)
          for(int d=0;d<=8;++d)
          {
            if(a+b+c+d==8)
            {
              assert(count<256);
              assert(count_right<165);

              vector_to_pattern_right[a*9*9*9 + b*9*9 + c*9 + d] = remap[count];
              pattern_to_vector[remap[count]] = a*9*9*9 + b*9*9 + c*9 + d;

              int bank=0;

              for(int y=0;y<8;++y)
                for(int x=0;x<8;++x)
                {
                  patterns[bank*pattern_bank_size+(remap[count]*8+y)*8+x]=(
                                  x < a ? 0 :
                            x < (a + b) ? 1 :
                        x < (a + b + c) ? 2 :
                                          3);
                }

              ++count;
              ++count_right;
            }
          }

    {
      FILE* in=fopen("gsmplay8x15.rgba","rb"); // https://github.com/pinobatch/bitmap-fonts/blob/master/8x16/gsmplay8x15.png
      assert(in);
      int w=0,h=0;
      uchar* font_rgba=loadFromRGBAFile(in,&w,&h);
      for(int i=0;i<256;++i)
      {
        int bank=1;
        int x0=((i/2)&15)*8;
        int y0=(i/32)*16+(i&1)*8;
        for(int y=0;y<8;++y)
          for(int x=0;x<8;++x)
          {
            int fc=font_rgba[((y0+y)*w+x0+x)*3+0];
            int pc=fc > 128 ? 3 : 0;
            // Add a drop shadow
            if(pc==0 && ((y0+y)&15)!=0 && x>0 && font_rgba[((y0+y-1)*w+x0+x-1)*3+0]>128)
              pc=1;
            patterns[bank*pattern_bank_size+(i*8+y)*8+x]=pc;
          }
      }
      {
        // Title screen graphics
        memset(title_screen_graphic,0,sizeof(title_screen_graphic));

/*

     ** * * ***  *** ***  
    *   * * *  * *   *  * 
    *   *** ***  **  ***  
    *    *  *  * *   * *  
     **  *  ***  *** *  * 

     **  **   **   ***  ***** *** ***  
    *   *  * *  * *       *   *   *  * 
    *   *  * ****  ***    *   *** ***  
    *   *  * *  *     *   *   *   * *  
     **  **  *  *  ***    *   *** *  * 

*/

        static const uint16_t flatfont[]= {
            //                   |          C
            0b0000000000000000,      0b1111111111111111,
            0b0000000000000000,      0b1111100000000000,
            0b0000000000000000,      0b1111100000000000,
            0b0000000000000000,      0b1111100000000000,
            0b0000000000000000,      0b1111111111111111,
            //       C           |          O
            0b1111111111111111,      0b1111111111111111,
            0b1111110000000000,      0b1111100000011111,
            0b1111110000000000,      0b1111100000011111,
            0b1111110000000000,      0b1111100000011111,
            0b1111111111111111,      0b1111111111111111,
            //       Y           |          A
            0b1111100000011111,      0b1111111111111111,
            0b1111100000011111,      0b1111100000011111,
            0b1111111111111111,      0b1111111111111111,
            0b0000011111100000,      0b1111100000011111,
            0b0000011111100000,      0b1111100000011111,
            //       B           |          S
            0b1111111111110000,      0b1111111111111111,
            0b1111100000111111,      0b1111100000000000,
            0b1111111111110000,      0b1111111111111111,
            0b1111100000111111,      0b0000000000011111,
            0b1111111111110000,      0b1111111111111111,
            //       E           |          T
            0b1111111111111111,      0b1111111111111111,
            0b1111100000000000,      0b0000011111000000,
            0b1111111111110000,      0b0000011111000000,
            0b1111100000000000,      0b0000011111000000,
            0b1111111111111111,      0b0000011111000000,
            //       R           |          E
            0b1111111111110000,      0b1111111111111111,
            0b1111100000111111,      0b1111100000000000,
            0b1111111111110000,      0b1111111111110000,
            0b1111100000111111,      0b1111100000000000,
            0b1111100000111111,      0b1111111111111111,
            //                   |          R
            0b0000000000000000,      0b1111111111110000,
            0b0000000000000000,      0b1111100000111111,
            0b0000000000000000,      0b1111111111110000,
            0b0000000000000000,      0b1111100000111111,
            0b0000000000000000,      0b1111100000111111,
        };

        for(int i=0;i<7;++i)
        {
          for(int y=0;y<flatfont_height;++y)
            for(int x=0;x<16;++x)
            {
/*
              if(flatfont[i*2*5 + y*2 + 1]&(1<<(15-x)))
                title_screen_graphic[(x+i*17+5)+(y+1)*128]|=1;
              if(flatfont[i*2*5 + y*2 + 0]&(1<<(15-x)))
                title_screen_graphic[(x+i*17+5)+(y+1)*128]|=2;
*/
              if(flatfont[i*2*5 + y*2 + 1]&(1<<(15-x)))
                title_screen_graphic[(x+i*17+5)+(y+0)*128]|=1;
              if(flatfont[i*2*5 + y*2 + 0]&(1<<(15-x)))
                title_screen_graphic[(x+i*17+5)+(y+0)*128]|=2;
            }
        }

        memset(title_screen_graphic_scaled,3,sizeof(title_screen_graphic_scaled));

        // Create scaled copies
        for(int i=0;i<num_title_screen_graphic_frames;++i)
        {
          const double w=128-i;
          for(int y=0;y<flatfont_height;++y)
            for(double x=0;x<128;++x)
            {
              int p=3;
              if(x>64-w/2.0 && x<64+w/2.0)
                p = title_screen_graphic[int((x-64+w/2.0)*(128.0/w))+y*128];
              assert(p>=0);
              assert(p<4);
              if(p==3)
                p=0;
              else if(p==0)
                p=3;
              title_screen_graphic_scaled[int(x)+(y+i*5)*128] = p;
            }

              //if(x<w)
              //  title_screen_graphic_scaled[x+(y+i*5)*128] =
              //    title_screen_graphic[int(double(x)*(128.0/double(w)))+y*128];
        }

        // Convert to 2bpp
        static const int h=(flatfont_height*num_title_screen_graphic_frames+7)&0xFF8;
        static uint8_t title_screen_graphic_scaled_2bpp[(128*h)/4];
        assert(sizeof(title_screen_graphic_scaled_2bpp)%16==0);

        memset(title_screen_graphic_scaled_2bpp,0,sizeof(title_screen_graphic_scaled_2bpp));

        //static uint8_t title_nametable[16*(h/8)];
        //memset(title_nametable,0,sizeof(title_nametable));

        for(int y=0;y<h;y+=8)
          for(int x=0;x<128;x+=8)
          {
            const int c=x/8+(y/8*16);
            for(int y2=0;y2<8;++y2)
              for(int x2=0;x2<8;++x2)
              {
                if((y+y2)<flatfont_height*num_title_screen_graphic_frames)
                {
                  const int i=(y+y2)/flatfont_height;
                  int p=title_screen_graphic_scaled[int(x+x2)+(((y+y2)%flatfont_height)+i*flatfont_height)*128];
                  //if(x<16 || x>=128-16) p+=1;
                  assert(p>=0);
                  assert(p<4);
                  title_screen_graphic_scaled_2bpp[c*16+y2+0]|=(p&1)<<(7-x2);
                  title_screen_graphic_scaled_2bpp[c*16+y2+8]|=((p&2)>>1)<<(7-x2);
                }
                else
                {
                  title_screen_graphic_scaled_2bpp[c*16+y2+0]|=1<<(7-x2);
                  title_screen_graphic_scaled_2bpp[c*16+y2+8]|=1<<(7-x2);
                }
              }
          }

        {
          FILE* out=fopen("title_screen_graphic_scaled_2bpp.bin","wb");
          fwrite(title_screen_graphic_scaled_2bpp,1,sizeof(title_screen_graphic_scaled_2bpp),out);
          fclose(out);
        }

/*
        {
          FILE* out=fopen("title_nametable.bin","wb");
          fwrite(title_nametable,1,sizeof(title_nametable),out);
          fclose(out);
        }
*/

/*
        //                 01234567
        const char* line1=" CYBER  ";
        const char* line2="COASTER ";
        for(int i=0;i<8;++i)
        {
          int j0=line2[i];
          int x0=(j0&15)*8;
          int y0=(j0/16)*16;

          for(int y=0;y<8;++y)
            for(int x=0;x<16;++x)
            {
              if(font_rgba[((y0+-1+(y*16)/7)*w+x0+x/2)*3+0])
                title_screen_graphic[(x+i*16)+y*128]|=1;
            }
        }
*/
      }
    }

    // Put large number digits into the patterns in bank 3
    {
      FILE* in=fopen("numbers_16x24.rgba","rb"); // Based on https://github.com/pinobatch/bitmap-fonts/blob/master/8x16/gsmplay8x15.png but modified by me.
      assert(in);
      int w=0,h=0;
      uchar* font_rgba=loadFromRGBAFile(in,&w,&h);
      for(int i=0;i<(w*h)/64;++i)
      {
        int bank=3;
        int x0=(i/3)*8;
        int y0=(i%3)*8;
        for(int y=0;y<8;++y)
          for(int x=0;x<8;++x)
          {
            int fc=font_rgba[((y0+y)*w+x0+x)*3+0];
            int pc=fc > 128 ? 3 : 0;
            patterns[bank*pattern_bank_size+(i*8+y)*8+x]=pc;
          }
      }
    }

    {
      FILE* in=fopen("upslant8x8.rgba","rb"); // https://raw.githubusercontent.com/pinobatch/bitmap-fonts/refs/heads/master/8x8/upslant8x8.png
      assert(in);
      int w=0,h=0;
      uchar* font_rgba=loadFromRGBAFile(in,&w,&h);
      for(int c=0;c<64;++c)
      {
        const int i=0x00+c;
        int bank=1;

        for(int y=0;y<8;++y)
          for(int x=0;x<8;++x)
          {
            int fc=font_rgba[((y+(c&~7))*w+(c&7)*8+x)*3+0];
            int pc=fc > 200 ? 3 : fc > 32 ? 2 : 0;
            patterns[bank*pattern_bank_size+(i*8+y)*8+x]=pc;
          }
      }
    }

    {
#if 0
      static const int dither[8][8] = {
        {  0, 32,  8, 40,  2, 34, 10, 42 },
        { 48, 16, 56, 24, 50, 18, 58, 26 },
        { 12, 44,  4, 36, 14, 46,  6, 38 },
        { 60, 28, 52, 20, 62, 30, 54, 22 },
        {  3, 35, 11, 43,  1, 33,  9, 41 },
        { 51, 19, 59, 27, 49, 17, 57, 25 },
        { 15, 47,  7, 39, 13, 45,  5, 37 },
        { 63, 31, 55, 23, 61, 29, 53, 21 },
      };
      const int bank=1;
      const int base=192;
      const int inset=3;
      memset(&patterns[bank*pattern_bank_size+base*64],0,64*64);
      for(int i=0;i<8;++i)
      {
        for(int x=(8-inset);x<(16-inset);++x)
          for(int y=0;y<8-inset;++y)
          {
            int c=(x-(8-inset))>i ? 0 :
                  1+(x-(8-inset)+4)/16+(dither[x&7][y&7]<((x-(8-inset)+4)&15)*4);
            assert(c>=0);
            assert(c<4);
            patterns[bank*pattern_bank_size+((i+base)*8+y+inset)*8+x-(8-inset)]=c;
          }

        for(int x=(8-inset);x<(16-inset);++x)
          for(int y=0;y<8-inset;++y)
          {
            int c=(x-(8-inset))>i ? 0 :
                  1+(x+8-(8-inset)+4)/16+(dither[x&7][y&7]<((x+8-(8-inset)+4)&15)*4);
            assert(c>=0);
            assert(c<4);
            patterns[bank*pattern_bank_size+((i+base+8)*8+y+inset)*8+x-(8-inset)]=c;
          }
      }
#endif
    }

    {
      // Create titlescreen version of the small font (since the backdop colour
      // is white).
      int bank=3;
      static uint8_t acme_char_table_titlescreen[256];
      memset(acme_char_table_titlescreen,0,sizeof(acme_char_table_titlescreen));
      //             01 23456789ABCEF0123456789ABCDEF0123456789ABCDEF
      const char* s=" !\"().,0123456789?ABCDEFGHIJKLMNOPQRSTUVWXYZ:;&";
      int pmap[4] = { 3, 1, 2, 0 };
      for(int i=0;i<48;++i)
      {
        int c=0x50+i;
        acme_char_table_titlescreen[s[i]]=c;
        for(int y=0;y<8;++y)
          for(int x=0;x<8;++x)
          {
            int p=patterns[1*pattern_bank_size+((s[i]-32)*8+y)*8+x];
            patterns[bank*pattern_bank_size+(c*8+y)*8+x]=pmap[p];
          }
      }
      // Make an arrow for the selection
      static uint8_t arrow[64] = {
          3,3,3,3,3,3,3,3,
          3,3,3,3,0,3,3,3,
          3,0,0,0,0,0,3,3,
          3,0,0,0,0,0,0,3,
          3,0,0,0,0,0,3,3,
          3,3,3,3,0,3,3,3,
          3,3,3,3,3,3,3,3,
          3,3,3,3,3,3,3,3,
        };
      memcpy(&patterns[bank*pattern_bank_size+0x7F*64],arrow,64);

      FILE* out=fopen("acme_char_table_titlescreen.bin","wb");
      fwrite(acme_char_table_titlescreen,1,sizeof(acme_char_table_titlescreen),out);
      fclose(out);
    }

    // Convert patterns to 2bpp
    for(int bank=0;bank<num_banks;++bank)
    {
      for(int i=0;i<256;++i)
      {
        for(int y=0;y<8;++y)
          for(int x=0;x<8;++x)
          {
            int p=patterns[bank*pattern_bank_size+(i*8+y)*8+x];
            assert(p>=0);
            assert(p<4);
            pattern_banks[bank*4096 + i*16 + y + 0] |= ((p & 1) >> 0) << (7-x);
            pattern_banks[bank*4096 + i*16 + y + 8] |= ((p & 2) >> 1) << (7-x);
          }
      }
    }

      // Speed indicator tiles
#if 1
    {
      static uint8_t speed_indicator_patterns[16*16] = {

        0b00000000, // 0
        0b00000000,
        0b01110111,
        0b01110111,
        0b01110111,
        0b01110111,
        0b01110111,
        0b01110111,

        0b00000000,
        0b00000000,
        0b00000000,
        0b00000000,
        0b00000000,
        0b00000000,
        0b00000000,
        0b00000000,

        0b00000000, // 1
        0b00000000,
        0b01110000,
        0b01110000,
        0b01110000,
        0b01110000,
        0b01110000,
        0b01110000,

        0b00000000,
        0b00000000,
        0b00000000,
        0b00000000,
        0b00000000,
        0b00000000,
        0b00000000,
        0b00000000,

        0b00000000, // 2
        0b00000000,
        0b00000000,
        0b00000000,
        0b00000000,
        0b00000000,
        0b00000000,
        0b00000000,

        0b00000000,
        0b00000000,
        0b01110111,
        0b01110111,
        0b01110111,
        0b01110111,
        0b01110111,
        0b01110111,

        0b00000000, // 3
        0b00000000,
        0b00000000,
        0b00000000,
        0b00000000,
        0b00000000,
        0b00000000,
        0b00000000,

        0b00000000,
        0b00000000,
        0b01110000,
        0b01110000,
        0b01110000,
        0b01110000,
        0b01110000,
        0b01110000,

        0b00000000, // 4
        0b00000000,
        0b01110111,
        0b01110111,
        0b01110111,
        0b01110111,
        0b01110111,
        0b01110111,

        0b00000000,
        0b00000000,
        0b01110111,
        0b01110111,
        0b01110111,
        0b01110111,
        0b01110111,
        0b01110111,

        0b00000000, // 5
        0b00000000,
        0b01110000,
        0b01110000,
        0b01110000,
        0b01110000,
        0b01110000,
        0b01110000,

        0b00000000,
        0b00000000,
        0b01110000,
        0b01110000,
        0b01110000,
        0b01110000,
        0b01110000,
        0b01110000,

        0b01110111, // 6
        0b01110111,
        0b01110111,
        0b01110111,
        0b01110111,
        0b01110111,
        0b00000000,
        0b00000000,

        0b00000000,
        0b00000000,
        0b00000000,
        0b00000000,
        0b00000000,
        0b00000000,
        0b00000000,
        0b00000000,

        0b01110000, // 7
        0b01110000,
        0b01110000,
        0b01110000,
        0b01110000,
        0b01110000,
        0b00000000,
        0b00000000,

        0b00000000,
        0b00000000,
        0b00000000,
        0b00000000,
        0b00000000,
        0b00000000,
        0b00000000,
        0b00000000,

        0b00000000, // 8
        0b00000000,
        0b00000000,
        0b00000000,
        0b00000000,
        0b00000000,
        0b00000000,
        0b00000000,

        0b01110111,
        0b01110111,
        0b01110111,
        0b01110111,
        0b01110111,
        0b01110111,
        0b00000000,
        0b00000000,

        0b00000000, // 9
        0b00000000,
        0b00000000,
        0b00000000,
        0b00000000,
        0b00000000,
        0b00000000,
        0b00000000,

        0b01110000,
        0b01110000,
        0b01110000,
        0b01110000,
        0b01110000,
        0b01110000,
        0b00000000,
        0b00000000,

        0b01110111, // 10
        0b01110111,
        0b01110111,
        0b01110111,
        0b01110111,
        0b01110111,
        0b00000000,
        0b00000000,

        0b01110111,
        0b01110111,
        0b01110111,
        0b01110111,
        0b01110111,
        0b01110111,
        0b00000000,
        0b00000000,

        0b01110000, // 11
        0b01110000,
        0b01110000,
        0b01110000,
        0b01110000,
        0b01110000,
        0b00000000,
        0b00000000,

        0b01110000,
        0b01110000,
        0b01110000,
        0b01110000,
        0b01110000,
        0b01110000,
        0b00000000,
        0b00000000,
      };

      const int bank=1;
      const int base=192;

      memset(&pattern_banks[bank*4096+base*16],0,64*16);

      //memcpy(&pattern_banks[bank*4096+base*16],&speed_indicator_patterns[0],sizeof(speed_indicator_patterns));

      // Put them into bank 3, too.
      memcpy(&pattern_banks[3*4096+(base-128)*16],&speed_indicator_patterns[0],sizeof(speed_indicator_patterns));
    }
#endif

    {
      // Bonus balls
      static uint8_t bonusball_patterns[] = {

        0b00000000, // Uncollected
        0b00111000,
        0b01000100,
        0b01000100,
        0b01000100,
        0b00111000,
        0b00000000,
        0b00000000,

        0b00000000,
        0b00000000,
        0b00000000,
        0b00000000,
        0b00000000,
        0b00000000,
        0b00000000,
        0b00000000,

        0b00000000, // Collected
        0b00111000,
        0b01100100,
        0b01000100,
        0b01000100,
        0b00111000,
        0b00000000,
        0b00000000,

        0b00000000,
        0b00000000,
        0b00111000,
        0b00111000,
        0b00111000,
        0b00000000,
        0b00000000,
        0b00000000,
      };

      const int bank=3;
      const int base=0xCC;

      memcpy(&pattern_banks[bank*4096+(base-128)*16],bonusball_patterns,sizeof(bonusball_patterns));
    }

      // Reticle tiles
#if 1
    {
      static uint8_t reticle_patterns[16] = {
        0b00000011, // 0
        0b00000011,
        0b00000011,
        0b00000011,
        0b00000011,
        0b00000011,
        0b11111111,
        0b11111111,

        0b00000000, // 1
        0b00000001,
        0b00000001,
        0b00000001,
        0b00000001,
        0b00000001,
        0b00000001,
        0b01111111,
      };

      const int bank=1;
      const int base=0xFE;

      memcpy(&pattern_banks[bank*4096+base*16],&reticle_patterns[0],sizeof(reticle_patterns));
    }
#endif

    {
      // Background star animation (constructed in 2bpp directly)

      const int bank=2;
      uint8_t* star_patterns=&pattern_banks[bank*4096];

      memset(star_patterns,0,4096);
      for(int frame=0;frame<32;++frame)
      {
        for(int y=0;y<32;++y)
          for(int x=0;x<32;++x)
          {
            const int c=(x/8)+(y/8)*4+(frame/2)*16;
            int frame2=frame-(frame/4)*2;
            assert(c>=0);
            assert(c<256);
            double u=-(x-16.5)/16.0;
            double v=+(y-16.5)/16.0;
            double a=fmodf(atan2(v,u)+M_PI+(double)frame2/16.0*M_PI*2.0/5.0,M_PI*2);
            assert(a>=0.0);
            assert(a<=M_PI*2);
            double b=abs(fmodf(a,M_PI*2.0/5.0)-M_PI*2.0/5.0/2.0);
            double u2=cos(b)*sqrt(u*u+v*v);
            double v2=sin(b)*sqrt(u*u+v*v);
            double w=u2+v2*2.0;
            if((w>0.7&&w<0.9)||w<0.6)
              star_patterns[c*16+(y&7)+(frame&1)*8]|=1<<(7-(x&7));
          }
      }
    }

    {
      // Background stars nametables
      memset(stars_nametables,0,sizeof(stars_nametables));

      for(int y=0;y<30;++y)
        for(int x=0;x<32;++x)
        {
          //int ofs=(0.5+0.5*cos(double(x/4+y/4)/4.0))*16;

          double d=sqrt(pow(abs((double)x-16),2)+pow(abs((double)y-15),2));

          //if(d>16)
          //  continue;

          int ofs=(x/4+(y+(x/4)*2)/4)*4;

          if(ofs&4)
            continue;

          for(int n=0;n<2;++n)
            stars_nametables[n*1024+x+y*32]=((x&3)+((y+(x/4)*2)&3)*4+n*16+16*ofs)&255;
        }

      // Attributes
      for(int y=0;y<8;++y)
        for(int x=0;x<8;++x)
        {
          int i=(y<4)?y:(7-y);
          for(int n=0;n<2;++n)
            stars_nametables[n*1024+960+x+y*8]=i|(i<<2)|(i<<4)|(i<<6);
        }

      FILE* out=fopen("stars_nametables.bin","wb");
      fwrite(stars_nametables,1,sizeof(stars_nametables),out);
      fclose(out);
    }

    // Fully opaque pattern for BG hide trick sprites
    {
      int bank=1;
      for(int i=0;i<8;++i)
      {
        // Index 1 gives opaque black, which is necessary since the
        // guard sprites can become visible if they enter the HUD region.
        pattern_banks[bank*4096+0xFF*16+0+i]=0xFF;
        pattern_banks[bank*4096+0xFF*16+8+i]=0x00;
      }
    }

    {
      int bank=3;
      for(int i=0;i<8;++i)
      {
        // Index 1 gives opaque black, which is necessary since the
        // guard sprites can become visible if they enter the HUD region.
        pattern_banks[bank*4096+0x4F*16+0+i]=0xFF;
        pattern_banks[bank*4096+0x4F*16+8+i]=0xFF;
      }
    }

    {
      FILE* out=fopen("patterns.bin","wb");
      fwrite(pattern_banks,1,sizeof(pattern_banks),out);
      fclose(out);
    }

    printf("count = %d\n",(int)count);
    printf("count_left = %d\n",(int)count_left);
    printf("count_right = %d\n",(int)count_right);

    // This is a composing table for "splatting" a new segment into a pattern.

    // Left
    for(int i=0;i<count_left;++i)
    {
      int v=pattern_to_vector[remap[i]];
      for(int j=0;j<8;++j)
      {
        int a=(v/9/9/9)%9;
        int b=(v/9/9)%9;
        int c=(v/9)%9;
        assert(a+b+c==8);
        int k = (b==0) ? 1 : 2;
        if(k==0) // Set A
        {
          // Not needed?
        }
        else if(k==1) // Set B
        {
          a=j;
          b=8-j;
          c=0;
        }
        else if(k==2) // Set C
        {
          a=MIN(a,j);
          b=j-a;
          c=8-j;
        }
        assert(a+b+c==8);
        strip_splat_table[j*256+remap[i]]=vector_to_pattern_left[a*9*9*9 + b*9*9 + c*9];
      }
    }

    // Right
    for(int i=0;i<count_right;++i)
    {
      int v=pattern_to_vector[remap[count_left+i]];
      for(int j=0;j<8;++j)
      {
        int d=(v/9/9/9)%9;
        int c=(v/9/9)%9;
        int b=(v/9)%9;
        int a=(v)%9;
        assert(a+b+c+d==8);
        int k = (b==0 && c==0) ? 1 : (c==0) ? 2 : 3;
        if(k==0) // Set A
        {
          // Not needed?
        }
        else if(k==1) // Set B
        {
          a=j+1;
          b=8-j-1;
          c=0;
          d=0;
        }
        else if(k==2) // Set C
        {
          a=MIN(a,j+1);
          b=j+1-a;
          c=8-j-1;
          d=0;
        }
        else if(k==3) // Set D
        {
          a=MIN(a,j+1);
          b=MAX(0,MIN(a+b,j+1)-a);
          c=j+1-(a+b);
          d=8-j-1;
        }
        assert(a+b+c+d==8);
        strip_splat_table[(7-j)*256+remap[count_left+i]]=vector_to_pattern_right[d*9*9*9 + c*9*9 + b*9 + a];
      }
    }

    {
      FILE* out=fopen("strip_splat_table.bin","wb");
      fwrite(strip_splat_table,1,sizeof(strip_splat_table),out);
      fclose(out);
    }

    // Sprite sets

    static uint8_t reticle_sprites[16];
    // Top-left
    reticle_sprites[0x0] = 32+(240-32-16)/2-8; // Y
    reticle_sprites[0x1] = 0xFE; // Tile
    reticle_sprites[0x2] = 0b00000000; // Attributes
    reticle_sprites[0x3] = 126; // X
    // Top-right
    reticle_sprites[0x4] = 32+(240-32-16)/2-8; // Y
    reticle_sprites[0x5] = 0xFE; // Tile
    reticle_sprites[0x6] = 0b01000000; // Attributes
    reticle_sprites[0x7] = 126+8; // X
    // Bottom-left
    reticle_sprites[0x8] = 32+(240-32-16)/2; // Y
    reticle_sprites[0x9] = 0xFE; // Tile
    reticle_sprites[0xA] = 0b10000000; // Attributes
    reticle_sprites[0xB] = 126; // X
    // Bottom-right
    reticle_sprites[0xC] = 32+(240-32-16)/2; // Y
    reticle_sprites[0xD] = 0xFE; // Tile
    reticle_sprites[0xE] = 0b11000000; // Attributes
    reticle_sprites[0xF] = 126+8; // X


    {
      memset(sprites_timeover,0xFF,sizeof(sprites_timeover));
      int j=0;
      {
        const char* s="TIME";
        for(int i=0;i<strlen(s);++i)
        {
          int p=(s[i]-32)*2+64;
          sprites_timeover[j*4+0] = 128; // Y
          sprites_timeover[j*4+1] = p; // Tile
          sprites_timeover[j*4+2] = 0x00; // Attributes
          sprites_timeover[j*4+3] = 128+i*8; // X
          ++j;
          sprites_timeover[j*4+0] = 128+8; // Y
          sprites_timeover[j*4+1] = p+1; // Tile
          sprites_timeover[j*4+2] = 0x00; // Attributes
          sprites_timeover[j*4+3] = 128+i*8; // X
          ++j;
        }
      }
      {
        const char* s="OVER!";
        for(int i=0;i<strlen(s);++i)
        {
          int p=(s[i]-32)*2+64;
          sprites_timeover[j*4+0] = 128+16; // Y
          sprites_timeover[j*4+1] = p; // Tile
          sprites_timeover[j*4+2] = 0x00; // Attributes
          sprites_timeover[j*4+3] = 128+i*8; // X
          ++j;
          sprites_timeover[j*4+0] = 128+16+8; // Y
          sprites_timeover[j*4+1] = p+1; // Tile
          sprites_timeover[j*4+2] = 0x00; // Attributes
          sprites_timeover[j*4+3] = 128+i*8; // X
          ++j;
        }
      }

      printf("sprites_timeover sprites: %d\n",j);

      int min_x=255,max_x=0,min_y=255,max_y=0;
      for(int i=0;i<j;++i)
      {
        min_x=MIN(min_x,sprites_timeover[i*4+3]);
        max_x=MAX(max_x,sprites_timeover[i*4+3]+8);
        min_y=MIN(min_y,sprites_timeover[i*4+0]);
        max_y=MAX(max_y,sprites_timeover[i*4+0]+16);
      }
      int offset_x=-min_x+134-(max_x-min_x)/2;
      int offset_y=-min_y+150;
      for(int i=0;i<j;++i)
      {
        sprites_timeover[i*4+3]+=offset_x;
        sprites_timeover[i*4+0]+=offset_y;
      }

      memcpy(&sprites_timeover[256-16],reticle_sprites,16);

      FILE* out=fopen("sprites_timeover.bin","wb");
      //fwrite(sprites_timeover,1,sizeof(sprites_timeover),out);
      fwrite(sprites_timeover,1,j*4,out);
      fclose(out);
    }

    {
      memset(sprites_crashed,0xFF,sizeof(sprites_crashed));
      int j=0;
      {
        const char* s="CRASH!";
        for(int i=0;i<strlen(s);++i)
        {
          int p=(s[i]-32)*2+64;
          sprites_crashed[j*4+0] = 128; // Y
          sprites_crashed[j*4+1] = p; // Tile
          sprites_crashed[j*4+2] = 0x00; // Attributes
          sprites_crashed[j*4+3] = 128+i*8; // X
          ++j;
          sprites_crashed[j*4+0] = 128+8; // Y
          sprites_crashed[j*4+1] = p+1; // Tile
          sprites_crashed[j*4+2] = 0x00; // Attributes
          sprites_crashed[j*4+3] = 128+i*8; // X
          ++j;
        }
      }

      printf("sprites_crashed sprites: %d\n",j);

      int min_x=255,max_x=0,min_y=255,max_y=0;
      for(int i=0;i<j;++i)
      {
        min_x=MIN(min_x,sprites_crashed[i*4+3]);
        max_x=MAX(max_x,sprites_crashed[i*4+3]+8);
        min_y=MIN(min_y,sprites_crashed[i*4+0]);
        max_y=MAX(max_y,sprites_crashed[i*4+0]+16);
      }
      int offset_x=-min_x+134-(max_x-min_x)/2;
      int offset_y=-min_y+150;
      for(int i=0;i<j;++i)
      {
        sprites_crashed[i*4+3]+=offset_x;
        sprites_crashed[i*4+0]+=offset_y;
      }

      memcpy(&sprites_crashed[256-16],reticle_sprites,16);

      FILE* out=fopen("sprites_crashed.bin","wb");
      //fwrite(sprites_crashed,1,sizeof(sprites_crashed),out);
      fwrite(sprites_crashed,1,j*4,out);
      fclose(out);
    }

    {
      memset(sprites_won,0xFF,sizeof(sprites_won));
      int j=0;
      {
        //const char* s="YOU WIN!";
        const char* s="FINISH!";
        for(int i=0;i<strlen(s);++i)
        {
          if(s[i]!=' ')
          {
            int p=(s[i]-32)*2+64;
            sprites_won[j*4+0] = 128; // Y
            sprites_won[j*4+1] = p; // Tile
            sprites_won[j*4+2] = 0x00; // Attributes
            sprites_won[j*4+3] = 128+i*8; // X
            ++j;
            sprites_won[j*4+0] = 128+8; // Y
            sprites_won[j*4+1] = p + 1; // Tile
            sprites_won[j*4+2] = 0x00; // Attributes
            sprites_won[j*4+3] = 128+i*8; // X
            ++j;
          }
        }
      }

      printf("sprites_won sprites: %d\n",j);

      int min_x=255,max_x=0,min_y=255,max_y=0;
      for(int i=0;i<j;++i)
      {
        min_x=MIN(min_x,sprites_won[i*4+3]);
        max_x=MAX(max_x,sprites_won[i*4+3]+8);
        min_y=MIN(min_y,sprites_won[i*4+0]);
        max_y=MAX(max_y,sprites_won[i*4+0]+16);
      }
      int offset_x=-min_x+134-(max_x-min_x)/2;
      int offset_y=-min_y+150;
      for(int i=0;i<j;++i)
      {
        sprites_won[i*4+3]+=offset_x;
        sprites_won[i*4+0]+=offset_y;
      }

      memcpy(&sprites_won[256-16],reticle_sprites,16);

      FILE* out=fopen("sprites_won.bin","wb");
      //fwrite(sprites_won,1,sizeof(sprites_won),out);
      fwrite(sprites_won,1,j*4,out);
      fclose(out);
    }

    {
      memset(sprites_paused,0xFF,sizeof(sprites_paused));
      int j=0;
      {
        const char* s="PAUSED";
        for(int i=0;i<strlen(s);++i)
        {
          if(s[i]!=' ')
          {
            int p=(s[i]-32)*2+64;
            sprites_paused[j*4+0] = 128; // Y
            sprites_paused[j*4+1] = p; // Tile
            sprites_paused[j*4+2] = 0x00; // Attributes
            sprites_paused[j*4+3] = 128+i*8; // X
            ++j;
            sprites_paused[j*4+0] = 128+8; // Y
            sprites_paused[j*4+1] = p + 1; // Tile
            sprites_paused[j*4+2] = 0x00; // Attributes
            sprites_paused[j*4+3] = 128+i*8; // X
            ++j;
          }
        }
      }

      printf("sprites_paused sprites: %d\n",j);

      int min_x=255,max_x=0,min_y=255,max_y=0;
      for(int i=0;i<j;++i)
      {
        min_x=MIN(min_x,sprites_paused[i*4+3]);
        max_x=MAX(max_x,sprites_paused[i*4+3]+8);
        min_y=MIN(min_y,sprites_paused[i*4+0]);
        max_y=MAX(max_y,sprites_paused[i*4+0]+16);
      }
      int offset_x=-min_x+134-(max_x-min_x)/2;
      int offset_y=-min_y+150;
      for(int i=0;i<j;++i)
      {
        sprites_paused[i*4+3]+=offset_x;
        sprites_paused[i*4+0]+=offset_y;
      }

      memcpy(&sprites_paused[256-16],reticle_sprites,16);

      FILE* out=fopen("sprites_paused.bin","wb");
      //fwrite(sprites_paused,1,sizeof(sprites_paused),out);
      fwrite(sprites_paused,1,j*4,out);
      fclose(out);
    }

    // HUD nametable initial contents
    {
      {
        const char* s="LAP";
        for(int i=0;i<strlen(s);++i)
          hud_nametable[2+i+2*32]=s[i]-32;
      }
      {
        //const char* s="1/3";
        const char* s="0/3";
        for(int i=0;i<strlen(s);++i)
          for(int y=0;y<2;++y)
            hud_nametable[2+1+3+i+(2+y)*32]=(s[i]-32)*2+y+64;
      }
      {
        const char* s="TIME";
        for(int i=0;i<strlen(s);++i)
          hud_nametable[16-5+i+2*32]=s[i]-32;
      }
/*
      {
        //const char* s="38";
        const char* s="00";
        for(int i=0;i<strlen(s);++i)
          for(int y=0;y<2;++y)
            hud_nametable[16+1+i+(2+y)*32]=(s[i]-32)*2+y+64;
      }
*/
      {
        const char* s="SPEED";
        for(int i=0;i<strlen(s);++i)
          hud_nametable[32-8+i+1*32]=s[i]-32;
      }

      // Bonus balls
      for(int i=0;i<8;++i)
          hud_nametable[5+i+0*32]=0xCC;

      FILE* out=fopen("hud_nametable.bin","wb");
      fwrite(hud_nametable,1,sizeof(hud_nametable),out);
      fclose(out);
    }
  }


  clipAndPlotBox(pixels, 0, 0, FRAME_WIDTH, FRAME_HEIGHT, RGB888toRGB565(50, 50, 50));

  clearVirtualFramebuffer();

  static const int N=4;

  static int strip_bounds[N*4];
  static uint8_t strip_indices[256];

  // Initialise the outermost rectangle
  for(int y=0;y<VIRTUAL_FRAME_HEIGHT;++y)
    strip_indices[y]=0;

  strip_bounds[0*4+0]=0;
  strip_bounds[0*4+1]=VIRTUAL_FRAME_WIDTH-1;
  strip_bounds[0*4+2]=0;
  strip_bounds[0*4+3]=VIRTUAL_FRAME_HEIGHT-1;

  int flip_point=0;

  static const int pickup_frames=24;
  static const int pickup_width=32, pickup_height=32;
  static uint8_t pickup_images[pickup_width*pickup_height*pickup_frames];
  static double scale_table[256];

  static uint8_t log_table[256];
  static uint8_t exp_table[256];
  static uint8_t log_z_table[256];

  const double min_log_z=0.0;
  // Max. divisor is 16
  const int max_divisor = 16;
  const double max_log_z=std::log((double(3*128)-1.0)/191.0*0.5*double(max_divisor-1)+1.0)/std::log(127.0);

  const double max_log_x=1.0; // log(127)/log(127)

  const double log_range = max_log_z + max_log_x;
  const double log_scale = 255.0 / log_range;

  const int log_bias=max_log_z*log_scale;

  // Determine strip bounds
  {
    static bool_t init_tables=TRUE;

    static int8_t rect_array[64*4*8];
    static int8_t level_metadata[32*8];

    if(init_tables)
    {
      init_tables=FALSE;

      memset(level_metadata,0,sizeof(level_metadata));

      memset(log_table,0,sizeof(log_table));
      memset(exp_table,0,sizeof(exp_table));
      memset(log_z_table,0,sizeof(log_z_table));

      // Set things up so that maximum Z scales by 1/16 and 
      // minimum Z scales by 1.

      printf("max_log_z = %g\n",max_log_z);
      printf("max_log_x = %g\n",max_log_x);
      printf("log_range = %g\n",log_range);
      printf("log_scale = %g\n",log_scale);

      for(int i=1;i<256;++i)
      {
        scale_table[i]=(double(i)-1.0)/254.0*7.0+1.0;
        int x=std::log(scale_table[i])/std::log(127.0)*log_scale;
        assert(x>=0);
        //assert(x<70);
        log_z_table[i]=x;
        printf("log_z_table[%d] = %d\n",i,(int)x);
      }

      for(int i=0;i<256;++i)
      {
        int j=i<=128 ? i : 256-i;
        int x=(i==0) ? 0 : MIN(255,std::log(std::abs(double(j)))/std::log(127.0)*log_scale+log_bias+0.5);
        log_table[i]=x;
        printf("log_table[%d] = %d\n",i,(int)x);
      }

      for(int i=0;i<256;++i)
      {
        int x=(i==0) ? 0 : MIN(127-4,std::exp(double(i-log_bias)/log_scale*std::log(127.0))*8.0+0.5);
        assert(x>=0);
        assert(x<128);
        exp_table[i]=x;
        printf("exp_table[%d] = %d\n",i,(int)x);
      }

      // Pickup images
      {
        static const int dither[8][8] = {
          {  0, 32,  8, 40,  2, 34, 10, 42 },
          { 48, 16, 56, 24, 50, 18, 58, 26 },
          { 12, 44,  4, 36, 14, 46,  6, 38 },
          { 60, 28, 52, 20, 62, 30, 54, 22 },
          {  3, 35, 11, 43,  1, 33,  9, 41 },
          { 51, 19, 59, 27, 49, 17, 57, 25 },
          { 15, 47,  7, 39, 13, 45,  5, 37 },
          { 63, 31, 55, 23, 61, 29, 53, 21 },
        };

        memset(pickup_images,0,sizeof(pickup_images));
        for(int i=0;i<pickup_frames;++i)
        {
          //const int frame=(log_z*(pickup_frames-1))/69;
          const int log_z=(i*69)/(pickup_frames-1);
          const double scale=std::exp(double(log_z-log_bias)/log_scale*std::log(127.0))*8.0+0.5;

          //float radius=1.0/scale_table[1+(i*254)/(pickup_frames-1)];
          double radius=1.0/scale;
          for(int y=0;y<pickup_height;++y)
            for(int x=0;x<pickup_width;++x)
            {
              double u=(double(x)+0.5-double(pickup_width)/2.0)/(double(pickup_width)/2.0)/radius;
              double v=(double(y)+0.5-double(pickup_height)/2.0)/(double(pickup_height)/2.0)/radius;
              double d=sqrt(u*u+v*v);
              if(d<1.0)
              {
                u=-abs(u);
                v=-abs(v);
                double w=sqrt(1.0-u*u-v*v);
                double ray[3]={0.0,0.0,-1.0};
                double t=ray[0]*u+ray[1]*v+ray[2]*w;
                ray[0]-=u*t*2.0;
                ray[1]-=v*t*2.0;
                ray[2]-=w*t*2.0;
                double l=pow(MAX(0.0,MIN(0.999,(-ray[0]-ray[1]+ray[2]*0.0-0.9)*2.0)),2.0);
                pickup_images[x+y*pickup_width + pickup_width*pickup_height*i]=1+int(l*2.0+double(dither[x&7][y&7])/64.0);
              }
            }
        }

        {
          // Determine unique pickup patterns

          static uint8_t metatiles[pickup_frames*4*4];
          memset(metatiles,0,sizeof(metatiles));

          int num_patterns=0;
          static uint8_t unique_patterns[256*64];
          memset(unique_patterns,0,sizeof(unique_patterns));

          static const int remap[16] = {
               0,  1,  5,  4,
               2,  3,  7,  6,
              10, 11, 15, 14,
               8,  9, 13, 12,
            };

          for(int i=0;i<pickup_frames;++i)
            for(int y0=0;y0<pickup_height/2;y0+=8) // Top-left only
              for(int x0=0;x0<pickup_width/2;x0+=8)
              {
                uint8_t pattern[64];
                for(int y=0;y<8;++y)
                  for(int x=0;x<8;++x)
                  {
                    pattern[x+y*8]=pickup_images[(x0+x)+(y0+y)*pickup_width + pickup_width*pickup_height*i];
                  }
                int j=0;
                for(;j<num_patterns;++j)
                  if(memcmp(pattern,unique_patterns+j*64,64)==0)
                    break;
                if(j==num_patterns)
                {
                  memcpy(unique_patterns+j*64,pattern,64);
                  ++num_patterns;
                }
                assert(j<256);
                metatiles[i*16+remap[x0/8+(y0/8)*4]]=0xC0+j;
                metatiles[i*16+remap[(3-x0/8)+(y0/8)*4]]=0xC0+j;
                metatiles[i*16+remap[x0/8+(3-(y0/8))*4]]=0xC0+j;
                metatiles[i*16+remap[(3-x0/8)+(3-(y0/8))*4]]=0xC0+j;
              }
            printf("pickup num_patterns=%d\n",num_patterns);
            {
              FILE* out=fopen("pickup_metatiles.bin","wb");
              fwrite(metatiles,1,sizeof(metatiles),out);
              fclose(out);
            }

            assert(num_patterns<64);

            // Convert to 2bpp
            static uint8_t pickup_images_2bpp[256*16];
            memset(pickup_images_2bpp,0,sizeof(pickup_images_2bpp));
            for(int i=0;i<num_patterns;++i)
            {
              for(int y=0;y<8;++y)
                for(int x=0;x<8;++x)
                {
                  uint8_t p=unique_patterns[i*64+y*8+x];
                  pickup_images_2bpp[i*16+0+y]|=((p&1)<<(7-x));
                  pickup_images_2bpp[i*16+8+y]|=(((p&2)>>1)<<(7-x));
                }
            }
            {
              FILE* out=fopen("pickup.bin","wb");
              fwrite(pickup_images_2bpp,1,num_patterns*16,out);
              fclose(out);
            }
        }

/*
        // Convert to 2bpp
        static uint8_t pickup_images_2bpp[32*16];
        memset(pickup_images_2bpp,0,sizeof(pickup_images_2bpp));
        for(int i=0;i<pickup_frames;i+=3)
        {
          for(int k=0;k<4;++k)
          {
            const int j=(i/3)*4+k;
            for(int y=0;y<8;++y)
              for(int x=0;x<8;++x)
              {
                if      (pickup_images[(x+(k&1)*8)+(y+(k/2)*8+pickup_height*(i+2))*pickup_width])
                {
                  pickup_images_2bpp[j*16+0+y]|=(1<<(7-x));
                  pickup_images_2bpp[j*16+8+y]|=(0<<(7-x));
                }
                else if (pickup_images[(x+(k&1)*8)+(y+(k/2)*8+pickup_height*(i+1))*pickup_width])
                {
                  pickup_images_2bpp[j*16+0+y]|=(0<<(7-x));
                  pickup_images_2bpp[j*16+8+y]|=(1<<(7-x));
                }
                else if (pickup_images[(x+(k&1)*8)+(y+(k/2)*8+pickup_height*(i+0))*pickup_width])
                {
                  pickup_images_2bpp[j*16+0+y]|=(1<<(7-x));
                  pickup_images_2bpp[j*16+8+y]|=(1<<(7-x));
                }
              }
          }
        }
        FILE* out=fopen("pickup.bin","wb");
        fwrite(pickup_images_2bpp,1,sizeof(pickup_images_2bpp),out);
        fclose(out);
*/
      }

      srand(0xEDD);

      // Rect array and pickup locations
      {
        int8_t* rects=&rect_array[64*4*0];
        // Level one
        double current_rect[4] = { 0, 0, 40, 40 }; // X, Y, W, H
        for(int i=0;i<64;++i)
        {
          tweenRectangle(i,current_rect,   4,11,   50,0,32,32);
          tweenRectangle(i,current_rect,   15+7*0,15+7*1,   -50,0,32,32);
          tweenRectangle(i,current_rect,   15+7*2,15+7*3,   -50,50,32,32);
          tweenRectangle(i,current_rect,   15+7*3,15+7*5,   50,-50,32,32);

          tweenRectangle(i,current_rect,   55,63,   0,0,40,40);

          rects[i*4+0]=current_rect[0]-current_rect[2]/2.0;
          rects[i*4+1]=current_rect[0]+current_rect[2]/2.0;
          rects[i*4+2]=current_rect[1]-current_rect[3]/2.0;
          rects[i*4+3]=current_rect[1]+current_rect[3]/2.0;

          if((i&7)==0)
          {
            level_metadata[32*0+i/8+0]=i; // Z
            level_metadata[32*0+i/8+8]=current_rect[0]+current_rect[2]/2.0*(frand()*2.0-1.0)*0.6; // X
            level_metadata[32*0+i/8+16]=current_rect[1]+current_rect[3]/2.0*(frand()*2.0-1.0)*0.6; // Y
          }
        }
        // Seconds per lap
        level_metadata[32*0+26]=1; // Tens
        level_metadata[32*0+27]=5; // Units
        // Layers palette
        level_metadata[32*0+28]=0x19;
        level_metadata[32*0+29]=0x28;
        level_metadata[32*0+30]=0x17;
        level_metadata[32*0+31]=0x2A;
      }
      {
        int8_t* rects=&rect_array[64*4*1];
        // Level two
        double current_rect[4] = { 0, 0, 40, 40 }; // X, Y, W, H
        for(int i=0;i<64;++i)
        {
          tweenRectangle(i,current_rect,   4,11,   0,0,32,8);

          tweenRectangle(i,current_rect,   20,21,   0,0,40,40);
          tweenRectangle(i,current_rect,   25,26,   0,20,32,8);

          tweenRectangle(i,current_rect,   33,34,   0,0,40,40);
          tweenRectangle(i,current_rect,   37,38,   0,-20,32,8);

          tweenRectangle(i,current_rect,   43,44,   0,0,40,40);
          tweenRectangle(i,current_rect,   48,49,   0,30,32,8);

          tweenRectangle(i,current_rect,   55,63,   0,0,40,40);

          rects[i*4+0]=current_rect[0]-current_rect[2]/2.0;
          rects[i*4+1]=current_rect[0]+current_rect[2]/2.0;
          rects[i*4+2]=current_rect[1]-current_rect[3]/2.0;
          rects[i*4+3]=current_rect[1]+current_rect[3]/2.0;

          if((i&7)==0)
          {
            level_metadata[32*1+i/8+0]=i; // Z
            level_metadata[32*1+i/8+8]=current_rect[0]+current_rect[2]/2.0*(frand()*2.0-1.0)*0.6; // X
            level_metadata[32*1+i/8+16]=current_rect[1]+current_rect[3]/2.0*(frand()*2.0-1.0)*0.6; // Y
          }
        }
        // Seconds per lap
        level_metadata[32*1+26]=1; // Tens
        level_metadata[32*1+27]=5; // Units
        // Layers palette
        level_metadata[32*1+28]=0x15;
        level_metadata[32*1+29]=0x28;
        level_metadata[32*1+30]=0x1C;
        level_metadata[32*1+31]=0x22;
      }
      {
        int8_t* rects=&rect_array[64*4*2];
        // Level three
        double current_rect[4] = { 0, 0, 50, 50 }; // X, Y, W, H
        for(int i=0;i<64;++i)
        {
          tweenRectangle(i,current_rect,   5,6,   20,0,8,50);
          tweenRectangle(i,current_rect,   6,7,   0,0,50,50);

          tweenRectangle(i,current_rect,   15,16,   -20,0,8,50);
          tweenRectangle(i,current_rect,   16,17,   0,0,50,50);

          tweenRectangle(i,current_rect,   22,23,   20,0,8,50);
          tweenRectangle(i,current_rect,   23,24,   0,0,50,50);

          tweenRectangle(i,current_rect,   27,28,   -20,0,8,50);
          tweenRectangle(i,current_rect,   28,29,   0,0,50,50);

          tweenRectangle(i,current_rect,   33,34,   -20,0,8,50);
          tweenRectangle(i,current_rect,   34,35,   0,0,50,50);

          tweenRectangle(i,current_rect,   36,37,   20,0,8,50);
          tweenRectangle(i,current_rect,   37,38,   0,0,50,50);

          tweenRectangle(i,current_rect,   39,40,   -20,0,8,50);
          tweenRectangle(i,current_rect,   40,41,   0,0,50,50);

          tweenRectangle(i,current_rect,   42,43,   10,0,8,50);
          tweenRectangle(i,current_rect,   43,44,   0,0,50,50);

          tweenRectangle(i,current_rect,   44,45,   -10,0,8,50);
          tweenRectangle(i,current_rect,   45,46,   0,0,50,50);

          tweenRectangle(i,current_rect,   46,47,   10,0,8,50);
          tweenRectangle(i,current_rect,   47,48,   0,0,50,50);

          tweenRectangle(i,current_rect,   48,49,   -10,0,8,50);
          tweenRectangle(i,current_rect,   49,50,   0,0,50,50);

          tweenRectangle(i,current_rect,   50,53,   0,0,20,20);

          tweenRectangle(i,current_rect,   53,56,   0,0,8,50);
          tweenRectangle(i,current_rect,   56,60,   10,0,8,50);


          tweenRectangle(i,current_rect,   55,63,   0,0,50,50);

          rects[i*4+0]=current_rect[0]-current_rect[2]/2.0;
          rects[i*4+1]=current_rect[0]+current_rect[2]/2.0;
          rects[i*4+2]=current_rect[1]-current_rect[3]/2.0;
          rects[i*4+3]=current_rect[1]+current_rect[3]/2.0;

          if((i&7)==0)
          {
            level_metadata[32*2+i/8+0]=i; // Z
            level_metadata[32*2+i/8+8]=current_rect[0]+current_rect[2]/2.0*(frand()*2.0-1.0)*0.6; // X
            level_metadata[32*2+i/8+16]=current_rect[1]+current_rect[3]/2.0*(frand()*2.0-1.0)*0.6; // Y
            if(i/8==3)
            {
              // Fix bad RNG
              level_metadata[32*2+i/8+8]=current_rect[0]; // X
              level_metadata[32*2+i/8+16]=current_rect[1]; // Y
            }
          }
        }
        // Seconds per lap
        level_metadata[32*2+26]=2; // Tens
        level_metadata[32*2+27]=0; // Units
        // Layers palette
        level_metadata[32*2+28]=0x2C;
        level_metadata[32*2+29]=0x1C;
        level_metadata[32*2+30]=0x31;
        level_metadata[32*2+31]=0x11;
      }
      {
        int8_t* rects=&rect_array[64*4*3];
        // Level four
        double current_rect[4] = { 0, 0, 50, 50 }; // X, Y, W, H
        for(int i=0;i<64;++i)
        {
          for(int j=5;j<55;++j)
            tweenRectangle(i,current_rect,   j,j+1,   cos(double(i)/2.0)*30, sin(double(i)/2.0)*(25+double(j)/2.0),32,32);

          tweenRectangle(i,current_rect,   55,63,   0,0,50,50);

          rects[i*4+0]=current_rect[0]-current_rect[2]/2.0;
          rects[i*4+1]=current_rect[0]+current_rect[2]/2.0;
          rects[i*4+2]=current_rect[1]-current_rect[3]/2.0;
          rects[i*4+3]=current_rect[1]+current_rect[3]/2.0;

          if((i&7)==0)
          {
            level_metadata[32*3+i/8+0]=i; // Z
            level_metadata[32*3+i/8+8]=current_rect[0]+current_rect[2]/2.0*(frand()*2.0-1.0)*0.6; // X
            level_metadata[32*3+i/8+16]=current_rect[1]+current_rect[3]/2.0*(frand()*2.0-1.0)*0.6; // Y
          }
        }
        // Seconds per lap
        level_metadata[32*3+26]=2; // Tens
        level_metadata[32*3+27]=0; // Units
        // Layers palette
        level_metadata[32*3+28]=0x00;
        level_metadata[32*3+29]=0x10;
        level_metadata[32*3+30]=0x00;
        level_metadata[32*3+31]=0x10;
      }
/*
      {
        int8_t* rects=&rect_array[64*4*4];
        // Level five
        double current_rect[4] = { 0, 0, 50, 50 }; // X, Y, W, H
        for(int i=0;i<64;++i)
        {
          for(int j=0;j<64;++j)
          {
            double th=double(i)/2.0+cos(double(i)/6.0)*5.0;
            tweenRectangle(i,current_rect,   j,j+1,   cos(th)*40, sin(th)*40,32,32);
          }

          rects[i*4+0]=current_rect[0]-current_rect[2]/2.0;
          rects[i*4+1]=current_rect[0]+current_rect[2]/2.0;
          rects[i*4+2]=current_rect[1]-current_rect[3]/2.0;
          rects[i*4+3]=current_rect[1]+current_rect[3]/2.0;

          if((i&7)==0)
          {
            level_metadata[32*4+i/8+0]=i; // Z
            level_metadata[32*4+i/8+8]=current_rect[0]+current_rect[2]/2.0*(frand()*2.0-1.0)*0.6; // X
            level_metadata[32*4+i/8+16]=current_rect[1]+current_rect[3]/2.0*(frand()*2.0-1.0)*0.6; // Y
          }
        }
        // Seconds per lap
        level_metadata[32*4+26]=1; // Tens
        level_metadata[32*4+27]=5; // Units
        // Layers palette
        level_metadata[32*4+28]=0x2C;
        level_metadata[32*4+29]=0x1C;
        level_metadata[32*4+30]=0x2B;
        level_metadata[32*4+31]=0x1B;
      }
*/

      {
        int8_t* rects=&rect_array[64*4*4];
        // Level five
        double current_rect[4] = { 0, 0, 64, 64 }; // X, Y, W, H
        for(int i=0;i<64;++i)
        {
          for(int j=0;j<64;++j)
          {
            double th=double(i)/2.0+cos(double(i)/6.0)*4.0;
            tweenRectangle(i,current_rect,   j,j+1,   cos(th*0.4)*40, 0,64,64);
          }

          rects[i*4+0]=current_rect[0]-current_rect[2]/2.0;
          rects[i*4+1]=current_rect[0]+current_rect[2]/2.0;
          rects[i*4+2]=current_rect[1]-current_rect[3]/2.0;
          rects[i*4+3]=current_rect[1]+current_rect[3]/2.0;

          if((i&7)==0)
          {
            level_metadata[32*4+i/8+0]=i; // Z
            level_metadata[32*4+i/8+8]=current_rect[0]+current_rect[2]/2.0*(frand()*2.0-1.0)*0.2; // X
            level_metadata[32*4+i/8+16]=current_rect[1]+current_rect[3]/2.0*(frand()*2.0-1.0)*0.2; // Y
          }
        }
        // Seconds per lap
        level_metadata[32*4+26]=2; // Tens
        level_metadata[32*4+27]=0; // Units
        // Layers palette
        level_metadata[32*4+28]=0x2C;
        level_metadata[32*4+29]=0x1C;
        level_metadata[32*4+30]=0x2B;
        level_metadata[32*4+31]=0x1B;
      }

      {
        int8_t* rects=&rect_array[64*4*5];
        // Level six
        double current_rect[4] = { 0, 0, 64, 64 }; // X, Y, W, H
        for(int i=0;i<64;++i)
        {
          const double t=0.7;

          tweenRectangle(i,current_rect,   8,9,   32.0*t,0,64.0*(1.0-t),64);
          tweenRectangle(i,current_rect,   9,10,   0,0,64,64);

          tweenRectangle(i,current_rect,   16,17,   -32.0*t,0,64.0*(1.0-t),64);
          tweenRectangle(i,current_rect,   17,18,   0,0,64,64);

          tweenRectangle(i,current_rect,   24,25,   32.0*t,0,64.0*(1.0-t),64);
          tweenRectangle(i,current_rect,   25,26,   0,0,64,64);

          tweenRectangle(i,current_rect,   32,33,   -32.0*t,0,64.0*(1.0-t),64);
          tweenRectangle(i,current_rect,   33,34,   0,0,64,64);

          tweenRectangle(i,current_rect,   40,41,   0,-32.0*t,64,64.0*(1.0-t));
          tweenRectangle(i,current_rect,   41,42,   0,0,64,64);

          tweenRectangle(i,current_rect,   48,49,   0,32.0*t,64,64.0*(1.0-t));
          tweenRectangle(i,current_rect,   49,50,   0,0,64,64);

          rects[i*4+0]=current_rect[0]-current_rect[2]/2.0;
          rects[i*4+1]=current_rect[0]+current_rect[2]/2.0;
          rects[i*4+2]=current_rect[1]-current_rect[3]/2.0;
          rects[i*4+3]=current_rect[1]+current_rect[3]/2.0;

          if(((i-3)&7)==0)
          {
            level_metadata[32*5+i/8+0]=i; // Z
            level_metadata[32*5+i/8+8]=current_rect[0]+current_rect[2]/2.0*(frand()*2.0-1.0)*0.2; // X
            level_metadata[32*5+i/8+16]=current_rect[1]+current_rect[3]/2.0*(frand()*2.0-1.0)*0.2; // Y
          }
        }
        // Seconds per lap
        level_metadata[32*5+26]=2; // Tens
        level_metadata[32*5+27]=0; // Units
        // Layers palette
        level_metadata[32*5+28]=0x2C;
        level_metadata[32*5+29]=0x1C;
        level_metadata[32*5+30]=0x28;
        level_metadata[32*5+31]=0x18;
      }

      static const int level_order[] = { 4, 5, 0, 2, 1, 3 };

      {
        // Character conversion table for ACME assembler
        static uint8_t acme_char_table[256];
        for(int i=0;i<256;++i)
          acme_char_table[i]=MAX(0,i-32);
        FILE* out=fopen("acme_char_table.bin","wb");
        fwrite(acme_char_table,1,sizeof(acme_char_table),out);
        fclose(out);
      }

      {
        FILE* out=fopen("log_table.bin","wb");
        fwrite(log_table,1,sizeof(log_table),out);
        fclose(out);
      }

      {
        FILE* out=fopen("exp_table.bin","wb");
        fwrite(exp_table,1,sizeof(exp_table),out);
        fclose(out);
      }

      {
        FILE* out=fopen("log_z_table.bin","wb");
        fwrite(log_z_table,1,sizeof(log_z_table),out);
        fclose(out);
      }

      {
        FILE* out=fopen("rect_array.bin","wb");
        //fwrite(rect_array,1,sizeof(rect_array),out);
        for(int i=0;i<sizeof(level_order)/sizeof(level_order[0]);++i)
          fwrite(&rect_array[256*level_order[i]],1,256,out);
        fclose(out);
      }

      {
        FILE* out=fopen("level_metadata.bin","wb");
        //fwrite(level_metadata,1,sizeof(level_metadata),out);
        for(int i=0;i<sizeof(level_order)/sizeof(level_order[0]);++i)
          fwrite(&level_metadata[32*level_order[i]],1,32,out);
        fclose(out);
      }

      // 6502 emulation init
      {
        {
          FILE* in=fopen("test_3d.bin","rb");
          fread(&ROM_6502[0],1,4096,in);
          fclose(in);
        }
        memcpy(&ROM_6502[0xC000-0x8000],&log_table[0],sizeof(log_table));
        memcpy(&ROM_6502[0xC100-0x8000],&exp_table[0],sizeof(exp_table));
        memcpy(&ROM_6502[0xC300-0x8000],&log_z_table[0],sizeof(log_z_table));
        memcpy(&ROM_6502[0xF000-0x8000],&strip_splat_table[0],sizeof(strip_splat_table));

        memset(RAM_6502,0,sizeof(RAM_6502));

        {
          FILE* in=fopen("strips.bin","rb");
          fread(&RAM_6502[0x0300],1,512,in);
          fclose(in);
        }
      }
    }

    // Rect array
    for(int i=0;i<4;++i)
    {
      int j=(i+g_rect_offset)%64;
      ROM_6502[0xC200-0x8000+i*4]=rect_array[j*4+0];
      ROM_6502[0xC201-0x8000+i*4]=rect_array[j*4+1];
      ROM_6502[0xC202-0x8000+i*4]=rect_array[j*4+2];
      ROM_6502[0xC203-0x8000+i*4]=rect_array[j*4+3];
    }

#if 1
    {
      // Run 6502 version of the below code
      RAM_6502[0x00]=0;
      RAM_6502[0x01]=g_cam_x;
      RAM_6502[0x02]=g_cam_y;
      RAM_6502[0x03]=0;
      RAM_6502[0x15]=84-(g_cam_sub_z/256); // Z
      RAM_6502[0x16]=32; // RASTER_UPPER
      RAM_6502[0x17]=223; // RASTER_LOWER
      do6502();

      //printf("-------\n");

      for(int i=1;i<N;++i)
      {
        strip_bounds[i*4+0]=RAM_6502[0x05+i*4+0];
        strip_bounds[i*4+1]=RAM_6502[0x05+i*4+1];
        strip_bounds[i*4+2]=RAM_6502[0x05+i*4+2];
        strip_bounds[i*4+3]=RAM_6502[0x05+i*4+3];

/*
        printf("strip_bounds[%d]=%d\n",i*4+0,strip_bounds[i*4+0]);
        printf("strip_bounds[%d]=%d\n",i*4+1,strip_bounds[i*4+1]);
        printf("strip_bounds[%d]=%d\n",i*4+2,strip_bounds[i*4+2]);
        printf("strip_bounds[%d]=%d\n",i*4+3,strip_bounds[i*4+3]);
*/

        //flip_point=RAM_6502[0x05+i*4+0];

        //for(int j=strip_bounds[i*4+2];j<strip_bounds[i*4+3];++j)
        //  strip_indices[j]=i;
      }

      flip_point=RAM_6502[0x18]*8+RAM_6502[0x19];

      //printf("-------\n");

      //for(int y=32;y<VIRTUAL_FRAME_HEIGHT;++y)
      //  strip_indices[y]=RAM_6502[y]>>5;
    }
#else
    for(int i=1;i<N;++i)
    {
      uint8_t z=i*84-(g_cam_sub_z/256);
      uint8_t log_z=log_z_table[z];
      assert(log_z<=log_bias);

      int8_t w=16;
      int8_t cx=int(cos((i+g_rect_offset)*2/2.0f)*32);
      int8_t cy=int(sin((i+g_rect_offset)*2/6.0f)*32);

      assert(cx>=-64+w);
      assert(cx<64-w);

      int8_t l=cx-w-g_cam_x;
      int8_t r=cx+w-g_cam_x;

      auto fx = [&](int x){
                  return (x<0?-1:+1)*(exp_table[MAX(0,log_table[uint8_t(x)]-log_z)]);
                  };


      uint8_t min_x=128+fx(l);
      uint8_t max_x=128+fx(r);


      assert(cy>=-64+w);
      assert(cy<64-w);

      int8_t t=cy-w-g_cam_y;
      int8_t b=cy+w-g_cam_y;

      auto fy = [&](int x){
                  return (x<0?-1:+1)*(exp_table[MAX(0,log_table[uint8_t(x)]-log_z)]);
                  };

      uint8_t min_y=128+fy(t);
      uint8_t max_y=128+fy(b);

      if(i>1)
      {
        // Clip with outer rectangle
        min_x=MAX(min_x,strip_bounds[(i-1)*4+0]);
        max_x=MIN(max_x,strip_bounds[(i-1)*4+1]);
        min_y=MAX(min_y,strip_bounds[(i-1)*4+2]);
        max_y=MIN(max_y,strip_bounds[(i-1)*4+3]);
      }

      flip_point=min_x;

      if(max_x<=min_x || max_y<=min_y)
        break;

      strip_bounds[i*4+0]=min_x;
      strip_bounds[i*4+1]=max_x;
      strip_bounds[i*4+2]=MAX(0,min_y);
      strip_bounds[i*4+3]=MIN(VIRTUAL_FRAME_HEIGHT-1,max_y);

      for(int j=strip_bounds[i*4+2];j<strip_bounds[i*4+3];++j)
        strip_indices[j]=i;
    }
#endif

  }


  // Determine the scroll points (basically the points where
  // the scanline IRQ should fire).
  static const int max_scroll_points=32+4+1;
  static uint8_t scroll_points[max_scroll_points*2];
  memset(scroll_points,0,sizeof(scroll_points));

  uint8_t num_scroll_points=0;

#if 1
  for(int y=0;y<38;++y)
  {
    scroll_points[num_scroll_points*2+0]=RAM_6502[0x80+y*2+1]; // Coarse scroll
    scroll_points[num_scroll_points*2+1]=RAM_6502[0x80+y*2+0]; // Scanlines delta
    ++num_scroll_points;
  }
#else
  // Force the top 32 scanlines to display the HUD (and leave the overscan blank)
  for(int i=0;i<32;++i)
    strip_indices[i]=4;

  // Force the bottom 16 scanlines to display the HUD (and leave the overscan blank)
  for(int i=0;i<16;++i)
    strip_indices[VIRTUAL_FRAME_HEIGHT-1-i]=4;

  int prev_y=0;
  for(int y=0;y<VIRTUAL_FRAME_HEIGHT;++y)
  {
    if(y==0)
    {
      assert(num_scroll_points<max_scroll_points);
      scroll_points[num_scroll_points*2+0]=strip_indices[y]<<5;
      ++num_scroll_points;
      prev_y=y;
      continue;
    }
    if(y==(VIRTUAL_FRAME_HEIGHT-1) || scroll_points[(num_scroll_points-1)*2+0]!=strip_indices[y]<<5)
    {
      assert(num_scroll_points<max_scroll_points);
      scroll_points[num_scroll_points*2+0]=strip_indices[y]<<5;
      assert((y-prev_y)>0);
      scroll_points[(num_scroll_points-1)*2+1]=y-prev_y-1;
      ++num_scroll_points;
      prev_y=y;
      continue;
    }
    if((y&7)==0 && strip_indices[y]!=4)
    {
      assert(num_scroll_points<max_scroll_points);
      scroll_points[num_scroll_points*2+0]=scroll_points[(num_scroll_points-1)*2+0];
      assert((y-prev_y)>0);
      scroll_points[(num_scroll_points-1)*2+1]=y-prev_y-1;
      ++num_scroll_points;
      prev_y=y;
      continue;
    }
  }
#endif

  static uint8_t strip_nametables[N*32];
  memset(strip_nametables,0,sizeof(strip_nametables));

  const int flip_point_fine=flip_point&7;
  const int flip_point_coarse=flip_point/8;

  //printf("vector_to_pattern_left[8*9*9*9 + 0*9*9 + 0*0] = $%02X\n",vector_to_pattern_left[8*9*9*9 + 0*9*9 + 0*0]);
  // Initialize for frontmost layer on the left side
  for(int i=0;i<flip_point_coarse;++i)
    strip_nametables[0*32+i]=0x2C;

  //printf("vector_to_pattern_right[0*9*9*9 + 0*9*9 + 0*9 + 8] = $%02X\n",vector_to_pattern_right[0*9*9*9 + 0*9*9 + 0*9 + 8]);
  // Initialize for frontmost layer on the right side
  for(int i=flip_point_coarse;i<32;++i)
    strip_nametables[0*32+i]=0x2D;


#if 1
  for(int x=0;x<32;++x)
  {
    strip_nametables[1*32+x]=RAM_6502[1*32+x];
    strip_nametables[2*32+x]=RAM_6502[2*32+x];
    strip_nametables[3*32+x]=RAM_6502[3*32+x];
  }
#else
  // Splat the layers

  // Second layer
  {
    if(strip_bounds[1*4+0] < strip_bounds[1*4+1])
    {
      int sb0=strip_bounds[1*4+0]-flip_point_fine;
      assert(sb0>=0);
      printf("sb0/8=%d, flip_point_coarse=%d\n",sb0/8,flip_point_coarse);
      assert(sb0/8<=flip_point_coarse);

      //int x=0;

      //printf("vector_to_pattern_left[8*9*9*9 + 0*9*9 + 0*0] = $%02X\n",vector_to_pattern_left[8*9*9*9 + 0*9*9 + 0*0]);
      // Outer left
      for(int x=0;x<sb0/8;++x)
      {
        strip_nametables[1*32+x]=0x2C;
      }

      //printf("vector_to_pattern_left[8*9*9*9 + 0*9*9 + 0*0] = $%02X\n",vector_to_pattern_left[8*9*9*9 + 0*9*9 + 0*0]);
      //if(x != flip_point_coarse)
      {
        int p=0x2C;
        strip_nametables[1*32+sb0/8]=strip_splat_table[(sb0&7)*256+p];
      }

      //printf("vector_to_pattern_left[0*9*9*9 + 8*9*9 + 0*9] = $%02X\n",vector_to_pattern_left[0*9*9*9 + 8*9*9 + 0*9]);
      // Inner left
      for(int x=sb0/8+1;x<flip_point_coarse;++x)
      {
        strip_nametables[1*32+x]=0x08;
      }

      int sb1=strip_bounds[1*4+1]-flip_point_fine;
      assert(sb1>=0);
      printf("sb1/8=%d, flip_point_coarse=%d\n",sb1/8,flip_point_coarse);
      assert(sb1/8>=flip_point_coarse);

      //printf("vector_to_pattern_right[0*9*9*9 + 0*9*9 + 8*9 + 0] = $%02X\n",vector_to_pattern_right[0*9*9*9 + 0*9*9 + 8*9 + 0]);
      // Inner right
      for(int x=flip_point_coarse;x<sb1/8;++x)
      {
        strip_nametables[1*32+x]=0x35;
      }

      //printf("vector_to_pattern_right[0*9*9*9 + 0*9*9 + 0*9 + 8] = $%02X\n",vector_to_pattern_right[0*9*9*9 + 0*9*9 + 0*9 + 8]);
      //if(x != flip_point_coarse)
      {
        int p=0x2D;
        strip_nametables[1*32+sb1/8]=strip_splat_table[(sb1&7)*256+p];
      }

      //printf("vector_to_pattern_right[0*9*9*9 + 0*9*9 + 0*9 + 8] = $%02X\n",vector_to_pattern_right[0*9*9*9 + 0*9*9 + 0*9 + 8]);
      // Outer right
      for(int x=sb1/8+1;x<32;++x)
      {
        strip_nametables[1*32+x]=0x2D;
      }
    }
    else
    {
      for(int x=0;x<32;++x)
        strip_nametables[1*32+x]=strip_nametables[0*32+x];
    }
  }


  // Third layer
  {
    if(strip_bounds[2*4+0] < strip_bounds[2*4+1])
    {
      int sb0=strip_bounds[2*4+0]-flip_point_fine;

      // Outer left
      for(int x=0;x<sb0/8;++x)
        strip_nametables[2*32+x]=strip_nametables[1*32+x];

      int p=strip_nametables[1*32+sb0/8];
      strip_nametables[2*32+sb0/8]=strip_splat_table[(sb0&7)*256+p];

      //printf("vector_to_pattern_left[0*9*9*9 + 0*9*9 + 8*9] = $%02X\n",vector_to_pattern_left[0*9*9*9 + 0*9*9 + 8*9]);
      // Inner left
      for(int x=sb0/8+1;x<flip_point_coarse;++x)
        strip_nametables[2*32+x]=0xFF;

      int sb1=strip_bounds[2*4+1]-flip_point_fine;

      // Inner right
      for(int x=flip_point_coarse;x<sb1/8;++x)
        strip_nametables[2*32+x]=0x59;

      p=strip_nametables[1*32+sb1/8];
      strip_nametables[2*32+sb1/8]=strip_splat_table[(sb1&7)*256+p];

      // Outer right
      for(int x=sb1/8+1;x<32;++x)
        strip_nametables[2*32+x]=strip_nametables[1*32+x];
    }
    else
    {
      for(int x=0;x<32;++x)
        strip_nametables[2*32+x]=strip_nametables[1*32+x];
    }
  }

  // Fourth layer
  {
    if(strip_bounds[3*4+0] < strip_bounds[3*4+1])
    {
      for(int x=0;x<flip_point_coarse;++x)
        strip_nametables[3*32+x]=strip_nametables[2*32+x];

      int sb1=strip_bounds[3*4+1]-flip_point_fine;

      //if(strip_bounds[3*4+1]-strip_bounds[3*4+0]!=0)
      {

      // Inner right
      for(int x=flip_point_coarse;x<sb1/8;++x)
        strip_nametables[3*32+x]=0xD1;

      int p=strip_nametables[2*32+sb1/8];
      strip_nametables[3*32+sb1/8]=strip_splat_table[(sb1&7)*256+p];

      // Outer right
      for(int x=sb1/8+1;x<32;++x)
        strip_nametables[3*32+x]=strip_nametables[2*32+x];
      }
    }
    else
    {
      for(int x=0;x<32;++x)
        strip_nametables[3*32+x]=strip_nametables[2*32+x];
    }
  }

#endif


  // Draw the screen

  int bank_offset=0;
  int attribute_palette[4];

  //bank_offset=((3-(frame_count&63)/16))*pattern_bank_size;

  static const int layer_palette_cycle[3]={0x2A,0x17,0x21};
  attribute_palette[0]=0x1D;

  attribute_palette[1]=(g_rect_offset+2)%64 ? layer_palette_cycle[(g_rect_offset+2)%3] : 0x20;
  attribute_palette[2]=(g_rect_offset+1)%64 ? layer_palette_cycle[(g_rect_offset+1)%3] : 0x20;
  attribute_palette[3]=(g_rect_offset+0)%64 ? layer_palette_cycle[(g_rect_offset+0)%3] : 0x20;


  static bool_t dump=TRUE;

  if(dump)
  {
    dump=FALSE;

    {
      FILE* out=fopen("strip_nametables.bin","wb");
      fwrite(strip_nametables,1,sizeof(strip_nametables),out);
      fclose(out);
    }

    {
      FILE* out=fopen("scroll_points.bin","wb");
      fwrite(scroll_points,1,sizeof(scroll_points),out);
      fclose(out);
    }
  }

  // Method simulating scanline IRQ
  int irq_count=0;
  int scanline_counter=0;
  int y_scroll=0;
  for(int y=0;y<VIRTUAL_FRAME_HEIGHT;++y)
  {
    if(scanline_counter==0)
    {
      y_scroll=(scroll_points[irq_count*2+0]>>5)*8;
      scanline_counter=scroll_points[irq_count*2+1]+1;
      ++irq_count;
    }
    for(int x=0;x<VIRTUAL_FRAME_WIDTH;++x)
    {
      if(x<8)
        plotPixelVirtual(x, y, 0x1D);
      else if(y_scroll/8<4)
      {
        int p=strip_nametables[y_scroll/8*32+(x-flip_point_fine)/8];

        assert(patterns[bank_offset+(p*8+(y&7))*8+((x-flip_point_fine)&7)]>=0);
        assert(patterns[bank_offset+(p*8+(y&7))*8+((x-flip_point_fine)&7)]<4);
        plotPixelVirtual(x, y, attribute_palette[patterns[bank_offset+(p*8+(y&7))*8+((x-flip_point_fine)&7)]]);
      }
      else
        plotPixelVirtual(x, y, 0x1D);
    }
    --scanline_counter;
    ++y_scroll;
  }

  // HUD mock-up
  {
    int bank_offset=1*pattern_bank_size;
    int hud_palette[4] = { 0x1D, 0x00, 0x10, 0x20 };

    for(int y=0;y<32;++y)
      for(int x=0;x<256;++x)
      {
        int p=hud_nametable[x/8+(y/8)*32];
        plotPixelVirtual(x,y, hud_palette[patterns[bank_offset+(p*8+(y&7))*8+(x&7)]]);
      }

/*
    {
      const char* s="LAP";
      for(int i=0;i<strlen(s);++i)
        for(int y=0;y<8;++y)
          for(int x=0;x<8;++x)
            plotPixelVirtual(16+i*8+x, 16+y, hud_palette[patterns[bank_offset+((s[i]-32)*8+(y&7))*8+(x&7)]]);
    }
    {
      //const char* s="1/3";
      char s[16];
      sprintf(s,"%01d/%01d",g_lap_count,g_laps);
      for(int i=0;i<strlen(s);++i)
        for(int y=0;y<16;++y)
          for(int x=0;x<8;++x)
          {
            plotPixelVirtual(16+8+3*8+i*8+x, 16+y, hud_palette[patterns[bank_offset+(((s[i]-32)*2+(y/8)+64)*8+(y&7))*8+(x&7)]]);
          }
    }
    {
      const char* s="TIME";
      for(int i=0;i<strlen(s);++i)
        for(int y=0;y<8;++y)
          for(int x=0;x<8;++x)
            plotPixelVirtual(128-4*8+i*8+x, 16+y, hud_palette[patterns[bank_offset+((s[i]-32)*8+(y&7))*8+(x&7)]]);
    }
    {
      //const char* s="38";
      char s[16];
      sprintf(s,"%02d",g_time_remaining);
      for(int i=0;i<strlen(s);++i)
        for(int y=0;y<16;++y)
          for(int x=0;x<8;++x)
          {
            plotPixelVirtual(128+8+i*8+x, 16+y, hud_palette[patterns[bank_offset+(((s[i]-32)*2+(y/8)+64)*8+(y&7))*8+(x&7)]]);
          }
    }
    {
      const char* s="SPEED";
      for(int i=0;i<strlen(s);++i)
        for(int y=0;y<8;++y)
          for(int x=0;x<8;++x)
            plotPixelVirtual(256-64+i*8+x, 16-8+y, hud_palette[patterns[bank_offset+((s[i]-32)*8+(y&7))*8+(x&7)]]);
    }
*/
    {
      static const int dither[8][8] = {
        {  0, 32,  8, 40,  2, 34, 10, 42 },
        { 48, 16, 56, 24, 50, 18, 58, 26 },
        { 12, 44,  4, 36, 14, 46,  6, 38 },
        { 60, 28, 52, 20, 62, 30, 54, 22 },
        {  3, 35, 11, 43,  1, 33,  9, 41 },
        { 51, 19, 59, 27, 49, 17, 57, 25 },
        { 15, 47,  7, 39, 13, 45,  5, 37 },
        { 63, 31, 55, 23, 61, 29, 53, 21 },
      };
      static const int ramp[6] = { 0x2C, 0x22, 0x24, 0x26, 0x27 };

      int inset=3;
      // Speed indicator
      const int w=9*8-inset*2;
      for(int x=0;x<w;++x)
        for(int y=0;y<16-inset*2;++y)
        {
          int u=(g_cam_speed*w)/top_speed;
          if(x<u)
          {
            int c=x<(8-inset) ? ramp[0] : x>=8*8-inset ? ramp[4] : ramp[(x-(8-inset)+4)/16+(dither[x&7][y&7]<((x-(8-inset)+4)&15)*4)];
            plotPixelVirtual(256-64-16+inset+x, 16+inset+y, c);
          }
        }
    }
  }

  {
    // Overlaid messages, constructed of sprites
    {
      int bank_offset=1*pattern_bank_size;
      int sprite_palette[4] = { 0x1D, 0x00, 0x10, 0x20 };

      uint8_t* sprites= g_game_mode == MODE_TIMEOVER ? sprites_timeover :
                        g_game_mode == MODE_CRASHED ? sprites_crashed :
                        g_game_mode == MODE_WON ? sprites_won :
                        nullptr;

      if(sprites)
        for(int i=0;i<64;++i)
        {
          uint8_t* s=sprites+i*4;
          for(int y=0;y<16;++y)
            for(int x=0;x<8;++x)
            {
              int p=patterns[bank_offset+(((s[1]&0xFE)+y/8)*8+(y&7))*8+(x&7)];
              if(p!=0)
                plotPixelVirtual(s[3]+x, s[0]+1+y, sprite_palette[p]);
            }
        }
    }
  }

#if 1
  {
    // Stars background
    static int frame=0;
    const int bank=2;
    int palette[4];
    if(frame&1)
    {
      palette[0] = 0x1D;
      palette[1] = 0x1D;
      palette[2] = 0x20;
      palette[3] = 0x20;
    }
    else
    {
      palette[0] = 0x1D;
      palette[1] = 0x20;
      palette[2] = 0x1D;
      palette[3] = 0x20;
    }
    int banks[4];
    banks[0]=(frame/4+0)&3;
    banks[1]=(frame/4+1)&3;
    banks[2]=(frame/4+2)&3;
    banks[3]=(frame/4+3)&3;
    for(int y=0;y<256;++y)
      for(int x=0;x<256;++x)
      {
        int p=stars_nametables[((frame/2)&1)*1024+(x/8)+(y/8)*32];
        if(p<64)
          p=(p&63)|(banks[0]*64);
        else if(p<128)
          p=(p&63)|(banks[1]*64);
        else if(p<192)
          p=(p&63)|(banks[2]*64);
        else
          p=(p&63)|(banks[3]*64);
        int c=((pattern_banks[bank*4096+p*16+(y&7)+0]>>(7-(x&7)))&1) |
             (((pattern_banks[bank*4096+p*16+(y&7)+8]>>(7-(x&7)))&1)<<1);
        assert(c>=0);
        assert(c<4);
        plotPixelVirtual(x,y,palette[c]);
      }
    ++frame;
  }
#endif

  {
    // Pickup item test
    uint8_t z=(1+2-(g_rect_offset%3))*84-(g_cam_sub_z/256);
    uint8_t log_z=log_z_table[z];
    assert(log_z<=log_bias);

    int8_t cx=0-g_cam_x;

    auto fx = [&](int x){
                return (x<0?-1:+1)*(exp_table[MAX(0,log_table[uint8_t(x)]-log_z)]);
                };

    uint8_t px=128+fx(cx);

    int8_t cy=0-g_cam_y;

    auto fy = [&](int x){
                return (x<0?-1:+1)*(exp_table[MAX(0,log_table[uint8_t(x)]-log_z)]);
                };

    uint8_t py=128+fy(cy);

    const int frame=(log_z*(pickup_frames-1))/69;

    static bool dumptable=true;
    if(dumptable)
    {
      dumptable=false;
      static uint8_t logz_to_pickupframe[256];
      memset(logz_to_pickupframe,0,sizeof(logz_to_pickupframe));
      for(int i=0;i<256;++i)
      {
        int frame=(i*(pickup_frames-1))/69;
        logz_to_pickupframe[i]=frame;
      }
      FILE* out=fopen("logz_to_pickupframe.bin","wb");
      fwrite(logz_to_pickupframe,1,sizeof(logz_to_pickupframe),out);
      fclose(out);
    }

    {
      uint8_t ramp[6] = { 0x1D, 0x05, 0x15, 0x25, 0x35, 0x20 };
      uint8_t palette[4][4] = {
                                { 0x1D, ramp[3], ramp[4], ramp[5] },
                                { 0x1D, ramp[3], ramp[3], ramp[3] },
                                { 0x1D, ramp[3], ramp[3], ramp[3] },
                                { 0x1D, ramp[3], ramp[3], ramp[2] },
                              };
      for(int y=0;y<pickup_height;++y)
        for(int x=0;x<pickup_width;++x)
        {
          uint8_t p=pickup_images[x+y*pickup_width + pickup_width*pickup_height*frame];
          assert(p<4);
          if(p)
            plotPixelVirtual(px+x-pickup_width/2,py+y-pickup_height/2,
                    palette[((x/(pickup_width/2))&1) + ((y/(pickup_height/2))&1)*2][p]);
        }
      uint8_t pickup_palettes[12];
      for(int i=0;i<4;++i)
      {
        pickup_palettes[0+i]=palette[0][i];
        pickup_palettes[4+i]=palette[1][i];
        pickup_palettes[8+i]=palette[3][i];
      }
      FILE* out=fopen("pickup_palettes.bin","wb");
      fwrite(pickup_palettes,1,sizeof(pickup_palettes),out);
      fclose(out);
    }

/*
    if(i>1)
    {
      // Clip with outer rectangle
      min_x=MAX(min_x,strip_bounds[(i-1)*4+0]);
      max_x=MIN(max_x,strip_bounds[(i-1)*4+1]);
      min_y=MAX(min_y,strip_bounds[(i-1)*4+2]);
      max_y=MIN(max_y,strip_bounds[(i-1)*4+3]);
    }
*/

  }

#if 1
  {
    // Title screen mockup

    for(int y=0;y<240;++y)
      for(int x=0;x<256;++x)
      {
        plotPixelVirtual(x,y,0x1D);
      }

/*
    for(int y=0;y<64;++y)
      for(int x=0;x<128;++x)
      {
        if(title_screen_graphic[x+y/8*128])
          plotPixelVirtual(128+x,128+y,0x20);
        else
          plotPixelVirtual(128+x,128+y,0x1D);
      }
*/

    static int logo_scroll_y[32*64];

    for(int i=0;i<32*64;++i)
      logo_scroll_y[i]=255;

    for(int frame=0;frame<64;++frame)
    {
      const int theta_i=frame&63;
      const int H=16;
      double theta=M_PI/2.0-fmodf(double(theta_i)*(M_PI/2.0)/32.0,M_PI/2.0);
      int mask=(int(-double(theta_i)*(M_PI/2.0)/32.0/(M_PI/2.0))&1)?2:1;
      double perspective=1.0/8.0;
      for(int y=-H;y<H;++y)
      {
        int scroll=255;

        {
          double ox=cos(theta-M_PI/2.0)*1.5*2;
          double oy=sin(theta-M_PI/2.0)*1.5*2;
          double r_dydz=double(y)/double(H);
          double rx=-cos(theta-M_PI/2.0)+sin(theta-M_PI/2.0)*r_dydz*perspective/2.0*1.3;
          double ry=-sin(theta-M_PI/2.0)-cos(theta-M_PI/2.0)*r_dydz*perspective/2.0*1.3;
          double t=(0.333/2-ox)/rx;
          if(t>1.0 && ox>0.333/2)
          {
            int u=(oy+ry*t)*-6.0*4+3.0;
            double w=MAX(0,MIN(128,128.0/t*1.25*2.2));
            if(u>=0 && u<=6 && w>116)
            {
              bool notblank=u>0.0 && u<6.0;
              scroll=notblank?(u-1+int(128-w)*5+(mask==2?64:0)):(u-1+int(128-w)*5+(mask==2?192:128));
            }
          }
        }
        {
          double ox=cos(theta)*1.5*2;
          double oy=sin(theta)*1.5*2;
          double r_dydz=double(y)/double(H);
          double rx=-cos(theta)+sin(theta)*r_dydz*perspective/2.0*1.3;
          double ry=-sin(theta)-cos(theta)*r_dydz*perspective/2.0*1.3;
          double t=(0.333/2-ox)/rx;
          if(t>1.0 && ox>0.333/2)
          {
            int u=(oy+ry*t)*-6.0*4+3.0;
            double w=MAX(0,MIN(128,128.0/t*1.25*2.2));
            if(u>=0 && u<=6 && w>116)
            {
              bool notblank=u>0.0 && u<6.0;
              scroll=notblank?(u-1+int(128-w)*5+(mask==1?64:0)):(u-1+int(128-w)*5+(mask==1?192:128));
            }
          }
        }

        if(y==(H-1))
          scroll=255;

        logo_scroll_y[frame*32+y+H]=scroll;
      }
    }

    static uint8_t logo_scroll_irq[64*64];
    memset(logo_scroll_irq,0,sizeof(logo_scroll_irq));

    for(int frame=0;frame<64;++frame)
      for(int y=0;y<32;++y)
      {
        int scroll=logo_scroll_y[frame*32+y];
        assert(scroll>=0);
        assert(scroll<256);
/*
  ; AAA AA AABBB BBBBB
  ; yyy NN YYYYY XXXXX
  ; ||| || ||||| +++++-- coarse X scroll
  ; ||| || +++++-------- coarse Y scroll
  ; ||| ++-------------- nametable select
  ; +++----------------- fine Y scroll
*/
        //logo_scroll_irq[frame*64+y*2+0]=((scroll&7)<<4)|((scroll&0b11000000)>>6);
        //logo_scroll_irq[frame*64+y*2+1]=(scroll&0b111000)<<2;

        //logo_scroll_irq[frame*64+y*2+0]=0;
        //logo_scroll_irq[frame*64+y*2+1]=0;//-scroll;

        if(y==0)
          scroll=64; // In nametable 2

        logo_scroll_irq[frame*64+y*2+0]=scroll;
        logo_scroll_irq[frame*64+y*2+1]=((scroll & 0xF8) << 2) | (0 >> 3);
      }

    static bool dump=true;
    if(dump)
    {
      dump=false;
      {
        FILE* out=fopen("logo_scroll_irq.bin","wb");
        fwrite(logo_scroll_irq,1,sizeof(logo_scroll_irq)/2,out); // NOTE: Divided by two!
        fclose(out);
      }
    }

    static int frame_count=0;

    //const int theta_i=int(double(frame_count)+32.0+20.0*cos(double(frame_count)/30.0))&63;
    const int theta_i=frame_count&63;

    const int H=16;
    double theta=M_PI/2.0-fmodf(double(theta_i)*(M_PI/2.0)/32.0,M_PI/2.0);
    int mask=(int(-double(theta_i)*(M_PI/2.0)/32.0/(M_PI/2.0))&1)?2:1;
    double perspective=1.0/8.0;
    for(int y=-H+1;y<H;++y)
    {
      //for(double x=0;x<128;++x) plotPixelVirtual(128+x-64,128+y,0x00);

/*
      static uint8_t ramps[4*2] = { 0x01, 0x0C,
                                    0x11, 0x1C,
                                    0x21, 0x2C,
                                    0x31, 0x3C,
                                  };
*/

      static uint8_t ramps[4*2] = { 0x01, 0x01,
                                    0x01, 0x01,
                                    0x01, 0x01,
                                    0x01, 0x01,
                                  };

      {
        double ox=cos(theta)*1.5*2;
        double oy=sin(theta)*1.5*2;
        double r_dydz=double(y)/double(H);
        double rx=-cos(theta)+sin(theta)*r_dydz*perspective/2.0*1.3;
        double ry=-sin(theta)-cos(theta)*r_dydz*perspective/2.0*1.3;
        double t=(0.333/2-ox)/rx;
        if(t>1.0 && ox>0.333/2)
        {
          uint8_t bgcol=ramps[MAX(0,MIN(3,int(theta*4.0/(M_PI/2.0))))*2+mask-1];

          int u=(oy+ry*t)*-6.0*4+3.0;
          double w=MAX(0,MIN(128,128.0/t*1.25*2.2));
          if(u>=0 && u<=6 && w>116)
            for(double x=0;x<128;++x)
            {
              bool notblank=u>0.0 && u<6.0;
              uint8_t p=notblank ? title_screen_graphic_scaled[int(x)+(u-1+int(128-w)*5)*128] : 0;
              if(x>64-w/2.0 && x<64+w/2.0)
                plotPixelVirtual(128+x-64,128+y,!notblank ? 0x1C : (p&(mask)) ? 0x20 : bgcol);
            }
        }
      }
      {
        double ox=cos(theta-M_PI/2.0)*1.5*2;
        double oy=sin(theta-M_PI/2.0)*1.5*2;
        double r_dydz=double(y)/double(H);
        double rx=-cos(theta-M_PI/2.0)+sin(theta-M_PI/2.0)*r_dydz*perspective/2.0*1.3;
        double ry=-sin(theta-M_PI/2.0)-cos(theta-M_PI/2.0)*r_dydz*perspective/2.0*1.3;
        double t=(0.333/2-ox)/rx;
        if(t>1.0 && ox>0.333/2)
        {
          uint8_t bgcol=ramps[MAX(0,MIN(3,(4-int(theta*4.0/(M_PI/2.0)))))*2+(1-(mask-1))];

          int u=(oy+ry*t)*-6.0*4+3.0;
          double w=MAX(0,MIN(128,128.0/t*1.25*2.2));
          if(u>=0 && u<=6 && w>116)
            for(double x=0;x<128;++x)
            {
              bool notblank=u>0.0 && u<6.0;
              uint8_t p=notblank ? title_screen_graphic_scaled[int(x)+(u-1+int(128-w)*5)*128] : 0;
              if(x>64-w/2.0 && x<64+w/2.0)
                plotPixelVirtual(128+x-64,128+y,!notblank ? 0x1C : (p&(mask^3)) ? 0x20 : bgcol);
            }
        }
      }
    }

    ++frame_count;
  }
#endif

#if 1
  {
    // Info screen mockup
    static uint8_t info_nametable[32*32];
    static uint8_t info_nametable2[32*32];
    static uint8_t ending_nametable[32*32];

    static int frame_count=0;
    static uint8_t sprites[64*4];
    static bool init=true;

    if(init)
    {
      init=false;

      xorshift32_state s;
      s.a=10;

      for(int i=0;i<64;++i)
      {
        xorshift32(&s);
        sprites[i*4+0]=(s.a>>8)&0xFF; // Y
        sprites[i*4+1]=i<32?0x57:0x5A; // Tile
        sprites[i*4+2]=0b00100000; // Attribute (behind BG)
        sprites[i*4+3]=s.a&0xFF; // X
      }

      {
        std::set<std::string> greetnames = {
            "KABUTO", "NEOMAN", "ISH", "IKS", "MEDO", "ALK", "ALKAMA",
            "LAS", "CUPE", "DFOX", "XTR1M", "LYCAN", "XTRIUM", "GASMAN",
            "DOJOE", "KAOMAU", "RANDOM", "TROPICALTREVOR", "BEFTEX",
            "CAPITALISM", "T$", "SUNSPIRE", "RBR", "TOPY", "LAYLA",
            "MRS.BEANBAG", "REALITY404", "TRUCK", "HAVOC", "SAGAMUSIX",
            "KEY REAL", "CCE", "JASMIN68K", "OHLI", "DSTAR", "LJ", "VIRGILL",
            "BLUEBERRY", "GOPHER", "H0FFMAN", "SMASH", "DESTOP", "LYNN",
            "JIX", "ALIEN", "ZANAGB", "CYRAXX", "NOSFE", "MANKELI",
            "MSQRT", "IQ", "GARGAJ", "PSYCHO", "EATDRINKFUCK", "WURSTGETRANK",
            "HELLMOOD", "SENSENSTAHL", "LIPSTIX", "GOBUSTO", "MR.D", "NNORM",
            "RED", "WYSIWTF", "TRIACE", "PURYX", "EVILPAUL", "FERRIS", 
            "PSENOUGH", "LLB", "ZAVIE", "FELL", "OKKIE", "YX", "NUCLEAR",
            "BOYC", "LUG00BER", "P_MALIN", "ADEPTAPRIL", "VISY", "EVVVIL",
            "NEU", "FRANKY", "SEVEN", "SLERPY", "MUFFINHOP"
          };

        std::string greets;
        for(const std::string& name : greetnames)
        {
          greets += name + " ";
        }
        printf("greetnames = \"%s\"\n",greets.c_str());

/*
 0123456789ABCDEF0123456789ABCDEF
"  ALIEN, ALK, ALKAMA, BEFTEX,   "
"  BLUEBERRY, CAPITALISM, CCE,   "
"  CUPE, DESTOP, DFOX, DOJOE,    "
"  DSTAR, GASMAN, GOPHER,        "
"  H0FFMAN, HAVOC, IKS, ISH,     "
"  JASMIN68K, JIX, KABUTO,       "
"  KAOMAU, KEY REAL, LAS,        "
"  LAYLA, LJ, LYCAN, LYNN,       "
"  MEDO, MRS. BEANBAG, NEOMAN,   "
"  OHLI, RANDOM, RBR, REALITY404,"
"  SAGAMUSIX, SMASH, SUNSPIRE,   "
"  T$, TOPY, TROPICAL TREVOR,    "
"  TRUCK, VIRGILL, XTR1M, XTRIUM,"
"  ZANAGB                        "
*/

/*
 0123456789ABCDEF0123456789ABCDEF
"  ADEPTAPRIL  ALIEN ALK ALKAMA  "
"  BEFTEX     BLUEBERRY    BOYC  "
"  CAPITALISM  CCE  CUPE CYRAXX  "
"  DESTOP   DFOX   DOJOE  DSTAR  "
"  EATDRINKFUCK EVILPAUL EVVVIL  "
"  FELL  FERRIS  FRANKY  GARGAJ  "
"  GASMAN GOBUSTO GOPHER H0FFMAN "
"  HAVOC  HELLMOOD  IKS  IQ ISH  "
"  JASMIN68K  JIX KABUTO KAOMAU  "
"  KEY REAL LAS LAYLA LIPSTIX LJ "
"  LLB   LUG00BER   LYCAN  LYNN  "
"  MANKELI MEDO MR.D MRS.BEANBAG "
"  MSQRT NEOMAN NEU NNORM NOSFE  "
"  NUCLEAR  OHLI OKKIE PSENOUGH  "
"  PSYCHO PURYX  P_MALIN RANDOM  "
"  RBR REALITY404 RED SAGAMUSIX  "
"  SENSENSTAHL   SEVEN   SLERPY  "
"  SMASH SUNSPIRE T$ TOPY TRIACE "
"  TROPICALTREVOR TRUCK VIRGILL  "
"  VISY   WURSTGETRANK  WYSIWTF  "
"  XTR1M XTRIUM YX ZANAGB ZAVIE  "
*/

/*
        //             0123456789ABCDEF0123456789ABCDEF
        const char* s="                                "
                      "           GREETINGS            " // 2
                      "           GREETINGS            " // 2
                      "                                " // 2
                      "  IN ALPHABETICAL ORDER:        " // 2
                      "                                " // 2
                      "              ALIEN ALK ALKAMA  "
                      "  BEFTEX    BLUEBERRY     BOYC  "
                      "              CCE  CUPE CYRAXX  "
                      "  DESTOP   DFOX   DOJOE  DSTAR  "
                      "  EATDRINKFUCK EVILPAUL EVVVIL  "
                      "  FELL  FERRIS  FRANKY  GARGAJ  "
                      "  GASMAN GOBUSTO GOPHER H0FFMAN "
                      "  HAVOC  HELLMOOD  IKS  IQ ISH  "
                      "  JASMIN68K  JIX KABUTO KAOMAU  "
                      "           LAS LAYLA LIPSTIX LJ "
                      "  LLB   LUG00BER   LYCAN  LYNN  "
                      "  MANKELI MEDO MR.D             "
                      "  MSQRT  BRANCH     NEOMAN NEU  "
                      "  NNORM  NOSFE  NUCLEAR  OKKIE  "
                      "  PSENOUGH PSYCHO PURYX P_MALIN "
                      "  RBR REALITY404 RED SAGAMUSIX  "
                      "  SENSENSTAHL   SEVEN           "
                      "  SMASH SUNSPIRE T$ TOPY TRIACE "
                      "  TROPICALTREVOR TRUCK VIRGILL  "
                      "  VISY   WURSTGETRANK  WYSIWTF  "
                      "  XTR1M XTRIUM YX ZANAGB ZAVIE  "
                      "                                " // 3
                      "  APOLOGIES TO ANYONE WE FORGOT " // 4
            ;
*/

        //             0123456789ABCDEF0123456789ABCDEF
        const char* s="                                "
                      "           GREETINGS            " // 2
                      "           GREETINGS            " // 2
                      "                                " // 2
                      "  IN ALPHABETICAL ORDER:        " // 2
                      "                                " // 2
                      "  ADEPTAPRIL  ALIEN  ALK  ALKAMA"
                      "  ASPEKT ATOMIM BEFTEX BLUEBERRY"
                      "  BOYC  CAPITALISM CUPE  CYRAXX "
                      "  DESTOP   DFOX   DOJOE   DSTAR "
                      "  EATDRINKFUCK EVILPAUL  EVVVIL "
                      "  FELL  FERRIS   FRANKY  GARGAJ "
                      "  GASMAN GOBUSTO  GOPHER H0FFMAN"
                      "  HAVOC  HELLMOOD  IKS  IQ  ISH "
                      "  JAMMER  JASMIN68K JIX  KABUTO "
                      "  KAOMAU  KEY-REAL  LAS  LAYLA  "
                      "  LIPSTIX LJ LLB  LUG00BER LYCAN"
                      "  LYNN   MANKELI   MEDO    MR.D "
                      "  MRS.BEANBAG  NEOMAN NEU  NNORM"
                      "  NOSFE  NUCLEAR  OKKIE PSENOUGH"
                      "  PSYCHO PURYX P_MALIN RBR R404 "
                      "  RED REN SAGA SENSENSTAHL SEVEN"
                      "  SMASH THE_CORPORATION T$ TOPY "
                      "  TRIACE  TROPICALTREVOR  TRUCK "
                      "  VIRGILL   VISY   WURSTGETRANK "
                      "  WYSIWTF      XTR1M     XTRIUM "
                      "  YX    YZI    ZANAGB     ZAVIE "
                      "                                " // 3
                      "  APOLOGIES TO ANYONE WE FORGOT " // 4
            ;
/*
        const char* s="                                " // 0
                      "                                " // 1
                      "                                " // 2
                      "                                " // 3
                      "                                " // 4
                      "                                " // 5
                      "    CYBER COASTER RELEASED AT   " // 6
                      "    REVISION 2025 IN GERMANY    " // 7
                      "                                " // 8
                      "                                " // 9
                      "    GREETINGS TO EVERYONE       " // A
                      "    SPECIAL THANKS TO:          " // B
                      "        NESDEV WIKI             " // C
                      "                                " // D
                      "                                " // E
                      "                                " // F
                      "                                " // 0
                      "                                " // 1
                      "                                " // 2
                      "                                " // 3
                      "                                " // 4
                      "                                " // 5
                      "                                " // 6
                      "                                " // 7
            ;
*/
        for(int i=0;i<MIN(strlen(s),32*32);++i)
          info_nametable[i]=s[i]-32;
      }

      drawBigTextInNametable(info_nametable,"GREETINGS",11,1);

      {

        //             0123456789ABCDEF0123456789ABCDEF
        const char* s="                                " // 0
                      "                                " // 2
                      "          HOW TO PLAY           " // 2
                      "          HOW TO PLAY           " // 2
                      "                                " // 2
                      "   AVOID  THE  WALLS, BEAT THE  "
                      "   TIME  LIMIT,  AND  COMPLETE  "
                      "   THREE  LAPS OF EACH LEVEL.   "
                      "   COLLECT  ALL OF THE SPHERES  "
                      "   IN  A  LEVEL  TO  EARN  THE  "
                      "   \"PERFECT\" RATING.            "
                      "                                " // 2
                      "                                " // 2
                      "   PRESS (A) TO ACCELERATE AND  "
                      "   (B)  TO BREAK.  PRESS START  "
                      "   THEN SELECT TO START OVER.   "
                      "                                "
                      "                                " // 2
                      "            CREDITS             " // 2
                      "            CREDITS             " // 2
                      "                                " // 2
                      "   CODE & GRAPHICS: FIZZER      "
                      "       MUSIC & SFX: NOBY        "
                      "                                " // 2
                      "                                "
                      "  MUSIC PLAYER BY SHIRU.        "
                      "  ORIGINAL FONTS BY PINOBATCH.  "
                      "  SPECIAL THANKS TO NESDEV WIKI " // 2
                      "                                " // 2
                      "                                " // 2
                      "                                " // 2
            ;

        for(int i=0;i<MIN(strlen(s),32*32);++i)
          info_nametable2[i]=s[i]-32;
      }

      drawBigTextInNametable(info_nametable2,"HOW TO PLAY",10,2);
      drawBigTextInNametable(info_nametable2,"CREDITS",12,18);

      {

        //             0123456789ABCDEF0123456789ABCDEF
        const char* s="                                " // 0
                      "                                " // 2
                      "                                " // 2
                      "                                " // 2
                      "                                " // 2
                      "                                " // 2
                      "        CONGRATULATIONS!        " // 2
                      "        CONGRATULATIONS!        "
                      "                                " // 2
                      "      ALL LEVELS COMPLETED      " // 2
                      "      ALL LEVELS COMPLETED      " // 2
                      "                                " // 2
                      "                                " // 2
                      "                                " // 2
                      "          100% PERFECT          " // 2
                      "          100% PERFECT          " // 2
                      "                                " // 2
                      "      ***  WAY TO GO! ***       " // 2
                      "                                " // 2
                      "                                " // 2
                      "                                " // 2
                      "                                " // 2
                      "       THANKS FOR PLAYING       " // 2
                      "       SEE YOU NEXT TIME!       " // 2
                      "                                " // 2
                      "                                " // 2
                      "                                " // 2
                      "                                " // 2
                      "                                " // 2
            ;

        for(int i=0;i<MIN(strlen(s),32*32);++i)
          ending_nametable[i]=s[i]-32;
      }

      drawBigTextInNametable(ending_nametable,"CONGRATULATIONS!",8,6);
      drawBigTextInNametable(ending_nametable,"- ALL LEVELS COMPLETED -",4,9);
      drawBigTextInNametable(ending_nametable,"100% PERFECT",10,14);

      {
        FILE* out=fopen("ending_nametable.bin","wb");
        fwrite(ending_nametable,1,sizeof(ending_nametable),out);
        fclose(out);
      }
      {
        FILE* out=fopen("info_nametable.bin","wb");
        fwrite(info_nametable,1,sizeof(info_nametable),out);
        fclose(out);
      }
      {
        FILE* out=fopen("info_nametable2.bin","wb");
        fwrite(info_nametable2,1,sizeof(info_nametable2),out);
        fclose(out);
      }
      {
        FILE* out=fopen("info_sprites.bin","wb");
        fwrite(sprites,1,sizeof(sprites),out);
        fclose(out);
      }
    }

    const int bank=1;

#if 0
    for(int i=0;i<32;++i)
    {
      uint8_t* pattern=&pattern_banks[bank*4096+0x57*16];
      for(int y=0;y<8;++y)
        for(int x=0;x<8;++x)
        {
          if(pattern[y]&(1<<(7-x)))
            plotPixelVirtual(sprites[i*4+3]+x,sprites[i*4+0]+y,0x15);
        }
      sprites[i*4+3]-=3;
    }

    for(int i=32;i<64;++i)
    {
      uint8_t* pattern=&pattern_banks[bank*4096+0x5A*16];
      for(int y=0;y<8;++y)
        for(int x=0;x<8;++x)
        {
          if(pattern[y]&(1<<(7-x)))
            plotPixelVirtual(sprites[i*4+3]+x,sprites[i*4+0]+y,0x15);
        }
      sprites[i*4+3]-=6;
    }

    int bg_palette[4] = { 0x1D, 0x00, 0x10, 0x20 };

    for(int y=0;y<240;++y)
      for(int x=0;x<256;++x)
      {
        plotPixelVirtual(x,y,0x04);
      }

    for(int y=0;y<240;++y)
      for(int x=0;x<256;++x)
      {
        int p=info_nametable[x/8+(y/8)*32];
        int c=patterns[bank*pattern_bank_size+(p*8+(y&7))*8+(x&7)];
        if(c>0)
          plotPixelVirtual(x,y, bg_palette[c]);
      }
#endif

    ++frame_count;
  }
#endif

  blitVirtualFramebuffer(pixels);

  for(int y=0;y<FRAME_HEIGHT;++y)
    for(int x=0;x<FRAME_WIDTH;++x)
    {
      if(x%(8*FRAME_ZOOM_FACTOR)==0 || y%(8*FRAME_ZOOM_FACTOR)==0)
      {
        //pixels[x+y*FRAME_WIDTH]|=RGB888toRGB565(100,100,100);
      }
    }

#if 0
  {
    static char debug_text[8192];
    memset(debug_text, 0, sizeof(debug_text));
    snprintf(debug_text, sizeof(debug_text),
                     "total_cycles      = %u\n"
                     "scanlines         = %u\n"
                     "camera            = %4d, %4d\n"
                     ,(uint32_t)total_cycles, (uint32_t)(double(total_cycles)/133.66+0.99),
                      g_cam_x,g_cam_y);

    plotString(pixels, 16, 32*FRAME_ZOOM_FACTOR+16, debug_text, 1);

    for(int i=1;i<N;++i)
    {
      snprintf(debug_text, sizeof(debug_text),
                       "%3d, %3d, %3d, %3d\n"
                       ,strip_bounds[i*4+0],strip_bounds[i*4+1],strip_bounds[i*4+2],strip_bounds[i*4+3]);

      plotString(pixels, 16, 32*FRAME_ZOOM_FACTOR+64+16*i, debug_text, 1);

      strip_bounds[i*4+0]=RAM_6502[0x05+i*4+0];
      strip_bounds[i*4+1]=RAM_6502[0x05+i*4+1];
      strip_bounds[i*4+2]=RAM_6502[0x05+i*4+2];
      strip_bounds[i*4+3]=RAM_6502[0x05+i*4+3];
    }
  }
#endif
}

extern bool_t extern_write_frame;


static void generateAllTables()
{
}

static void init()
{
  generateAllTables();
}


int main(int argc, char* argv[])
{
  printf("Compiled on " __DATE__ " at " __TIME__ ".\n");

#ifdef __GNUC__
# ifdef __GNUC_MINOR__
#   ifdef __GNUC_PATCHLEVEL__
  printf("Compiled with GCC %d.%d.%d\n", __GNUC__, __GNUC_MINOR__, __GNUC_PATCHLEVEL__);
#   endif
# endif
#endif

  printf("Compiled against SDL version %d.%d.%d\n", SDL_MAJOR_VERSION, SDL_MINOR_VERSION, SDL_PATCHLEVEL);

  const SDL_version* sdl_version = SDL_Linked_Version();

  if(sdl_version)
    printf("Running dynamically-linked SDL version %d.%d.%d\n", sdl_version->major, sdl_version->minor, sdl_version->patch);
  else
    printf("SDL_Linked_Version() returned NULL.\n");

  if(SDL_Init(SDL_INIT_TIMER | SDL_INIT_VIDEO))
      return -1;

  SDL_WM_SetCaption("", NULL);
  
  atexit(SDL_Quit);
  
  SDL_Surface* surface = SDL_SetVideoMode(FRAME_WIDTH, FRAME_HEIGHT, 16, SDL_HWSURFACE | SDL_DOUBLEBUF | SDL_ASYNCBLIT | SDL_ANYFORMAT);

  if(!surface)
      return -1;

  init();

  sysInitialise();

  initRenderstate(&g_rs);

  while(keep_going == 1)
  {
      SDL_Event event;

      while(SDL_PollEvent(&event))
      {
          if(event.type == SDL_QUIT)
          {
              keep_going = 0;
          }
      }

      frame_time = SDL_GetTicks() - g_gs.start_level_time;

      while(frame_time >= g_gs.next_frame)
      {
          if(!sysUpdateInput())
            return 0;

          Uint8 *key = SDL_GetKeyState(NULL);

          if(g_game_mode == MODE_INGAME)
          {
            int16_t speed=1;

            if(key[SDLK_LEFT])
              g_cam_x=MAX(-64,MIN(63,(int)g_cam_x-speed));
            if(key[SDLK_RIGHT])
              g_cam_x=MAX(-64,MIN(63,(int)g_cam_x+speed));
            if(key[SDLK_UP])
              g_cam_y=MAX(-64,MIN(63,(int)g_cam_y-speed));
            if(key[SDLK_DOWN])
              g_cam_y=MAX(-64,MIN(63,(int)g_cam_y+speed));

            if(key['a'])
            {
              if(g_cam_speed > top_speed-8)
                g_cam_speed=top_speed;
              else
                g_cam_speed+=8;
            }

            if(key['z'])
            {
              if(g_cam_speed < 32)
                g_cam_speed=0;
              else
                g_cam_speed-=32;
            }

            uint16_t prev_g_cam_sub_z=g_cam_sub_z;

            g_cam_sub_z+=g_cam_speed;

            while(g_cam_sub_z>=84*256)
            {
              // Check for collisions
              if( g_cam_x<*(int8_t*)&ROM_6502[0xC200-0x8000+0*4] || 
                  g_cam_x>*(int8_t*)&ROM_6502[0xC201-0x8000+0*4] || 
                  g_cam_y<*(int8_t*)&ROM_6502[0xC202-0x8000+0*4] || 
                  g_cam_y>*(int8_t*)&ROM_6502[0xC203-0x8000+0*4])
              {
                g_game_mode=MODE_CRASHED;
                g_cam_sub_z=prev_g_cam_sub_z;
              }
              else
              {
                if((g_rect_offset+0)%64==0)
                {
                  if(g_lap_count==g_laps)
                    g_game_mode=MODE_WON;
                  else
                  {
                    g_time_remaining+=20;
                    g_lap_count+=1;
                  }
                }
                g_rect_offset+=1;
                g_cam_sub_z-=84*256;
              }
            }

            ++frame_count;

            if(frame_count%60==0)
              --g_time_remaining;

            if(g_time_remaining==0)
              g_game_mode=MODE_TIMEOVER;
          }

          g_gs.next_frame += 15;
      }
      
      SDL_LockSurface(surface);

      uint16_t* pixels = (uint16_t*)surface->pixels;

      drawIngameScreen(pixels, &g_rs, &g_gs);

      SDL_UnlockSurface(surface);

      SDL_UpdateRect(surface, 0, 0, 0, 0);
      SDL_Flip(surface);

      SDL_Delay(5);

  }

  return 0;
}
