  ; acme.exe --cpu 6502 -o main.bin
  * = $CB00

  ; SFX indices
  SFX_CRASH     =  1 ; *
  SFX_PICKUP    =  2 ; *
  SFX_CURSOR    =  3
  SFX_TIMEOVER  =  4 ; *
  SFX_5SEC      =  5 ; *
  SFX_4SEC      =  6 ; *
  SFX_3SEC      =  7 ; *
  SFX_2SEC      =  8 ; *
  SFX_1SEC      =  9 ; *
  SFX_RACESTART = 10 ; *
  SFX_NEWLAP    = 11 ; *
  SFX_FINISH    = 12 ; *
  SFX_PERFECT   = 13 ; *
  SFX_PAUSE     = 14 ; *
  SFX_UNPAUSE   = 15 ; *

  ; Constants
  WINDOW_YMIN = 48
  WINDOW_YMAX = 224

  NUM_LEVELS  = 6

  X_SCROLL_OFFSET = 31

  JOYPAD1 = $4016
  JOYPAD2 = $4017

  TOP_SPEED     = $0800
  ACCELERATION  = 8
  DECELERATION  = 32
  MAX_LAPS      = 3

  GAME_ENABLE_MOVEMENT        = 1<<0
  GAME_ENABLE_COUNTER_UPDATE  = 1<<1
  GAME_ENABLE_STRIP_UPDATE    = 1<<2
  GAME_ENABLE_STATIC_SPRITES  = 1<<3
  GAME_ENABLE_PAUSING         = 1<<4

  INTERMISSION_MODE_GETREADY  = 0
  INTERMISSION_MODE_TRYAGAIN  = 1
  INTERMISSION_MODE_NEXTLEVEL = 2

  INTERMISSION_COLOUR_0 = $36
  INTERMISSION_COLOUR_1 = $25
  INTERMISSION_COLOUR_2 = $14
  INTERMISSION_COLOUR_3 = $03

  ; ROM
  ENDING_NAMETABLE       = $EC00
  INFO_SPRITES          = $8400
  ;SPRITES_TIMEOVER      = $9000
  ;SPRITES_CRASHED       = $9100
  ;SPRITES_WON           = $9200
  HUD_NAMETABLE         = $BF00
  SPRITES_PAUSED        = $9400
  PICKUPSPRITES_ROUTINE = $A100
  STARS_NAMETABLES      = $AF00
  LOGZ_TO_PICKUPFRAME   = $A500
  LEVEL_METADATA_ARRAY  = $A600
  LOGO_SCROLL_IRQ       = $A700
  titlescreen           = $B700
  LOG_TABLE             = $C000
  EXP_TABLE             = $C100
  LOG_Z_TABLE           = $C200
  RECT_ARRAY_STORE      = $C300
  MAIN2                 = $CB00

  ; FamiTone2
  FamiToneInit          = $E000
  FamiToneSfxInit       = $E3DE
  FamiToneUpdate        = $E124
  FamiToneMusicPlay     = $E093
  FamiToneMusicStop     = $E0B7
  FamiToneMusicPause    = $E100
  FamiToneSfxPlay       = $E418
  FamiToneSamplePlay    = $E195
  FT_SFX_STRUCT_SIZE	= 15
  FT_SFX_CH0			= FT_SFX_STRUCT_SIZE*0
  FT_SFX_CH1			= FT_SFX_STRUCT_SIZE*1
  FT_SFX_CH2			= FT_SFX_STRUCT_SIZE*2
  FT_SFX_CH3			= FT_SFX_STRUCT_SIZE*3


  ; RAM
  ; Zeropage (first 32 bytes are for variables, rest is reserved)
  RECT_ARRAY_INDEX  = $00
  CAM_X             = $01
  CAM_Y             = $02
  LOG_Z             = $03
  TEMP              = $04
  STRIP_BOUNDS      = $05 ; Bounds are inclusive-exclusive.
  Z                 = $15
  COUNT             = $16
  SCANLINE          = $17
  FLIP_POINT_COARSE = $18
  FLIP_POINT_FINE   = $19
  ; The following variable is safely overlaid with the nametable staging area
  RECT_ARRAY_BASE   = $20 ; 2B

  ; Variables overlaid with the nametable staging area
  PICKUP_XPROJ      = $20
  PICKUP_YPROJ      = $21
  PICKUP_FRAME      = $22
  PICKUP_BOUNDS     = $25 ; 4B

  ; PPU VRAM addresses
  PPU_VRAM_TIME_DIGIT_1       = $2090+(WINDOW_YMIN/8-3)*32 ; Big version
  PPU_VRAM_TIME_DIGIT_2       = $2092+(WINDOW_YMIN/8-3)*32
  PPU_VRAM_LAPS_DIGIT_1       = $2086+(WINDOW_YMIN/8-2)*32
  PPU_VRAM_SPEED_INDICATOR_1  = $2097+(WINDOW_YMIN/8-2+0)*32
  PPU_VRAM_SPEED_INDICATOR_2  = $2097+(WINDOW_YMIN/8-2+1)*32
  PPU_VRAM_BONUSBALLS         = $2085+(WINDOW_YMIN/8-4)*32

  ; Non-zeropage ingame transient variables (they are allowed to be slow)
  FRAME_COUNT               = $0550
  BUTTONS                   = $0551 ; A, B, Select, Start, Up, Down, Left, Right
  RECT_ARRAY_INDEX_BASE     = $0552
  COLOUR_OFFSET             = $0553
  CAM_SUB_Z                 = $0554 ; 2B
  CAM_SPEED                 = $0556 ; 2B
  COUNTER_FRAMES            = $0558
  COUNTER_UNITS             = $0559
  COUNTER_TENS              = $055A
  GAME_ENABLES              = $055B
  ;STATIC_SPRITES_HIGH       = $055C
  TEMP_1                    = $055D
  LAP_COUNT                 = $055E
  PREV_BUTTONS              = $055F
  GAME_PAUSED               = $0560
  TEMP_2                    = $0561
  TEMP_3                    = $0562
  SCROLLPOINTS_END          = $0563
  SCROLLPOINTS_END_PENDING  = $0564
  LAYER_PALETTE_TEMP        = $0565 ; 3B
  LAYER_PALETTE             = $0568 ; 8B
  CAM_Z                     = $0570
  IN_SIGHTS_COUNT           = $0571
  BONUSBALLS_PENDING        = $0573
  BONUSBALL_TOWRITE         = $0574
  PENDING_SFX               = $0575

  LEVEL_METADATA            = $0580 ; 32B

  ; Non-zeropage persistent variables
  INTERMISSION_MODE     = $05F0
  LEVEL_INDEX           = $05F1
  BONUSBALLS            = $05F2
  PERFECT_COUNT         = $05F3

  NMI_COUNTER           = $05FF

  SPRITES_STAGING       = $0200
  RECT_ARRAY            = $0700

  ; $20 - $7F : Nametable staging
  ; $80 - $D0 : Scrollpoints array

  ; Initialise FamiTone2

	ldx #$00
	ldy #$85
	lda #1 ; NTSC
	jsr FamiToneInit
	ldx #$00
	ldy #$99
	lda #1 ; NTSC
	jsr FamiToneSfxInit

	lda #3
	jsr FamiToneMusicPlay

  jsr titlescreen

  lda #0
  sta LEVEL_INDEX

  ; -------------- Intermission screen --------------

intermission

  lda INTERMISSION_MODE
  cmp #INTERMISSION_MODE_NEXTLEVEL
  bne +
  ; Count a perfect level
  lda #$FF
  cmp BONUSBALLS
  bne +
  inc PERFECT_COUNT
+

  ; Detect final level completed
  lda #NUM_LEVELS
  cmp LEVEL_INDEX
  bne +
  jsr endingscreen
  jsr titlescreen
  lda #0
  sta LEVEL_INDEX
  jmp intermission
+

  ; MMC3: Set mirroring to horizontal
  lda #$01
  sta $A000

  sei ; Disable interrupts

  ; Clear PPUADDR latch by reading PPUSTATUS
  bit $2002

  lda #$00
  sta $2001 ; Disable rendering

  ; Wait for vertical blank.
;- bit $2002
;  bpl -

  ; Set the sprites palette
  lda #$3F
  sta $2006
  lda #$10
  sta $2006

  ldx #$00
- lda intermission_sprites_palette,x
  sta $2007
  inx
  cpx #$08
  bne -

  ; Reset PPUSCROLL
  bit $2002
  lda #$00
  sta $2005 ; X scroll
  lda #$00
  sta $2005 ; Y scroll

  ; BG and sprites must use different pattern tables.
  ;     VPHBSINN
  ldx #%10010000
  stx $2000 ; PPUCTRL

  ; Show nothing for one frame.
  ldx #%00000000
  stx $2001 ; PPUMASK

  ; Clear the zeropage and sprites staging area
  ldx #$00
  txa
- sta $00,x
  sta SPRITES_STAGING,x
  inx
  bne -

  ; Clear the ingame variables
  ldx #$30
- dex
  sta $0550,x
  bne -

  ; Set up stars background nametables
  lda #$20
  sta $2006
  lda #$00
  sta $2006
  ldx #$00
!for i, 0, 3 {
- lda STARS_NAMETABLES+i*256,x
  sta $2007
  inx
  bne -
}

  lda #$28
  sta $2006
  lda #$00
  sta $2006
  ldx #$00
!for i, 4, 7 {
- lda STARS_NAMETABLES+i*256,x
  sta $2007
  inx
  bne -
}

  lda INTERMISSION_MODE
  cmp #INTERMISSION_MODE_GETREADY
  bne +
  ldx #0
  jsr init_music
+

  jsr read_joypad
  lda BUTTONS
  sta PREV_BUTTONS

  lda INTERMISSION_MODE
  cmp #INTERMISSION_MODE_NEXTLEVEL
  bne +
  lda #$FF
  cmp BONUSBALLS
  bne +
  lda #SFX_PERFECT
  sta PENDING_SFX
+

intermission_loop

  jsr update_music

  ; Set up text sprites in staging area

  lda INTERMISSION_MODE

  ldx #$00

  cmp #INTERMISSION_MODE_NEXTLEVEL
  beq +
  cmp #INTERMISSION_MODE_GETREADY
  beq ++

  lda #$00
  sta TEMP
  lda #106
  sta TEMP_1 ; Y
  lda #100
  sta TEMP_2 ; X
  lda #(string_1_end-string_1_begin)
  sta TEMP_3 ; N
  ldy #(string_1_begin-STRINGS)
  jsr make_string_sprites

  jmp +++

+

  lda #$FF
  cmp BONUSBALLS
  bne not_perfect

perfect
  lda #$01
  sta TEMP
  lda #106
  sta TEMP_1 ; Y
  lda #104
  sta TEMP_2 ; X
  lda #(string_7_end-string_7_begin)
  sta TEMP_3 ; N
  ldy #(string_7_begin-STRINGS)
  jsr make_string_sprites

  ; Animate
  lda FRAME_COUNT
  lsr
  and #3
  asl
  asl
  tay
  lda #%00000000
  sta SPRITES_STAGING+2,y
  sta SPRITES_STAGING+2+16,y

  jmp +++

not_perfect
  lda #$00
  sta TEMP
  lda #106
  sta TEMP_1 ; Y
  lda #100
  sta TEMP_2 ; X
  lda #(string_3_end-string_3_begin)
  sta TEMP_3 ; N
  ldy #(string_3_begin-STRINGS)
  jsr make_string_sprites
  jmp +++

++
  lda #$00
  sta TEMP
  lda #106
  sta TEMP_1 ; Y
  lda #100
  sta TEMP_2 ; X
  lda #(string_6_end-string_6_begin)
  sta TEMP_3 ; N
  ldy #(string_6_begin-STRINGS)
  jsr make_string_sprites

+++

  lda #$00
  sta TEMP
  lda #(115+8*2+3+6)
  sta TEMP_1 ; Y
  lda #100
  sta TEMP_2 ; X
  lda #(string_2_end-string_2_begin)
  sta TEMP_3 ; N
  ldy #(string_2_begin-STRINGS)
  jsr make_string_sprites

  lda #$00
  sta TEMP
  lda #(115+8*1+6)
  sta TEMP_1 ; Y
  lda #104
  sta TEMP_2 ; X
  lda #(string_5_end-string_5_begin)
  sta TEMP_3 ; N
  ldy #(string_5_begin-STRINGS)
  jsr make_string_sprites

  lda LEVEL_INDEX
  clc
  adc #17
  sta SPRITES_STAGING+1,x ; Tile
  inx
  lda TEMP_1
  sta SPRITES_STAGING-1,x ; Y
  inx
  lda #%00000000
  sta SPRITES_STAGING,x ; Attributes
  inx
  lda TEMP_2
  clc
  adc #8
  sta SPRITES_STAGING,x ; X
  inx

  ; Wait for NMI
  lda NMI_COUNTER
- cmp NMI_COUNTER
  beq -

  ; ----------------------------------------------------------
  ; V-blank

  sta $E000 ; MMC3 IRQ acknowledge + disable

  ; -------------- Joypad input --------------

  lda BUTTONS
  sta PREV_BUTTONS
  jsr read_joypad

  ; Write sprite data to OAM
  lda #$00
  sta $2003 ; OAMADDR
  lda #(SPRITES_STAGING>>8)
  sta $4014 ; OAMDMA
  ; DMA wait..

  ; -------------- Handle inputs --------------

  ; Start
  lda #(1<<4)
  bit BUTTONS
  beq +
  lda #(1<<4)
  bit PREV_BUTTONS
  bne +
  jmp gamestart
+

  ; MMC3: Set CHR banks
  lda #4
  ldx #$00
  stx $8000 ; Select R0
  sta $8001 ; Set CHR bank 0 at $0000-$07FF
  lda #6
  inx       ; X = 1
  stx $8000 ; Select R1
  sta $8001 ; Set CHR bank 2 at $0800-$0FFF

  lda FRAME_COUNT
  lsr
  lsr
  clc
  and #3
  ora #8
  ldx #2    ; X = 2
  stx $8000 ; Select R2
  sta $8001 ; Set CHR bank at $1000-$13FF
  clc
  adc #1
  and #3
  ora #8
  inx       ; X = 3
  stx $8000 ; Select R3
  sta $8001 ; Set CHR bank at $1400-$17FF
  adc #1
  and #3
  ora #8
  inx       ; X = 4
  stx $8000 ; Select R4
  sta $8001 ; Set CHR bank at $1800-$1BFF
  adc #1
  and #3
  ora #8
  inx       ; X = 5
  stx $8000 ; Select R5
  sta $8001 ; Set CHR bank at $1C00-$1FFF

  ; Update palette
  lda #$3F
  sta $2006
  lda #$00
  sta $2006

  ldx #$00
  lda #1
  bit FRAME_COUNT
bne +
- lda intermission_palette_0,x
  sta $2007
  inx
  cpx #$10
  bne -
  jmp ++
+
- lda intermission_palette_1,x
  sta $2007
  inx
  cpx #$10
  bne -
++
  ; Workaround for potential palette corruption bug
  lda #$3F
  sta $2006
  lda #$00
  sta $2006
  sta $2006
  sta $2006

  ; BG and sprites must use different pattern tables.
  lda FRAME_COUNT
  ;lsr
  ;and #1
  and #2
  ;     VPHBSINN
  ora #%10010000
  sta $2000 ; PPUCTRL

  ; Enable BG+sprites
  ldx #%00011000
  stx $2001 ; PPUMASK

  inc FRAME_COUNT

  ; Set PPUSCROLL
  bit $2002
  lda FRAME_COUNT
  eor #$FF
  sta $2005 ; X scroll
  lda #$00
  sta $2005 ; Y scroll

  jmp intermission_loop

  ; -------------- Main game --------------

gamestart

  ; MMC3: Set mirroring to horizontal
  lda #$01
  sta $A000

  sei ; Disable interrupts

  ; Clear PPUADDR latch by reading PPUSTATUS
  bit $2002

  lda #$00
  sta $2001 ; Disable rendering

  ; BG and sprites must use different pattern tables.
  ;     VPHBSINN
  ldx #%10001000
  stx $2000 ; PPUCTRL

  ; MMC3: Set CHR banks

  ; MMC3: Set CHR banks for HUD display
  ldy #$00
  sty $8000 ; Select R0
  ldy #$04
  sty $8001 ; Set CHR bank 4 at $0000-$07FF

  ldy #$01
  sty $8000 ; Select R1
  ldy #$0C
  sty $8001 ; Set CHR bank 12 at $0800-$0FFF

  ldy #4    ; Y = 4
  ldx #2    ; X = 2
  stx $8000 ; Select R2
  sty $8001 ; Set CHR bank 4 at $1000-$13FF
  iny       ; Y = 5
  inx       ; X = 3
  stx $8000 ; Select R3
  sty $8001 ; Set CHR bank 5 at $1400-$17FF
  iny       ; Y = 6
  inx       ; X = 4
  stx $8000 ; Select R4
  sty $8001 ; Set CHR bank 6 at $1800-$1BFF
  iny       ; Y = 7
  inx       ; X = 5
  stx $8000 ; Select R5
  sty $8001 ; Set CHR bank 7 at $1C00-$1FFF


  ; Set the HUD palette
  lda #$3F
  sta $2006
  lda #$04
  sta $2006

  ldx #$00
- lda hud_palette,x
  sta $2007
  inx
  cpx #12
  bne -

  ; Set the sprites palettes
  lda #$3F
  sta $2006
  lda #$10
  sta $2006

  ; Reticle
  lda #$1D
  sta $2007
  lda #$1D
  sta $2007
  lda #$00
  sta $2007
  lda #$20
  sta $2007

  ; Bonus pickup item
  ldx #$00
- lda pickup_palettes,x
  sta $2007
  inx
  cpx #12
  bne -

  ; Workaround for potential palette corruption bug
  lda #$3F
  sta $2006
  lda #$00
  sta $2006
  sta $2006
  sta $2006

  ; Set up PPU write address for nametable 0, $2000
  lda #$20
  sta $2006
  lda #$00
  sta $2006
  ; Write the first nametable row, which stays constant
  ; over all frames.
  ldx #32
  lda #$2C
- sta $2007
  dex
  bne -

  ; Set up PPU write address for HUD location in attribute table 0
  lda #$23
  sta $2006
  lda #$C8
  sta $2006
  ; Write two rows of attributes
  ldx #16
  lda #%01010101
- sta $2007
  dex
  bne -

  ; Set up PPU write address for HUD speed indicator attributes
  lda #$23
  sta $2006
  lda #$D5
  sta $2006
  lda #%10101010
  sta $2007
  sta $2007
  sta $2007

  ; Set up PPU write address for HUD bonus balls attributes
  lda #$23
  sta $2006
  lda #$C9
  sta $2006
  lda #%11111111
  sta $2007
  sta $2007
  sta $2007

  ; Copy initial HUD contents to $2080 in PPU RAM
  lda #$20
  sta $2006
  lda #$80
  sta $2006
  ; Two blank rows
  ldx #$40
  lda #$00
- sta $2007
  dex
  bne -
  ldx #$00
  ; Three rows from ROM
- lda HUD_NAMETABLE,x
  sta $2007
  inx
  cpx #$80
  bne -

  ; Wait for vertical blank.
;- bit $2002
;  bpl -

  ; Show nothing for one frame.
  ldx #%00000000
  stx $2001 ; PPUMASK

  ; Clear the zeropage and sprites staging area
  ldx #$00
  txa
- sta $00,x
  sta SPRITES_STAGING,x
  inx
  bne -

  ; Clear the ingame variables
  ldx #$30
- dex
  sta $0550,x
  bne -

  ; Copy in-game IRQ routine from ROM to RAM - assume it is exactly 80 bytes in size.
  ldx #$4F
- lda $FD00,x
  sta $0500,x
  dex
  bpl -

  ; Initialise the staging scrollpoints array
  ldx #$00
- lda #((4<<5)+0) ; Coarse Y scroll
  sta $80,x
  inx
  lda #07 ; Scanlines delta
  sta $80,x
  inx
  cpx #48
  bne -

  txa
  clc
  adc #($40-2)
  sta SCROLLPOINTS_END

  ; Initialise rect array
  lda #(RECT_ARRAY_STORE & $FF)
  sta RECT_ARRAY_BASE+0
  lda #(RECT_ARRAY_STORE>>8)
  clc
  adc LEVEL_INDEX
  sta RECT_ARRAY_BASE+1

  ldy #$00
- lda (RECT_ARRAY_BASE),y
  sta RECT_ARRAY,y
  iny
  bne -

  ; Initialise level metadata
  lda LEVEL_INDEX
  ; Multiply by 32
!for i, 1, 5 {
  asl
}
  tax
  ldy #$00
  ; Copy 32 bytes of data
- lda LEVEL_METADATA_ARRAY,x
  sta LEVEL_METADATA,y
  inx
  iny
  cpy #$20
  bne -

  ; Initialise cycling layers palette from level metadata
  ldx #$00
!for i, 0, 1 {
  lda LEVEL_METADATA+28
  sta LAYER_PALETTE,x
  inx
  lda LEVEL_METADATA+29
  sta LAYER_PALETTE,x
  inx
  lda LEVEL_METADATA+30
  sta LAYER_PALETTE,x
  inx
  lda LEVEL_METADATA+31
  sta LAYER_PALETTE,x
  inx
}

  ; Initialise timer variables
  lda #0
  sta COUNTER_TENS
  lda #0
  sta COUNTER_UNITS
  lda #59
  sta COUNTER_FRAMES

  lda #$00
  sta BONUSBALLS

  lda #INTERMISSION_MODE_TRYAGAIN
  sta INTERMISSION_MODE

  lda #(GAME_ENABLE_MOVEMENT|GAME_ENABLE_STRIP_UPDATE|GAME_ENABLE_COUNTER_UPDATE|GAME_ENABLE_PAUSING)
  sta GAME_ENABLES

  jsr read_joypad
  lda BUTTONS
  sta PREV_BUTTONS

  ; Wait for NMI
  lda NMI_COUNTER
- cmp NMI_COUNTER
  beq -

  ; Enable BG but not sprites.
  ldx #%00001000
  stx $2001 ; PPUMASK

mainloop

  ; Reset PPUSCROLL to place the HUD at the top of the screen.
  bit $2002
  lda #$00
  sta $2005 ; X scroll
  lda #$20
  sta $2005 ; Y scroll

  bit $2002
  ldx #$00
  stx $2006 ; A

  ; Enable MMC3 IRQ
  lda #(WINDOW_YMIN-1) ; Reserve the top lines for HUD and overscan
  sta $C000 ; IRQ latch
  sta $C001 ; IRQ reload
  sta $E001 ; IRQ enable

  cli ; Enable interrupts

  ; -------------------------------------------

  ; Reset IRQ self-patch
  ; The scrollpoints array (max. 64 bytes) is double-buffered.

  ; Copy from the staging scrollpoints array to the active one
  ldx #63
- lda $80,x
  sta $C0,x
  dex
  bpl -

  lda $C0 ; Scrollpoints array: Coarse scroll Y
  sta $0501
  lda #$41
  sta $FF ; Reset scrollpoints array index

  ; Restore IRQ
  lda FLIP_POINT_FINE ; Fine X
  eor #$FF
  sta $0506
  lda #$05
  sta $0508
  lda #$EA ; NOP
  sta $0524

  ; ----------------------------------------------------------
  ; Start of frame

  ; -------------- Joypad input --------------

  lda BUTTONS
  sta PREV_BUTTONS
  jsr read_joypad

  ; -------------- Handle misc. inputs --------------

  ; Start
  lda #(1<<4)
  bit BUTTONS
  beq +
  lda #(1<<4)
  bit PREV_BUTTONS
  bne +

  lda #(GAME_ENABLE_PAUSING)
  bit GAME_ENABLES
  beq +++

  ; Pause or unpause
  inc GAME_PAUSED
  lda #1
  bit GAME_PAUSED
  beq ++
  ; Pause
  lda #SFX_PAUSE
  sta PENDING_SFX
  ;lda #(SPRITES_PAUSED>>8)
  ;sta STATIC_SPRITES_HIGH
  ldx #$00
  lda #$00
- sta SPRITES_STAGING,x
  inx
  bne -
- lda sprites_paused,x
  sta SPRITES_STAGING+$10,x
  inx
  cpx #(sprites_paused_end-sprites_paused)
  bne -
  lda #(GAME_ENABLE_STATIC_SPRITES|GAME_ENABLE_PAUSING)
  sta GAME_ENABLES
  jmp +
++
  ; Unpause
  lda #SFX_UNPAUSE
  sta PENDING_SFX
  lda #(GAME_ENABLE_MOVEMENT|GAME_ENABLE_STRIP_UPDATE|GAME_ENABLE_COUNTER_UPDATE|GAME_ENABLE_PAUSING)
  sta GAME_ENABLES

  jmp +
+++
  jmp intermission
+

  lda #1
  bit GAME_PAUSED
  beq +
  ; Select
  lda #(1<<5)
  bit BUTTONS
  beq +
  lda #(1<<5)
  bit PREV_BUTTONS
  bne +
  ; Reset the level
  ;lda #6
  ;sta LEVEL_INDEX
  jmp intermission
+

  ; -------------- Music --------------

  jsr update_music

  ; -------------- Timer countdown --------------

  lda LAP_COUNT
  beq ++

  lda #GAME_ENABLE_COUNTER_UPDATE
  bit GAME_ENABLES
  beq ++

  dec COUNTER_FRAMES
  lda #$FF
  cmp COUNTER_FRAMES
  bne +
  ; Tick down one second
  lda #59
  sta COUNTER_FRAMES
  dec COUNTER_UNITS
  lda COUNTER_TENS
  bne +++
  jsr sound_alarm
+++
  lda #$FF
  cmp COUNTER_UNITS
  bne +
  lda #9
  sta COUNTER_UNITS
  dec COUNTER_TENS
+

  ; Check to see if time has run out
  lda COUNTER_UNITS
  ora COUNTER_TENS
  bne ++
  ; Time over!
  lda #SFX_TIMEOVER
  sta PENDING_SFX
  ; Set the game mode to display TIME OVER
  ;lda #(SPRITES_TIMEOVER>>8)
  ;sta STATIC_SPRITES_HIGH
  ldx #$00
  lda #$00
- sta SPRITES_STAGING,x
  inx
  bne -
- lda sprites_timeover,x
  sta SPRITES_STAGING+$10,x
  inx
  cpx #(sprites_timeover_end-sprites_timeover)
  bne -
  lda #(GAME_ENABLE_STATIC_SPRITES)
  sta GAME_ENABLES
++

  ; -------------- Handle game inputs --------------

  lda #GAME_ENABLE_MOVEMENT
  bit GAME_ENABLES
  bne +
  jmp movement_done
+

  ; Right
  lda #(1<<0)
  bit BUTTONS
  beq +
  lda #$70
  cmp CAM_X
  beq +
  inc CAM_X
+

  ; Left
  lda #(1<<1)
  bit BUTTONS
  beq +
  lda #$91
  cmp CAM_X
  beq +
  dec CAM_X
+

  ; Down
  lda #(1<<2)
  bit BUTTONS
  beq +
  lda #$70
  cmp CAM_Y
  beq +
  inc CAM_Y
+

  ; Up
  lda #(1<<3)
  bit BUTTONS
  beq +
  lda #$91
  cmp CAM_Y
  beq +
  dec CAM_Y
+

  ; -------------- Forward movement --------------

  ; A
  lda #(1<<7)
  bit BUTTONS
  beq ++
  ; Increase speed
  clc
  lda CAM_SPEED+0
  adc #(ACCELERATION & $FF)
  sta CAM_SPEED+0
  lda CAM_SPEED+1
  adc #(ACCELERATION >> 8)
  sta CAM_SPEED+1
  cmp #(TOP_SPEED >> 8)
  bne ++
  lda #$FF
  sta CAM_SPEED+0
  lda #((TOP_SPEED >> 8) - 1)
  sta CAM_SPEED+1
++

  ; B
  lda #(1<<6)
  bit BUTTONS
  beq ++
  ; Decrease speed
  sec
  lda CAM_SPEED+0
  sbc #(DECELERATION & $FF)
  sta CAM_SPEED+0
  lda CAM_SPEED+1
  sbc #(DECELERATION >> 8)
  sta CAM_SPEED+1
  cmp #$FF
  bne ++
  lda #$00
  sta CAM_SPEED+0
  sta CAM_SPEED+1
++

  ; Update camera Z position
  clc
  lda CAM_SUB_Z+0
  adc CAM_SPEED+0
  sta CAM_SUB_Z+0
  lda CAM_SUB_Z+1
  sta TEMP_1
  adc CAM_SPEED+1
  sta CAM_SUB_Z+1
  cmp #84
  bcc ++

  ; Collision detection
  lda CAM_X
  ldx RECT_ARRAY_INDEX_BASE
  cmp RECT_ARRAY,x ; min_x
  bmi crash
  inx
  cmp RECT_ARRAY,x ; max_x
  bpl crash
  inx
  lda CAM_Y
  cmp RECT_ARRAY,x ; min_y
  bmi crash
  inx
  cmp RECT_ARRAY,x ; max_y
  bpl crash
  inx
  jmp no_crash

crash
  ; A collision occurred
  jsr crash_occurred
  jmp ++

no_crash
  ; Check to see if a lap has been completed
  lda RECT_ARRAY_INDEX_BASE
  bne +
  ; Completed a lap
  lda #MAX_LAPS
  cmp LAP_COUNT
  bne +++
  ; All laps done
  lda #SFX_FINISH
  sta PENDING_SFX
  jsr set_nextlevel
  jmp ++
+++
  ; Count a lap
  lda #SFX_NEWLAP
  sta PENDING_SFX
  jsr bump_lap_timers
  inc LAP_COUNT
  lda LAP_COUNT
  cmp #1
  bne ++++
  ; The first lap has a special SFX.
  lda #SFX_RACESTART
  sta PENDING_SFX
++++
+
  ; Proceed
  lda CAM_SUB_Z+1
  sec
  sbc #84
  sta CAM_SUB_Z+1
  inc RECT_ARRAY_INDEX_BASE
  inc RECT_ARRAY_INDEX_BASE
  inc RECT_ARRAY_INDEX_BASE
  inc RECT_ARRAY_INDEX_BASE
  inc COLOUR_OFFSET
  lda CAM_Z
  clc
  adc #1
  and #63
  sta CAM_Z
++

movement_done

  ; ----------------------------------------------------------
  ; 3D projection and rectangle clipping goes here.
  ; During this phase there must be no use of the Y register
  ; or any instructions lasting for more than 6 cycles.

  ; Start generating data for the next frame.

  lda RECT_ARRAY_INDEX_BASE
  sta RECT_ARRAY_INDEX

  lda #84
  sec
  sbc CAM_SUB_Z+1
  sta Z

  ; -------------- Projection and clipping --------------

  lda #$08
  sta FLIP_POINT_COARSE

!for i, 1, 3 {

  ; Fetch Log(Z)
  ldx Z
  lda LOG_Z_TABLE,x
  sta LOG_Z

  ; Load L
  ldx RECT_ARRAY_INDEX
  lda RECT_ARRAY,x
  inc RECT_ARRAY_INDEX

  ; Subtract camera X
  sec
  sbc CAM_X
  beq +
  sta TEMP

  ; Exp(Log(X)-Log(Z))
  tax
  lda LOG_TABLE,x
  sec
  sbc LOG_Z
  tax
  lda EXP_TABLE,x
  
  bit TEMP
  bpl +
  ; Negate A
  eor #$FF
  clc
  adc #1
+ clc
  adc #(128+4)

!if i > 1 {
  ; Clip against outer rectangle
  cmp STRIP_BOUNDS+(i-1)*4+0 ; min_x
  bcs +
  lda STRIP_BOUNDS+(i-1)*4+0 ; min_x
}
+ sta STRIP_BOUNDS+i*4+0 ; min_x

  ; Load R
  ldx RECT_ARRAY_INDEX
  lda RECT_ARRAY,x
  inc RECT_ARRAY_INDEX

  ; Subtract camera X
  sec
  sbc CAM_X
  beq +
  sta TEMP

  ; Exp(Log(X)-Log(Z))
  tax
  lda LOG_TABLE,x
  sec
  sbc LOG_Z
  tax
  lda EXP_TABLE,x
  
  bit TEMP
  bpl +
  ; Negate A
  eor #$FF
  clc
  adc #1
+ clc
  adc #(128+4)

!if i > 1 {
  ; Clip against outer rectangle
  cmp STRIP_BOUNDS+(i-1)*4+1 ; max_x
  bcc +
  lda STRIP_BOUNDS+(i-1)*4+1 ; max_x
}
+ sta STRIP_BOUNDS+i*4+1 ; max_x

  lda STRIP_BOUNDS+i*4+0 ; min_x
  cmp STRIP_BOUNDS+i*4+1 ; max_x
  bcs +
  sta FLIP_POINT_COARSE ; Temporary
+

  ; --- Upper ---

  ; Load T
  ldx RECT_ARRAY_INDEX
  lda RECT_ARRAY,x
  inc RECT_ARRAY_INDEX

  ; Subtract camera Y
  sec
  sbc CAM_Y
  beq +
  sta TEMP

  ; Exp(Log(Y)-Log(Z))
  tax
  lda LOG_TABLE,x
  sec
  sbc LOG_Z
  tax
  lda EXP_TABLE,x
  
  bit TEMP
  bpl +
  ; Negate A
  eor #$FF
  clc
  adc #1
+ clc
  adc #128

!if i > 1 {
  ; Clip against outer rectangle
  cmp STRIP_BOUNDS+(i-1)*4+3 ; max_y
  bcc +
  lda STRIP_BOUNDS+(i-1)*4+3 ; max_y
} else {
  ; Clip against the window
  cmp #WINDOW_YMAX
  bcc +
  lda #WINDOW_YMAX
}
+ sta STRIP_BOUNDS+i*4+2 ; min_y

  ; --- Lower ---

  ; Load B
  ldx RECT_ARRAY_INDEX
  lda RECT_ARRAY,x
  inc RECT_ARRAY_INDEX

  ; Subtract camera Y
  sec
  sbc CAM_Y
  beq +
  sta TEMP

  ; Exp(Log(Y)-Log(Z))
  tax
  lda LOG_TABLE,x
  sec
  sbc LOG_Z
  tax
  lda EXP_TABLE,x
  
  bit TEMP
  bpl +
  ; Negate A
  eor #$FF
  clc
  adc #1
+ clc
  adc #128

!if i > 1 {
  ; Clip against outer rectangle
  cmp STRIP_BOUNDS+(i-1)*4+3 ; max_y
  bcc +
  lda STRIP_BOUNDS+(i-1)*4+3 ; max_y
} else {
  ; Clip against the window
  cmp #WINDOW_YMAX
  bcc +
  lda #WINDOW_YMAX
}
+ sta STRIP_BOUNDS+i*4+3 ; max_y

!if i < 3 {
  lda #84
  clc
  adc Z
  sta Z
}
}

  ; -------------- Scrollpoints --------------

  ldx #$00

  ; Insert a dummy scanline to hide garbage caused by bankswitch
  ; at split between HUD and ingame-view.
  lda #(4<<5) ; Coarse Y scroll
  sta $80,x
  inx
  lda #1 ; Scanlines delta
  sta $80,x
  inx

  lda #WINDOW_YMIN
  sta SCANLINE

!for i, 1, 3 {

  lda STRIP_BOUNDS+i*4+2 ; min_y
  sec
  sbc SCANLINE

  bcc +++

  sta TEMP
  ; Divide by 8
  lsr
  lsr
  lsr
  beq ++

  ; Middle scrollpoints (if any)
  sta COUNT
- lda #(((i-1)<<5)+X_SCROLL_OFFSET) ; Coarse Y scroll
  sta $80,x
  inx
  lda #7 ; Scanlines delta
  sta $80,x
  inx
  dec COUNT
  bne -

++
  ; End scrollpoint
  lda #(((i-1)<<5)+X_SCROLL_OFFSET) ; Coarse Y scroll
  sta $80,x
  lda TEMP
  and #7
  beq +
  inx
  sec
  sbc #1 ; Scanlines delta
  sta $80,x
  inx
+
  lda STRIP_BOUNDS+i*4+2 ; min_y
  sta SCANLINE
+++
}

  ; --- Lower ---

!for i, 3, 0 {

!if i = 0 {
  lda #WINDOW_YMAX
} else {
  lda STRIP_BOUNDS+i*4+3 ; max_y
}
  sec
  sbc SCANLINE

  bcc +++

  sta TEMP
  ; Divide by 8
  lsr
  lsr
  lsr
  beq ++

  ; Middle scrollpoints (if any)
  sta COUNT
- lda #((i<<5)+X_SCROLL_OFFSET) ; Coarse Y scroll
  sta $80,x
  inx
  lda #7 ; Scanlines delta
  sta $80,x
  inx
  dec COUNT
  bne -

++
  ; End scrollpoint
  lda #((i<<5)+X_SCROLL_OFFSET) ; Coarse Y scroll
  sta $80,x
  lda TEMP
  and #7
  beq +
  inx
  sec
  sbc #1 ; Scanlines delta
  sta $80,x
  inx
+
  lda STRIP_BOUNDS+i*4+3 ; max_y
  sta SCANLINE
+++
}

  ; Show the HUD for the remaining (16) scanlines
  lda #((4<<5)+0) ; Coarse Y scroll
  sta $80,x
  inx
  lda #09 ; Scanlines delta
  sta $80,x
  inx
  stx SCROLLPOINTS_END_PENDING

  lda #((4<<5)+0) ; Coarse Y scroll
  sta $80,x
  inx
  lda #$20 ; Scanlines delta (more than necessary because no further IRQs are needed)
  sta $80,x
  inx

  ; -------------- Sprites staging --------------

  lda #GAME_ENABLE_STATIC_SPRITES
  bit GAME_ENABLES
  bne +
  jsr do_dynamic_sprites
+

  ; Guide reticle sprite

  lda #$80
  ora BONUSBALL_TOWRITE
  sta BONUSBALL_TOWRITE

  lda #%00000000
  sta TEMP

  ; Check sights
  lda #2
  cmp IN_SIGHTS_COUNT
  bne +
  ; Check for collision
  lda BONUSBALLS
  eor BONUSBALLS_PENDING
  beq ++
  ; A bonusball has been acquired
  lda #SFX_PICKUP
  sta PENDING_SFX
  lda BONUSBALLS_PENDING
  sta BONUSBALLS
  lda BONUSBALL_TOWRITE
  and #$0F
  sta BONUSBALL_TOWRITE
++
  lda #%11000000
  sta TEMP
+

  ; Top-left
  ldx #$00
  lda #(32+(240-32-16)/2-8)
  sta SPRITES_STAGING,x ; Y
  inx
  lda #$FE
  sta SPRITES_STAGING,x ; Tile index
  inx
  lda #%00000000
  eor TEMP
  sta SPRITES_STAGING,x ; Attributes
  inx
  lda #126
  sta SPRITES_STAGING,x ; X
  inx
  ; Top-right
  lda #(32+(240-32-16)/2-8)
  sta SPRITES_STAGING,x ; Y
  inx
  lda #$FE
  sta SPRITES_STAGING,x ; Tile index
  inx
  lda #%01000000
  eor TEMP
  sta SPRITES_STAGING,x ; Attributes
  inx
  lda #(126+8)
  sta SPRITES_STAGING,x ; X
  inx
  ; Bottom-left
  lda #(32+(240-32-16)/2)
  sta SPRITES_STAGING,x ; Y
  inx
  lda #$FE
  sta SPRITES_STAGING,x ; Tile index
  inx
  lda #%10000000
  eor TEMP
  sta SPRITES_STAGING,x ; Attributes
  inx
  lda #(126)
  sta SPRITES_STAGING,x ; X
  inx
  ; Bottom-right
  lda #(32+(240-32-16)/2)
  sta SPRITES_STAGING,x ; Y
  inx
  lda #$FE
  sta SPRITES_STAGING,x ; Tile index
  inx
  lda #%11000000
  eor TEMP
  sta SPRITES_STAGING,x ; Attributes
  inx
  lda #(126+8)
  sta SPRITES_STAGING,x ; X
  inx

  ; -------------- Nametables staging --------------

  lda FLIP_POINT_COARSE
  and #7
  sta FLIP_POINT_FINE
  lda FLIP_POINT_COARSE
  lsr
  lsr
  lsr
  sta FLIP_POINT_COARSE

  ; Nametable-generating routine is in RAM because it uses self-modifying code.
  jsr $0300

  inc FRAME_COUNT

  ; -------------- "Early V-Blank" preparation --------------
  ; Wait for second-to-last scrollpoint to hit
- lda $FF
  cmp SCROLLPOINTS_END
  bcc -

  ; Patch the IRQ handler so that it turned rendering off in an appropriate cycle window.
  lda #$00
  sta $0506
  lda #$01
  sta $0508

  inc SCROLLPOINTS_END
  inc SCROLLPOINTS_END

  ; Wait for the final scrollpoint to hit
- lda $FF
  cmp SCROLLPOINTS_END
  bcc -

  ; Rendering is now safely turned off.

  lda SCROLLPOINTS_END_PENDING
  clc
  adc #$40
  sta SCROLLPOINTS_END ; From previous frame

  ;
  ; CutterCross:
  ;
  ;   The initial render disable write should land during dots 319-340 of the scanline.
  ;
  ;   The behavior is CPU/PPU alignment dependent, but the worst-case alignment only has
  ;   a safe zone for dots 319-340 of a scanline for the render disable write.
  ;
  ;   [For the case where you're disabling rendering and leaving it that way until vBlank,
  ;    the corruption behavior is slightly different and there's an added safe zone of
  ;    dots 62-254, but that's not very relevant to a midframe palette swap.]
  ;
  ; https://forums.nesdev.org/viewtopic.php?t=23209
  ;
  ; CutterCross:
  ;
  ;   Disabling rendering midframe can also corrupt palette memory if the internal PPU
  ;   register v is in the ranges $3C00-$3FFF or $7C00-$7FFF when the disable write occurs.
  ;   i.e. Reading from the bottom right nametable.
  ;

  ; Turn rendering off
  ;lda #%00000000
  ;sta $2001 ; PPUMASK

  ; ----------------------------------------------------------
  ; (Early) V-Blank

  sta $E000 ; MMC3 IRQ acknowledge + disable

  ; Clear PPUADDR latch by reading PPUSTATUS
  bit $2002

  ; BG and sprites must use different pattern tables.
  ; VRAM address increment by 32 for HUD update
  ;     VPHBSINN
  ldx #%10001100 ; 8x8 sprites - 8x16 causes issues when OAM decays and MMC3 IRQ is in use.
  stx $2000 ; PPUCTRL

  ; Update HUD nametable (big numbers)

  ; First digit
  lda #(PPU_VRAM_TIME_DIGIT_1 >> 8)
  sta $2006
  lda #(PPU_VRAM_TIME_DIGIT_1 & $FF)
  sta $2006
  lda COUNTER_TENS
  ; Multiply by 6
  asl
  clc
  adc COUNTER_TENS
  asl
  ora #$80
  tax
!for i, 0, 2 {
  stx $2007
  inx
}
  lda #((PPU_VRAM_TIME_DIGIT_1+1) >> 8)
  sta $2006
  lda #((PPU_VRAM_TIME_DIGIT_1+1) & $FF)
  sta $2006
!for i, 0, 2 {
  stx $2007
  inx
}

  ; Second digit
  lda #(PPU_VRAM_TIME_DIGIT_2 >> 8)
  sta $2006
  lda #(PPU_VRAM_TIME_DIGIT_2 & $FF)
  sta $2006
  lda COUNTER_UNITS
  ; Multiply by 6
  asl
  clc
  adc COUNTER_UNITS
  asl
  ora #$80
  tax
!for i, 0, 2 {
  stx $2007
  inx
}
  lda #((PPU_VRAM_TIME_DIGIT_2+1) >> 8)
  sta $2006
  lda #((PPU_VRAM_TIME_DIGIT_2+1) & $FF)
  sta $2006
!for i, 0, 2 {
  stx $2007
  inx
}

  ; Laps counter
  lda #(PPU_VRAM_LAPS_DIGIT_1 >> 8)
  sta $2006
  lda #(PPU_VRAM_LAPS_DIGIT_1 & $FF)
  sta $2006
  lda LAP_COUNT
  asl
  clc
  adc #$60
  tax
  stx $2007
  inx
  stx $2007

  ; BG and sprites must use different pattern tables.
  ; VRAM address increment by 1 for speed indicator update
  ;     VPHBSINN
  ldx #%10001000 ; 8x8 sprites - 8x16 causes issues when OAM decays and MMC3 IRQ is in use.
  stx $2000 ; PPUCTRL

  ; Speed indicator row 1
  lda #(PPU_VRAM_SPEED_INDICATOR_1 >> 8)
  sta $2006
  lda #(PPU_VRAM_SPEED_INDICATOR_1 & $FF)
  sta $2006

  ldx #7
  ldy #0
- lda CAM_SPEED+0
  bne +
  lda CAM_SPEED+1
  beq ++++
+ lda #$80
  bit CAM_SPEED+0
  bne ++
  lda #$C1
  cpy #3
  bcc +
  lda #$C3
  cpy #6
  bcc +
  lda #$C5
  jmp +
++
  lda #$C0
  cpy #3
  bcc +
  lda #$C2
  cpy #6
  bcc +
  lda #$C4

+ cpy CAM_SPEED+1
  bcc ++
  beq +++
++++
  lda #$00
++
  and #$CE
+++
  sta $2007
  iny
  dex
  bne -


  ; Speed indicator row 2
  lda #(PPU_VRAM_SPEED_INDICATOR_2 >> 8)
  sta $2006
  lda #(PPU_VRAM_SPEED_INDICATOR_2 & $FF)
  sta $2006

  ldx #7
  ldy #0
- lda CAM_SPEED+0
  bne +
  lda CAM_SPEED+1
  beq ++++
+ lda #$80
  bit CAM_SPEED+0
  bne ++
  lda #$C7
  cpy #3
  bcc +
  lda #$C9
  cpy #6
  bcc +
  lda #$CB
  jmp +
++
  lda #$C6
  cpy #3
  bcc +
  lda #$C8
  cpy #6
  bcc +
  lda #$CA

+ cpy CAM_SPEED+1
  bcc ++
  beq +++
++++
  lda #$00
++
  and #$CE
+++
  sta $2007
  iny
  dex
  bne -


  ; Bonusballs
  bit BONUSBALL_TOWRITE
  bmi +
  lda #(PPU_VRAM_BONUSBALLS >> 8)
  sta $2006
  lda #(PPU_VRAM_BONUSBALLS & $FF)
  clc
  adc BONUSBALL_TOWRITE
  sta $2006
  lda #$CD
  sta $2007
+

  ; Sprites

  ; BG and sprites must use different pattern tables.
  ; VRAM address increment by 1 for strips update
  ;     VPHBSINN
  ldx #%10001000 ; 8x8 sprites - 8x16 causes issues when OAM decays and MMC3 IRQ is in use.
  stx $2000 ; PPUCTRL

  ; Strips nametable

  lda #GAME_ENABLE_STRIP_UPDATE
  bit GAME_ENABLES
  beq +

  ; Set up PPU write address for nametable 0, $2000+32
  lda #$20
  sta $2006
  lda #($00+32)
  sta $2006

  jsr unrolled_strips_nametable_copy

+

  ; Enable BG+sprites
  ldx #%00011000
  stx $2001 ; PPUMASK

  ; Set the BG palette
  lda #$3F
  sta $2006
  lda #$00
  sta $2006

  lda #$1D ; Backdrop (black)
  sta $2007

  ; Layers palette cycle
  lda COLOUR_OFFSET
  and #3
  eor #3
  tax
  lda LAYER_PALETTE,x
  sta LAYER_PALETTE_TEMP+0
  inx
  lda LAYER_PALETTE,x
  sta LAYER_PALETTE_TEMP+1
  inx
  lda LAYER_PALETTE,x
  sta LAYER_PALETTE_TEMP+2

  lda COLOUR_OFFSET
  sec
  adc #1
  and #63
  cmp #3
  bcs +
  tax
  lda #$20
  sta LAYER_PALETTE_TEMP,x
+

  lda LAYER_PALETTE_TEMP+0
  sta $2007
  lda LAYER_PALETTE_TEMP+1
  sta $2007
  lda LAYER_PALETTE_TEMP+2
  sta $2007

  ; Workaround for potential palette corruption bug
  lda #$3F
  sta $2006
  lda #$00
  sta $2006
  sta $2006
  sta $2006

  ; -------------------------------------------

  ; MMC3: Set CHR banks for HUD display
  ldy #$00
  sty $8000 ; Select R0
  ldy #$04
  sty $8001 ; Set CHR bank 4 at $0000-$07FF

  ldy #$01
  sty $8000 ; Select R1
  ldy #$0C
  sty $8001 ; Set CHR bank 12 at $0800-$0FFF

  ; Write sprite data to OAM
  lda #$00
  sta $2003 ; OAMADDR
  lda #(SPRITES_STAGING>>8)
  sta $4014 ; OAMDMA
  ; DMA wait..

  jmp mainloop

unrolled_strips_nametable_copy
  ; Copy the nametables by reading from fixed addresses in the
  ; zeropage. This is faster than using a "popslide".
  ; Only three rows of nametable need to be copied.
!for i, 0, 95 {
  ; Nametable staging is at $20-$7F
  lda $20+i ; 3 cycles
  sta $2007 ; 4 cycles
}
  rts

read_joypad
  ; https://www.nesdev.org/wiki/Controller_reading_code
; At the same time that we strobe bit 0, we initialize the ring counter
; so we're hitting two birds with one stone here
    lda #$01
    ; While the strobe bit is set, buttons will be continuously reloaded.
    ; This means that reading from JOYPAD1 will only return the state of the
    ; first button: button A.
    sta JOYPAD1
    sta BUTTONS
    lsr          ; now A is 0
    ; By storing 0 into JOYPAD1, the strobe bit is cleared and the reloading stops.
    ; This allows all 8 buttons (newly reloaded) to be read from JOYPAD1.
    sta JOYPAD1
readjoy_loop
    lda JOYPAD1
    lsr          ; bit 0 -> Carry
    rol BUTTONS  ; Carry -> bit 0; bit 7 -> Carry
    bcc readjoy_loop
  rts


make_string_sprites
- lda STRINGS,y
  sta SPRITES_STAGING+1,x ; Tile
  inx
  lda TEMP_1
  sta SPRITES_STAGING-1,x ; Y
  inx
  lda TEMP
  sta SPRITES_STAGING,x ; Attributes
  inx
  lda TEMP_2
  sta SPRITES_STAGING,x ; X
  inx
  clc
  adc #8
  sta TEMP_2
  lda STRINGS,y
  bne +
  dex
  dex
  dex
  dex
+ iny
  dec TEMP_3
  bne -
  rts

set_nextlevel
  lda #83
  sta CAM_SUB_Z+1
  ldx #$00
  lda #$00
- sta SPRITES_STAGING,x
  inx
  bne -
- lda sprites_won,x
  sta SPRITES_STAGING+$10,x
  inx
  cpx #(sprites_won_end-sprites_won)
  bne -
  lda #(GAME_ENABLE_STATIC_SPRITES)
  sta GAME_ENABLES
  lda #INTERMISSION_MODE_NEXTLEVEL
  sta INTERMISSION_MODE
  inc LEVEL_INDEX
  rts

bump_lap_timers
  ; Units
  lda COUNTER_UNITS
  clc
  adc LEVEL_METADATA+27 ; Units
  cmp #10
  bcc +
  sbc #10
  inc COUNTER_TENS
+ sta COUNTER_UNITS
  ; Tens
  lda COUNTER_TENS
  clc
  adc LEVEL_METADATA+26 ; Tens
  sta COUNTER_TENS
  rts

crash_occurred
  lda #SFX_CRASH
  sta PENDING_SFX
  ; Set the game mode to display CRASH!
  lda #83
  sta CAM_SUB_Z+1
  ldx #$00
  lda #$00
- sta SPRITES_STAGING,x
  inx
  bne -
- lda sprites_crashed,x
  sta SPRITES_STAGING+$10,x
  inx
  cpx #(sprites_crashed_end-sprites_crashed)
  bne -
  lda #(GAME_ENABLE_STATIC_SPRITES)
  sta GAME_ENABLES
  rts

!convtab raw
!text "by Edd Biddulph & Roope Makinen"

  ; -------------- Ending screen --------------

endingscreen

  ; MMC3: Set mirroring to vertical
  lda #$00
  sta $A000

  sei ; Disable interrupts
  stx $E000 ; Disable scanline counter interrupts

  ; Clear PPUADDR latch by reading PPUSTATUS
  bit $2002

  lda #$00
  sta $2001 ; Disable rendering

  ; Wait for vertical blank.
;- bit $2002
;  bpl -

  ; Clear PPUADDR latch by reading PPUSTATUS
  bit $2002

  ; MMC3: Set charbanks
  ldx #$00
  ldy #$00
  stx $8000 ; Select R0
  sty $8001 ; Set CHR bank 0 at $0000-$07FF
  iny
  iny       ; Y = 2
  inx       ; X = 1
  stx $8000 ; Select R1
  sty $8001 ; Set CHR bank 2 at $0800-$0FFF

  iny
  iny       ; Y = 4
  inx       ; X = 2
  stx $8000 ; Select R2
  sty $8001 ; Set CHR bank 4 at $1000-$13FF
  iny       ; Y = 5
  inx       ; X = 3
  stx $8000 ; Select R3
  sty $8001 ; Set CHR bank 5 at $1400-$17FF
  iny       ; Y = 6
  inx       ; X = 4
  stx $8000 ; Select R4
  sty $8001 ; Set CHR bank 6 at $1800-$1BFF
  iny       ; Y = 7
  inx       ; X = 5
  stx $8000 ; Select R5
  sty $8001 ; Set CHR bank 7 at $1C00-$1FFF

  ldy #$00
  inx       ; X = 6
  stx $8000 ; Select R6
  sty $8001 ; Set PRG bank 0 at $8000-$9FFF
  iny       ; Y = 1
  inx       ; X = 7
  stx $8000 ; Select R7
  sty $8001 ; Set PRG bank 1 at $A000-$BFFF

  ; Reset PPUSCROLL
  bit $2002
  lda #$00
  sta $2005 ; X scroll
  lda #$00
  sta $2005 ; Y scroll

  ;     VPHBSINN
  ldx #%10011000
  stx $2000 ; PPUCTRL

  ; Show nothing for one frame.
  ldx #%00000000
  stx $2001 ; PPUMASK

  ; Clear the zeropage
  ldx #$00
  txa
- sta $00,x
  inx
  bne -

  ; Clear the ingame variables
  ldx #$30
- dex
  sta $0550,x
  bne -

  ; Set up info nametables
  lda #$20
  sta $2006
  lda #$00
  sta $2006
  tax
!for i, 0, 3 {
- lda ENDING_NAMETABLE+i*256,x
  sta $2007
  inx
  bne -
}

  lda #NUM_LEVELS
  cmp PERFECT_COUNT
  beq +
  ; Blank out the "100% PERFECT" message
  lda #$21
  sta $2006
  lda #$80
  sta $2006
  lda #$00
  tax
- sta $2007
  inx
  bne -
+

  ; Copy sprites initial state to staging area
  ldx #$00
- lda INFO_SPRITES,x
  sta SPRITES_STAGING,x
  inx
  bne -

  ldx #1
  jsr init_music

  jsr read_joypad
  lda BUTTONS
  sta PREV_BUTTONS

  ; ----------------------------------------------------------

  ; Wait for NMI
  lda NMI_COUNTER
- cmp NMI_COUNTER
  beq -

  ; Set the BG palette
  lda #$3F
  sta $2006
  lda #$00
  sta $2006

  lda #$11
  sta $2007
  lda #$1D
  sta $2007
  lda #$21
  sta $2007
  lda #$20
  sta $2007

  ; Set the sprites palette
  lda #$3F
  sta $2006
  lda #$10
  sta $2006

  lda #$11 ; Aliased to $3F00
  sta $2007
  lda #$1D
  sta $2007
  lda #$28
  sta $2007
  lda #$28
  sta $2007

  ; Workaround for potential palette corruption bug
  lda #$3F
  sta $2006
  lda #$00
  sta $2006
  sta $2006
  sta $2006

  ; Reset PPUSCROLL
  bit $2002
  lda #$00
  sta $2005 ; X scroll
  lda #$00
  sta $2005 ; Y scroll

endingscreen_loop

  jsr update_music

  ; Wait for NMI
  lda NMI_COUNTER
- cmp NMI_COUNTER
  beq -

  ; ----------------------------------------------------------
  ; V-blank

  sta $E000 ; MMC3 IRQ acknowledge + disable

  ; Enable BG+sprites
  ldx #%00011000
  stx $2001 ; PPUMASK

  ; -------------- Joypad input --------------

  lda BUTTONS
  sta PREV_BUTTONS
  jsr read_joypad

  ; Write sprite data to OAM
  lda #$00
  sta $2003 ; OAMADDR
  lda #(SPRITES_STAGING>>8)
  sta $4014 ; OAMDMA
  ; DMA wait..

  ; -------------- Handle inputs --------------

  ; Start
  lda #(1<<4)
  bit BUTTONS
  beq +
  lda #(1<<4)
  bit PREV_BUTTONS
  bne +
  rts
+

  ; Update sprite positions
  ldx #(31*4)
-
  ; Back layer
  lda SPRITES_STAGING+0,x
  clc
  adc #3
  sta SPRITES_STAGING+0,x
  ; Front layer
  lda #$54
  sta SPRITES_STAGING+128+1,x
  lda SPRITES_STAGING+128+0,x
  clc
  adc #6
  sta SPRITES_STAGING+128+0,x
!for i, 0, 3 {
  dex
}
  bpl -


  inc FRAME_COUNT

  jmp endingscreen_loop


init_music
  txa
	jsr FamiToneMusicPlay
  rts

update_music
  lda PENDING_SFX
  beq +
  sec
  sbc #1
	ldx #FT_SFX_CH0
	jsr FamiToneSfxPlay
  lda #$00
  sta PENDING_SFX
+
  jsr FamiToneUpdate
  rts

do_dynamic_sprites
  ; -------------- Bonus pickup item sprites --------------

  lda #$00
  sta IN_SIGHTS_COUNT
  lda BONUSBALLS
  sta BONUSBALLS_PENDING
  lda #$FF
  sta BONUSBALL_TOWRITE

  lda #1
  sta TEMP

  ; Up to 8 pickups, 3 bytes per pickup (structure-of-arrays), 24 bytes in total.
  ldx #$00
- lda TEMP
  bit BONUSBALLS
  bne +++++
  lda CAM_Z
  clc
  adc #2
  and #63
  sec
  sbc LEVEL_METADATA,x ; Z
  cmp #0
  beq +
  cmp #1
  beq ++
  cmp #2
  beq +++
+++++
  inx
  clc
  rol TEMP
  cpx #8
  bne -
  jmp no_bonus_sprites

+
!for i, 0, 3 {
  lda STRIP_BOUNDS+2*4+i
  sta PICKUP_BOUNDS+i
}
  ; Perform clipping for the min Y bound since the
  ; strippoints algorithm doesn't do that.
  lda PICKUP_BOUNDS+2 ; min_y
  cmp STRIP_BOUNDS+1*4+2 ; min_y
bcs +
  lda STRIP_BOUNDS+1*4+2 ; min_y
  sta PICKUP_BOUNDS+2 ; min_y
+
  lda #(84*3) ; Sub-Z
  sec
  sbc CAM_SUB_Z+1
  jmp ++++
++
!for i, 0, 3 {
  lda STRIP_BOUNDS+1*4+i
  sta PICKUP_BOUNDS+i
}
  ; Perform clipping for the min Y bound since the
  ; strippoints algorithm doesn't do that.
  lda PICKUP_BOUNDS+2 ; min_y
  cmp #WINDOW_YMIN
bcs +
  lda #WINDOW_YMIN
  sta PICKUP_BOUNDS+2 ; min_y
+
  lda #(84*2) ; Sub-Z
  sec
  sbc CAM_SUB_Z+1
  jmp ++++
+++
  lda #0
  sta PICKUP_BOUNDS+0 ; min_x
  lda #255
  sta PICKUP_BOUNDS+1 ; max_x
  lda #WINDOW_YMIN
  sta PICKUP_BOUNDS+2 ; min_y
  lda #WINDOW_YMAX
  sta PICKUP_BOUNDS+3 ; max_y
  lda #(84*1) ; Sub-Z
  sec
  sbc CAM_SUB_Z+1
  ; Test for passing by
  cmp #8
  bcs +
  sta TEMP_1
  lda TEMP
  ora BONUSBALLS_PENDING
  sta BONUSBALLS_PENDING
  stx BONUSBALL_TOWRITE
  lda TEMP_1
+
++++

  stx TEMP

  ; Fetch Log(Z)
  tax
  lda LOG_Z_TABLE,x
  sta LOG_Z

  ; Determine base sprite pattern and palette
  tax
  lda LOGZ_TO_PICKUPFRAME,x
  sta PICKUP_FRAME

  ldx TEMP

  lda LEVEL_METADATA+8,x ; X
  sta TEMP_1
  lda LEVEL_METADATA+16,x ; Y
  sta TEMP_2


  ; Subtract camera X
  lda TEMP_1 ; Pickup X (two's-complement)
  sec
  sbc CAM_X
  sta TEMP

  ; Get absolute value
  bit TEMP
  bpl +
  ; Negate A
  eor #$FF
  clc
  adc #1
+ clc
  ; Update "in sights" state
  cmp #5
  bcs +
  inc IN_SIGHTS_COUNT
+

  lda TEMP
  beq +

  ; Exp(Log(X)-Log(Z))
  tax
  lda LOG_TABLE,x
  sec
  sbc LOG_Z
  tax
  lda EXP_TABLE,x
  
  ; Apply sign
  bit TEMP
  bpl +
  ; Negate A
  eor #$FF
  clc
  adc #1
+ clc
  adc #(128+4)
  sta PICKUP_XPROJ

  ; Sprite rolloff on the left and right sides of the screen isn't quite
  ; possible because the exp table pre-clamps. So just remove the sprite if it
  ; ends up on the sides.
  cmp #10
  bcc no_bonus_sprites
  cmp #251
  bcs no_bonus_sprites

  ; Subtract camera Y
  lda TEMP_2 ; Pickup Y (two's-complement)
  sec
  sbc CAM_Y
  sta TEMP

  ; Get absolute value
  bit TEMP
  bpl +
  ; Negate A
  eor #$FF
  clc
  adc #1
+ clc
  ; Update "in sights" state
  cmp #5
  bcs +
  inc IN_SIGHTS_COUNT
+

  lda TEMP
  beq +

  ; Exp(Log(Y)-Log(Z))
  tax
  lda LOG_TABLE,x
  sec
  sbc LOG_Z
  tax
  lda EXP_TABLE,x
  
  ; Apply sign
  bit TEMP
  bpl +
  ; Negate A
  eor #$FF
  clc
  adc #1
+ clc
  adc #128
  sta PICKUP_YPROJ

  jsr PICKUPSPRITES_ROUTINE
  jmp bonus_sprites_done

no_bonus_sprites

  ldx #16
  lda #$00
- sta SPRITES_STAGING,x
  inx
  cpx #192
  bne -

bonus_sprites_done

  rts

sound_alarm
  ; Sound countdown imminence alarms
  lda #5
  cmp COUNTER_UNITS
  bne +
  lda #SFX_5SEC
  sta PENDING_SFX
  rts
+ lda #4
  cmp COUNTER_UNITS
  bne +
  lda #SFX_4SEC
  sta PENDING_SFX
  rts
+ lda #3
  cmp COUNTER_UNITS
  bne +
  lda #SFX_3SEC
  sta PENDING_SFX
  rts
+ lda #2
  cmp COUNTER_UNITS
  bne +
  lda #SFX_2SEC
  sta PENDING_SFX
  rts
+ lda #1
  cmp COUNTER_UNITS
  bne +
  lda #SFX_1SEC
  sta PENDING_SFX
+ rts

sprites_timeover
!binary "sprites_timeover.bin"
sprites_timeover_end

sprites_crashed
!binary "sprites_crashed.bin"
sprites_crashed_end

sprites_won
!binary "sprites_won.bin"
sprites_won_end

sprites_paused
!binary "sprites_paused.bin"
sprites_paused_end

STRINGS
!convtab "acme_char_table.bin"
string_1_begin
  !text "TRY AGAIN"
string_1_end
string_2_begin
  !text "LET'S GO!"
string_2_end
string_3_begin
  !text "WELL DONE"
string_3_end
;string_4_begin
;  !text "PRESS"
;string_4_end
string_5_begin
  !text "LEVEL"
string_5_end
string_6_begin
  !text "GET READY"
string_6_end
string_7_begin
  !text "PERFECT!"
string_7_end
;string_8_begin
;  !text "SELECT TO"
;string_8_end
;string_9_begin
;  !text "RETRY"
;string_9_end

pickup_palettes
!binary "pickup_palettes.bin"

intermission_palette_0
!byte $1D, INTERMISSION_COLOUR_0, $1D, INTERMISSION_COLOUR_0
!byte $1D, INTERMISSION_COLOUR_1, $1D, INTERMISSION_COLOUR_1
!byte $1D, INTERMISSION_COLOUR_2, $1D, INTERMISSION_COLOUR_2
!byte $1D, INTERMISSION_COLOUR_3, $1D, INTERMISSION_COLOUR_3

intermission_palette_1
!byte $1D, $1D, INTERMISSION_COLOUR_0, INTERMISSION_COLOUR_0
!byte $1D, $1D, INTERMISSION_COLOUR_1, INTERMISSION_COLOUR_1
!byte $1D, $1D, INTERMISSION_COLOUR_2, INTERMISSION_COLOUR_2
!byte $1D, $1D, INTERMISSION_COLOUR_3, INTERMISSION_COLOUR_3

intermission_sprites_palette
!byte $1D, $00, $10, $20
!byte $1D, $05, $15, $25

hud_palette
!byte $1D, $1D, $10, $20 ; HUD text palette
!byte $1D, $2A, $28, $16 ; HUD speed indicator palette
!byte $1D, $14, $24, $20 ; HUD bonus balls palette
