// Pangolink

#include <cmath>
#include <cassert>
#include <cstdio>
#include <cstdlib>
#include <memory>
#include <cstring>

using namespace std;

/*

  System RAM layout:

  ; Zeropage
  $0000-$0004 : Misc. variables
  $0005-$0014 : Strip bounds
  $0015-$0019 : Misc. variables
  $0020-$007F : Nametable staging
  $0080-$00FE : Scrollpoints (staging and active)
  $00FF-$00FF : Scrollpoint index

  $0100-$01FF : Stack
  $0200-$02FF : Sprites staging area
  $0300-$04FF : Stripgen routine (512 bytes)
  $0500-$054F : IRQ handler (80 bytes)
  $0550-$057F : "Slow" ingame variables
  $0580-$05BF : Current level metadata
  $05A0-$05EF : Free (80 bytes)
  $05F0-$05FE : 15 bytes of "Slow" persistent variables
  $05FF-$05FF : NMI counter
  $0600-$06FF : FamiTone2 scratch area a.k.a. FT_BASE_ADR
  $0700-$07FF : Current level rects data

  PRG ROM layout (default banks):

  ----------------- Bank Window R6 -----------------
  $8000-$83FF : Info nametable
  $8400-$84FF : Info sprites
  $8500-$98FF : FamiTone2 music data - Limit: 5120 bytes
  $9900-$A0FF : FamiTone2 sound effects data
  ----------------- Bank Window R7 -----------------
  $A100-$A4FF : Pickup sprites construction routine
  $A500-$A5FF : Log(Z)-to-pickupframe table
  $A600-$A6FF : Level metadata (256/8=32 bytes per level)
  $A700-$AEFF : Logo scroll IRQ table
  $AF00-$BEFF : Stars nametables
  $B700-$BEFF : Titlescreen program (titlescreen.bin)
  $BF00-$BF7F : HUD initial nametable
  $BF80-$BFFF : Free (128 bytes)
  ----------------- Bank Window (-2) -----------------
  $C000-$C0FF : Log table
  $C100-$C1FF : Exp table
  $C200-$C2FF : Log(Z) table
  $C300-$CAFF : Rect array (4 bytes per rect, 64 rects per course, 2048 bytes = 8 levels)
  $CB00-$DFFF : Main program (main.bin) - Limit: 5376 bytes
  ----------------- Bank Window (-1) -----------------
  $E000-$E7FF : FamiTone2 replayer routine
  $E800-$EBFF : Info nametable 2
  $EC00-$EFFF : Ending nametable
  $F000-$F7FF : Stripsplat table
  $F800-$F9FF : Stripgen routine (strips.bin, is copied to RAM)
  $FD00-$FDFF : IRQ routine (irq.bin, is copied to RAM)
  $FF00-$FFEF : Reset routine (reset.bin)
  $FFF0-$FFF9 : Minimal NMI handler
  $FFFA-$FFFB : NMI vector
  $FFFC-$FFFD : Reset vector
  $FFFE-$FFFF : IRQ vector (points at RAM)

  CHR ROM:

  Bank 0: Strip patterns
  Bank 1: HUD font and misc. in-game patterns
  Bank 2: Star background patterns
  Bank 3: Free

*/

static const uint16_t org_address                 = 0x8000;

static const uint16_t info_nametable_address      = 0x8000;
static const uint16_t info_sprites_address        = 0x8400;
static const uint16_t music_address               = 0x8500;
static const uint16_t sfx_address                 = 0x9900;
//static const uint16_t sprites_timeover_address    = 0x9000;
//static const uint16_t sprites_crashed_address     = 0x9100;
//static const uint16_t sprites_won_address         = 0x9200;
static const uint16_t hud_nametable_address       = 0xBF00;
//static const uint16_t sprites_paused_address      = 0x9400;
static const uint16_t stars_nametables_address    = 0xAF00;
static const uint16_t pickupsprites_address       = 0xA100;
static const uint16_t logz_to_pickupframe_address = 0xA500;
static const uint16_t level_metadata_address      = 0xA600;
static const uint16_t logo_scroll_irq_address     = 0xA700;
static const uint16_t titlescreen_address         = 0xB700;
static const uint16_t logtable_address            = 0xC000;
static const uint16_t exptable_address            = 0xC100;
static const uint16_t logztable_address           = 0xC200;
static const uint16_t rects_address               = 0xC300;
static const uint16_t main_address                = 0xCB00;
static const uint16_t famitone2_address           = 0xE000;
static const uint16_t info_nametable2_address     = 0xE800;
static const uint16_t ending_nametable_address    = 0xEC00;
static const uint16_t stripsplat_address          = 0xF000;
static const uint16_t stripgen_address            = 0xF800;
static const uint16_t irq_address                 = 0xFD00;
static const uint16_t reset_address               = 0xFF00;
static const uint16_t nmi_address                 = 0xFFF0;


int main(int argc, char* argv[])
{
  printf("Compiled on " __DATE__ " at " __TIME__ ".\n");

  // 4 PRG banks (3 banks would be fine too but that does not work
  // on Everdrive N8 PRO for some reason).
  static const int prg_size=16384*2;

  assert(prg_size%16384==0);

  // 4 CHR banks
  static const int chr_size=256*16*4;

  assert(chr_size%8192==0);

  static const int mapper=4; // MMC3

  struct iNESHeader
  {
    uint8_t id[4];
    uint8_t num_prg;
    uint8_t num_chr;
    uint8_t flags_6;
    uint8_t flags_7;
    uint8_t flags_8;
    uint8_t flags_9;
    uint8_t flags_10;
    uint8_t padding[5];
  };

  static_assert(sizeof(iNESHeader)==16);

  iNESHeader header;

  memset(&header,0,sizeof(header));

  header.id[0]=0x4E;
  header.id[1]=0x45;
  header.id[2]=0x53;
  header.id[3]=0x1A;

  header.num_prg=prg_size/16384;
  header.num_chr=chr_size/8192;

  header.flags_6=0b00000000|(mapper << 4); // Horizontal mirroring (also set on MMC3 at runtime)
  header.flags_7=0b00000000|(mapper & 0xF0);
  header.flags_8=0;
  header.flags_9=0;
  header.flags_10=0;

  static uint8_t main_banks[16384*2];
  memset(main_banks,0,sizeof(main_banks));

  {
    FILE* in=fopen("info_nametable.bin","rb");
    assert(in);
    fseek(in,0,SEEK_END);
    const int len=ftell(in);
    fseek(in,0,SEEK_SET);
    assert(len==1024);
    fread(main_banks+info_nametable_address-org_address,1,len,in);
    fclose(in);
  }

  {
    FILE* in=fopen("info_nametable2.bin","rb");
    assert(in);
    fseek(in,0,SEEK_END);
    const int len=ftell(in);
    fseek(in,0,SEEK_SET);
    assert(len==1024);
    fread(main_banks+info_nametable2_address-org_address,1,len,in);
    fclose(in);
  }

  {
    FILE* in=fopen("ending_nametable.bin","rb");
    assert(in);
    fseek(in,0,SEEK_END);
    const int len=ftell(in);
    fseek(in,0,SEEK_SET);
    assert(len==1024);
    fread(main_banks+ending_nametable_address-org_address,1,len,in);
    fclose(in);
  }

  {
    FILE* in=fopen("info_sprites.bin","rb");
    assert(in);
    fseek(in,0,SEEK_END);
    const int len=ftell(in);
    fseek(in,0,SEEK_SET);
    assert(len==256);
    fread(main_banks+info_sprites_address-org_address,1,len,in);
    fclose(in);
  }

  // Main
  {
    FILE* in=fopen("main.bin","rb");
    assert(in);
    fseek(in,0,SEEK_END);
    const int len=ftell(in);
    fseek(in,0,SEEK_SET);
    assert(len<=(0xE000-main_address));
    fread(main_banks+main_address-org_address,1,len,in);
    fclose(in);
  }

/*
  {
    FILE* in=fopen("sprites_timeover.bin","rb");
    assert(in);
    fseek(in,0,SEEK_END);
    const int len=ftell(in);
    fseek(in,0,SEEK_SET);
    assert(len==256);
    fread(main_banks+sprites_timeover_address-org_address,1,len,in);
    fclose(in);
  }

  {
    FILE* in=fopen("sprites_crashed.bin","rb");
    assert(in);
    fseek(in,0,SEEK_END);
    const int len=ftell(in);
    fseek(in,0,SEEK_SET);
    assert(len==256);
    fread(main_banks+sprites_crashed_address-org_address,1,len,in);
    fclose(in);
  }

  {
    FILE* in=fopen("sprites_won.bin","rb");
    assert(in);
    fseek(in,0,SEEK_END);
    const int len=ftell(in);
    fseek(in,0,SEEK_SET);
    assert(len==256);
    fread(main_banks+sprites_won_address-org_address,1,len,in);
    fclose(in);
  }
*/

  {
    FILE* in=fopen("hud_nametable.bin","rb");
    assert(in);
    fseek(in,0,SEEK_END);
    const int len=ftell(in);
    fseek(in,0,SEEK_SET);
    assert(len==32*4);
    fread(main_banks+hud_nametable_address-org_address,1,len,in);
    fclose(in);
  }

/*
  {
    FILE* in=fopen("sprites_paused.bin","rb");
    assert(in);
    fseek(in,0,SEEK_END);
    const int len=ftell(in);
    fseek(in,0,SEEK_SET);
    assert(len==256);
    fread(main_banks+sprites_paused_address-org_address,1,len,in);
    fclose(in);
  }
*/

  {
    FILE* in=fopen("stars_nametables.bin","rb");
    assert(in);
    fseek(in,0,SEEK_END);
    const int len=ftell(in);
    fseek(in,0,SEEK_SET);
    assert(len==2048);
    fread(main_banks+stars_nametables_address-org_address,1,len,in);
    fclose(in);
  }

  {
    FILE* in=fopen("pickupsprites.bin","rb");
    assert(in);
    fseek(in,0,SEEK_END);
    const int len=ftell(in);
    fseek(in,0,SEEK_SET);
    assert(len<=1024);
    fread(main_banks+pickupsprites_address-org_address,1,len,in);
    fclose(in);
  }

  {
    FILE* in=fopen("logz_to_pickupframe.bin","rb");
    assert(in);
    fseek(in,0,SEEK_END);
    const int len=ftell(in);
    fseek(in,0,SEEK_SET);
    assert(len==256);
    fread(main_banks+logz_to_pickupframe_address-org_address,1,len,in);
    fclose(in);
  }

  {
    FILE* in=fopen("level_metadata.bin","rb");
    assert(in);
    fseek(in,0,SEEK_END);
    const int len=ftell(in);
    fseek(in,0,SEEK_SET);
    assert(len<=256);
    fread(main_banks+level_metadata_address-org_address,1,len,in);
    fclose(in);
  }

  {
    FILE* in=fopen("logo_scroll_irq.bin","rb");
    assert(in);
    fseek(in,0,SEEK_END);
    const int len=ftell(in);
    fseek(in,0,SEEK_SET);
    assert(len<=2048);
    fread(main_banks+logo_scroll_irq_address-org_address,1,len,in);
    fclose(in);
  }

  {
    FILE* in=fopen("titlescreen.bin","rb");
    assert(in);
    fseek(in,0,SEEK_END);
    const int len=ftell(in);
    fseek(in,0,SEEK_SET);
    assert(len<=2048);
    fread(main_banks+titlescreen_address-org_address,1,len,in);
    fclose(in);
  }

  {
    FILE* in=fopen("log_table.bin","rb");
    assert(in);
    fread(main_banks+logtable_address-org_address,1,256,in);
    fclose(in);
  }

  {
    FILE* in=fopen("exp_table.bin","rb");
    assert(in);
    fread(main_banks+exptable_address-org_address,1,256,in);
    fclose(in);
  }

  {
    FILE* in=fopen("log_z_table.bin","rb");
    assert(in);
    fread(main_banks+logztable_address-org_address,1,256,in);
    fclose(in);
  }

  // Strip splat table
  {
    FILE* in=fopen("strip_splat_table.bin","rb");
    assert(in);
    fseek(in,0,SEEK_END);
    const int len=ftell(in);
    fseek(in,0,SEEK_SET);
    assert(len==256*8);
    fread(main_banks+stripsplat_address-org_address,1,len,in);
    fclose(in);
  }

  // Strip generation routine in ROM (gets copied to RAM)
  {
    FILE* in=fopen("strips.bin","rb");
    assert(in);
    fseek(in,0,SEEK_END);
    const int len=ftell(in);
    fseek(in,0,SEEK_SET);
    assert(len<=512);
    fread(main_banks+stripgen_address-org_address,1,len,in);
    fclose(in);
  }

  // IRQ handler in ROM (gets copied to RAM)
  {
    FILE* in=fopen("irq.bin","rb");
    assert(in);
    fseek(in,0,SEEK_END);
    const int len=ftell(in);
    fseek(in,0,SEEK_SET);
    assert(len<=80);
    fread(main_banks+irq_address-org_address,1,len,in);
    fclose(in);
  }

  // Reset routine
  {
    FILE* in=fopen("reset.bin","rb");
    assert(in);
    fseek(in,0,SEEK_END);
    const int len=ftell(in);
    fseek(in,0,SEEK_SET);
    assert(len<=256-16);
    fread(main_banks+reset_address-org_address,1,len,in);
    fclose(in);
  }

  // Rects
  {
    FILE* in=fopen("rect_array.bin","rb");
    assert(in);
    fseek(in,0,SEEK_END);
    const int len=ftell(in);
    fseek(in,0,SEEK_SET);
    assert(len<=2048);
    fread(main_banks+rects_address-org_address,1,len,in);
    fclose(in);
  }

  // FamiTone2
  {
    FILE* in=fopen("famitone2/famitone2_asm6.bin","rb");
    assert(in);
    fseek(in,0,SEEK_END);
    const int len=ftell(in);
    fseek(in,0,SEEK_SET);
    assert(len<=2048);
    fread(main_banks+famitone2_address-org_address,1,len,in);
    fclose(in);
  }

  // Music
  {
    FILE* in=fopen("fizzy3_lau8.bin","rb");
    assert(in);
    fseek(in,0,SEEK_END);
    const int len=ftell(in);
    fseek(in,0,SEEK_SET);
    assert(len<=5120);
    fread(main_banks+music_address-org_address,1,len,in);
    fclose(in);
  }

  // SFX
  {
    FILE* in=fopen("fizzy3_lau7_SFX.bin","rb");
    assert(in);
    fseek(in,0,SEEK_END);
    const int len=ftell(in);
    fseek(in,0,SEEK_SET);
    assert(len<=2048);
    fread(main_banks+sfx_address-org_address,1,len,in);
    fclose(in);
  }

  // Minimal NMI
  main_banks[nmi_address - org_address + 0] = 0xEE; // INC $05FF
  main_banks[nmi_address - org_address + 1] = 0xFF;
  main_banks[nmi_address - org_address + 2] = 0x05;
  main_banks[nmi_address - org_address + 3] = 0x40; // RTI

  *(uint16_t*)(main_banks + 0xFFFA - org_address) = nmi_address; // NMI
  *(uint16_t*)(main_banks + 0xFFFC - org_address) = reset_address; // RESET
  *(uint16_t*)(main_banks + 0xFFFE - org_address) = 0x0500; // IRQ/BRK (in RAM)


  static uint8_t prg[prg_size]; // All banks combined
  memset(prg,0,sizeof(prg));

  // In the below code, "bank" refers to an 8kb MMC3 bank, not
  // a 16kb iNES bank.

  // Fixed banks (-1) and (-2)
  memcpy(&prg[prg_size-0x2000*2],&main_banks[0x2000*2],0x2000);
  memcpy(&prg[prg_size-0x2000*1],&main_banks[0x2000*3],0x2000);

  // "Default"-mapped banks 0 and 1
  memcpy(&prg[0x2000*0],&main_banks[0x2000*0],0x2000);
  memcpy(&prg[0x2000*1],&main_banks[0x2000*1],0x2000);

/*
  // Music data banks 2 and 3
  {
    //FILE* in=fopen("danger_streets.bin","rb");
    //FILE* in=fopen("fizzy3.bin","rb");
    //FILE* in=fopen("fizzy3_per2.bin","rb");
    FILE* in=fopen("fizzy3_per3.bin","rb");
    assert(in);
    fseek(in,0,SEEK_END);
    const int len=ftell(in);
    fseek(in,0,SEEK_SET);
    assert(len<0x1000);
    fread(&prg[0x2000*2],1,0x1000,in);
    fclose(in);
  }

  // SFX data
  {
    FILE* in=fopen("fizzy3_per2_SFXb.bin","rb");
    assert(in);
    fseek(in,0,SEEK_END);
    const int len=ftell(in);
    fseek(in,0,SEEK_SET);
    assert(len<0x1000);
    fread(&prg[0x2000*2+0x1000],1,0x1000,in);
    fclose(in);
  }
*/

  static uint8_t chr[chr_size];
  memset(chr,0,sizeof(chr));

  {
    FILE* in=fopen("patterns.bin","rb");
    assert(in);
    fseek(in,0,SEEK_END);
    const int len=ftell(in);
    fseek(in,0,SEEK_SET);
    assert(len<=chr_size);
    fread(chr,1,sizeof(chr),in);
    fclose(in);
  }

  {
    FILE* in=fopen("pickup.bin","rb");
    assert(in);
    fseek(in,0,SEEK_END);
    const int len=ftell(in);
    fseek(in,0,SEEK_SET);
    assert(len<=1024);
    fread(chr+4096*1+0xC0*16,1,len,in);
    fclose(in);
  }

  {
    FILE* in=fopen("title_screen_graphic_scaled_2bpp.bin","rb");
    assert(in);
    fseek(in,0,SEEK_END);
    const int len=ftell(in);
    fseek(in,0,SEEK_SET);
    assert(len<=2048);
    fread(chr+4096*3+0x80*16,1,len,in);
    fclose(in);
  }

  FILE* out=fopen("output.nes","wb");
  fwrite(&header,1,sizeof(header),out);
  fwrite(prg,1,sizeof(prg),out);
  fwrite(chr,1,sizeof(chr),out);
  fclose(out);
}
